/*
  diskerase.c  --  writes a binary pattern to a device (disk)
  Copyright (C) 1998,1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include	"diskerase.h"

int display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Writes pattern periodically to a device.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] device\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -f\t force erasing\n" );
  fprintf( HELP_CHANNEL, "  -h\t help, write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -v\t be talkative on %s\n", STD_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation settings to %s and exit sucessfully\n", VERSION_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -H\t SIZE\t write hash marks to %s (one per SIZE bytes)\n", STD_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -p\t PATT\t write byte sequence PATT to device\n" );
  fprintf( HELP_CHANNEL, "  -s\t STRING\t write string STRING to device (without the terminating `\\0')\n" );
  return( 0 );
}


int display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "Compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "BUFFER_SIZE: %d\n", BUFFER_SIZE );
  return( 0 );
}


int ss_str_stp( char* str1, char* str2 ) {
/*
SOLYGA --------------------
SOLYGA int ss_str_stp(str1,str2)
SOLYGA string function stop:
SOLYGA writes '\0' to first char in str1 matching one of the chars from str2,
SOLYGA returns number of deleted characters

SOLYGA started      : 97/03/28
Steffen Solyga <solyga@tetibm2.ee.TU-Berlin.DE>
*/
  char *p1,*p2;
  int n=0;
  for (p1=str1;;p1++) {
    if (*p1=='\0') return(n);
    for (p2=str2;;p2++) {
      if (*p2=='\0') break;
      if (*p2==*p1) {
        *p1='\0';
        while(*(p1+++n)!='\0');
        return(n);
      }
    }
  }
}


char *ss_str_dsc( char* str1, char* str2) {
/*
SOLYGA --------------------
SOLYGA char *ss_str_dsc(str1,str2)
SOLYGA string function discard:
SOLYGA returns pointer to first char of str1 not contained in str2

SOLYGA started      : 97/02/09
SOLYGA last modified: Sat Jun 28 22:40:06 GMT 1997 @beast

Steffen Solyga <solyga@tetibm2.ee.TU-Berlin.DE>
*/
  char *p1,*p2;
  for (p1=str1;;p1++) {
    if (*p1=='\0') return(p1);
    for (p2=str2;;p2++) {
      if (*p2=='\0') return(p1);
      if (*p2==*p1) break;
    }
  }
}


int ss_str_cntchr( char* str, char chr ) {
/*
SOLYGA --------------------
SOLYGA int ss_str_cntchr(str,chr)
SOLYGA string function string contains character:
SOLYGA returns 1 if so, 0 else

SOLYGA started      : Thu Jul 3 19:37:48 GMT 1997 @beast
SOLYGA last modified: Thu Jul 3 19:37:48 GMT 1997 @beast
Steffen Solyga <solyga@tetibm2.ee.TU-Berlin.DE>
*/
  char *p=str;
  for(p=str;*p!='\0';p++) if(*p==chr) return(1);
  return(0);
}


int main( int argc, char** argv ) {
/*
SOLYGA --------------------
SOLYGA main() diskerase
SOLYGA writes a string to entire drive

SOLYGA started      : Mon Dec 21 20:57:05 CET 1998 @beast
*/
  int c;
  int retval= 0;	/* return value */
  int of_verbose= 0;	/* option flag: verbose */
  int of_force= 0;	/* option flag: force */
  int of_string= 0;	/* option flag: string specified */
  int of_pattern= 0;	/* option flag: pattern specified */
  int of_hash= 0;	/* option flag: print hashes */
  int fd;
  char* device;
  char inbuf[]= "  ";			/* for question/answer */
  unsigned char buffer[BUFFER_SIZE];	/* uchar for extension to all bytes */
  ssize_t buffer_size;			/* size used */
  ssize_t pattern_size;			/* size of pattern */
  ssize_t nbw= 0;
  long long int tnbw= 0;
  long long hash_size;
  long long int nhw= 0;			/* number of hashes written */
  int i;
  int is_string;			/* for question */
  /* initialize buffer */
  *buffer= DEFAULT_BYTE;
  pattern_size= 1;
  while( (c=getopt(argc,argv,"fhH:p:s:vV")) != EOF ) {
    switch( c ) {
      case 'f': /* force erasing */
        of_force= 1;
        break;
      case 'H': /* print hash marks after given number of bytes */
        of_hash= 1;
        {
          char* p= ss_str_dsc( optarg, DIVIDERS );
          char* pold= p;
          hash_size= strtoq( p, &p, 0 );
          if( p==pold || hash_size==0 ) {
            fprintf( ERROR_CHANNEL, "%s: Bad size format `%s'.\n",
                     *argv, optarg );
            retval= 1; goto DIE_NOW;
          }
          switch( *(p=ss_str_dsc(p,DIVIDERS)) ) {
            case '\0': break;
            case 'G' : hash_size*= 1024;
            case 'M' : hash_size*= 1024;
            case 'k' : hash_size*= 1024; break;
            default  :
              fprintf( ERROR_CHANNEL, "%s: Bad unit `%s'.\n", *argv, p );
              retval= 1; goto DIE_NOW;
          }
        }
        break;
      case 'h': /* display help to HELP_CHANNEL and exit sucessfully */
        display_help( *argv );
        retval= 0; goto DIE_NOW;
      case 'p': /* get pattern to write */
        if( of_string ) {
          fprintf( ERROR_CHANNEL, "%s: String and pattern specified.\n",
                   *argv );
          retval= 1; goto DIE_NOW;
        }
        of_pattern= 1;
        pattern_size= 0;
        {
          long val;
          char* p= optarg;
          char* pold;
          while( 1 ) {
            if( *(p=ss_str_dsc(p,DIVIDERS)) == '\0' ) break;
            if( pattern_size >= BUFFER_SIZE ) {
              fprintf( ERROR_CHANNEL, "%s: Pattern too long.\n", *argv );
              retval= 1; goto DIE_NOW;
            }
            pold= p;
            val= strtol( p, &p, 0 );
            if( p == pold ) {
              ss_str_stp( pold, DIVIDERS );
              fprintf( ERROR_CHANNEL, "%s: `%s' is not a number.\n",
                       *argv, pold );
              retval= 1; goto DIE_NOW;
            }
            if( val<0L || val>255L ) {
              ss_str_stp( pold, DIVIDERS );
              fprintf( ERROR_CHANNEL, "%s: `%s' does not fit in one byte.\n",
                       *argv, pold );
              retval= 1; goto DIE_NOW;
            }
            buffer[pattern_size++]= (unsigned char) val;
          }
        }
        break;
      case 's': /* get string to write */
        if( of_pattern ) {
          fprintf( ERROR_CHANNEL, "%s: Pattern and string specified.\n", 
                   *argv );
          retval= 1; goto DIE_NOW;
        }
        of_string= 1;
        if( (pattern_size=strlen(optarg)) > BUFFER_SIZE ) {
          fprintf( ERROR_CHANNEL, "%s: String too long.\n",
                   *argv );
          retval= 1; goto DIE_NOW;
        }
        for( i=0; i<pattern_size; i++ ) buffer[i]= optarg[i];
        break;
      case 'v': /* be verbose */
        of_verbose= 1;
        break;
      case 'V': /* print version info and exit successfully */
        display_version( *argv );
        retval= 0; goto DIE_NOW;
      case '?': /* refer to help and exit unsuccessfully */
        fprintf( ERROR_CHANNEL, "Try `%s -h' for more information.\n", *argv );
        exit(1);
      default : /* program error */
        fprintf( ERROR_CHANNEL, "Steffen, check the options string!\n" );
        retval= 1; goto DIE_NOW;
    }
  }
  if( pattern_size == 0 ) {
    fprintf( ERROR_CHANNEL, "%s: String or pattern too short.\n", *argv );
    retval= 1; goto DIE_NOW;
  }
#ifdef	DEBUG
fprintf( DEBUG_CHANNEL, "DEBUG: pattern is " );
for( i=0; i<pattern_size; i++ )
  fprintf( DEBUG_CHANNEL, "0x%02x ", buffer[i] );
fprintf( DEBUG_CHANNEL, "\n" );
#endif
  if( argc <= optind ) {
    fprintf( ERROR_CHANNEL, "%s: No device specified.\n", *argv );
    retval= 1; goto DIE_NOW;
  }
  device= argv[optind];
  /* fill buffer */
  buffer_size= ( BUFFER_SIZE / pattern_size ) * pattern_size;
  for( i=pattern_size; i<buffer_size; i++ ) buffer[i]= buffer[i%pattern_size];
#ifdef	DEBUG
fprintf( DEBUG_CHANNEL, "DEBUG: pattern size= %d\n", pattern_size );
fprintf( DEBUG_CHANNEL, "DEBUG: buffer size = %d\n", buffer_size );
#endif
  /* check for ascii */
  is_string= 1;
  for( i=0; i<pattern_size; i++ ) is_string&= IS_CHAR(buffer[i]);
  /* ask user */
  if( !of_force ) {
    fprintf( STD_CHANNEL, "You're going to overwrite entire device `%s' with ", device );
    if( is_string ) {
      fprintf( STD_CHANNEL, "string `" );
      for( i=0; i<pattern_size; i++ )
        fprintf( STD_CHANNEL, "%c", buffer[i] );
      fprintf( STD_CHANNEL, "'.\n" );
    }
    else {
      fprintf( STD_CHANNEL, "pattern " );
      for( i=0; i<pattern_size-1; i++ )
        fprintf( STD_CHANNEL, "0x%02x ", buffer[i] );
      fprintf( STD_CHANNEL, "0x%02x.\n", buffer[i] );
    }
    fprintf( STD_CHANNEL, "This will destroy its data without any chance of recovery.\n" );
    fprintf( STD_CHANNEL, "Do you know about what you're doing? [y/n]: " );
    fflush( STD_CHANNEL );
    read(0,inbuf,2);
    if( *inbuf != 'y' ) { retval= 0; goto DIE_NOW; }
  }
  if( of_verbose ) {
    fprintf( STD_CHANNEL, "Writing one hash per %lld bytes.\n", hash_size );
  }
  /* open */
  if( (fd=open(device,O_WRONLY)) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' for writing. %s.\n",
             *argv, device, sys_errlist[errno] );
    retval= 1; goto DIE_NOW;
  }
  /* write */
  tnbw= 0;
  while( 1 ) {
    if( (nbw=write(fd,buffer,buffer_size)) == -1 ) {
      /* breakpoint for whole disks like /dev/hda */
      if( of_hash ) fprintf( STD_CHANNEL, "\n" );
      if( of_verbose )
        fprintf( STD_CHANNEL, "%s `%s'.\n", sys_errlist[errno], device );
      break;
    }
    tnbw+= nbw;
    if( of_hash  &&  tnbw >= (1+nhw)*hash_size ) {
      fprintf( STD_CHANNEL, "#" ); fflush( STD_CHANNEL );
      nhw++;
    }
    if( nbw < buffer_size ) {
      /* breakpoint for partitions like /dev/hda1 */
      if( of_hash && nhw>0 ) fprintf( STD_CHANNEL, "\n" );
      break;
    }
  }
  if( of_verbose )
    fprintf( STD_CHANNEL, "%lld bytes written.\n", tnbw );
  /* close */
  if( close(fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Couldn't close `%s' successfully.\n",
             *argv, device );
    retval= 1; goto DIE_NOW;
  }
DIE_NOW:
  exit( retval );
}
