/***
 * CopyPolicy: GNU Public License 2 applies
 * Copyright (C) by Monty (xiphmont@mit.edu)
 *
 * paranoia.c; other than the two blocks of new code in cdda2wav.c,
 * this and lpc.c is where the paranoia functions are.  
 * See README.paranoia */

/* KNOWN BUGS: ************************************************************

   Paranoia deals suboptimally if the beginning/end of a CD track is
     badly toasted (it guesses).  

   Scratches still occasionally sneak through auto-detection; if
   isolated pops are still appearing in the output of a damaged disc,
   use the 'assume all data is potentially damaged' -z commandline
   option

   After finding a scratch, we are unable to check previously written
   n frames in the same area of the surface for undetected scratches
   due to being unable to seek back in the written stream

** IMPROVEMENTS TO COME: **************************************************

   None of this is particularly well optimized at the moment.  I
     wanted it to be solid before fast.  

   Cache retries like before?  Right now a bad read (sync dies in the
     overlap) will result in the next match failing because the code
     currently wants two good reads in a *row*, not two good reads
     period. 

   dynamic overlap management; watch the jitter and optimize 

   IIR based culling (pop detection); right now we just use averaged deltas

***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <math.h>
#include <sys/types.h>
#include "mycdrom.h"
#include "interface.h"
#include "cdda2wav.h"
#include "resample.h"
#include "global.h"
#include "paranoia.h"
#include "lpc.h"

long endlap=0;
int paranoid=-1;
int scratch=-1;
int repair=-1;
float p_cull_mult=8.;
int scratched_force=0;

/* cast hacks: kids, don't do this at home! It's a bad habit. */
/* Don't use globals either. :-) */

/* How many longs in test1 and test2 match beginning at [0]? */
long paranoia_lapcount(long *test1, long *test2, long leng){
  long count=0;
  while(count++<leng)
    if(*(test1++)!=*(test2++))
      return(count-1);
  return(leng);
}

/* Looks for the best overlap exceeding P_OVERLAP; returns the
   length of overlap in *overlapped, and the position where overlap begins
   relative to test1[0]; negative alignment means the overlap began at
   test1[0] */

long paranoia_bi_find(long *test1, long *test2, long limit, long *overlapped){
  long count;
  long scanahead=0;
  long best=0;
  long bestpos=0;

  /* We also need to handle long stretches of silence */

  if(test1[0]==test2[0])
    for(scanahead=0;scanahead<limit-P_OVERLAP;scanahead++)
      if(test1[scanahead]!=test1[scanahead+1] ||
	 test2[scanahead]!=test2[scanahead+1])
	break;

  if(scanahead==limit-P_OVERLAP){
    *overlapped=limit-P_OVERLAP;
    return(0);
  }

  for(count=0;count<limit-P_OVERLAP-scanahead;count++){
    if((*overlapped=paranoia_lapcount(test1+scanahead,test2+count+scanahead,
				      limit-count-scanahead)+scanahead)>best){
      best=*overlapped;
      bestpos=-count;
    }
    if(count+best>=limit)break;
    if(count>0)
      if((*overlapped=paranoia_lapcount(test1+count+scanahead,test2+scanahead,
				       limit-count-scanahead)+scanahead)>best){
	best=*overlapped;
	bestpos=count;
      }
    if(count+best>=limit)break;
  }
  if(best>=P_OVERLAP){
    *overlapped=best;
    return(bestpos);
  }
  
  return(INT_MAX);
}

/* This returns the *new* overlap and alignment */
/* Alignment here is bytes into paranoia that the new overlap
   begins */

long paranoia_uni_find(long *test1, long *test2, long verify, long sofar,
		      long paralign, long limit, long *overlapped){
  long count;
  long best=-1;
  long bestpos=0;
  long scanahead=verify-P_OVERLAP;
  
  for(;scanahead<sofar;scanahead++)
    if(test1[scanahead-1]!=test1[scanahead])break;

  if(scanahead==sofar){
#ifdef P_DEBUG
  fprintf(stderr,"uni_find: all silence.  Returning\n");
#endif

    *overlapped=sofar-verify;
    return(paralign);
  }

  for(count=0;count<limit-P_OVERLAP;count++){
    long check_bytes=min(sofar-scanahead,limit-count);
    if((*overlapped=paranoia_lapcount(test1+scanahead,
				      test2+count,
				      check_bytes))>best){
      best=*overlapped-P_OVERLAP;
      bestpos=count+P_OVERLAP;
    }
  }
  if(best>=0){
    *overlapped=best;
#ifdef P_DEBUG
    fprintf(stderr,"uni_find: Returning alignment:%ld overlap:%ld\n",
	bestpos,*overlapped  );
#endif
    return bestpos;
  }  
  return(INT_MAX);
}

long paranoia_correlate(long *m,long *n,long l1,long l2,long cutoff){
  long limit=min(l1,l2);
  long acc=0,count=0;

  limit=min(limit,cutoff);

  while(count++<limit)if(*(--m)==*(--n))acc++;
  return(acc);
}

/* Find alignment of a scratch */
/* The CD in question is probably fairly toasted, and we're recovering
   alignment as best as possible. This function is expensive and
   currently just uses brute force. */

long paranoia_uni_find_tolerant(long *ref,long *rabble,long verify,long sofar,
				 long paralign,long rlimit){
  long backoff,count,best=0,bestpos=0,test;

  /* Brute force for now.... do jump tables some other time */

  for(backoff=0;backoff<=CD_FRAMESAMPLES-P_OVERLAP;backoff+=P_OVERLAP){
    /* If we don't sync right at the limit, search more distance, but not
       more than a sector */
    
    for(count=0;count<paralign-P_OVERLAP;count++){
      /* Check oneup and onedown */
      
      /*Down*/
      test=paranoia_correlate(ref+verify-backoff,
			      rabble+paralign-count-1-backoff,
			      verify-backoff,
			      paralign-count-1-backoff,P_OVERLAP);
      if(test>best){
	best=test;
	bestpos=paralign-count-1;
      }
      
      /*Up*/
      if(count+paralign<=rlimit){
	test=paranoia_correlate(ref+verify-backoff,
				rabble+paralign+count-backoff,
				verify-backoff,
				paralign+count-backoff,P_OVERLAP);
	if(test>best){
	  best=test;
	  bestpos=paralign+count;
	}
      }
    }
    
    /* Well, this is a bit arbitrary... */
    if(best>32){
#ifdef P_DEBUG
      fprintf(stderr,"Uni tolerant returning:%ld acc:%ld\n",bestpos,best);
#endif
      
      return bestpos;
    }  
  }
  return(INT_MAX);
}

long paranoia_overlap_tolerant(long *build,long *suspectbits,long *sc,
			       long pverify,long psofar,long alignment,
			       long plimit){
  long count;
  long offset=alignment-pverify;
  long acc=0;
  long ret=-1;

  for(count=psofar-1;count>=pverify;count--){ 
    if(build[count]==sc[count+offset]){	
      acc++; 
      if(acc==P_OVERLAP)ret=count-pverify+1;
    }else
      if(suspectbits && ret>-1)suspectbits[count]=-1;
  }

  return(ret);
}

#define P_AV_LENGTH 32

void paranoia_cull(long *paranoia,long *suspectbits,long limit){
  if(suspectbits){
    long count;

    long accL=0;
    long accR=0;
    long queueL[P_AV_LENGTH];
    long queueR[P_AV_LENGTH];
    long prevL;
    long prevR;
    long prevpos;
    long leng=0;
    long ptr=0;
    short *p=(short *)paranoia;

    /* OK; we look for samples that have abnormally high slew */
    /* Go backward; this accounts for the fact that audio attack is a forward
       scanning phenomenon and decay is usually a slow exponential */
    /* Don't worry about the very end; that's what the overlap is for */

    count=limit-1;
    while(suspectbits[count] && count>=0)count--;

    prevpos=count;
    prevL=p[count<<1];
    prevR=p[(count<<1)+1];
    count--;

    for(;count>=0;count--){
      if(suspectbits[count]==0){
	long valL=p[count<<1];
	long valR=p[(count<<1)+1];
	long slewL=labs(valL-prevL)/(prevpos-count);
	long slewR=labs(valR-prevR)/(prevpos-count);
	long valid=-1;

	if(leng==P_AV_LENGTH){
	  if(slewL>accL*p_cull_mult/leng ||
	     slewR>accR*p_cull_mult/leng)valid=0;
	  
	  if(valid==0 && count>0 && !suspectbits[count-1]){
	    int nextL=p[(count-1)<<1];
	    int nextR=p[((count-1)<<1)+1];
	    
	    if((nextL<valL && valL<prevL) ||
	       (nextL>valL && valL>prevL))
	      if((nextR<valR && valR<prevR) ||
		 (nextR>valR && valR>prevR))
		valid=0;
	  }
	  
	  if(!valid)
	    suspectbits[count]=-1;
	  else{
	    accL-=queueL[ptr];
	    accR-=queueR[ptr];
	    leng--;
	  }
	}
	
	if(suspectbits[count]==0){
	  leng++;
	  accL+=queueL[ptr]=slewL;
	  accR+=queueR[ptr++]=slewR;
	  if(ptr>=P_AV_LENGTH)ptr=0;
	  prevL=valL;
	  prevR=valR;
	  prevpos=count;
	}
      }	
    }
  }
}

void paranoia_spackle(long *paranoia,long *suspectbits,long limit){
  lpc_smart_spackle(paranoia,suspectbits,limit);
}

void indicator(long sofar,long total,char happiness,const char *s){
  static int state=0;
  static char happistate=')';
  const int els=52;
  char buffer[80];
  static char statussave[53];
  static int begin=0;

  if(!begin){
    begin=-1;
    memset(statussave,'.',els+1);
  }

  if(global.verbose>1){
    int bars=(float)sofar/total*els+.5;
    if(!s)state++;
    if(state>6)state=0;
    if(happiness==0)happiness=happistate;
    happistate=happiness;

    sprintf(buffer,"\r    (==[ :-%c ]==[",happiness);
    switch(happiness){
    case '|':
      if(statussave[bars]=='.')
	statussave[bars]=':';
      break;
    case '0':
      if(statussave[bars]=='V'||statussave[bars]=='#')
	statussave[bars]='#';
      else
	statussave[bars]='*';
      break;
    case '(':
      if(statussave[bars]=='*'||statussave[bars]=='#')
	statussave[bars]='#';
      else
	statussave[bars]='V';
      break;
    }

    if(s){
      /* User supplied message */
      int togo=els-strlen(s)-3;
      strcat(buffer," ");
      strcat(buffer,s);
      strcat(buffer," ");
      while(togo--)strcat(buffer," ");
      strcat(buffer," ]==) ");
    }else{
      /* progress indicator */
      int spaces=els-bars;
      int count=0;
      while(bars--)strncat(buffer,statussave+(count++),1);

      if(spaces>0){
	switch(state){
	case 0:
	  strcat(buffer," ");
	  break;
	case 1:case 6:
	  strcat(buffer,".");
	  break;
	case 2:case 5:
	  strcat(buffer,"o");
	  break;
	case 3:case 4:
	  strcat(buffer,"O");
	  break;
	}
	spaces--;
      }
      while(spaces--) strcat(buffer," ");
      strcat(buffer,"]==) ");
    }
    fprintf(stderr,buffer);
  }
}



