/*
 * Copyright: GNU Public License 2 applies
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * CDDA2WAV (C) 1995-1997 Heiko Eissfeldt heiko@colossus.escape.de
 * parts    (C) Peter Widow
 * parts    (C) Thomas Niederreiter
 * parts    (C) RSA Data Security, Inc.
 *
 * last changes:
 *   18.12.93 - first version,	OK
 *   01.01.94 - generalized & clean up HE
 *   10.06.94 - first linux version HE
 *   12.06.94 - wav header alignment problem fixed HE
 *   12.08.94 - open the cdrom device O_RDONLY makes more sense :-)
 *		no more floating point math
 *		change to sector size 2352 which is more common
 *		sub-q-channel information per kernel ioctl requested
 *		doesn't work as well as before
 *		some new options (-max -i)
 *   01.02.95 - async i/o via semaphores and shared memory
 *   03.02.95 - overlapped reading on sectors
 *   03.02.95 - generalized sample rates. all integral divisors are legal
 *   04.02.95 - sun format added
 *              more divisors: all integral halves >= 1 allowed
 *		floating point math needed again
 *   06.02.95 - bugfix for last track and not d0
 *              tested with photo-cd with audio tracks
 *		tested with xa disk 
 *   29.01.96 - new options for bulk transfer
 *   01.06.96 - tested with enhanced cd
 *   01.06.96 - tested with cd-plus
 *   02.06.96 - support pipes
 *   02.06.96 - support raw format
 *   04.02.96 - security hole fixed
 *   22.04.97 - large parts rewritten
 *   28.04.97 - make file names DOS compatible
 *   01.09.97 - add speed control
 *   20.10.97 - add find mono option
 *
 * TODO: 
 *       more items in file TODO
 */

#include "config.h"
#include <unistd.h>

#define USE_GETOPT_LONG

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <math.h>
#include <fcntl.h>
#include <time.h>
#include <limits.h>
#include <sys/ioctl.h>

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/wait.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include "semshm.h"
#include "wav.h"	/* wav file header structures */
#include "sun.h"	/* sun audio file header structures */
#include "raw.h"	/* raw file handling */
#include "interface.h"  /* low level cdrom interfacing */
#include "cdda2wav.h"
#include "resample.h"
#include "toc.h"
#include "setuid.h"
#include "ringbuff.h"
#include "global.h"
#include "paranoia.h"
#include "mycdrom.h"


static const char * optstring = "D:A:I:O:c:b:r:a:t:i:d:o:n:v:l:E:C:y:S:"
#ifdef MD5_SIGNATURES
                         "M:"
#endif
                         "smw"
#ifdef	ECHO_TO_SOUNDCARD
                         "e"
#endif
                         "NqxPRBVhFGXYZzH";

#ifdef USE_GETOPT_LONG
#include <getopt.h>
struct option options [] = {
	{"device",required_argument,NULL,'D'},
	{"auxdevice",required_argument,NULL,'A'},
	{"interface",required_argument,NULL,'I'},
	{"output-format",required_argument,NULL,'O'},
	{"channels",required_argument,NULL,'c'},
	{"bits-per-sample",required_argument,NULL,'b'},
	{"rate",required_argument,NULL,'r'},
	{"divider",required_argument,NULL,'a'},
	{"track",required_argument,NULL,'t'},
	{"index",required_argument,NULL,'i'},
	{"duration",required_argument,NULL,'d'},
	{"offset",required_argument,NULL,'o'},
	{"sectors-per-request",required_argument,NULL,'n'},
	{"buffers-in-ring",required_argument,NULL,'l'},
	{"output-endianess",required_argument,NULL,'E'},
	{"cdrom-endianess",required_argument,NULL,'C'},
	{"verbose-level",required_argument,NULL,'v'},
#ifdef MD5_SIGNATURES
	{"md5",required_argument,NULL,'M'},
#endif
	{"stereo",no_argument,NULL,'s'},
	{"mono",no_argument,NULL,'m'},
	{"wait",no_argument,NULL,'w'},
	{"find-extremes",no_argument,NULL,'F'},
	{"find-mono",no_argument,NULL,'G'},
#ifdef	ECHO_TO_SOUNDCARD
	{"echo",no_argument,NULL,'e'},
#endif
	{"no-write",no_argument,NULL,'N'},
	{"no-infofile",no_argument,NULL,'H'},
	{"quiet",no_argument,NULL,'q'},
	{"max",no_argument,NULL,'x'},

	{"no-overlap",no_argument,NULL,'P'},
	{"no-paranoia",no_argument,NULL,'X'},
	{"no-scratch-recovery",no_argument,NULL,'Y'},
	{"no-scratch-repair",no_argument,NULL,'Z'},
	{"force-scratch-repair",no_argument,NULL,'z'},
	{"tune-scratch-trigger",required_argument,NULL,'y'},
	{"speed-select",required_argument,NULL,'S'},

	{"dump-rates",no_argument,NULL,'R'},
	{"bulk",no_argument,NULL,'B'},
	{"version",no_argument,NULL,'V'},
	{"help",no_argument,NULL,'h'},
	{NULL,0,NULL,0}
};

#endif /* GETOPT_LONG */


/* global variables */
global_t global;

/* static variables */
static int (*InitAudioFile)( int audioflag, long channels, unsigned long rate,
			    long nBitsPerSample,
			    unsigned long expected_bytes);
static int (*ExitAudioFile)( int audioflag, unsigned long nBytesDone );
static unsigned long (*GetSndHdrSize)( void );

static unsigned long nSamplesDone = 0;

static	int child_pid = 0xff;

static char fname_base[200] = FILENAME;
static long nSamplesToDo;
static long current_track;
static int bulk = 0;

long SamplesNeeded( long amount, long undersampling_val)
{
  long retval = ((undersampling_val * 2 + Halved)*amount)/2;
  if (Halved && (nSamplesToDo & 1))
    retval += 2;
  return retval;
}

#ifdef INFOFILES
int write_info_file(char *fname_baseval, long int track, 
		    unsigned long int SamplesDone, int numbered)
{
  FILE *info_fp;
  char fname[200];
  char datetime[30];
  time_t utc_time;
  struct tm *tmptr;

  /* write info file */
  if (!strcmp(fname_baseval,"standard output")) return 0;
  if (numbered)
    sprintf(fname, "%s_%02ld.inf", fname_baseval, track);
  else
    sprintf(fname, "%s.inf", fname_baseval);
  info_fp = fopen (fname, "w");
  if (!info_fp)
    return -1;

#ifdef MD5_SIGNATURES
  if (global.md5blocksize)
    MD5Final (global.MD5_result, &global.context);
#endif

  utc_time = time(NULL);
  tmptr = localtime(&utc_time);
  if (tmptr) {
    strftime(datetime, sizeof(datetime), "%x %X", tmptr);
  } else {
    strncpy(datetime, "unknown", sizeof(datetime));
  }
  fprintf(info_fp, "#created by cdda2wav "VERSION" %s\n"
	  "Track = %ld\n"
	  "Length = %ld milliseconds\n"
	  "ISRC = %s\n"
#ifdef MD5_SIGNATURES
	  "MD-5(%d) = %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n"
#endif
	  "Pre-emphasis = %s\n"
	  "Channels = %d\n"
	  "Copy permitted = %s\n"
	  "Endianess = %s\n"
	  , datetime
	  , track
	  , (SamplesDone*10 + 220) /441	/* in milliseconds */
	  , g_toc[track-1].ISRC
#ifdef MD5_SIGNATURES
	  , global.md5blocksize
	  , global.MD5_result[0]
	  , global.MD5_result[1]
	  , global.MD5_result[2]
	  , global.MD5_result[3]
	  , global.MD5_result[4]
	  , global.MD5_result[5]
	  , global.MD5_result[6]
	  , global.MD5_result[7]
	  , global.MD5_result[8]
	  , global.MD5_result[9]
	  , global.MD5_result[10]
	  , global.MD5_result[11]
	  , global.MD5_result[12]
	  , global.MD5_result[13]
	  , global.MD5_result[14]
	  , global.MD5_result[15]
#endif
	  , g_toc[track-1].bFlags & 1 ? "yes" : "no"
	  , g_toc[track-1].bFlags & 8 ? 4 : 2
	  , g_toc[track-1].bFlags & 2 ? "yes" : "no"
	  , global.need_big_endian ? "big" : "little"
	  );
  fclose(info_fp);
  return 0;
}
#endif

static void CloseAudio(char *fname_baseval, long int track, int bulkflag, 
			int channels_val, unsigned long nSamples)
{
#ifdef INFOFILES
      /* define sizes in the header of the audio file */
      if (!global.no_infofile)
            write_info_file(fname_baseval,track,nSamples,bulkflag);
#endif

      /* define length */
      ExitAudioFile( global.audio, nSamples*global.OutSampleSize*channels_val );

      close (global.audio);
      global.audio = -1;
}

static unsigned int track = 1;

/* On terminating:
 * define size-related entries in audio file header, update and close file */
static void CloseAll (void)
{
  int chld_return_status;
  int amichild;

  /* terminate child process first */
  amichild = child_pid == 0;

  if (amichild) {
      /* kill parent process */
/*      kill (getppid(), SIGINT);*/
      return;
  } else
      kill(child_pid,SIGINT);


  /* wait for child to terminate */
  wait(&chld_return_status);

  /* do general clean up */

#ifdef	ECHO_TO_SOUNDCARD
  if (global.echo) {
      close(global.soundcard_fd);
  }
#endif
  if (global.audio>=0) {
    if (bulk) {
      /* finish sample file for this track */
      CloseAudio(fname_base, current_track, bulk, global.channels,
  	  global.nSamplesDoneInTrack);
    } else {
      /* finish sample file for this track */
      CloseAudio(fname_base, track, bulk, global.channels,
  	  nSamplesToDo);
    }
  }

  /* tell minimum and maximum amplitudes, if required */
  if (global.findminmax) {
    fprintf(stderr, "minimum amplitude :%d/-32768, maximum amplitude :%d/32767\n", 
	global.minamp, global.maxamp);
  }

  /* tell mono or stereo recording, if required */
  if (global.findmono) {
    fprintf(stderr, "Audio samples are originally %s.\n", global.ismono ? "mono" : "stereo");
  }

  /* free shared memory and semaphores */
  free_semshm();

  /* switch to original mode and close device */
  if (global.cd_fd >= 0) {
      EnableCdda (0);
      close(global.cd_fd);
      global.cd_fd = -1;
  }
}


/* report a fatal error, clean up and exit */
void FatalError (const char *szMessage, ...)
{
  va_list marker;

  va_start(marker, szMessage);

  vfprintf (stderr, szMessage, marker);

  va_end(marker);
  exit (1);
}

/* request a shared memory block */
int shm_request(key_t key, unsigned int size, unsigned char **memptr)
{
  int   ret_val;
  int   shmflag;
  int   SHMEM_ID;
  int    cmd;
  struct shmid_ds buf;

  shmflag = IPC_CREAT | 0600;
  ret_val = shmget(key,(signed)size,shmflag);
  if ( ret_val == -1 )
  {
    perror("shmget");
    return -1;
  }

  SHMEM_ID = ret_val;
  cmd = IPC_STAT;
  ret_val = shmctl(SHMEM_ID,cmd,&buf);
#ifdef IPCTST
  fprintf(stderr, "%d: shmctl STAT= %d, SHM_ID: %d, key %ld cuid %d cgid %d mode %3o size %d\n",
          getpid(),ret_val,SHMEM_ID,
          buf.shm_perm.key,buf.shm_perm.cuid,buf.shm_perm.cgid,
          buf.shm_perm.mode,buf.shm_segsz);
#endif

  *memptr = (unsigned char *) shmat(SHMEM_ID, NULL, 0);
  if (*memptr == (unsigned char *) -1) {
    *memptr = NULL;
    return -1;
  }
  return SHMEM_ID;
}


/* open the audio output file and prepare the header. 
 * the header will be defined on terminating (when the size
 * is known). So hitting the interrupt key leaves an intact
 * file.
 */
static void OpenAudio (char *fname, double rate, long nBitsPerSample, 
		       long channels_val, unsigned long expected_bytes)
{
  if (global.audio == -1) {

    global.audio = open (fname, O_CREAT | O_WRONLY | O_TRUNC
#ifdef SYNCHRONOUS_WRITE
		       | O_SYNC
#endif
		  , 0644);
    if (global.audio == -1) {
      perror("open audio sample file");
      FatalError ("Could not open file %s\n", fname);
    }
  }
  InitAudioFile( global.audio, channels_val, (unsigned long)rate, nBitsPerSample, expected_bytes );

#ifdef MD5_SIGNATURES
  if (global.md5blocksize)
    MD5Init (&global.context);
  global.md5count = global.md5blocksize;
#endif
}


static void set_offset(unsigned char *p, int offset)
{
  *(int *)(p - OFF) = offset;
}


static int get_offset(unsigned char *p)
{
  return *(int *)(p - OFF);
}


static void usage( void )
{
  fprintf( stderr,
"cdda2wav [-c chans] [-s] [-m] [-b bits] [-r rate] [-a divider] [-S speed]\n"
"         [-t track[+endtrack]] [-i index] [-o offset] [-d duration] [-F] [-G]\n"
"         [-x] [-q] [-w] [-v] [-R] [-P] [-X] [-Y] [-Z] [-y threshold] [-z]\n"
"	  [-e] [-n sectors] [-N] [-H] [-l buffers] [-D device] [-I interface]\n"
"         [-O audiotype] [-B] [-E output-endianess] [-A auxdevice] [audiofile]\n"
"Version "VERSION"\n"
"cdda2wav copies parts from audio cd's directly to wav or au files.\n"
"It requires a supported cdrom drive to work.\n"
"options: -D device   : set the cdrom or scsi generic device.\n"
"         -A auxdevice: set the aux device (typically /dev/cdrom).\n"
"         -I interface: specify the interface for cdrom access.\n"
"                     : (generic_scsi or cooked_ioctl).\n"
"         -c channels : set 1 for mono, or 2 for stereo recording.\n"
"         -s          : set to stereo recording.\n"
"         -m          : set to mono recording.\n"
"         -x          : set to maximum quality (stereo/16-bit/44.1 KHz).\n"
"         -b bits     : set bits per sample per channel (8, 12 or 16 bits).\n"
"         -r rate     : set rate in samples per second. -R gives all rates\n"
"         -a divider  : set rate to 44100Hz / divider. -R gives all rates\n"
"         -R          : dump a table with all available sample rates\n"
"         -S speed    : set the cdrom drive to a given speed during reading\n"
"         -P          : switch off overlap sampling; implies -X -Y -Z\n"
"         -X          : switch off extra alignment paranoia; implies -Y -Z\n"
"         -Y          : switch off scratch detection; implies -Z\n"
"         -Z          : switch off scratch reconstruction\n"
"         -y threshold: tune scratch detection filter; low values catch\n"
"                     : more scratches but may also catch legitimate values\n"
"         -z          : disable 'smart' scratch auto-detect and force \n"
"                     : paranoia to treat all data as possibly scratched\n\n"
"         -n sectors  : read sectors per request.\n"
"         -l buffers  : use a ring buffer with 'l' elements.\n"
"         -t track[+end track]\n"
"                     : select start track (and optionally end track).\n"
"         -i index    : select start index.\n"
"         -o offset   : start 'offset' sectors behind start track/index.\n"
"                       one sector equivalents 1/75 second.\n"
"         -O audiotype: set wav or sun or cdr (aka raw) audio format. Default is wav.\n"
"         -C endianess: set little or big input sample endianess.\n"
"         -E endianess: set little or big output sample endianess.\n"
"         -d duration : set recording time in seconds or 0 for whole track.\n"
"         -B          : record each track into a seperate file.\n"
"         -w          : wait for audio signal, then start recording.\n"
"         -F          : find extrem amplitudes in samples.\n"
"         -G          : find if input samples are mono.\n"
#ifdef	ECHO_TO_SOUNDCARD
"         -e          : echo audio data to sound device SOUND_DEV.\n"
#endif
"         -v level    : print informations on current cd (level: 0-63).\n"
"         -N          : no file operation.\n"
"         -H          : no info file generation.\n"
#ifdef MD5_SIGNATURES
"         -M count    : calculate MD-5 checksum for 'count' bytes.\n"
#endif
"         -q          : quiet operation, no screen output.\n"
#ifdef USE_GETOPT_LONG
"         -V          : print version information.\n"
"         -h          : this help screen.\n"  
#endif
"defaults: %s, %d bit, %g Hz, track 1, no offset, one track,\n"
"          type %s '%s', don't wait for signal, not quiet, not verbose,\n"
"          use %s, device %s, aux %s\n"
"parameters: (optional) a file name or - for standard output.\n",
	  CHANNELS-1?"stereo":"mono", BITS_P_S, 44100.0 / UNDERSAMPLING, 
          AUDIOTYPE, FILENAME, DEF_INTERFACE, CD_DEVICE, AUX_DEVICE);
  exit( 1 );
}

static void init_globals(void)
{
  strncpy(global.dev_name, CD_DEVICE, sizeof(global.dev_name));	/* device name */
  strncpy(global.aux_name, AUX_DEVICE, sizeof(global.aux_name));/* auxiliary cdrom device */
  global.audio    = -1;		/* audio file desc */
  global.cd_fd	  = -1;		/* cdrom file desc */
  global.no_file  =  0;		/* flag no_file */
  global.no_infofile  =  0;	/* flag no_infofile */
  global.quiet	  =  0;		/* flag quiet */
  global.verbose  =  3 + 32;	/* verbose level */
  global.sh_bits  =  0;		/* sh_bits: sample bit shift */
  global.Remainder=  0;		/* remainder */
  global.SkippedSamples =  0;		/* skipped samples */
  global.OutSampleSize  =  0;		/* output sample size */
  global.channels = CHANNELS;	/* output sound channels */
  global.nSamplesDoneInTrack = 0; /* written samples in current track */
  global.outputendianess = NONE; /* user specified output endianess */
  global.littleendian = -1; /* user specified output endianess */
  global.findminmax  =  0;	/* flag find extrem amplitudes */
  global.maxamp = INT_MIN;	/* maximum amplitude */
  global.minamp = INT_MAX;	/* minimum amplitude */
  global.speed = DEFAULT_SPEED; /* use default */ 
  global.findmono  =  0;	/* flag find if samples are mono */
  global.ismono  =  1;		/* flag if samples are mono */
}


static unsigned long SectorBurst;
#if (SENTINEL > CD_FRAMESIZE_RAW)
#error block size for overlap check has to be < sector size
#endif

#if      defined _POSIX_PRIORITY_SCHEDULING
#include <sched.h>
#endif

/* and finally: the MAIN program */
int main( long argc, char *argv [] )
{
  long lSector;
  long BeginAtSample;
  long lSector_p1;
  long lSector_p2;
  long sector_offset = 0;
  unsigned long SamplesToWrite; 
  unsigned long endtrack = 1;
  double rectime = DURATION;
  unsigned int cd_index = 1;
  double rate = 44100 / UNDERSAMPLING;
  double int_part;
  int bits = BITS_P_S;
  int just_the_toc = 0;
  char fname[200];
  char int_name[100] = DEF_INTERFACE;
  char audio_type[10] = AUDIOTYPE;
  long i;
  unsigned char *p;
  int c;
  int long_option_index=0;
  int am_i_cdda2wav;

  /* init global variables */
  init_globals();

  /* When being invoked as list_audio_tracks, just dump a list of
     audio tracks. */
  am_i_cdda2wav = strcmp(argv[0],"list_audio_tracks");
  if (!am_i_cdda2wav) global.verbose = SHOW_JUSTAUDIOTRACKS;

  /* Control those set-id privileges... */
  initsecurity();
  dontneedroot();

  /* command options parsing */
#ifdef USE_GETOPT_LONG
  while ( (c = getopt_long (argc,argv,optstring,options,&long_option_index)) 
		!= EOF)
#else
  while ( (c = getopt(argc, argv, optstring)) != EOF )
#endif /* USE_GETOPT_LONG */
	{
    switch (c) {
      case 'D':    /* override device */
        strncpy( global.dev_name, optarg, sizeof(global.dev_name) );
        global.dev_name[sizeof(global.dev_name)-1]=0;
	break;
      case 'A':    /* override device */
        strncpy( global.aux_name, optarg, sizeof(global.aux_name) );
        global.aux_name[sizeof(global.aux_name)-1]=0;
	break;
      case 'I':    /* override interface */
	strncpy( int_name, optarg, sizeof(int_name) );
        int_name[sizeof(int_name)-1]=0;
	break;

      /* the following options are relevant to 'cdda2wav' only */
#ifdef MD5_SIGNATURES
      case 'M':
        if (!am_i_cdda2wav) break;
	global.md5blocksize = atoi( optarg );
        break;
#endif
      case 'O':    /* override audio type */
        if (!am_i_cdda2wav) break;
	strncpy( audio_type, optarg, 4);
        audio_type[sizeof(audio_type)-1]=0;
	break;
      case 'C':    /* override input endianess */
        if (!am_i_cdda2wav) break;
	if (strcasecmp(optarg, "little") == 0) {
	   global.littleendian = 1;
	} else 
	if (strcasecmp(optarg, "big") == 0) {
	   global.littleendian = 0;
	} else {
	   usage();
	}
	break;
      case 'E':    /* override output endianess */
        if (!am_i_cdda2wav) break;
	if (strcasecmp(optarg, "little") == 0) {
	   global.outputendianess = LITTLE;
	} else 
	if (strcasecmp(optarg, "big") == 0) {
	   global.outputendianess = BIG;
	} else {
	   usage();
	}
	break;
      case 'c':    /* override channels */
        if (!am_i_cdda2wav) break;
	global.channels = atoi( optarg );
	break;
      case 'S':    /* override drive speed */
        if (!am_i_cdda2wav) break;
	global.speed = atoi( optarg );
	break;
      case 'l':    /* install a ring buffer with 'buffers' elements */
        if (!am_i_cdda2wav) break;
        buffers = atoi( optarg );
        break;
      case 'b':    /* override bits */
        if (!am_i_cdda2wav) break;
	bits = atoi( optarg );
	break;
      case 'r':    /* override rate */
        if (!am_i_cdda2wav) break;
	rate = atoi( optarg );
	break;
      case 'a':    /* override rate */
        if (!am_i_cdda2wav) break;
	if (strtod( optarg, NULL ) != 0.0)
	    rate = fabs(44100.0 / strtod( optarg, NULL ));
	else {
	    fprintf(stderr, "-a requires a nonzero, positive divider.\n");
	    usage();
	}
	break;
      case 't':    /* override start track */
        if (!am_i_cdda2wav) break;
	{
	char * endptr;
	char * endptr2;
	track = strtoul( optarg, &endptr, 10 );
	endtrack = strtoul( endptr, &endptr2, 10 );
	if (endptr2 == endptr)
		endtrack = track;
	break;
	}
      case 'i':    /* override start index */
        if (!am_i_cdda2wav) break;
	cd_index = atoi( optarg );
	break;
      case 'd':    /* override recording time */
        if (!am_i_cdda2wav) break;
	rectime = strtod( optarg, NULL );
	break;
      case 'o':    /* override offset */
        if (!am_i_cdda2wav) break;
	sector_offset = atol( optarg );
	break;
      case 'n':    /* read sectors per request */
        if (!am_i_cdda2wav) break;
	nsectors = atol( optarg );
	break;
      case 's':    /* stereo */
        if (!am_i_cdda2wav) break;
	global.channels = 2;
	break;
      case 'm':    /* mono */
        if (!am_i_cdda2wav) break;
	global.channels = 1;
	break;
      case 'x':    /* max */
        if (!am_i_cdda2wav) break;
	global.channels = 2; bits = 16; rate = 44100;
	break;
      case 'w':    /* wait for some audio intensity */
        if (!am_i_cdda2wav) break;
	waitforsignal = 1;
	break;
      case 'F':    /* find extreme amplitudes */
        if (!am_i_cdda2wav) break;
	global.findminmax = 1;
	break;
      case 'G':    /* find if mono */
        if (!am_i_cdda2wav) break;
	global.findmono = 1;
	break;
      case 'e':	   /* echo to sound acrd */
        if (!am_i_cdda2wav) break;
#ifdef	ECHO_TO_SOUNDCARD
	global.echo = 1;
#else
	fprintf(stderr, "There is no sound support compiled into %s.\n",argv[0]);
#endif
	break;	
      case 'v':    /* tell us more */
        if (!am_i_cdda2wav) break;
	global.verbose = atoi( optarg );
	break;
      case 'q':    /* be quiet */
        if (!am_i_cdda2wav) break;
	global.quiet = 1;
	global.verbose = 0;
	break;
      case 'N':    /* don't write to file */
        if (!am_i_cdda2wav) break;
	global.no_file = 1;
	break;
      case 'H':    /* don't write to file */
        if (!am_i_cdda2wav) break;
	global.no_infofile = 1;
	break;
      case 'B':    /* bulk transfer */
        if (!am_i_cdda2wav) break;
	bulk = 1;
	if (rectime == 0.0)
	  rectime = 99999.0;
	break;
      case 'P':    /* prevent overlap reading */
        if (!am_i_cdda2wav) break;
	overlap = 0;
	break;
      case 'X':    /* prevent extra paranoia */
        if (!am_i_cdda2wav) break;
 	paranoid = 0;
 	break;
      case 'Y':    /* prevent scratch tolerance */
        if (!am_i_cdda2wav) break;
 	scratch = 0;
 	break;
      case 'Z':    /* prevent scratch repair */
        if (!am_i_cdda2wav) break;
 	repair = 0;
	break;
      case 'z':    /* force scratch repair */
        if (!am_i_cdda2wav) break;
 	scratched_force = -1;
	break;
      case 'y':    /* tune scratch repair */
        if (!am_i_cdda2wav) break;
 	p_cull_mult = atof(optarg);
	break;
      case 'R':    /* list available rates */
        if (!am_i_cdda2wav) break;
	{ int ii;
	  fprintf(stderr, "Cdda2wav version "VERSION": available rates are:\n"
               "Rate   Divider      Rate   Divider      "
               "Rate   Divider      Rate   Divider\n");
	  for (ii = 1; ii <= 44100 / 880 / 2; ii++) {
	    long i2 = ii;
	    fprintf(stderr, "%-7g  %2ld         %-7g  %2ld.5       ",
                    44100.0/i2,i2,44100/(i2+0.5),i2);
	    i2 += 25;
	    fprintf(stderr, "%-7g  %2ld         %-7g  %2ld.5\n",
                    44100.0/i2,i2,44100/(i2+0.5),i2);
	    i2 -= 25;
	  }
	}
	exit(0);
#ifdef USE_GETOPT_LONG
	break;
      case 'V':
        if (!am_i_cdda2wav) break;
	fputs ("Cdda2wav version "VERSION"\n", stderr);
	exit (0);
	break;
      case 'h':
	usage();
	break;
#if 0
      case '?':
	fputs ("use cdda2wav --help to get more information.", stderr);
	exit (1);
	break;
#endif
#endif
      default:
        usage();
    }
  }

  /* check all parameters */
  if (global.verbose < 0 || global.verbose > SHOW_MAX) {
    fprintf(stderr, "Error: incorrect verbose level setting: %d\n",global.verbose);
    usage();
  }
  if (global.verbose == 0) global.quiet = 1;

  if ( global.channels != 1 && global.channels != 2 ) {
    fprintf(stderr, "Error: incorrect channel setting: %d\n",global.channels);
    usage();
  }

  if ( bits != 8 && bits != 12 && bits != 16 ) {
    fprintf(stderr, "Error: incorrect bits_per_sample setting: %d\n",bits);
    usage();
  }

  if ( rate < 827.0 || rate > 44100.0 ) {
    fprintf(stderr, "Error: incorrect sample rate setting: %g\n",rate);
    usage();
  }

  if ( modf( 2*44100.0 / rate, &int_part) >= 0.5 ) {
      int_part += 1.0;
      fprintf( stderr, "Nearest available sample rate is %g Hertz\n",
	      2*44100.0 / int_part);
  }
  Halved = ((int) int_part) & 1;
  rate = 2*44100.0 / int_part;
  undersampling = (int) int_part / 2.0;
  samples_to_do = undersampling;

  if ( rectime < 0 ) {
    fprintf(stderr, "Error: incorrect recording time setting: %f\n",rectime);
    usage();
  }

  if (!strcmp((char *)int_name,"generic_scsi"))
      interface = GENERIC_SCSI;
  else if (!strcmp((char *)int_name,"cooked_ioctl"))
      interface = COOKED_IOCTL;
  else  {
    fprintf(stderr, "Error: incorrect interface setting: %s\n",int_name);
    usage();
  }

#ifdef MD5_SIGNATURES
  if (global.md5blocksize < 0) {
    fprintf(stderr, "Error: incorrect i MD-5 count: %d\n",global.md5blocksize);
    usage();
  }
#endif

  /* check * init audio file */
  if (!strncmp(audio_type,"wav",3)) {
    global.need_big_endian = 0;
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitWav;
    ExitAudioFile = ExitWav;
    GetSndHdrSize = GetWavHdrSize;
  } else if (!strncmp(audio_type, "sun", 3)) {
    /* Enhanced compatibility */
    strcpy(audio_type, "au");

    global.need_big_endian = 1;
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitSun;
    ExitAudioFile = ExitSun;
    GetSndHdrSize = GetSunHdrSize;
  } else if (!strncmp(audio_type, "cdr", 3) || 
             !strncmp(audio_type, "raw", 3)) {
    global.need_big_endian = 1;	/* for cd writer compatibility */
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitRaw;
    ExitAudioFile = ExitRaw;
    GetSndHdrSize = GetRawHdrSize;
  } else {
    fprintf(stderr, "Error: incorrect audio type setting: %3s\n", audio_type);
    usage();
  }
  if (global.no_file) fname_base[0] = '\0';

  if (!bulk) {
    strcat(fname_base, ".");
    strcat(fname_base, audio_type);
  }

  /* all options processed. Now a file name may follow */
  if ( optind < argc ) {
    if (!strcmp(argv[optind],"-")) {
      /* pipe mode */
      bulk = 0;
      global.audio = dup (fileno(stdout));
      strncpy( fname_base, "standard output", sizeof(fname_base) );
    } else {
      /* filename given */
      strncpy( fname_base, argv[optind], sizeof(fname_base)-8 );
    }
    fname_base[sizeof(fname_base)-1]=0;
  }

  /* setup interface and open cdrom device */
  /* request semaphores and shared memory */
  SetupInterface( int_name );

  /* get table of contents */
  tracks = ReadToc( g_toc );
  have_CD_extra = FixupTOC(tracks + 1);

  if ( global.verbose == SHOW_JUSTAUDIOTRACKS ) {
    int z;

    for (z = 0; z < tracks; z++)
	if ((g_toc[z].bFlags & CDROM_DATA_TRACK) == 0)
	  printf("%02d\t%06u\n", g_toc[z].bTrack, g_toc[z].dwStartSector);
    exit(0);
  }

  if ( global.verbose != 0 )
    fputs( "Cdda2wav version "VERSION
#if defined _POSIX_PRIORITY_SCHEDULING || defined ECHO_TO_SOUNDCARD
           " plus"
#endif
#if defined _POSIX_PRIORITY_SCHEDULING
           " real time scheduling"
#endif
#if defined ECHO_TO_SOUNDCARD
           " soundcard support"
#endif
           "\n", stderr );

  if ( global.verbose & (SHOW_TOC | SHOW_STARTPOSITIONS) )
    DisplayToc ();

  if (just_the_toc) exit(0);

  if (nsectors < 1+overlap) {
      fprintf(stderr, "Error: overlap (%d) is >= nsectors (%d)\n",overlap, nsectors);
      exit(-1);
  }
  if ( global.verbose != 0 && overlap == 0)
      fprintf(stderr, "No overlap sampling active\n");

  /* paranoia reigns supreme */
  sync();

  /* try to get some extra kicks */
  needroot(0);
  nice(-20);
  dontneedroot();
  dontneedgroup();

  /* switch cdrom to audio mode */
  EnableCdda (1);

  atexit ( CloseAll );

  if ( !FirstTrack () )
    FatalError ( "This disk has no audio tracks\n" );

  if ( global.verbose & (SHOW_MCN | SHOW_ISRC) )
    Read_MCN_ISRC();

  /* check if start track is in range */
  if ( track < 1 || track > tracks ) {
    fprintf(stderr, "Error: incorrect start track setting: %d\n",track);
    usage();
  }

  /* check if end track is in range */
  if ( endtrack < track || endtrack > tracks ) {
    fprintf(stderr, "Error: incorrect end track setting: %ld\n",endtrack);
    usage();
  }

  if (cd_index != 1)
    sector_offset += ScanIndices( track, cd_index );
  else if (global.verbose & SHOW_INDICES)
    ScanIndices( track, cd_index );

  lSector = GetStartSector ( track );
  lSector_p1 = GetEndSector ( track ) + 1;

  if ( lSector < 0 )
    FatalError ( "track %d not found\n", track );

  lSector += sector_offset;
  /* check against end sector of track */
  if ( lSector >= lSector_p1 ) {
    fputs( "sector offset exceeds track size (ignored)\n", stderr );
    lSector -= sector_offset;
  }

  if (track != endtrack) {
    if (CheckTrackrange( track, endtrack) == 1) {
      lSector_p2 = GetEndSector ( endtrack ) + 1;

      if (lSector_p2 >= 0) {
        rectime = (lSector_p2 - lSector) / 75.0;
      } else {
        fputs( "end track is no valid audio track (ignored)\n", stderr );
      }
    } else {
      fputs( "track range does not consist of audio tracks only (ignored)\n", stderr );
    }
  }
  lSector_p2 = GetLastSectorOnCd( track );
  if ( rectime == 0.0 ) {
    /* set time to track time */
    nSamplesToDo = (lSector_p1 - lSector) * CD_FRAMESAMPLES;
    rectime = (lSector_p1 - lSector) / 75.0;
  } else {
    /* Prepare the maximum recording duration.
     * It is defined as the biggest amount of
     * adjacent audio sectors beginning with the
     * specified track/index/offset. */

    if ( rectime > (lSector_p2 - lSector) / 75.0 ) {
      rectime = (lSector_p2 - lSector) / 75.0;
      lSector_p1 = lSector_p2;
    }

    /* calculate # of samples to read */
    nSamplesToDo = (long)(rectime*44100.0 + 0.5);
  }

  global.OutSampleSize = (1+bits/12);
  if (nSamplesToDo/undersampling == 0L) {
      fprintf( stderr, "time interval too short. Duration > %g secs!\n", 
	       undersampling/44100.0);
      exit(-2);
  }
  SamplesToWrite = nSamplesToDo*2/(int)int_part;

  init_soundcard(rate, bits);

  if (global.speed != 0)
     SelectSpeed(global.speed);

  if ( global.verbose & SHOW_SUMMARY ) {
      fprintf (stderr, "recording %.5f seconds %s with %d bits @ %g Hz"
	      ,rectime ,global.channels == 1 ? "mono":"stereo", bits, rate);
      if (*fname_base) fprintf(stderr, " ->'%s'...", fname_base );
      fputs("\n", stderr);
      if ( !waitforsignal )
	if (!bulk)
	  fprintf(stderr, "samplefile size will be %lu bytes.\n", GetSndHdrSize() +
	  SamplesToWrite*global.OutSampleSize*global.channels  ); 
	else {
	  int tracks_included;
	  tracks_included = GetTrack(
			      (unsigned) (lSector + nSamplesToDo/CD_FRAMESAMPLES -1))
				     - track + 1;
	  fprintf(stderr, "samplefiles size total will be %lu bytes. %d tracks\n", 
		  tracks_included * GetSndHdrSize() +
	  SamplesToWrite*global.OutSampleSize*global.channels, tracks_included  ); 
	}
  }

  current_track = track;

  if ( !global.no_file ) {
    if (bulk)
      sprintf(fname, "%s_%02ld.%s",fname_base,current_track,audio_type);
    else
      strcpy(fname,fname_base);
    OpenAudio( fname, rate, bits, global.channels, 
	      (unsigned)(SamplesToWrite*global.OutSampleSize*global.channels) );
  }

  global.Remainder = (75 % nsectors)+1;

  global.sh_bits = 16 - bits;		/* shift counter */

  i = nSamplesToDo;
  if (Halved && (i&1))
      i += 2;

  BeginAtSample = lSector * CD_FRAMESAMPLES;

  /* Everything is set up. Now fork and let one process read cdda sectors
     and let the other one store them in a wav file */

  /* forking */
  child_pid = fork();

  if (child_pid < 0) {
    perror("fork()");
  }
  /*********************** fork **************************************/
  if (child_pid == 0) {
    /* child READER section */

    unsigned total_retries = 0;

#if      defined _POSIX_PRIORITY_SCHEDULING

    int sched_fifo_min, sched_fifo_max;
    struct sched_param sched_parms;

    sched_fifo_min = sched_get_priority_min(SCHED_FIFO);
    sched_fifo_max = sched_get_priority_max(SCHED_FIFO);
    sched_parms.sched_priority = sched_fifo_max - 1;
    needroot(0);
    if (0 != sched_setscheduler(getpid(), SCHED_FIFO, &sched_parms) )
	perror("cannot set realtime scheduling policy");
    dontneedroot();
#endif

    while (1) {
      unsigned char *newbuf;
      int added_size;
      int offset;
      unsigned int retry_count = 0;

      /* get next buffers */
#ifdef DEBUG_BUFFER
fprintf(stderr, "wait for empty buffer, ");
#endif
      p = get_next_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "got empty\n");
#endif

      /* how many sectors should be read */
      if (overlap && nsectors*CD_FRAMESAMPLES > i) {
	SectorBurst = min(nsectors,
			  overlap + (i + CD_FRAMESAMPLES-1) / CD_FRAMESAMPLES);
      } else {
	SectorBurst = min(nsectors,
			  (i + CD_FRAMESAMPLES-1) / CD_FRAMESAMPLES);
      }
      if (lSector_p2 < lSector+SectorBurst-1)
        SectorBurst = lSector_p2 - lSector + 1;

#if 0
fprintf(stderr,"reading from sec %6ld to %6ld,  %6ld...\n",
       lSector, lSector+SectorBurst-1, i/CD_FRAMESAMPLES );
#endif

#define MAX_READRETRY 1

      {
        do {
          ReadCdRom( p, lSector, SectorBurst );
        } while (NULL ==
                 (newbuf = synchronize( p, SectorBurst*CD_FRAMESAMPLES,
                                       nSamplesToDo-i )) &&
                 ++retry_count < MAX_READRETRY);
      }

      /* how much has been added? */
      if (newbuf) {
        offset = newbuf - p;
      } else {
        offset = overlap * CD_FRAMESIZE_RAW;
      }
      set_offset(p,offset);

#ifdef DEBUG_BUFFER
fprintf(stderr, "defining buffer, ");
#endif
      define_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "defined\n");
#endif

      added_size = SectorBurst * CD_FRAMESAMPLES - offset/4;

#if 0
fprintf(stderr, "added_size (secs =%d), (samples=%d)\n",added_size / CD_FRAMESAMPLES,added_size);
#endif
      if (retry_count) {
        fprintf(stderr,"retries while reading from sec %6ld to %6ld,  %6ld...\n",
                lSector, lSector+SectorBurst-1, i/CD_FRAMESAMPLES );
        total_retries += retry_count;
      }

      lSector += SectorBurst - overlap;

      i -= added_size;

    }
    if (total_retries) {
      fprintf(stderr,"%u retries while reading\n",total_retries);
    }
    exit(0);
  } else {
    /******************* parent WRITER  section *********************/
    int child_state;
    
#define left_in_this_track(pos, track) (max(0,g_toc[track].dwStartSector*CD_FRAMESAMPLES - (pos)))

#if      defined _POSIX_PRIORITY_SCHEDULING

    int sched_fifo_min, sched_fifo_max;
    struct sched_param sched_parms;

    sched_fifo_min = sched_get_priority_min(SCHED_FIFO);
    sched_fifo_max = sched_get_priority_max(SCHED_FIFO);
    sched_parms.sched_priority = sched_fifo_max - 2;
    needroot(0);
    if (0 != sched_setscheduler(getpid(), SCHED_FIFO, &sched_parms) )
	perror("cannot set realtime scheduling policy");
    dontneedroot();
#endif

    /* don't need these anymore.  Good security policy says we get rid
       of them ASAP */
    neverneedroot();
    neverneedgroup();

    while (nSamplesDone < nSamplesToDo) {
      int current_offset;

      /* get oldest buffers */
#ifdef DEBUG_BUFFER
fprintf(stderr, "wait defined buffer, ");
#endif
      p = get_oldest_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "got defined\n");
#endif

      current_offset = get_offset(p);
      p += current_offset;

      /* how many bytes are available */
      SectorBurst = nsectors*CD_FRAMESAMPLES - current_offset/4;
      SectorBurst = min((nSamplesToDo-nSamplesDone),SectorBurst);

      /* when track end is reached, close current file and start a new one */

#if 0
fprintf(stderr, "before loop: track %ld limit at %d, we have up to %ld\n",
	current_track,
	g_toc[current_track+1 -1].dwStartSector*CD_FRAMESAMPLES, 
	SectorBurst + nSamplesDone + BeginAtSample); 
#endif

      while ((nSamplesDone < nSamplesToDo) && (SectorBurst != 0)) {
	long unsigned int how_much = SectorBurst;
	long unsigned int left_in_track = left_in_this_track(BeginAtSample+nSamplesDone,
							     current_track);

	if (bulk)
	  how_much = min(how_much, left_in_track);

#if 0
fprintf(stderr, "in loop: left %lu, shall write %lu samples\n", left_in_track, how_much);
#endif


#ifdef MD5_SIGNATURES
	if (global.md5count) {
	  MD5Update (&global.context, p, min(global.md5count,how_much));
	  global.md5count -= min(global.md5count,how_much);
	}
#endif
	if ( SaveBuffer ( p,
			 how_much,
			 &nSamplesDone) ) {
	  /* kill child */
	  kill(child_pid, SIGINT);
	  exit(2);
	}
#if 0
fprintf(stderr, "1wrote %08ld from %08ld samples\n", nSamplesDone, 
	SectorBurst + BeginAtSample); 
#endif
        global.nSamplesDoneInTrack += how_much;
	SamplesToWrite -= nSamplesDone;

	/* move residual samples upto buffer start */
	if (how_much < SectorBurst) 
	  memmove(p,
		  p + how_much*4,
		  (SectorBurst - how_much) * 4);
	
	if (left_in_track < SectorBurst) {

	  if (bulk) {
	    /* finish sample file for this track */
	    CloseAudio(fname_base, current_track, bulk, global.channels,
	  	  global.nSamplesDoneInTrack);
          } else if (SamplesToWrite == 0) {
	    /* finish sample file for this track */
	    CloseAudio(fname_base, track, bulk, global.channels,
	  	  nSamplesToDo);
	  }

	  if (global.verbose)
	    fprintf( stderr, "track %2ld successfully recorded\n", current_track);

          global.nSamplesDoneInTrack = 0;
	  if ( bulk && SamplesToWrite > 0 ) {
	    if ( !global.no_file ) {

	      /* build next filename */
	      sprintf(fname, "%s_%02ld.%s",fname_base,current_track+1,audio_type);
	      OpenAudio( fname, rate, bits, global.channels,
			(g_toc[current_track].dwStartSector -
			 g_toc[current_track-1].dwStartSector)*CD_FRAMESIZE_RAW);
	    }
	  }
	  current_track++;
	  jitterShift = 0;
	}
	SectorBurst -= how_much;

      }  /* end while */

#ifdef DEBUG_BUFFER
fprintf(stderr, "drop buffer, ");
#endif
      drop_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "dropped\n");
#endif
    } /* end while */
    if (global.verbose)
      fprintf( stderr, "track %2ld successfully recorded\n", current_track);

    /* kill child */
    kill(child_pid, SIGINT);
    wait(&child_state);
    exit(0);
  }
  if (!global.quiet) fputs( "\n", stderr );

  return 0;
}
