/*
  ckdisk.c  --  short description
  Copyright (C) 2005 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: ckdisk.c,v 1.3 2005/07/22 00:45:59 solyga Exp $
 */

#include	"ckdisk.h"


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Write pattern to / read pattern from device.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] device\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -w\t write to device; default is read/verify\n" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level on %s (max %d)\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr", VERBOSE_LEVEL_MAX );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -s SIZE\t set blocksize; default is %d (0x%x) bytes\n", BLK_SZE, BLK_SZE );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  BLK_SZE :  %4d (0x%x)\n", BLK_SZE, BLK_SZE );
  fprintf( VERSION_CHANNEL, "  BLKS    :  %4d (0x%x)\n", BLKS, BLKS );
  return( 0 );
}


ssize_t
my_read( int fd, void* buf, size_t count ) {
/*
 * (attempts to) read exactly count bytes from fd into buf
 * returns number of bytes read or -1 on error
 * retval < count indicates EOF (or EAGAIN when non-blocking)
 * started 1998-01-01
 */
  unsigned char* p= buf;
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= count;
  do {
    if( (nbr=read(fd,p+tnbr,rem)) == -1 ) {
      if( errno == EAGAIN ) return( tnbr );
      else                  return( -1 );
    }
    tnbr+= nbr;
    rem-= nbr;
  } while( nbr>0 && rem>0 );
  return( tnbr );
}


ssize_t
my_write( int fd, void* buf, size_t count ) {
/*
 * (attempts to) write exactly count bytes from buf to fd
 * returns number of bytes written or -1 on error
 * retval < count indicates nothing special (see write(2))
 * started 1998-01-01
 */
  unsigned char* p= buf;
  ssize_t nbw;
  ssize_t tnbw= 0;
  size_t rem= count;
  do {
    if( (nbw= write( fd, p+tnbw, rem )) == -1 ) return( -1 );
    tnbw+= nbw;
    rem-= nbw;
  } while( nbw>0 && rem>0 );
  return( tnbw );
}


char*
str_dsc( char* str1, char* str2) {
/*
returns pointer to first char of str1 not contained in str2
started 1997-02-09
*/
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


long
getoptarg( char opt, char* arg, long min, long max, char* pn ) {
  char* p;
  long val= strtol( arg, &p, 0 );
  if( p == arg ) { /* nothing read */
    fprintf( ERROR_CHANNEL,
             "%s: Argument of option %c (`%s') must be a number.\n",
             pn, opt, arg );
    exit( RETVAL_ERROR );
  }
  if( val<min || val>max ) {
    fprintf( ERROR_CHANNEL,
             "%s: Argument value %ld of option %c must be in [%ld,%ld].\n",
             pn, val, opt, min, max );
    exit( RETVAL_ERROR );
  }
  if( *str_dsc(p," \t") != '\0' ) {
    fprintf( ERROR_CHANNEL,
             "%s: Argument `%s' of option %c must be an integer.\n",
             pn, arg, opt );
    exit( RETVAL_ERROR );
  }
  return( val ) ;
}


char*
basename( char* name ) {
/*
 * strip directory from name
 * returns pointer to stripped name
 * hint: basename("/usr/bin/") == ""
 *   basename(1) would return "bin" !!
 */
  char* p= name;
  while( *p != '\0' ) p++;
  while( p > name ) {
    if( *(p-1) == '/' ) break;
    else p--;
  }
  return( p );
}


int sigrcvd = 0;


void
handler( int bla ) {
  sigrcvd = 1;
}


int
main( int argc, char** argv ) {
/*
 * main() ckdisk
 * started      : Wed Jul 20 00:40:29 CEST 2005 @beast
 */
  char* fpn = *argv;
  char* pn = basename( fpn );		/* prg_name is Makefile constant */
  int retval = RETVAL_OK;
  int c;				/* getopt() */
  int verbose = 0;			/* verbosity level */
  int do_wr = 0;			/* write? (default is read) */
  int blk_sze = BLK_SZE;
  int blks = BLKS;
  char* fn = NULL;
  int fd;
  int blk;				/* block counter */
  unsigned char* rbuf = NULL;		/* read buffer */
  unsigned char* wbuf = NULL;		/* write buffer */
  ssize_t nbr;				/* number of bytes read */
  ssize_t nbw;				/* number of bytes written */
  ssize_t tnbr;				/* total number of bytes read */
  ssize_t tnbw;				/* total number of bytes written */
  int i;


/* process options */
  *argv= pn;				/* give getop() the cut name */
  while( (c=getopt(argc,argv,"hs:vwV")) != EOF ) {
    switch( c ) {
      case 'h': /* display help and exit sucessfully */
        display_help( pn );
        retval = RETVAL_OK; goto DIE_NOW;
      case 's': /* change block size */
        blk_sze = (int) getoptarg( c, optarg, 1, INT_MAX-1, pn );
        break;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'w': /* write to device (default is read/verify) */
        do_wr = 1;
        break;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        display_version( pn );
        retval = RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, pn );
        retval = RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 pn, MY_EMAIL_ADDRESS );
        retval = RETVAL_BUG; goto DIE_NOW;
    }
  }

  if( optind >= argc ) {
    display_help( pn );
    retval = RETVAL_ERROR; goto DIE_NOW;
  }
  fn = argv[optind];

  if( (wbuf=malloc(blk_sze)) == NULL ) {
    fprintf( ERROR_CHANNEL,
             "%s: Cannot allocate %d bytes for write buffer. %s.\n",
             pn, blk_sze, strerror(errno) );
    retval = RETVAL_MALLOC; goto DIE_NOW;
  }
  for( i=0; i<blk_sze; i++ ) {
    wbuf[i] = (unsigned char) i;
  }

  signal( SIGINT, handler );

/* do the work */

  switch( do_wr ) {
    case 0: /* read from device and verify */
      if( verbose ) {
        fprintf( VERBOSE_CHANNEL,
                 "%s: Reading from '%s' blocks of %d (0x%x) bytes.\n",
                 pn, fn, blk_sze, blk_sze );
      }
      if( (rbuf=malloc(blk_sze)) == NULL ) {
        fprintf( ERROR_CHANNEL,
                 "%s: Cannot allocate %d bytes for read buffer. %s.\n",
                 pn, blk_sze, strerror(errno) );
        retval = RETVAL_MALLOC; goto DIE_NOW;
      }
      if( (fd=open(fn,OPEN_RD_FLAGS)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot open '%s' read-only. %s.\n",
                 pn, fn, strerror(errno) );
        retval = RETVAL_ERROR; goto DIE_NOW;
      }
      tnbr = 0;
      for( blk=0;;blk++ ) {
        if( sigrcvd ) {
          retval = RETVAL_SIG;
          break;
        }
        if( (nbr=my_read(fd,rbuf,blk_sze)) == -1 ) {
          if( errno == ENOSPC ) break;
          fprintf( ERROR_CHANNEL,
                   "%s: Read error at address %d (0x%x, block %d). %s.\n",
                   pn, blk*blk_sze, blk*blk_sze, blk, strerror(errno) );
          retval = RETVAL_ERROR;
          break;
        }
        tnbr += nbr;
        if( nbr == 0 ) break;
        for( i=0; i<nbr; i++ ) {
          if( rbuf[i] != wbuf[i] ) {
            if( verbose && blk>0 ) fprintf( ERROR_CHANNEL, "\n" );
            fprintf( ERROR_CHANNEL,
                     "%s: At offset %d (0x%x) of `%s': read 0x%02x, expected 0x%02x.\n",
                     pn, blk*blk_sze+i, blk*blk_sze+i, fn, rbuf[i], wbuf[i] );
            retval = RETVAL_DIFF;
            break;
          }
        }
        if( retval == RETVAL_DIFF ) break;
        if( verbose && ( blk%blks == 0 ) ) {
          fprintf( VERBOSE_CHANNEL, "." );
          fflush( VERBOSE_CHANNEL );
        }
      }
      if( verbose && retval!=RETVAL_DIFF ) fprintf( VERBOSE_CHANNEL, "\n" );
      if( close(fd) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot close '%s'. %s.\n",
                 pn, fn, strerror(errno) );
        retval = RETVAL_ERROR; goto DIE_NOW;
      }
      if( verbose ) {
        fprintf( VERBOSE_CHANNEL, "%s: %d (0x%x) bytes read.\n",
                 pn, tnbr, tnbr );
      }
      break;
    case 1: /* write to device */
      if( verbose ) {
        fprintf( VERBOSE_CHANNEL,
                 "%s: Writing to '%s' blocks of %d (0x%x) bytes.\n",
                 pn, fn, blk_sze, blk_sze );
      }
      if( (fd=open(fn,OPEN_WR_FLAGS,PERMISSIONS)) == -1 ) {
        fprintf( ERROR_CHANNEL,
                 "%s: Cannot open '%s' write-only. %s.\n",
                 pn, fn, strerror(errno) );
        retval = RETVAL_ERROR; goto DIE_NOW;
      }
      tnbw = 0;
      for( blk=0;;blk++ ) {
        if( sigrcvd ) {
          retval = RETVAL_SIG;
          break;
        }
        if( (nbw=my_write(fd,wbuf,blk_sze)) == -1 ) {
          if( errno == ENOSPC ) break;
          fprintf( ERROR_CHANNEL,
                   "%s: Write error at address %d (0x%x, block %d). %s.\n",
                   pn, blk*blk_sze, blk*blk_sze, blk, strerror(errno) );
          retval = RETVAL_ERROR;
          break;
        }
        tnbw += nbw;
        if( verbose && ( blk%blks == 0 ) ) {
          fprintf( VERBOSE_CHANNEL, "." );
          fflush( VERBOSE_CHANNEL );
        }
      }
      if( verbose ) fprintf( VERBOSE_CHANNEL, "\n" );
      if( close(fd) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot close '%s'. %s.\n",
                 pn, fn, strerror(errno) );
        retval = RETVAL_ERROR; goto DIE_NOW;
      }
      if( verbose ) {
        fprintf( VERBOSE_CHANNEL, "%s: %d (0x%x) bytes written.\n",
                 pn, tnbw, tnbw );
      }
      break;
    default: /* error */
        fprintf( ERROR_CHANNEL, "%s: Mode bug! E-mail me at %s.\n",
                 pn, MY_EMAIL_ADDRESS );
        retval = RETVAL_BUG; goto DIE_NOW;
  }

DIE_NOW:
  if( wbuf != NULL ) free( wbuf );
  if( rbuf != NULL ) free( rbuf );
  close( fd );
  exit( retval );
}
