/*
 * playcdr.c: Modified recplay.c 
 * by 23.10.95 T.Niederreiter
 * 
 * Plays now 16 bit, 44.1kHz, Stereo - raw files in big-endian-format
 * (known as CDR-Files).
 * New switch -x to play little endian raw files
 * 
 * A simple recording and playback program for the Linux Sound Driver 2.0
 * or later.
 * 
 * Copyright by Hannu Savolainen 1993
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */
#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <stdlib.h>
#ifndef __STDC__
#include <getopt.h>
#endif /* __STDC__ */
#include <fcntl.h>
#ifdef __STDC__
#include <string.h>
#else /* __STDC__ */
#include <strings.h>
#endif /* __STDC__ */
#include <sys/soundcard.h>

#define DEFAULT_DSP_SPEED 44100
#define FRAMESIZE 2352       /* Framesize in CDR-Format */

#define AUDIO "/dev/dsp"

int timelimit = 0, dsp_speed = DEFAULT_DSP_SPEED, dsp_stereo = 1;
int samplesize = 16;
int quiet_mode = 0;
int swap = 1;
int audio, abuf_size;
int startframe = 0;
char *audiobuf, c;

char audio_name[20] = AUDIO;

void playcdr (char *name);
extern void describe_error(void);

int
main (int argc, char *argv[])
{
  char *command;
  int tmp;

  command = argv[0];

  while ((c = getopt (argc, argv, "qs:St:b:d:xf:")) != EOF)
    switch (c)
      {
      case 'S':
	dsp_stereo = 1;
	break;
      case 'q':
	quiet_mode = 1;
	break;
      case 's':
	dsp_speed = atoi (optarg);
	if (dsp_speed < 300)
	  dsp_speed *= 1000;
	break;
      case 't':
	timelimit = atoi (optarg);
	break;
      case 'b':
	samplesize = atoi (optarg);
	break;

      case 'd':
        strncpy(audio_name, optarg, 20);
        break;
    
      case 'x':
	swap = 0;
	break;

      case 'f':
        startframe = atoi (optarg);
        break;

      default:
	fprintf (stderr, 
	 "Usage: %s [-qSx] [-t secs] [-s Hz] [-b 8|12|16] [-d device] [-f startframe] [filename]\n", command);
	exit (-1);
      }


  audio = open (audio_name, O_WRONLY, 0);
  if (audio == -1)
    {
      perror (audio_name);
      describe_error();
      exit (-1);
    }

  tmp = samplesize;
  ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &samplesize);
  if (tmp != samplesize)
  {
  	fprintf(stderr, "Unable to set the sample size\n");
  	exit(-1);
  }

  if (ioctl (audio, SNDCTL_DSP_STEREO, &dsp_stereo)==-1)
  {
      fprintf (stderr, "%s: Unable to set mono/stereo\n", command);
      perror (audio_name);
      exit (-1);
  }

  if (ioctl (audio, SNDCTL_DSP_SPEED, &dsp_speed) == -1)
    {
      fprintf (stderr, "%s: Unable to set audio speed\n", command);
      perror (audio_name);
      exit (-1);
    }
  if (!quiet_mode)
    {
      fprintf (stderr, "Speed %d Hz ", dsp_speed);
      if (dsp_stereo)
	fprintf (stderr, "(stereo)\n");
      else
	fprintf (stderr, "(mono)\n");
      if (samplesize != 8)
        fprintf(stderr, "%d bits per sample\n", samplesize);
    }

  ioctl (audio, SNDCTL_DSP_GETBLKSIZE, &abuf_size);
  if (abuf_size < 1024 || abuf_size > (2*65536))
    {
      if (abuf_size == -1)
	perror (audio_name);
      else
	fprintf (stderr, "Invalid audio buffers size %d\n", abuf_size);
      exit (-1);
    }

  if ((audiobuf = malloc (abuf_size)) == NULL)
    {
      fprintf (stderr, "Unable to allocate input/output buffer\n");
      exit (-1);
    }

  if (optind > argc - 1)
    playcdr (NULL);
  else
    while (optind <= argc - 1)
      {
	playcdr (argv[optind++]);
      }

  close (audio);
  return 0;
}

void
playcdr (char *name)
{
  int fd, l, ii;

  int count, c;
  char tmpswap;

  if (!timelimit)
    count = 0x7fffffff;
  else
    {
      count = timelimit * dsp_speed;
      if (dsp_stereo)
	count *= 2;
      if (samplesize != 8)
        count *= 2;
    }

      if (!name)
	{
	  fd = 0;
	  name = "stdin";
	}
      else
	{
	  if ((fd = open (name, O_RDONLY, 0)) == -1)
	    {
	      perror (name);
	      exit (-1);
	    }
	}

      if (startframe != 0) 
      {
        if (lseek(fd,startframe*FRAMESIZE,SEEK_SET) == -1)
        {
          perror(name);
          exit (-1);
        }
      }

      while (count)
	{
	  c = count;

	  if (c > abuf_size)
	    c = abuf_size;

	  if ((l = read (fd, audiobuf, c))
	      > 0)
	    {

/* T.Niederreiter: Swap byte-order */

	      if (swap)
	        {
		for (ii=0; ii<l; ii+=2) {
                 tmpswap=audiobuf[ii];
		 audiobuf[ii]=audiobuf[ii+1];
		 audiobuf[ii+1]=tmpswap;
                 }
	        }

	      if (write (audio, audiobuf, l) != l)
		{
		  perror (audio_name);
		  exit (-1);
		}
	      count -= l;
	    }
	  else
	    {
	      if (l == -1)
		{
		  perror (name);
		  exit (-1);
		}
	      count = 0;	/* Stop */
	    }

	}			/* while (count) */
      if (fd != 0)
	close (fd);
}


void describe_error(void)
{
	switch (errno)
	{
	case ENXIO:
	case ENODEV:
		fprintf(stderr,	"The device file was found in /dev but\n"
				"there is no driver for it in the kernel.\n"
				"\n"
				"There is several possible reasons:\n"
				"- The sound driver is not present\n"
				"  in the kernel you are currently running.\n"
				"- The driver is in the kernel but there is\n"
				"  no soundcard on your system.\n"
				"- Configuration of the sound driver doesn't\n"
				"  match the hardware.\n"
				"\n"
				"Try cat /dev/sndstat for further info.\n");
		break;

	case ENOSPC:
		fprintf(stderr,	"The soundcard driver was not installed\n"
				"properly. The sound_mem_init() routine\n"
				"was not called during the boot procedure.\n");
		break;

	case ENOENT:
		fprintf(stderr,	"The device file is missing from /dev.\n"
				"Run the script at the end of file\n"
				"sound/linux/Readme (distributed with\n"
				"the sound driver.\n");
		break;

	default: ;
	}
}
