/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char rcsid[] = "$Id: dao.cc,v 1.3 1998/05/22 18:30:25 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/shm.h>

#include "dao.h"
#include "util.h"

// buffer size in blocks
int BUFFER_SIZE = 250;

struct Message {
  long type;
  char *buffer;
  long bufLen;
};

static MSGLEN = sizeof(Message) - sizeof(long);

static int TERMINATE = 0;

RETSIGTYPE terminationRequest(int sig)
{
  TERMINATE = 1;
  signal(sig, terminationRequest);
}

static void writeSlave(CdrDriver *cdr, int testMode, long total, int msgId)
{
  long ret;
  long cnt = 0;
  long blkCount = 0;
  long len = 0;
  char *buf = NULL;
  Message msg;

  while (1) {

    //fprintf(stderr, "Slave: waiting for master.\n");

    do {
      if ((ret = msgrcv(msgId, (msgbuf *)&msg, MSGLEN, 1, 0)) < 0 &&
	  errno != EINTR) {
	fprintf(stderr, "Slave: msgrcv failed: %s\n", strerror(errno));
	exit(1);
      }
    } while (ret < 0 && errno == EINTR);

    len = msg.bufLen;
    buf = msg.buffer;

    if (len == 0) {
      // termination request
      fprintf(stdout, "\n");
      if (testMode) {
	fprintf(stdout, "Read %ld blocks.\n", blkCount);
      }
      else {
	fprintf(stdout, "Wrote %ld blocks.\n", blkCount);
      }
      exit(0);
    }

    cnt += len * AUDIO_BLOCK_LEN;
    blkCount += len;

    //fprintf(stderr, "Slave: writing buffer %p (%ld).\n", buf, len);

    msg.type = 2;
    msg.buffer = NULL;

    if (!testMode) {

      if (cdr->writeData(buf, len) != 0) {
	fprintf(stderr, "Slave: write of audio data failed.\n");

	cdr->flushCache();

	msg.bufLen = 1;	// notify master about error
      }
      else {
	fprintf(stdout, "Wrote %ld of %ld MB.\r", cnt >> 20, total >> 20);
	fflush(stdout);

	msg.bufLen = 0; // indicates OK
      }
    }
    else {
      fprintf(stdout, "Read %ld of %ld MB.\r", cnt >> 20, total >> 20);
      fflush(stdout);

      msg.bufLen = 0; // indicates OK
    }

    do {
      if ((ret = msgsnd(msgId, (msgbuf*)&msg, MSGLEN, 0)) < 0 && 
	  errno != EINTR) {
	fprintf(stderr, "Slave: msgsnd failed: %s\n", strerror(errno));
	exit(3);
      }
    } while (ret < 0 && errno == EINTR);
  }
}

int writeDiscAtOnce(const Toc *toc, CdrDriver *cdr, int swap, int testMode)
{
  char *buf = NULL;
  int bufSize = BUFFER_SIZE * AUDIO_BLOCK_LEN;
  long length = toc->length().lba();
  long total = length * AUDIO_BLOCK_LEN;
  long n;
  int pid = 0;
  int status;
  int err = 0;
  int msgId = -1;
  int shmId1 = -1;
  int shmId2 = -1;
  char *sharedMem1 = NULL;
  char *sharedMem2 = NULL;
  char *sbuf1 = NULL;
  char *sbuf2 = NULL;
  Message msg;

  //signal(SIGINT, terminationRequest);
  signal(SIGINT, SIG_IGN);


  if ((msgId = msgget(IPC_PRIVATE, 0600)) < 0) {
    fprintf(stderr, "Cannot create message queue: %s\n", strerror(errno));
    err = 1; goto fail;
  }

  if ((shmId1 = shmget(IPC_PRIVATE, bufSize, 0600)) < 0 ||
      (shmId2 = shmget(IPC_PRIVATE, bufSize, 0600)) < 0) {
    fprintf(stderr, "Cannot create shared memory segments: %s\n",
	    strerror(errno));
    err = 1; goto fail;
  }
  if ((sharedMem1 = shmat(shmId1, 0, 0)) <= 0 ||
      (sharedMem2 = shmat(shmId2, 0, 0)) <= 0) {
    fprintf(stderr, "Cannot get shared memory: %s\n", strerror(errno));
    err = 1; goto fail;
  }
  sbuf1 = sharedMem1;
  sbuf2 = sharedMem2;
  buf = sbuf1;

  if (toc->openData() != 0) {
    fprintf(stderr, "Cannot open audio data.\n");
    err = 1; goto fail;
  }

  n = (length > BUFFER_SIZE ? BUFFER_SIZE : length);

  //fprintf(stderr, "Master: reading buffer %p.\n", buf);
  if (toc->readData(buf, n) != n) {
    fprintf(stderr, "Read of audio data failed.\n");
    err = 1; goto fail;
  }

  if (swap) {
    swapSamples((Sample *)buf, n * SAMPLES_PER_BLOCK);
  }

  if (!testMode) {
    if (cdr->initDao() != 0) {
      err = 1; goto fail;
    }
  }

  if ((pid = fork()) == 0) {
    // we are the new writing process
    writeSlave(cdr, testMode, total, msgId);
  }
  else if (pid < 0) {
    fprintf(stderr, "fork failed: %s\n", strerror(errno));
    err = 1; goto fail;
  }

  if (TERMINATE) {
    err = 1; goto fail;
  }

  if (!testMode) {
    if (cdr->startDao() != 0) {
      err = 1; goto fail;
    }
  }

  if (!testMode) {
    fprintf(stdout, "Writing audio data...\n");
  }

  do {
    // notify slave that it can write buffer 'bufSel'
    msg.type = 1;
    msg.bufLen = n;
    msg.buffer = buf;
    do {
      if ((status = msgsnd(msgId, (msgbuf*)&msg, MSGLEN, 0) < 0) &&
	  errno != EINTR) {
	fprintf(stderr, "msgsnd failed: %s\n", strerror(errno));
	err = 1; goto fail;
      }
    } while (status < 0 && errno == EINTR);

    length -= n;
    n = (length > BUFFER_SIZE ? BUFFER_SIZE : length);

    // switch to other buffer
    buf = (buf == sbuf1 ? sbuf2 : sbuf1);

    //fprintf(stderr, "Master: reading buffer %p.\n", buf);
    if (toc->readData(buf, n) != n) {
      fprintf(stderr, "Read of audio data failed.\n");
      err = 1; goto fail;
    }

    if (swap) {
      swapSamples((Sample *)buf, n * SAMPLES_PER_BLOCK);
    }

    // wait for slave to finish writing of previous buffer
    //fprintf(stderr, "Master: waiting for slave.\n");
    do {
      if ((status = msgrcv(msgId, (msgbuf*)&msg, MSGLEN, 2, 0)) < 0 &&
	  errno != EINTR) {
	fprintf(stderr, "msgrcv failed: %s\n", strerror(errno));
	err = 1; goto fail;
      }
    } while (status < 0 && errno == EINTR);

    if (msg.bufLen != 0) {
      fprintf(stderr, "Writing failed - buffer under run?\n");
      err = 1;
    }

  } while (length > 0 && err == 0 && !TERMINATE);

  // send finish message
  msg.type = 1;
  msg.bufLen = 0;
  msg.buffer = NULL;

  do {
    if ((status = msgsnd(msgId, (msgbuf*)&msg, MSGLEN, 0)) < 0) {
      if (errno != EINTR) {
	fprintf(stderr, "msgsnd failed: %s\n", strerror(errno));
	err = 1; goto fail;
      }
    }
  } while (status < 0 && errno == EINTR);

  // wait for writing process to finish
  if (wait(&status) <= 0) {
    fprintf(stderr, "wait failed: %s\n", strerror(errno));
    err = 1; goto fail;
  }
  pid = 0;

  if (WIFEXITED(status)) {
    if (WEXITSTATUS(status) != 0) {
      err = 1; goto fail;
    }
  }
  else {
    fprintf(stderr, "Child process terminated abnormally.\n");
    err = 1; goto fail;
  }

  if (!testMode && err == 0 && !TERMINATE) {
    if (cdr->finishDao() != 0) {
      err = 1; goto fail;
    }
  }

fail:
  toc->closeData();

  //  if (buf != NULL) {
  //    free(buf);
  //  }

  if (pid != 0) {
    kill(pid, SIGKILL);
  }

  if (msgId >= 0) {
    if (msgctl(msgId, IPC_RMID, NULL) != 0) {
      fprintf(stderr, "Cannot remove message queue: %s\n", strerror(errno));
    }
  }

  if (shmId1 >= 0) {
    if (sharedMem1 != NULL) {
      if (shmdt(sharedMem1) != 0) {
	fprintf(stderr, "shmdt: %s\n", strerror(errno));
      }
    }
    if (shmctl(shmId1, IPC_RMID, NULL) != 0) {
      fprintf(stderr, "Cannot remove shared memory: %s\n", strerror(errno));
    }
  }

  if (shmId2 >= 0) {
    if (sharedMem2 != NULL) {
      if (shmdt(sharedMem2) != 0) {
	fprintf(stderr, "shmdt: %s\n", strerror(errno));
      }
    }
    if (shmctl(shmId2, IPC_RMID, NULL) != 0) {
      fprintf(stderr, "Cannot remove shared memory: %s\n", strerror(errno));
    }
  }

  signal(SIGINT, SIG_DFL);


  return err;
}

