/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char rcsid[] = "$Id: PlextorReader.cc,v 1.1 1998/05/22 14:00:08 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>

#include "PlextorReader.h"

#include "Toc.h"

PlextorReader::PlextorReader(ScsiIf *scsiIf, Toc *toc) : CdrDriver(scsiIf, toc)
{
  driverName_ = "Plextor CD-ROM Reader - Version 1.0";
  
  speed_ = 0;
  simulate_ = 0;
}

// static constructor
CdrDriver *PlextorReader::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new PlextorReader(scsiIf, toc);
}

// sets speed
// return: 0: OK
//         1: illegal speed
int PlextorReader::speed(int /*s*/)
{
  return 0;
}

int PlextorReader::initDao()
{
  fprintf(stderr, "Writing is not supported by this driver.\n");
  return 1;
}

int PlextorReader::startDao()
{
  return 1;
}

int PlextorReader::finishDao()
{
  return 1;
}

Toc *PlextorReader::readDiscToc()
{
  Toc *toc = NULL;
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 100 * 8 + 4;
  unsigned char data[100 * 8 + 4];
  unsigned char *p = NULL;
  int nofTracks = 0;
  long trackStart[100];
  int i;
  long discStart = 0;

  // read disc toc
  memset(cmd, 0, 10);
  cmd[0] = 0x43; // READ TOC
  cmd[6] = 0; // return info for all tracks
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;
  cmd[9] = 0; // return tack information

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read disc toc.\n");
    return NULL;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read disc toc.\n");
      return NULL;
    }
  }
  
  
  nofTracks = data[3] - data[2] + 1;
  if (nofTracks > 99) {
    fprintf(stderr, "Got illegal toc data.\n");
    return NULL;
  }


  for (i = 0, p = data + 4; i <= nofTracks; i++, p += 8) {
    if (i < nofTracks && (p[1] & 0x0f) != 0x0f && (p[1] & 0x04) != 0) {
      fprintf(stderr, "Disc contains data track - aborting.\n");
      return NULL;
    }
    trackStart[i] = p[4] << 24;
    trackStart[i] |= p[5] << 16;
    trackStart[i] |= p[6] << 8;
    trackStart[i] |= p[7];

  }

  toc = new Toc;

  long lastpregap = 0;
  long pregap = 0;

  for (i = 0, p = data + 4; i < nofTracks; i++, p += 8) {
    Track t(Track::AUDIO);

    Msf trackLength(trackStart[i+1] - trackStart[i]);

    printf("Analyzing track %d: start %s, ", i + 1, Msf(trackStart[i]).str());
    printf("length %s...\n", trackLength.str());

    if ((p[1] & 0x0f) != 0x0f) {
      t.preEmphasis(p[1] & 0x01);
      t.copyPermitted(p[1] & 0x02);
      t.audioType(p[1] & 0x08);
    }

    if (i == 0 && trackStart[i] != 0) {
      discStart = trackStart[i];
      t.append(SubTrack(SubTrack::DATA,
			AudioData(AudioData::SILENCE, 
				  Msf(discStart).samples())));
    }

    lastpregap = pregap;
    pregap = 0;

    if (i < nofTracks - 1) {
      // find pregap of next track
      pregap = findPregap(i + 2, trackStart[i], 
			  trackStart[i+1] - trackStart[i] + 150);
      //printf("Pregap: %ld, track start: %ld\n", pregap,  trackStart[i]);
      if (pregap >= trackStart[i+1]) {
	pregap = 0;
      }
      else if (pregap > 0) {
	pregap = trackStart[i+1] - pregap;
      }
    }

    if (lastpregap > 0) {
      printf("Found pre-gap: %s\n", Msf(lastpregap).str());
    }

    trackLength = Msf(trackStart[i+1] - trackStart[i] - pregap + lastpregap);

    t.append(SubTrack(SubTrack::DATA,
		      AudioData("data.wav", 
				Msf(trackStart[i] - discStart
				    - lastpregap).samples(), 
				trackLength.samples())));
    if (i == 0) {
      t.start(Msf(discStart));
    }
    else {
      t.start(Msf(lastpregap));
    }

    if (readIsrc(i + 1, &t)) {
      printf("Found ISRC code.\n");
    }

    toc->append(&t);

  }

  if (readCatalog(toc)) {
    printf("Found disc catalogue number.\n");
  }

  return toc;
}

// tries to read catalog number from disc and adds it to 'toc'
// return: 1 if valid catalog number was found, else 0

int PlextorReader::readCatalog(Toc *toc)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  char catalog[14];
  int i;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x02; // get media catalog number
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read sub channel data.\n");
    return 0;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read sub channel data.\n");
      return 0;
    }
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 13; i++) {
      catalog[i] = data[0x09 + i];
    }
    catalog[13] = 0;

    if (toc->catalog(catalog) == 0) {
      return 1;
    }
  }

  return 0;
}

// plays one audio block starting from given position
void PlextorReader::playAudioBlock(long start)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;

  // play one audio block
  memset(cmd, 0, 10);
  cmd[0] = 0x45; // PLAY AUDIO
  cmd[2] = start >> 24;
  cmd[3] = start >> 16;
  cmd[4] = start >> 8;
  cmd[5] = start;
  cmd[8] = 1; // 1 block

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot play audio block.\n");
    return;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot play audio block.\n");
      return;
    }
  }
}

// tries to read ISRC code of given track number and stores it in
// given track.
// return: 1 if valid ISRC code was found, else 0
int PlextorReader::readIsrc(int trackNr, Track *track)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  char isrc[13];
  int i;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x03; // get ISRC code
  cmd[6] = trackNr;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read ISRC code.\n");
    return 0;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read ISRC code.\n");
      return 0;
    }
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 12; i++) {
      isrc[i] = data[0x09 + i];
    }
    isrc[12] = 0;

    if (track->isrc(isrc) == 0) {
      return 1;
    }
  }

  return 0;
}

int PlextorReader::readSubChannelData(int *trackNr, int *indexNr, long *relPos)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x00; // get sub Q channel data
  cmd[6] = 0;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read sub Q channel data.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read sub Q channel data.\n");
      return 1;
    }
  }

  *trackNr = data[6];
  *indexNr = data[7];
  *relPos = 0;
  *relPos |= data[0x0c] << 24;
  *relPos |= data[0x0d] << 16;
  *relPos |= data[0x0e] << 8;
  *relPos |= data[0x0f];
  
  return 0;
}

long PlextorReader::findPregap(int trackNr, long trackStart, long trackLength)
{
  int actTrack;
  int actIndex;
  long start = trackStart - 150;
  long end = trackStart + trackLength - 1;
  long mid;
  long relPos;

  while (start < end) {
    mid = start + ((end - start) / 2);
    
    //printf("Checking block %ld... ", mid);
    playAudioBlock(mid);
    if (readSubChannelData(&actTrack, &actIndex, &relPos) != 0) {
      return 0;
    }
    //printf("Found track %d, index %d, pos %ld\n", actTrack, actIndex,
    //       relPos);
    if (actTrack < trackNr) {
      playAudioBlock(mid+1);
      if (readSubChannelData(&actTrack, &actIndex, &relPos) != 0) {
	return 0;
      }
      if (actTrack == trackNr) {
	if (actIndex == 0) {
	  //printf("Found pregap at %ld\n", mid + 1);
	  return mid;
	}
	else {
	  return 0;
	}
      }
      else {
	start = mid + 1;
      }
      
    }
    else {
      end = mid;
    }
  }

  return 0;
}
