/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char rcsid[] = "$Id: CdrDriver.cc,v 1.2 1998/05/22 18:29:16 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>

#include "CdrDriver.h"

CdrDriver::StringTable CdrDriver::senseKeys[] = {
  { 0x00, "NO SENSE" },
  { 0x01, "RECOVERED ERROR" },
  { 0x02, "NOT READY" },
  { 0x03, "MEDIUM ERROR" },
  { 0x04, "HARDWARE ERROR" },
  { 0x05, "ILLEGAL REQUEST" },
  { 0x06, "UNIT ATTENTION" },
  { 0x08, "BLANK CHECK" },
  { 0x09, "VENDOR SPECIFIC" },
  { 0x0b, "ABORTED COMMAND" },
  { 0x0d, "VOLUME OVERFLOW" },
  { 0x0e, "MISCOMPARE" },
  { 0x00, (char *)0 }};


const char *CdrDriver::getFromStringTable(const StringTable *t, int code) const
{
  while (t->message != NULL) {
    if (t->code == code) {
      return t->message;
    }

    t += 1;
  }

  return NULL;
}

// Prints decoded sense message, if 'ignoreUnitAttention' is != 0 and sense
// code indicates unit attention nothing will be printed and 0 will be
// returned.
// return: 0: OK, no error
//         1: sense key indicates error
int CdrDriver::decodeSense(const unsigned char *buf, int len,
			   int ignoreUnitAttention) const
{
  int code = buf[2] & 0x0f;
  const char *msg;

  if (code == 0 || (ignoreUnitAttention != 0 && code == 0x06)) {
    return 0;
  }

  msg = getFromStringTable(senseKeys, code);

  fprintf(stderr, "\nSCSI command failed:\n");
  fprintf(stderr, "  sense key 0x%x: %s.\n", code,
	  msg ? msg : "unknown code");

  if (len > 0x0c && buf[7] != 0) {
    fprintf(stderr, "  additional sense 0x%x\n", (unsigned char)buf[0x0c]);
  }

  return 1;
}

// checks if unit is ready
// return: 0: OK
//         1: scsi command failed
//         2: not ready
//         3: not ready, no disc in drive
//         4: not ready, tray out

int CdrDriver::testUnitReady(int ignoreUnitAttention) const
{
  unsigned char cmd[6];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 6);

  switch (scsiIf_->sendCmd(cmd, 6, NULL, 0, NULL, 0)) {
  case 1:
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if ((sense[2] & 0x0f) == 0x02) {
      // not ready
      return 2;
    }
    else if (decodeSense(sense, senseLen, ignoreUnitAttention) != 0) {
      return 1;
    }
    else {
      return 0;
    }
  }

  return 0;
}

// start unit ('startStop' == 1) or stop unit ('startStop' == 0)
// return: 0: OK
//         1: scsi command failed

int CdrDriver::startStopUnit(int startStop) const
{
  unsigned char cmd[6];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 6);

  cmd[0] = 0x1b;
  
  if (startStop != 0) {
    cmd[4] |= 0x01;
  }

  switch (scsiIf_->sendCmd(cmd, 6, NULL, 0, NULL, 0)) {
  case 1:
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      return 1;
    }
  }

  return 0;
}


// blocks or unblocks tray
// return: 0: OK
//         1: scsi command failed

int CdrDriver::preventMediumRemoval(int block) const
{
  unsigned char cmd[6];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 6);

  cmd[0] = 0x1e;
  
  if (block != 0) {
    cmd[4] |= 0x01;
  }

  switch (scsiIf_->sendCmd(cmd, 6, NULL, 0, NULL, 0)) {
  case 1:
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      return 1;
    }
  }

  return 0;
}

// reset device to initial state
// return: 0: OK
//         1: scsi command failed

int CdrDriver::rezeroUnit() const
{
  unsigned char cmd[6];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 6);

  cmd[0] = 0x01;
  
  switch (scsiIf_->sendCmd(cmd, 6, NULL, 0, NULL, 0)) {
  case 1:
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      return 1;
    }
  }

  return 0;
}

// flushs cache of drive which inidcates end of write action
// return: 0: OK
//         1: scsi command failed
int CdrDriver::flushCache() const
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);

  cmd[0] = 0x35; // FLUSH CACHE
  
  switch (scsiIf_->sendCmd(cmd, 10, NULL, 0, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot flush cache.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot flush cache.\n");
      return 1;
    }
  }

  return 0;
}

// Writes data to target, the block length depends on the actual writing mode
// and is stored internally. 'len' is number of blocks to write.
// return: 0: OK
//         1: scsi command failed
int CdrDriver::writeData(const char *buf, long len) const
{
  assert(blockLength_ > 0);

  int blocksPerCmd = scsiIf_->maxDataLen() / blockLength_;
  assert(blocksPerCmd > 0);

  int nwritten = 0;
  int writeLen = 0;
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);
  cmd[0] = 0x2a; // WRITE1
  
  while (len > 0) {
    writeLen = (len > blocksPerCmd ? blocksPerCmd : len);

    cmd[7] = writeLen >> 8;
    cmd[8] = writeLen & 0xff;

    switch(scsiIf_->sendCmd(cmd, 10, 
			    (unsigned char *)(buf + (nwritten * blockLength_)),
			    writeLen * blockLength_, NULL, 0)) {
    case 1:
      fprintf(stderr, "Write data failed.\n");
      return 1;

    case 2:
      sense = scsiIf_->getSense(senseLen);

      if (decodeSense(sense, senseLen, 0) != 0) {
	fprintf(stderr, "Write data failed.\n");
      	return 1;
      }
    }

    len -= writeLen;
    nwritten += writeLen;
  }
      
  return 0;
}

unsigned char CdrDriver::ascii2Isrc(char c) const
{
  if (isdigit(c)) {
    return (c - '0') & 0x3f;
  }
  else if (isupper(c)) {
    return (c - 'A' + 17) & 0x3f;
  }
  else {
    return 0;
  }
}
