/*
 * Copyright: GNU Public License 2 applies
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * CDDA2WAV (C) 1995-1998 Heiko Eissfeldt heiko@colossus.escape.de
 * parts    (C) Peter Widow
 * parts    (C) Thomas Niederreiter
 * parts    (C) RSA Data Security, Inc.
 *
 * last changes:
 *   18.12.93 - first version,	OK
 *   01.01.94 - generalized & clean up HE
 *   10.06.94 - first linux version HE
 *   12.06.94 - wav header alignment problem fixed HE
 *   12.08.94 - open the cdrom device O_RDONLY makes more sense :-)
 *		no more floating point math
 *		change to sector size 2352 which is more common
 *		sub-q-channel information per kernel ioctl requested
 *		doesn't work as well as before
 *		some new options (-max -i)
 *   01.02.95 - async i/o via semaphores and shared memory
 *   03.02.95 - overlapped reading on sectors
 *   03.02.95 - generalized sample rates. all integral divisors are legal
 *   04.02.95 - sun format added
 *              more divisors: all integral halves >= 1 allowed
 *		floating point math needed again
 *   06.02.95 - bugfix for last track and not d0
 *              tested with photo-cd with audio tracks
 *		tested with xa disk 
 *   29.01.96 - new options for bulk transfer
 *   01.06.96 - tested with enhanced cd
 *   01.06.96 - tested with cd-plus
 *   02.06.96 - support pipes
 *   02.06.96 - support raw format
 *   04.02.96 - security hole fixed
 *   22.04.97 - large parts rewritten
 *   28.04.97 - make file names DOS compatible
 *   01.09.97 - add speed control
 *   20.10.97 - add find mono option
 *   Jan/Feb 98 - conversion to use Joerg Schillings SCSI library
 *
 */

#include "sndconfig.h"
#if defined (HAVE_UNISTD_H) && (HAVE_UNISTD_H == 1)
#include <sys/types.h>
#include <unistd.h>
#endif

#if defined(_GNU_C_SOURCE) || defined(__USE_GNU)
#define USE_GETOPT_LONG
#endif

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <math.h>
#if defined (HAVE_FCNTL_H) && (HAVE_FCNTL_H == 1)
#include <fcntl.h>
#endif
#if defined (TIME_WITH_SYS_TIME)
#include <sys/time.h>
#include <time.h>
#else
#if defined (HAVE_SYS_TIME_H) && (HAVE_SYS_TIME_H == 1)
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif
#if defined (HAVE_LIMITS_H) && (HAVE_LIMITS_H == 1)
#include <limits.h>
#endif
#if defined (HAVE_SYS_IOCTL_H) && (HAVE_SYS_IOCTL_H == 1)
#include <sys/ioctl.h>
#endif

#if defined (HAVE_SYS_WAIT_H) && (HAVE_SYS_WAIT_H == 1)
#include <sys/wait.h>
#endif
#include <saveargs.h>
#include <scsitransp.h>

#include "semshm.h"	/* semaphore functions */
#include "wav.h"	/* wav file header structures */
#include "sun.h"	/* sun audio file header structures */
#include "raw.h"	/* raw file handling */
#include "interface.h"  /* low level cdrom interfacing */
#include "cdda2wav.h"
#include "resample.h"
#include "toc.h"
#include "setuid.h"
#include "ringbuff.h"
#include "global.h"
#include "mycdrom.h"


static const char * optstring = "D:A:I:O:c:b:r:a:t:i:d:o:n:v:l:E:C:S:p:"
#ifdef MD5_SIGNATURES
                         "M:"
#endif
                         "smw"
#ifdef	ECHO_TO_SOUNDCARD
                         "e"
#endif
                         "NqxPRBVhFGHTJ";

#ifdef USE_GETOPT_LONG
#include <getopt.h>	/* for get_long_opt () */
struct option options [] = {
	{"device",required_argument,NULL,'D'},
	{"auxdevice",required_argument,NULL,'A'},
	{"interface",required_argument,NULL,'I'},
	{"output-format",required_argument,NULL,'O'},
	{"channels",required_argument,NULL,'c'},
	{"bits-per-sample",required_argument,NULL,'b'},
	{"rate",required_argument,NULL,'r'},
	{"divider",required_argument,NULL,'a'},
	{"track",required_argument,NULL,'t'},
	{"index",required_argument,NULL,'i'},
	{"duration",required_argument,NULL,'d'},
	{"offset",required_argument,NULL,'o'},
	{"sectors-per-request",required_argument,NULL,'n'},
	{"buffers-in-ring",required_argument,NULL,'l'},
	{"output-endianess",required_argument,NULL,'E'},
	{"cdrom-endianess",required_argument,NULL,'C'},
	{"verbose-level",required_argument,NULL,'v'},
#ifdef MD5_SIGNATURES
	{"md5",required_argument,NULL,'M'},
#endif
	{"stereo",no_argument,NULL,'s'},
	{"mono",no_argument,NULL,'m'},
	{"wait",no_argument,NULL,'w'},
	{"find-extremes",no_argument,NULL,'F'},
	{"find-mono",no_argument,NULL,'G'},
#ifdef	ECHO_TO_SOUNDCARD
	{"echo",no_argument,NULL,'e'},
#endif
	{"no-write",no_argument,NULL,'N'},
	{"no-infofile",no_argument,NULL,'H'},
	{"quiet",no_argument,NULL,'q'},
	{"max",no_argument,NULL,'x'},

	{"no-overlap",no_argument,NULL,'P'},
	{"speed-select",required_argument,NULL,'S'},

	{"dump-rates",no_argument,NULL,'R'},
	{"bulk",no_argument,NULL,'B'},
	{"deemphasize",no_argument,NULL,'T'},
	{"playback-realtime",required_argument,NULL,'p'},
	{"verbose-SCSI",no_argument,NULL,'V'},
	{"version",no_argument,NULL,'J'},
	{"help",no_argument,NULL,'h'},
	{NULL,0,NULL,0}
};

#endif /* USE_GETOPT_LONG */


/* global variables */
global_t global;

/* static variables */
static int (*InitAudioFile)( int audioflag, long channels, unsigned long rate,
			    long nBitsPerSample,
			    unsigned long expected_bytes);
static int (*ExitAudioFile)( int audioflag, unsigned long nBytesDone );
static unsigned long (*GetSndHdrSize)( void );

static unsigned long nSamplesDone = 0;

static	int child_pid = 0xff;

static unsigned long nSamplesToDo;
static unsigned int current_track;
static int bulk = 0;


static void RestrictPlaybackRate( long newrate )
{
       global.playback_rate = newrate;

       if ( global.playback_rate < 25 ) global.playback_rate = 25;   /* filter out insane values */
       if ( global.playback_rate > 250 ) global.playback_rate = 250;

       if ( global.playback_rate < 100 )
               global.nsectors = (global.nsectors*global.playback_rate)/100;
}


long SamplesNeeded( long amount, long undersampling_val)
{
  long retval = ((undersampling_val * 2 + Halved)*amount)/2;
  if (Halved && (nSamplesToDo & 1))
    retval += 2;
  return retval;
}

#ifdef INFOFILES
static void output_indices(FILE *fp, index_list *p)
{
  int ci = 0;

  fprintf(fp, "Index =");
  if (p == NULL) {
    fprintf(fp, "0\n");
    return;
  }

  while (p != NULL) {
    int frameoff = p->frameoffset;

    ci++;
    if ( ci > 8 && (ci % 8) == 1) fputs("\nIndex = ", fp);
    if (frameoff == -1) {
      fprintf(fp, "unavai, ");
    } else {
      fprintf(fp, "%6d, ", frameoff);
    }
    p = p->next;
  }
  fputs("\n", fp);
}

static int write_info_file(char *fname_baseval, unsigned int track, 
		    unsigned long int SamplesDone, int numbered)
{
  FILE *info_fp;
  char fname[200];
  char datetime[30];
  time_t utc_time;
  struct tm *tmptr;

  /* write info file */
  if (!strcmp(fname_baseval,"standard output")) return 0;

  if (numbered)
    sprintf(fname, "%s_%02u.inf", fname_baseval, track);
  else {
    char *pp;

    strncpy(fname, fname_baseval, sizeof(fname) -1);
    fname[sizeof(fname) -1] = 0;
    pp = strrchr(fname, '.');
    if (pp == NULL) {
      pp = fname + strlen(fname);
    }
    strncpy(pp, ".inf", sizeof(fname) - 1 - (pp - fname));
  }
  info_fp = fopen (fname, "w");
  if (!info_fp)
    return -1;

#ifdef MD5_SIGNATURES
  if (global.md5blocksize)
    MD5Final (global.MD5_result, &global.context);
#endif

  utc_time = time(NULL);
  tmptr = localtime(&utc_time);
  if (tmptr) {
    strftime(datetime, sizeof(datetime), "%x %X", tmptr);
  } else {
    strncpy(datetime, "unknown", sizeof(datetime));
  }
  fprintf(info_fp, "#created by cdda2wav "VERSION" %s\n"
	  "Track = %u"
	  , datetime
	  , track
	  );
  if (global.tracktitle[track-1] != NULL) {
    fprintf(info_fp, "Title = '%s'", global.tracktitle[track-1]);
  }
  fprintf(info_fp, 
	  "\n"
	  "Length = %ld milliseconds\n"
	  "ISRC = %s\n"
#ifdef MD5_SIGNATURES
	  "MD-5 = (%d) %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n"
#endif
	  "Pre-emphasis = %s\n"
	  "Channels = %d\n"
	  "Copy permitted = %s\n"
	  "Endianess = %s\n"
	  , (SamplesDone*10 + 220) /441	/* in milliseconds */
	  , g_toc[track-1].ISRC
#ifdef MD5_SIGNATURES
	  , global.md5blocksize
	  , global.MD5_result[0]
	  , global.MD5_result[1]
	  , global.MD5_result[2]
	  , global.MD5_result[3]
	  , global.MD5_result[4]
	  , global.MD5_result[5]
	  , global.MD5_result[6]
	  , global.MD5_result[7]
	  , global.MD5_result[8]
	  , global.MD5_result[9]
	  , global.MD5_result[10]
	  , global.MD5_result[11]
	  , global.MD5_result[12]
	  , global.MD5_result[13]
	  , global.MD5_result[14]
	  , global.MD5_result[15]
#endif
	  , g_toc[track-1].bFlags & 1 ? "yes" : "no"
	  , g_toc[track-1].bFlags & 8 ? 4 : 2
	  , g_toc[track-1].bFlags & 2 ? "yes" : "no"
	  , global.need_big_endian ? "big" : "little"
	  );
    fprintf(info_fp, "# index list\n");
    output_indices(info_fp, global.trackindexlist[track-1]);
    fprintf(info_fp, 
	  "Media Catalog Number (UPC/EAN) = %s\n"
	  "CDDB discid = 0x%08lx\n"
	  "Album title = %s\n"
	  , MCN
	  , (unsigned long) global.cddb_id
	  , global.disctitle != NULL ? global.disctitle : (const unsigned char *)""
	  );
  fclose(info_fp);
  return 0;
}
#endif

static void CloseAudio(char *fname_baseval, unsigned int track, int bulkflag, 
			int channels_val, unsigned long nSamples)
{
#ifdef INFOFILES
      /* define sizes in the header of the audio file */
      if (!global.no_infofile)
            write_info_file(fname_baseval,track,nSamples,bulkflag);
#endif

      /* define length */
      ExitAudioFile( global.audio, nSamples*global.OutSampleSize*channels_val );

      close (global.audio);
      global.audio = -1;
}

static unsigned int track = 1;

/* On terminating:
 * define size-related entries in audio file header, update and close file */
static void CloseAll (void)
{
  int chld_return_status = 0;
  int amichild;

  /* terminate child process first */
  amichild = child_pid == 0;

  if (amichild) {
#ifdef DEBUG_CLEANUP
fprintf(stderr, "Child terminating, \n");
#endif

    if (global.iloop > 0) {
      /* set to zero */
      global.iloop = 0;
      return;
    }

    if (global.nosynccount > 0 && global.nosynccount <= 10) {
      fprintf(stderr,   "Sorry, I was not able to synchronize, so written data is\n"
                        "not reliable :-(.\n");
    } else if (global.nosynccount > 10) {
      fprintf(stderr,   "Sorry, I was not able to synchronize, so written data is\n"
                        "not reliable :-(. You might consider switching to a\n"
			"better suited program like Monty's cdparanoia, which\n"
                        "does a lot more tricks to dissect the correct pieces from\n"
                        "this stream of random numbers:\n"
                        "http://www.mit.edu/afs/sipb/user/xiphmont/cdparanoia/index.html\n");
    }
    /* switch to original mode and close device */
    EnableCdda (0);
    _exit(0);
  }
#ifdef DEBUG_CLEANUP
fprintf(stderr, "Parent terminating, \n");
#endif

#ifdef DEBUG_CLEANUP
fprintf(stderr, "Parent wait for child death, \n");
#endif
#if defined(HAVE_SYS_WAIT) && (HAVE_SYS_WAIT == 1)
  /* wait for child to terminate */
  if (0 > wait(&chld_return_status)) {
    perror("");
  } else {
    if (WIFEXITED(chld_return_status)) {
      if (WEXITSTATUS(chld_return_status)) {
        fprintf(stderr, "Child exited with %d\n", WEXITSTATUS(chld_return_status));
      }
    }
    if (WIFSIGNALED(chld_return_status)) {
      fprintf(stderr, "Child exited due to signal %d\n", WTERMSIG(chld_return_status));
    }
    if (WIFSTOPPED(chld_return_status)) {
      fprintf(stderr, "Child is stopped due to signal %d\n", WSTOPSIG(chld_return_status));
    }
  }
#endif

#ifdef DEBUG_CLEANUP
fprintf(stderr, "Parent child death, state:%d\n", chld_return_status);
#endif
  /* do general clean up */

  if (global.audio>=0) {
    if (bulk) {
      /* finish sample file for this track */
      CloseAudio(global.fname_base, current_track, bulk, global.channels,
  	  global.nSamplesDoneInTrack);
    } else {
      /* finish sample file for this track */
      CloseAudio(global.fname_base, track, bulk, global.channels,
  	  (unsigned int) nSamplesToDo);
    }
  }

  /* tell minimum and maximum amplitudes, if required */
  if (global.findminmax) {
    fprintf(stderr, "minimum amplitude :%d/-32768, maximum amplitude :%d/32767\n", 
	global.minamp, global.maxamp);
  }

  /* tell mono or stereo recording, if required */
  if (global.findmono) {
    fprintf(stderr, "Audio samples are originally %s.\n", global.ismono ? "mono" : "stereo");
  }
}


/* report a fatal error, clean up and exit */
void FatalError (const char *szMessage, ...)
{
  va_list marker;

  va_start(marker, szMessage);

  vfprintf (stderr, szMessage, marker);

  va_end(marker);

  if (child_pid == 0) {
    /* kill the parent too */
    kill(getppid(), SIGINT);
  }
  exit (1);
}


/* open the audio output file and prepare the header. 
 * the header will be defined on terminating (when the size
 * is known). So hitting the interrupt key leaves an intact
 * file.
 */
static void OpenAudio (char *fname, double rate, long nBitsPerSample, 
		       long channels_val, unsigned long expected_bytes)
{
  if (global.audio == -1) {

    global.audio = open (fname, O_CREAT | O_WRONLY | O_TRUNC | O_NONBLOCK
#ifdef SYNCHRONOUS_WRITE
		       | O_SYNC
#endif
		  , 0644);
    if (global.audio == -1) {
      perror("open audio sample file");
      FatalError ("Could not open file %s\n", fname);
    }
  }
  InitAudioFile( global.audio, channels_val, (unsigned long)rate, nBitsPerSample, expected_bytes );

#ifdef MD5_SIGNATURES
  if (global.md5blocksize)
    MD5Init (&global.context);
  global.md5count = global.md5blocksize;
#endif
}


static void set_offset(myringbuff *p, int offset)
{
#ifdef DEBUG_SHM
  fprintf(stderr, "Write offset %d at %p\n", offset, &p->offset);
#endif
  p->offset = offset;
}


static int get_offset(myringbuff *p)
{
#ifdef DEBUG_SHM
  fprintf(stderr, "Read offset %d from %p\n", p->offset, &p->offset);
#endif
  return p->offset;
}


static void usage( void )
{
  fprintf( stderr,
"cdda2wav [-c chans] [-s] [-m] [-b bits] [-r rate] [-a divider] [-S speed]\n"
"         [-t track[+endtrack]] [-i index] [-o offset] [-d duration] [-F] [-G]\n"
"         [-x] [-q] [-w] [-v] [-R] [-P] [-T] [-J]\n"
"	  [-e] [-n sectors] [-N] [-H] [-l buffers] [-D device] [-I interface]\n"
"         [-O audiotype] [-B] [-E output-endianess] [-A auxdevice] [audiofile]\n"
"Version "VERSION"\n"
"cdda2wav copies parts from audio cd's directly to wav or au files.\n"
"It requires a supported cdrom drive to work.\n"
"options: -D device   : set the cdrom or scsi device (as Bus,Id,Lun).\n"
"         -A auxdevice: set the aux device (typically /dev/cdrom).\n"
"         -I interface: specify the interface for cdrom access.\n"
"                     : (generic_scsi or cooked_ioctl).\n"
"         -c channels : set 1 for mono, or 2 for stereo recording.\n"
"         -s          : set to stereo recording.\n"
"         -m          : set to mono recording.\n"
"         -x          : set to maximum quality (stereo/16-bit/44.1 KHz).\n"
"         -b bits     : set bits per sample per channel (8, 12 or 16 bits).\n"
"         -r rate     : set rate in samples per second. -R gives all rates\n"
"         -a divider  : set rate to 44100Hz / divider. -R gives all rates\n"
"         -R          : dump a table with all available sample rates\n"
"         -S speed    : set the cdrom drive to a given speed during reading\n"
"         -P          : switch off overlap sampling;\n"
#ifdef NOTYET
"         -X          : switch off extra alignment paranoia; implies -Y -Z\n"
"         -Y          : switch off scratch detection; implies -Z\n"
"         -Z          : switch off scratch reconstruction\n"
"         -y threshold: tune scratch detection filter; low values catch\n"
"                     : more scratches but may also catch legitimate values\n"
"         -z          : disable 'smart' scratch auto-detect and force \n"
"                     : paranoia to treat all data as possibly scratched\n\n"
#endif
"         -n sectors  : read sectors per request.\n"
"         -l buffers  : use a ring buffer with 'l' elements.\n"
"         -t track[+end track]\n"
"                     : select start track (and optionally end track).\n"
"         -i index    : select start index.\n"
"         -o offset   : start 'offset' sectors behind start track/index.\n"
"                       one sector equivalents 1/75 second.\n"
"         -O audiotype: set wav or sun or cdr (aka raw) audio format. Default is wav.\n"
"         -C endianess: set little or big input sample endianess.\n"
"         -E endianess: set little or big output sample endianess.\n"
"         -d duration : set recording time in seconds or 0 for whole track.\n"
"         -B          : record each track into a seperate file.\n"
"         -w          : wait for audio signal, then start recording.\n"
"         -F          : find extrem amplitudes in samples.\n"
"         -G          : find if input samples are mono.\n"
"         -T          : undo pre-emphasis in input samples.\n"
#ifdef	ECHO_TO_SOUNDCARD
"         -e          : echo audio data to sound device SOUND_DEV.\n"
#endif
"         -v level    : print informations on current cd (level: 0-63).\n"
"         -N          : no file operation.\n"
"         -H          : no info file generation.\n"
#ifdef MD5_SIGNATURES
"         -M count    : calculate MD-5 checksum for 'count' bytes.\n"
#endif
"         -q          : quiet operation, no screen output.\n"
"         -p percent  : play (echo) audio at a new pitch rate.\n"
"         -J          : print version information.\n"
#ifdef USE_GETOPT_LONG
"         -h          : this help screen.\n"  
#endif
"defaults: %s, %d bit, %g Hz, track 1, no offset, one track,\n"
"          type %s '%s', don't wait for signal, not quiet, not verbose,\n"
"          use %s, device %s, aux %s\n"
"parameters: (optional) a file name or - for standard output.\n",
	  CHANNELS-1?"stereo":"mono", BITS_P_S, 44100.0 / UNDERSAMPLING, 
          AUDIOTYPE, FILENAME, DEF_INTERFACE, CD_DEVICE, AUX_DEVICE);
  exit( 1 );
}

static void init_globals(void)
{
  strncpy(global.dev_name, CD_DEVICE, sizeof(global.dev_name));	/* device name */
  strncpy(global.aux_name, AUX_DEVICE, sizeof(global.aux_name));/* auxiliary cdrom device */
  strncpy(global.fname_base, FILENAME, sizeof(global.fname_base));/* auxiliary cdrom device */
  global.audio    = -1;		/* audio file desc */
  global.cooked_fd  = -1;	/* cdrom file desc */
  global.no_file  =  0;		/* flag no_file */
  global.no_infofile  =  0;	/* flag no_infofile */
  global.no_cddbfile  =  0;	/* flag no_cddbfile */
  global.quiet	  =  0;		/* flag quiet */
  global.verbose  =  3 + 32 + 64;	/* verbose level */
  global.sh_bits  =  0;		/* sh_bits: sample bit shift */
  global.Remainder=  0;		/* remainder */
  global.iloop    =  0;		/* todo counter */
  global.SkippedSamples =  0;	/* skipped samples */
  global.OutSampleSize  =  0;	/* output sample size */
  global.channels = CHANNELS;	/* output sound channels */
  global.nSamplesDoneInTrack = 0; /* written samples in current track */
  global.buffers = 2;           /* buffers to use */
  global.nsectors = NSECTORS;   /* sectors to read in one request */
  global.overlap = 1;           /* amount of overlapping sectors */
  global.outputendianess = NONE; /* user specified output endianess */
  global.littleendian = -1; /* user specified output endianess */
  global.findminmax  =  0;	/* flag find extrem amplitudes */
  global.maxamp = INT_MIN;	/* maximum amplitude */
  global.minamp = INT_MAX;	/* minimum amplitude */
  global.speed = DEFAULT_SPEED; /* use default */ 
  global.findmono  =  0;	/* flag find if samples are mono */
  global.ismono  =  1;		/* flag if samples are mono */
  global.deemphasize  =  0;	/* flag undo pre-emphasis in samples */
  global.playback_rate = 100;   /* new fancy selectable sound output rate */
  global.nosynccount = 0;       /* counter for unsuccessful synchronizations */
  global.cddb_id = 0;           /* disc identifying id for CDDB database */
  global.disctitle = NULL;
  global.creator = NULL;
  global.copyright_message = NULL;
  memset(global.tracktitle, 0, sizeof(global.tracktitle));
  memset(global.trackindexlist, 0, sizeof(global.trackindexlist));
}

#if !defined (HAVE_STRCASECMP) || (HAVE_STRCASECMP != 1)
#include <ctype.h>
static int strcasecmp(const char *s1, const char *s2)
{
  while (s1 && s2 && tolower(*s1) == tolower(*s2)) {
    s1++;
    s2++;
  }
  if (s1 == NULL && s2 == NULL) return 0;
  if (s1 == NULL) return -1;
  if (s2 == NULL) return +1;
  return tolower(*s1) - tolower(*s2);
}
#endif

static unsigned long SectorBurst;
#if (SENTINEL > CD_FRAMESIZE_RAW)
#error block size for overlap check has to be < sector size
#endif

#if      defined _POSIX_PRIORITY_SCHEDULING
#include <sched.h>
#endif

/* and finally: the MAIN program */
int main( int argc, char *argv [] )
{
  long lSector;
  long BeginAtSample;
  long lSector_p1;
  long lSector_p2;
  long sector_offset = 0;
  unsigned long SamplesToWrite; 
  unsigned long endtrack = 1;
  double rectime = DURATION;
  unsigned int cd_index = 1;
  double rate = 44100 / UNDERSAMPLING;
  double int_part;
  int bits = BITS_P_S;
  int just_the_toc = 0;
  char fname[200];
  char int_name[100] = DEF_INTERFACE;
  char audio_type[10] = AUDIOTYPE;
  myringbuff *p;
  int c;
#ifdef USE_GETOPT_LONG
  int long_option_index=0;
#endif /* USE_GETOPT_LONG */
  int am_i_cdda2wav;
  char * env_p;

  save_args(argc, argv);

  /* init global variables */
  init_globals();

  /* When being invoked as list_audio_tracks, just dump a list of
     audio tracks. */
  am_i_cdda2wav = strcmp(argv[0],"list_audio_tracks");
  if (!am_i_cdda2wav) global.verbose = SHOW_JUSTAUDIOTRACKS;

  /* Control those set-id privileges... */
  initsecurity();

  env_p = getenv("CDR_DEVICE");
  if (env_p != NULL) {
    strncpy( global.dev_name, env_p, sizeof(global.dev_name) );
    global.dev_name[sizeof(global.dev_name)-1]=0;
  }

  /* command options parsing */
#ifdef USE_GETOPT_LONG
  while ( (c = getopt_long (argc,argv,optstring,options,&long_option_index)) 
		!= EOF)
#else
  while ( (c = getopt(argc, argv, optstring)) != EOF )
#endif /* USE_GETOPT_LONG */
	{
    switch (c) {
      case 'D':    /* override device */
        strncpy( global.dev_name, optarg, sizeof(global.dev_name) );
        global.dev_name[sizeof(global.dev_name)-1]=0;
	break;
      case 'A':    /* override device */
        strncpy( global.aux_name, optarg, sizeof(global.aux_name) );
        global.aux_name[sizeof(global.aux_name)-1]=0;
	break;
      case 'I':    /* override interface */
	strncpy( int_name, optarg, sizeof(int_name) );
        int_name[sizeof(int_name)-1]=0;
	break;

      /* the following options are relevant to 'cdda2wav' only */
#ifdef MD5_SIGNATURES
      case 'M':
        if (!am_i_cdda2wav) break;
	global.md5blocksize = strtoul( optarg, NULL, 10 );
        break;
#endif
      case 'O':    /* override audio type */
        if (!am_i_cdda2wav) break;
	strncpy( audio_type, optarg, 4);
        audio_type[sizeof(audio_type)-1]=0;
	break;
      case 'C':    /* override input endianess */
        if (!am_i_cdda2wav) break;
	if (strcasecmp(optarg, "little") == 0) {
	   global.littleendian = 1;
	} else 
	if (strcasecmp(optarg, "big") == 0) {
	   global.littleendian = 0;
	} else 
	if (strcasecmp(optarg, "guess") == 0) {
	   global.littleendian = -2;
	} else {
	   usage();
	}
	break;
      case 'E':    /* override output endianess */
        if (!am_i_cdda2wav) break;
	if (strcasecmp(optarg, "little") == 0) {
	   global.outputendianess = LITTLE;
	} else 
	if (strcasecmp(optarg, "big") == 0) {
	   global.outputendianess = BIG;
	} else {
	   usage();
	}
	break;
      case 'c':    /* override channels */
        if (!am_i_cdda2wav) break;
	global.channels = strtol( optarg, NULL, 10);
	break;
      case 'S':    /* override drive speed */
        if (!am_i_cdda2wav) break;
	global.speed = strtoul( optarg, NULL, 10);
	break;
      case 'l':    /* install a ring buffer with 'buffers' elements */
        if (!am_i_cdda2wav) break;
        global.buffers = strtoul( optarg, NULL, 10);
        break;
      case 'b':    /* override bits */
        if (!am_i_cdda2wav) break;
	bits = strtol( optarg, NULL, 10);
	break;
      case 'r':    /* override rate */
        if (!am_i_cdda2wav) break;
	rate = strtol( optarg, NULL, 10);
	break;
      case 'a':    /* override rate */
        if (!am_i_cdda2wav) break;
	if (strtod( optarg, NULL ) != 0.0)
	    rate = fabs(44100.0 / strtod( optarg, NULL ));
	else {
	    fprintf(stderr, "-a requires a nonzero, positive divider.\n");
	    usage();
	}
	break;
      case 't':    /* override start track */
        if (!am_i_cdda2wav) break;
	{
	char * endptr;
	char * endptr2;
	track = strtoul( optarg, &endptr, 10 );
	endtrack = strtoul( endptr, &endptr2, 10 );
	if (endptr2 == endptr)
		endtrack = track;
	break;
	}
      case 'i':    /* override start index */
        if (!am_i_cdda2wav) break;
	cd_index = strtoul( optarg, NULL, 10);
	break;
      case 'd':    /* override recording time */
        if (!am_i_cdda2wav) break;
	rectime = strtod( optarg, NULL );
	break;
      case 'o':    /* override offset */
        if (!am_i_cdda2wav) break;
	sector_offset = strtol( optarg, NULL, 10);
	break;
      case 'n':    /* read sectors per request */
        if (!am_i_cdda2wav) break;
	global.nsectors = strtoul( optarg, NULL, 10);
	break;

       /*-------------- RS 98 -------------*/
      case 'p':    /* specify playback pitch rate */
        global.playback_rate = strtol( optarg, NULL, 10);
        RestrictPlaybackRate( global.playback_rate );
        break;

      case 's':    /* stereo */
        if (!am_i_cdda2wav) break;
	global.channels = 2;
	break;
      case 'm':    /* mono */
        if (!am_i_cdda2wav) break;
	global.channels = 1;
	break;
      case 'x':    /* max */
        if (!am_i_cdda2wav) break;
	global.channels = 2; bits = 16; rate = 44100;
	break;
      case 'w':    /* wait for some audio intensity */
        if (!am_i_cdda2wav) break;
	waitforsignal = 1;
	break;
      case 'F':    /* find extreme amplitudes */
        if (!am_i_cdda2wav) break;
	global.findminmax = 1;
	break;
      case 'G':    /* find if mono */
        if (!am_i_cdda2wav) break;
	global.findmono = 1;
	break;
      case 'e':	   /* echo to sound acrd */
        if (!am_i_cdda2wav) break;
#ifdef	ECHO_TO_SOUNDCARD
	global.echo = 1;
#else
	fprintf(stderr, "There is no sound support compiled into %s.\n",argv[0]);
#endif
	break;	
      case 'v':    /* tell us more */
        if (!am_i_cdda2wav) break;
	global.verbose = strtol( optarg, NULL, 10);
	break;
      case 'q':    /* be quiet */
        if (!am_i_cdda2wav) break;
	global.quiet = 1;
	global.verbose = 0;
	break;
      case 'N':    /* don't write to file */
        if (!am_i_cdda2wav) break;
	global.no_file = 1;
	break;
      case 'H':    /* don't write to file */
        if (!am_i_cdda2wav) break;
	global.no_infofile = 1;
	global.no_cddbfile = 1;
	break;
      case 'B':    /* bulk transfer */
        if (!am_i_cdda2wav) break;
	bulk = 1;
	break;
      case 'P':    /* prevent overlap reading */
        if (!am_i_cdda2wav) break;
	global.overlap = 0;
	break;
      case 'T':    /* do deemphasis on the samples */
        if (!am_i_cdda2wav) break;
	global.deemphasize = 1;
	break;
      case 'R':    /* list available rates */
        if (!am_i_cdda2wav) break;
	{ int ii;
	  fprintf(stderr, "Cdda2wav version "VERSION": available rates are:\n"
               "Rate   Divider      Rate   Divider      "
               "Rate   Divider      Rate   Divider\n");
	  for (ii = 1; ii <= 44100 / 880 / 2; ii++) {
	    long i2 = ii;
	    fprintf(stderr, "%-7g  %2ld         %-7g  %2ld.5       ",
                    44100.0/i2,i2,44100/(i2+0.5),i2);
	    i2 += 25;
	    fprintf(stderr, "%-7g  %2ld         %-7g  %2ld.5\n",
                    44100.0/i2,i2,44100/(i2+0.5),i2);
	    i2 -= 25;
	  }
	}
	exit(0);
	break;
      case 'V':
        if (!am_i_cdda2wav) break;
	jes_verbose++;
	break;
      case 'J':
        if (!am_i_cdda2wav) break;
	fputs (VERSION, stdout);
	exit (0);
	break;
#ifdef USE_GETOPT_LONG
      case 'h':
	usage();
	break;
#if 0
      case '?':
	fputs ("use cdda2wav --help to get more information.", stderr);
	exit (1);
	break;
#endif
#endif
      default:
        usage();
    }
  }

  /* check all parameters */
  if (global.verbose < 0 || global.verbose > SHOW_MAX) {
    fprintf(stderr, "Error: incorrect verbose level setting: %d\n",global.verbose);
    usage();
  }
  if (global.verbose == 0) global.quiet = 1;

  if ( global.channels != 1 && global.channels != 2 ) {
    fprintf(stderr, "Error: incorrect channel setting: %d\n",global.channels);
    usage();
  }

  if ( bits != 8 && bits != 12 && bits != 16 ) {
    fprintf(stderr, "Error: incorrect bits_per_sample setting: %d\n",bits);
    usage();
  }

  if ( rate < 827.0 || rate > 44100.0 ) {
    fprintf(stderr, "Error: incorrect sample rate setting: %g\n",rate);
    usage();
  }

  if ( modf( 2*44100.0 / rate, &int_part) >= 0.5 ) {
      int_part += 1.0;
      fprintf( stderr, "Nearest available sample rate is %g Hertz\n",
	      2*44100.0 / int_part);
  }
  Halved = ((int) int_part) & 1;
  rate = 2*44100.0 / int_part;
  undersampling = (int) int_part / 2.0;
  samples_to_do = undersampling;

  if ( rectime < 0 ) {
    fprintf(stderr, "Error: incorrect recording time setting: %f\n",rectime);
    usage();
  }

  if (!strcmp((char *)int_name,"generic_scsi"))
      interface = GENERIC_SCSI;
  else if (!strcmp((char *)int_name,"cooked_ioctl"))
      interface = COOKED_IOCTL;
  else  {
    fprintf(stderr, "Error: incorrect interface setting: %s\n",int_name);
    usage();
  }

  /* check * init audio file */
  if (!strncmp(audio_type,"wav",3)) {
    global.need_big_endian = 0;
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitWav;
    ExitAudioFile = ExitWav;
    GetSndHdrSize = GetWavHdrSize;
  } else if (!strncmp(audio_type, "sun", 3)) {
    /* Enhanced compatibility */
    strcpy(audio_type, "au");

    global.need_big_endian = 1;
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = InitSun;
    ExitAudioFile = ExitSun;
    GetSndHdrSize = GetSunHdrSize;
  } else if (!strncmp(audio_type, "cdr", 3) || 
             !strncmp(audio_type, "raw", 3)) {
    global.need_big_endian = 1;	/* for cd writer compatibility */
    if (global.outputendianess != NONE)
	global.need_big_endian = global.outputendianess == BIG;
    InitAudioFile = (int (*)(int audio, long channels, unsigned long myrate, long nBitsPerSample,
             unsigned long expected_bytes))InitRaw;
    ExitAudioFile = (int (*)(int audio, unsigned long nBytesDone)) ExitRaw;
    GetSndHdrSize = GetRawHdrSize;
  } else {
    fprintf(stderr, "Error: incorrect audio type setting: %3s\n", audio_type);
    usage();
  }
  if (global.no_file) global.fname_base[0] = '\0';

  if (!bulk) {
    strcat(global.fname_base, ".");
    strcat(global.fname_base, audio_type);
  }

  /* all options processed. Now a file name may follow */
  if ( optind < argc ) {
    if (!strcmp(argv[optind],"-")) {
      /* pipe mode */
      if (bulk == 1) {
        fprintf(stderr, "bulk mode disabled while outputting to a pipe\n");
        bulk = 0;
      }
      global.audio = dup (fileno(stdout));
      strncpy( global.fname_base, "standard output", sizeof(global.fname_base) );
    } else {
      /* filename given */
      strncpy( global.fname_base, argv[optind], sizeof(global.fname_base)-8 );
    }
    global.fname_base[sizeof(global.fname_base)-1]=0;
  }

  /* setup interface and open cdrom device */
  /* request semaphores and shared memory */
  SetupInterface( );
  atexit ( free_semshm );

  /* get table of contents */
  tracks = ReadToc( g_toc );
  if (tracks == 0) {
    fprintf(stderr, "No track in table of contents! Aborting...\n");
    exit(10);
  }

  have_CD_extra = FixupTOC(tracks + 1);
  global.cddb_id = calc_cddb_id();

  if ( global.verbose == SHOW_JUSTAUDIOTRACKS ) {
    unsigned int z;

    for (z = 0; z < tracks; z++)
	if ((g_toc[z].bFlags & CDROM_DATA_TRACK) == 0)
	  printf("%02d\t%06u\n", g_toc[z].bTrack, g_toc[z].dwStartSector);
    exit(0);
  }

  if ( global.verbose != 0 )
    fputs( "#Cdda2wav version "VERSION
#if defined _POSIX_PRIORITY_SCHEDULING || defined ECHO_TO_SOUNDCARD
           " plus"
#endif
#if defined _POSIX_PRIORITY_SCHEDULING
           " real time scheduling"
#endif
#if defined ECHO_TO_SOUNDCARD
           " soundcard support"
#endif
           "\n", stderr );

  if ( global.verbose & (SHOW_TOC | SHOW_STARTPOSITIONS) )
    DisplayToc ();

  if (just_the_toc) exit(0);

  if (global.nsectors < 1+global.overlap) {
      fprintf(stderr, "Error: overlap (%d) is >= nsectors (%d)\n",global.overlap, global.nsectors);
      exit(-1);
  }
  if ( global.verbose != 0 && global.overlap == 0)
      fprintf(stderr, "No overlap sampling active\n");

  /* paranoia reigns supreme */
#if defined(HAVE_SYNC) && (HAVE_SYNC == 1)
  sync();
#endif

  /* try to get some extra kicks */
  needroot(0);
#if defined(HAVE_NICE) && (HAVE_NICE == 1)
  nice(-20);
#endif
  dontneedroot();

  /* switch cdrom to audio mode */
  EnableCdda (1);

  atexit ( CloseAll );

  if ( !FirstTrack () )
    FatalError ( "This disk has no audio tracks\n" );

  if ( global.verbose & (SHOW_MCN | SHOW_ISRC) )
    Read_MCN_ISRC();

  /* check if start track is in range */
  if ( track < 1 || track > tracks ) {
    fprintf(stderr, "Error: incorrect start track setting: %d\n",track);
    usage();
  }

  /* check if end track is in range */
  if ( endtrack < track || endtrack > tracks ) {
    fprintf(stderr, "Error: incorrect end track setting: %ld\n",endtrack);
    usage();
  }

  if (cd_index != 1)
    sector_offset += ScanIndices( track, cd_index );
  else if (global.verbose & SHOW_INDICES)
    ScanIndices( track, cd_index );

  lSector = GetStartSector ( track );
  lSector_p1 = GetEndSector ( track ) + 1;

  if ( lSector < 0 )
    FatalError ( "track %d not found\n", track );

  lSector += sector_offset;
  /* check against end sector of track */
  if ( lSector >= lSector_p1 ) {
    fputs( "sector offset exceeds track size (ignored)\n", stderr );
    lSector -= sector_offset;
  }

  if ( lSector < 0L ) {
    fputs( "negative start sector! Set to zero.\n", stderr );
    lSector = 0L;
  }

  lSector_p2 = GetLastSectorOnCd( track );
  if (bulk == 1 && track == endtrack && rectime == 0.0)
     rectime = 99999.0;
  if ( rectime == 0.0 ) {
    /* set time to track time */
    nSamplesToDo = (lSector_p1 - lSector) * CD_FRAMESAMPLES;
    rectime = (lSector_p1 - lSector) / 75.0;
    if (CheckTrackrange( track, endtrack) == 1) {
      lSector_p2 = GetEndSector ( endtrack ) + 1;

      if (lSector_p2 >= 0) {
        rectime = (lSector_p2 - lSector) / 75.0;
        nSamplesToDo = (long)(rectime*44100.0 + 0.5);
      } else {
        fputs( "end track is no valid audio track (ignored)\n", stderr );
      }
    } else {
      fputs( "track range does not consist of audio tracks only (ignored)\n", stderr );
    }
  } else {
    /* Prepare the maximum recording duration.
     * It is defined as the biggest amount of
     * adjacent audio sectors beginning with the
     * specified track/index/offset. */

    if ( rectime > (lSector_p2 - lSector) / 75.0 ) {
      rectime = (lSector_p2 - lSector) / 75.0;
      lSector_p1 = lSector_p2;
    }

    /* calculate # of samples to read */
    nSamplesToDo = (long)(rectime*44100.0 + 0.5);
  }

  global.OutSampleSize = (1+bits/12);
  if (nSamplesToDo/undersampling == 0L) {
      fprintf( stderr, "time interval too short. Duration > %g secs!\n", 
	       undersampling/44100.0);
      exit(-2);
  }
  SamplesToWrite = nSamplesToDo*2/(int)int_part;

  init_soundcard(rate, bits);

  if (global.speed != 0 && SelectSpeed != NULL)
     SelectSpeed(global.speed);

  if ( global.verbose & SHOW_SUMMARY ) {
     if ( !waitforsignal ) {
	if (!bulk)
	  fprintf(stderr, "samplefile size will be %lu bytes.\n",
           GetSndHdrSize() + SamplesToWrite*global.OutSampleSize*global.channels  ); 
	else {
	  int tracks_included;
	  tracks_included = GetTrack(
			      (unsigned) (lSector + nSamplesToDo/CD_FRAMESAMPLES -1))
				     - track + 1;
	  fprintf(stderr, "samplefiles size total will be %lu bytes. %d tracks\n", 
		  tracks_included * GetSndHdrSize() +
	  SamplesToWrite*global.OutSampleSize*global.channels, tracks_included  ); 
	}
     }
     fprintf (stderr, "recording %.5f seconds %s with %d bits @ %g Hz"
	      ,rectime ,global.channels == 1 ? "mono":"stereo", bits, rate);
     if (*global.fname_base) fprintf(stderr, " ->'%s'...", global.fname_base );
     fputs("\n", stderr);
  }

  current_track = track;

  if ( !global.no_file ) {
    if (bulk)
      sprintf(fname, "%s_%02u.%s",global.fname_base,current_track,audio_type);
    else
      strcpy(fname,global.fname_base);
    OpenAudio( fname, rate, bits, global.channels, 
	      (unsigned)(SamplesToWrite*global.OutSampleSize*global.channels) );
  }

  global.Remainder = (75 % global.nsectors)+1;

  global.sh_bits = 16 - bits;		/* shift counter */

  global.iloop = nSamplesToDo;
  if (Halved && (global.iloop&1))
      global.iloop += 2;

  BeginAtSample = lSector * CD_FRAMESAMPLES;

#if defined(HAVE_SEMCTL) && (HAVE_SEMCTL == 1)
#else
  init_pipes();
#endif

  /* Everything is set up. Now fork and let one process read cdda sectors
     and let the other one store them in a wav file */

  /* forking */
  child_pid = fork();

  if (child_pid < 0) {
    perror("fork()");
  }
  /*********************** fork **************************************/
  if (child_pid == 0) {
    /* child READER section */

    unsigned total_retries = 0;
    struct sigaction sigac;

#if      defined _POSIX_PRIORITY_SCHEDULING

    int sched_fifo_min, sched_fifo_max;
    struct sched_param sched_parms;

    sched_fifo_min = sched_get_priority_min(SCHED_FIFO);
    sched_fifo_max = sched_get_priority_max(SCHED_FIFO);
    sched_parms.sched_priority = sched_fifo_max - 1;
    needroot(0);
    if (-1 == sched_setscheduler(getpid(), SCHED_FIFO, &sched_parms)
        && global.quiet != 1)
	perror("cannot set realtime scheduling policy");
    dontneedroot();
#endif

#if defined(HAVE_SEMCTL) && (HAVE_SEMCTL == 1)
#else
    init_child();
#endif

    while (global.iloop) {
      unsigned char *newbuf;
      unsigned int added_size;
      int offset;
      unsigned int retry_count = 0;

      /* get next buffers */
#ifdef DEBUG_BUFFER
fprintf(stderr, "CH: wait for empty buffer, ");
#endif
      p = get_next_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "CH: got empty\n");
#endif

      /* how many sectors should be read */
      if (global.overlap && global.nsectors*CD_FRAMESAMPLES > global.iloop) {
	SectorBurst = min(global.nsectors,
			  global.overlap + (global.iloop + CD_FRAMESAMPLES-1) / CD_FRAMESAMPLES);
      } else {
	SectorBurst = min(global.nsectors,
			  (global.iloop + CD_FRAMESAMPLES-1) / CD_FRAMESAMPLES);
      }
      if (lSector_p2 < lSector+(int)SectorBurst-1)
        SectorBurst = lSector_p2 - lSector + 1;

#if 0
fprintf(stderr,"CH: reading from sec %6ld to %6ld,  %6ld...\n",
       lSector, lSector+SectorBurst-1, global.iloop/CD_FRAMESAMPLES );
#endif

#define MAX_READRETRY 1

      do {
          ReadCdRom( p->data, lSector, SectorBurst );
      } while (NULL ==
                 (newbuf = synchronize( p->data, SectorBurst*CD_FRAMESAMPLES, nSamplesToDo-global.iloop )) &&
                 ++retry_count < MAX_READRETRY);

      /* how much has been added? */
      if (newbuf) {
        offset = newbuf - ((unsigned char *)p->data);
      } else {
        offset = global.overlap * CD_FRAMESIZE_RAW;
      }
      set_offset(p,offset);

#ifdef DEBUG_BUFFER
fprintf(stderr, "CH: defining buffer, ");
#endif
      define_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "CH: defined\n");
#endif

      added_size = SectorBurst * CD_FRAMESAMPLES - offset/4;

#if 0
fprintf(stderr, "CH: added_size (secs =%d), (samples=%d)\n",added_size / CD_FRAMESAMPLES,added_size);
#endif
      if (retry_count) {
        fprintf(stderr,"retries while reading from sec %6ld to %6ld,  %6ld...\n",
                lSector, lSector+SectorBurst-1, global.iloop/CD_FRAMESAMPLES );
        total_retries += retry_count;
      }

      lSector += SectorBurst - global.overlap;

      if (global.iloop >= added_size)
        global.iloop -= added_size;
      else
        global.iloop = 0;

    }
    if (total_retries) {
      fprintf(stderr,"%u retries while reading\n",total_retries);
    }
#if 0
fprintf(stderr, "CH: done\n");
#endif
    exit(0);
  } else {
    /******************* parent WRITER  section *********************/
    int child_state;
    
#define left_in_this_track(pos, track) (max(0,(int)g_toc[track].dwStartSector*CD_FRAMESAMPLES - (int)(pos)))

#if      defined _POSIX_PRIORITY_SCHEDULING

    int sched_fifo_min, sched_fifo_max;
    struct sched_param sched_parms;

    sched_fifo_min = sched_get_priority_min(SCHED_FIFO);
    sched_fifo_max = sched_get_priority_max(SCHED_FIFO);
    sched_parms.sched_priority = sched_fifo_max - 2;
    needroot(0);
    if (-1 == sched_setscheduler(getpid(), SCHED_FIFO, &sched_parms)
        && global.quiet != 1)
	perror("cannot set realtime scheduling policy");
    dontneedroot();
#endif

    /* don't need these anymore.  Good security policy says we get rid
       of them ASAP */
    neverneedroot();
    neverneedgroup();

#if defined(HAVE_SEMCTL) && (HAVE_SEMCTL == 1)
#else
    init_parent();
#endif
    while (nSamplesDone < nSamplesToDo) {
      int current_offset;

      /* get oldest buffers */
#ifdef DEBUG_BUFFER
fprintf(stderr, "PA: wait defined buffer, ");
#endif
      p = get_oldest_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "PA: got defined\n");
#endif

      current_offset = get_offset(p);
      /*p += current_offset;*/

      /* how many bytes are available */
      SectorBurst = global.nsectors*CD_FRAMESAMPLES - current_offset/4;
      SectorBurst = min((nSamplesToDo-nSamplesDone),SectorBurst);

      /* when track end is reached, close current file and start a new one */

#if 0
fprintf(stderr, "PA: before loop: track %d limit at %d, we have up to %ld\n",
	current_track,
	g_toc[current_track+1 -1].dwStartSector*CD_FRAMESAMPLES, 
	SectorBurst + nSamplesDone + BeginAtSample); 
#endif

      while ((nSamplesDone < nSamplesToDo) && (SectorBurst != 0)) {
	long unsigned int how_much = SectorBurst;
	long unsigned int left_in_track = left_in_this_track(BeginAtSample+nSamplesDone,
							     current_track);

	if (bulk)
	  how_much = min(how_much, left_in_track);

#if 0
fprintf(stderr, "PA: in loop: left %lu, shall write %lu samples\n", left_in_track, how_much);
#endif


#ifdef MD5_SIGNATURES
	if (global.md5count) {
	  MD5Update (&global.context, ((unsigned char *)p->data) +current_offset, min(global.md5count,how_much));
	  global.md5count -= min(global.md5count,how_much);
	}
#endif
	if ( SaveBuffer ( p->data + current_offset/4,
			 how_much,
			 &nSamplesDone) ) {
	  /* kill child */
	  kill(child_pid, SIGINT);
	  exit(2);
	}
#if 0
fprintf(stderr, "PA: 1wrote %08ld from %08ld samples\n", nSamplesDone, 
	SectorBurst + BeginAtSample); 
#endif
        global.nSamplesDoneInTrack += how_much;
	SamplesToWrite -= nSamplesDone;

	/* move residual samples upto buffer start */
	if (how_much < SectorBurst) 
	  memmove((char *)(p->data) + current_offset,
		  (char *)(p->data) + current_offset + how_much*4,
		  (SectorBurst - how_much) * 4);
	
	if (left_in_track < SectorBurst) {

	  if (bulk) {
	    /* finish sample file for this track */
	    CloseAudio(global.fname_base, current_track, bulk, global.channels,
	  	  global.nSamplesDoneInTrack);
          } else if (SamplesToWrite == 0) {
	    /* finish sample file for this track */
	    CloseAudio(global.fname_base, track, bulk, global.channels,
	  	  (unsigned int) nSamplesToDo);
	  }

	  if (global.verbose) {
	    if (global.tracktitle[current_track -1] != NULL) {
	      fprintf( stderr, "track '%s' successfully recorded\n", global.tracktitle[current_track-1]);
            } else {
	      fprintf( stderr, "track %2u successfully recorded\n", current_track);
            }
          }

          global.nSamplesDoneInTrack = 0;
	  if ( bulk && SamplesToWrite > 0 ) {
	    if ( !global.no_file ) {

	      /* build next filename */
	      sprintf(fname, "%s_%02u.%s",global.fname_base,current_track+1,audio_type);
	      OpenAudio( fname, rate, bits, global.channels,
			(g_toc[current_track].dwStartSector -
			 g_toc[current_track-1].dwStartSector)*CD_FRAMESIZE_RAW);
	    }
	  }
	  current_track++;
	  jitterShift = 0;
	}
	SectorBurst -= how_much;

      }  /* end while */

#ifdef DEBUG_BUFFER
fprintf(stderr, "PA: drop buffer, ");
#endif
      drop_buffer();
#ifdef DEBUG_BUFFER
fprintf(stderr, "dropped\n");
#endif
    } /* end while */
  if (global.verbose) {
    if (global.tracktitle[current_track -1] != NULL) {
      fprintf( stderr, "track '%s' successfully recorded\n", global.tracktitle[current_track-1]);
    } else {
      fprintf( stderr, "track %2u successfully recorded\n", current_track);
    }
  }

    exit(0);
  }
  if (!global.quiet) fputs( "\n", stderr );

  return 0;
}
