/*
 *       vtscreensaver.c  An svgalib screensaver for consoles,
 *       (c)1997 Jesse Reichler
 *       This program is free software; you can redistribute it and/or
 *       modify it under the terms of the GNU General Public License
 *       as published by the Free Software Foundation.
 *
 *        with keyboard interrupt checking based on:
 *         [ switchd.c,     a monitor keyboard and switch to a vc on inactivity
 *           which is Copyright (c) 1996 by Ervin Walter <00edwalter@bsuvc.bsu.edu>
 *           This program is free software; you can redistribute it and/or
 *           modify it under the terms of the GNU General Public License
 *           as published by the Free Software Foundation; either version
 *           2 of the License, or (at your option) any later version.
 *         ]
 *
 *
 *      
 *
 *       MIGHT require svgalib 1.2.11 or greater for propper svgalib support.
 *       MIGHT require kernal 2.0 or better.
 *
 *       WILL fail on svgalib progs if your svgalib is not configured properly to restore textmode from
 *         the svgalib program you invoke.  Nonsvgalib programs should work fine as screensavers.
 *
 *       MUST be a free tty to dynamically allocate.
 *        just as X allocates the first free tty, vtscreensaver will do the same.
 *        (you may have to MAKEDEV /dev/tty? to properly set their premissions before they were useable)
 *       
 *
 *       SECURITY WARNING-> this program includes a call to exec(), which will run ANY
 *                          program passed to it, and so this program should ONLY be
 *                          runnable by root.  It checks and insists that SETUID is not set, for safety.
 *
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/vt.h>
#include <sys/kd.h>
#include <sys/ioctl.h>
#include <linux/tty.h>
#include <signal.h>
#include <time.h>
#include <wait.h>
#include <dirent.h>

#define VERSION "0.2"
#define TTYBASENAME "/dev/tty%d"
#define PROCDIR "/proc"
#define PROGNAME "vtscreensaver"




void usage(int stat)
{
   fprintf(stderr,
      "\nvtscreensaver version " VERSION "\n"
      "Usage: vtscreensaver [options] inactivity_delay program_to_run_plus_args\n"
      "   options: -[dgks]\n"
      "      d  Debug mode (prints verbose messages).\n"
      "      g  invoke even if current vt is in Graphics mode.\n"
      "      k  vtscreensaver takes responsibility for monitoring Keyboard\n"
      "         (possible performance hit).\n"
      "      s  vtscreensaver assumes target program is Safe, and does no monitoring).\n"
      "         (opposite of -k option).\n"
      "      p  instructs vtscreensaver not to restart the target each time, but instead\n"
      "         to Preserve it and switch to it on subsequent inactivity periods.\n"
      "      P  same as -p but target is suspended when switched.\n"
      "         [caution: can cause svgalib programs to hang - see README].\n"
      "      x  do not run vtscreensaver, force eXit of any running copies.\n"
      "      i  invoke immediately (can only be used on prev. run vtscreensaver process).\n\n"
      "ex: vtscreensaver -dg 60 /usr/local/bin/particle.exe 30 10\n\n"
      "    causes the program particle.exe, which may be an svgalib program, to be\n"
      "    invoked with the arguments 10 and 5, after 60-90seconds of keyboard inactivity\n"
      "    (the variable range is due to periodic checking every delay/2 seconds).\n\n");
   exit (stat);
}




// globals we need
int ssvt=-1;         // global memory of screensavers vt
int deadchild=2;     // global flag for a dead child
int delaytime;       // global delaytime between invocations (used for safety checking)
int safetychecktime; // global time between safety checks
int childpid;        // global child pid (needed for safety killing)
int debugmode=0;     // global debugmode flag
int preserve=0;      // global flag for preserve mode
int calledvt=-1;     // global id of vt we are invoked from
long presstime=0L;   // global count of delay between presses


// non-int function declarations
void childies(int dummy);
void safetyalarm(int dummy);
void setsafetyalarm();
void setchildalarm();
long timesincepress();
void switchtovt(int vtno);
void vtredirecto(int vtno);
void waitonkeyboard();
void killpid(int pid);
void debugvt(int vtno);
void waitforvt(int vtno);
void setquitter();
void unsetquitter();
void quit(int dummy);
int killcopies();
int invokeold();
void invoke(int dummy);



void main (int argc, char **argv)
{
  int pid;
  int i;
  int times_invoked=0;
  char progname[100];
  int evengraphics=0;
  int startargs=1;
  int vtmonitor=0;
  int delaybetween;
  int retv;
  

  // Verify that only REAL root may run
  if (getuid() != 0)
    {
      fprintf(stderr, "vtscreensaver: must be root, and setuid is not allowed.\n");
      exit(3);
    }


  // check for proper usage
  if (argc-1 <startargs)
    usage(1);
  
  // get any options specified, either as one string or multiples, and adjust startargs counter
  while (argv[startargs][0]=='-')
    {
      // options were specified
      if (strchr(argv[startargs],'d')!=NULL)
	debugmode=1;
      if (strchr(argv[startargs],'g')!=NULL)
	evengraphics=1;
      if (strchr(argv[startargs],'p')!=NULL)
	preserve=1;
      if (strchr(argv[startargs],'P')!=NULL)
	preserve=2;
      if (strchr(argv[startargs],'k')!=NULL)
	vtmonitor=1;
      if (strchr(argv[startargs],'s')!=NULL)
	vtmonitor=-1;
      if (strchr(argv[startargs],'x')!=NULL)
	{
	  retv=killcopies();
	  if (retv>0)
	    printf("vtscreensaver signing off...\n");
	  else
	    printf("no copies of vtscreensaver were found running...\n");
	  exit(1);
	}
      if (strchr(argv[startargs],'i')!=NULL)
	{
	  retv=invokeold();
	  if (retv>0)
	    printf("invoking vtscreensaver...\n");
	  else
	    printf("vtscreensaver was not found running...\n");
	  exit(1);
	}

      if (argc-1 <++startargs)
	usage(1);
    }



  // check again for proper usage
  if (argc-1<startargs)
    usage(1);


  // Get delaytime commandline Arguments
  delaytime = atoi(argv[startargs++]);
  delaybetween = delaytime/2;

  // check for acceptable delay times
  if (delaytime<10 && debugmode==0)
    {
    printf("vtscreensaver error: minimum delay time in nondebug mode is 10 seconds\n");
    printf("                     (to ensure that you have enough time to log in and\n");
    printf("                     kill me in an emergency).\n");
    exit(-1);
    }
  else if (delaytime<10 && debugmode==1)
    {
    printf("vtscreensaver error: minimum delay time in debug mode is 10 seconds.\n");
    exit(-1);
    }



  // set safetycheck time
  if (preserve!=0)
    safetychecktime=1; // preserve must intercept keystrokes
  else if (debugmode)
    safetychecktime=5; // auto set of a good safetychecktime
  else if (vtmonitor==1)
    safetychecktime=1;
  else if (vtmonitor==-1)
    safetychecktime=0;
  else
    safetychecktime=delaytime;
  

  // check again for proper usage
  if (argc-1<startargs)
    usage(1);
  if (argv[startargs][0]=='-') // when they reverse the order of the options
    usage(1);

  // now get the program to run, rest of args will be passed on to exec
  if (strlen(argv[startargs])>80)
    {
    fprintf(stderr, "vtscreensaver err: target progname too large (>80).\n");
    exit(0);
    }
  strcpy(progname,argv[startargs]);



  // tell the world we are starting
  printf("Starting vtscreensaver...\n");
  // announce more if we are in debugmode
  if (debugmode)
    {
      printf("\n(running with debug option engaged) will invoke '%s' after %d-%d seconds of inactivity, a maximum of 5 times, and checks for child exit every 5 seconds.\n",progname,delaytime,delaytime+delaybetween);
      if (preserve==2)
	printf("\nWARNING: you have the -P, Preserve Suspend option set.  Most svgalib programs will hang if they are explicitly suspended by vtscreensaver AND are already suspended automatically by svgalib when using the -p option.\n\n");
    }







  // Fork to create daemon
  if ((pid = fork()) < 0)
    {
      perror("vtscreensaver err: Couldn't fork() daemon process, exiting.");
      exit(5);
    }
  else
    {
      if (pid != 0)
	exit(0);
    }


  // clean up a bit
  setsid(); // release the tty
  umask(0); // created files will have no permissions

  nice(10); // make it a lower priority process


  // set a termination handler
  killcopies();
  setquitter();


  // now start the monitoring loop
  while(1) 
    {
      // has enought innactive time passed?
      if (timesincepress() >= delaytime) 
	{
	  // run the screensaver
	  if (evengraphics || intextmode())
	    {
	      if (debugmode)
                printf("Debugmode engaging, try no. %d of %d\r\n",times_invoked,5);

	      // execute our beneficiary
              runprogram(progname,&argv[startargs]);

	      if (debugmode && ++times_invoked==5)
		{
		  printf("vtscreensaver debugmode is done testing.  exiting.\r\n");
		  exit(1);
		}
	    }
	  else
	    {
	      // even though the screensaver will save properly over
	      // another svgalib prog and over X we assume user doesnot want this
	    }
	}
      // go to sleep for a bit
      usleep(delaybetween*1000000);
    }

}











int runprogram(char *progname, char *const argv[])
{
  // run the program specified
  int retv=0;
  int pid;
  int count;
  int vtno;
  int wstatus;


  // save the current vt, before the screen saver kicks in
  calledvt=getcurrentvt();

  if (deadchild==1)
    {
      // we can be in this situation while running in preserve mode
      // the child died while waiting in the background.
      //   wait((int *)0);
      waitpid(childpid,&wstatus,WUNTRACED);
      if (WIFSTOPPED(wstatus))
	{
	  // the child is just temporarily stopped
          kill(childpid,SIGCONT);
	  deadchild=0;
	}
      else
	deadchild=2; // child is dead and was waited for
    }

  if (deadchild==0)
    {
      // we are in a preserve mode, and child is still alive
      //  so we just switch to its vt
      switchtovt(ssvt);
    }
  else
    {
      // there is no living child, so we need to create one
      // fork the child

      if ((pid = fork()) < 0)
	{
	  perror("vtscreensaver runprogram err: Couldn't fork() daemon process, exiting.");
	  exit(5);
	}

      if (pid == 0)
	{
	  // Child: executes the program

          // first make sure we don't respond the quit signals of parent
          unsetquitter();

	  //  dynamically create a new vt, run the target there, and then pop back to this one.
	  vtno=newvt();
	  vtredirecto(vtno);

	  // now re can execute it
          retv=execvp(progname,argv);

	  // if we are here there was an error exec'ing
          fprintf(stderr,"Error executing '%s' which returned %d from execvp. Check file and path.\r\n",progname,retv);
          exit(0);
        }
     // parent of fork goes here
     deadchild=0;
     childpid=pid;
    }


  // register (or reregister as the case may be) the child watcher
  setchildalarm();


  if (preserve==0)
    {
      // set the safety alarm watcher
      if (safetychecktime>1)
        setsafetyalarm();
      else if (safetychecktime==1)
        {
          waitonkeyboard();
          killpid(childpid);
	  // turn off the alarm since it's no longer needed
	  alarm((long)0L);
	  signal(SIGALRM,SIG_DFL);
        }
      else
        {
          // a darling trusted child - how rare!
        }

      // now formally wait for the death by calling wait()
      wait((int *)0);
      deadchild=2; // child is dead and was waited for

      // before we leave, unregister the signal watchers
      signal(SIGCHLD,SIG_DFL);
      signal(SIGALRM,SIG_DFL);
      alarm((long)0L);
    }
  else
    {
      // For the preserve option, we do not wait for death, instead we
      //  watch keyboard very closely, and jump out when we get a key
      usleep(1*1000000); // give svgalib time to change to its vt
      ssvt=getcurrentvt();
      waitonkeyboard();
      usleep(2*100000); // give svgalib time to leave its vt
      if (preserve==2)
        {
          // suspend child before leaving
          kill(childpid,SIGSTOP);
        }
      else if (deadchild==1)
        {
          // be nice and check here, even though if it dies later
          //  we still detect it.
          wait((int *)0);
          deadchild=2; // child is dead and was waited for
        }
    }

  // restore the called vt, before we did screen saver.
  switchtovt(calledvt);

  retv=getcurrentvt();
  if (calledvt!=retv)
   {
     printf("vtscreensaver warning: currentvt=%d  and we wanted to be at %d\n",retv,calledvt);
     //  exit(0);
   }

  calledvt=-1;
  return(retv);
}

















// vt ancillary functions


int intextmode()
{
  // check if the current vt is in textmode
  long conmode;
  int fd2;
  
  if ((fd2 = open("/dev/console",O_WRONLY)) < 0)
    {
      perror("vtscreensaver err: Can't open /dev/console.  exiting.\n");
      exit(4);
    }
  ioctl(fd2, KDGETMODE, &conmode);
  close(fd2);
  
  if (conmode==KD_TEXT)
    return 1;
  else
    return 0;
}



int getcurrentvt()
 {
 // find out what the current vt # is
 int fd;
 int vtno;
 struct vt_stat vts;

 if ((fd = open("/dev/console",O_WRONLY)) < 0)
   {
     perror("vtscreensaver err: Can't open /dev/console.  exiting.\n");
     exit(4);
   }

 (void) ioctl(fd, VT_GETSTATE, &vts);
 vtno=vts.v_active;

 close(fd);
 return vtno;
 }



void switchtovt(int vtno)
 {
 // switch to a given terminal number
 int fd;
 int fd2;
 char vtname[80];

 if (debugmode)
   {
     printf(". vtscreensaver trying to switch to vt %d.\r\n",vtno);
     debugvt(vtno);
   }

 if ((fd = open("/dev/console",O_WRONLY)) < 0)
   {
     perror("vtscreensaver err: Can't open /dev/console.  exiting.\n");
     exit(4);
   }

 if (ioctl(fd, VT_ACTIVATE, vtno) < 0) 
   {
     perror("vtscreensaver err: Can't switch vt's.  exiting.\n");
     close(fd);
     exit(6);
   }

 // switch to the vt
 waitforvt(vtno);

 if (debugmode)
   printf(". done switching to vt %d.\r\n",vtno);

 close(fd);
 }



void vtredirecto(int vtno)
{
  // redirect stdout to the specified vt
  // if -1 is passed, redirects to /dev/null
  char vtname[80];

  if (vtno==-1)
    {
      if (freopen("/dev/null","w",stdout) == NULL)
	perror("vtscreensaver: Can't redirect stdout to /dev/null\n");
      return;
    }

  sprintf(vtname,TTYBASENAME,vtno);
  if (freopen(vtname,"w",stdout) == NULL)
    perror("vtscreensaver: Can't redirect stdout to switched tty.\n");
}



int newvt()
{
  // find and switch to a new unused vt
  int fd;
  int vtno;
  char vtname[80];

  if ((fd = open("/dev/console",O_WRONLY)) < 0)
    {
      perror("switchd: Can't open /dev/console\n");
      exit(4);
    }

  // find a free vt
  if (ioctl(fd, VT_OPENQRY, &vtno) < 0)
    {
      perror("vtscreensaver err: Can't find free vt.\n");
      close(fd);
      exit(6);
    }
  if (debugmode)
    printf("vtscreensaver opening new tty #%d.\n",vtno);

  close(fd);


  // now switch to it
  switchtovt(vtno);


  // now redirect to it
  sprintf(vtname,TTYBASENAME,vtno);
  if ((fd = open(vtname, O_RDWR)) ==-1)
    { 
      perror("vtscreensaver err: Can't open a switched vt.  exiting.\n");
      exit (4);
    }
  close(0);  close(1);  close(2);
  dup(fd);
  // and close the file handle
  close(fd);

  return vtno;
}



void debugvt(int vtno)
{
 int fd;
 struct vt_mode vtd;

 if ((fd = open("/dev/console",O_WRONLY)) < 0)
   {
     perror("vtscreensaver err: Can't open /dev/console.  exiting.\n");
     exit(4);
   }

 (void) ioctl(fd, VT_GETMODE, &vtd);

 printf(".. debug info for vt %d.\r\n",vtno);
 printf(".. debugvt mode=%d  (AUTO=%d PROCESS=%d ACKACQ=%d).\r\n",vtd.mode,VT_AUTO,VT_PROCESS,VT_ACKACQ);
 printf(".. debugvt waitv=%d.\r\n",vtd.waitv);
 printf(".. debugvt sigs=%d,%d.\r\n",vtd.relsig,vtd.acqsig);

 close(fd);
}



void waitforvt(int vtno)
{
 // A manual replacement for   ioctl(fd, VT_WAITACTIVE, vtno)
 int fd;
 struct vt_mode vtd;
 int curvt;


 // I have found the svgalib hangs sometimes if we use the recommended 
 //  ioctl(fd, VT_WAITACTIVE, vtno).  But we still need to wait somehow, and I think
 //  that resending the VT_ACTIVATE signal (inside the loop below) is nesc. as well
 //  so maybe the activate signal is just getting lost.

 if (debugmode)
   printf(". vtscreensaver manually waiting for succesful switch to vt %d.\r\n",vtno);

 if ((fd = open("/dev/console",O_WRONLY)) < 0)
   {
     perror("vtscreensaver err: Can't open /dev/console.  exiting.\n");
     exit(4);
   }

 // wait until we get there
 while ((curvt=getcurrentvt())!=vtno)
   {
     if (debugmode)
       printf("... still waiting for %d to be ready (we're are %d).\r\n",vtno,curvt);
     usleep(1*100000);
     ioctl(fd, VT_ACTIVATE, vtno); // send another activation signal
   }


 // ioctl(fd, VT_WAITACTIVE, vtno); // no need for this since we are there
 close(fd);
}














// keyboard interupt checking


long timesincepress()
{
  // Return as estimate of how long it has been since the last keypress
  //  uses some static variables.
  // The idea to check /proc/interrupts is from Erwin Walter's switchd program
  FILE *intfp;
  char line[80], kbdnumstr[10];
  long kbdcount;
  long curtime,elapsedtime;
  static long oldkbdcount=0L;
  static long lasttime=-1L;

  // compute how long it has been since our last keyboard check
  curtime=time(NULL);
  if (lasttime==-1L)
    lasttime=curtime;
  elapsedtime=curtime-lasttime;
  lasttime=curtime;

  // open and find the keyboard interrupt counter
  if ((intfp = fopen("/proc/interrupts", "r")) != NULL)
    {
      while (fgets(line, 80, intfp) != NULL)
	{
	  if (strstr(line, "keyboard") != NULL)
	    {
	      //  get the current keyboard count

              // Andres Kies points out that truncating the keyboard count at 9,
              //  as seen in Ervin Walter's switchd, loses the last two digits of
              //  the interrupt count, and is unnesc. as atoi stops at the end
              //  of the number automatically.
	      kbdcount = atoi(line+3);

              // check it against last counte
              if (kbdcount != oldkbdcount)
             	{
         	  oldkbdcount = kbdcount;
	          presstime = 0L;
	        }
              else
                {
                  // some time has passed since the last key was pressed
                  presstime += elapsedtime;
                }
              fclose(intfp);
	      return presstime;
	    }
	}
      // we should never have gotten here, because it means there was no keyboard line
      // in the proc file!
      fclose(intfp);
      fprintf(stderr,"vtscreensaver err: Could't find keyboard line in /proc/interrupts.\n");
      exit(0);
    }

  // We couldn't open the /proc/interrupts file
  fprintf(stderr,"vtscreensaver err: Could't open /proc/interrupts file.\n");
  exit(0);
}
















// signal watchers

void childies(int dummy)
{
  // a debug mode signal watcher which helps us make sure
  //  that target program is safe.
  deadchild=1;
  if (debugmode)
    printf(". child (target program) died.\r\n");
}


void safetyalarm(int dummy)
{
  // a safety check to make sure that the child we invoked is
  //  willing to terminate on a key press

  if (timesincepress()<delaytime)
    {
      // uh oh, child is not willing to give up on keypress
      // so we want to kill the child.
      killpid(childpid);
      // turn off the alarm since it's no longer needed
      alarm((long)0L);
      signal(SIGALRM,SIG_DFL);
    }
  else
    {
      // no need to panic yet, so reset the alarm
      alarm((long)safetychecktime);
    }
}



void setsafetyalarm()
{
  // set the safety alarm, so that it does not turn off after invoking once (the default)
  struct sigaction sa;
  int retv;

  sa.sa_handler=safetyalarm;
  sa.sa_flags=SA_RESTART;
  sa.sa_restorer=NULL;
  sa.sa_mask=0;
  retv=sigaction(SIGALRM,&sa,NULL);
  alarm((long)safetychecktime);
}


void setchildalarm()
{
  // set the child alarm, so that we are informed when a child dies
  struct sigaction sa;
  int retv;

  sa.sa_handler=childies;
  sa.sa_flags=SA_ONESHOT;
  sa.sa_restorer=NULL;
  sa.sa_mask=0;
  retv=sigaction(SIGCHLD,&sa,NULL);
}


void setquitter()
{
  // set the quiting signal watcher, to be invoked when we are killed
  signal(SIGKILL,quit);
  signal(SIGTERM,quit);
  signal(SIGUSR1,invoke);
}

void unsetquitter()
{
  // unset the quiting signal watcher, to be invoked when we are killed
  signal(SIGKILL,SIG_DFL);
  signal(SIGTERM,SIG_DFL);
  signal(SIGUSR1,SIG_DFL);
}






void quit(int dummy)
{
  // we have to shutdown, so cleanup

  // kill the child
  if (deadchild==0)
    {
      killpid(childpid);
      wait((int *)0);
    }
  else if (deadchild==1)
    wait((int *)0);

  // restore a vt if we are still switched away
  if (calledvt!=-1)
    switchtovt(calledvt);

  exit(1);
}



void invoke(int dummy)
{
  // we have to shutdown, so cleanup

  // first trick to reset kbd counter
  //  wait enough time for keyboard interrupt to be updated
  usleep(1*1000000);
  presstime=0L;
  timesincepress();

  // now by faking the presstime, we force execution
  presstime=delaytime+1;

  // reset the signal handler
  signal(SIGUSR1,invoke);
}









     

void killpid(int pid)
{
  // assasinate the pid passed
  kill (pid,SIGTERM);
  if (debugmode)
    {
      // debugmode announces and is paranoid
      printf("Manually killing target program because debug mode is paranoid.\r\n");
      usleep(3*1000000); // wait for child to die
      if (!deadchild)
	{
	  fprintf(stderr,"vtscreensaver debugmode error: couldn't kill child. exiting.\r\n");
	  kill (pid,SIGKILL);  // ...
	  exit(0);
	}
    }
}



void waitonkeyboard()
{
  // just sit here until user hits a key, check frequently
  while (1)
    {
      // sleep for 1/10 second
      usleep(100000);

      if (timesincepress()<delaytime)
	break;
      if (deadchild==1)
	break;
    }
}









int killcopies()
{
  // we want to kill any previously running versions of vtscreensaver
  // RETURN the number of copies killed
 int fd;
 struct dirent *d;
 DIR *pdir;
 char pname[80],fname[255];
 char string[255];
 int pnum;
 int ourid;
 FILE *fp;
 int numkilled=0;


 // first open the dir
 if ((pdir=opendir(PROCDIR))==NULL)
   {
     if (debugmode)
       printf("vtscreensaver warning: couldn't open %s to check for prev. running copies.\n",PROCDIR);
     return;
   }

 // get our own id
 ourid=getpid();

 // now check for proc files
 for (;;)
   {
     d=readdir(pdir);
     if (d==NULL)
       break;
     if (strlen(d->d_name)>=80)
       continue;
     strcpy(pname,d->d_name);
     if (pname[0]<'0' || pname[0]>'9')
       continue;
     // okay a candidate, now try to open its cmdline file
     sprintf(fname,"%s/%s/cmdline",PROCDIR,pname);
     if ((fp = fopen(fname, "r")) == NULL)
       continue;
     fgets(string,80,fp);
     if (strstr(string,PROGNAME)!=NULL)
       {
         // okay we found a copy
         pnum=atoi(pname);
         if (ourid==pnum)
           continue; // we don't want to commit suicide
         if (debugmode)
           printf("vtscreensaver found a running copy [pid%d] which is now being killed.\r\n",pnum);
         killpid(pnum);
	 ++numkilled;
       }
     fclose(fp);
   }

 closedir(pdir);
 return(numkilled);
}



int invokeold()
{
// send an invoke signal to a sleeping vtscreensaver process
 int fd;
 struct dirent *d;
 DIR *pdir;
 char pname[80],fname[255];
 char string[255];
 int pnum;
 int ourid;
 FILE *fp;
 int invoked=0;


 // first open the dir
 if ((pdir=opendir(PROCDIR))==NULL)
   {
     if (debugmode)
       printf("vtscreensaver warning: couldn't open %s to check for prev. running copies.\n",PROCDIR);
     return;
   }

 // get our own id
 ourid=getpid();

 // now check for proc files
 for (;;)
   {
     d=readdir(pdir);
     if (d==NULL)
       break;
     if (strlen(d->d_name)>=80)
       continue;
     strcpy(pname,d->d_name);
     if (pname[0]<'0' || pname[0]>'9')
       continue;
     // okay a candidate, now try to open its cmdline file
     sprintf(fname,"%s/%s/cmdline",PROCDIR,pname);
     if ((fp = fopen(fname, "r")) == NULL)
       continue;
     fgets(string,80,fp);
     fclose(fp);
     if (strstr(string,PROGNAME)!=NULL)
       {
         // okay we found a copy
         pnum=atoi(pname);
         if (ourid==pnum)
           continue; // we don't want to commit suicide
         if (debugmode)
           printf("vtscreensaver found a running copy [pid%d] which is now being invoked.\r\n",pnum);
         kill(pnum,SIGUSR1);
	 invoked=1;
         break;
       }
   }

 closedir(pdir);
 return(invoked);
}
