// particle, an svgalib display hack
// ver 0.01, 7/17/97 is (c)Jesse Reichler, 1997
//
// This program is
//  free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation.
//


#include <stdio.h>
#include <vga.h>
#include <vgagl.h>
#include <math.h>
#include <stdlib.h>




int usage()     
{
  printf("Particle can be run as:\n");
  printf("particle.exe initial_particles [path_length slowdown sunmass dotsize svgamode quitkeys]\n\n");
  printf("     initial_particles - the number of starting (and thus maximal) single\n");
  printf("                         unit mass particles. [NO DEFAULT, range=20-200]\r\n");
  printf("     path_length       - the length of the trail left by particles, which\r\n");
  printf("                         makes it easier to see. [default=200, range=1-4000]\r\n");
  printf("     slowdown          - as particle coallesce, the program runs faster\r\n");
  printf("                         because there are fewer calculations.  Making this\r\n");
  printf("                         larger balances this speed so that the system\r\n");
  printf("                         doesn't go too fast. [deafult=10, range=0-]\r\n");
  printf("     sunmass           - mass of the invisible sun [default=0, range=0-]\r\n");
  printf("     dotsize           - size of the particle displayed [default=auto, range=1-3]\r\n");
  printf("     svgamode          - explicit svgalib display mode to run in [default=auto]\r\n");
  printf("     quitkeys          - if 0, only escape quits; if 1, any key quits.  [default=1]\r\n\r\n");
  printf("An x can be used in place of a parameter to use the default value.\r\n\r\n");
  printf("Particle looks for the highest (256-color) resolution mode that your graphics\r\n");
  printf("card supports, and tries to use that mode.  This mode selection may be\r\n");
  printf("overriden by passing the svgamode parameter, or defining the environment\r\n");
  printf("variable GSVGAMODE to one of the svgalib modes, as defined in vga.h.\r\n");
  printf("i.e. setenv GSVGAMODE xxx, where xxx is (from vga.h):\r\n");
  printf("     45   (1600x1200x256)\r\n");
  printf("     13   (1280x1024x256)\r\n");
  printf("     12   (1024x768x256)\r\n");
  printf("     11   (800x600x256)\r\n");
  printf("     10   (640x480x256)\r\n\r\n");
  exit(1);
}







// limits
#define MAX_COLOR 255
#define MAX_PARTS 250
#define MAX_PATHLEN 4000




// screen dimensions
int dim_x;
int dim_y;
int small_x;
int small_y;
int kdimx;
int kdimy;


// default starting parameters
int startparticles=20;
int pathlen=200;
int slowdown=10;
int manualsvgamode=-1;


// some internal constants
float INIT_MOMENTUM=.05;
int INIT_MASS=1;
int MAXMASS=13;
float GRAV=0.1;
float xdiv=1.0;
float ydiv=1.0;
float TOOCLOSE=1.0;


// global vars we need
int pathoff=0;


// functions
long int myrandom(long mlim);
int picksvgalibmode(int);
int trymode(int);

// different size pixels
void mygl_setpixel1(int x,int y,int cl);
void mygl_setpixel2(int x,int y,int cl);
void mygl_setpixel3(int x,int y,int cl);

// global pixel function pointer
void (*drawparticle)(int,int,int)=mygl_setpixel1;




// Particle class

class particle
{
public:
  float x;
  float y;
  float x_momentum;
  float y_momentum;
  float mass;
  int tag;
  short int xpath[MAX_PATHLEN];
  short int ypath[MAX_PATHLEN];
  int ptype;
  
  particle (float, float, float, float, float, int,int);
  inline int gravize (float, float, float, int);
  inline void move ();
};



particle :: particle (float qy, float qx, float qy_momentum, float qx_momentum, float qmass, int qtag, int qtype)
{
  y = qy;
  x = qx;
  y_momentum = qy_momentum;
  x_momentum = qx_momentum;
  mass = qmass; 
  tag = qtag;
  for (int i=0;i<MAX_PATHLEN;++i)
    {
      xpath[i]=-1;
      ypath[i]=-1;
    }
  ptype=qtype;
}


inline int particle :: gravize (float fy, float fx, float fmass, int onlyonce)
{
  // Update momemtum calculations for a particle, and calculate collisions

  float xd,yd;
  float rdist;
  float rdistsq;
  float xo,yo;
  float newmass;
  float gfact;
  
  xd=x-fx;
  yd=y-fy;
  rdist=sqrt(pow(xd,2) + pow(yd,2));
  if (rdist<.001)
    {
      x=x+myrandom(5)-2;
      y=y+myrandom(5)-2;
    }
  else
    {
      // gravity calculation
      rdistsq=pow(rdist,2);
      gfact=(GRAV*mass*fmass)/rdistsq;
      xo=gfact*xd;
      yo=gfact*yd;
      x_momentum -= xo/xdiv;
      y_momentum -= yo/ydiv;
      
    }
  
  if (rdist<TOOCLOSE && onlyonce==1)
    {
      // collision
      newmass=mass+fmass;
      if (newmass<=MAXMASS)
	{
	  // delete it and grow the other
	  mass=newmass;
	  return(-1);
	}
      else
	{
	  // create a new one
	  x_momentum=0;
	  y_momentum=0;
	  return(1);
	}
    }
  return(0);
}




inline void particle :: move ()
{
  // Move a particle based on its previously calculated momentum
  int ix;
  int iy;
  int eraseoff;
  
  // erase end of trail
  eraseoff=(pathoff+1)%pathlen;
  ix=xpath[eraseoff];
  if (ix!=-1)
    {
      iy=ypath[eraseoff];
      drawparticle(ix,iy,0);
    }
  
  x += x_momentum;
  y += y_momentum;
  
  if (x > dim_x || x<small_x)
    {
      if (x> dim_x)
	x=dim_x-myrandom(6);
      if (x<small_x)
	x=small_x+myrandom(6);
      x_momentum=0;
    }
  if (y > dim_y || y<small_y)
    {
      if (y> dim_y)
	y=dim_y-myrandom(6);
      if (y<small_y)
	y=small_y+myrandom(6);
      y_momentum=0;
    }
  
  ix = (int)x;
  iy = (int)y;
  xpath[pathoff]=ix;
  ypath[pathoff]=iy;
  
  drawparticle(ix,iy,(tag%MAX_COLOR)+1);
}



















main (int argc,char **argv)
{
  particle **group;
  int mass;
  int onlyonce;
  int retv;
  int count,count2;
  float newmass;
  float xm,ym;
  int ix,iy;
  int ptype;
  int newparts;
  int pc;
  float yacc,xacc;
  int ky;
  int c;
  int x,y;
  int newtag;
  int xd2,yd2;
  int realparts;
  int tot_part=0;
  int i;
  long dummy4;
  int dotsize=-1;
  particle *sun;
  float sunmass=0.0;
  int quitkeys=1;
 

  // no parameters gets usage message
  if (argc==1)
    usage();


  // parse commandline switches
  if (argc>1)
    {
      if (strcmp(argv[1],"--help")==0 || strcmp(argv[1],"?")==0 || strcmp(argv[1],"-help")==0 || strcmp(argv[1],"-h")==0)
	usage();
      
      startparticles=atoi(argv[1]);
      if (startparticles > MAX_PARTS)
	startparticles=MAX_PARTS;
      if (startparticles<MAXMASS+1)
	startparticles=MAXMASS+1;
    }
  if (argc>2)
    {
      if (strcmp(argv[2],"x")!=0)
	{
	  pathlen=atoi(argv[2]);
	  if (pathlen>MAX_PATHLEN)
	    pathlen=MAX_PATHLEN;
	  if (pathlen<1)
	    pathlen=1;
	}
    }
  if (argc>3)
    {
      if (strcmp(argv[3],"x")!=0)
	{
	  slowdown=atoi(argv[3]);
	  if (slowdown<0)
	    slowdown=0;
	}
    }

  if (argc>4)
    {
      if (strcmp(argv[4],"x")!=0)
	  sunmass=atof(argv[4]);
    }

  if (argc>5)
    {
      if (strcmp(argv[5],"x")!=0)
	{
	  dotsize=atoi(argv[5]);
	  if (dotsize<1)
	    dotsize=1;
	  if (dotsize>3)
	    dotsize=3;
	}
    }

  if (argc>6)
    {
      if (strcmp(argv[6],"x")!=0)
	  manualsvgamode=atoi(argv[6]);
    }

  if (argc>7)
    {
      if (strcmp(argv[7],"x")!=0)
	  quitkeys=atoi(argv[7]);
    }



  // initialize svgalib
  vga_init();
  // now pick a resolution
  if (!picksvgalibmode(manualsvgamode))
    {
      // couldn't find a satisfactory one
      vga_setmode(TEXT);
      fprintf(stderr,"particle error: no satisfactory svgalib mode.\r\n");
      exit(0);
    }


  // set the particle drawing function pointer if manually specified dot size
  if (dotsize!=-1)
    {
      switch(dotsize)
	{
	case 1:
	  drawparticle=&mygl_setpixel1;
	  break;
	case 2:
	  drawparticle=&mygl_setpixel2;
	  break;
	case 3:
	  drawparticle=&mygl_setpixel3;
	  break;
	}
    }


  // safe random seed
  srand((unsigned int) (time((long *)0)%320000L));
  

  
  
  // make the particles
  group = new particle *[MAX_PARTS];
  for (count=0;count<MAX_PARTS;++count)
    group[count]=NULL;

  // resolution
  xd2=dim_x/2;
  yd2=dim_y/2;
  kdimx=(dim_x)+small_x;
  kdimy=(dim_y)+small_y;
  xacc=0; yacc=0;

  for (i = 0; i < startparticles; i++)
    {
      x=myrandom (dim_x);
      y=myrandom (dim_y);
      mass=INIT_MASS;
      if (INIT_MOMENTUM>0)
	{
	  xacc=((xd2-x)/1.0)*(double)INIT_MOMENTUM;
	  yacc=((yd2-y)/1.0)*(double)INIT_MOMENTUM;
	}
      group [i] = new particle (y, x, yacc, xacc, mass, 1,1);
      ++tot_part;
    }


  // create the sun
  if (sunmass!=0.0)
    sun=new particle(yd2,xd2,0.0,0.0,sunmass,254,1);


  // Now loop forever, moving the particles
  for (;;)
    {
      // kcount loop just decreases how frequently we scan keyboard by 1/3
      for (int kcount=1;kcount<3;++kcount)
	{
	  // now iterate through matrix of pairs
	  for (i = 0; i < tot_part; i++)
	    {
	      if (group[i]==NULL)
		continue;
	      for ( int j = 0; j < tot_part; j++)
		{
		  if (group[j]==NULL)
		    continue;
		  if (i == j) continue;

		  // set flag for who's responsible for dying, etc.
		  if (i<j)
		    onlyonce=1;
		  else
		    onlyonce=0;

		  // update effects of gravity
		  retv=group [i] -> gravize (group [j] -> y, group [j] -> x, group [j] -> mass,onlyonce);
		  
		  // now check for collision - coallesce
		  if (retv==-1)
		    {
		      // particle is destroyed, so we have to kill the entire path
		      for (count=0;count<pathlen;++count)
			{
			  ix=group[j]->xpath[count];
			  if (ix!=-1)
			    {
			      iy=group[j]->ypath[count];
			      drawparticle(ix,iy,0);
			    }
			}
		      delete group[j];
		      group[j]=NULL;
		    }

		  // now check for collision - explode
		  else if (retv==1)
		    {
		      // create new particles
		      newmass=group[i]->mass + group[j]->mass;

		      // first blow off the two main parents
		      newmass=newmass-2;
		      group[i]->mass=1;
		      group[i]->tag+=1;
		      group[j]->mass=1;
		      group[j]->tag+=1;
		      group[i]->x_momentum=0;
		      group[i]->y_momentum=0;
		      group[j]->x_momentum=0;
		      group[j]->y_momentum=0;
		      ptype=1;
		      if (group[i]->tag < group[j]->tag)
			newtag=group[i]->tag;
		      else
			newtag=group[j]->tag;
		      
		      // now blow off the rest of the mass
		      for (count2=0;count2<newmass;++count2)
			{
			  for (count=0;count<tot_part;++count)
			    {
			      if (group[count]==NULL)
				break;
			    }
			  if (count>tot_part)
			    ++tot_part;
			  
			  // create new particle
			  xm=0; ym=0;
			  group[count]= new particle (group[i]->y+myrandom(30)-15, group[i]->x+myrandom(30)-15, ym, xm, 1, (int)newtag,ptype);
			}
		    }
		}
	      if (sunmass!=0.0)
		retv=group [i] -> gravize (sun->y, sun->x, sun->mass ,0);

	    }
	  
	  // now draw updated particles, and count how many there really are
	  for (i = 0; i < tot_part; i++)
	    {
	      realparts=1;
	      if (group[i]!=NULL)
		{
		  ++realparts;
		  group [i] -> move ();
		}
	      
	    }

	  // update the global pathoff path tracking offset
	  pathoff=(pathoff+1)%pathlen;

	  // do any slowdown to account for speed increase with less particles
	  if (slowdown>0)
	    {
	      dummy4=0;
	      for (int dummy1=realparts+1;dummy1<startparticles;++dummy1)
		{
		  for (int dummy2=realparts+1;dummy2<startparticles;++dummy2)
		    {
		      for (int dummy3=1;dummy3<=slowdown;++dummy3)
			{
			  // just dummy calculations to slow down system
			  dummy4+=dummy1;
			  dummy4+=dummy2;
			}
		    }
		}
	    }

	}
      
      
      // test if the user wants to exit
      ky=vga_getkey();
      if (quitkeys && ky!=-1 && ky!=0)
	    break;
      if (ky==27)
	break;
    }

  // delete all particles
  delete [] group;

  // delete the sun
  if (sunmass!=0.0)
    delete sun;
  
  // put the screen back into text mode before we leave
  vga_setmode(TEXT);
  return 1;
}









int picksvgalibmode(int manualsvgamode)
{
  // first pick the best mode
  int vgamode;

  // FIRST try the commandline request
  if (manualsvgamode>0)
    {
      if (trymode(manualsvgamode)==1)
	return 1;
    }

  // svgalib documentation tells us the environment variable GSVGAMODE might
  //  contain a prefered mode, so next ask for it and see if it is good enough.
  vgamode=vga_getdefaultmode();
  if (trymode(vgamode)==1)
    return 1;


  // Otherwise pick the best one we know
  if (trymode(G1600x1200x256)==1)
    {
      drawparticle=mygl_setpixel2;
      return 1;
    }

  if (trymode(G1280x1024x256)==1)
    {
      drawparticle=mygl_setpixel2;
      return 1;
    }

  if (trymode(G1024x768x256)==1)
    return 1;

  if (trymode(G640x480x256)==1)
    return 1;

  if (trymode(G800x600x256)==1)
    return 1;

  if (trymode(G640x480x256)==1)
    return 1;

  return 0;
}


int trymode(int smnum)
{
  // See ifwe can use this mode
  // RETURN 1 if this avgalib mode is good enough
  //        0 if not
  vga_modeinfo *vi;
  int retv;

  retv=vga_hasmode(smnum);
  if (retv==0)
    return 0;

  vi=vga_getmodeinfo(smnum);
  if (vi==NULL)
    return 0;

  // make sure we have enough colors
  if (vi->colors<255)
    return(0);

  // now set mode
  vga_setmode(smnum);
  gl_setcontextvga(smnum);

  dim_x= (vi->width)-3;
  dim_y = (vi->height)-3;
  small_x=3;
  small_y=3;

  return 1;
}






long int myrandom(long mlim)
{
  // just return a random mod mlim
  return (random()%mlim);
}







void mygl_setpixel1(int x,int y,int cl)
{
  gl_setpixel(x,y,cl);
}


void mygl_setpixel2(int x,int y,int cl)
{
  gl_setpixel(x,y,cl);
  gl_setpixel(x-1,y,cl);
  gl_setpixel(x+1,y,cl);
  gl_setpixel(x,y-1,cl);
  gl_setpixel(x,y+1,cl);
}

void mygl_setpixel3(int x,int y,int cl)
{
  gl_setpixel(x,y,cl);
  
  gl_setpixel(x-1,y,cl);
  gl_setpixel(x+1,y,cl);
  gl_setpixel(x,y-1,cl);
  gl_setpixel(x,y+1,cl);
  
  gl_setpixel(x-2,y,cl);
  gl_setpixel(x+2,y,cl);
  gl_setpixel(x,y-2,cl);
  gl_setpixel(x,y+2,cl);
  
  gl_setpixel(x-1,y-1,cl);
  gl_setpixel(x+1,y+1,cl);
  gl_setpixel(x-1,y+1,cl);
  gl_setpixel(x+1,y-1,cl);
}
