/*-----------------------------------------------------------------------------
 |	termcolor - simple termcap/terminfo hacker to provide colors on std terms
 |
 |	This program allows one to make immediate changes to his/her termcap and
 |	terminfo files, changing the regular entries used by most programs to
 |	colorful counterparts. You can, for instance, redefine 'bold' to mean
 |	Bright Yellow on black, so that every program that uses bold (most do)
 |	will display Bright Yellow instead.
 |
 |	BUGS:
 |		There are some oversimplifications:
 |
 |		(1) Only the first terminal name is checked on /etc/termcap.
 |		    If you port this to another unix pay special attention
 |          to this, since many unices use a the terminal name in 
 |			the second field name of /etc/termcap.
 |		(2) Since my time is scarce, sequences as 'end-bold', 'end-underline'
 |			and such were grossly simplified as 'cease all attribs'. This
 |			will cause "stacked" attrib (eg: bold+standout) to behave
 |			incorrectly. In practice, I've never seen this situation.
 |		(3)	This is my first program using Lex, so, please be nice... :)
 |
 |	Author:
 |		Marco Paganini
 |		paganini@ism.com.br
 |		(Undernet IRC, as Lobsang)
 |
 |  If you  use  this program, send me some email. If you want to make 
 |  changes/additions to  this  program, send your changes to me and I 
 |  will incorporate then into the next version.
 +----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "termcolor.h"

void	main(int ac, char **av)
{
	char	bold     [COLOR_STR_LEN];
	char	reverse  [COLOR_STR_LEN];
	char	underline[COLOR_STR_LEN];
	char	standout [COLOR_STR_LEN];
	char	*term;
	int		ix;
	int		proceed = 0;
	
	bold[0] = reverse[0] = underline[0] = standout[0] = '\0';

	/** Check TERM var **/

	if ((term = getenv("TERM")) == NULL)
	{
		fprintf(stderr, "termcolor: The TERM variable must be set!\n");
		exit(1);
	}

	/** Parse command line **/

	for (ix = 1; ix < ac; ix++)
	{
		if (strcmp(av[ix], "-b") == 0)
		{
			strcpy(bold, av[++ix]);
			if (ansi_color(bold, 0) == NULL)
			{
				printf("termcolor: bad bold colorpair %s\n");
				exit(1);
			}
			proceed = 1;
		}
		else
		if (strcmp(av[ix], "-r") == 0)
		{
			strcpy(reverse, av[++ix]);
			if (ansi_color(reverse, 0) == NULL)
			{
				printf("termcolor: bad reverse colorpair %s\n");
				exit(1);
			}
			proceed = 1;
		}
		else
		if (strcmp(av[ix], "-u") == 0)
		{
			strcpy(underline, av[++ix]);
			if (ansi_color(underline, 0) == NULL)
			{
				printf("termcolor: bad underline colorpair %s\n");
				exit(1);
			}
			proceed = 1;
		}
		else
		if (strcmp(av[ix], "-s") == 0)
		{
			strcpy(standout, av[++ix]);
			if (ansi_color(standout, 0) == NULL)
			{
				printf("termcolor: bad standout colorpair %s\n");
				exit(1);
			}
			proceed = 1;
		}
		else
		{
			usage();
			exit(1);
		}
	}

	if (!proceed)
	{
		usage();
		exit(1);
	}

	/** Do the changes **/

	if (change_termcap(TERMCAP, term, bold, reverse, underline, standout) != 0)
		exit(-1);

	if (change_terminfo(term, bold, reverse, underline, standout) != 0)
		exit(-1);

	exit(0);
}

/*---------------------------------------------------------------------------
 | ansi_color(color_pair, mode)
 |
 |	This function will get a color pair in the form "color:color" and return
 |  a string containg the ansi sequence to generate that color (suitable for
 |	termcap/terminfo).
 |
 |	mode: 0 = Termcap style output
 |        1 = Terminfo style output
 |
 |	NULL is returned if the function cannot determine the color sequence
 |  (usually, invalid colors).
 +-----------------------------------------------------------------------*/

char	*ansi_color(char *colorpair, int mode)
{
	static	char	cpair[80];
 	char			temp_colorpair[80];

	static	ANSI_COLORS	ansi_table[] = 
	{
		{ "black", 		30, 40 },
		{ "red",   		31, 41 },
		{ "green", 		32, 42 },
		{ "yellow",		33, 43 },
		{ "blue",  		34, 44 },
		{ "magenta",	35,	45 },
		{ "cyan",		36, 46 },
		{ "white",		37, 47 },
		{ "",			-1, -1 }
	};

	int		ix;
	int		hifore = 0;
	char	*pstr;
	char	fore[20];
	char    back[20];

	/** Break the colors from fore:back and check hilite **/

	strcpy(temp_colorpair, colorpair);	/** Don't mess original **/

	pstr = strtok(temp_colorpair, ":");

	if (pstr == NULL)
		return NULL;

	if (strncmp(pstr, "hi", 2) == 0)
	{
		hifore = 1;
		strcpy(fore, pstr + 2);
	}
	else
		strcpy(fore, pstr);

	pstr = strtok(NULL, ":");
	if (pstr == NULL)
		return NULL;

	strcpy(back, pstr);

	/** Now we will generate the string **/

	strcpy(cpair, (mode == 0 ? TCANSI_COLOR_PREFIX : TIANSI_COLOR_PREFIX));

	if (hifore)
		strcat(cpair, ANSI_HILITE);

	/** Foreground **/

	for (ix = 0; ansi_table[ix].fore != -1; ix++)
	{
		if (strcmp(fore, ansi_table[ix].color_name) == 0)
		{
			sprintf(cpair + strlen(cpair), "%d;", ansi_table[ix].fore);
			break;
		}
	}

	if (ansi_table[ix].fore == -1)		/** No match **/
		return NULL;

	/** Background **/

	for (ix = 0; ansi_table[ix].fore != -1; ix++)
	{
		if (strcmp(back, ansi_table[ix].color_name) == 0)
		{
			sprintf(cpair + strlen(cpair), "%d", ansi_table[ix].back);
			break;
		}
	}

	if (ansi_table[ix].fore == -1)		/** No match **/
		return NULL;

	/** End the spec **/

	strcat(cpair, ANSI_COLOR_SUFFIX);

	return(cpair);
}

/*----------------------------------------------------------------------------*/

void	usage(void)
{
	printf("Termcolor - Termcap/terminfo hacker\n");
	printf("V%s - Nov/1995 by Marco Paganini (paganini@ism.com.br)\n", VERS);
	printf("\nUsage: termcolor -b colorpair -r colorpair -u colorpair -s colorpair\n\n");
	printf("Where:\n");
	printf("  -b: bold color\n");
	printf("  -r: reverse color\n");
	printf("  -u: underline color\n");
	printf("  -s: standout color\n");
	printf("\nColorpairs:\n");
	printf("  Colorpairs can be given in the form [hi]forecolor:backcolor\n");
	printf("  Prefix foreground colors with 'hi' for hilite.\n");
	printf("\nExamples:\n");
	printf("  termcolor -b white:blue -r black:green -u hiyellow:black\n");
	printf("  termcolor -s hiblue:black\n");
}

/** EOF **/
