/*
 *       open.c open a vt to run a new command (or shell).
 *       
 *	 Copyright (c) 1994 by Jon Tombs <jon@gtex02.us.es>
 *
 *       This program is free software; you can redistribute it and/or
 *       modify it under the terms of the GNU General Public License
 *       as published by the Free Software Foundation; either version
 *       2 of the License, or (at your option) any later version.
 *
 * 4 May 1995 (tjd): 	fixed setuid, added setgid checking, and
 * 			fixed some random bugs.
 */

#include "open.h"
#ifdef TTY
#include <grp.h>
#include <pwd.h>
#endif

const char *OPENversion = "open: 1.4 (c) Jon Tombs 1995";

#ifndef VTNAME
#error vt device name must be defined in open.h
#endif

int 
main(int argc, char *argv[])
{
   int fd = 0;
   int opt;
   pid_t pid;
   struct vt_stat vt;
   int vtno     = -1;
   char show    = FALSE;
   char login   = FALSE;
   char verbose = FALSE;
   char do_wait	= FALSE;
   char vtname[sizeof VTNAME + 2]; /* allow 999 possible VTs */
   char *cmd, *def_cmd = NULL;
   uid_t euid,ruid;
   gid_t rgid;

#ifdef TTY
   uid_t tty_uid=-1;
   gid_t tty_gid=-1;
   struct group *ttyg;
   struct passwd *ttyn;
#endif

   euid=geteuid();
   ruid=getuid();
   rgid=getgid();

   /* safety first... */
   seteuid(ruid);

   /* no need to be running sgid, make sure we're not! */
   setgid(rgid);

   /*
    * I don't like using getopt for this, but otherwise this gets messy.
    * POSIX/Gnu getopt forces the use of -- to separate child/program
    * options. RTFM.
    */
   while ((opt = getopt(argc, argv, "c:lsvw")) != -1) {
      switch (opt) {
	case 'c':
	  vtno = (int) atol(optarg);
	  if (vtno <= 0 ||
		vtno > (!ruid ? MAX_NR_CONSOLES : MAX_NR_USER_CONSOLES)) {
	    fprintf(stderr, "open: %s illegal vt number\n", optarg); 
	    return 5;
	  }
	  break;
	case 'l':
	  login = TRUE;
	  break;
	case 's':
	  show = TRUE;
	  break;
	case 'v':
	  verbose = TRUE;
	  break;	    	    
	case 'w':
	  do_wait = TRUE;
	  break;
	default:
	  usage(1);
	
      }
   }

   if (!(argc > optind)) {
      def_cmd = getenv("SHELL");
      if (def_cmd == NULL)
	usage(0);
   }

   /* if we're not root, make sure we're on the console */
   if(ruid)
   {
	if ((fd = open("/dev/tty",O_WRONLY,0)) < 0) {
		perror("open: Can't open /dev/tty\n");
		return(3);
	}
   
	if(ioctl(fd,VT_GETSTATE,&vt) < 0)
	{
	  fprintf(stderr,"This program may only be run on the console.\n");
	  close(fd);
          exit(1);
	}
    }
    else
    {
	if ((fd = open("/dev/console",O_WRONLY,0)) < 0) {
		perror("open: Failed to open /dev/console");	
		return(2);
	}
    }

   if (vtno == -1) {

     if ((ioctl(fd, VT_OPENQRY, &vtno) < 0) || (vtno == -1)) {
        perror("open: Cannot find a free VT");
        close(fd);
        return(3);
     }

     if (ioctl(fd, VT_GETSTATE, &vt) < 0) {
	perror("open: can't get VTstate");
        close(fd);
        return(4);
     }
   }

   sprintf(vtname, VTNAME, vtno);

#ifdef TTY
   /* try to fix the tty modes if running suid root */

   if(!euid)
   {
	if((ttyn=getpwnam(TTYUSER)))
	{
		tty_uid=ttyn->pw_uid;
	}
	else
	{
		char buf[256];
		sprintf(buf,"Warning: can't get uid for '%s'",TTYUSER);
		perror(buf);
		tty_uid=-1;
	}

	if((ttyg=getgrnam(TTY)))
	{
		tty_gid=ttyg->gr_gid;
	}
	else
	{
		char buf[256];
		sprintf(buf,"Warning: can't get gid for '%s'",TTY);
		perror(buf);
		tty_gid=-1;
	}
	
	seteuid(euid);
	chown(vtname,ruid,tty_gid);

	if(tty_gid != (gid_t) -1)
		chmod(vtname,TTYMODE);

	/* drop permissions again.  keep saved-id in case of wait. */
	seteuid(ruid);
   }
#endif

   /* We assume getty has made any in use VT non accessable */
   if (access(vtname, R_OK | W_OK) < 0) {
      fprintf(stderr, "open: Failed to open %s read/write (%s)\n", vtname,
	     strerror(errno));
      return (5);
   }

   if (verbose)
	fprintf(stderr,	"open: using VT %s\n", vtname);
	
   cmd = malloc(strlen(argv[optind] + 2));

   if (login)
      strcpy(cmd, "-");
   else
      cmd[0] = '\0';

   if (def_cmd)
      strcat(cmd, def_cmd);
   else
   {
      strcat(cmd, argv[optind]);
      if (login) argv[optind] = cmd++;
   }
   	
   if((pid=fork()) == 0) {
      /* be paranoid */
      setuid(ruid);

      /* leave current vt */
#ifdef   ESIX_5_3_2_D
      if (setpgrp() < 0) {
#else
      if (setsid() < 0) {
#endif      
        fprintf(stderr, "open: Unable to set new session (%s)\n",
	strerror(errno));
      }
      close(0);
      close(1);
      close(2);
      close(fd);	 

      /* and grab new one */
      if ((fd = open(vtname, O_RDWR)) == -1) { /* Shouldn't happen */
        _exit (4); /* silently die */
      }
      dup(fd); dup(fd);

      if (show) {
	/* 
         * Can't tell anyone if any of these fail, so throw away
	 * the return values 
         */
        (void) ioctl(fd, VT_ACTIVATE, vtno);
        /* wait to be really sure we have switched */
	(void) ioctl(fd, VT_WAITACTIVE, vtno);
      }
      if (def_cmd)
         execlp(def_cmd, cmd, NULL);
      else
	 execvp(cmd, &argv[optind]);
   }
   if ( pid < 0 ) {
      perror("open: fork() error");
      return(6);
   }


   if ( do_wait ) {
      wait(NULL);
      if (show) { /* Switch back... */
	 (void) ioctl(fd, VT_ACTIVATE, vt.v_active);
	 /* wait to be really sure we have switched */
	 (void) ioctl(fd, VT_WAITACTIVE, vt.v_active);
      }
#ifdef TTY
      /* restore owners/permissions on tty */
      if(!euid)
      {
	seteuid(euid);
	chown(vtname,tty_uid,tty_gid);
	if(tty_gid != (gid_t) -1)
		chmod(vtname,TTYMODE);
	setuid(ruid);
      }
#endif
   }

   close (fd);
   return 0;
}
      

void usage(int stat)
{
   fprintf(stderr,
      "Usage: open [-c vtnumer ][-l] [-s] [-v] -- command_line\n");
   exit (stat);
}


