#include<stdio.h>
#include<sys/types.h>
#include<dirent.h>
#include<unistd.h>
#include<string.h>
#include<fcntl.h>
#include<sys/ioctl.h>
#include<linux/vt.h>
#include<sys/wait.h>
#include<signal.h>
#include<sys/stat.h>
#include<malloc.h>
#include<stdlib.h>
#include<linux/kd.h>

#define MAXCONS 64 /* this is indeed max + 1 as devices begin with 1 */
#define PIDFILE "/var/run/consd.pid"

/* default arguments for various gettys - %d is replaced by tty number */
char default_args[][100]=
{ "/sbin/agetty 38400 tty%d linux",
  "/sbin/getty_ps tty%d 38400",
  
  /* add your favorite gettys here */
   
  /* list must be terminated with empty string */
  ""
};

int pid_list[MAXCONS];
int debug=0;
int finish=0;
int maxcons=MAXCONS-1;
char getty_args[1024]="";
int telep=0;
int bindkey=0;
int aggressive=0;

int find_tty(int pid)
{ int i;

  for(i=0;i<MAXCONS;++i)
  { if(pid_list[i]==pid)return i;
  }
  
  return -1;
}

void signal_handler(int a)
{ if(a==SIGTERM||a==SIGINT)finish=1;
  if(a==SIGHUP)telep=1;
  signal(SIGHUP,signal_handler);
}

int do_bindkey(void)
{ int fd;

  fd=open("/dev/tty0",O_RDWR);
  if(fd<0)
  { perror("open /dev/tty0 failed");
    return -1;
  }

  ioctl(fd,KDSIGACCEPT,SIGHUP);
  close(fd);
  return 0;
}

int check_getty(int pid)
{ char buf[1024];
  FILE*f;
  int i=0;
  
  sprintf(buf,"/proc/%d/stat",pid);
  f=fopen(buf,"r");
  if(f==NULL)return -1;
  fscanf(f,"%d %s",&i,buf);
  fclose(f);
  if(strstr(buf,"getty")==NULL)return -1;
  
  return 1;
}

int fg_console(void)
{ int fd;
  struct vt_stat v;
  
  fd=open("/dev/tty0",O_RDWR);
  if(fd<0)
  { if(debug)perror("open /dev/tty0 failed");
    return -1;
  }
  
  ioctl(fd,VT_GETSTATE,&v);
  close(fd);
  
  return v.v_active;
}

int find_getty_below_vt(int vt)
{ int i;

  for(i=vt-1;i>=0;--i)
  { if(pid_list[i]>0&&check_getty(pid_list[i])>0)return i;
  }
  
  return -1;
}

int find_getty_above_vt(int vt)
{ int i;

  for(i=MAXCONS-1;i>vt;--i)
  { if(pid_list[i]>0&&check_getty(pid_list[i])>0)return i;
  }
  
  return -1;
}

int write_message(char*text,int tty)
{ char buf[20];
  int fd;
  
  sprintf(buf,"/dev/tty%d",tty);
  fd=open(buf,O_WRONLY);
  if(fd<0)
  { if(debug)perror("write_message: open failed");
    return -1;
  }
  
  write(fd,text,strlen(text));
  close(fd);
  
  return 0;
}

int count_gettys(int*minvt)
{ DIR*d;
  FILE*f;
  int count=0;
  int i;
  char buf[1024];
  struct dirent*de;
  int tty_dev=0;
  char dummy=0;
  int m=1000;
  
  d=opendir("/proc");
  if(d==NULL)
  { if(debug)fprintf(stderr,"consd: can't read /proc");
    return -1;
  }
  
  while((de=readdir(d))!=NULL)
  { sprintf(buf,"/proc/%s/stat",de->d_name);
    f=fopen(buf,"r");
    if(f==NULL)continue;
    fscanf(f,"%d %s %c %d %d %d %d",&i,buf,&dummy,&i,&i,&i,&tty_dev);
    fclose(f);
    /* only count gettys on devices 4,1 to 4,64 */
    if(strstr(buf,"getty")&&tty_dev>1024&&tty_dev<1024+MAXCONS)
    { count++;
      if(m>tty_dev-1024)m=tty_dev-1024;    
    }
  }
  
  closedir(d);
  
  if(minvt)*minvt=m;
  return count;
}

void insert_tty_in_arg(char*arg,int tty)
{ char buf[1024];
  
  /* safety check: ensure string in arg does not become longer */
  if(tty<0||tty>99)return;
  
  while(strstr(arg,"%d"))
  { sprintf(buf,arg,tty);
    strcpy(arg,buf);
  }
}

char** setup_getty_args(int tty)
{ char buf[1024];
  char**args;
  int i;
  char*p;

  strcpy(buf,getty_args);
  insert_tty_in_arg(buf,tty);
  
  args=malloc(100*sizeof(char*));
  if(args==NULL)return NULL;
  
  i=0;
  for(p=strtok(buf," ");p;p=strtok(NULL," "))
  { args[i]=malloc(1024);
    if(args[i])strcpy(args[i],p);
    ++i;
    if(i==99)break;
  }
  args[i]=NULL;
  
  return args;
}

int new_getty(int vt)
{ int i;
  char**args;

  if(debug)fprintf(stderr,"starting new getty on vt %d\n",vt);
  
  if(debug)write_message("vt opened by consd\n",vt);
  
  i=fork();
  if(i==0)
  { i=setsid();
    if(i<0)
    { if(debug)perror("setsid failed");
      exit(-1);
    }
    
    args=setup_getty_args(vt);
    if(args==NULL||args[0]==NULL)
    { if(debug)fprintf(stderr,"setup_getty_args failed\n");
      exit(-1);
    }
    if(debug)
    { char**p=args;
    
      fprintf(stderr,"executing: ");
      while(*p)
      { fprintf(stderr,"%s ",*p);
        ++p;
      }
      fprintf(stderr,"\n");
    }
    execv(args[0],args);
    
    if(debug)perror("exec failed");
    exit(-1);
  }
  sleep(1);
  if(i<0)return i;
  
  if(waitpid(i,NULL,WNOHANG)==i)
  { write_message("\nwarning: getty exited immediately\n",vt);
    if(debug)fprintf(stderr,"warning: getty exited immediately\n");
    return -1;
  }
    
  pid_list[vt]=i;
  return i;
}

int deallocvt(int i)
{ int fd;

  fd=open("/dev/tty0",O_RDWR);
  if(fd<0)
  { if(debug)perror("open /dev/tty0 failed");
    return -1;
  }
  
  i=ioctl(fd,VT_DISALLOCATE,i);
  close(fd);
  
  if(i<0)
  { if(debug)perror("ioctl VT_DISALLOCATE failed\n");
  }
  
  return i;
}

int newvt(void)
{ int fd;
  int i;
  
  fd=open("/dev/tty0",O_RDWR);
  if(fd<0)
  { if(debug)perror("open /dev/tty0 failed");
    return -1;
  }
  
  i=0;
  ioctl(fd,VT_DISALLOCATE,0);
  ioctl(fd,VT_OPENQRY,&i);
  close(fd);
  if(i<=0)return -1;
  
  return i;
}

void kill_getty(int vt)
{ if(debug)fprintf(stderr,"killing getty on vt %d\n",vt);
  kill(pid_list[vt],SIGTERM);
  waitpid(pid_list[vt],NULL,0);
  pid_list[vt]=0;
  if(debug)fprintf(stderr,"getty on vt %d killed\n",vt);
  write_message("\nvt closed by consd\n",vt);
  deallocvt(vt);
}

int teleport_to_free(void)
{ int tovt;
  int fd;
  char buf[2]="\7";
  
  fd=open("/dev/tty0",O_RDWR);
  if(fd<0)
  { if(debug)perror("open /dev/tty0 failed");
    return -1;
  }
  
  if(count_gettys(&tovt)>0)
  { ioctl(fd,VT_ACTIVATE,tovt);
    close(fd);
    return 0;                  
  }
  
  write(fd,buf,1);
  close(fd);
  
  return -1;
}

void checkvt(void)
{ int i;
  int nr_gettys;

  nr_gettys=count_gettys(NULL);
  if(debug)fprintf(stderr,"checkvt: %d gettys counted\n",nr_gettys);
  if(nr_gettys==0)
  { i=newvt();
    if(i>maxcons)
    { if(debug)fprintf(stderr,"checkvt: maxcons reached\n");
    }
    else if(i>=0)
    { if(debug)fprintf(stderr,"checkvt: opening new vt\n");
      new_getty(i);
    }
    else
    { if(debug)fprintf(stderr,"checkvt: newvt failed\n");
    }
  }
  if(nr_gettys>1)
  { i=find_getty_above_vt(0);
    if(i>0)
    { if(debug)fprintf(stderr,"unnecessary waiting getty found on vt %d\n",i);
      if(i==fg_console())
      { if(debug)fprintf(stderr,"not killing it because it's on foreground console\n");
      }
      else kill_getty(i);
    }
  }
  if(nr_gettys==0&&aggressive!=0)
  { deallocvt(0);
  }
  if(nr_gettys>0&&aggressive!=0)
  { i=newvt();
    if(i>0)
    { if(find_getty_above_vt(i)>0)
      { if(debug)fprintf(stderr,"unused console %d in the middle found, starting getty there\n",i);
        new_getty(i);
      }
    }
  }
  if(debug)fprintf(stderr,"checkvt: end\n");
}

void pidfile(void)
{ struct stat buf;
  int pid=0;
  char str[100];
  FILE*f;
  
  if(stat(PIDFILE,&buf)>=0)
  { if(debug)fprintf(stderr,"pidfile exists");
    f=fopen(PIDFILE,"r");
    if(f)
    { fscanf(f,"%d",&pid);
      fclose(f);
      sprintf(str,"/proc/%d/stat",pid);
      if(pid!=getpid()&&stat(str,&buf)>=0)
      { fprintf(stderr,"consd already running\n");
        exit(1);
      }
    }
    unlink(PIDFILE);
  }
  
  f=fopen(PIDFILE,"w");
  if(f==NULL)
  { fprintf(stderr,"cannot write pidfile %s\n",PIDFILE);
    exit(1);
  }
  fprintf(f,"%d\n",getpid());
  fclose(f);
}

int does_getty_exist(char*args)
{ char buf[1024];
  int i;
  struct stat sbuf;
  
  for(i=0;i<1024;++i)
  { if(args[i]=='\0')break;
    if(args[i]==' ')break;
    buf[i]=args[i];
  }
  buf[i]='\0';
  
  if(stat(buf,&sbuf)<0)return 0;
  return 1;
}

int replacen(char*text)
{ int i;
  char*p;
  char*q1;
  char*q2;

  if(debug)fprintf(stderr,"replacen: analysing %s\n",text);
    
  i=0;
  while((p=strstr(text,"tty"))!=NULL)
  { if(debug)fprintf(stderr,"replacen: candidate found: %s\n",p);
    if(p[3]<'1'||p[3]>'9')goto rx;
    if(p[4]<'0'||p[4]>'9')
    { /* need to shift string */
      q1=p+strlen(p); /* -> '\0' */
      q2=q1+1;
      while(q1>p+3)*q2--=*q1--;     
    }
    else if(p[5]>='0'||p[5]<='9')goto rx;
    p[3]='%';
    p[4]='d';
    ++i;
   rx:
    text=p+1;
  }
  
  return i;
}

void guess_from_inittab(void)
{ FILE*f;
  char buf[1024];
  char*p;
  int i;
  
  if(debug)fprintf(stderr,"trying to guess from /etc/inittab...\n");
  f=fopen("/etc/inittab","r");
  if(f==NULL)
  { if(debug)perror("/etc/inittab");
    return;
  }
  while(fgets(buf,1024,f))
  { if(*buf=='\0'||*buf=='#')continue;
    i=strlen(buf);
    if(buf[i-1]=='\n')buf[i-1]='\0';
      
    if(strstr(buf,"getty"))
    { if(debug)fprintf(stderr,"found line with getty: %s\n",buf);
      p=strrchr(buf,':');
      if(p==NULL)
      { if(debug)fprintf(stderr,"line has no colon, ignored\n");
        continue;
      }
      ++p;
      if(!does_getty_exist(p))
      { if(debug)fprintf(stderr,"getty %s does not exist\n",p);
        continue;
      }
      if(replacen(p)!=0)
      { fclose(f);
        strcpy(getty_args,p);
        if(debug)fprintf(stderr,"getty command line guessed from inittab: %s\n",getty_args);
        return;
      }
      if(debug)fprintf(stderr,"line does not contain tty[1-9] or tty[1-9][0-9]\n");
    }
  }
  fclose(f);
}

void check_getty_args(void)
{ int i;

  if(strlen(getty_args)==0)
  { if(debug)fprintf(stderr,"no getty in command line, guessing...\n");
    guess_from_inittab();
    if(does_getty_exist(getty_args))return;
    for(i=0;default_args[i][0]!='\0';++i)
    { strcpy(getty_args,default_args[i]);
      if(debug)fprintf(stderr,"trying %s...\n",getty_args);
      if(does_getty_exist(getty_args))return;
    }
    fprintf(stderr,"No default getty found. Please specify getty path and arguments explicitely.\n");
    exit(1);
  }
  if(does_getty_exist(getty_args))return;
  fprintf(stderr,"Program does not exist\n");
  exit(1);
}

void usage(void)
{ fprintf(stderr,"consd 1.5.8 (C) 1998-2009 by Frank Gockel, Martin Berentsen\n");
  fprintf(stderr,"usage: consd [-d] [-h] [-a] [-b] [-m max] [-p getty args ...]\n\n");
  fprintf(stderr,"    -d: turn on debugging\n");
  fprintf(stderr,"    -h: print this help\n");
  fprintf(stderr,"    -a: be more aggressive on unused consoles\n");
  fprintf(stderr,"    -b: bind to spawn_console key\n");
  fprintf(stderr,"    -m: set maximum console number\n");
  fprintf(stderr,"    -p: everything behind this is used as getty command line (%%d is replaced\n");
  fprintf(stderr,"        with the tty number). There are no further options valid behind -p.\n");
}

void checkmax(void)
{ int i;
  struct stat sbuf;
  char buf[20];
  
  for(i=1;i<=maxcons;++i)
  { sprintf(buf,"/dev/tty%d",i);
    if(stat(buf,&sbuf)<0)
    { if(i<2)
      { fprintf(stderr,"There's something wrong here - can't find %s\n",buf);
        exit(1);
      }
      --i;
      if(maxcons>i)
      { maxcons=i;
        fprintf(stderr,"You have only consoles up to tty%d in /dev. I won't use more.\n",i);
        break;
      }
    }
  }
}

int main(int argc, char*argv[])
{ int i,t,n;

  if(getuid())
  { fprintf(stderr,"You are not root. Sorry.\n");
    exit(1);
  }

  i=1;
  while(i<argc)
  { if(strcmp(argv[i],"-d")==0)
    { debug=1;
    }
    else if(strcmp(argv[i],"-h")==0)
    { usage();
      exit(1);
    }
    else if(strcmp(argv[i],"-a")==0)
    { aggressive=1;
    }
    else if(strcmp(argv[i],"-b")==0)
    { bindkey=1;
    }
    else if(strcmp(argv[i],"-m")==0)
    { ++i;
      if(argv[i]==NULL)
      { 
       m_err:
        fprintf(stderr,"option -m requires an integer argument [1-%d]\n",
                MAXCONS-1);
        exit(1);
      }
      maxcons=atoi(argv[i]);
      if(maxcons<1||maxcons>=MAXCONS)goto m_err;
    }
    else if(strcmp(argv[i],"-p")==0)
    { ++i;
      if(argv[i]==NULL)
      { fprintf(stderr,"option -p requires at least one argument\n");
        exit(1);
      }
      strcpy(getty_args,argv[i]);
      while(argv[++i])
      { strcat(getty_args," ");
        strcat(getty_args,argv[i]);
      }
      break;
    }
    else
    { fprintf(stderr,"unknown option %s\n",argv[i]);
      exit(1);
    }
    ++i;
  }

  check_getty_args();
  if(debug)fprintf(stderr,"getty args=%s\n",getty_args);
  checkmax();
  
  if(debug==0)
  { if(fork())exit(0);
  }

  pidfile();
  
  for(i=0;i<MAXCONS;++i)pid_list[i]=0;
  
  signal(SIGCHLD,signal_handler);
  signal(SIGTERM,signal_handler);
  signal(SIGINT,signal_handler);
  signal(SIGHUP,signal_handler);

  /* make sure gettys from init are already up when consd takes over
     the console management :) If missing, it may cause strange errors,
     but no real problems... This is for people starting consd from bootup
     scripts. */
  sleep(10);
  
  if(bindkey)do_bindkey();
    
  while(!finish)
  { checkvt();
    sleep(10);
    
    while((i=waitpid(-1,NULL,WNOHANG))>0)
    { t=find_tty(i);
      if(t>=0)
      { if(debug)fprintf(stderr,"process %d on tty %d died\n",i,t);
        pid_list[t]=0;
        n=newvt();
        if(debug)fprintf(stderr,"newvt returns %d\n",n);
        if(find_getty_below_vt(t)>0)
        { if(t==fg_console())
          { if(debug)fprintf(stderr,"vt %d is foreground console, restarting\n",t);
            new_getty(t);
          }
          else
          { if(debug)fprintf(stderr,"vt %d is not foreground and getty on vt below found, not restarting\n",t);
          }
        }
        else
        { if(debug)fprintf(stderr,"no waiting getty on vt below %d found, restarting\n",t);
          new_getty(t);
        }
        i=find_getty_above_vt(t);
        if(i>0)
        { if(debug)fprintf(stderr,"getty on vt %d, above %d, found\n",i,t);
          if(i==fg_console())
          { if(debug)fprintf(stderr,"not killing it off because it's on foreground console\n");
          }
          else kill_getty(i);
        }
      }
    }
    
    if(telep)
    { telep=0;
      teleport_to_free();
    }
    
    signal(SIGCHLD,signal_handler);
    signal(SIGHUP,signal_handler);

  }
  
  if(debug)fprintf(stderr,"consd exiting\n");
  
  for(i=0;i<MAXCONS;++i)
  { if(pid_list[i]>0&&check_getty(pid_list[i])>0)
    { kill_getty(i);
    }
  }
  
  unlink(PIDFILE);
  return 0;
}
