/*
Some of this code was extracted straight out of XFree86.

This contains several files:
README.RIVA128        - This file
ClockProg.RIVA128.c   - Source for program to get/set the pixel clocks.
clockselect.c         - C file containing all the nessesary code to switch 
                        clocks. Intended to be dropped into SVGATextMode.

This code does set the pixel clock for the RIVA128. It has a programmable
clock that can basically be any requested value to within 1Mhz, most of the 
time within 0.05Mhz of the requested value.

The only problem I've found so far is that it seems to _overwrite_ the
default clocks. ie. If you're in mode 80x25, then setting the pixel clock
with this program will _change_ that clock so that if you try to switch back
to that mode with the VGA clocks, it won't work because the clock has changed.

Other than the clock setting, this video card seems to work just like an 
ET6000. Any code that it uses to set the screen width seems to works fine.

===============================================================================
Here is the bit from nVidia, extracted out XFree86 source:
It contains the source code licence for the extracted code.
*/

 /***************************************************************************\
|*                                                                           *|
|*        Copyright (c) 1996-1998 NVIDIA, Corp.  All rights reserved.        *|
|*                                                                           *|
|*     NOTICE TO USER:   The source code  is copyrighted under  U.S. and     *|
|*     international laws.   NVIDIA, Corp. of Sunnyvale, California owns     *|
|*     the copyright  and as design patents  pending  on the design  and     *|
|*     interface  of the NV chips.   Users and possessors of this source     *|
|*     code are hereby granted  a nonexclusive,  royalty-free  copyright     *|
|*     and  design  patent license  to use this code  in individual  and     *|
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include,  in the user documenta-     *|
|*     tion and  internal comments to the code,  notices to the end user     *|
|*     as follows:                                                           *|
|*                                                                           *|
|*     Copyright (c) 1996-1998  NVIDIA, Corp.    NVIDIA  design  patents     *|
|*     pending in the U.S. and foreign countries.                            *|
|*                                                                           *|
|*     NVIDIA, CORP.  MAKES  NO REPRESENTATION ABOUT  THE SUITABILITY OF     *|
|*     THIS SOURCE CODE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT     *|
|*     EXPRESS OR IMPLIED WARRANTY OF ANY KIND.  NVIDIA, CORP. DISCLAIMS     *|
|*     ALL WARRANTIES  WITH REGARD  TO THIS SOURCE CODE,  INCLUDING  ALL     *|
|*     IMPLIED   WARRANTIES  OF  MERCHANTABILITY  AND   FITNESS   FOR  A     *|
|*     PARTICULAR  PURPOSE.   IN NO EVENT SHALL NVIDIA, CORP.  BE LIABLE     *|
|*     FOR ANY SPECIAL, INDIRECT, INCIDENTAL,  OR CONSEQUENTIAL DAMAGES,     *|
|*     OR ANY DAMAGES  WHATSOEVER  RESULTING  FROM LOSS OF USE,  DATA OR     *|
|*     PROFITS,  WHETHER IN AN ACTION  OF CONTRACT,  NEGLIGENCE OR OTHER     *|
|*     TORTIOUS ACTION, ARISING OUT  OF OR IN CONNECTION WITH THE USE OR     *|
|*     PERFORMANCE OF THIS SOURCE CODE.                                      *|
|*                                                                           *|
 \***************************************************************************/

#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <fcntl.h>
#include "messages.h"
#include "vga_prg.h"

#ifdef DOS
#include <dpmi.h>
#endif

/* PCI stuff -- copied largely from the MGAProbe() code */
#include "include/Xmd.h"
#include "vgaPCI.h"

#define PCI_VENDOR_NVIDIA_SGS   0x12d2
#define PCI_CHIP_RIVA128        0x0018
#define PCI_VENDOR_ID_NVIDIA           0x10de
#define PCI_DEVICE_ID_NVIDIA_TNT       0x0020
#define PCI_DEVICE_ID_NVIDIA_TNT2      0x0028
#define PCI_DEVICE_ID_NVIDIA_UTNT2     0x0029
#define PCI_DEVICE_ID_NVIDIA_VTNT2     0x002C
#define PCI_DEVICE_ID_NVIDIA_UVTNT2    0x002D
#define PCI_DEVICE_ID_NVIDIA_ITNT2     0x00A0

vgaPCIInformation *vgaPCIInfo;

//=== Constant section ===
//============ Most of these constants come from the NV3 Xserver (XF86_SVGA) =====================
#define PRAMDAC_BASE       0x00680000
#define PRAMDAC_PLL_COEFF  0x00000508
#define PRAMDAC_PLL_COEFF_SELECT   0x0000050C

#define NV3_MIN_CLOCK_IN_KHZ  25000    // Not sure about this, but it seems reasonable
#define NV3_MAX_CLOCK_IN_KHZ 230000
#define NV4_MAX_CLOCK_IN_KHZ 350000

static int max_clock, is_nv3, pll_coeff;

/* NTSC cards have approx 14.3Mhz. Need to detect, but leave for now*/
#define PLL_INPUT_FREQ 13500 
#define M_MIN 7
#define M_MAX 13

#define P_MIN 0
#define P_MAX 4 /* Not sure about this. Could be 4 */

//=== Function section ===
// From xc/programs/Xserver/hw/xfree86/vga256/drivers/nv/riva_hw.c in XFree86 3.3.6
/*
 * Calculate the Video Clock parameters for the PLL.
 */
static int CalcVClock
(
    int           clockIn,
    int           double_scan,
    int          *clockOut,
    int          *mOut,
    int          *nOut,
    int          *pOut/*,
    RIVA_HW_INST *chip*/
)
{
    unsigned lowM, highM, highP;
    unsigned DeltaNew, DeltaOld;
    unsigned VClk, Freq;
    unsigned M, N, P;

    DeltaOld = 0xFFFFFFFF;

    VClk     = (unsigned)clockIn;
    if (double_scan)
        VClk *= 2;

    if (/*chip->CrystalFreqKHz*/PLL_INPUT_FREQ == 14318)
    {
        lowM  = 8;
        highM = 14 - (/*chip->Architecture == NV_ARCH_03*/is_nv3);
    }
    else
    {
        lowM  = 7;
        highM = 13 - (/*chip->Architecture == NV_ARCH_03*/is_nv3);
    }

    highP = 4 - (/*chip->Architecture == NV_ARCH_03*/is_nv3);
    for (P = 0; P <= highP; P ++)
    {
        Freq = VClk << P;
        if ((Freq >= 128000) && (Freq <= /*chip->MaxVClockFreqKHz*/max_clock))
        {
            for (M = lowM; M <= highM; M++)
            {
                N    = (VClk * M / /*chip->CrystalFreqKHz*/PLL_INPUT_FREQ) << P;
                Freq = (/*chip->CrystalFreqKHz*/PLL_INPUT_FREQ * N / M) >> P;
                if (Freq > VClk)
                    DeltaNew = Freq - VClk;
                else
                    DeltaNew = VClk - Freq;
                if (DeltaNew < DeltaOld)
                {
                    *mOut     = M;
                    *nOut     = N;
                    *pOut     = P;
                    *clockOut = Freq;
                    DeltaOld  = DeltaNew;
                }
            }
        }
    }
    return (DeltaOld != 0xFFFFFFFF);

}
           
// Set the clock to the given speed (in KHz)
Bool RIVA128ClockSelect( int clockspeed )
{
  int *ptr;

  int out;
  int m, n, p, value;
  int i = 0;
  pciConfigPtr pcr = NULL;
  int fd;

  vgaPCIInfo = vgaGetPCIInfo();

  if (vgaPCIInfo && vgaPCIInfo->AllCards) {
     	while ((pcr = vgaPCIInfo->AllCards[i++])) {
  		if (pcr->_vendor == PCI_VENDOR_NVIDIA_SGS)
  		{
                       if (pcr->_device == PCI_CHIP_RIVA128)
                       {
                               is_nv3 = 1;
                               pll_coeff = 0x10010100;
                               max_clock = NV3_MAX_CLOCK_IN_KHZ;
                               break;
                       }
               }
               if (pcr->_vendor == PCI_VENDOR_ID_NVIDIA)
               {
                       if (pcr->_device == PCI_DEVICE_ID_NVIDIA_TNT ||
                               pcr->_device == PCI_DEVICE_ID_NVIDIA_TNT2 ||
                               pcr->_device == PCI_DEVICE_ID_NVIDIA_UTNT2 ||
                               pcr->_device == PCI_DEVICE_ID_NVIDIA_VTNT2 ||
                               pcr->_device == PCI_DEVICE_ID_NVIDIA_UVTNT2 ||
                               pcr->_device == PCI_DEVICE_ID_NVIDIA_ITNT2)
                       {
                               is_nv3 = 0;
                               pll_coeff = 0x00010700;
                               max_clock = NV4_MAX_CLOCK_IN_KHZ;
                                break;
                       }

  		}
  	}
  }
  else 
  {
    PERROR(("No RIVA128/TNT/TNT2 chip found in PCI info!\n"));
    return FALSE;
  }

  if ( (!pcr) )
  {
    PERROR(("No RIVA128/TNT/TNT2 chip found in PCI info!\n"));
    return FALSE;
  }
  
  PDEBUG(("PCI BASE0 = 0x%x\n", pcr->_base0));

#ifndef DOS
  // Uses memory mapped registers, so must map memory
  fd = open( "/dev/mem", O_RDWR );
  
  if( fd == -1 )
  {
    PERROR(( "Error opening /dev/mem" ));
    return FALSE;
  }
    
  // mmap the programmable RAMDAC into our address space
  ptr = (int*)mmap( 0, 0x1000, PROT_READ | PROT_WRITE, MAP_SHARED, fd, (off_t)(pcr->_base0) + PRAMDAC_BASE );

  if( ptr == (int*)-1 )
  {
    PERROR(( "Error mmap'ing /dev/mem" ));
    return FALSE;
  }
    
  close( fd );
#else
  ptr=(int*) malloc(0x4000);
  if (!ptr || __djgpp_map_physical_memory(ptr, 0x1000, (off_t)(pcr->_base0)))
  {
      PERROR(("mmap() failed %d\n", ptr));
      return FALSE;
  }
#endif

  // Calculate the clock  
  //NV3ClockSelect( (float)clockspeed, &out, &m, &n, &p );
  CalcVClock ((float) clockspeed, 0, &out, &m, &n, &p);
    
  value = (m) + (n<<8) + (p<<16);
  
  // But of debug info
  PDEBUG(( "Wanted %dkHz, got %dkHz (m=%d, n=%d, p=%d, value=0x%08X)\n",
    clockspeed, (int)out, m, n, p, value ));
    
  // Default value is 0x00000100 (NV3)
  // X uses 0x10010100 (NV3) or 0x10000700 (NV4)
  // We use 0x00010100 (NV3) or 0x00010700 (NV4)
  ptr[PRAMDAC_PLL_COEFF_SELECT/4] = pll_coeff;  // could use |=

  // Divide by 4 because we're dealing with integers
  ptr[PRAMDAC_PLL_COEFF/4] = value;
  
#ifndef DOS
  // Unmap memory
  munmap( ptr, 0x1000 );
#endif

  // All done
  return TRUE;
}
