/****************************************************************************
*****************************************************************************
* The Burn Distribution, (C) Terry Porter <tjporter@odyssey.apana.org.au>
* This distribution comes under the GNU general public license agreement.
*****************************************************************************
*
* Atmel AT89C51 routines
*
* By Jason Nunn <jsno@downunder.net.au>, October 2000
* Darwin, Northern Territory, Australia
*
* =======================================================================
* uses both 6821 IC's.
*
* these were used in my auto-play ATAPI CDROM project (Onhold).
*
*****************************************************************************
****************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include "common.h"
#include "if.h"
#include "pio.h"
#include "target_2051.h"
#include "target_at89c51_defs.h"
#define  nBUSY SEL_OUT

int sig[3];

/****************************************************************************
* misc routines
****************************************************************************/
void set_addr_at89c51(int addr)
{
  chip_select(CHIP2);
  porta_data(addr & 0xff); /*set LS address on Port A, Chip 2*/
  portb_data(addr >> 8);   /*set MS address on Port B, Chip 2*/
  chip_select(CHIP1);
}

int set_prog_enable_volt_at89c51(void)
{
  if(sig[0] == 0x1e && sig[1] == 0x51)
  {
    if(sig[2] == 0xff)
    {
      porta_bit_high(ctrl_5v);
      porta_bit_high(ctrl_12v);
      return 1;
    }

    if(sig[2] == 0x05)
    {
      porta_bit_low(ctrl_5v);
      porta_bit_low(ctrl_12v);
      return 1;
    }
  }

  printf("Unknown signature, Unknown voltage interface "
         "Unable to perform operation.\n");
  return 0;
}

/****************************************************************************
* read signature
****************************************************************************/
int get_sig_cell_at89c51(int addr)
{
  int s;

  set_addr_at89c51(addr);  /*set manufacturer addr*/
  porta_bit_low(p2_7);     /*clock data out*/
  s = read_sig_cell_2051();
  porta_bit_high(p2_7);

  return s;
}

void read_sig_cell_at89c51(void)
{
  set_portb_tristate();  /*make chip 1, port B high high imped.*/

/*set signature mode*/
  porta_bit_low(p2_6);
  porta_bit_high(p2_7);
  porta_bit_low(p3_6_p3_7);
  porta_bit_high(prog);
  porta_bit_low(ctrl_5v);  /*program enable now = 5v*/

  printf("READING SIGS NOW, Please wait...\n"
         "Sig: ");

  sig[0] = get_sig_cell_at89c51(0x30); /*read manufacturer field*/
  sig[1] = get_sig_cell_at89c51(0x31); /*read IC type field*/
  sig[2] = get_sig_cell_at89c51(0x32); /*read Voltage type field*/
  putchar('\n');
  power_off_target_2051();
}

/****************************************************************************
* lock IC (lock bits 1 and 2 only)
****************************************************************************/
void lock_at89c51(void)
{
  read_sig_cell_at89c51();

/*set write lock - bit 1*/
  porta_bit_high(p2_6);
  porta_bit_high(p2_7);
  porta_bit_high(p3_6_p3_7);
  porta_bit_high(prog);
  if(!set_prog_enable_volt_at89c51())
    return;

  printf("Setting write lock (bit 1), Please wait...\n");
  porta_bit_low(prog);
  usleep(10000);
  porta_bit_high(prog);

/*set write lock - bit 2*/
  porta_bit_low(p3_6_p3_7);

  printf("Setting write lock (bit 2), Please wait...\n");
  porta_bit_low(prog);
  usleep(10000);
  porta_bit_high(prog);

  printf("Done.\n");
}

/****************************************************************************
*
****************************************************************************/
void read_image_from_at89c51(void)
{
  int addr;

  alloc_image_buffer();
  set_portb_tristate(); /*make chip 1, port B high high imped.*/

/*set read mode*/
  porta_bit_low(p2_6);
  porta_bit_high(p2_7);
  porta_bit_high(p3_6_p3_7);
  porta_bit_high(prog);
  porta_bit_low(ctrl_5v);  /*program enable now = 5v*/

  printf("READING NOW, Please wait...\n");
  for(addr = 0;addr < target_size;addr++)
  {
    int data = 0xff;

    print_read_ticker(addr,0xf);
    set_addr_at89c51(addr);  /*set address*/
    porta_bit_low(p2_7);     /*clock data out*/
    for(;;)
    {
      portb_data(data);
      if(!(read_status() & PE))
        break;
      data--;
    }
    porta_bit_high(p2_7);
    image_buf[addr] = data;
  }

  print_read_ticker_final(addr);
}

/****************************************************************************
* main display function
****************************************************************************/
void display_at89c51(void)
{
  read_image_from_at89c51();
  print_hex_dump();
}

/****************************************************************************
* Read and save to file
****************************************************************************/
void read_at89c51(void)
{
  read_image_from_at89c51();
  write_image_to_file();
}

/****************************************************************************
* erase IC
****************************************************************************/
void erase_at89c51(void)
{
  read_sig_cell_at89c51();

/*set erase mode*/
  porta_bit_high(p2_6);
  porta_bit_low(p2_7); 
  porta_bit_low(p3_6_p3_7);
  porta_bit_high(prog);
  if(!set_prog_enable_volt_at89c51())
    return;

/*erase*/
  printf("ERASING NOW, Please wait...\n");
  porta_bit_low(prog);
  usleep(10000);
  porta_bit_high(prog);

  printf ("Done.\n");
}

/****************************************************************************
* main write function
****************************************************************************/
void program_at89c51(void)
{
  int addr,failed_burn = 0;

  read_image_from_file();
  erase_at89c51();
  usleep(1);

/*set mode's that won't change during write and verify cycles*/
  porta_bit_low(p2_6);
  porta_bit_high(p3_6_p3_7);
  porta_bit_high(prog);

  printf("PROGRAMMING NOW, Please wait...\n");
  for(addr = 0;addr < target_size;addr++)
  {
    int timeout;

    print_read_ticker(addr,0xf);

    if(image_buf[addr] == 0xff)
      continue;

    set_addr_at89c51(addr);      /*set address*/

   /*
    * write cycle
    */
    set_portb_output();
    portb_data(image_buf[addr]); /*set data*/
    porta_bit_high(p2_7);
    if(!set_prog_enable_volt_at89c51())
      return;

    porta_bit_low(prog);         /*program, clock data in*/
    porta_bit_high(prog);

    /*wait for the /BUSY line to go high*/
    timeout = 0;
    while(!(read_status() & nBUSY))
    {
      timeout++;

      if(timeout > 1000)
      {
        printf("IC won't program. Burn Failed. Aborting.\n");
        return;
      }
    }

   /*
    * verify cycle
    */
    set_portb_tristate();
    porta_bit_low(ctrl_5v);   /*VPP = 5v*/
    porta_bit_low(ctrl_12v);

    porta_bit_low(p2_7);     /*clock data out*/
    portb_data(image_buf[addr]);
    if(read_status() & PE)
      failed_burn++;
    porta_bit_high(p2_7);
  }

  print_read_ticker_final(addr);
  print_result(failed_burn);
}
