/****************************************************************************
*****************************************************************************
* The Burn Distribution, (C) Terry Porter <tjporter@odyssey.apana.org.au>
* This distribution comes under the GNU general public license agreement.
*****************************************************************************
*
* Atmel AT24Cx (and compatables) routines
* AT24C08 only at the moment
*
* (nb/ STTI also make a compatable IC family as well).
*
* By Jason Nunn <jsno@downunder.net.au>, June 2000
* Darwin, Northern Territory, Australia
*
* =======================================================================
* uses only chip 1
*
* you'll find these IC's in computer monitors ('Viewmaster' Monitors for
* people living in Australia), both as a data store by the signal
* processing micro and by the vga cards that read data from it
* (plug and pray feature in winshit.. ever wondered how winshit can
* detect the presence of a different monitor ?... *wick*).
*
* apparently, these IC's are also used in smart cards.
*
* - port1 is always set as output (already been set to output in burn.c.)
*
* - i could implement page writing and sequential reading, but to keep
* the code as simple as possible, and to support a wide range of IC's
* with the same code base, for now, i've only implemented random read
* and byte write commands.
*****************************************************************************
****************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include "common.h"
#include "if.h"
#include "pio.h"
#include "target_at24cx_defs.h"

/*
 * macros to set line states
 */
#define BUS_H portb_bit_high(AT24CX_BUS)
#define BUS_L portb_bit_low(AT24CX_BUS)

#define CLK_H porta_bit_high(AT24CX_CLK)
#define CLK_L porta_bit_low(AT24CX_CLK)

/****************************************************************************
* init routines
*
* this sets the initial line states
****************************************************************************/
void init_at24cx(void)
{
  printf("Initing... ");
  set_portb_output();

  BUS_H;
  CLK_H;

  printf("done.\n");
}

/****************************************************************************
* frame ambles.
*
* a start condition takes IC out of standby mode, and is ready to accept
* our frame. a stop puts it back into standby mode.
*
* we talk to this family in serial frames.. we use this ambles at the
* start and end of each frame.
****************************************************************************/
void set_start_condition(void)
{
  BUS_H;
  CLK_H;

  BUS_L;
  CLK_L;
}

void set_stop_condition(void)
{
  BUS_L;
  CLK_H;
  BUS_H;
}

/****************************************************************************
* probe the databus and read it's state..
*
* set PBx lines AND Dx lines high for comparitor. also, databus will   
* be set to high in this operation as well (pulled up by PB0). the IC
* will pull down databus with it's internal FET to indicate lows.
****************************************************************************/
int probe_bus(void)
{
  portb_data(0xff);
  
  if(!(read_status() & PE))
    return 1;
   
  return 0;
}

/****************************************************************************
* read frame word (8 bits .. MS first)
*
* set databus high (so that IC pulls databus down to indicate lows), and
* negative trigger data out of IC.
****************************************************************************/
int read_word(void)
{
  int data = 0,x = 0;

  BUS_H;
  for(;;)
  {
    CLK_H;

    if(probe_bus())
      data |= 1;

    CLK_L;

    if(++x == 8)
      break;

    data <<= 1;
  }

  return data;
}

/****************************************************************************
* serially send data word via databus
*
* databus can only change state when clock is LOW
* data is feed MS first
*
* for writing data, clock pulse is triggered on a positive edge
****************************************************************************/
void write_word(int data)
{
  int x;

 /*
  * clock 8 bits in (MS first)
  */
  for(x = 0x80;x > 0;x >>= 1)
  {
    if(data & x)
      BUS_H;
    else
      BUS_L;

    CLK_H;           /*clock data in*/
    CLK_L;
  }

 /*
  * now try to sense an acknowledge
  */
  BUS_H;
  CLK_H;
  if(!probe_bus())
  {
    CLK_L;
    return;
  }

  printf("IC failed to acknowledge after sending word. Aborting.\n");
  exit(-1);
}

/****************************************************************************
* compose device word (this a IC specific, and varies from chip to chip)
****************************************************************************/
void send_device_word(int addr,int read_f)
{
  switch(target_id)
  {
   /*
    * send device address word for AT24C08
    *
    *     1  0  1  0  A2 P1 P0 RW
    *
    * 4 pages, 256 bytes per page
    *
    * first send device word, then address word
    */
    case TARGET_24C08:
      write_word(AT24CX_PREAMBLE | ((addr >> 7) & 0x6) | read_f);
      break;
  }
}

/****************************************************************************
* BYTE READ FRAME
****************************************************************************/
void random_read_frame(int addr)
{
 /*
  * send a "dummy" write frame
  */
  set_start_condition();
  send_device_word(addr,AT24CX_WRITE_F);
  write_word(addr); /*send address word*/

 /*
  * now do a "current read"
  */
  set_start_condition();
  send_device_word(addr,AT24CX_READ_F);
  image_buf[addr] = read_word();

 /*
  * send no ack
  */
  BUS_H;
  CLK_H;
  CLK_L;

  set_stop_condition();
}

/****************************************************************************
* BYTE WRITE FRAME
****************************************************************************/
void byte_write_frame(int addr)
{
 /*
  * send "write" frame header
  */
  set_start_condition();
  send_device_word(addr,AT24CX_WRITE_F);
  write_word(addr);  /*send address word*/

 /*
  * send data
  */
  write_word(image_buf[addr]);

  set_stop_condition();

 /*
  * IC enter internally timed write cycle for 10us
  *
  * how the manual says to do this, is ignore the wait and
  * push on with iterative looping if frame fails. .. but i'm nice ;)
  */
  usleep(10);
}

/****************************************************************************
*
****************************************************************************/
void read_image_from_at24cx(void)
{
  int addr;

  alloc_image_buffer();

  printf("READING NOW, Please wait...\n");
  for(addr = 0;addr < target_size;addr++)
  {
    random_read_frame(addr);
    print_read_ticker(addr,0x3);
  }
  print_read_ticker_final(addr);
}

/****************************************************************************
* main display function
****************************************************************************/
void display_at24cx(void)
{
  read_image_from_at24cx();
  print_hex_dump();
}

/****************************************************************************
* Read and save to file
****************************************************************************/
void read_at24cx(void)
{
  read_image_from_at24cx();
  write_image_to_file();
}

/****************************************************************************
* main write function
*
* when writing data to the IC, this is how it's done here-
*
* 1) write entire block of data
* 2) read back what you have written
* 3) then verify the two images
* 4) if they match, then burn successful
*
****************************************************************************/
void program_at24cx(void)
{
  int addr;
  char *old_image_buf;

  read_image_from_file();

  printf("PROGRAMMING NOW, Please wait...\n");
  for(addr = 0;addr < target_size;addr++)
  {
    print_write_ticker(addr,0x3);
    byte_write_frame(addr);
  }
  print_write_ticker_final(addr);

  printf("VERIFYING NOW, Please wait...\n");
  old_image_buf = image_buf;
  read_image_from_at24cx();

  if(!memcmp(image_buf,old_image_buf,target_size))
    printf("Burn Successful.\n");
  else
    printf("Burn failed.\n");
}
