/****************************************************************************
*****************************************************************************
* The Burn Distribution, (C) Terry Porter <tjporter@odyssey.apana.org.au>
* This distribution comes under the GNU general public license agreement.
*****************************************************************************
*
* Amtel AT892051 read, verify and burn routines
*
* =====================================================================
*
* this code uses chip 1 only.
*
* nb/ these routines have a target length as a parameter, so it's easy to
* burn other family members with different flash sizes.
*
* CTRL_5V | CTRL_12V | RST/VPP
* --------+----------+---------
*    L    |    L     |  5V
*    H    |    L     |  0V
*    L    |    H     |  12V
*    H    |    H     |  12V
*
* nb/ - terry: through out the code, i've changed the low states of CTRL_5V
* to high states and visa-versa. -- jsno
*
* nb2/ have to think about making power_off_target_2051() independant of
* 2051 as it's called everywhere -- jsno.
*
*****************************************************************************
****************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include "common.h"
#include "if.h"
#include "pio.h"
#include "target_2051_defs.h"

#define  nBUSY_2051 SEL_OUT

/****************************************************************************
* low level funcs used in these file and others (these really could be
* turned into macros)
****************************************************************************/

/*+ Power off the target chip +*/
void
power_off_target_2051 (void)
{
  porta_bit_high (ctrl_5v); /* Turn of power to the chip */
  porta_bit_low (ctrl_12v);
}

/*+ Resets chip +*/
void reset_target_2051(void)
{
  porta_bit_low (ctrl_5v);  /*Reset the 2055 = 5v*/
  usleep (100);             /*give the xtal time to start*/
  porta_bit_high (ctrl_5v); /*restart 2055   = 0v*/
}

/*+ Programming pulse +*/
void
program_pulse (void)
{
  porta_bit_low (p3_2_prog);  /*Just one pulse will do, its self timed*/
  porta_bit_high (p3_2_prog); /*inside the 2051*/
}

/*+ Clock the 2051 +*/
void
Clock_2051 (void)
{
  porta_bit_high (clock_2051); /*one clock pulse*/
  porta_bit_low (clock_2051);
}

/****************************************************************************
* read atmel x051 signature
****************************************************************************/
int read_sig_cell_2051(void)
{
  int data = 0xff;

  for(;;)
  {
    portb_data(data);

    if(!(read_status () & PE))  /*a /PE == matched*/
      break;

    if(!data)
    {
      printf("?? ");
      return -1;
    }

    data--;
  }

  printf("%02x ",data);

  return data;
}

void read_sig_2051(void)
{
  int x;

  set_portb_tristate();    /*make PBx/Port1 high impedence*/
  porta_bit_low(ctrl_5v);  /*cntl voltage = 5V, IC is reset, PC now 0*/

  porta_bit_low(clock_2051);
  porta_bit_high(p3_2_prog);
  porta_bit_low(p3_3);
  porta_bit_low(p3_4);
  porta_bit_low(p3_7_p3_5);

  printf("READING SIGS NOW, Please wait...\n"
         "Sig: ");
  for(x = 0;x < 2;x++)
  {
    read_sig_cell_2051();
    Clock_2051();
  }
  putchar('\n');
}

/****************************************************************************
* subroutine for display_2051() and dump_2051()
*
* Read the 2051, by clocking the xtal pin and testing the 74688 comparitor
* against a generated 0 - 255 bit number from the PC
****************************************************************************/
void read_image_from_2051(void)
{
  int address;

  alloc_image_buffer();

/* we want to read the 2051, so make this port tri state */
  set_portb_tristate ();

/* Port1 is permanemtly an output, for the 2051 timing signals */

/* Initalise the 2051 for a read cycle */
  porta_bit_low (clock_2051);
  porta_bit_high (p3_2_prog);
  porta_bit_low (p3_3);
  porta_bit_low (p3_4);
  porta_bit_high (p3_7_p3_5);
  porta_bit_low (ctrl_5v); /*cntl voltage = 5V, IC is reset, PC now 0*/

/* data is now available, just needs to be clocked */
  printf ("READING NOW, Please wait...\n");
  for (address = 0; address < target_size; address++)
    {
      int chip_data = 0xff; /*Inital data to test for, 
                              faster on erased bytes*/

      print_read_ticker(address,0xf);

     /*
      * dummy send to port1, which is tri-state to get data
      * to the comparator and not intefere with port 2
      * increment chip_data
      */
      for(;;)
      {
        portb_data(chip_data);
        if(!(read_status () & PE))  /*a /PE == matched*/
          break;
        chip_data--;
      }

      image_buf[address] = chip_data;
      Clock_2051();
    }

  print_read_ticker_final(address);
}

/****************************************************************************
* dump contents of 2051 on stdout
****************************************************************************/
void
display_2051 (void)
{
  read_image_from_2051();
  print_hex_dump();
}

/****************************************************************************
* Read the 2051 and save to file
****************************************************************************/
void
read_2051 (void)
{
  read_image_from_2051();
  write_image_to_file();
}

/****************************************************************************
*
****************************************************************************/
/*+ Erase the 2051 +*/
void
erase_2051 (void)
{
  printf("ERASING NOW, Please wait...\n");

  porta_bit_high (p3_3);      /*set up proper combination of control sigs*/
  porta_bit_low (p3_4);
  porta_bit_low (p3_7_p3_5);
  porta_bit_low (clock_2051);

  porta_bit_high (p3_2_prog); /*Prog pin high (not programming)*/
  porta_bit_high (ctrl_12v);  /*Turn on 12 volts to the "RST" pin*/
  porta_bit_low (p3_2_prog);  /*Program for 10 milli secs (i think its
                                self timed)*/
  usleep (10000);

  porta_bit_high (p3_2_prog); /* Program off*/

  printf ("Done.\n");
}

/****************************************************************************
* Program the 2051
*
* cycle chart for voltage control
*
* 12V ------------+ +-----------+ +-------------+ +---------
*       burn'in   | |           | |             | |
*                 | |           | |             | |
*                 | |           | |             | |
* 5V              +-+           +-+             +-+
*                 verify'in
*
****************************************************************************/
void
program_2051 (void)
{
  int address,failed_burn = 0;

  read_image_from_file();

  /* Erase the 2051 */
  erase_2051 ();

  usleep (1);
/* Initalise  the 2051 for a burn cycle*/
/* reset the 2051 address counter to zero */
  porta_bit_high (ctrl_5v);
  porta_bit_low (clock_2051);
  porta_bit_high (p3_2_prog);

/*set flags that are constant though out write/verify process*/
  porta_bit_low (p3_3);
  porta_bit_high (p3_7_p3_5);
  porta_bit_low (ctrl_5v);    /*cntl voltage now = 5v*/

  printf ("PROGRAMMING NOW, Please wait...\n");
  for (address = 0; address < target_size; address++)
    {
      print_write_ticker(address,0xf);

     /*
      * if data is == 0xff, no need to burn. All 2051 locns
      * are 0xff after an erase cycle
      */
      if (image_buf[address] != 0xff)
	{
          int timeout;

         /*
          * write cycle (make port2 output, set program modes)
          */
	  set_portb_output ();
	  porta_bit_high (p3_4);
	  porta_bit_high (ctrl_12v);
          portb_data(image_buf[address]);
	  program_pulse ();

          /*wait for the /BUSY_2051 line to go high*/
	  timeout = 0;
	  while(!(read_status() & nBUSY_2051))
	    {
	      timeout++;

              if (timeout > 1000)
	      {
	        printf ("IC won't program. Burn Failed. Aborting.\n");
	        return;
	      }
	    }

         /*
          * verify cycle (make port2 tristate, set read mode, set cntl
          *               voltage to 5v)
          */
	  set_portb_tristate();
	  porta_bit_low (p3_4);
	  porta_bit_low (ctrl_12v);

	  /*
           * data should now be compared between 2051 and pc parallel
           * port, by the comparitor dummy send to port2, as setting it to
           * a input has destroyed the prev data
           */
	  portb_data(image_buf[address]);

	  if(read_status () & PE)  /*Check for failures and count them*/
	    failed_burn++;
	}

      Clock_2051(); /* increment the 2051 address counter */
    }

  print_write_ticker_final(address);
  print_result(failed_burn);
}
