/****************************************************************************
*****************************************************************************
* The Burn Distribution, (C) Terry Porter <tjporter@odyssey.apana.org.au>
* This distribution comes under the GNU general public license agreement.
*****************************************************************************
*
* Routines to access the 6821 PIO chip , send data to its pins etc
*
*****************************************************************************
****************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include "if.h"
#include "pio_defs.h"

/****************************************************************************
* misc routines
****************************************************************************/
int chip_selected;

/*+ selects which PIO IC will respond to commands +*/
void chip_select(int c)
{
  chip_selected = c;

  switch(c)
  {
    case CHIP1:  /*chip 1 (U3 on schematic)*/
      turnon(CS);
      break;

    case CHIP2:  /*chip 2 (U1 on schematic)*/
      turnoff(CS);
      break;
  }
}

/****************************************************************************
* 6821 portA IO control routines
****************************************************************************/

/*+ Sets up portA, all pins as inputs (used to tri state the port) +*/
void set_porta_tristate(void)
{
/* set port a 6821_a as input */

/* select portA, control register  0 1 */
  turnoff (RS1);
  turnon (RS0);

/* set control register to select "data direction register B", 
   send 0x00 to the 6821 */
  port_data (0x00);

/* select portA, data direction register  0 0 */
  turnoff (RS1);
  turnoff (RS0);

/* set all bits to input */
  port_data (0x00);

/* set the control register to "peripheral register" */
  port_data (0x04);

/* Leave in peripheral register/ddr mode */
  turnoff (RS1);
  turnoff (RS0);

/* PortA is now in input mode, and although its pullups make it appear high,
 * it wont clash with data on its pins.
 */
}

/*+ Sets up portA, all pins as outputs +*/
void set_porta_output(void)
{
/* select portA, control register  0 1 */
  turnoff (RS1);
  turnon (RS0);

/* set control register to select "data direction register A" */
  port_data (0x00);

/* select portA, data direction register  0 0 */
  turnoff (RS1);
  turnoff (RS0);

/* set all bits to output */
  port_data (0xff);

/* setup portA for data output*/
/* select portA, control register  0 1 */
  turnoff (RS1);
  turnon (RS0);

/* set control register to select "output register A" */
  port_data (0x04);

/* Data sent to portA, will now appear on its pins */
}

/****************************************************************************
* 6821 portA misc routines
****************************************************************************/
int c1_pa_reg,c2_pa_reg;

/*+ Sends data to chip 1, port A pins +*/
void porta_data(int data)
{
/* address peripheral register A, 0 0 */
  turnoff(RS1);
  turnoff(RS0);

  port_data(data);

  switch(chip_selected)
  {
    case CHIP1:
      c1_pa_reg = data;  /*update the cached register*/
      break;

    case CHIP2:
      c2_pa_reg = data;
      break;
  }
}

/*+ Sets a particular bit of chip 1, port A+*/
void porta_bit_high(int data)
{
  turnoff(RS1);
  turnoff(RS0);

  switch(chip_selected)
  { 
    case CHIP1:
      c1_pa_reg |= data;
      port_data(c1_pa_reg);
      break;

    case CHIP2:
      c2_pa_reg |= data;
      port_data(c2_pa_reg);
      break;
  }
}

/*+ Resets a particular bit of chip1, port A*/
void porta_bit_low(int data)
{
  turnoff(RS1);
  turnoff(RS0);

  switch(chip_selected)
  { 
    case CHIP1:
      c1_pa_reg &= ~data;
      port_data(c1_pa_reg);
      break;

    case CHIP2:
      c2_pa_reg &= ~data;
      port_data(c2_pa_reg);
      break;
  }
}

/****************************************************************************
* 6821 portB IO control routines
****************************************************************************/

/*+ Sets up portB, all pins as inputs (used to tri state the port) +*/
void set_portb_tristate(void)
{
/* select portB, control register  1 1 */
  turnon (RS1);
  turnon (RS0);

/* set control register to select "data direction register B", send 0x00 to the 6821 */
  port_data (0x00);

/* select portB, data direction register  1 0 */
  turnon (RS1);
  turnoff (RS0);

/* set all bits to input */
  port_data (0x00);

/* now select the peripheral register */
/* select portB, control register  1 1 */
  turnon (RS1);
  turnon (RS0);

/* set control register to select "output register", send 0x04 to the 6821 */
  port_data (0x04);

/* select peripheral reg 1 0 , and send 0x00 to zero data  */
  turnon (RS1);
  turnoff (RS0);
  port_data (0x00);

/* PortB is now in input mode, it wont clash with data on its pins. */
}

/*+ Sets up portB, all pins as outputs +*/
void set_portb_output(void)
{
/* set port B of 6821 chip 1 as output */

/* select portB, control register  1 1 */
  turnon (RS1);
  turnon (RS0);

/* set control register to select "data direction register", send 0x00 to the 6821 */
  port_data (0x00);

/* select portB, data direction register  1 0 */
  turnon (RS1);
  turnoff (RS0);

/* set all bits to output */
  port_data (0xff);

/* setup portB for data output*/
/* select portB, control register  1 1 */
  turnon (RS1);
  turnon (RS0);

/* set control register to select "output register", send 0x04 to the 6821 */
  port_data (0x04);

/* select peripheral reg 1 0 , and send 0x00 to zero data  */
  turnon (RS1);
  turnoff (RS0);
  port_data (0x00);

/* Data sent to portB, will now appear on its pins */
}

/****************************************************************************
* 6821 portB misc routines
****************************************************************************/
int c1_pb_reg,c2_pb_reg;

/*+ Sends data to chip 1, portB pins +*/
void portb_data(int data)
{
/* address peripheral register B, 1 0 */
  turnon(RS1);
  turnoff(RS0);

  port_data(data);

  switch(chip_selected)
  {
    case CHIP1:
      c1_pb_reg = data;  /*update the cached register*/
      break;

    case CHIP2:
      c2_pb_reg = data;
      break;
  }
}

/*+ Sets a particular bit of chip 1, portB+*/
void portb_bit_high(int data)
{
  turnon(RS1);
  turnoff(RS0);

  switch(chip_selected)
  { 
    case CHIP1:
      c1_pb_reg |= data;
      port_data(c1_pb_reg);
      break;

    case CHIP2:
      c2_pb_reg |= data;
      port_data(c2_pb_reg);
      break;
  }
}

/*+ Resets a particular bit of chip 1, portB+*/
void portb_bit_low(int data)
{
  turnon(RS1);
  turnoff(RS0);

  switch(chip_selected)
  {
    case CHIP1:
      c1_pb_reg &= ~data;
      port_data(c1_pb_reg);
      break;

    case CHIP2:
      c2_pb_reg &= ~data;
      port_data(c2_pb_reg);
      break;
  }
}

/****************************************************************************
* init routines
****************************************************************************/

/*+ Initalise any critical states of the pio in use here +*/
void
initalise_pio(void)
{
  turnon(E); /* E clock must be high to start */

 /*
  * set porta A and B of chip 2 to output, and set port A of chip 1
  * to output.
  *
  * these will always be outputs, and will never be tristated, or
  * used as inputs, so we set them now and forget about them.
  *
  * nb/ the only port you will see tristated in the code is port B, chip 1.
  *     when reading data via U2 (74688).
  */
  chip_select(CHIP2);
  set_porta_output();
  set_portb_output();

  chip_select(CHIP1);
  set_porta_output();
}

void deinitalise_pio(void)
{
  turnoff(E);
}

/****************************************************************************
* terry... i haven't tested these relay routines. could you give them the
* once over? ;)
* nb/ moved 2051 dependant code into burn.c and target_2051.c
* -- jsno
****************************************************************************/

/*+ Switches (energises) relay to switch contacts to burn,read chip etc, and
 * switches out the xtal, with the burner taking over +*/
void
relay_burn (void)
{
  printf("Relays ON.\n");

  chip_select(CHIP1);
  porta_bit_high (relay);
  usleep (50000);         /*Time for relay contacts to settle (50ms)*/

/* now burn, read the target, whatever */
}


/*+ De energises relay so the chip pins connect to the project being developed
 * including switching in the xtal, and resetting the chip +*/
void
relay_project (void)
{
  printf("Relays OFF.\n");

  chip_select(CHIP1);
  porta_bit_low (relay);
  usleep (50000);           /*Time for relay contacts to settle (50ms)*/

/* project has control of the target now */
}

/****************************************************************************
* Test for POWER ON, or programmer not connected
****************************************************************************/
void test_for_power_off(void)
{
  printf("Detecting unit... ");

 /*
  * Note the internal parallel port pull up on "PE", pin12, makes this line
  * look high if its not connected to anything, or if the comparitor has no
  * power              
  */

  chip_select(CHIP1);
  set_portb_output();
  portb_data(0xaa);  /*Set port 2 bits high,low etc */
  porta_data(0x55);  /*Different data to pc data, via a dummy load to port1*/

 /*
  * Compare different words , one from port2, the other from 
  * the parallel port if the power is on, /peq will be high 
  * if the numbers are different we get a high (0x20), if the power is off
  * then /peq will be low via the pull down resistor of 470 Ohms.
  */
  if(!(read_status () & PE))
  {
    printf ("no.\nPOWER is off. Apply power to unit and retry, Aborting.\n");
    exit(-1);
  }

  printf("yes.\n");
}
