/****************************************************************************
*****************************************************************************
*****************************************************************************
* The Burn Distribution, (C) Terry Porter <tjporter@odyssey.apana.org.au>
* This distribution comes under the GNU general public license agreement.
*
* ================================================================
* main listing.
*
*****************************************************************************
*****************************************************************************
****************************************************************************/
#include <stdio.h>
#include "version.h"
#include "common.h"
#include "if.h"
#include "pio.h"
#include "target_2051.h"
#include "target_at24cx.h"
#include "target_at89c51.h"

#define ERASE_TARGET    1
#define PROGRAM_TARGET  2
#define READ_TARGET     3
#define DISPLAY_TARGET  4
#define READ_SIG_TARGET 5
#define LOCK_TARGET     6

int     func = 0;  /*function id, enumerated by one of the above defs*/

/****************************************************************************
*
****************************************************************************/
void na_message(void)
{
  printf("Not applicable to this family (or not implemented yet).\n");
}

/***************************************************************************/
/*+ target handler for amtel x051 family +*/
/***************************************************************************/
void do_target_x051(void)
{
  power_off_target_2051();
  test_for_power_off();
  relay_burn();           /*now, switch target to PIO control*/

  switch(func)
  {
    case ERASE_TARGET:
      erase_2051();
      break;

    case PROGRAM_TARGET:
      program_2051();
      break;

    case READ_TARGET:
      read_2051();
      break;

    case DISPLAY_TARGET:
      display_2051();
      break;

    case READ_SIG_TARGET:
      read_sig_2051();
      break;

    default:
      na_message();
      break;
  }

  power_off_target_2051();
  relay_project(); /*hand control of IC back to application*/
  reset_target_2051();
}

/***************************************************************************/
/*+ target handler for amtel AT24Cx serial EEPROM family +*/
/***************************************************************************/
void do_target_at24cx()
{
  test_for_power_off();
  relay_burn();
  init_at24cx();

  switch(func)
  {
    case PROGRAM_TARGET:
      program_at24cx();
      break;

    case READ_TARGET:
      read_at24cx();
      break;

    case DISPLAY_TARGET:
      display_at24cx();
      break;

    default:
      na_message();
      break;
  }

  relay_project();
}

/***************************************************************************/
/*+ target handler for the Atmel AT89C51 (12V program enable type)+*/
/***************************************************************************/
void do_target_at89c51(void)
{
  power_off_target_2051();  /*same voltage signals apply to this IC*/
  test_for_power_off();
  relay_burn();

  switch(func)
  {
    case ERASE_TARGET:
      erase_at89c51();
      break;

    case PROGRAM_TARGET:
      program_at89c51();
      break;

    case READ_TARGET:
      read_at89c51();
      break;

    case DISPLAY_TARGET:
      display_at89c51();
      break;

    case READ_SIG_TARGET:
      read_sig_cell_at89c51();
      break;

    case LOCK_TARGET:
      lock_at89c51();
      break;

    default:
      na_message();
      break;
  }

  power_off_target_2051();
  relay_project();
  reset_target_2051();
}

/***************************************************************************/
/*+ prints help +*/
/***************************************************************************/
void print_help(void)
{
  printf(
    "Burn comes with ABSOLUTELY NO WARRANTY; for details, read the\n"
    "COPYING file that comes with this distribution. This is free\n"
    "software, and you are welcome to redistribute it under certain\n"
    "conditions.\n\n"
    "Usage: burn -m <target> [-hewrdsl] [-f <filename.bin>]\n"
    "\n"
    "       burn -h ....................... This display.\n"
    "       burn -m <t> -e ................ Erase IC (if applicable)\n"
    "       burn -m <t> -w -f <filename> .. burn an image to IC\n"
    "       burn -m <t> -r -f <filename> .. Dump IC to file\n"
    "       burn -m <t> -d ................ view IC\n"
    "       burn -m <t> -s ................ view signature of IC (if applicable)\n"
    "       burn -m <t> -l ................ Lock IC (if applicable)\n"
    "\n"
    "Current Targets: 2051,4051,24C08,89C51\n\n");
}

/***************************************************************************/
/*+ program entry point +*/
/***************************************************************************/
int main(int argc, char *argv[])
{
  int x;

  printf("Burn Programmer " VERSION " (for Unit revision: 2051_proto R1 & Universal R1)\n"
         "Copyright (C) Terry Porter <tjporter@odyssey.apana.org.au>\n"
         "\n");

  if(argc == 1)
  {
    print_help();
    exit(-1);
  }

  for(x = 1;x < argc;x++)
  {
    if(argv[x][0] == '-')
    {
      int nx;

      nx = x + 1;
      switch(argv[x][1])
      {
        case 'h':
          print_help();
          exit(0);

        case 'm':
          if(nx < argc)
          {
            if     (!strcmp(argv[nx],"2051"))
              target_id = TARGET_2051;
            else if     (!strcmp(argv[nx],"4051"))
              target_id = TARGET_4051;
            else if(!strcmp(argv[nx],"24C08"))
              target_id = TARGET_24C08;
            else if(!strcmp(argv[nx],"89C51"))
              target_id = TARGET_89C51;
            else
            {
              printf("Unknown target, Aborting.\n");
              exit(-1);
            }
          }
          break;

        case 'e':
          func = ERASE_TARGET;
          break;

        case 'w':
          func = PROGRAM_TARGET;
          break;

        case 'r':
          func = READ_TARGET;
          break;

        case 'd':
          func = DISPLAY_TARGET;
          break;

        case 's':
          func = READ_SIG_TARGET;
          break;

        case 'l':
          func = LOCK_TARGET;
          break;

        case 'f':
          if(nx < argc)
            filename = argv[nx];
          break;

        default:
          printf("Unknown parameter, aborting.\n");
          exit(-1);
      }

    }
  }

  if(!target_id || !func)
  {
    printf("Missing parameter. Aborting.\n");
    exit(-1);
  }

  if((func != DISPLAY_TARGET) &&
     (func != ERASE_TARGET) &&
     (func != READ_SIG_TARGET) &&
     (func != LOCK_TARGET))
    if(filename == NULL)
    {
      printf("Filename not specified, Aborting.\n");
      exit(-1);
    }

 /*
  * start interface and init our PIO
  */
  IF_init();
  initalise_pio();

 /*
  * main burn code below
  */
  switch(target_id)
  {
    case TARGET_2051:
      printf("Target: Atmel AT89C2051 (2kb flash RAM)\n");
      target_size = 2048;
      do_target_x051();
      break;

    case TARGET_4051:
      printf("Target: Atmel AT89C4051 (4kb flash RAM)\n");
      target_size = 4096;
      do_target_x051();
      break;

    case TARGET_24C08:
      printf("Target: Atmel AT24C08 and compatables (1kb serial EEPROM)\n");
      target_size = 1024;
      do_target_at24cx();
      break;

    case TARGET_89C51:
      printf("Target: Atmel AT89C51 (4kb flash RAM)\n");
      target_size = 4096;
      do_target_at89c51();
      break;

  }

  deinitalise_pio();

  return 1;
}
