/**************************************************************************
*
* /dev/burn device for Linux 2.x kernels
* by Jason Nunn <jsno@downunder.net.au>, Darwin, NT, Australia. April 2000
* This software comes under the GNU GPL. See COPYING.
*
* A Linux parallel port interface module for AT892051 Programmer and Burn
* (by Terry Porter <tjporter@odyssey.apana.org.au>), but can be for any
* solid state application that you which to control via parrallel port and
* via /dev/burn
*
* See README for installation instructions.
*
* (nb/ code adapted from Generic parallel printer driver, by Jim Weigand
*  and Linus Torvalds)
*
* cmdline:
*
* insmod burnm.o parport = 1
*
* =====================================================================
* API:
*
* the /dev/burnm is a 2 byte fixed size file. you can move to and
* read/write to any of the bytes-
*
* - reading from any position in the file returns 1 byte status register
* - writing to file at offset 0 writes 1 byte to data register
* - writing to file at offset 1 writes 1 byte to ctrl register
*
* reading:
*                        Offset 0   Offset 1
*                      +----------+----------+
*                      |  Status  |  Status  |
*                      +----------+----------+
*
* writing:
*                        Offset 0   Offset 1
*                      +----------+----------+
*                      |   Data   |   Ctrl   |
*                      +----------+----------+
*
**************************************************************************/
#include <linux/version.h>
#include <linux/module.h>
#include <linux/modversions.h>
#include <linux/init.h>
#include <linux/autoconf.h>
#include <linux/parport.h>
#include <asm/uaccess.h>

#define BURNM_VER     "0.0"
#define BURNM_MAJOR   200
#define MAX_FILE_SIZE 2

struct pardevice *dev = NULL;

static loff_t burnm_lseek(struct file *file,loff_t offset,int origin)
{
  switch(origin)
  {
    case 0:
      if(offset < MAX_FILE_SIZE)
      {
        file->f_pos = offset;
        return file->f_pos;
      }
    case 1:
    default:
      return -EINVAL;
  }
}

static ssize_t burnm_read(
  struct file *file,char *buf,size_t length,loff_t *ppos)
{
  unsigned char val;

  parport_claim(dev);
  val = parport_read_status(dev->port);
  parport_release(dev);

  if(copy_to_user(buf,&val,sizeof(unsigned char)))
    return -EFAULT;

  return 1;
}

static ssize_t burnm_write(
  struct file *file, const char *buf,size_t count,loff_t *ppos)
{
  register int ret = 1;
  unsigned char val;

  if(copy_from_user(&val,buf,sizeof(unsigned char)))
    return -EFAULT;

  parport_claim(dev);
  if(file->f_pos == 0)
    parport_write_data(dev->port,*((unsigned char *)&val));
  else if(file->f_pos == 1)
    parport_write_control(dev->port,*((unsigned char *)&val));
  else
    ret = -EINVAL;
  parport_release(dev);

  return ret;
}

static int burnm_open(struct inode *inode,struct file *file)
{
  MOD_INC_USE_COUNT;
  return 0;
}

static int burnm_release(struct inode *inode,struct file *file)
{
  MOD_DEC_USE_COUNT;
  return 0;
}

/**************************************************************************
* init code
**************************************************************************/
static struct file_operations burnm_fops =
{
  burnm_lseek,
  burnm_read,
  burnm_write,
  NULL,        /*readdir*/
  NULL,        /*poll*/
  NULL,        /*ioctl*/
  NULL,        /*mmap*/
  burnm_open,
  NULL,        /*flush*/
  burnm_release,
  NULL,        /*fsync*/
  NULL,        /*fasync*/
  NULL,        /*check_media_change*/
  NULL,        /*revalidate*/
  NULL         /*lock*/
};

static char *parport = NULL;
MODULE_PARM(parport, "1-" __MODULE_STRING(1) "s");

int burnm_init(struct parport *port)
{
  /*
   * force irq not to be inited. we won't need it
   */
  port->irq = PARPORT_IRQ_NONE;

  dev = parport_register_device(port,"burnm",NULL,NULL,NULL,0,(void *)dev);
  if(dev == NULL)
  {
    printk(KERN_INFO "burnm: no device found.\n");
    return -ENODEV;
  } 

  if(register_chrdev(BURNM_MAJOR,"burnm",&burnm_fops))
  {
    printk("burnm: unable to get major %d.\n",BURNM_MAJOR);
    return -EIO;
  }

  printk(KERN_INFO "burnm: using %s.\n",port->name);
  return 0;
}

int init_module(void)
{
  struct parport *port;

  printk(KERN_ERR "burnm: version " BURNM_VER ", By Jason Nunn.\n");

  port = parport_enumerate();
  if(port != NULL)
  {
    if(!parport)
      return burnm_init(port);
    else
    {
      char *ep;
      unsigned long r;

      r = simple_strtoul(parport,&ep,0);
      if(ep != parport)
        for(;;)
        {
          if(port == NULL)
            break;

          if(r == port->number)
            return burnm_init(port);

          port = port->next;
        }
    }
  }

  printk(KERN_ERR "burnm: Device not found.\n");
  return -ENODEV;
}

void cleanup_module(void)
{
  unregister_chrdev(BURNM_MAJOR,"burnm");
  if(dev != NULL)
    parport_unregister_device(dev);

  printk(KERN_ERR "burnm: uninstalled.\n");
}
