/********************************************************************

    File:       ups.c

    Purpose:    Read UPS status.

                Copyright 1996, Bob Hauck
                Copyright 1998, Michael Robinton
                
                This program is free software; you can redistribute it
                and/or modify it under the terms of the GNU General Public
                License as published by the Free Software Foundation;
                either version 2 of the License, or (at your option) any
                later version.

                This program is distributed in the hope that it will be
                useful, but WITHOUT ANY WARRANTY; without even the implied
                warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
                PURPOSE.  See the GNU General Public License for more details.

                You should have received a copy of the GNU General Public
                License along with this program; if not, write to the Free
                Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
                USA.

    Language:   GCC 2.7.0

    Author:     Bob Hauck
    Modified:   Michael Robinton
    Modified:   Russ Magee                       

    Revision 1.4  2003/07/21 18:56:00  bobh
    Add <stdlib.h>

    Revision 1.3  2000/06/09 14:54:00  Michael Robinton/Russ Magee    
    Integrate Russ Magee's pin remapping contribution.        
    RM..            
    Removed pin assignment #defines -- now read from upsd.conf                
    Changed all code to use logical UPS lines, supported by mapping.c,h        

    Revision 1.2  1998/05/22 10:39:00  miker
    Add logging status string VERSION.
    UPS_Setup UPS kill needs to be in the OFF
    state after an attempt is made to kill the power.
    This is a pulsed signal.
    
    Revision 1.1  1996/11/23 16:45:12  bobh
    Initial revision

**********************************************************************/
#include <sys/ioctl.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "common.h"
#include "mapping.h"
#include "ups.h"

/*  Status strings for logging. The VERSIONs don't have to match
 *  since the VERSION message is sent but never decoded by NET_CHECK */
char *StatusString [] =
{
    "OK",
    "ON BATTERY",
    "LOW BATTERY",
    "UPS_ERROR",
    VERSION
};

/*-------------------------------------------------------------------*
     UPS_Open

     Open the UPS monitor device (serial port).

     Parameters:  DevName - Name of the device to open.

     Returns:     File descriptor that was opened.
 *-------------------------------------------------------------------*/
int UPS_Open (char *DevName)
    {
    int fd;
    
    if ((fd = open (DevName, O_RDWR | O_NDELAY)) < 0)
        {
        LogError ("opendev", strerror (errno));
        }

    return fd;
    }


/*-------------------------------------------------------------------*
     UPS_Setup

     Set up the port for monitoring.  Optionally kill the UPS
     power (only works if the UPS is on battery).

     Parameters:  fd   - File descriptor of monitor device.
                  kill - TRUE to shut down UPS power.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void UPS_Setup (int fd, int kill)
{

  /* Most UPS units need power from the PC for status lines so we set
     it here */
  UPS_Assert(fd, L_UPSPWR);

  if (!kill)
  {
    UPS_Deassert(fd, L_KILL);
    sleep (1);
  }
  if (kill)
  {
    LogError ("kill", "Attempting to kill the power!\n");
    UPS_Deassert(fd, L_KILL);
    sleep (5);
    UPS_Assert(fd, L_KILL);
    sleep (5);
    UPS_Deassert(fd, L_KILL);
    sleep (1);

    /* Hmmm..... If you have a power outage, you won't make it!
     */
    exit (0);
  }
}


/*-------------------------------------------------------------------*
     UPS_Check

     Check the status of the UPS (Ok, On Battery, or Low Battery).
     
     Parameters:  fd   - File descriptor of monitor device.

     Returns:     S_* status code.
 *-------------------------------------------------------------------*/
int UPS_Check (int fd)
{
  int flags;
  int status;

  flags = UPS_Read(fd);
  /*  printf("LogStatus: %04x\n", flags); fflush(stdout);*/
  status = (flags & (B_ONBAT | B_LOBAT));

  if ((flags & B_ERR) == 0)
  {
    return S_ERROR;        /* bad UPS cable? */
  }

  switch (status)
  {
    /*  On battery, but battery ok
     */
  case B_ONBAT:
    status = S_ONBAT;
    break;

    /*  On battery, and battery is low
     */
  case (B_LOBAT | B_ONBAT):
    status = S_LOBAT;
    break;

    /*  Everything is a-ok
     */
  default:
    status = S_OK;
  }

  return status;
}
