/********************************************************************

    File:       net.c

    Purpose:    Support for remote monitoring of a UPS over the
                network.  Expects another copy of upsd to be running
                on the "master" server (the one connected to the
                UPS).
                
                Copyright 1996, Bob Hauck
                Copyright 1998,2000, Michael Robinton
                
                This program is free software; you can redistribute it
                and/or modify it under the terms of the GNU General Public
                License as published by the Free Software Foundation;
                either version 2 of the License, or (at your option) any
                later version.

                This program is distributed in the hope that it will be
                useful, but WITHOUT ANY WARRANTY; without even the implied
                warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
                PURPOSE.  See the GNU General Public License for more details.

                You should have received a copy of the GNU General Public
                License along with this program; if not, write to the Free
                Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
                USA.

    Language:   GCC 2.7.0

    Author:     Bob Hauck
    Modified:   Michael Robinton

    Revision 1.9  2003/07/21 18:56:00 bobh
    Add <stdlib.h> for exit(), leave the socklen_t warning until I can
    figure out if there are any Linuxes left that don't have it in libc.

    Revision 1.8  2000/06/08 15:06:00 mikr
    Add some socket.h definitions that are not present in 
    older libraries, used only in this module
    
    Revision 1.7  2000/06/07 08:25:00 mikr
    Revise NET_Check ReadLine, and TCPOpen to use non-blocking I/O so that 
    socksified operation works properly.

    Revision 1.6  1998/05/23 14:51:00  mikr
    Change strcmp to strncmp in NET_Check to limit comparison for bogus 
    received strings.
    Add latch to snuff repeated connect fails to down server in TCPOpen.
    Add 'close (fd)' to TCPOpen to remove stale sockets clogging kernel.
    Revise NET_Check to log bogus server messages.
    Change 'Status' to 'NetStatus' to reflect usage.

    Revision 1.5  1997/01/20 22:35:48  bobh
    Fix header order for Alpha systems.  Clean up assorted warnings.

    Revision 1.4  1996/12/15 02:08:01  bobh
    Clean up some of the comments.

    Revision 1.3  1996/12/03 03:08:56  bobh
    Handle SIGINT, use waitpid() on SIGTERM to make sure we
    get an zombies, use SO_REUSEADDR on the server socket.

    Revision 1.2  1996/11/24 18:27:09  bobh
    Forgot to handle interrupted system calls...we will see
    those when children die.

    Revision 1.1  1996/11/23 16:45:12  bobh
    Initial revision

**********************************************************************/
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include "common.h"
#include "ups.h"

#ifndef SHUT_RDWR
enum
{
  SHUT_RD = 0,          /* No more receptions.  */
#define SHUT_RD         SHUT_RD
  SHUT_WR,              /* No more transmissions.  */
#define SHUT_WR         SHUT_WR   
  SHUT_RDWR             /* No more receptions or transmissions.  */
#define SHUT_RDWR       SHUT_RDWR
};
#endif      

#ifndef socklen_t
typedef unsigned int socklen_t;
#endif

#ifndef INADDR_NONE
#define INADDR_NONE 0xffffffff
#endif

#ifndef INADDR_ANY
#define INADDR_ANY 0x00000000
#endif

#ifndef htons
extern unsigned short htons (unsigned short);
#endif

#ifndef htonl
extern unsigned long  htonl (unsigned long);
#endif

/*  File descriptor for "master" mode.
 */
static int ServerFD = -1;


/*-------------------------------------------------------------------*
     SigTerm

     Handle SIGTERM, SIGINT and SIGCHLD.

     Parameters:  sig - Signal number being received.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void SigTerm (int Sig)
    {
    if (Sig == SIGCHLD)
        {
        signal (SIGCHLD, SigTerm);
        while (waitpid (-1, NULL, WNOHANG) > 0)
            ;
        }
    else
        {
        if (ServerFD >= 0)
            {
            close (ServerFD);
            exit (0);
            }
        }
    }


/*-------------------------------------------------------------------*
     TCPOpen

     Open a client-mode socket to communicate with the master
     server.  Local function, not exported.

     Parameters:  HostAddr - Remote host address in network format.
                  Port     - Remote port number to connect.
                  time_out - connect failure time out

     Returns:     The open file descriptor or -1 on error.
 *-------------------------------------------------------------------*/
static int TCPOpen (unsigned long HostAddr, unsigned short Port, int time_out)
    {
    static int		snuff_msg = 1;
    int			flags, n, error, sts, fd;
    socklen_t		len;
    fd_set		rset;
    struct sockaddr_in	server_addr;
    struct timeval	tval;

    bzero (&server_addr, sizeof (server_addr));
    server_addr.sin_family = AF_INET;

    if (Port <= 0)
        {
        LogError ("tcp_open", "must specify a port");
        return -1;
        }

    server_addr.sin_port = htons (Port);
    if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
        {
        if (snuff_msg <= 1)
            {
            ++snuff_msg;
            LogError ("tcp_open", "can't create socket");
            }
         return -1;
        }

    sts = 1;    
    setsockopt (fd, SOL_SOCKET, SO_REUSEADDR, &sts, sizeof (sts));
    bcopy (&HostAddr, &server_addr.sin_addr, sizeof (HostAddr));

/**************** socket open, process connect non-blocking *********/
    
    flags = fcntl (fd, F_SETFL, 0);
    fcntl (fd, F_SETFL, flags | O_NONBLOCK);    
    error = 0;
    if ((n = connect (fd,
                 (struct sockaddr *) &server_addr,
                 sizeof (server_addr)) < 0))
        {
        if (errno != EINPROGRESS )
            goto tcp_errx;
        }

    if (n == 0)
        goto done;
    
    FD_ZERO (&rset);
    FD_SET (fd, &rset);
    tval.tv_sec = time_out;
    tval.tv_usec = 0;
    
    if ((n = select (fd + 1, &rset, NULL, NULL, &tval)) == 0 )
    	{
    	shutdown ( fd, SHUT_RDWR );
    	errno = ETIMEDOUT;
    	goto tcp_errn;
    	}
    	
    if (FD_ISSET(fd, &rset))
        {
        len = sizeof (error);
        if (getsockopt (fd, SOL_SOCKET, SO_ERROR, &error, &len) < 0)
            goto tcp_errx;
        } else
            goto tcp_errx;
    
  done:
    fcntl (fd, F_SETFL, flags);
    if (error)
        {
        shutdown ( fd, SHUT_RDWR );
        errno = error;
        goto tcp_errn;
        }
            
    if (snuff_msg)
        {
        snuff_msg = 0;
        LogError ("tcp_open", "connected to server");
        }
    return fd;

  tcp_errx:
    shutdown ( fd, SHUT_RDWR );
  tcp_errn:
/* just in case */
    close (fd);
    if (snuff_msg <= 1)
        {
        ++snuff_msg;
        LogError ("tcp_open", "can't connect to server");
        }
    return -1;
    }


/*-------------------------------------------------------------------*
     ReadLine

     Read a line of data from a remote host.  Local function, not
     exported.

     Parameters:  FD     - File descriptor of socket to read.
                  Buffer - Buffer in which to place data.
                  Len    - Length of Buffer.
                  time_out - connect failure time out

     Returns:     Count of bytes read.
 *-------------------------------------------------------------------*/
static int ReadLine (int FD, char *Buffer, int Len, int time_out)
    {
    int			rc;
    char		c;
    int			i;
    fd_set		rset;
    struct timeval	tval;
    
    tval.tv_sec = time_out;
    tval.tv_usec = 0;

    for (i = 0; i < Len; ++i)
        {
        FD_ZERO (&rset);
        FD_SET (FD, &rset);
        if ((select(FD + 1, &rset, NULL, NULL, &tval)) == 0)
            return -1;
            
        if ((rc = read (FD, &c, 1)) == 1)
             {
             *Buffer++ = c;
             if (c == '\n')
                 break;
             }
        else if (rc == 0)
            {
            if (i == 1)
                return 0;
            else
                break;
            }
        else
            {
            return -1;
            }
        }
    
    *Buffer = 0;
    return i;
    }


/*-------------------------------------------------------------------*
     NET_GetServerAddr

     Translate server name or IP in dotted-quad notation to an
     internet address in network format.

     Parameters:  Server - Server name or IP.

     Returns:     Server address or IN_ADDRNONE on failure.
 *-------------------------------------------------------------------*/
unsigned long NET_GetServerAddr (char *Server)
    {
    struct hostent *hp;
    unsigned long  inaddr;
    
    if ((inaddr = inet_addr (Server)) == INADDR_NONE)
        {
        if ((hp = gethostbyname (Server)) == NULL)
            {
            LogError ("get_addr", "host name error");
            return -1;
            }

        bcopy (hp->h_addr, (char *)&inaddr, hp->h_length);
        }

    return inaddr;
    }


/*-------------------------------------------------------------------*
     NET_Check

     Get status of a remote UPS.

     Parameters:  Server - Server address in network format.
                  Port   - Remote IP port to connect to.
                  time_out - connect failure time out

     Returns:     S_* status code.
 *-------------------------------------------------------------------*/
int NET_Check (unsigned long Server, int Port, int time_out)
    {
    char       buffer [64];
    int        i;
    int status = S_NOCHANGE;
    int fd     = TCPOpen (Server, Port, time_out);

    if (fd >= 0)
        {
        if (ReadLine (fd, buffer, sizeof (buffer), time_out) > 0)
            {
            for (i = S_OK; i <= S_ERROR; ++i)
                {
                if (strncmp (buffer, StatusString [i], sizeof (buffer)) == 0)
                    break;
                }

            if (i > S_ERROR)
                LogError ( "netcheck", "Bogus message from unknown sender" );
            else
                status = i;
            } else
                LogError ( "netcheck", "Read timed out" );

    	shutdown ( fd, SHUT_RDWR );
	close (fd);
        }
    else
        LogError ("netcheck", "Failed to connect" );

    return status;
    }


/*-------------------------------------------------------------------*
     NET_Serve

     Provide UPS status to remote hosts on the network.
     
     Parameters:  Port    - Port number to listen on.
                  MaxWait - Max time to wait for connection in secs.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void NET_Serve (int Port, int MaxWait)
    {
    fd_set         fdset;
    size_t         cli_len;
    int            newfd;
    int            sts;
    int            child;
    long           endtime;
    struct timeval timeout;
    static struct sockaddr_in serv_addr;
    static struct sockaddr_in cli_addr;

    if (ServerFD < 0)
        {
        /*  No socket is open yet
         */
        bzero (&serv_addr, sizeof (serv_addr));
        serv_addr.sin_family      = AF_INET;
        serv_addr.sin_addr.s_addr = htonl (INADDR_ANY);
        serv_addr.sin_port        = htons (Port);
        
        if ((ServerFD = socket (AF_INET, SOCK_STREAM, 0)) < 0)
            {
            LogError ("serve", "can't open server socket");
            exit (1);
            }

        sts = 1;
        
        setsockopt (ServerFD, SOL_SOCKET, SO_REUSEADDR, &sts, sizeof (sts));
        
        if (bind (ServerFD,
                  (struct sockaddr *) &serv_addr,
                  sizeof (serv_addr))
            < 0)
            {
            LogError ("serve", "can't bind server socket");
            close (ServerFD);
            ServerFD = -1;
            exit (1);
            }

        /*  Set up signal handlers and listen on the open socket.
         */
        listen (ServerFD, 5);
        signal (SIGCHLD, SigTerm);
        signal (SIGTERM, SigTerm);
        signal (SIGINT,  SigTerm);        
        }

    /*  Wait for a client to connect.  Don't wait longer than
     *  MaxWait seconds.
     */
    endtime = MaxWait + time (NULL);
    sts     = 1;

    while (sts > 0)
        {
        FD_ZERO (&fdset);
        FD_SET  (ServerFD, &fdset);

        timeout.tv_sec  = endtime - time (NULL);
        timeout.tv_usec = 0;

        if (timeout.tv_sec < 0)
            break;
        else
            sts = select (ServerFD + 1, &fdset, 0, 0, &timeout);

        if (sts < 0 && errno == EINTR)
            {
            /*  Interrupted system call...probably a child died.
             */
            sts = 1;
            continue;
            }
        else if (sts > 0)
            {
            /*  We have a client...fork and send the data.
             */
            cli_len = sizeof (cli_addr);
            newfd = accept (ServerFD, (struct sockaddr *) &cli_addr, &cli_len);
            child = fork ();
            
            if (child > 0)
                {
                close (newfd);
                }
            else if (child < 0)
                {
                LogError ("serve", "fork error");
                return;
                }
            else
                {
                write (newfd,
                       StatusString [NetStatus],
                       strlen (StatusString [NetStatus]));
        	close (newfd);        
                exit (0);
                }
            }
        }
    
    return;
    }
