/*
 * Copyright (C) 1996, Tom Bjorkholm  <tomb@mydata.se>
 *
 * 
 * upsd.c -- a program to interact with the (Powerbox) UPS
 *
 *
 * Compile with: gcc -O2 -g -Wall -static -o upsd upsd.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

static char usage[] =
 " Usage:\n"
 " ======\n"
 "         upsd device fail_action return_action battery_low_action\n"
 "   or    upsd device shutdown\n"
 "\n"
 " where:\n"
 "      device ......: /dev/cua1 or whatever serial port device file to use\n"
 "                     to communicate with the UPS\n"
 "      fail_action..: bash script to run when the net power fails\n"
 "                     i.e. to start a controlled shutdown\n"
 "      return_action: bash script to run when the net power returns\n"
 "                     i.e. to cancel the shutdown\n"
 " battery_low_action: bash script to run when the UPS reports\n"
 "                     \'battery low\'. i.e. \'express powerdown\'.\n"
 "      shutdown.....: shut down UPS (do not become daemon) \n"
 "\n"
 " NOTE: Sending a SIGPWR to the upsd process will turn off the battery\n"
 "       backup.\n"
 "\n"
 " Documentation:\n"
 " ==============\n"
 "\n"
 " Nine pin serial port: 1 Carrier Detect\n"
 "                       2 Receive Data\n"
 "                       3 Transmit Data\n"
 "                       4 Data Terminal Ready\n"
 "                       5 Signal Ground\n"
 "                       6 Data Set Ready\n"
 "                       7 Request To Send\n"
 "                       8 Clear To Send\n"
 "                       9 Ring Indicator\n"
 "\n"
 " Powerbox UPS pinout truth table:\n"
 "              Pin:  1,2    4,8    2,6   4,9    3,7    \n"
 " Normal.........:  open   open   close close   0 V\n"
 " Back-up........:  close  open   open  close   0 V\n"
 " Battery low....:         close        open    0 V\n"
 " Shutdown signal:                              5-12 V\n"
 "\n"
 " Cable:\n"
 "  UPS side:                       Computer Side:\n"
 "  (power fail.) 2 ----------------- 6 DTS\n"
 "                1 -+--------------- 4 DTR (used as 5 V supply)\n"
 "                8 -+\n"
 "                6 -+--------------- 5 Ground\n"
 "                9 -+\n"
 "  (turn off...) 3 -+\n"
 "  (turn off...) 7 ----------------- 7 RTS\n"
 "  (battery low) 4 ----------------- 8 CTS \n"
 "\n"
 " Power fail (2 = computer 6, DTS) is high if net power failes\n"
 "                                  is grounded if net power is OK\n"
 " Battery low (4 = computer 8, CTS) is high if battery is low\n"
 "                                   is low if battery is OK\n"
 " Turn off 3 ( = computer 5, Ground) is ground for turn off signal\n"
 " Turn off 7 ( = computer 7, RTS) is pulsed high to turn off UPS.\n"
 "\n"
 "ABSOLUTELY NO WARANTY\n"
 "Copying policy: GPL\n";

/* Do not change these defines */
#define MCR_DTR  0x01
#define MCR_RTS  0x02
#define MSR_DCTS 0x01
#define MSR_DDSR 0x02
#define MSR_DRI  0x04
#define MSR_DDCD 0x08
#define MSR_CTS  0x10
#define MSR_DSR  0x20
#define MSR_RI   0x40
#define MSR_DCD  0x80


/*
 * Configurable define section starts.
 */

/* Line to use as +5 V supply */
#define PLUS5V MCR_DTR

/* Polarity of +5 V bit */
#define PLUS5V_ACTIVE 0xff  /* 0xff or 0x0 */

/* Shut down signal */
#define SHUTDOWN MCR_RTS

/* Shut down signal polarity */
#define SHUTDOWN_ACTIVE 0x0    /* 0xff or 0x0 */

/* Shut down signal pulse length in microsesconds */
#define SHUTDOWN_PULSE 32000

/* What's a good state: i.e. net power OK */
/* good state if: (Modem_status & GOOD_STATE_MASK) == GOOD_STATE_RESULT */
#define GOOD_STATE_MASK (MSR_DSR | MSR_CTS)
#define GOOD_STATE_RESULT 0

/* What's a net fail state: i.e. net power failed, UPS backup working */
/* fail state if: (Modem_status & FAIL_STATE_MASK) == FAIL_STATE_RESULT */
#define FAIL_STATE_MASK (MSR_DSR)
#define FAIL_STATE_RESULT (MSR_DSR)

/* What's a battery low state: i.e. network not ok, battery will fail soon */
/* battery low if: (Modem_status & BATTERY_LOW_MASK) == BATTERY_LOW_RESULT */
#define BATTERY_LOW_MASK  (MSR_CTS)
#define BATTERY_LOW_RESULT (MSR_CTS)

/* Action to take on undefine state: i.e. state not defined above */
/* action 0 = As good state */
/* action 1 = As a net power fail */
/* action 2 = As battery low */
#define UNDEFINED_STATE_ACTION 1

/* When can we trust to get an interrupt */
#define IRQ_FROM_NORMAL 1  /* If state changes from normal we get an irq */
#define IRQ_FROM_FAIL   0
#define IRQ_FROM_BATTERY_LOW 0

/*
 * Configurable define section ends.
 */

#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<linux/tty.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <asm/io.h>
#include <syslog.h>
#include <signal.h>
#include <sys/mman.h>

#define __KERNEL__
#include<linux/serial.h>

#define MSR (base + 6)
#define MCR (base + 4)

static int base = 0;

void alarm_handler(int unused)
{
    (void) unused;
    signal(SIGALRM, alarm_handler);    
}

void shutoffups(int unused)
{
    unsigned char shut;
    unsigned char mc;
    int i;
    (void) unused;
    signal(SIGPWR,shutoffups);
    mc = inb(MCR);
    syslog(LOG_EMERG," shutting down ups\n");
    usleep(50000);
    for(i=0; i<5; i++) {
	if (SHUTDOWN_ACTIVE) {
	    mc &= (~(SHUTDOWN));
	    shut = mc | (SHUTDOWN);
	} else {
	    mc |= (SHUTDOWN);
	    shut = mc & (~(SHUTDOWN));
	}
	outb(shut, MCR);
	usleep(SHUTDOWN_PULSE);
	outb(mc, MCR);
	usleep(SHUTDOWN_PULSE);
    }
}

void plus5v_on(void)
{
    unsigned char mc=inb(MCR);
    if (SHUTDOWN_ACTIVE) {
	mc &= (~(SHUTDOWN));
    } else {
	mc |= (SHUTDOWN);
    }
    if (PLUS5V_ACTIVE) {
       mc |= (PLUS5V);
    } else {
       mc &= (~(PLUS5V));
    }
    outb(mc, MCR);
}

/* return state 0 = good, 1 = fail, 2 battery low */
int what_state(void)
{
    unsigned char mc=inb(MSR);
    if ((mc & (BATTERY_LOW_MASK)) == (BATTERY_LOW_RESULT)) return 2;
    if ((mc & (FAIL_STATE_MASK)) ==  (FAIL_STATE_RESULT)) return 1;
    if ((mc & (GOOD_STATE_MASK)) ==  (GOOD_STATE_RESULT)) return 0;
    return UNDEFINED_STATE_ACTION;
}

void exe(char *command)
{
    int pid=fork();
    syslog(LOG_NOTICE, "executing: %s\n", command);
    if (pid < 0) {
	syslog(LOG_EMERG, "fork() failed\n");
	sleep(5);
	exe(command);
    } else {
	if (pid == 0) { /* child */
	    execlp("bash", "bash", command, 0);
	}
    }
}

void action(int to_state, char **argv)
{
    char *mes[] = {"Power has returned", "Power failure", "Battery low"};
    syslog(LOG_ALERT,mes[to_state]);
    switch(to_state) {
    case 0: exe(argv[3]); break;
    case 1: exe(argv[2]); break;
    case 2: exe(argv[4]); break;
    default: syslog(LOG_ALERT,"Internal error: action(%d)\n",to_state); break;
    }
}

void ups_wait(int state, int fd)
{
    int irq_ok = 0;
    switch(state) {
    case 0: irq_ok = IRQ_FROM_NORMAL; break;
    case 1: irq_ok = IRQ_FROM_FAIL; break;
    case 2: irq_ok = IRQ_FROM_BATTERY_LOW; break;
    default: syslog(LOG_ALERT,"Internal error: ups_wait(%d)\n",state); break;
    }
    if (irq_ok) {
	/* we do not really trust the interrupt, but can poll much slower */
	alarm(120);
	ioctl(fd,TIOCMIWAIT,TIOCM_RNG | TIOCM_DSR | TIOCM_CD | TIOCM_CTS);
	alarm(0);
    } else {
	sleep(2);
    }
}

void daemonify(void)
{
    int pid, i;
    int stack_user[1000]; /* grab some extra memory, before we mlockall */
    chdir("/");
    if ((pid = fork()) < 0) {
	fprintf(stderr,"upsd: could not fork, exiting\n");
	exit(1);
    }
    if (pid) exit(0);  /* parent exiting */
    for (i = 0; i < 3; i++) close(i); 
    setpgrp(); 
    pid = fork(); /* might fail, which is not good, but it will still work */
    if (pid > 0) exit(0);
    /* we need to be in memory, the disk might be unmounted */
    stack_user[1]=15;
    stack_user[500]=15;
    mlockall(MCL_CURRENT | MCL_FUTURE);
    /* locking might fail, but there is nothing we can do about it */
    /* so there is no use in checking :-/ */
}

int sanity(int argc, char **argv)
{
    /* we do a sanity check while we can write to stderr and tell the user */
    int fd, base, shutdown = 0;
    struct serial_struct ss;
    openlog("upsd", LOG_CONS, LOG_DAEMON); 
    if ((argc != 3) && (argc != 5)) {
	fprintf(stderr,"%s\n", usage);
	syslog(LOG_ALERT, "bad command line, exiting\n");
	closelog();
	exit(1);
    }
    if (argc == 3) {
	shutdown = 1;
	if (strcmp(argv[2],"shutdown")) {
	    fprintf(stderr,"%s\n", usage);
	    syslog(LOG_ALERT, "bad command line, exiting\n");
	    closelog();
	    exit(1);
	}
    }
    if (getuid()) {
	fprintf(stderr,"upsd: Can only run as root. Exiting\n");
	syslog(LOG_ALERT,"Can only run as root. Exiting\n");
	exit(1);
    }
    if ((fd=open(argv[1],O_RDWR)) < 0) {
	fprintf(stderr,"upsd: Could not open %s \n", argv[1]);
	perror("upsd: exiting");
	syslog(LOG_ALERT,"Could not open %s \n", argv[1]);
	syslog(LOG_ALERT,"%m -- exiting\n");
	closelog();
	exit(1);	
    }
    if (ioctl(fd,TIOCGSERIAL, &ss)) {
	perror("upsd: ioctl(TIOCSERGSTRUCT) failed -- exiting");
	syslog(LOG_ALERT,"ioctl(TIOCSERGSTRUCT) failed:%m -- exiting\n");
	closelog();
	exit(1);	
    }
    base=ss.port;
    if(ioperm(base,8,1)) {
	perror("upsd: ioperm() failed -- exiting\n");
	syslog(LOG_ALERT,"ioperm() failed:%m -- exiting\n");
	closelog();
	exit(1);	
    }
    close(fd);
    closelog();
    return shutdown;
}

int main(int argc, char **argv)
{
    int fd;
    int state, oldstate;
    int shutdown = 0;
    struct serial_struct ss;
    shutdown = sanity(argc, argv);
    if (!shutdown) daemonify();
    openlog("upsd", LOG_CONS, LOG_DAEMON); 
    if ((fd=open(argv[1],O_RDWR)) < 0) {
	syslog(LOG_ALERT,"Could not open %s \n", argv[1]);
	syslog(LOG_ALERT,"%m -- exiting\n");
	return 1;	
    }
    if (ioctl(fd,TIOCGSERIAL, &ss)) {
	syslog(LOG_ALERT,"ioctl(TIOCSERGSTRUCT) failed:%m -- exiting\n");
	return 1;	
    }
    base=ss.port;
    if(ioperm(base,8,1)) {
	syslog(LOG_ALERT,"ioperm() failed:%m -- exiting\n");
	return 1;	
    }
    syslog(LOG_NOTICE, " ups serial port at base 0x%x", base);
    if (shutdown) {
	shutoffups(0);
	return 0;
    }
    signal(SIGPWR,shutoffups);
    signal(SIGCHLD,SIG_IGN);
    signal(SIGALRM,alarm_handler);
    plus5v_on();
    state = what_state();
    if (state) action(state, argv);
    for (;;) {
	oldstate = state;
	state = what_state();
	if (state != oldstate) action(state,argv);
	ups_wait(state, fd);
    }
    close(fd);
    return 0;
}
