/*
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *               BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *            -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 *  Parts of the code are from Miquel van Smoorenburg's powerd.c
 *  Other parts are original from Christian Holtje <docwhat@uiuc.edu>
 *  Smart-UPS code from Pavel Korensky's apcd.c, apcd.h, and upsnet.c.
 *  I believe that it is okay to say that this is Public Domain, just
 *  give credit, where credit is due.
 *
 *  Disclaimer:  We make NO claims to this software, and take no
 *               resposibility for it's use/misuse.
 *
 *  Cable for a smarter Back-UPS and SmartUPS from APC only.
 *
 *  Computer Side   |  Description of Cable           |  UPS Side
 *  DB9f  |  DB25f  |                                 |    DB9m
 *   4    |   20    |  DTR (5vcc)             *below  |    n/c
 *   8    |    5    |  CTS (low-batt)         *below  |     5
 *   2    |    3    |  RxD (other line-fail)  *below  |     3
 *   5    |    7    |  Ground (Signal)                |     4
 *   1    |    8    |  CD (line-fail from ups)        |     2
 *   7    |    4    |  RTS (shutdown ups)             |     1
 *  n/c   |    1    |  Frame/Case Gnd (optional)      |     9
 *
 *  The "*below" one needs solder type DB connectors with standard
 *  hoods, two (2) 4.7K ohm 1/4 watt %5, one (1) foot of 3/32" (inch)
 *  shrink wrap, rosin core solder, and three (3) to five (5) feet of
 *  22AWG multi-stranded five (5) conductor cable. 
 *
 *  9/20/96 2:00am
 *
 */

/*
 * Add here an identifer that is unique for tracking changes
 * made by each person other than myself.
 *
 * Christopher J. Reimer <reimer@doe.carleton.ca>       CJR
 * Werner Panocha <wpanocha@t-online.de>                w.p.
 * Brian Schau <bsc@fleggaard.dk>                       BSC
 * Marko Sakari Asplund <Marko.Asplund@cern.ch>         MSA
 * Riccardo Facchetti <fizban@tin.it>			-RF
 * Jonathan H N Chin <jc254@newton.cam.ac.uk>		JHNC
 */

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pwd.h>

#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#undef __DEBUG
#undef VERBOSE_THREADS

int master_present;
int socketfd;
int newsocketfd;

UPSINFO myUPS;
struct sockaddr_in my_adr;
int masterlen;
struct sockaddr_in master_adr;
struct hostent *mastent;

int server_fifo_fd;
int client_fifo_fd;

FILE *procfile = NULL;  /* initial value set to NULL for terminate */
FILE *logfile = NULL;   /* initial value set to NULL for terminate */
FILE *nlgf = NULL;      /* initial value set to NULL for terminate */

time_t last_time_on_line;
time_t last_time_annoy;
time_t last_time_delay;
time_t last_time_procfs;
time_t last_time_logging;
time_t last_time_net;
time_t last_time_access;
time_t last_time_nologon;

int debug_net = FALSE;

int flags;
int killcount;

int le_bit   = TIOCM_LE;
int dtr_bit  = TIOCM_DTR;
int rts_bit  = TIOCM_RTS;
int st_bit   = TIOCM_ST;
int sr_bit   = TIOCM_SR;
int cts_bit  = TIOCM_CTS;
int cd_bit   = TIOCM_CD;
int rng_bit  = TIOCM_RNG;
int dsr_bit  = TIOCM_DSR;

struct termios oldtio;
struct termios newtio;

/*********************************************************************/
void powerfail (int ok)
{
	/*  If apcupsd terminates here, it will never get a chance to
	 *  report the event of returning mains-power.
	 *  I think apcupsd has no need to force terminate() by itself.
	 *  It will receive a SIGTERM from init, when system goes down.
	 *  This signal is trapped and will trigger apcupsd's terminate()
	 *  function.
	 *  (Was already reported/fixed in my 2.7.1 diff,)
	 *  (so I do it again now) w.p.

	if (ok == 2) terminate(0); }

	 *  Closes procfile and logfile to preserve information.
	 */

	if (ok == 2)
		clear_files();

	/*
	 *  The network slaves apcupsd needs to terminate here for now.
	 *  This sloppy, but it works. If you are networked, then the
	 *  master must fall also. This is required so that the UPS
	 *  can reboot the slaves.
	 */

	if (ok == 3)
		terminate(0);
}
	
/*********************************************************************/
void logonfail (int ok)
{
	int lgnfd;

	unlink(NOLOGIN);
	if (ok == 0 && ((lgnfd = open(NOLOGIN, O_CREAT|O_WRONLY, 0644)) >= 0)) {
		write(lgnfd, POWERFAIL "\n", sizeof(POWERFAIL));
		close(lgnfd);
	}
}

/*********************************************************************/
int make_file (const char *s)
{
	int makefd;

	if ((makefd = open(s,O_CREAT|O_WRONLY, 0644)) >= 0)
		close(makefd);
	return makefd;
}

/*********************************************************************/
int remove_file (const char *s)
{
	unlink(s);
	return *s;
}

#ifdef NEW_THREADS
/**********************************************************************
 * the thread_terminate function and trapping signals allows threads
 * to cleanly exit.
 *********************************************************************/
void thread_terminate (int sig) {
	/*
	 * Before doing anything else restore the signal handlers,
	 * stopping daemon work.
	 *
	 * -RF
	 */

	restore_signals();

	detach_ipc();

	/*
	 * Nothing to do. This is a thread: cleanup is managed by the father
	 * process.
	 */
	_exit(0);
}

/*********************************************************************/
void clean_threads (void) {
	int i;
	int status;

	for (i=0; i<MAX_THREADS; i++) {
		/*
		 * Stop if there's no more childs.
		 */
		if (child_pid[i] == 0)
			break;

		/*
		 * Terminate the child.
		 */
		kill(child_pid[i], SIGTERM);

		/*
		 * Make sure the process is dead.
		 */
		waitpid(child_pid[i], &status, 0);
	}
}
#endif /* NEW_THREADS */

/**********************************************************************
 * the terminate function and trapping signals allows apcupsd
 * to exit and cleanly close logfiles, and reset the tty back
 * to its original settings. You may want to add this
 * to all configurations.  Of course, the file descriptors
 * must be global for this to work, I also set them to
 * NULL initially to allow terminate to determine whether
 * it should close them.
 *********************************************************************/
void terminate (int sig) {
	/*
	 * XXX - Is someone able to explain me why here we find a sleep for 10
	 * seconds ?
	 *
	 * -RF
	 */
	sleep(10);
#ifdef NEW_THREADS
	restore_signals();
#endif /* NEW_THREADS */
	if (logfile != NULL)
		logprintf("apcupsd exiting, signal %u\n", sig);

	clear_files();

	if (myUPS.fd != -1) {
		/*
		 * If the ups device is opened:
		 * 1 - Close it
		 * 2 - Delete lock file
		 *
		 * -RF
		 */
		tcflush(myUPS.fd, TCIFLUSH);
		tcsetattr(myUPS.fd, TCSANOW, &oldtio);
		close(myUPS.fd);
	}

#ifdef NEW_THREADS
#ifdef NEW_HTTP
	if (myUPS.enable_access.type == TRUE)
		httpcleanup();
#endif /* NEW_HTTP */
	clean_threads();
	destroy_ipc();
#endif /* NEW_THREADS */
	delete_lockfile(&myUPS);
	_exit(0);
}

/*********************************************************************/
void clear_files (void)
{
	if (logfile != NULL) {
		fflush(logfile);
		fclose(logfile);
	}
	if (procfile != NULL) {
		fflush(procfile);
		fclose(procfile);
	}
}

/*********************************************************************/
void make_pid (void)
{
	pid_t pid;
	FILE *pidf;

	if ((pidf = fopen(APCPID, "w")) != NULL) {
		fprintf(pidf, "%d\n", pid = getpid());
		fclose(pidf);
	}
}

/*********************************************************************/
/*                     Main program.                                 */
/*********************************************************************/
int main (int argc, char **argv)
{
	int KILL = FALSE;

	if ((getuid() != 0) && (geteuid() != 0)) {
		fprintf(stderr, "%s: needs superuser privileges.\n", argv[0]);
		exit(1);
	}

	if ((argc >= 2) && ((strcmp(argv[1], "-v") == 0) ||
	    (strcmp(argv[1], "--version") == 0))) {
		fprintf(stderr, "%s v%s Andre Hedrick et al.\n",
		        argv[0], APCUPSD_RELEASE);
		fprintf(stderr, "Usage: %s <option>\n\a", argv[0]);
		fprintf(stderr, "NOTE : %s ONLY one option at a time!!!\n\n", argv[0]);
		fprintf(stderr, "       killpower      :\n");
		fprintf(stderr, "       -d | --debug   :\n");
		fprintf(stderr, "       -v | --version :\n");
		goto error_out;
	} else if (argc >= 2  && ((strcmp(argv[1], "-d") == 0) ||
	           (strcmp(argv[1], "--debug") == 0))) {
		debug_net = TRUE;

#ifdef DEBUG
		fprintf(stderr, "%s: Built on %s at %s\n", argv[0], __DATE__, __TIME__);
#endif
		KILL = FALSE;
		if (check_for_config(argv[0], &myUPS) == FAILURE)
			goto error_out;

		/*
		 * We don't need to delete_lockfile() here.
		 * check_stale_lockfile() help us do it in create_lockfile() and
		 * delete_lockfile()
		 * -RF
		 */

		if ((myUPS.class.type == NETSLAVE) || (myUPS.class.type == NETMASTER)) {
			fprintf(stderr, "%s Net Debug mode enabled.\n\a", argv[0]);
		} else {
			fprintf(stderr, "%s Network mode not enabled, bye.\n\a", argv[0]);
			goto error_out;
		}
		if (myUPS.logtime == 0) {
			myUPS.logtime = 500;
			fprintf(stderr, "%s Net Debug requires a logtime--newtime: %d.\n\a", argv[0], myUPS.logtime);
		}
	} else if (argc >= 2  && (strcmp(argv[1], "killpower") == 0)) {
		KILL = TRUE;
		if (check_for_config(argv[0], &myUPS) == FAILURE)
			goto error_out;
		else
			delete_lockfile(&myUPS);
	} else if (argc >= 2) {
		fprintf(stderr, "Usage: %s <option>\n\a", argv[0]);
		fprintf(stderr, "NOTE : %s ONLY one option at a time!!!\n\n", argv[0]);
		fprintf(stderr, "       killpower      :\n");
		fprintf(stderr, "       -d | --debug   :\n");
		fprintf(stderr, "       -v | --version :\n");
		exit(1);
	} else {
		KILL = FALSE;
		if (check_for_config(argv[0], &myUPS) == FAILURE)
			goto error_out;
		else
			delete_lockfile(&myUPS);
	}

	if (!((myUPS.slave_count > 0) && (KILL == FALSE))) {
		debug_net = FALSE;
	}

	if (myUPS.logtime != 0)
		logfile = fopen(LOG_FILE,"a+");
	else
		unlink(LOG_FILE);

	if (myUPS.proctime != 0)
		procfile = fopen(UPS_STAT,"w");
	else
		unlink(UPS_STAT);

	master_present = 0;

	switch(myUPS.sharenet.type) {
		case DISABLE:
		case SHARE:
			if (setup_serial(&myUPS, KILL))
				goto error_out;
			break;          /* MSA the missing break error......core dump fix? */
		case NET:
			switch(myUPS.class.type) {
				case NO_CLASS:
				case STANDALONE:
				case SHARESLAVE:
				case SHAREMASTER:
				case SHARENETMASTER:
					break;
				case NETSLAVE:
					if ((KILL == 1) || (prepare_slave(&myUPS)))
						goto error_out;
					break;
				case NETMASTER:
					if (setup_serial(&myUPS, KILL))
						goto error_out;
					if ((KILL == 0) && (prepare_master(&myUPS)))
						goto error_out;
					break;
				default:
					fprintf(stderr, "%s: NET Class Error %s\n\a", argv[0], strerror(errno));
					goto error_out;
			}
			break;
		case SHARENET:
			if (setup_serial(&myUPS, KILL))
				goto error_out;
			if ((KILL == 0) && (prepare_master(&myUPS)))
				goto error_out;
			break;
		default:
			exit(1);
	}

	if (myUPS.fd != -1)
		kill_serial(&myUPS, KILL);
	else
		kill_net(&myUPS, KILL);

#ifdef NEW_THREADS
	if (init_ipc() == FAILURE) {
		logprintf("%s: init_ipc failed.\n", argv[0]);
		fprintf(stderr, "%s: init_ipc failed.\n", argv[0]);
		terminate(0);
	}
#endif /* NEW_THREADS */

	/*
	 *  Become a daemon.
	 */
	switch(fork()) {
		case 0:                       /* I am the child. */
			setsid();
			break;
		case -1:                      /* Failed to become daemon. */
			fprintf(stderr, "%s: can't fork.\n", argv[0]);
			exit(1);
		default:                     /* I am the parent. */
			exit(0);
	}

#ifdef NEW_THREADS
	logopen();
#endif /* NEW_THREADS */

	make_pid();

	/*
	 * Okay.
	 * We are a child and our father is just died without touching the lockfile
	 * that now is a stale lock file.
	 * We have to re-lock the device and is so simple.
	 * No problem in doing it here. All the lock functions check for APC_NET
	 * and now here is the conceptually best site to do the locking:
	 * just after the fork.
	 *
	 * Another way of doing the lockfile transition through the fork() could be:
	 *
	 * ...
	 * open_and_flock_lockfile();
	 * switch(fork()) { ... }
	 * update_unlock_and_close_lockfile();
	 * ...
	 *
	 * But this implies that we should write two more lock functions to get the
	 * same functionality of the create_lockfile() below.
	 * The main difference is that with the current code we do not "own" the
	 * lock file during the fork(), but this is IMHO not an issue because if
	 * someone stole it, we are anyway in troubles since someone other is
	 * willing to control the UPS device so it is better to bail out.
	 * On the other hand if we want to be fascist, we could implement the "flock
	 * the lockfile during fork()" way.
	 * -RF
	 */
	if (create_lockfile(&myUPS, KILL) == LCKERROR) {
		fprintf(stderr, "%s: failed to reacquire lock file on device %s\n", argv[0], myUPS.device);
		logprintf("%s: failed to reacquire lock file on device %s\n", argv[0], myUPS.device);
		terminate(0);
	}

#ifdef NEW_THREADS
	init_signals();
#endif /* NEW_THREADS*/

	if (myUPS.fd != -1) {
		prep_serial(&myUPS);
	}

#ifndef NEW_THREADS

	if (myUPS.fd != -1) {
		do_serial(&myUPS);
	} else {
		do_net(&myUPS);
	}

#else /* NEW_THREADS */

	/*
	 * Now update the shared area with myUPS data.
	 */
	write_shmarea(&myUPS);

#ifdef NEW_HTTP
	if (myUPS.class.type == NETSLAVE) {
		myUPS.enable_http.type = FALSE;
	}
#endif /* NEW_HTTP */

	/*
	 * From now ... we must _only_ start up threads!
	 * No more unchecked writes to myUPS because the threads must rely
	 * on write locks and up to date data in shared structure.
	 */

	if (myUPS.fd != -1) {
		start_thread(do_serial);
#ifdef VERBOSE_THREADS
		fprintf(stderr, "%s: start_thread(do_serial).\n", argv[0]);
#endif /* VERBOSE_THREADS */
	} else {
		start_thread(do_net);
#ifdef VERBOSE_THREADS
		fprintf(stderr, "%s: start_thread(do_net).\n", argv[0]);
#endif /* VERBOSE_THREADS */
	}

	if (myUPS.slave_count) {
		start_thread(do_slaves);
#ifdef VERBOSE_THREADS
		fprintf(stderr, "%s: start_thread(do_slaves).\n", argv[0]);
#endif /* VERBOSE_THREADS */
	}

	if (myUPS.reports) {
		/*
		 * Set up logging timers.
		 */
		if (logfile != NULL)
			time(&last_time_logging);
		/*
		 * Set up procfs timers.
		 */
		if (procfile != NULL)
			time(&last_time_procfs);
		start_thread(do_reports);
	}

#ifdef NEW_HTTP
	if (myUPS.enable_http.type == TRUE)
		start_thread(do_http);
#endif /* NEW_HTTP */

	wait_for_termination();
	terminate(0);
	logclose();
#endif /* NEW_THREADS */

	/*
	 *  Error! (shouldn't happen)
	 */
error_out:
	fprintf(stderr, "%s: Error! (shouldn't happen).\n", argv[0]);

        exit(1);
}
