/*
 *  apcnet.c  -- network parts for apcupsd package
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *               BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *            -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#undef DEBUG

/*
 * these defenitions simplify code for the network communication layer
 * MAGIC_PAD is the size required so that APC_MAGIC strings
 * are boundary aligned - nice for having boundary alignment
 * for the ints that are stored in the network packet.
 */
#define MAGIC_PAD sizeof(int)-APC_MAGIC_SIZE%sizeof(int)
/* NETDATA_SIZE is the size of the network packet between client & server */
#define NETDATA_SIZE NUM_MAGIC*(APC_MAGIC_SIZE+MAGIC_PAD)+NUM_INT*sizeof(int)

extern int master_present;
extern int socketfd;
extern int newsocketfd;

extern struct sockaddr_in my_adr;
extern int masterlen;
extern struct sockaddr_in master_adr;
extern struct hostent *mastent;

extern FILE *procfile;
extern FILE *logfile;

extern time_t last_time_on_line;
extern time_t last_time_annoy;
extern time_t last_time_delay;
extern time_t last_time_procfs;
extern time_t last_time_logging;
extern time_t last_time_net;

extern int debug_net;

/*********************************************************************/
/* write_struct_net takes a netdata structure and copies it to       */
/* malloc'ed memory so that the netdata structure is properly network*/
/* byte-ordered and padded.  It was written to be "dropped in" to the*/
/* current code and have a minimum amount of impact - swap write()   */
/* for write_struct_net when writing netdat structures across the    */
/* network.                                                          */
/*********************************************************************/
int write_struct_net (int wsocketfd, struct netdata *write_struct, size_t size)
{
	void *write_string;
	char *magic_array;
	int *int_array;
	int i;
	int wbytes;

	if ((write_string=malloc(NETDATA_SIZE)) == NULL) {
		logprintf("write_struct_net(): unable to malloc %d bytes for write_string\n", NETDATA_SIZE);
		exit(1);
	}

	magic_array=(char *)write_string;
	i=0;
	strcpy((char *)&magic_array[i*(APC_MAGIC_SIZE+MAGIC_PAD)], write_struct->apcmagic);
	i++;
	strcpy((char *)&magic_array[i*(APC_MAGIC_SIZE+MAGIC_PAD)], write_struct->usermagic);
	i++;
	if (i != NUM_MAGIC) {
		logprintf("write_struct_net(): index error: i = %d; expected %d magics\n", i, NUM_MAGIC);
		exit(1);
	}

	int_array=(int *)(write_string+NUM_MAGIC*(APC_MAGIC_SIZE+MAGIC_PAD));
	i=0;
	int_array[i]=htonl(write_struct->status);
	i++;
	int_array[i]=htonl(write_struct->lowbatt);
	i++;
	int_array[i]=htonl(write_struct->battery);
	i++;
	int_array[i]=htonl(write_struct->shutdown);
	i++;
	int_array[i]=htonl(write_struct->load);
	i++;
	int_array[i]=htonl(write_struct->timeout);
	i++;
	int_array[i]=htonl(write_struct->runtime);
	i++;
	int_array[i]=htonl(write_struct->nologin);
	i++;
	int_array[i]=htonl(write_struct->emergency);
	i++;
	int_array[i]=htonl(write_struct->remote);
	i++;
	int_array[i]=htonl(write_struct->order);
	i++;
	int_array[i]=htonl(write_struct->check_time);
	i++;
	int_array[i]=htonl(write_struct->release_me);
	i++;
	if (i != NUM_INT) {
		logprintf("write_struct_net(): index error: i = %d; expected %d ints\n", i, NUM_INT);
		exit(1);
	}

	wbytes=write(wsocketfd, write_string, NETDATA_SIZE);

	if (wbytes != NETDATA_SIZE) {
		logprintf("write_struct_net(): incorrect write: wrote %d bytes, expected %d\n", wbytes, NETDATA_SIZE);
	}

	/* logprintf("wrote netdata structure:\n"); */
	/* log_struct(write_struct); */
	free(write_string);
	return(wbytes);
}

/*********************************************************************/
/* read_struct_net reads a packet from the network an copies it to a */
/* netdata structure.  read_struct_net translates the ints to local  */
/* byte-ordering and removes padding.  It was written to be "dropped */
/* in" to the current code and have a minimum impact - swap read()   */
/* for read_struct_net when reading netdat structures from the       */
/* network.                                                          */
/*********************************************************************/
int read_struct_net (int rsocketfd, struct netdata *read_struct, size_t size)
{
	void *read_string;
	char *magic_array;
	int *int_array;
	int i;
	int rbytes;

	if ((read_string=malloc(NETDATA_SIZE)) == NULL) {
		logprintf("read_struct_net(): unable to malloc %d bytes for read_string\n", NETDATA_SIZE);
		exit(1);
	}

	rbytes=read(rsocketfd, read_string, NETDATA_SIZE);

	if (rbytes != NETDATA_SIZE) {
		logprintf("read_struct_net(): incorrect read: read %d bytes, expected %d\n", rbytes, NETDATA_SIZE);
		free(read_string);
		return(rbytes);
	}

	magic_array=(char *)read_string;
	i=0;
	strcpy(read_struct->apcmagic, (char *)&magic_array[i*(APC_MAGIC_SIZE+MAGIC_PAD)]);
	i++;
	strcpy(read_struct->usermagic, (char *)&magic_array[i*(APC_MAGIC_SIZE+MAGIC_PAD)]);
	i++;
	if (i != NUM_MAGIC) {
		logprintf("read_struct_net(): index error: i = %d; expected %d magics\n", i, NUM_MAGIC);
		exit(1);
	}

	int_array=(int *)(read_string+NUM_MAGIC*(APC_MAGIC_SIZE+MAGIC_PAD));
	i=0;
	read_struct->status=ntohl(int_array[i]);
	i++;
	read_struct->lowbatt=ntohl(int_array[i]);
	i++;
	read_struct->battery=ntohl(int_array[i]);
	i++;
	read_struct->shutdown=ntohl(int_array[i]);
	i++;
	read_struct->load=ntohl(int_array[i]);
	i++;
	read_struct->timeout=ntohl(int_array[i]);
	i++;
	read_struct->runtime=ntohl(int_array[i]);
	i++;
	read_struct->nologin=ntohl(int_array[i]);
	i++;
	read_struct->emergency=ntohl(int_array[i]);
	i++;
	read_struct->remote=ntohl(int_array[i]);
	i++;
	read_struct->order=ntohl(int_array[i]);
	i++;
	read_struct->check_time=ntohl(int_array[i]);
	i++;
	read_struct->release_me=ntohl(int_array[i]);
	i++;
	if (i != NUM_INT) {
		logprintf("read_struct_net(): index error: i = %d; expected %d ints\n", i, NUM_INT);
		exit(1);
	}

	/* logprintf("read netdata structure:\n"); */
	/* log_struct(read_struct); */
	free(read_string);
	return(rbytes);
}

/*********************************************************************/
/* This prints each network packet to the log file.  It was writting */
/* for my debugging but it might be nice to leave it in place for    */
/* anyone that wants to run apcupsd in debug mode                    */
/*********************************************************************/
void log_struct (struct netdata *logstruct)
{
	logprintf("\tapcmagic = %s\n", logstruct->apcmagic);
	logprintf("\tstatus = %x\n", logstruct->status);
	logprintf("\tlowbatt = %x\n", logstruct->lowbatt);
	logprintf("\tbattery = %x\n", logstruct->battery);
	logprintf("\tshutdown = %x\n", logstruct->shutdown);
	logprintf("\tload = %x\n", logstruct->load);
	logprintf("\ttimeout = %x\n", logstruct->timeout);
	logprintf("\truntime = %x\n", logstruct->runtime);
	logprintf("\tnologin = %x\n", logstruct->nologin);
	logprintf("\temergency = %x\n", logstruct->emergency);
	logprintf("\tremote = %x\n", logstruct->remote);
	logprintf("\torder = %x\n", logstruct->order);
	logprintf("\tusermagic = %s\n", logstruct->usermagic);
	logprintf("\tcheck_time = %x\n", logstruct->check_time);
	logprintf("\trelease_me = %x\n\n", logstruct->release_me);
}

/*********************************************************************/
int reconnect_master (UPSINFO *ups, int who)
{
	struct hostent *slavent;
	struct netdata master_magic_data;
	int j = 0;
	int c = 0;

	if ((slavent = gethostbyname(ups->slaves[who].name)) == NULL) {
		logprintf("Can't re-resolve slave name %s\n", ups->slaves[who].name);
		return(0);
	} else {
		c = 0;
		j = 1;
		while((c == 0) && (j<=5)) {
			ups->slaves[who].socket = socket(AF_INET, SOCK_STREAM, 0);
			if (ups->slaves[who].socket < 0) {
				logprintf("Can't re-allocate socket\n");
				return(0);
			}
			/* memset is more portable than bzero */
			memset((void *) &ups->slaves[who].addr, 0,
			sizeof(struct sockaddr_in));
			ups->slaves[who].addr.sin_family = AF_INET;
			memcpy((void *)&ups->slaves[who].addr.sin_addr,
			(void *)slavent->h_addr,
			sizeof(struct in_addr));
			ups->slaves[who].addr.sin_port = htons(ups->NetUpsPort);
			if (connect(ups->slaves[who].socket,
			            (struct sockaddr *) &ups->slaves[who].addr,
			            sizeof(ups->slaves[who].addr)) < 0) {
				logprintf("Can't reconnect to slave %s.\n",
				ups->slaves[who].name);
				close(ups->slaves[who].socket);
				sleep(10);
				j++;
			} else {
				c = 1;
				read_struct_net(ups->slaves[who].socket,
				                &master_magic_data, sizeof(master_magic_data));
				if ((strcmp(APC_MAGIC, master_magic_data.apcmagic) == 0) &&
				    (strcmp(ups->slaves[who].magic, master_magic_data.usermagic) == 0)) {
					logprintf("Got reconnect to slave %s.\n",
					          ups->slaves[who].name);
				} else {
					close(ups->slaves[who].socket);
					return(0);
				}
				logprintf("Got reconnect to slave %s.\n", ups->slaves[who].name);
				master_magic_data.check_time = ups->nettime - 5;
				sleep(1);
				write_struct_net(ups->slaves[who].socket,
				                 &master_magic_data, sizeof(master_magic_data));
				close(ups->slaves[who].socket);
				return(1);
			}
		}
	}
	return(0);
}

/*********************************************************************/
int prepare_master (UPSINFO *ups)
{
	struct hostent *slavent;
	struct netdata master_magic_data;
	int i = 0;
	int j = 0;
	int c = 0;

	for (i=0; i<ups->slave_count; i++) {
		if ((slavent = gethostbyname(ups->slaves[i].name)) == NULL) {
			logprintf("Can't resolve slave name %s\n",
			ups->slaves[i].name);
			ups->slaves[i].present = 0;
		} else {
			c = 0;
			j = 1;
			while((c == 0) && (j<=30)) {
				ups->slaves[i].socket = socket(AF_INET, SOCK_STREAM, 0);
				if (ups->slaves[i].socket < 0) {
					logprintf("Can't allocate socket\n");
					ups->slaves[i].present = 0;
					return(1);
				}
				/* memset is more portable than bzero */
				memset((void *) &ups->slaves[i].addr, 0,
				       sizeof(struct sockaddr_in));
				ups->slaves[i].addr.sin_family = AF_INET;
				memcpy((void *)&ups->slaves[i].addr.sin_addr,
				       (void *)slavent->h_addr,
				       sizeof(struct in_addr));
				ups->slaves[i].addr.sin_port = htons(ups->NetUpsPort);
				if (connect(ups->slaves[i].socket,
				            (struct sockaddr *) &ups->slaves[i].addr,
				            sizeof(ups->slaves[i].addr)) < 0) {
					logprintf("Can't connect to slave %s.\n",
					ups->slaves[i].name);
					ups->slaves[i].present = 0;
					close(ups->slaves[i].socket);
					sleep(10);
					j++;
				} else {
					c = 1;
					read_struct_net(ups->slaves[i].socket,
					&master_magic_data, sizeof(master_magic_data));
					if (strcmp(APC_MAGIC, master_magic_data.apcmagic) == 0)
						strcpy(ups->slaves[i].magic,
						       master_magic_data.usermagic);
					master_magic_data.check_time = ups->nettime - 5;
					sleep(1);
					write_struct_net(ups->slaves[i].socket,
					                 &master_magic_data, sizeof(master_magic_data));
					close(ups->slaves[i].socket);
					ups->slaves[i].present = 1;
				}
			}
		}
	}
	return(0);
}

/*********************************************************************/
void send_to_slaves (UPSINFO *ups)
{
	int i = 0;
	int slaves_down = 0;
	int slave_error = 0;

	for (i=0; i<ups->slave_count; i++) {
		switch(send_to_each(ups, i)) {
			case 7:
				logprintf("%s closed socket stream, try later.\n", ups->slaves[i].name);
				break;
			case 6:
				logprintf("Can't reconnect to slave %s, try later.\n", ups->slaves[i].name);
				ups->slaves[i].present = 2;
				break;
			case 5:
				logprintf("Got slave shutdown from %s.\n", ups->slaves[i].name);
				slaves_down++;
				break;          
			case 4:
				logprintf("Can't write to slave %s.\n", ups->slaves[i].name);
				close(ups->slaves[i].socket);
				break;
			case 3:
				logprintf("Can't read magic from slave %s.\n", ups->slaves[i].name);
				close(ups->slaves[i].socket);
				break;
			case 2:
				logprintf("Can't connect to slave %s.\n", ups->slaves[i].name);
				slave_error++;
				close(ups->slaves[i].socket);
				break;
			case 1:
				logprintf("Can't open stream socket.\n");
				break;
			case 0:
				close(ups->slaves[i].socket);
				break;
			default:
				logprintf("Unknown Error.\n");
				break;
		}
	}

	if ((slave_error) && (ups->ShutDown == 1)) {
		for (i=0; i<ups->slave_count; i++) {
			if (ups->slaves[i].present == 3) {
				int j = 0;
				for (j=0; j<5; j++) {
					if (!send_to_each(ups, i))
						goto next_slave;
				}
next_slave:
				if ((ups->slaves[i].present == 3) &&
					(slaves_down != ups->slave_count)) {
					logprintf("Can't connect to slave %s for SHUTDOWN.\n", ups->slaves[i].name);
					slaves_down++;
				}
			}
		}
	}

	if (slaves_down == ups->slave_count)
		ups->slave_bool = FALSE;

	time(&last_time_net);
	ups->slave_bool = TRUE;
}

/*********************************************************************/
int send_to_each (UPSINFO *ups, int who)
{
	struct netdata read_data;
	struct netdata send_data;

	if (ups->slaves[who].present == 1) {
		goto do_send;
	} else if (ups->slaves[who].present == 3) {
		ups->slaves[who].present = 1;
		goto do_send;
	} else if ((ups->slaves[who].present == 2) && (ups->remote_state != 2)) {
		return(7);
	} else if ((ups->slaves[who].present == 2) && (ups->remote_state == 2)) {
		ups->slaves[who].present = reconnect_master(ups, who);
	} else if (!ups->slaves[who].present) {
		ups->slaves[who].present = reconnect_master(ups, who);
		if (!ups->slaves[who].present)
			return(6);
	} else {
		goto do_send;
	}

do_send:

	if ((ups->slaves[who].socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		return(1);

	if ((connect(ups->slaves[who].socket,
	             (struct sockaddr *) &ups->slaves[who].addr,
	             sizeof(ups->slaves[who].addr))) == -1) {
		ups->slaves[who].present = 3;
		return(2);
	}

	if ((read_struct_net(ups->slaves[who].socket, &read_data,
	                     sizeof(read_data))) != sizeof(read_data))
		return(3);

	ups->slaves[who].present = read_data.release_me;

	strcpy(send_data.apcmagic, APC_MAGIC);
	send_data.status	= ups->LineUp;
	send_data.lowbatt	= ups->BattUp;
	send_data.battery	= ups->BatteryUp;
	send_data.shutdown	= ups->ShutDown;
	send_data.load		= ups->load;
	send_data.timeout	= ups->timedout;
	send_data.runtime	= ups->timelout;
	send_data.nologin	= ups->nologin_file;
	send_data.emergency	= ups->emergencydown;
	send_data.remote	= ups->remotedown;
	send_data.order		= who;
	strcpy(send_data.usermagic, ups->slaves[who].magic);
	if (ups->LineUp)
		send_data.check_time = 5;
	else
		send_data.check_time = ups->nettime - 5;

	send_data.release_me	= read_data.release_me;

	if ((write_struct_net(ups->slaves[who].socket, &send_data,
	                      sizeof(send_data))) != sizeof(send_data))
		return(4);
	else {
		if (ups->BatteryUp == 0 && ups->ShutDown == 1) {
			read_struct_net(ups->slaves[who].socket, &send_data,
			                sizeof(send_data));
			if (send_data.shutdown == 2 )
				return(4);
		}
		return(0);
	}

	return(-1);
}


/*********************************************************************/
/* slaves                                                            */
/*********************************************************************/

/* Bind socket, fill adress ******************************************/
int reconnect_slave(UPSINFO *ups)
{
	struct netdata slave_magic_data;
	int c = 0;
	int i = 1;

	/* Open socket for network communication */
	if ((socketfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		logprintf("Can't open stream socket, reconnect.\n");
		master_present = 0;
		return(1);
	}

	/* memset is more portable than bzero */
	memset((char *) &my_adr, 0, sizeof(struct sockaddr_in));
	my_adr.sin_family = AF_INET;
	my_adr.sin_addr.s_addr = htonl(INADDR_ANY);
	my_adr.sin_port = htons(ups->NetUpsPort);

	if (bind(socketfd, (struct sockaddr *) &my_adr, sizeof(my_adr)) < 0) {
		logprintf("Can't bind local address, reconnect.\n");
		master_present = 0;
		close(socketfd);
		return(1);
	}

	while((c == 0) && (i<=10)) {
		listen(socketfd, 1);
		masterlen = sizeof(master_adr);
		newsocketfd = accept(socketfd,
		                     (struct sockaddr *) &master_adr,
		                     &masterlen);
		if (newsocketfd < 0) {
			logprintf("Accept error %d try %d error, reconnect.\n", i, errno);
			i++;
		} else
			c = 1;
	}

	if (c == 0) {
		master_present = 0;
		return(1);
	}

	if ((mastent = gethostbyname(ups->master_name)) == NULL) {
		logprintf("Can't resolve master name %s, reconnect.\n", ups->master_name);
		master_present = 0;
		close(newsocketfd);
		close(socketfd);
		return(1);
	}

	/*
	 *  Let's add some basic security
	 */

	if (memcmp((void *) &master_adr.sin_addr,
	           (void *)mastent->h_addr,
	           sizeof(struct in_addr)) != 0) {
		logprintf("reconnect_slave(): Unauthorised attempt from %s, reconnect.\n",
		inet_ntoa(master_adr.sin_addr));
		close(newsocketfd);
		close(socketfd);
		master_present = 0;
		return(1);
	}

	strcpy(slave_magic_data.apcmagic, APC_MAGIC);
	slave_magic_data.status		= 0;
	slave_magic_data.lowbatt	= 0;
	slave_magic_data.battery	= 0;
	slave_magic_data.shutdown	= 0;
	slave_magic_data.load		= 0;
	slave_magic_data.timeout	= 0;
	slave_magic_data.runtime	= 0;
	slave_magic_data.nologin	= 0;
	slave_magic_data.emergency	= 0;
	slave_magic_data.remote		= 0;
	slave_magic_data.order		= 0;
	strcpy(slave_magic_data.usermagic, ups->usermagic);
	slave_magic_data.check_time	= 0;
	slave_magic_data.release_me	= 1;

	if ((write_struct_net(newsocketfd, &slave_magic_data,
	                      sizeof(slave_magic_data)))== -1) {
	    close(newsocketfd);
		logprintf("Write to Master socket failure, reconnect.\n");
		master_present = 0;
		return(1);
	}

	sleep(1);

	if ((read_struct_net(newsocketfd, &slave_magic_data,
	                     sizeof(slave_magic_data))) == -1) {
		close(newsocketfd);
		logprintf("Read from Mastersocket failure, reconnect.\n");
		master_present = 0;
		return(1);
	}

	ups->nettime = slave_magic_data.check_time;
	close(newsocketfd);
	time(&last_time_net);
	master_present = 1;
	unlink(APC_RE_NET);
	return(0);
}

/* Bind socket, fill address ******************************************/
int prepare_slave (UPSINFO *ups)
{
	FILE * apc_re_net;
	struct netdata slave_magic_data;
	int renetfd = 0;
	int c = 0;
	int i = 1;

	if ((apc_re_net = fopen(APC_RE_NET, "r")) != NULL) {
		fclose(apc_re_net);
		if (reconnect_slave(ups)) {
			unlink(APC_RE_NET);
			if ((renetfd = open(APC_RE_NET, O_CREAT|O_WRONLY, 0644)) >= 0) {
				write(renetfd, REDO_NET "\n", sizeof(REDO_NET));
				close(renetfd);
			}
			close(socketfd);
		}
		return(1);
	}

	/* Open socket for network communication */
	if ((socketfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		logprintf("Can't open stream socket.\n");
		master_present = 0;
		return(1);
	}

	memset((char *) &my_adr, 0, sizeof(struct sockaddr_in));
	my_adr.sin_family = AF_INET;
	my_adr.sin_addr.s_addr = htonl(INADDR_ANY);
	my_adr.sin_port = htons(ups->NetUpsPort);

	if (bind(socketfd,
	         (struct sockaddr *) &my_adr,
	         sizeof(my_adr)) < 0) {
		logprintf("Can't bind local address.\n");
		close(socketfd);
		master_present = 0;
		return(1);
	}

	while((c == 0) && (i<=10)) {
		listen(socketfd, 1);
		masterlen = sizeof(master_adr);
		newsocketfd = accept(socketfd,
		                     (struct sockaddr *) &master_adr,
		                     &masterlen);
		if (newsocketfd < 0) {
			logprintf("Accept error %d try %d error.\n", i, errno);
			i++;
		} else
			c = 1;
	}

	if (c == 0) {
		master_present = 0;
		return(1);
	}

	if ((mastent = gethostbyname(ups->master_name)) == NULL) {
		logprintf("Can't resolve master name %s.\n", ups->master_name);
		close(newsocketfd);
		close(socketfd);
		master_present = 0;
		return(1);
	}

	/*
	 *  Let's add some basic security
	 */

	if (memcmp((void *) &master_adr.sin_addr,
	           (void *)mastent->h_addr,
	           sizeof(struct in_addr)) != 0) {
		logprintf("prepare_slave(): Unauthorised attempt from %s.\n",
		inet_ntoa(master_adr.sin_addr));
		close(newsocketfd);
		close(socketfd);
		master_present = 0;
		return(1);
	}

	strcpy(slave_magic_data.apcmagic, APC_MAGIC);
	slave_magic_data.status		= 0;
	slave_magic_data.lowbatt	= 0;
	slave_magic_data.battery	= 0;
	slave_magic_data.shutdown	= 0;
	slave_magic_data.load		= 0;
	slave_magic_data.timeout	= 0;
	slave_magic_data.runtime	= 0;
	slave_magic_data.nologin	= 0;
	slave_magic_data.emergency	= 0;
	slave_magic_data.remote		= 0;
	slave_magic_data.order		= 0;
	strcpy(slave_magic_data.usermagic, ups->usermagic);
	slave_magic_data.check_time	= 0;
	slave_magic_data.release_me	= 1;

	if ((write_struct_net(newsocketfd, &slave_magic_data,
	                      sizeof(slave_magic_data)))== -1) {
		close(newsocketfd);
		logprintf("Write to Master socket failure\n");
		master_present = 0;
		return(1);
	}

	sleep(1);

	if ((read_struct_net(newsocketfd, &slave_magic_data,
	                     sizeof(slave_magic_data))) == -1) {
		close(newsocketfd);
		logprintf("Read from Mastersocket failure\n");
		master_present = 0;
		return(1);
	}

	ups->nettime = slave_magic_data.check_time;
	close(newsocketfd);
	time(&last_time_net);
	master_present = 1;
	return(0);
}

#define WACKY_NETWORK_ATTEMPS

/*********************************************************************/
int get_from_master (UPSINFO *ups)
{
	struct netdata write_data;
	struct netdata get_data;
	int order = 0;
	int c = 0;
	int i = 1;

	if (!master_present) {
		reconnect_slave(ups);

		if (!master_present)
			return(0);
	}

#ifdef WACKY_NETWORK_ATTEMPS

	while((c == 0) && (i<=10)) {
		listen(socketfd, 1);
		newsocketfd = accept(socketfd, (struct sockaddr *) &master_adr, &masterlen);

#ifdef DEBUG
		fprintf(stderr, "%s: newsocketfd = %d.\n", ups->argvalue, newsocketfd);
#endif

		if (newsocketfd < 0) {
			logprintf("Accept error %d try %d error, reconnect.\n", i, errno);
			i++;
		} else if (!newsocketfd) {
			logprintf("NO CONNECT on try %d.\n", i);
			continue;
		} else {
			c = 1;
			logprintf("%s: GOT Socket Accept on try %d newsocketfd = %d.\n", ups->argvalue, i, newsocketfd);
		}
	}
	if (c == 0) {
		logprintf("Socket Accept Error.\n");
		fprintf(stderr, "%s: Socket Accept Error..\n\a", ups->argvalue);
		close(newsocketfd);
		return(0);
	}
#else
	if (listen(socketfd, 1) == -1) {
		logprintf("Listen Failure\n");
		return(0);
	}
#ifdef DEBUG
	fprintf(stderr, "%s: Listening.\n", ups->argvalue);
#endif

	if ((newsocketfd = accept(socketfd,
	                          (struct sockaddr *) &master_adr,
	                          &masterlen)) < 0) {
		logprintf("Socket Accept Error.\n");
		close(newsocketfd);
		return(0);
	}
#endif
	/*
	 *  Let's add some basic security
	 */

	if (memcmp((void *) &master_adr.sin_addr,
	           (void *)mastent->h_addr,
	           sizeof(struct in_addr)) != 0) {
		logprintf("get_from_master(): Unauthorised attempt from %s.\n",
		inet_ntoa(master_adr.sin_addr));
		close(newsocketfd);
		return(0);
	}

	strcpy(write_data.apcmagic, APC_MAGIC);
	write_data.status	= 0;
	write_data.lowbatt	= 0;
	write_data.battery	= 0;
	write_data.shutdown	= 0;
	write_data.load		= 0;
	write_data.timeout	= 0;
	write_data.runtime	= 0;
	write_data.nologin	= 0;
	write_data.emergency	= 0;
	write_data.remote	= 0;
	write_data.order	= 0;
	strcpy(write_data.usermagic, ups->usermagic);
	write_data.check_time	= 0;
	write_data.release_me	= ups->remote_state;

	if ((write_struct_net(newsocketfd, &write_data, sizeof(write_data))) == -1) {
		close(newsocketfd);
		logprintf("Write to Master socket failure.\n");
		return(0);
	}

	sleep(1);

	if ((read_struct_net(newsocketfd, &get_data, sizeof(get_data))) == -1) {
		close(newsocketfd);
		logprintf("Read from Mastersocket failure.\n");
		return(0);
	}

	if ((strcmp(APC_MAGIC, get_data.apcmagic) == 0) &&
	    (strcmp(ups->usermagic, get_data.usermagic) == 0)) {
		ups->LineUp        = get_data.status;
		ups->BattUp        = get_data.lowbatt;
		ups->BatteryUp     = get_data.battery;
		ups->ShutDown      = get_data.shutdown;
		ups->load          = get_data.load;
		ups->timedout      = get_data.timeout;
		ups->timelout      = get_data.runtime;
		ups->nologin_file  = get_data.nologin;
		ups->emergencydown = get_data.emergency;
		ups->remotedown    = get_data.remote;
		order              = get_data.order;
		ups->nettime       = get_data.check_time;
		ups->remote_state  = get_data.release_me;
	} else {
		logprintf("GFM APC    MAGIC is: %s\n", get_data.apcmagic);
		logprintf("GFM MASTER MAGIC is: %s\n", get_data.usermagic);
		logprintf("GFM SLAVE  MAGIC is: %s\n", ups->usermagic);
		close(newsocketfd);
		return(0);
	}

	if (ups->nologin_file == TRUE) {
		make_file(NOLOGIN);
		logonfail(0);
	}

	/*
	 *  Did lowbatt_bit go high? Then the battery power is failing.
	 *  Normal Power down during Power Failure
	 *  Any Kind of Shutdown
	 */
	if (ups->ShutDown == 1) {
		get_data.shutdown = 2;
		write_struct_net(newsocketfd, &get_data, sizeof(get_data));
		if (ups->timedout) {
			system(ups->timeout);
			system(ups->doshutdown);
		} else if (ups->load) {
			system(ups->loadlimit);
			system(ups->doshutdown);
		} else if (ups->timelout) {
			system(ups->runlimit);
			system(ups->doshutdown);
		} else if (ups->emergencydown) {
			system(ups->emergency);
		} else if (ups->remotedown) {
			system(ups->remote_shutdown);
		} else {
			system(ups->failing);
			system(ups->doshutdown);
		}
		close(newsocketfd);
		close(socketfd);
		make_file(PWRFAIL);
		powerfail(3);
		logonfail(0);
		return(0);
	}

	close(newsocketfd);
	return(1);
}

/*********************************************************************/
void kill_net (UPSINFO *ups, int killpwr)
{
	if (killpwr) {
		fprintf(stderr,
		        "%s: Waiting For ShareUPS Master to shutdown.\n",
		        ups->argvalue);
		sleep(60);
		fprintf(stderr,
		        "%s: Great Mains Returned, Trying a REBOOT!\n",
		        ups->argvalue);
		system(ups->reboot_cmd);
		fprintf(stderr,
		        "%s: Drat!! Failed to have power killed by Master!\n",
		        ups->argvalue);
		fprintf(stderr,
		        "%s: Perform CPU-reset or power-off\n",
		        ups->argvalue);
		terminate(0);
	}
}

#ifndef NEW_THREADS
/*********************************************************************/
void do_net (UPSINFO *ups)
{
	if (logfile != NULL)
		time(&last_time_logging);

	if (procfile != NULL)
		time(&last_time_procfs);

	ups->remote_state = TRUE;

	while(1) {
		if (ups->enable_access.type == TRUE)
			(void) pipe_requests(ups);

		get_from_master(ups);

		if (ups->remote_state == 2) {
			terminate(0);
		}

		if (!ups->LineUp)
			time(&last_time_annoy);
		if (!ups->LineUp)
			time(&last_time_delay);

		/*
		 *  Did status_bit go high? Then the power is failing.
		 */
		if (ups->OldLineUp == 0 && ups->LineUp != 0) {
			system(ups->powerout);
			if (logfile != NULL)
				updating_logs(logfile, ups);
			if (procfile != NULL)
				updating_procfs(procfile, ups);
		}
		/*
		 *  Did the second test verify the power is failing?
		 */
		if ((ups->LineUp) && (ups->OldLineUp) && (ups->BatteryUp == 0)) {
			system(ups->onbatt);
			if (logfile != NULL)
				updating_logs(logfile, ups);
			if (procfile != NULL)
				updating_procfs(procfile, ups);
		}
		/*
		 *  Did status_bit go down again? Then the power is back.
		 *  No shutdown to cancel.
		 */
		if ((ups->OldLineUp > 0 && ups->LineUp == 0) && ups->ShutDown == 0) {
			system(ups->mainsback);
			if (ups->nologin_file != TRUE) {
				logonfail(1);
				remove_file(NOLOGIN);
			}
		}
		/*
		 *  Announce to LogOff, with initial delay ....
		 */
		if ((ups->BatteryUp == 1 &&
		    ups->ShutDown == 0 &&
		    ups->LineUp > 0) &&
		    ((time(NULL) - last_time_delay) > ups->delay) &&
		    ((time(NULL) - last_time_annoy) > ups->annoy) &&
		    (ups->nologin_file == TRUE)) {
			system(ups->annoyme);
			time(&last_time_annoy);
		}

		if ((logfile != NULL) &&
		    ((time(NULL) - last_time_logging) > ups->logtime)) {
			updating_logs(logfile, ups);
		}             /* Do Logging */

		if ((procfile != NULL) &&
		    ((time(NULL) - last_time_procfs) > ups->proctime)) {
			updating_procfs(procfile, ups);
		}          /* Do Procfs */

		/*
		 *  Reset count, remember status and sleep 2 seconds.
		 */
		ups->OldLineUp = ups->LineUp;
		ups->OldBattUp = ups->BattUp;
		sleep(2);
	}
}

#else /* NEW_THREADS */

/*********************************************************************/
void timer_net (int sig)
{
	UPSINFO *ups = &myUPS;

	read_andlock_shmarea(ups);

	ups->remote_state = TRUE;

	while(1) {
		get_from_master(ups);

		if (ups->remote_state == 2)
			terminate(0);

		if (!ups->LineUp)
			time(&last_time_annoy);
		if (!ups->LineUp)
			time(&last_time_delay);

		/*
		 *  Did status_bit go high? Then the power is failing.
		 */
		if (ups->OldLineUp == 0 && ups->LineUp != 0) {
			system(ups->powerout);
		}
		/*
		 *  Did the second test verify the power is failing?
		 */
		if ((ups->LineUp) && (ups->OldLineUp) && (ups->BatteryUp == 0)) {
			system(ups->onbatt);
		}
		/*
		 *  Did status_bit go down again? Then the power is back.
		 *  No shutdown to cancel.
		 */
		if ((ups->OldLineUp > 0 && ups->LineUp == 0) && ups->ShutDown == 0) {
			system(ups->mainsback);
			if (ups->nologin_file != TRUE) {
				logonfail(1);
				remove_file(NOLOGIN);
			}
		}
		/*
		 *  Announce to LogOff, with initial delay ....
		 */
		if ((ups->BatteryUp == 1 &&
		    ups->ShutDown == 0 &&
		    ups->LineUp > 0) &&
		    ((time(NULL) - last_time_delay) > ups->delay) &&
		    ((time(NULL) - last_time_annoy) > ups->annoy) &&
		    (ups->nologin_file == TRUE)) {
			system(ups->annoyme);
			time(&last_time_annoy);
		}

		/*
		 *  Reset count, remember status and sleep 2 seconds.
		 */
		ups->OldLineUp = ups->LineUp;
		ups->OldBattUp = ups->BattUp;
		sleep(2);
	}

	write_andunlock_shmarea(ups);

	alarm(TIMER_NET);
}

/*********************************************************************/
void do_net (void) {
	init_thread_signals();
	init_timer(TIMER_NET, timer_net);
	sleep_forever();
}

/*********************************************************************/
void timer_slaves (int sig) {
	UPSINFO *ups = &myUPS;
	int now = time(NULL);

	read_andlock_shmarea(ups);

#if 0
	/*
	 * For now we are sending data to slaves _every_ time we can.
	 * This is for test pourposes and because I must think to a way to
	 * signal this thread that it must send data even if the delay time is
	 * not enought.
	 * -RF
	 */
	ups->send_slaves_urg = TRUE;

	if (((now - last_time_net) > ups->nettime) ||
			ups->send_slaves_urg == TRUE) {
		send_to_slaves(ups);
        }
	ups->send_slaves_urg = FALSE;
#else
	if ((now - last_time_net) > ups->nettime) {
		send_to_slaves(ups);
        }
#endif
	write_andunlock_shmarea(ups);

	alarm(TIMER_SLAVES);
}

/*********************************************************************/
void do_slaves (void) {
	init_thread_signals();
	init_timer(TIMER_SLAVES, timer_slaves);
	sleep_forever();
}
#endif /* NEW_THREADS */
