/* 	$Id: CircMeter.c,v 1.4 1997/11/23 17:48:09 pirx Exp $	 */

/*
 * Author: Achim Bangert (abangert@ix.urz.uni-heidelberg.de)
 */

/*

Copyright (c) 1987  X Consortium

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from the X Consortium.

*/

#include <stdio.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "CircMeterP.h"

#define MAX(a, b) (((a) > (b)) ? (a) : (b))
#define STRLEN(str) (((str) == NULL) ? 0 : strlen(str))

static XtResource resources[] = {
#define offset(field) XtOffsetOf(CircMeterRec, field)
  /* {name, class, type, size, offset, default_type, default_addr}, */
  {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
   offset(circ_meter.foreground), XtRString, XtDefaultForeground},
  {XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
   offset(circ_meter.font), XtRString, XtDefaultFont},
  {XtNheadline, XtCHeadline, XtRString, sizeof(String),
   offset(circ_meter.headline), XtRString, NULL},
  {XtNjustify, XtCJustify, XtRJustify, sizeof(XtJustify),
   offset(circ_meter.justify), XtRImmediate, (XtPointer) XtJustifyLeft},
  {XtNplacement, XtCPlacement, XtRJustify, sizeof(XtJustify),
   offset(circ_meter.placement), XtRImmediate, (XtPointer) XtJustifyRight},
  {XtNspacing, XtCSpacing, XtRInt, sizeof(int),
   offset(circ_meter.spacing), XtRImmediate, (XtPointer) 4},
  {XtNnumberSectors, XtCNumberSectors, XtRInt, sizeof(int), 
   offset(circ_meter.num_sectors), XtRImmediate, (XtPointer) 0},
  {XtNcolor1, XtCColor1, XtRPixel, sizeof(Pixel), 
   offset(circ_meter.sector[0].color), XtRString, (XtPointer) "blue"},
  {XtNarc1, XtCArc1, XtRInt, sizeof(int),
   offset(circ_meter.sector[0].arc), XtRImmediate, (XtPointer) 0},
  {XtNlabel1,  XtCLabel1, XtRString, sizeof(String),
   offset(circ_meter.sector[0].label), XtRString, NULL},
  {XtNcolor2, XtCColor2, XtRPixel, sizeof(Pixel), 
   offset(circ_meter.sector[1].color), XtRString, (XtPointer) "green"},
  {XtNarc2, XtCArc2, XtRInt, sizeof(int),
   offset(circ_meter.sector[1].arc), XtRImmediate, (XtPointer) 0},
  {XtNlabel2,  XtCLabel2, XtRString, sizeof(String),
   offset(circ_meter.sector[1].label), XtRString, NULL},
  {XtNcolor3, XtCColor3, XtRPixel, sizeof(Pixel), 
   offset(circ_meter.sector[2].color), XtRString, (XtPointer) "yellow"},
  {XtNarc3, XtCArc3, XtRInt, sizeof(int),
   offset(circ_meter.sector[2].arc), XtRImmediate, (XtPointer) 0},
  {XtNlabel3, XtCLabel3, XtRString, sizeof(String),
   offset(circ_meter.sector[2].label), XtRString, NULL},
  {XtNcolor4, XtCColor4, XtRPixel, sizeof(Pixel), 
   offset(circ_meter.sector[3].color), XtRString, (XtPointer) "red"},
  {XtNarc4, XtCArc4, XtRInt, sizeof(int),
   offset(circ_meter.sector[3].arc), XtRImmediate, (XtPointer) 0},
  {XtNlabel4, XtCLabel4, XtRString, sizeof(String),
   offset(circ_meter.sector[3].label), XtRString, NULL}
#undef offset
};

static void Initialize(Widget, Widget, ArgList, Cardinal *);
/* initialize private widget data */

static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal *);
/* make necessary changes in response to changed resources */

static void Redisplay(Widget, XEvent *, Region);
/* redraw widget */

static void Resize(Widget);
/* change layout to accomodate new widget size */

static void Destroy(Widget);
/* release shared GC */

CircMeterClassRec circMeterClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"CircMeter",
    /* widget_size		*/	sizeof(CircMeterRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* circMeter fields */
    /* empty			*/	0
  }
};

WidgetClass circMeterWidgetClass = (WidgetClass) &circMeterClassRec;

static void redraw_labels(Widget);
/* redraw headline and labels */

static void redraw_sectors(Widget);
/* redraw sectors */

static void GetGC(CircMeterWidget);
/* allocate GC for drawing */

static void layout(CircMeterWidget, int);
/* calculate sizes and positions of widget's components */

static void text_extent(CircMeterPart *);
/* calculate width and height of widget's text part */

static int justify_text(Dimension, Dimension, XtJustify);
/* justify text in field */

static void Initialize(Widget request, Widget new, ArgList args, 
		       Cardinal *num_args)
/* initialize private widget data */
{
  CircMeterWidget cmw = (CircMeterWidget) new;
  int layout_mode = XcmwGET_TEXT_EXTENT;
  int i;

  if(cmw->circ_meter.headline)
    cmw->circ_meter.headline = XtNewString(cmw->circ_meter.headline);
  for(i = 0; i < cmw->circ_meter.num_sectors; ++i)
    if(cmw->circ_meter.sector[i].label)
      cmw->circ_meter.sector[i].label =
	XtNewString(cmw->circ_meter.sector[i].label);
  if(cmw->circ_meter.placement != XtJustifyLeft &&
     cmw->circ_meter.placement != XtJustifyRight)
    {
      XtWarning("CircMeter Widget: Specified placement not supported, "
                "using XtJustifyRight instead");
      cmw->circ_meter.placement = XtJustifyRight;
    }
  if(cmw->core.width == 0)
    /* we are free to use our preferred width */
    layout_mode |= XcmwSET_WIDTH;
  if(cmw->core.height == 0)
    /* use preferred height */
    layout_mode |= XcmwSET_HEIGHT;
  layout(cmw, layout_mode);
  GetGC(cmw);
}

static Boolean SetValues(Widget current, Widget request, Widget new, 
			 ArgList args, Cardinal *num_args)
/* make necessary changes in response to changed resources */
{
  CircMeterWidget cmw = (CircMeterWidget) new;
  CircMeterPart *cmp = &(cmw->circ_meter);
  CircMeterWidget old_cmw = (CircMeterWidget) current;
  CircMeterPart *old_cmp = &(old_cmw->circ_meter);
  Boolean change_layout = False;
  Boolean redisplay = False;
  int i;

  if(cmp->foreground != old_cmp->foreground)
    redisplay = True;
  if(cmp->font != old_cmp->font)
    {
      XSetFont(XtDisplay(new), cmp->gc, cmp->font->fid);
      change_layout = True;
    }
  if(cmp->justify != old_cmp->justify ||
     cmp->placement != old_cmp->placement ||
     cmp->spacing != old_cmp->spacing || 
     cmp->num_sectors != old_cmp->num_sectors)
    {
      if(cmp->placement != XtJustifyLeft && cmp->placement != XtJustifyRight)
	{
	  XtWarning("CircMeter Widget: Specified placement not supported, "
                "using XtJustifyRight instead");
	  cmp->placement = XtJustifyRight;
	}
      change_layout = True;
    }
  if(old_cmp->headline != cmp->headline) 
    {
      XtFree((char *) old_cmp->headline);
      /* that's okay because default is NULL */
      if(cmp->headline)
	cmp->headline = XtNewString(cmp->headline);
      change_layout = True;
    }
  for(i = 0; i < cmp->num_sectors; ++i)
    {
      if(old_cmp->sector[i].label != cmp->sector[i].label) 
	{
	  XtFree((char *) old_cmp->sector[i].label);
	  if(cmp->sector[i].label)
	    cmp->sector[i].label = XtNewString(cmp->sector[i].label);
	  change_layout = True;
	}
      if(old_cmp->sector[i].color != cmp->sector[i].color ||
	 old_cmp->sector[i].arc != cmp->sector[i].arc)
	redisplay = True;
    }
  if(change_layout)
    layout(cmw, XcmwGET_TEXT_EXTENT);
  return redisplay || change_layout;
}

static void redraw_labels(Widget w)
/* redraw headline and labels */
{
  CircMeterPart *cmp = &(((CircMeterWidget) w)->circ_meter);
  int text_y = cmp->font->max_bounds.ascent + cmp->spacing;
  int i;

  if(cmp->headline_len)
    {
      XSetForeground(XtDisplay(w), cmp->gc, cmp->foreground);
      XDrawString(XtDisplay(w), XtWindow(w), cmp->gc, 
		  cmp->headline_x, text_y, cmp->headline, cmp->headline_len);
      text_y += cmp->font_height;
    }
  for(i = 0; i < cmp->num_sectors; ++i)
    if(cmp->sector[i].label_len)
      {
	XSetForeground(XtDisplay(w), cmp->gc, cmp->sector[i].color);
	XDrawString(XtDisplay(w), XtWindow(w), cmp->gc, 
		    cmp->sector[i].label_x, text_y, cmp->sector[i].label, 
		    cmp->sector[i].label_len);
	text_y += cmp->font_height;
      }
}

static void redraw_sectors(Widget w)
/* redraw sectors */
{
  CircMeterPart *cmp = &(((CircMeterWidget) w)->circ_meter);
  int delta_phi, phi = 450 * 64;
  int i;
  
  /*
   * angles range from 0 to 360 * 64, 0 corresponding to the three-o'clock
   * position; arcs are drawn starting from phi to phi + delta_phi; negative
   * delta_phi indicates clockwise motion
   */
  for(i = 0; i < cmp->num_sectors; ++i)
    {
      delta_phi = -(cmp->sector[i].arc * 360 * 64) / 1000;
      XSetForeground(XtDisplay(w), cmp->gc, cmp->sector[i].color);
      XFillArc(XtDisplay(w), XtWindow(w), cmp->gc, 
	       cmp->circ_x, cmp->spacing, cmp->circ_radius, 
	       cmp->circ_radius, phi, delta_phi);
      phi += delta_phi;
    }
  /*
   * complete full circle with arc in background color
   */
  delta_phi = 90 * 64 - phi;
  XSetForeground(XtDisplay(w), cmp->gc, w->core.background_pixel);
  XFillArc(XtDisplay(w), XtWindow(w), cmp->gc, 
	   cmp->circ_x, cmp->spacing, cmp->circ_radius, 
	   cmp->circ_radius, phi, delta_phi);
}

static void Redisplay(Widget w, XEvent *event, Region region)
/* redraw widget */
{
  CircMeterPart *cmp = &(((CircMeterWidget) w)->circ_meter);
  int text_x = cmp->spacing;

  if(cmp->placement == XtJustifyRight) 
    text_x += cmp->circ_radius + cmp->spacing; 
  if(XRectInRegion(region, text_x, cmp->spacing, 
		   cmp->text_width, cmp->text_height) != RectangleOut)
    /* region intersects rectangle containing labels */
    redraw_labels(w);
  if(XRectInRegion(region, cmp->circ_x, cmp->spacing, 
		   cmp->circ_radius, cmp->circ_radius) != RectangleOut)
    /* region intersects rectangle containing sectors */
    redraw_sectors(w);
}

static void Resize(Widget w)
/* change layout to accomodate new widget size */
{
  layout((CircMeterWidget) w, 0);
}

static void Destroy(Widget w)
/* release shared GC */
{
  CircMeterWidget cmw = (CircMeterWidget) w;
  int i;

  XtReleaseGC(w, cmw->circ_meter.gc);
  XtFree((char *) cmw->circ_meter.headline);
  for(i = 0; i < cmw->circ_meter.num_sectors; ++i)
    XtFree((char *) cmw->circ_meter.sector[i].label);
}

static void GetGC(CircMeterWidget cmw)
/* allocate GC for drawing */
{
  XGCValues values;
  XtGCMask value_mask;

  values.background = cmw->core.background_pixel;
  values.function = GXcopy;
  values.line_width = 0;
  values.fill_style = FillSolid;
  values.fill_rule = EvenOddRule;
  values.font = cmw->circ_meter.font->fid;
  values.arc_mode = ArcPieSlice;
  value_mask = GCBackground | GCFunction | GCLineWidth | GCFillStyle | 
    GCFillRule | GCFont | GCArcMode;
  cmw->circ_meter.gc = XtAllocateGC((Widget) cmw, 0, value_mask,
				    &values, GCBackground | GCForeground | 
				    GCFont, ~value_mask);
}

static void layout(CircMeterWidget cmw, int mode)
/* calculate sizes and positions of widget's components */
{
  CircMeterPart *cmp = &(cmw->circ_meter);
  Dimension text_x;
  int i;

  if(mode & XcmwGET_TEXT_EXTENT)
    /* compute max. text width */
    text_extent(cmp);
  if(mode & XcmwSET_HEIGHT)
    /* use preferred height */
    cmw->core.height = 
      MAX(cmp->text_height, XcmwDEFAULT_HEIGHT) + cmp->spacing * 2;
  cmp->circ_radius = cmw->core.height - cmp->spacing * 2;
  if(mode & XcmwSET_WIDTH)
    /* use preferred width */
    cmw->core.width = cmw->core.height + cmp->text_width + cmp->spacing;
  if(cmp->placement == XtJustifyRight)
    /* place labels to the right of sectors */
    {
      cmp->circ_x = cmp->spacing;
      text_x = cmw->core.height;
    }
  else
    /* place labels at the left side */
    {
      cmp->circ_x = cmp->text_width + cmp->spacing * 2;
      text_x = cmp->spacing;
    }
  if(cmp->headline_len)
    cmp->headline_x = text_x + 
      justify_text(cmp->text_width, cmp->headline_width, cmp->justify);
  for(i = 0; i < cmp->num_sectors; ++i)
    if(cmp->sector[i].label_len)
      cmp->sector[i].label_x = text_x + 
	justify_text(cmp->text_width, cmp->sector[i].label_width, 
		     cmp->justify);
}

static void text_extent(CircMeterPart *cmp)
/* calculate width and height of widget's text part */
{
  int i;

  cmp->text_height = 0;
  cmp->font_height = cmp->font->max_bounds.ascent + 
    cmp->font->max_bounds.descent;
  if((cmp->headline_len = STRLEN(cmp->headline)))
    {
      cmp->text_width = cmp->headline_width = 
	XTextWidth(cmp->font, cmp->headline, cmp->headline_len);
      cmp->text_height += cmp->font_height;
    }
  else
    cmp->text_width = 0;
  for(i = 0; i < cmp->num_sectors; ++i)
    if((cmp->sector[i].label_len = STRLEN(cmp->sector[i].label)))
      {
	cmp->sector[i].label_width = 
	  XTextWidth(cmp->font, cmp->sector[i].label, 
		     cmp->sector[i].label_len);
	cmp->text_width = MAX(cmp->text_width, cmp->sector[i].label_width);
	cmp->text_height += cmp->font_height;
      }
  if(cmp->text_width == 0)
    /* cancel spacing for text of zero width */
    cmp->text_width = -cmp->spacing;
}

static int justify_text(Dimension field_width, Dimension text_width,
			XtJustify justify)
/* justify text in field */
{
  switch(justify)
    {
    case XtJustifyCenter:
      return (field_width - text_width) / 2;
    case XtJustifyRight:
      return field_width - text_width;
    default:
      return 0;
    }
}

/*
 * convenience functions
 */

void XcmwUpdateSectors(Widget w, int *arcs)
/* update arcs and redraw sectors */
{
  CircMeterWidget cmw = (CircMeterWidget) w;
  Boolean redraw = False;
  int i;

  for(i = 0; i < cmw->circ_meter.num_sectors; ++i)
    if(cmw->circ_meter.sector[i].arc != arcs[i])
      {
	cmw->circ_meter.sector[i].arc = arcs[i];
	redraw = True;
      }
  if(redraw && XtIsRealized(w))
    redraw_sectors(w);
}



