/*========================================================================
 * PROGRAM: xloadtime
 *
 * VERSION: $Id: xloadtime.c,v 1.1.1.1 1996/08/09 15:06:31 darren Exp $
 *
 * AUTHOR : Darren J Moffat <darren@xarius.demon.co.uk>
 *        : Trent Piepho <xyzzy@speakeasy.org>
 *
 * DESCRIP: Xload with integral clock, uses Linux /proc/loadavg
 *
 *- Version History ------------------------------------------------------
 *
 * $Log: xloadtime.c,v $
 * Revision 2       ~1996/10/01  11:11:11  zyzzy
 * Revision 1.1.1.1  1996/08/09  15:06:31  darren
 * Initial Version
 *
 *
 *======================================================================*/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/StripChart.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Scrollbar.h>


/*-----------------------------------------------------------------------+
 | Local Function Prototypes                                             |
 +----------------------------------------------------------------------*/
void set_time(void);
void get_load(Widget w, XtPointer client, XtPointer load_value);
void set_cpu(void);
int get_cpus(void);

/*-----------------------------------------------------------------------+
 | Global Variables                                                      |
 +----------------------------------------------------------------------*/
int update;						/* Number of Seconds to next update */
Widget curtime;					/* Label Widget for Current Time */
Widget cpu[4];					/* Scrollbar(s) for CPU usage */
int ncpu;						/* Number of CPUs */
int proc_stat;					/* file handle for /proc/stat */

typedef struct {
	Boolean ampm;				/* use 12-hour time or 24-hour time */
	Boolean cpu;				/* display CPU usage bar(s) */
} AppData;

AppData app_data;				/* application resources */

static XtResource resources[] =
{
	{ "12hour", "12Hour", XtRBoolean, sizeof(Boolean),
		XtOffsetOf(AppData, ampm), XtRImmediate, (XtPointer) False
	},
	{ "cpu", "Cpu", XtRBoolean, sizeof(Boolean),
		XtOffsetOf(AppData, cpu), XtRImmediate, (XtPointer) False
	}
};

void my_XawScrollbarSetThumb(Widget w, float top, float shown)
{
  Arg arglist[2];

  if(sizeof(float)>sizeof(XtArgVal)) {
    XtSetArg(arglist[0], XtCTopOfThumb, &top);
    XtSetArg(arglist[1], XtNshown, &shown);
   }
  else {
    XtArgVal *l_top=(XtArgVal*)&top;
    XtArgVal *l_shown=(XtArgVal*)&shown; 
    XtSetArg(arglist[0], XtNtopOfThumb, *l_top);
    XtSetArg(arglist[1], XtNshown, *l_shown);
   }
  XtSetValues(w, arglist, 2);
}


/*-----------------------------------------------------------------------+
 | get_load                                                              |
 |                                                                       |
 | Sets the value of the StripChart to the current machine load average  |
 | Also sets the time every 60 seconds,  This is done here to avoid the  |
 | need to setup a seprate thread or using sigalarm.                     |
 |                                                                       |
 | PARAMS                                                                |
 |	w			In	StripChart Widget     					             |
 |	loadavg		In	File handle of /proc/loadavg                         |
 |	load_value	Out	Current Value of StripChart                          |
 |                                                                       |
 | GLOBALS                                                               |
 |	update		Read                                                     |
 |                                                                       |
 | FILES                                                                 |
 |	loadavg		Read	/proc/loadavg for current value                  |
 |                                                                       |
 +----------------------------------------------------------------------*/
void get_load(Widget w, XtPointer loadavg, XtPointer load_value)
{
	static int clicks = 0;
	char str[32];

	lseek((int) loadavg, 0, SEEK_SET);
	read((int) loadavg, str, 31);
	(void) sscanf(str, "%lf", (double *) load_value);

	if(app_data.cpu) set_cpu();

	if (clicks * update >= 60) {
		set_time();
		clicks = 0;
	} else {
		++clicks;
	}

}								/* get_load */


/* returns number of CPUs. */
int get_cpus(void)
{
	char str[256];
	char *pos;
	int len;

	lseek(proc_stat, 0, SEEK_SET);
	read(proc_stat, str, 255);
	sscanf(str, "cpu %*d %*d %*d %*d\n%n", &len);
	pos = str+len;

	if(strncmp(pos,"cpu",3)) return 1;
	sscanf(pos, "cpu%*d %*d %*d %*d %*d\n%n", &len); pos+=len;
	if(strncmp(pos,"cpu",3)) return 1;
	sscanf(pos, "cpu%*d %*d %*d %*d %*d\n%n", &len); pos+=len;
	if(strncmp(pos,"cpu",3)) return 2;
	sscanf(pos, "cpu%*d %*d %*d %*d %*d\n%n", &len); pos+=len;
	if(strncmp(pos,"cpu",3)) return 3;
	else return 4;
}

/* Gets CPU usage from proc, and sets scrollbars. */
void set_cpu(void)
{
	char str[256];
	char *pos;
	int len;
	int c;
	int u,n,s,i;
	static int lu[4]={0,0,0,0},li[4]={0,0,0,0};

	lseek(proc_stat, 0, SEEK_SET);
	read(proc_stat, str, 255);
	sscanf(str, "cpu %d %d %d %d\n%n", &u, &n, &s, &i, &len);
	pos = str+len;

	for(c=0;c<ncpu;c++)  {
		int x;
		if(ncpu>1) {
			sscanf(pos, "cpu%*d %d %d %d %d\n%n", &u, &n, &s, &i, &len); 
			pos+=len;
		}

		u+=n+s; i+=u;     /* u = used CPU, i = total CPU */
		x = 256*(u-lu[c])/(i-li[c]);
		my_XawScrollbarSetThumb(cpu[c], 0.0, x/256.0);

		lu[c]=u; li[c]=i;
	};

	return;
}


/*-----------------------------------------------------------------------+
 | set_time                                                              |
 |                                                                       |
 | Sets the label of the curtime Widget to be the current local time     |
 |                                                                       |
 | GLOBALS                                                               |
 |	curtime		Write                                            |
 |                                                                       |
 +----------------------------------------------------------------------*/
void set_time(void)
{
	struct tm *timeinfo;
	char timebuf[9];
	long t;

	time(&t);
	timeinfo = localtime(&t);

	if (app_data.ampm == True) {
		(void) sprintf(timebuf, "%2d:%02d %s",
				(timeinfo->tm_hour % 12) ? (timeinfo->tm_hour % 12) : 12,
					   timeinfo->tm_min,
					   (timeinfo->tm_hour >= 12) ? "PM" : "AM");
	} else {
		(void) sprintf(timebuf, "%02d:%02d",
					   timeinfo->tm_hour,
					   timeinfo->tm_min);
	};

	XtVaSetValues(curtime,
				  XtNlabel, &timebuf,
				  NULL);

}								/* set_time */


/*-----------------------------------------------------------------------+
 | main                                                                  |
 +----------------------------------------------------------------------*/
int main(int argc, char **argv)
{
	int i;
	XtAppContext app_context;
	Widget topLevel, paned, load;
	int loadavg;

	static XrmOptionDescRec options[] =
	{
		{"-update", "*update", XrmoptionSepArg, NULL},
		{"-jumpscroll", "*jumpScroll", XrmoptionSepArg, NULL},
		{"-scale", "*minScale", XrmoptionSepArg, NULL},
		{"-12hour", "*12Hour", XrmoptionNoArg, "True"},
		{"-cpu", "*Cpu", XrmoptionNoArg, "True"}};

	if((loadavg = open("/proc/loadavg", O_RDONLY)) == -1) {
		perror("couldn't open /proc/loadavg");
		exit(1);
	};

	if((proc_stat = open("/proc/stat", O_RDONLY)) == -1) {
		perror("couldn't open /proc/stat");
		exit(1);
	};

	topLevel = XtVaAppInitialize(&app_context,
								 "XLoad",
								 options,
								 XtNumber(options),
								 &argc, argv,
								 NULL,
								 NULL);

	XtVaGetApplicationResources(topLevel,
								&app_data,
								resources,
								XtNumber(resources),
								NULL);

	if(app_data.cpu)  {
		ncpu = get_cpus();
		paned = XtVaCreateManagedWidget("paned", panedWidgetClass, topLevel,
										XtNborderWidth, 1,
										XtNorientation, XtorientHorizontal,
										NULL);

		for(i=0;i<ncpu;i++)  {
			char name[8];
			sprintf(name,"cpu%d",i);
			cpu[i] = XtVaCreateManagedWidget(name, scrollbarWidgetClass, paned, 
											 XtNorientation, XtorientVertical,
											 XtNsensitive, False,
											 XtNminimumThumb, 1,
											 XtNthickness, 10,
											 XtNthumb, None,
											 NULL);
		}
		set_cpu();
	} else {
		paned = topLevel;
	}

	paned = XtVaCreateManagedWidget("paned", panedWidgetClass, paned,
									XtNorientation, XtorientVertical,
									NULL);

	curtime = XtVaCreateManagedWidget("curtime", labelWidgetClass, paned,
									  XtNborderWidth, 0,
									  NULL);

	load = XtVaCreateManagedWidget("load", stripChartWidgetClass, paned,
								   XtNborderWidth, 0,
								   NULL);

	XtAddCallback(load, XtNgetValue, get_load, (XtPointer) loadavg);

	XtVaGetValues(load, XtNupdate, &update, NULL);

	set_time();

	XtRealizeWidget(topLevel);
	XtAppMainLoop(app_context);

	exit(0);
}
