/***************************************
  $Header: /home/amb/procmeter/RCS/statistics.c 2.17 1998/01/06 19:39:04 amb Exp $

  ProcMeter - A performance metering/logging program for Linux - Version 2.4.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1994,95,96,97,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <time.h>
#include <sys/time.h>

#include <sys/stat.h>
#include <unistd.h>

#include "procmeter.h"

/*+ The statistics that come from /proc. +*/
Statistics ProcStats[NProcStats]=
{
 /* CPU       */  {"cpu"      , "stat"    , "%"   , 0 ,   20  , 0 , 0 },
 /* CPU_USER  */  {"cpu-user" , "stat"    , "%"   , 0 ,   20  , 0 , 0 },
 /* CPU_NICE  */  {"cpu-nice" , "stat"    , "%"   , 0 ,   20  , 0 , 0 },
 /* CPU_SYS   */  {"cpu-sys"  , "stat"    , "%"   , 0 ,   20  , 0 , 0 },
 /* CPU_IDLE  */  {"cpu-idle" , "stat"    , "%"   , 0 ,   20  , 0 , 0 },
 /* LOAD      */  {"load"     , "loadavg" , "1"   , 0 ,    1  , 0 , 0 },
 /* PROC      */  {"proc"     , "loadavg" , "10"  , 0 ,   10  , 0 , 0 },
 /* CONTEXT   */  {"context"  , "stat"    , "100" , 0 ,  100  , 0 , 0 },
 /* SWAP      */  {"swap"     , "stat"    , "50"  , 0 ,   50  , 0 , 0 },
 /* SWAP_IN   */  {"swap-in"  , "stat"    , "50"  , 0 ,   50  , 0 , 0 },
 /* SWAP_OUT  */  {"swap-out" , "stat"    , "50"  , 0 ,   50  , 0 , 0 },
 /* PAGE      */  {"page"     , "stat"    , "50"  , 0 ,  100  , 0 , 0 },
 /* PAGE_IN   */  {"page-in"  , "stat"    , "50"  , 0 ,  100  , 0 , 0 },
 /* PAGE_OUT  */  {"page-out" , "stat"    , "50"  , 0 ,  100  , 0 , 0 },
 /* DISK      */  {"disk"     , "stat"    , "25"  , 0 ,   25  , 0 , 0 },
 /* DISK_RD   */  {"disk-rd"  , "stat"    , "25"  , 0 ,   25  , 0 , 0 },
 /* DISK_WR   */  {"disk-wr"  , "stat"    , "25"  , 0 ,   25  , 0 , 0 },
 /* INTR      */  {"intr"     , "stat"    , "100" , 0 ,  100  , 0 , 0 },
 /* MEM_FREE  */  {"mem-free" , "meminfo" , "1 MB", 0 , 1<<10 , 0 , 0 }, /* The units and scale may get reset later */
 /* MEM_USED  */  {"mem-used" , "meminfo" , "1 MB", 0 , 1<<10 , 0 , 0 }, /* The units and scale may get reset later */
 /* MEM_BUFF  */  {"mem-buff" , "meminfo" , "1 MB", 0 , 1<<10 , 0 , 0 }, /* The units and scale may get reset later */
 /* MEM_CACHE */  {"mem-cache", "meminfo" , "1 MB", 0 , 1<<10 , 0 , 0 }, /* The units and scale may get reset later */
 /* MEM_SWAP  */  {"mem-swap" , "meminfo" , "1 MB", 0 , 1<<10 , 0 , 0 }, /* The units and scale may get reset later */
 /* LPKT      */  {"lpkt"     , "net/dev" , "100" , 0 ,  100  , 0 , 0 },
 /* FPKT      */  {"fpkt"     , "net/dev" , "50"  , 0 ,   50  , 0 , 0 },
 /* FPKT_RX   */  {"fpkt-rx"  , "net/dev" , "50"  , 0 ,   50  , 0 , 0 },
 /* FPKT_TX   */  {"fpkt-tx"  , "net/dev" , "50"  , 0 ,   50  , 0 , 0 },
 /* FPKT_COLL */  {"fpkt-coll", "net/dev" , "50"  , 0 ,   50  , 0 , 0 },
 /* SPKT      */  {"spkt"     , "net/dev" , "5"   , 0 ,    5  , 0 , 0 },
 /* SPKT_RX   */  {"spkt-rx"  , "net/dev" , "5"   , 0 ,    5  , 0 , 0 },
 /* SPKT_TX   */  {"spkt-tx"  , "net/dev" , "5"   , 0 ,    5  , 0 , 0 },
 /* BATT_LIFE */  {"batt-life", "apm"     , "%"   , 0 ,   20  , 0 , 0 },
 /* BATT_LVL  */  {"batt-lvl" , "apm"     , "HLCU", 0 ,    1  , 0 , 0 }
};

/*+ A scaling to convert the load to an integer value. +*/
#define LOAD_SCALE 1000

/*+ The other statistics. +*/
Statistics *OtherStats=NULL;

/*+ The number of other statistics. +*/
long NOtherStats=0;

/*+ The interval between updates. +*/
static long interval;

/*+ The values for the /proc statistics, +*/
static long values[2][NProcStats], /*+ an array containing the previous and current values. +*/
            *current=values[0],    /*+ a pointer to the array of current values. +*/
            *previous=values[1];   /*+ a pointer to the array of previous values. +*/

/*+ The values for other statistics. +*/
static long *other=NULL;

/*+ The format of the statistics in /proc/net/dev. (changed in V ~2.1.28). +*/
static char *proc_net_dev_format=NULL;

/*+ The new format of the statistics in /proc/net/meminfo. (changed in V ~2.1.41). +*/
static int proc_meminfo_V2_1_41=0;


/*++++++++++++++++++++++++++++++++++++++
  Confirm the format of the files in /proc.
  ++++++++++++++++++++++++++++++++++++++*/

void VerifyProcStatistics(void)
{
 int error=0;
 FILE* f;
 char line[256];
 int i;

 /* Verify the statistics from /proc/stat */

 f=fopen("/proc/stat","r");
 if(!f)
    error=fprintf(stderr,"ProcMeter: Failed to open the '/proc/stat' file.\n"),
          fprintf(stderr,"           No cpu, cpu-user, cpu-nice, cpu-sys & cpu-idle\n"),
          fprintf(stderr,"           No disk, disk-rd & disk-wr\n"),
          fprintf(stderr,"           No page, page-in & page-out\n"),
          fprintf(stderr,"           No swap, swap-in & swap-out\n"),
          fprintf(stderr,"           No intr\n"),
          fprintf(stderr,"           No context\n");
 else
    if(!fgets(line,256,f))
       error=fprintf(stderr,"ProcMeter: Failed to read from the '/proc/stat' file.\n"),
             fprintf(stderr,"           No cpu, cpu-user, cpu-nice, cpu-sys & cpu-idle\n"),
             fprintf(stderr,"           No disk, disk-rd & disk-wr\n"),
             fprintf(stderr,"           No page, page-in & page-out\n"),
             fprintf(stderr,"           No swap, swap-in & swap-out\n"),
             fprintf(stderr,"           No intr\n"),
             fprintf(stderr,"           No context\n");
    else
      {
       unsigned long d0,d1,d2,d3;

       if(sscanf(line,"cpu %lu %lu %lu %lu",&current[CPU_USER],&current[CPU_NICE],&current[CPU_SYS],&current[CPU_IDLE])==4)
          ProcStats[CPU].avail=ProcStats[CPU_USER].avail=ProcStats[CPU_NICE].avail=ProcStats[CPU_SYS].avail=ProcStats[CPU_IDLE].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected 'cpu' line in the '/proc/stat' file.\n"),
                fprintf(stderr,"           No cpu, cpu-user, cpu-nice, cpu-sys & cpu-idle\n");

       fgets(line,256,f);
       if(sscanf(line,"disk %lu %lu %lu %lu",&d0,&d1,&d2,&d3)==4)
          ProcStats[DISK].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected 'disk' line in the '/proc/stat' file.\n"),
                fprintf(stderr,"           No disk\n");

       fgets(line,256,f);
       if(sscanf(line,"disk_rio %lu %lu %lu %lu",&d0,&d1,&d2,&d3)==4)
         {
          ProcStats[DISK_RD].avail=1;

          fgets(line,256,f);
          if(sscanf(line,"disk_wio %lu %lu %lu %lu",&d0,&d1,&d2,&d3)==4)
             ProcStats[DISK_WR].avail=1;
         }

       while(line[0]=='d')      /* kernel version > ~1.3.0 */
          fgets(line,256,f);

       if(sscanf(line,"page %lu %lu",&current[PAGE_IN],&current[PAGE_OUT])==2)
          ProcStats[PAGE].avail=ProcStats[PAGE_IN].avail=ProcStats[PAGE_OUT].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected 'page' line in the '/proc/stat' file.\n"),
                fprintf(stderr,"           No page, page-in & page-out\n");

       fgets(line,256,f);
       if(sscanf(line,"swap %lu %lu",&current[SWAP_IN],&current[SWAP_OUT])==2)
          ProcStats[SWAP].avail=ProcStats[SWAP_IN].avail=ProcStats[SWAP_OUT].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected 'swap' line in the '/proc/stat' file.\n"),
                fprintf(stderr,"           No swap, swap-in & swap-out\n");

       fgets(line,256,f);
       if(sscanf(line,"intr %lu",&current[INTR])==1)
          ProcStats[INTR].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected 'intr' line in the '/proc/stat' file.\n"),
                fprintf(stderr,"           No intr\n");

       fgets(line,256,f);
       if(sscanf(line,"ctxt %lu",&current[CONTEXT])==1)
          ProcStats[CONTEXT].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected 'ctxt' line in the '/proc/stat' file.\n"),
                fprintf(stderr,"           No context\n");
      }

 if(f)
    fclose(f);

 /* Verify the statistics from /proc/loadavg */

 f=fopen("/proc/loadavg","r");
 if(!f)
    error=fprintf(stderr,"ProcMeter: Failed to open the '/proc/loadavg' file.\n"),
          fprintf(stderr,"           No load\n"),
          fprintf(stderr,"           No proc\n");
 else
    if(!fgets(line,256,f))
       error=fprintf(stderr,"ProcMeter: Failed to read from the '/proc/loadavg' file.\n"),
             fprintf(stderr,"           No load\n"),
             fprintf(stderr,"           No proc\n");
    else
      {
       double d;

       fgets(line,256,f);
       if(sscanf(line,"%lf %*f %*f %*d/%ld",&d,&current[PROC])==2) /* kernel version > ~1.2.0 */
          ProcStats[LOAD].avail=ProcStats[PROC].avail=1;
       else
          if(sscanf(line,"%lf",&d)==1) /* kernel version < ~1.2.0 */
             ProcStats[LOAD].avail=1;
          else
             error=fprintf(stderr,"ProcMeter: Unexpected 'load' line in the '/proc/loadavg' file.\n"),
                   fprintf(stderr,"           No load\n"),
                   fprintf(stderr,"           No proc\n");
      }

 if(f)
    fclose(f);

 /* Verify the statistics from /proc/meminfo */

 f=fopen("/proc/meminfo","r");
 if(!f)
    error=fprintf(stderr,"ProcMeter: Failed to open the '/proc/meminfo' file.\n"),
          fprintf(stderr,"           No mem-free, mem-used, mem-buff, mem-cache & mem-swap\n");
 else
    if(!fgets(line,256,f))
       error=fprintf(stderr,"ProcMeter: Failed to read from the '/proc/meminfo' file.\n"),
             fprintf(stderr,"           No mem-free, mem-used, mem-buff, mem-cache & mem-swap\n");
    else
       if(strcmp(line,"        total:   used:    free:   shared:  buffers:\n") && /* kernel version < ~2.0.0 */
          strcmp(line,"        total:    used:    free:  shared: buffers:  cached:\n") && /* kernel version > ~2.0.0 */
          strncmp(line,"MemTotal:",9)) /* kernel version > ~2.1.41 */
          error=fprintf(stderr,"ProcMeter: Unexpected first line in '/proc/meminfo'.\n"),
                fprintf(stderr,"           No mem-free, mem-used, mem-buff, mem-cache & mem-swap\n");
       else
         {
          unsigned long mem_tot=0;

          proc_meminfo_V2_1_41=!strncmp(line,"MemTotal:",9);

          if(proc_meminfo_V2_1_41)
            {
             unsigned long swap_tot=0,swap_free=0;
             sscanf(line,"MemTotal: %lu",&mem_tot);
             if(fgets(line,256,f) && sscanf(line,"MemFree: %lu",&current[MEM_FREE])==1)
               {ProcStats[MEM_FREE].avail=ProcStats[MEM_USED].avail=1; current[MEM_USED]=mem_tot-current[MEM_FREE];}
             fgets(line,256,f); /* MemShared */
             if(fgets(line,256,f) && sscanf(line,"Buffers: %lu",&current[MEM_BUFF])==1)
                ProcStats[MEM_BUFF].avail=1;
             if(fgets(line,256,f) && sscanf(line,"Cached: %lu",&current[MEM_CACHE])==1)
                ProcStats[MEM_CACHE].avail=1;
             if(fgets(line,256,f) && sscanf(line,"SwapTotal: %lu",&swap_tot)==1 &&
                fgets(line,256,f) && sscanf(line,"SwapFree: %lu",&swap_free)==1)
               {ProcStats[MEM_SWAP].avail=1; current[MEM_SWAP]=swap_tot-swap_free;}
             if(!(ProcStats[MEM_FREE].avail&ProcStats[MEM_USED].avail&ProcStats[MEM_BUFF].avail&ProcStats[MEM_CACHE].avail&ProcStats[MEM_SWAP].avail))
               {
                error=fprintf(stderr,"ProcMeter: Unexpected line in the '/proc/meminfo' file.\n"),
                      fprintf(stderr,"           No mem-free, mem-used, mem-buff, mem-cache & mem-swap\n");
                ProcStats[MEM_FREE].avail=ProcStats[MEM_USED].avail=ProcStats[MEM_BUFF].avail=ProcStats[MEM_CACHE].avail=ProcStats[MEM_SWAP].avail=0;
               }
            }
          else
            {
             fgets(line,256,f);
             if(sscanf(line,"Mem: %lu %lu %lu %*u %lu %lu",&mem_tot,&current[MEM_USED],&current[MEM_FREE],&current[MEM_BUFF],&current[MEM_CACHE])==5)
                ProcStats[MEM_FREE].avail=ProcStats[MEM_USED].avail=ProcStats[MEM_BUFF].avail=ProcStats[MEM_CACHE].avail=1;
             else
                if(sscanf(line,"Mem: %lu %lu %lu %*u %lu",&mem_tot,&current[MEM_USED],&current[MEM_FREE],&current[MEM_BUFF])==4)
                   ProcStats[MEM_FREE].avail=ProcStats[MEM_USED].avail=ProcStats[MEM_BUFF].avail=1;
                else
                   error=fprintf(stderr,"ProcMeter: Unexpected 'Mem' line in the '/proc/meminfo' file.\n"),
                         fprintf(stderr,"           No mem-free, mem-used, mem-buff & mem-cache\n");

             fgets(line,256,f);
             if(sscanf(line,"Swap: %*u %lu",&current[MEM_SWAP])==1)
                ProcStats[MEM_SWAP].avail=1;
             else
                error=fprintf(stderr,"ProcMeter: Unexpected 'Swap' line in the '/proc/meminfo' file.\n"),
                      fprintf(stderr,"           No mem-swap\n");

             mem_tot>>=10;
             current[MEM_FREE ]>>=10;
             current[MEM_USED ]>>=10;
             current[MEM_BUFF ]>>=10;
             current[MEM_CACHE]>>=10;
             current[MEM_SWAP ]>>=10;
            }

          if(ProcStats[MEM_FREE].avail && (mem_tot>>=14))
            {
             long scale=1;
             char str[6];

             while(mem_tot)
               {mem_tot>>=1; scale<<=1;}

             sprintf(str,"%ld MB",scale);

             scale<<=10;

             strcpy(ProcStats[MEM_FREE ].units,str); ProcStats[MEM_FREE ].scale=scale;
             strcpy(ProcStats[MEM_USED ].units,str); ProcStats[MEM_USED ].scale=scale;
             strcpy(ProcStats[MEM_BUFF ].units,str); ProcStats[MEM_BUFF ].scale=scale;
             strcpy(ProcStats[MEM_CACHE].units,str); ProcStats[MEM_CACHE].scale=scale;
             strcpy(ProcStats[MEM_SWAP ].units,str); ProcStats[MEM_SWAP ].scale=scale;
            }
         }

 if(f)
    fclose(f);

 /* Verify the statistics from /proc/net/dev */

 f=fopen("/proc/net/dev","r");
 if(!f)
    error=fprintf(stderr,"ProcMeter: Failed to open the '/proc/net/dev' file.\n"),
          fprintf(stderr,"           No lpkt\n"),
          fprintf(stderr,"           No fpkt, fpkt-rx, fpkt-tx & fpkt-coll\n"),
          fprintf(stderr,"           No spkt, spkt-rx & spkt-tx\n");
 else
    if(!fgets(line,256,f))
       error=fprintf(stderr,"ProcMeter: Failed to read from the '/proc/net/dev' file.\n"),
             fprintf(stderr,"           No lpkt\n"),
             fprintf(stderr,"           No fpkt, fpkt-rx, fpkt-tx & fpkt-coll\n"),
             fprintf(stderr,"           No spkt, spkt-rx & spkt-tx\n");
    else
       if(strcmp(line,"Inter-|   Receive                  |  Transmit\n") && /* all existing kernel versions */
          strcmp(line,"Inter-|   Receive                           |  Transmit\n")) /* possible future change */
          error=fprintf(stderr,"ProcMeter: Unexpected header line 1 in the '/proc/net/dev' file.\n"),
                fprintf(stderr,"           No lpkt\n"),
                fprintf(stderr,"           No fpkt, fpkt-rx, fpkt-tx & fpkt-coll\n"),
                fprintf(stderr,"           No spkt, spkt-rx & spkt-tx\n");
       else
         {
          fgets(line,256,f);
          if(strcmp(line," face |packets errs drop fifo frame|packets errs drop fifo colls carrier\n") && /* kernel version < ~2.1.28 */
             strcmp(line," face |bytes    packets errs drop fifo frame|bytes    packets errs drop fifo colls carrier\n")) /* kernel version > ~2.1.28 */
             error=fprintf(stderr,"ProcMeter: Unexpected header line 2 in the '/proc/net/dev' file.\n"),
                   fprintf(stderr,"           No lpkt\n"),
                   fprintf(stderr,"           No fpkt, fpkt-rx, fpkt-tx & fpkt-coll\n"),
                   fprintf(stderr,"           No spkt, spkt-rx & spkt-tx\n");
          else
            {
             if(!strcmp(line," face |packets errs drop fifo frame|packets errs drop fifo colls carrier\n"))
                proc_net_dev_format="%lu %*u %*u %*u %*u %lu %*u %*u %*u %lu";
             else
                proc_net_dev_format="%*u %lu %*u %*u %*u %*u %*u %lu %*u %*u %*u %lu";

             while(fgets(line,256,f))
               {
                int i;
                char intf[2];
                long rx=0,tx=0,coll=0;

                for(i=strlen(line);i>6 && line[i]!=':';i--); ++i;
                if(sscanf(line,"%1s",intf)==1 && sscanf(&line[i],proc_net_dev_format,&rx,&tx,&coll)==3)
                  {
                   if(*intf=='l' || *intf=='d') /* 'lo' or 'dummy' devices. */
                      ProcStats[LPKT].avail=1;
                   else
                      if(*intf=='s' || *intf=='p' || *intf=='f' || *intf=='i') /* 'sl' or 'ppp'/'plip' or 'flip' or 'isdn'/'ippp' devices. */
                         ProcStats[SPKT].avail=ProcStats[SPKT_RX].avail=ProcStats[SPKT_TX].avail=1;
                      else /* other devices */
                         ProcStats[FPKT].avail=ProcStats[FPKT_RX].avail=ProcStats[FPKT_TX].avail=ProcStats[FPKT_COLL].avail=1;
                  }
                else
                   if(strcmp(&line[i]," No statistics available.\n"))
                      error=fprintf(stderr,"ProcMeter: Unexpected device line in the '/proc/net/dev' file.\n"),
                            fprintf(stderr,"           No ?pkt, ?pkt-rx & ?pkt-tx (maybe)\n");
               }
            }
         }

 if(f)
    fclose(f);

 /* Verify the statistics from /proc/apm */

 f=fopen("/proc/apm","r");
 if(!f)
    ;                           /* Don't bother giving an error message for 99% of systems. */
 else
    if(!fgets(line,256,f))
       error=fprintf(stderr,"ProcMeter: Failed to read from the '/proc/apm' file.\n"),
             fprintf(stderr,"           No batt-life, batt-lvl\n");
    else
      {
       float ver;

       fgets(line,256,f);
       if(sscanf(line,"%f %*f %*x %*x %*x %lx %ld%%",&ver,&current[BATT_LVL],&current[BATT_LIFE])==3 && ver>1.199999 && ver<1.20001)
          ProcStats[BATT_LIFE].avail=ProcStats[BATT_LVL].avail=1;
       else
          error=fprintf(stderr,"ProcMeter: Unexpected line in the '/proc/apm' file.\n"),
                fprintf(stderr,"           No batt-life, batt-lvl\n");
      }

 if(f)
    fclose(f);

 /* Print an error message if needed. */

 if(error)
   {
    fprintf(stderr,"\n"
                   "ProcMeter: Warning, one or more incompatibilities in /proc were encountered in\n"
                   "           the initialisation of the meters.  This will lead to one or more of\n"
                   "           the meters being unavailable.\n");
    fprintf(stderr,"           Please report this to ProcMeter's author amb@gedanken.demon.co.uk,\n"
                   "           stating the version of ProcMeter, the error message, the contents\n"
                   "           of the file in error in /proc and the version of the Linux kernel.\n"
                   "\n");

    for(i=0;i<NProcStats;i++)
       if(ProcStats[i].avail)
          break;
    if(i==NProcStats)
      {
       fprintf(stderr,"ProcMeter: There are no statistics available\n"
                      "           Are you sure that /proc is mounted?\n");
       exit(1);
      }
   }

 /* Get ready to run. */

 for(i=0;i<NProcStats;i++)
    current[i]=previous[i]=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Verify the statistics in the named file.

  char *filename The name of the file containing the statistics.
  ++++++++++++++++++++++++++++++++++++++*/

void VerifyOtherStatistics(char *filename)
{
 int error=0;
 FILE* f;
 char line[256];
 struct stat buf;
 time_t now;
 char *datfile=(char*)malloc(strlen(filename)+5);
 char *deffile=(char*)malloc(strlen(filename)+5);

 strcpy(datfile,filename);
 strcpy(deffile,datfile);

 if(strlen(datfile)>4 && !strcmp(datfile+strlen(datfile)-4,".dat"))
    strcpy(deffile+strlen(datfile)-4,".def");
 else
    strcat(deffile,".def");

 while(stat(deffile,&buf))
   {
    fprintf(stderr,"ProcMeter: Waiting for the file '%s' to appear.\n",deffile);
    sleep(1);
   }

 now=time(NULL);
 while((now-buf.st_mtime)<1)
   {
    fprintf(stderr,"ProcMeter: Waiting for the file '%s' to stabilise.\n",deffile);
    sleep(1);
    stat(deffile,&buf);
    now=time(NULL);
   }

 f=fopen(deffile,"r");
 if(!f)
    error=fprintf(stderr,"ProcMeter: Failed to open the '%s' file.\n",deffile),
          fprintf(stderr,"           No statistics from '%s'\n",datfile);
    else
       while(fgets(line,256,f))
          if(*line!='#')
            {
             NOtherStats++;
             other=(long*)realloc(other,sizeof(long)*(NOtherStats+1));
             OtherStats=(Statistics*)realloc(OtherStats,sizeof(Statistics)*(NOtherStats+1));
             if(sscanf(line,"%15s %7s %ld",OtherStats[NOtherStats].name,
                                           OtherStats[NOtherStats].units,
                                          &OtherStats[NOtherStats].scale)!=3)
               {
                int i;
                line[strlen(line)-1]=0;
                NOtherStats--;
                error=fprintf(stderr,"ProcMeter: Error parsing the line '%s' from the '%s' file.\n",line,deffile),
                      fprintf(stderr,"           No statistics from '%s'\n",datfile);
                for(i=NOtherStats;i>0;i--)
                   if(OtherStats[i].file==datfile)
                      OtherStats[i].avail=0;
               }
             else
               {
                OtherStats[NOtherStats].file=datfile;
                OtherStats[NOtherStats].avail=!error;
                OtherStats[NOtherStats].used=0;
                OtherStats[NOtherStats].value=0;
               }
            }

 if(f)
    fclose(f);
}


/*++++++++++++++++++++++++++++++++++++++
  Find out the statistics by looking in /proc and the other files.
  ++++++++++++++++++++++++++++++++++++++*/

void GetStatistics(void)
{
 FILE* f;
 char line[256];

 static int which=0;
 static struct timeval t[2],*cur_time,*prev_time;

 /* Select which of the two sets of values to use. */

 previous=values[which];
 prev_time=&t[which];

 which=which^1;

 current=values[which];
 cur_time=&t[which];

 /* Get the time */

 gettimeofday(cur_time,NULL);

 /* Get the statistics from /proc/stat */

 if((ProcStats[CPU].used || ProcStats[CPU_USER].used || ProcStats[CPU_NICE].used || ProcStats[CPU_SYS].used || ProcStats[CPU_IDLE].used) ||
    (ProcStats[SWAP].used || ProcStats[SWAP_IN].used || ProcStats[SWAP_OUT].used) ||
    (ProcStats[PAGE].used || ProcStats[PAGE_IN].used || ProcStats[PAGE_OUT].used) ||
    (ProcStats[DISK].used || ProcStats[DISK_RD].used || ProcStats[DISK_WR].used) ||
    (ProcStats[INTR].used || ProcStats[CONTEXT].used))
   {
    unsigned long d0,d1,d2,d3;

    f=fopen("/proc/stat","r");
    if(!f)
      {fprintf(stderr,"ProcMeter: Failed to open the '/proc/stat' file.\n");exit(1);}

    fgets(line,256,f);
    if(ProcStats[CPU].avail)
       sscanf(line,"cpu %lu %lu %lu %lu",&current[CPU_USER],&current[CPU_NICE],&current[CPU_SYS],&current[CPU_IDLE]);
    fgets(line,256,f);
    if(ProcStats[DISK].avail)
      {sscanf(line,"disk %lu %lu %lu %lu",&d0,&d1,&d2,&d3);current[DISK]=d0+d1+d2+d3;}
    if(ProcStats[DISK_RD].avail)
      {fgets(line,256,f);
       sscanf(line,"disk_rio %lu %lu %lu %lu",&d0,&d1,&d2,&d3);current[DISK_RD]=d0+d1+d2+d3;}
    if(ProcStats[DISK_WR].avail)
      {fgets(line,256,f);
       sscanf(line,"disk_wio %lu %lu %lu %lu",&d0,&d1,&d2,&d3);current[DISK_WR]=d0+d1+d2+d3;}
    while(line[0]=='d')      /* kernel version > ~1.3.x */
       fgets(line,256,f);
    if(ProcStats[PAGE].avail)
       sscanf(line,"page %lu %lu",&current[PAGE_IN],&current[PAGE_OUT]);
    fgets(line,256,f);
    if(ProcStats[SWAP].avail)
       sscanf(line,"swap %lu %lu",&current[SWAP_IN],&current[SWAP_OUT]);
    fgets(line,256,f);
    if(ProcStats[INTR].avail)
       sscanf(line,"intr %lu",&current[INTR]);
    fgets(line,256,f);
    if(ProcStats[CONTEXT].avail)
       sscanf(line,"ctxt %lu",&current[CONTEXT]);

    fclose(f);
   }

 /* Get the statistics from /proc/loadavg */

 if(ProcStats[LOAD].used || ProcStats[PROC].used)
   {
    double d;

    f=fopen("/proc/loadavg","r");
    if(!f)
      {fprintf(stderr,"ProcMeter: Failed to open the '/proc/loadavg' file.\n");exit(1);}

    fgets(line,256,f);
    sscanf(line,"%lf %*f %*f %*d/%ld",&d,&current[PROC]);
    current[LOAD]=(long)(d*LOAD_SCALE);

    fclose(f);
   }

 /* Get the statistics from /proc/meminfo */

 if(ProcStats[MEM_FREE].used || ProcStats[MEM_USED].used || ProcStats[MEM_BUFF].used || ProcStats[MEM_CACHE].used || ProcStats[MEM_SWAP].used)
   {
    f=fopen("/proc/meminfo","r");
    if(!f)
      {fprintf(stderr,"ProcMeter: Failed to open the '/proc/meminfo' file.\n");exit(1);}

    if(proc_meminfo_V2_1_41)
      {
       unsigned long mem_tot=0,swap_tot=0,swap_free=0;
       fgets(line,256,f);
       sscanf(line,"MemTotal: %lu",&mem_tot);
       fgets(line,256,f);
       sscanf(line,"MemFree: %lu",&current[MEM_FREE]);
       current[MEM_USED]=mem_tot-current[MEM_FREE];
       fgets(line,256,f); /* MemShared */
       fgets(line,256,f);
       sscanf(line,"Buffers: %lu",&current[MEM_BUFF]);
       fgets(line,256,f);
       sscanf(line,"Cached: %lu",&current[MEM_CACHE]);
       fgets(line,256,f);
       sscanf(line,"SwapTotal: %lu",&swap_tot);
       fgets(line,256,f);
       sscanf(line,"SwapFree: %lu",&swap_free);
       current[MEM_SWAP]=swap_tot-swap_free;
      }
    else
      {
       fgets(line,256,f);
       fgets(line,256,f);
       if(ProcStats[MEM_FREE].avail)
          sscanf(line,"Mem: %*u %lu %lu %*u %lu %lu",&current[MEM_USED],&current[MEM_FREE],&current[MEM_BUFF],&current[MEM_CACHE]);
       fgets(line,256,f);
       if(ProcStats[MEM_SWAP].avail)
          sscanf(line,"Swap: %*u %lu",&current[MEM_SWAP]);

       current[MEM_FREE ]>>=10;
       current[MEM_USED ]>>=10;
       current[MEM_BUFF ]>>=10;
       current[MEM_CACHE]>>=10;
       current[MEM_SWAP ]>>=10;
      }

    fclose(f);
   }

 /* Get the statistics from /proc/net/dev */

 if((ProcStats[LPKT].used) ||
    (ProcStats[FPKT].used || ProcStats[FPKT_RX].used || ProcStats[FPKT_TX].used || ProcStats[FPKT_COLL].used) ||
    (ProcStats[SPKT].used || ProcStats[SPKT_RX].used || ProcStats[SPKT_TX].used))
   {
    f=fopen("/proc/net/dev","r");
    if(!f)
      {fprintf(stderr,"ProcMeter: Failed to open the '/proc/net/dev' file.\n");exit(1);}

    current[LPKT]=0;
    current[FPKT_RX]=current[FPKT_TX]=0;
    current[SPKT_RX]=current[SPKT_TX]=0;
    current[FPKT_COLL]=0;
    fgets(line,256,f);
    fgets(line,256,f);
    while(fgets(line,256,f))
      {
       int i;
       char intf[2];
       long rx=0,tx=0,coll=0;

       sscanf(line,"%1s",intf);
       for(i=strlen(line);i>6 && line[i]!=':';i--); ++i;
       sscanf(&line[i],proc_net_dev_format,&rx,&tx,&coll);
       if(*intf=='l' || *intf=='d') /* 'lo' or 'dummy' devices. */
          current[LPKT]+=tx;
       else
          if(*intf=='s' || *intf=='p' || *intf=='f' || *intf=='i') /* 'sl' or 'ppp'/'plip' or 'flip' or 'isdn'/'ippp' devices. */
            {current[SPKT_RX]+=rx;current[SPKT_TX]+=tx;}
          else /* other devices */
            {current[FPKT_RX]+=rx;current[FPKT_TX]+=tx;current[FPKT_COLL]+=coll;}
      }

    fclose(f);
   }

 /* Get the statistics from /proc/apm */

 if(ProcStats[BATT_LIFE].used || ProcStats[BATT_LVL].used)
   {
    long lvl;

    f=fopen("/proc/apm","r");
    if(!f)
      {fprintf(stderr,"ProcMeter: Failed to open the '/proc/apm' file.\n");exit(1);}

    fgets(line,256,f);
    sscanf(line,"%*f %*f %*x %*x %*x %lx %ld%%",&lvl,&current[BATT_LIFE]);

    if(lvl==0xff)               /* known to be unknown */
       current[BATT_LVL]=0;
    else if(lvl&0x04)           /* critical */
       current[BATT_LVL]=1;
    else if(lvl&0x02)           /* low */
       current[BATT_LVL]=2;
    else if(lvl&0x01)           /* high */
       current[BATT_LVL]=3;
    else                        /* other means unknown */
       current[BATT_LVL]=0;

    fclose(f);
   }

 /* The derived values */

 if(ProcStats[CPU].avail)
    current[CPU]=current[CPU_USER]+current[CPU_NICE]+current[CPU_SYS];

 if(ProcStats[PAGE].avail)
    current[PAGE]=current[PAGE_IN]+current[PAGE_OUT];

 if(ProcStats[SWAP].avail)
    current[SWAP]=current[SWAP_IN]+current[SWAP_OUT];

 if(ProcStats[MEM_BUFF].avail)
    current[MEM_USED]-=current[MEM_BUFF];
 if(ProcStats[MEM_CACHE].avail)
    current[MEM_USED]-=current[MEM_CACHE];

 if(ProcStats[FPKT].avail)
    current[FPKT]=current[FPKT_RX]+current[FPKT_TX];
 if(ProcStats[SPKT].avail)
    current[SPKT]=current[SPKT_RX]+current[SPKT_TX];

 /* Get the other statistics. */

 if(NOtherStats)
   {
    int i;

    for(i=1;i<=NOtherStats;)
      {
       int j,used=0;

       for(j=i;j<=NOtherStats;j++)
          if(OtherStats[j].file!=OtherStats[i].file)
             break;
          else
             if(OtherStats[j].used)
                used=1;

       if(used)
         {
          f=fopen(OtherStats[i].file,"r");
          if(!f)
            {fprintf(stderr,"ProcMeter: Failed to open the '%s' file.\n",OtherStats[i].file);exit(1);}

          for(;i<j;i++)
             if(!fgets(line,256,f))
               {fprintf(stderr,"ProcMeter: Failed to read from the '%s' file.\n",OtherStats[i].file);exit(1);}
             else
                sscanf(line,"%ld",&other[i]);

          fclose(f);
         }
       else
          i=j;
      }
   }

 /* Get the time interval. */

 interval=(cur_time->tv_sec-prev_time->tv_sec)*1000+(cur_time->tv_usec-prev_time->tv_usec)/1000;
}


/*++++++++++++++++++++++++++++++++++++++
  Return the current value of the statistic.

  double CurrentValue Returns the current value for the specified statistic.

  int type The type of the Statistics, +ve for ProcStats, -ve for OtherStats.
  ++++++++++++++++++++++++++++++++++++++*/

double CurrentValue(int type)
{
 int tot;
 double value;

 if(type<0)
    value=other[-type];
 else
    switch(type)
      {
      case CPU:
      case CPU_USER:
      case CPU_NICE:
      case CPU_SYS:
      case CPU_IDLE:
       tot=current[CPU]+current[CPU_IDLE]-previous[CPU]-previous[CPU_IDLE];
       if(tot)
          value=100.0*(double)(current[type]-previous[type])/(double)tot;
       else
          value=0.0;
       if(value>100.0)
          value=100.0;
       break;

      case LOAD:
       value=(double)current[type]/(double)LOAD_SCALE;
       break;

      case PROC:

      case MEM_USED:
      case MEM_FREE:
      case MEM_BUFF:
      case MEM_CACHE:
      case MEM_SWAP:

      case BATT_LIFE:
      case BATT_LVL:
       value=(double)current[type];
       break;

      default:
       if(interval)
          value=1000.0*(double)(current[type]-previous[type])/interval;
       else
          value=0.0;
      }

 if(value<0.0)
    value=0.0;

 return(value);
}
