/* File: 	proc_driver.c
 *
 * Description: Code for the /proc driver module
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 2002, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	01/12/2002	Created
 *            
 *
 */

#include <stdio.h>		
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <varargs.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef __linux__
#include <sys/mkdev.h>		/* To get major/minor numbers 		 */
#endif
#include <sys/param.h>		/* To get PAGESIZE i.e. system page	 */
#include <dirent.h>		/* To get the /proc directory listing	 */
#include <unistd.h>

#include "os_info.h"

#include "tree.h"	/* Generic m-ary tree package include file */
#include "list.h"	/* Simple list package			   */
#include "debug.h"	/* Macro based debug package		   */

#include "machine_info.h"	/* Some process wide definitions	   */

/* ----------------------- /proc routines ----------------- */


typedef struct _proc_scanner_info
{
        DIR     *dirp;              /* Directory ptr, opened  once */
	int      first;
} Proc_scanner_info;

int 
proc_init_pid_scanner( machine, dir_path )
Machine_info *machine;
char         *dir_path;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) malloc( sizeof( Proc_scanner_info ));
	if ( psi == NULL )
	{
		fprintf( stderr, "Out of memory allocating proc info scanner\n");
		return -1;
	}

	machine->data = (void *) psi;


	psi->dirp = opendir( dir_path );
	if ( psi->dirp == NULL )
	{
		perror("opendir");
		return  -1;
	}

	psi->first = 1;

	/* printf("proc scanner intialized\n"); */

	return 0;
}

int 
proc_get_first_pid( machine )
Machine_info *machine;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) machine->data;

	if ( !psi )       return -1;
	if ( !psi->dirp ) return -1;

	if ( psi->first )
	    psi->first = 0;
	else
	    rewinddir( psi->dirp );

	return  machine->get_next_pid( machine );
}

int
proc_get_next_pid( machine )
Machine_info *machine;
{
	int 		done    = 0;
	int 		pid     = -1;
	char		*fname;
	char		*e;		/* End of number in file name */
	struct  dirent 	*direntp;
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) machine->data;

	if ( !psi ) return -1;

	while ( ! done )
	{
	    while ( (direntp = readdir( psi->dirp )) != NULL )
	    {
		fname = direntp->d_name;

		if ( 	( fname[0] == '.' ) &&   /* Skip . */
			( fname[1] == NULL_CHAR ))
		{
			continue;
		}

		if ( 	( fname[0] == '.' ) &&   /* Skip .. */
			( fname[1] == '.' ) &&
			( fname[2] == NULL_CHAR ))
		{
			continue;
		}
		
		/* If not a number, then skip it. Some systems have
		 * other directories/files in the /proc directory
		 */

		pid = strtol( fname, &e, 10 );
		if ( e == fname )
		{
			continue;		/* not a number */
		}

		break;
	    }

	    if ( direntp == NULL )
	    	pid  = -1;

	    done = 1;
	}

	return pid;
}

int 
proc_close_pid_scanner( machine )
Machine_info *machine;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) machine->data;

	if ( !psi ) return -1;

	closedir( psi->dirp );

	return 0;
}

int 
proc_cleanup_pid_scanner( machine, pid )
Machine_info *machine;
int pid;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) machine->data;

	if ( !psi ) return -1;

	/* This is a stub on internal scanner */

	return 0;
}

int 
proc_cache_process_details( machine, process_list ) 	/* Stub for now */
Machine_info *machine;
lnode *process_list;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;
	psi = (Proc_scanner_info *) machine->data;
	if ( !psi ) return -1;
	return 0;
}

int 
proc_get_pid_main_details( machine, pid ) 	/* Stub for now */
Machine_info *machine;
int pid;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;
	psi = (Proc_scanner_info *) machine->data;
	if ( !psi ) return -1;
	return 0;
}

int 
proc_get_pid_extra_details( machine, pid )
Machine_info *machine;
int pid;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) machine->data;

	if ( !psi ) return -1;

	/* Stub for now */

	return 0;
}

int 
proc_release_process_details( machine, process_list ) 	/* Stub for now */
Machine_info *machine;
lnode *process_list;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;
	psi = (Proc_scanner_info *) machine->data;
	if ( !psi ) return -1;
	return 0;
}

int 
proc_get_machine_status( machine, pid )
Machine_info *machine;
int pid;
{
	Proc_scanner_info *psi;

	if ( !machine ) return -1;

	psi = (Proc_scanner_info *) machine->data;

	if ( !psi ) return -1;

	/* Stub for now */

	return 0;
}
