/* File: 	popup_menu.c
 *
 * Description: Routines to create and process popup menu's.
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1995, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	4/14/95	Created
 *
 */

#include <sys/time.h>
#include <unistd.h>
#include <errno.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/RowColumn.h>


#include "libXs.h"

#include "debug.h"

#include <Xpsi/Tree.h>

#include "tree.h"
#include "list.h"

#include "treeps.h"
#include "user_group.h"

Widget setup_node_popup_menu();
Widget setup_tree_popup_menu();

void 		do_node();
void 		do_tree_popup();

void 		do_renice();
void 		do_renice_subtree();

xs_menu_struct RenicePopupMenu[]={
	{"+1   (Lowers Priority)", "L", NULL_STR, do_renice, "1",  
					NULL, 0, NULL_STR},
	{"+5", NULL, NULL_STR, do_renice, "5",  
					NULL, 0, NULL_STR},
	{"-1   (Raises Priority)", "R", NULL_STR, do_renice, "-1",
					NULL, 0, NULL_STR},
	{"-5", "P", NULL_STR, do_renice, "-5",  
					NULL, 0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL,
                                                           0, NULL_STR},
	{"+1   On Subtree", NULL, NULL_STR, do_renice_subtree, "1",  
					NULL, 0, NULL_STR},
	{"+5   On Subtree", NULL, NULL_STR, do_renice_subtree, "5",  
					NULL, 0, NULL_STR},
	{"-1   On Subtree", NULL, NULL_STR, do_renice_subtree, "-1",  
					NULL, 0, NULL_STR},
	{"-5   On Subtree", NULL, NULL_STR, do_renice_subtree, "-5",  
					NULL, 0, NULL_STR},

};

xs_menu_struct NodePopupData[]={
#ifdef EXTERNAL_CONTROL_PANELS
	{"Process Controls","P", NULL_STR, do_node, "ctrl", NULL, 
							0, NULL_STR},
#else
	{"Man Page","M", NULL_STR, do_node, "man", NULL, 
							0, NULL_STR},
#ifdef LINUX
	{"Trace System Calls","T", NULL_STR, do_node, "strace", NULL, 
							0, NULL_STR},
#else
	{"Trace System Calls","T", NULL_STR, do_node, "truss", NULL, 
							0, NULL_STR},
#endif
	{"Change Nice Value","N", NULL_STR, NULL, NULL_STR, 
			(struct _menu_struct *) RenicePopupMenu,
			XtNumber(RenicePopupMenu), "Change Process Nice Value(s)"},
#endif
	{"External Script","M", NULL_STR, do_node, "starter", NULL, 
							0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL,
                                                           0, NULL_STR},
	{"Select Process", "t", NULL_STR, do_node, "select",  NULL, 
							0, NULL_STR},
	{"Display Details","D", NULL_STR, do_node, "details", NULL, 
							0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL,
                                                           0, NULL_STR},
	{"Signal Process", "S", NULL_STR, do_node, "signal",  NULL, 
							0, NULL_STR},
	{"Kill Process", "K", NULL_STR, do_node, "kill",  NULL, 
							0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL,
                                                           0, NULL_STR},
	{"Hide Subtree",   "H", NULL_STR, do_node, "hide",    NULL, 
							0, NULL_STR},
	{"Show Subtree",   "S", NULL_STR, do_node, "show",    NULL, 
							0, NULL_STR},
	{"Outline Subtree","O", NULL_STR, do_node, "outline", NULL, 
							0, NULL_STR},
};

xs_menu_struct TreePopupData[]={
	{"Show All Hidden Processes",  "S", NULL_STR, 
			do_tree_popup,"show_all",NULL,0,NULL_STR},
	{"Toggle Tree Outline","O",NULL_STR,
			do_tree_popup,"outline",NULL,0,NULL_STR},
};

extern Popup_on_pid;
extern tnode *Head;

extern User_group_info *Users;

extern tnode *display_node_widget();
extern tnode *redisplay_node();

extern Debug;

tnode *outline_all();

int authorizeRenice();

extern int errno;
extern char **Environment;

Widget
setup_node_popup_menu( parent )
Widget parent;
{
	Widget 	menu;
	Arg 	wargs[10];
	int 	n;

	DEBUG0(1, "setup_node_popup_menu\n" );

	n = 0;
	menu = XmCreatePopupMenu( parent, "ProcessMenu", wargs, n );

	xs_create_menu_buttons( NULL, menu, NodePopupData, 
						    XtNumber(NodePopupData));

	return( menu );
}

Widget
setup_tree_popup_menu( parent )
Widget parent;
{
	Widget 	menu;
	Arg 	wargs[10];
	int 	n;

	DEBUG0(1, "setup_tree_popup_menu\n" );

	n = 0;
	menu = XmCreatePopupMenu( parent, "TreeMenu", wargs, n );

	xs_create_menu_buttons( NULL, menu, TreePopupData, 
						XtNumber(TreePopupData));
	return( menu );
}

doMan( programName )
char *programName;
{
	char cmdStr[256];

	DEBUG1(1, "doMan: on(%s)\n", programName );
	
	sprintf( cmdStr, 
	          "%s %s %s %s %s> /dev/null 2>&1 &", 
		  CMDTOOL_PATH,
		  CMDTOOL_ARGS,
		  MAN_CMD_PATH,
		  MAN_CMD_ARGS,
		  programName );

	spawnExternal( cmdStr );
}

doShellProgram( programName )
char *programName;
{
	char cmdStr[256];

	DEBUG1(1, "doShellProgram: on(%s)\n", programName );
	
	sprintf( cmdStr, 
	          "%s %s %s> /dev/null 2>&1 &", 
		  CMDTOOL_PATH,
		  CMDTOOL_ARGS,
		  programName );

	spawnExternal( cmdStr );
}

void
doStarter( pid, cmdName )
int 	pid;
char   *cmdName;
{
	char cmdStr[256];

	DEBUG2(1, "doStarter: on %s(%d)\n", cmdName, pid );
	
	sprintf( cmdStr, 
	          "tpsExtAppStarter %d %s > /dev/null 2>&1 &",
		  pid,
		  cmdName );

	spawnExternal( cmdStr );
}
void
doControlPanel( pid, cmdName )
int 	pid;
char   *cmdName;
{
	char cmdStr[256];

	DEBUG2(1, "doControlPanel: on %s(%d)\n", cmdName, pid );
	
	sprintf( cmdStr, 
	          "processControlPanel -p %d -n %s > /dev/null 2>&1 &",
		  pid,
		  cmdName );

	spawnExternal( cmdStr );
}


doStrace( pid )
int pid;
{
	char cmdStr[256];
	int effective_uid=0;

	DEBUG1(1, "doStrace: on(%d)\n", pid );

	effective_uid = geteuid();

	if ( effective_uid != 0 )
	{
	    user_alert(TreeWidget,"Cannot Trace Process!! Treeps needs to be setuid root.");
	    return;
	}
	else
	{
	    sprintf( cmdStr, 
	          "%s %s %s %s %d > /dev/null 2>&1 &", 
		  CMDTOOL_PATH,
		  CMDTOOL_ARGS,
		  TRACE_CMD_PATH,
		  TRACE_CMD_ARGS,
		  pid );
	}

	spawnTrustedExternal( cmdStr );
}

doTruss( pid )
int pid;
{
	char cmdStr[256];
	int effective_uid=0;

	DEBUG1(1, "doTruss: on(%d)\n", pid );

	effective_uid = geteuid();

	if ( effective_uid != 0 )
	{
	    user_alert(TreeWidget,"Cannot Trace Process!! Treeps needs to be setuid root.");
	    return;
	}
	else
	{
	    sprintf( cmdStr, 
	          "%s %s %s %s %d > /dev/null 2>&1 &", 
		  CMDTOOL_PATH,
		  CMDTOOL_ARGS,
		  TRUSS_CMD_PATH,
		  TRUSS_CMD_ARGS,
		  pid );
	}

	spawnTrustedExternal( cmdStr );
}

void 
do_node( w, client_data, call_data)
Widget w;
caddr_t client_data; 
XmPushButtonCallbackStruct  *call_data;
{
	int        	   new_value;
	int	   	   old_value;
	tnode	   	  *t_ptr;
	Process_tree_info *pti_ptr;
	XEvent	 	  *event;
	XButtonEvent	  *but_event;
	int		   doManCmd=0;
	int		   doCtrlPanel=0;
	int		   doStarterApp=0;
	sysProcInfo *pp;

	DEBUG1( 1, "do_node client data(%s)\n", client_data);

	/* Perform a one time only action on the process */

	old_value = Process_action;

	if ( strcmp( client_data, "ctrl") == 0 )
	{
	 	doCtrlPanel++;
	}
	else if ( strcmp( client_data, "man" ) == 0 )
	{
	 	doManCmd++;
	}
	else if ( strcmp( client_data, "select") == 0 )
	{
	 	new_value = ACTION_PROCESS_SELECT;
	}
	else if ( strcmp( client_data, "details") == 0 )
	{
	 	new_value = ACTION_PROCESS_DETAILS;
	}
	else if ( strcmp( client_data, "signal") == 0 )
	{
	 	new_value = ACTION_PROCESS_SIGNAL;
	}
	else if ( strcmp( client_data, "kill") == 0 )
	{
	 	new_value = ACTION_KILL_PROCESS;
	}
	else if ( strcmp( client_data, "outline") == 0 )
	{
	 	new_value = ACTION_OUTLINE_SUBTREE;
	}
	else if ( strcmp( client_data, "hide") == 0 )
	{
	 	new_value = ACTION_HIDE_SUBTREE;
	}
	else if ( strcmp( client_data, "show") == 0 )
	{
	 	new_value = ACTION_SHOW_SUBTREE;
	}
	else if ( strcmp( client_data, "strace") == 0 )
	{
		doStrace( Popup_on_pid );
	 	return;
	}
	else if ( strcmp( client_data, "truss") == 0 )
	{
		doTruss( Popup_on_pid );
	 	return;
	}
	else if ( strcmp( client_data, "starter") == 0 )
	{
		doStarterApp++;
	}


	Process_action = new_value;

	/* Corrupt the event pointer button to appear as if a selection
	 * was performed. We then lookup the node associated with the
	 * popup menu and call select_node to perform the desired 
	 * single event. After the one shot event, we set the Process_action
	 * back.
	 */

	event = call_data->event;

	but_event = (XButtonEvent *)event;

	but_event->button = 1;

	t_ptr = walk_tree( Head, match_key, Popup_on_pid );
	if ( t_ptr == NULL )
	{
		DEBUG1(0,"do_node: Failed to find process(%d)\n",Popup_on_pid);
		Process_action = old_value;
		return;
	}

	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr == NULL )
	{
		DEBUG1(0, "do_node: process(%d) NULL pti_ptr\n", Popup_on_pid);
		Process_action = old_value;
		return;
	}

	if ( doManCmd )
	{
		pp = pti_ptr->pp;
		doMan( PROCESS_CMD_NAME(pp) );
	}
	else if ( doCtrlPanel )
	{
		pp = pti_ptr->pp;
		doControlPanel( Popup_on_pid, PROCESS_CMD_NAME(pp) );
	}
	else if ( doStarterApp )
	{
		pp = pti_ptr->pp;
		doStarter( Popup_on_pid, PROCESS_CMD_NAME(pp) );
	}
	else
		select_node( pti_ptr->label_w, t_ptr, event );

	Process_action = old_value;
}


tnode
*setNiceValue( t_ptr, niceDelta )
tnode *t_ptr;
int niceDelta;
{
	Process_tree_info *pti_ptr;
	char		   cmd[256];
	int		   currentNiceValue, newNiceValue;

	DEBUG2( 1, "setNiceValue pid(%d) delta(%d)\n", t_ptr->key, niceDelta);

	if ( t_ptr == NULL )
	{
	    DEBUG0(0,"setNiceValue: Null t_ptr!!\n");
       	    return( NULL_TNODE_PTR );
	}

	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr == NULL )
	{
	    DEBUG1(0, "setNiceValue: process(%d) NULL pti_ptr\n", t_ptr->key );
       	    return( NULL_TNODE_PTR );
	}

	currentNiceValue = PROCESS_NICE( pti_ptr->pp );

#ifdef LINUX

	newNiceValue = currentNiceValue + niceDelta;

	sprintf(cmd, "%s %d %s %d >/dev/null 2>&1 &", 
			RENICE_CMD_PATH,	
			newNiceValue, 
			RENICE_CMD_ARGS,	
			t_ptr->key);
#else
	newNiceValue = niceDelta;

	sprintf(cmd, "%s %s %d %s %d >/dev/null 2>&1 &", 
			RENICE_CMD_PATH,	
			RENICE_CMD_ARG1,	
			newNiceValue, 
			RENICE_CMD_ARGS,	
			t_ptr->key);
#endif

	spawnTrustedExternal( cmd );

       	return( NULL_TNODE_PTR );
}

void 
do_renice( w, client_data, call_data)
Widget w;
caddr_t client_data; 
XmPushButtonCallbackStruct  *call_data;
{
	tnode	   	  *t_ptr;
	int		   delta;
	int		   allow_renice=0;


	DEBUG1( 1, "do_renice client data(%s)\n", client_data);

	t_ptr = walk_tree( Head, match_key, Popup_on_pid );
	if ( t_ptr == NULL )
	{
	    DEBUG1(0,"do_renice: Failed to find process(%d)\n",Popup_on_pid);
	    return;
	}

	allow_renice = authorizeRenice( t_ptr, client_data );

	if ( allow_renice )
	{
	    if ( client_data[0] == '-' )
		delta = 0 - (int) (client_data[1] - '0');
	    else
		delta = client_data[0] - '0';
		
	    setNiceValue( t_ptr, delta );
	}
}

char *
GIS( index )		/* Generate international string */
int index;
{
    return( 
"Cannot Increase Nice Value!! Treeps needs to be setuid root");
}

int 
authorizeRenice( t_ptr, client_data )
tnode   *t_ptr;
caddr_t client_data; 
{
	int 	 	  real_uid, effective_uid;
	int 		  allow_renice=0;
	sysProcInfo       *pp;
	Process_tree_info *pti_ptr;


	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr == NULL )
	{
	    DEBUG1(0, "do_renice_subtree: process(%d) NULL pti_ptr\n", 
							    Popup_on_pid );
	    return( 0 );
	}

	effective_uid = geteuid();
	real_uid      = getuid();

	pp = pti_ptr->pp;
        if ( pp == NULL )
              return( 0 );

	if ( real_uid == PROCESS_USER_ID(pp) )
        {
	     if ( client_data[0] == '-' )  
	     {
	         if ( effective_uid != 0 )
	     	    user_alert(TreeWidget, GIS(1) );
	         else if ( !TRUST_DECREASE_OWN_NICE_VALUE )
	     	  user_alert(TreeWidget,
		  "Cannot Increase Nice Value!! - Ask your System Administrator");
		 else
             	    allow_renice++;
	     }
	     else
             	allow_renice++;
        }
        else
        {
	    if ( !TRUST_CHANGE_ANY_PROCESS_NICE_VALUE )
	        user_alert(TreeWidget,
				"Cannot Change Nice Value!! - Not Your Process");
	    else if ( effective_uid != 0 )
	       user_alert(TreeWidget, GIS(1) );
	    else
                allow_renice++;
	}

	return( allow_renice );
}

void 
do_renice_subtree( w, client_data, call_data)
Widget w;
caddr_t client_data; 
XmPushButtonCallbackStruct  *call_data;
{
	tnode *t_ptr;
	int   delta;
	int   allow_renice;

	t_ptr = walk_tree( Head, match_key, Popup_on_pid );
	if ( t_ptr == NULL )
	{
	    DEBUG1(0,"do_renice_subtree: Failed to find process(%d)\n",
						Popup_on_pid);
	    return;
	}

	DEBUG1( 1, "do_renice_subtree client data(%s)\n", client_data);

	allow_renice = authorizeRenice( t_ptr, client_data );

	if ( allow_renice )
	{
	    if ( client_data[0] == '-' )
		delta = 0 - (int) (client_data[1] - '0');
	    else
		delta = client_data[0] - '0';

	    walk_tree( t_ptr, setNiceValue, delta );
	}
}

void 
do_tree_popup( w, client_data, call_data)
Widget w;
caddr_t client_data; 
XmPushButtonCallbackStruct  *call_data;
{
	Process_tree_info  *pti_ptr;

	if ( strcmp( client_data, "show_all") == 0 )
	{
		XpsiDoTreeLayout( TreeWidget, False );
		walk_tree( Head, display_node_widget, 0 );
		XpsiDoTreeLayout( TreeWidget, True );
	}
	else if ( strcmp( client_data, "outline") == 0 )
	{
		XpsiDoTreeLayout( TreeWidget, False );

		pti_ptr = (Process_tree_info *)Head->data;
		if ( pti_ptr->outline == True )
			walk_tree( Head, outline_all, False );
		else
			walk_tree( Head, outline_all, True );
		walk_tree( Head, redisplay_node, 0 );
		XpsiDoTreeLayout( TreeWidget, True );
	}
}

tnode
*outline_all( ptr, flag )
tnode *ptr;
int    flag;
{
	Process_tree_info  *pti_ptr;

	pti_ptr = (Process_tree_info *) ptr->data;

	if ( pti_ptr )
	{
		if ( flag )
			pti_ptr->outline = True;
		else
			pti_ptr->outline = False;
	}

	return( NULL );
}


int
spawnExternal( cmd )
char *cmd;
{
        pid_t child_pid;
        int status;
        char *argv[4];

        child_pid = fork();
        if ( child_pid == -1 )
                return 1;

        if ( child_pid == 0 )
        {
	    /* Alter ownerships, signal handling, ... */

            setuid( getuid() ); 

            argv[0] = "sh";
            argv[1] = "-c";
            argv[2] = cmd;
            argv[3] = 0;

            execve("/bin/sh", argv, Environment);

            exit(127);
        }

        while(1)
        {
               	if (waitpid(child_pid, &status, 0) == -1) 
	       	{
                   if (errno != EINTR)
                       return 1;
               	} 
		else
		{
                   break;
		}
         }
	return 0;
}

int
spawnTrustedExternal( cmd )
char *cmd;
{
	system( cmd );
}
