/* File: 	menu_bar.c
 *
 * Description: Routines to create and process menu's.
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1994, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	7/31/94	Created
 *
 */

#include <sys/types.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>

#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/Core.h>
#include <X11/CoreP.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/SeparatoG.h>
#include <Xm/RowColumn.h>
#include <Xm/MessageB.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/Text.h>
#include <Xm/Label.h>

#include <Xpsi/Tree.h>  /* our own custom Tree widget              */

#include "libXs.h"

#include "tree.h"       /* Generic m-ary tree package include file */
#include "list.h"	/* Generic list package                    */

#include "user_group.h"

#include "treeps.h"

#include "color_code.h"

#include "button_bar.h"

#include "debug.h"

Widget setup_menu();

void 		do_item();
void 		do_view();
void 		do_orient();
void 		do_conn();
void 		do_parent_placement();
void 		do_auto_size();
void 		do_hier();
void 		do_dec();
void 		do_user();
void 		do_proc();
void 		do_color();
void 		do_action();
void 		do_deco();
void 		do_sig();
void 		do_help();
void 		do_sendmail();
void 		do_mailaddr();
void 		do_timer_done();
void 		do_short_tic();
void 		do_long_tic();
void 		do_all_displayed_active();
void 		do_displayManPage();
void		do_find_dialog();
void 		do_start_search();
void 		do_get_process_name();
void 		do_get_process_pid();

#ifdef SVR4_MP
void do_lwp_display();
#endif

void		popup_pref_dialog();

void 		do_popdown();

extern Widget Top_level;


xs_menu_struct FileMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Halt",   "H", NULL_STR, do_item, "halt", NULL, 0, NULL_STR},
	{"Run",    "R", NULL_STR, do_item, "run", NULL, 0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"System Info", "s", NULL_STR, do_item, "sysInfoLaunchPad", NULL, 0,NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"Timers ...", "T", NULL_STR, do_item, "timers", NULL, 0,NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
/* {"Preferences ...", "P", NULL_STR, do_item, "prefs", NULL, 0,NULL_STR}, */
	{"Refresh Window", "W", NULL_STR, do_item, "refresh", NULL, 0,NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"Exit", "E", NULL_STR, do_item, "exit", NULL, 0, NULL_STR},
};

xs_menu_struct TreeOrientMenuData[]={
	{"Radio",  "0", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Left to Right",  "L", NULL_STR, do_orient, "left", NULL, 
							0, NULL_STR},
	{"Right to Left",  "R", NULL_STR, do_orient, "right",  NULL, 
							0, NULL_STR},
	{"Top to Bottom",  "T", NULL_STR, do_orient, "top", NULL, 
							0, NULL_STR},
	{"Bottom to Top",  "B", NULL_STR, do_orient, "bottom", NULL, 
							0, NULL_STR},
	{"Star Topology",  "S", NULL_STR, do_orient, "star", NULL, 
							0, NULL_STR},
};

xs_menu_struct TreePlacementMenuData[]={
	{"Radio",  "0", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Above First Child", "F", NULL_STR,
			do_parent_placement, "first", NULL, 0, NULL_STR},
	{"Centered Above Children","M",NULL_STR,
			do_parent_placement,"middle", NULL, 0, NULL_STR},
	{"Above Last Child",  "L", NULL_STR, 
			do_parent_placement, "last", NULL, 0, NULL_STR},
};

xs_menu_struct autoSizeMenuData[]={
	{"Radio",  "0", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Auto Size On", "n", NULL_STR,
			do_auto_size, "on", NULL, 0, NULL_STR},
	{"Auto Size Off","f",NULL_STR,
			do_auto_size,"off", NULL, 0, NULL_STR},
};


xs_menu_struct TreeConnectMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Diagonal",  "D", NULL_STR, do_conn, "diagonal", NULL, 
							0, NULL_STR},
	{"Tier",      "e", NULL_STR, do_conn, "tier",    NULL, 
							0, NULL_STR},
	{"Pathway",   "P", NULL_STR, do_conn, "pathway", NULL, 
							0, NULL_STR},
	{"None",      "N", NULL_STR, do_conn, "none", NULL, 0, NULL_STR},
};

xs_menu_struct HierarchyMenuData[]={
#ifdef SMART_PROCESS_ATTACHMENT
	{"Radio",              "2", NULL_STR, NULL, NULL_STR, NULL, 
#else
	{"Radio",              "1", NULL_STR, NULL, NULL_STR, NULL, 
#endif
							0, NULL_STR},
	{"Parent Child Only",  "C", NULL_STR, do_hier, "child", NULL, 
							0, NULL_STR},
	{"Child and Group Leader",  "G", NULL_STR, do_hier, "group", 
						NULL, 0, NULL_STR},
#ifdef SMART_PROCESS_ATTACHMENT

	{"Smart Attachment",  "S", NULL_STR, do_hier, "smart", 
						NULL, 0, NULL_STR},
#endif

};

xs_menu_struct DescendantsMenuData[]={
	{"Radio",              "1", NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},
	{"Shown if Displayable","S", NULL_STR, do_dec, "filter", NULL, 
							0, NULL_STR},
	{"Shown Always",        "A", NULL_STR, do_dec, "always", NULL, 
							0, NULL_STR},
};

#ifdef SVR4_MP

xs_menu_struct ShowLWPMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Display LWP's", "D", NULL_STR,
			do_lwp_display, "on", NULL, 0, NULL_STR},
	{"Hide LWP's","H",NULL_STR,
			do_lwp_display,"off", NULL, 0, NULL_STR},
};

#endif


xs_menu_struct ViewMenuData[]={
	{"Tree Orientation", "O", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)TreeOrientMenuData, 
			XtNumber(TreeOrientMenuData), "Orientation"},
	{"Tree Connection Style", "C", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)TreeConnectMenuData, 
			XtNumber(TreeConnectMenuData), "Connection"},
	{"Tree Parent Placement", "P", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)TreePlacementMenuData, 
			XtNumber(TreeConnectMenuData), "Parent Placement"},
	{"Process Hierarchy", "H", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)HierarchyMenuData, 
			XtNumber(HierarchyMenuData), "Hierarchy"},
	{"Descendants", "D", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)DescendantsMenuData, 
			XtNumber(DescendantsMenuData), "Descendants"},
	{"Main Window", "W", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)autoSizeMenuData, 
			XtNumber(DescendantsMenuData), "Auto Size"},
#ifdef SVR4_MP
	{"Light Weight Processes", "L", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)ShowLWPMenuData, 
			XtNumber(ShowLWPMenuData), "LWP Visibility"},
#endif
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"Display List ...", "L", NULL_STR, do_view, "dlist", NULL, 
							0, NULL_STR},
	{"Color Map ...", "M", NULL_STR, do_view, "map", NULL, 
							0, NULL_STR},
	{"Group/User Tree ...", "G", NULL_STR, do_view, "gutree", NULL, 
							0, NULL_STR},
};

xs_menu_struct UsersMenuData[]={
	{"Radio",  "0", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Login User", "L", NULL_STR, do_user, "login", NULL, 0, NULL_STR},
	{"All Users",  "A", NULL_STR, do_user, "all",    NULL, 0, NULL_STR},
	{"Root",       "R", NULL_STR, do_user, "root", NULL, 0, NULL_STR},
	{"User/Group Selection", "G", NULL_STR, do_user, "User/Group Selection", NULL, 0, NULL_STR},
};

xs_menu_struct ColorMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"TearOff",  NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},
	{"Real User Id",      "U", NULL_STR, do_color, "uid", NULL, 
							0, NULL_STR},
	{"Effective User Id", "E", NULL_STR, do_color, "euid", NULL, 
							0, NULL_STR},
	{"Real Group Id",     "G", NULL_STR, do_color, "gid", NULL, 
							0, NULL_STR},
	{"Effective Group Id","f", NULL_STR, do_color, "egid", NULL, 
							0, NULL_STR},
	{"Total CPU Time",    "T", NULL_STR, do_color, "tcpu", NULL, 
							0, NULL_STR},
	{"Current CPU Load",  "L", NULL_STR, do_color, "load", NULL, 
							0, NULL_STR},
	{"Process Status", "S", NULL_STR, do_color, "status", NULL, 
							0, NULL_STR},
	{"Resident Memory","R", NULL_STR, do_color, "resident", NULL, 
							0, NULL_STR},
	{"Image Size",     "I", NULL_STR, do_color, "image", NULL, 
							0, NULL_STR},
	{"Priority",       "P", NULL_STR, do_color, "priority", NULL, 
							0, NULL_STR},
};

xs_menu_struct ShowMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Daemons",        "D", NULL_STR, do_proc, "daemons", NULL, 
							0, NULL_STR},
	{"Foreground",     "F", NULL_STR, do_proc, "foreground", NULL, 
							0, NULL_STR},
};



/* ---------------- Platform Depenedent Signal Menus ---------------- */

#ifdef LINUX

xs_menu_struct SignalMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"TearOff",  NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},

	{"Hangup                   1",  "H", NULL, do_sig, "1", NULL, 0, NULL_STR},
	{"Interrupt                2",  "I", NULL, do_sig, "2", NULL, 0, NULL_STR},
	{"Quit                     3",  "Q", NULL, do_sig, "3", NULL, 0, NULL_STR},
	{"Illegal Instruction      4",  "l", NULL, do_sig, "4", NULL, 0, NULL_STR},
	{"Trace Trap               5",  "T", NULL, do_sig, "5", NULL, 0, NULL_STR},
	{"Abort                    6",  "A", NULL, do_sig, "6", NULL, 0, NULL_STR},
	{"IOT Trap                 6",  "O", NULL, do_sig, "6", NULL, 0, NULL_STR},
	{"BUS Error                7",  "B", NULL, do_sig, "7", NULL, 0, NULL_STR},
	{"Float Pt. Exception      8",  "F", NULL, do_sig, "8", NULL, 0, NULL_STR},
	{"Kill                     9",  "K", NULL, do_sig, "9", NULL, 0, NULL_STR},
	{"User Signal 1           10",  "1", NULL, do_sig, "10", NULL, 0, NULL_STR},
	{"Segmentation Violation  11",  "g", NULL, do_sig, "11", NULL, 0, NULL_STR},
	{"User Signal 2           12",  "2", NULL, do_sig, "12", NULL, 0, NULL_STR},
	{"Broken Pipe             13",  "k", NULL, do_sig, "13", NULL, 0, NULL_STR},
	{"Alarm Clock             14",  "m", NULL, do_sig, "14", NULL, 0, NULL_STR},
	{"Termination             15",  "r", NULL, do_sig, "15", NULL, 0, NULL_STR},
	{"Stack Fault             16",  "u", NULL, do_sig, "16", NULL, 0, NULL_STR},
	{"Child Status Changed    17",  "h", NULL, do_sig, "17", NULL, 0, NULL_STR},
	{"Continue                18",  "o", NULL, do_sig, "18", NULL, 0, NULL_STR},
	{"Stop                    19",  "S", NULL, do_sig, "19", NULL, 0, NULL_STR},
	{"Keyboard Stop           20",  "y", NULL, do_sig, "20", NULL, 0, NULL_STR},
	{"Background TTY Read     21",  "R", NULL, do_sig, "21", NULL, 0, NULL_STR},
	{"Background TTY Write    22",  "W", NULL, do_sig, "22", NULL, 0, NULL_STR},
	{"Urgent Socket Condition 23",  "t", NULL, do_sig, "23", NULL, 0, NULL_STR},
	{"CPU Limit Exceeded      24",  "C", NULL, do_sig, "24", NULL, 0, NULL_STR},
	{"File Size Limit         25",  "F", NULL, do_sig, "25", NULL, 0, NULL_STR},
	{"Virtual Alarm Clock     26",  "V", NULL, do_sig, "26", NULL, 0, NULL_STR},
	{"Profiling Alarm Clock   27",  "f", NULL, do_sig, "27", NULL, 0, NULL_STR},
	{"Window Size Changed     28",  "i", NULL, do_sig, "28", NULL, 0, NULL_STR},
	{"Pollable Event Occured  29",  "E", NULL, do_sig, "29", NULL, 0, NULL_STR},
	{"I/O now Possible        29",  "n", NULL, do_sig, "29", NULL, 0, NULL_STR},
	{"Power Failure           30",  "P", NULL, do_sig, "30", NULL, 0, NULL_STR},
};

#else

#ifdef SOLARIS

xs_menu_struct SignalMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"TearOff",  NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},

	{"Hangup                   1",  "H", NULL, do_sig, "1", NULL, 0, NULL_STR},
	{"Interrupt                2",  "I", NULL, do_sig, "2", NULL, 0, NULL_STR},
	{"Quit                     3",  "Q", NULL, do_sig, "3", NULL, 0, NULL_STR},
	{"Illegal Instruction      4",  "l", NULL, do_sig, "4", NULL, 0, NULL_STR},
	{"Trace Trap               5",  "T", NULL, do_sig, "5", NULL, 0, NULL_STR},
	{"Abort                    6",  "A", NULL, do_sig, "6", NULL, 0, NULL_STR},
	{"IOT Trap                 6",  "O", NULL, do_sig, "6", NULL, 0, NULL_STR},
	{"EMT Instruction          7",  "M", NULL, do_sig, "7", NULL, 0, NULL_STR},
	{"Float Pt. Exception      8",  "F", NULL, do_sig, "8", NULL, 0, NULL_STR},
	{"Kill                     9",  "K", NULL, do_sig, "9", NULL, 0, NULL_STR},
	{"Bus Error               10",  "B", NULL, do_sig, "10", NULL, 0, NULL_STR},
	{"Segmentation Violation  11",  "g", NULL, do_sig, "11", NULL, 0, NULL_STR},
	{"Bad System Call Arg     12",  "d", NULL, do_sig, "12", NULL, 0, NULL_STR},
	{"Broken Pipe             13",  "k", NULL, do_sig, "13", NULL, 0, NULL_STR},
	{"Alarm Clock             14",  "m", NULL, do_sig, "14", NULL, 0, NULL_STR},
	{"Termination             15",  "r", NULL, do_sig, "15", NULL, 0, NULL_STR},
	{"Sig User 1              16",  "1", NULL, do_sig, "16", NULL, 0, NULL_STR},
	{"Sig User 2              17",  "2", NULL, do_sig, "17", NULL, 0, NULL_STR},
	{"Child Status Change     18",  "h", NULL, do_sig, "18", NULL, 0, NULL_STR},
	{"Power Fail              19",  "P", NULL, do_sig, "19", NULL, 0, NULL_STR},
	{"Window Size Change      20",  "W", NULL, do_sig, "20", NULL, 0, NULL_STR},
	{"Urgent Socket Condition 21",  "t", NULL, do_sig, "21", NULL, 0, NULL_STR},
	{"Pollable Event Occured  22",  "E", NULL, do_sig, "22", NULL, 0, NULL_STR},
	{"I/O Now Possible        22",  "n", NULL, do_sig, "22", NULL, 0, NULL_STR},
	{"Stop                    23",  "S", NULL, do_sig, "23", NULL, 0, NULL_STR},
	{"User Stop from tty      24",  "y", NULL, do_sig, "24", NULL, 0, NULL_STR},
	{"Continue Stopped Proc   25",  "o", NULL, do_sig, "25", NULL, 0, NULL_STR},
	{"Background TTY Read     26",  "R", NULL, do_sig, "26", NULL, 0, NULL_STR},
	{"Background TTY Write    27",  "W", NULL, do_sig, "27", NULL, 0, NULL_STR},
	{"Virtual Timer Expired   28",  "V", NULL, do_sig, "28", NULL, 0, NULL_STR},
	{"Profiling Timer Expired 29",  "f", NULL, do_sig, "29", NULL, 0, NULL_STR},
	{"Exceed CPU Limit        30",  "C", NULL, do_sig, "30", NULL, 0, NULL_STR},
	{"Exceed File Size Limit  31",  "F", NULL, do_sig, "31", NULL, 0, NULL_STR},
/*
 *
 * The following are new in Solaris 2.6+, they may require thr_kill to be sent
 * to the desired thread. This functionality is planned for 1.1+
 *
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"Process's LWP's Blocked 32",  "L", NULL, do_sig, "32", NULL, 0, NULL_STR},
	{"LWP Thread Signal       33",  "a", NULL, do_sig, "33", NULL, 0, NULL_STR},
	{"CPR Freeze              34",  "z", NULL, do_sig, "34", NULL, 0, NULL_STR},
	{"CPR Thaw                35",  "w", NULL, do_sig, "35", NULL, 0, NULL_STR},
	{"Thread Cancel           36",  "n", NULL, do_sig, "36", NULL, 0, NULL_STR},
	{"Resource Lost           37",  "u", NULL, do_sig, "37", NULL, 0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"Real Time 1(highest)    38",  NULL, NULL, do_sig, "38", NULL, 0, NULL_STR},
	{"Real Time 2             39",  NULL, NULL, do_sig, "39", NULL, 0, NULL_STR},
	{"Real Time 3             40",  NULL, NULL, do_sig, "40", NULL, 0, NULL_STR},
	{"Real Time 4             41",  NULL, NULL, do_sig, "41", NULL, 0, NULL_STR},
	{"Real Time 5             42",  NULL, NULL, do_sig, "42", NULL, 0, NULL_STR},
	{"Real Time 6             43",  NULL, NULL, do_sig, "43", NULL, 0, NULL_STR},
	{"Real Time 7             44",  NULL, NULL, do_sig, "44", NULL, 0, NULL_STR},
	{"Real Time 8(lowest)     45",  NULL, NULL, do_sig, "45", NULL, 0, NULL_STR},
*/

};

#else  

#ifdef UW7

xs_menu_struct SignalMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"TearOff",  NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},

	{"Hangup                   1",  "H", NULL, do_sig, "1", NULL, 0, NULL_STR},
	{"Interrupt                2",  "I", NULL, do_sig, "2", NULL, 0, NULL_STR},
	{"Quit                     3",  "Q", NULL, do_sig, "3", NULL, 0, NULL_STR},
	{"Illegal Instruction      4",  "l", NULL, do_sig, "4", NULL, 0, NULL_STR},
	{"Trace Trap               5",  "T", NULL, do_sig, "5", NULL, 0, NULL_STR},
	{"Abort                    6",  "A", NULL, do_sig, "6", NULL, 0, NULL_STR},
	{"IOT Trap                 6",  "O", NULL, do_sig, "6", NULL, 0, NULL_STR},
	{"EMT Instruction          7",  "M", NULL, do_sig, "7", NULL, 0, NULL_STR},
	{"Float Pt. Exception      8",  "F", NULL, do_sig, "8", NULL, 0, NULL_STR},
	{"Kill                     9",  "K", NULL, do_sig, "9", NULL, 0, NULL_STR},
	{"Bus Error               10",  "B", NULL, do_sig, "10", NULL, 0, NULL_STR},
	{"Segmentation Violation  11",  "g", NULL, do_sig, "11", NULL, 0, NULL_STR},
	{"Bad System Call Arg     12",  "d", NULL, do_sig, "12", NULL, 0, NULL_STR},
	{"Broken Pipe             13",  "k", NULL, do_sig, "13", NULL, 0, NULL_STR},
	{"Alarm Clock             14",  "m", NULL, do_sig, "14", NULL, 0, NULL_STR},
	{"Termination             15",  "r", NULL, do_sig, "15", NULL, 0, NULL_STR},
	{"Sig User 1              16",  "1", NULL, do_sig, "16", NULL, 0, NULL_STR},
	{"Sig User 2              17",  "2", NULL, do_sig, "17", NULL, 0, NULL_STR},
	{"Child Status Change     18",  "h", NULL, do_sig, "18", NULL, 0, NULL_STR},
	{"Power Fail              19",  "P", NULL, do_sig, "19", NULL, 0, NULL_STR},
	{"Window Size Change      20",  "W", NULL, do_sig, "20", NULL, 0, NULL_STR},
	{"Urgent Socket Condition 21",  "t", NULL, do_sig, "21", NULL, 0, NULL_STR},
	{"Pollable Event Occured  22",  "E", NULL, do_sig, "22", NULL, 0, NULL_STR},
	{"I/O Now Possible        22",  "n", NULL, do_sig, "22", NULL, 0, NULL_STR},
	{"Stop                    23",  "S", NULL, do_sig, "23", NULL, 0, NULL_STR},
	{"User Stop from tty      24",  "y", NULL, do_sig, "24", NULL, 0, NULL_STR},
	{"Continue Stopped Proc   25",  "o", NULL, do_sig, "25", NULL, 0, NULL_STR},
	{"Background TTY Read     26",  "R", NULL, do_sig, "26", NULL, 0, NULL_STR},
	{"Background TTY Write    27",  "W", NULL, do_sig, "27", NULL, 0, NULL_STR},
	{"Virtual Timer Expired   28",  "V", NULL, do_sig, "28", NULL, 0, NULL_STR},
	{"Profiling Timer Expired 29",  "f", NULL, do_sig, "29", NULL, 0, NULL_STR},
	{"Exceed CPU Limit        30",  "C", NULL, do_sig, "30", NULL, 0, NULL_STR},
	{"Exceed File Size Limit  31",  "F", NULL, do_sig, "31", NULL, 0, NULL_STR},
};

#else  /* System V.4, pre UnixWare 7 */

xs_menu_struct SignalMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"TearOff",  NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},
	{"Hangup          1",  "H", NULL, do_sig, "1", NULL, 0, NULL_STR},
	{"Interrupt       2",  "I", NULL, do_sig, "2", NULL, 0, NULL_STR},
	{"Quit            3",  "Q", NULL, do_sig, "3", NULL, 0, NULL_STR},
	{"Kill            9",  "K", NULL, do_sig, "9", NULL, 0, NULL_STR},
	{"Alarm Clock    14",  "A", NULL, do_sig, "14", NULL, 0, NULL_STR},
	{"Terminated     15",  "T", NULL, do_sig, "15", NULL, 0, NULL_STR},
	{"User Signal 1  16",  "1", NULL, do_sig, "16", NULL, 0, NULL_STR},
	{"User Signal 2  17",  "2", NULL, do_sig, "17", NULL, 0, NULL_STR},
	{"Child Status   18",  "C", NULL, do_sig, "18", NULL, 0, NULL_STR},
	{"Power Fail     19",  "P", NULL, do_sig, "19", NULL, 0, NULL_STR},
	{"Window Size    20",  "W", NULL, do_sig, "20", NULL, 0, NULL_STR},
};

#endif /* End of non Unixware 7 */

#endif /* End of non linux part */

#endif  /* End of platform dependent signal defs */

xs_menu_struct ActionMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Select",         	   "t",NULL_STR,do_action,"select",NULL,
							0,NULL_STR},
	{"Display Process Details","D",NULL_STR,do_action,"details",NULL,
							0,NULL_STR},
	{"Signal Process",         "S",NULL_STR,do_action,"signal",NULL,
							0,NULL_STR},
	{"Kill Process",           "K",NULL_STR,do_action,"kill",NULL,
							0,NULL_STR},
	{"Outline Subtree",        "O",NULL_STR,do_action,"outline",NULL,
							0,NULL_STR},
	{"Hide Subtree",           "H",NULL_STR,do_action,"hide",NULL,
							0,NULL_STR},
	{"ShoW Subtree",           "W",NULL_STR,do_action,"show",NULL,
							0,NULL_STR},
};

xs_menu_struct DecorationMenuData[]={
	{"Radio",  "1", NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"No Decorations",   "N", NULL_STR, do_deco,"none",NULL, 0,NULL_STR},
	{"Distinguished", "D", NULL_STR, do_deco,"distinguished",NULL, 
								0,NULL_STR},
};

xs_menu_struct ProcMenuData[]={
	{"Show", "S", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)ShowMenuData, 
			XtNumber(ShowMenuData), "Show"},
	{"Color Based On", "C", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)ColorMenuData, 
			XtNumber(ColorMenuData), "Color Based On"},
	{"Selection Action", "A", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)ActionMenuData, 
			XtNumber(ActionMenuData), "Selection Action"},
	{"Signal",  "S", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)SignalMenuData, 
			XtNumber(SignalMenuData), "Signal"},
	{"Decorations",  "D", NULL_STR, NULL, NULL_STR, 
		(struct _menu_struct *)DecorationMenuData, 
			XtNumber(DecorationMenuData), "Decorate"},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							   0, NULL_STR},
	{"Find",   "F", NULL_STR, do_find_dialog,"find",NULL, 0,NULL_STR},
};

xs_menu_struct RelatedDocsMenuData[]={
	{"TearOff",  NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 
							0, NULL_STR},
#ifdef SVR4_MP
	{"priocntl      Man Page",  "", NULL, do_displayManPage, "priocntl", NULL, 0, NULL_STR},
#else
	{"procinfo      Man Page",  "", NULL, do_displayManPage, "procinfo", NULL, 0, NULL_STR},
#endif
	{"ps            Man Page",  "", NULL, do_displayManPage, "ps", NULL, 0, NULL_STR},
	{"netstat       Man Page",  "", NULL, do_displayManPage, "netstat", NULL, 0, NULL_STR},
	{"renice        Man Page",  "", NULL, do_displayManPage, "renice", NULL, 0, NULL_STR},
#ifdef LINUX
	{"strace        Man Page",  "", NULL, do_displayManPage, "strace", NULL, 0, NULL_STR},
#else
	{"trace         Man Page",  "", NULL, do_displayManPage, "trace", NULL, 0, NULL_STR},
	{"truss         Man Page",  "", NULL, do_displayManPage, "truss", NULL, 0, NULL_STR},
#endif
	{"top           Man Page",  "", NULL, do_displayManPage, "top", NULL, 0, NULL_STR},
#ifdef SVR4_MP
	{"proc          Man Page",  "", NULL, do_displayManPage, "proc", NULL, 0, NULL_STR},
#else
	{"proc          Man Page",  "", NULL, do_displayManPage, "5 proc", NULL, 0, NULL_STR},
#endif
	{"Signal        Man Page",  "", NULL, do_displayManPage, "7 signal", NULL, 0, NULL_STR},
};

xs_menu_struct HelpMenuData[]={
	{"About",           "A", NULL_STR,   do_help, "about", NULL, 
							0, NULL_STR},
	{"Menu Bar",        "M", NULL_STR,   do_help, "menu_bar", NULL, 
							0, NULL_STR},
	{"Popup Menus",     "P", NULL_STR,   do_help, "popups", NULL, 
							0, NULL_STR},
	{"Keys",            "K", NULL_STR,   do_help, "keys", NULL, 
							0, NULL_STR},
	{"Buttons",         "B", NULL_STR,   do_help, "buttons", NULL, 
							0, NULL_STR},
	{"Decorations",     "D", NULL_STR,   do_help, "decorations", NULL, 
							0, NULL_STR},
	{"Copyrights",     "C", NULL_STR,   do_help, "copyrights", NULL, 
							0, NULL_STR},
	{"Sep",         NULL_STR, NULL_STR, NULL, NULL_STR, NULL, 0, NULL_STR},
	{"Related Docs", "R", NULL_STR, NULL, 
			NULL_STR, (struct _menu_struct *)RelatedDocsMenuData, 
			XtNumber(RelatedDocsMenuData), "Related Docs"},
};

xs_menu_struct PulldownData[]={
	{"Program", "P", NULL_STR, NULL, 
			NULL_STR, (struct _menu_struct *)FileMenuData, 
			XtNumber(FileMenuData), "Program"},
	{"Views",   "V", NULL_STR, NULL, 
			NULL_STR, (struct _menu_struct *)ViewMenuData, 
			XtNumber(ViewMenuData), "Views"},
	{"Users",   "U", NULL_STR, NULL, 
			NULL_STR, (struct _menu_struct *)UsersMenuData,
			XtNumber(UsersMenuData), "Users"},
	{"Processes","r", NULL_STR, NULL, 
			NULL_STR, (struct _menu_struct *)ProcMenuData,
			XtNumber(ProcMenuData), "Processes"},
	{"Help", "H", NULL_STR, NULL, 
			NULL_STR, (struct _menu_struct *)HelpMenuData, 
			XtNumber(HelpMenuData), "Help"},
};


Widget Menu_widget;	/* Set during setup menu */

extern Widget create_label();
extern void process_tip_enter();

extern Debug;

extern Widget TreeWidget;
extern Widget Main_menu;
extern lnode *Process_list;

extern Button_item  *Main_Buttons;
extern int	     Main_Button_cnt;

char *get_color_menu_str2();

Widget	Timer_st_value;		/* Short tic text widget */
Widget	Timer_lt_value;		/* Long tic text widget  */

Widget  Find_pn_value;		/* Finds process name text widget */
Widget  Find_pid_value;		/* Finds process pid text widget */
Widget  Find_msg;		/* Find message area */

int 	In_Find_dialog=0;

Widget  Find_pw;

void start_find();

char 			*Find_process_name=NULL;
int   			 Find_process_id=-1;
Process_tree_info 	*Find_pti_ptr=0;

void leave_find_dialog();		/* take down process tip */
void enter_find_dialog();		/* can do process tip */

extern int MaxSig;

extern User_group_info  *Users;

extern int 	Decorated_tree;
extern void 	change_decor();
extern Boolean 	AutoSizeMainWindow;
extern int	Process_attach_mode;

extern int refresh_main_window();


Widget
setup_menu( parent )
Widget parent;
{
	Widget 	menuMW;
	Widget 	sep;
	Arg 	wargs[10];
	int 	n;
	uid_t   euid;

	DEBUG0(1, "setup_menu\n" );

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	menuMW = XmCreateMenuBar( parent, "menubar", wargs, n );
	XtManageChild( menuMW );

	Menu_widget = menuMW;

	xs_create_menu_buttons( NULL, menuMW, PulldownData,
					XtNumber(PulldownData));
	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, menuMW); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep = XtCreateManagedWidget( "separator1", xmSeparatorGadgetClass, 
						parent, wargs, n );

	return( sep );
}

set_menu_toggle( w, item_label, notify )
Widget w;
char *item_label;
Boolean notify;
{
	Widget  	item;
	char    	buf[80];
	String  	name, item_name;
	int		invoked_callback=False;

	if ( w == NULL )
		return( False );

	/* Find the menu_label(must be unique!!) then call 
	 * XmToggleButtonSetState() to set the new selection and invoke the 
	 * callback to do the actual work. The routine returns true, if it 
	 * found the menu item, set it and did the callback, otherwise it 
	 * returns False.
	 */

	buf[0] = '*'; buf[1] = NULL_CHAR;
	strcat( buf, item_label );

	name = &(buf[0]);

	DEBUG1( 1, "Searching for menu label(%s)\n", name );

	item = XtNameToWidget( w, name );
	if ( item )
	{
		item_name = XtName( item );
		DEBUG2( 1,"Match (%s), on item_name(%s)\n", name, item_name);
	}
	else
	{
		DEBUG1(1, "set_menu_toggle: Failed to find item(%s)\n", 
								item_label );
		return( False );
	}

	DEBUG1(1, "Setting Current Toggle to (%s)\n", item_name );

	XmToggleButtonSetState( item, True, notify );
	invoked_callback = True;

	return( invoked_callback );
}

unset_menu_toggle( w, item_label)
Widget w;
char *item_label;
{
	Widget  	item;
	char    	buf[80];
	String  	name, item_name;

	if ( w == NULL )
		return( False );

	/* Find the menu_label(must be unique!!) then call 
	 * XmToggleButtonSetState() to set the new selection 
	 */

	buf[0] = '*'; buf[1] = NULL_CHAR;
	strcat( buf, item_label );

	name = &(buf[0]);

	DEBUG1( 1, "Searching for menu label(%s)\n", name );

	item = XtNameToWidget( w, name );
	if ( item )
	{
		item_name = XtName( item );
		DEBUG2( 1,"Match (%s), on item_name(%s)\n", name, item_name);
	}
	else
	{
		DEBUG1(1, "set_menu_toggle: Failed to find item(%s)\n", 
								item_label );
		return( False );
	}

	DEBUG1(1, "Setting Current Toggle to (%s)\n", item_name );

	XmToggleButtonSetState( item, False, False );

	return( True );
}

set_start_up_menu_toggles()
{
	char *str;
	int   save_color_based_on;

	if ( Show_all )
	{
		set_menu_toggle( Main_menu, "All Users", True );
	}

	if ( Show_daemons == True )
	{
		set_menu_toggle( Main_menu, "Daemons", True );
	}

	if ( Process_attach_mode == GROUP_ATTACH )
		set_menu_toggle( Main_menu, "Child and Group Leader", True );
	else if ( Process_attach_mode == SMART_ATTACH )
		set_menu_toggle( Main_menu, "Smart Attachment", True );
	else
		set_menu_toggle( Main_menu, "Parent Child Only", True );
	
	if ( AutoSizeMainWindow != True )
		set_menu_toggle( Main_menu, "Auto Size Off", True );

	if ( Color_based_on != COLOR_BY_EFFECTIVE_USER_ID )
	{
		str = get_color_menu_str2( Color_based_on );

		save_color_based_on = Color_based_on;

		/* Color_based_on = 0;  Force change */

		set_menu_toggle( Main_menu, str, True );

		Color_based_on = save_color_based_on; /* In case above fails */
	}

	if ( Current_placement == PARENT_CENTERED )
		set_menu_toggle( Main_menu, "Centered Above Children", True );
	else if ( Current_placement == PARENT_ABOVE_FIRST_CHILD )
		set_menu_toggle( Main_menu, "Above First Child", True );
	else if ( Current_placement == PARENT_ABOVE_LAST_CHILD )
		set_menu_toggle( Main_menu, "Above Last Child", True );
}


void 
do_item( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int     	new_value=0;
	static  Widget  last_value_w=NULL;
	static  int     first_timer=1;
	static  int     first_time_refresh=1;
	static  int     ignore_next=0;
	static  int     first_time_prefs=1;
	int		n;
	Arg		wargs[4];

	DEBUG1( 1, "do_item client data(%s)\n", client_data);

	if ( ignore_next )
	{
		ignore_next = 0;
		return;
	}

	if ( strcmp(client_data, "exit") == 0 )
	{
		treeps_exit(0);
	}
	else if ( strcmp(client_data, "halt") == 0 )
	{
		new_value = 0;
	}
	else if ( strcmp(client_data, "run") == 0 )
	{
		new_value = 1;
	}
	else if ( strcmp(client_data, "timers") == 0 )
	{
		if ( first_timer )	/* Remove toggle indicator */
		{
	    		n = 0;
 	    		XtSetArg( wargs[n], XmNindicatorOn, False); n++;
	    		XtSetValues( w, wargs, n );

			first_timer = 0;
		}

		popup_timer_controls( w );

		/* Set button back to last selection */

		ignore_next = 1;

		if ( last_value_w != NULL )
			XmToggleButtonSetState( last_value_w, True, True );

		return;
	}
	else if ( strcmp(client_data, "sysInfoLaunchPad") == 0 )
	{
		if ( first_timer )	/* Remove toggle indicator */
		{
	    		n = 0;
 	    		XtSetArg( wargs[n], XmNindicatorOn, False); n++;
	    		XtSetValues( w, wargs, n );

			first_timer = 0;
		}

		spawnExternal( "sysInfoLaunchPad &" ); 

		/* Set button back to last selection */

		ignore_next = 1;

		if ( last_value_w != NULL )
			XmToggleButtonSetState( last_value_w, True, True );

		return;
	}
	else if ( strcmp(client_data, "prefs") == 0 )
	{
		if ( first_time_prefs )	/* Remove toggle indicator */
		{
	    		n = 0;
 	    		XtSetArg( wargs[n], XmNindicatorOn, False); n++;
	    		XtSetValues( w, wargs, n );

			first_time_prefs = 0;
		}

		popup_pref_dialog( w );

		/* Set button back to last selection */

		ignore_next = 1;

		if ( last_value_w != NULL )
			XmToggleButtonSetState( last_value_w, True, True );

		return;
	}
	else if ( strcmp(client_data, "refresh") == 0 )
	{
		if ( first_time_refresh )	/* Remove toggle indicator */
		{
	    		n = 0;
 	    		XtSetArg( wargs[n], XmNindicatorOn, False); n++;
	    		XtSetValues( w, wargs, n );

			first_time_refresh = 0;
		}

		refresh_main_window();

		/* setNewView(); experimental, sets new view in scroll window*/

		/* Set button back to last selection */

		ignore_next = 1;

		if ( last_value_w != NULL )
			XmToggleButtonSetState( last_value_w, True, True );

		return;
	}


	if ( new_value != Monitor_processes )
	{
		set_button_from_menu_label( w );
		Monitor_processes = new_value;
	}

	last_value_w = w;
}

set_button_from_menu_label( w )
Widget w;
{
	Boolean rc;
	char    *label_str;
	int      n;
	Arg      wargs[3];
	XmString label_xmstr;

	n = 0;
	XtSetArg( wargs[n], XmNlabelString, &label_xmstr ); n++;
	XtGetValues( w, wargs, n );

	rc = XmStringGetLtoR( label_xmstr, XmSTRING_DEFAULT_CHARSET,&label_str);

	DEBUG2(3, "XmstringGet returned(%d) str(%s)\n", rc, label_str);

	if ( rc )
		set_button( Main_Buttons, Main_Button_cnt, label_str );

	if ( label_str )
		free( label_str );
}


void 
do_view( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int new_value=0;

	DEBUG1( 1, "do_view client data(%s)\n", client_data);

	if ( strcmp(client_data, "map") == 0 )
	{
		arm_button( Main_Buttons, Main_Button_cnt,"Color Map");
		set_cmap_callback();
		show_colors();
	}
	else if ( strcmp(client_data, "gutree") == 0 )
	{
		arm_button( Main_Buttons, Main_Button_cnt, "gutree" );
		set_gutree_callback();
		popup_user_tree( Users );
	}
	else if ( strcmp(client_data, "dlist") == 0 )
	{
		arm_button( Main_Buttons, Main_Button_cnt, "Fields" );
		set_fields_callback();
		show_display_list();
	}
}


void 
do_orient( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	TreeOrientation orient;
	static int firstStarLayout=0;

	DEBUG1( 1, "do_view client data(%s)\n", client_data);

	orient = LEFT_TO_RIGHT;
	if ( strcmp( client_data, "star" ) == 0 )
	{
	    if ( ! firstStarLayout )
	    {
	        firstStarLayout++;
		show_help( w, "star_layout" );
	    }

	    orient = STAR_TOPOLOGY;
	}
	else if ( strcmp( client_data, "left" ) == 0 )
	{
		orient = LEFT_TO_RIGHT;
	}
	else if ( strcmp( client_data, "right" ) == 0 )
	{
		orient = RIGHT_TO_LEFT;
	}
	else if ( strcmp( client_data, "top" ) == 0 )
	{
		orient = TOP_TO_BOTTOM;
	}
	else if ( strcmp( client_data, "bottom" ) == 0 )
	{
		orient = BOTTOM_TO_TOP;
	}

	if ( orient != Current_orientation )
	{
		Current_orientation = orient;
		set_new_orientation( orient );
	}
}

void 
do_conn( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	ConnectionStyle conn;

	DEBUG1( 1, "do_view client data(%s)\n", client_data);

	conn = DIAGONAL_CONNECTION;
	if ( strcmp( client_data, "diagonal" ) == 0 )
	{
		conn = DIAGONAL_CONNECTION;
	}
	else if ( strcmp( client_data, "tier" ) == 0 )
	{
		conn = TIER_CONNECTION;
	}
	else if ( strcmp( client_data, "pathway" ) == 0 )
	{
		conn = PATHWAY_CONNECTION;
	}
	else if ( strcmp( client_data, "none" ) == 0 )
	{
		conn = NO_CONNECTION;
	}

	if ( conn != Current_connection )
	{
		Current_connection = conn;
		set_new_conn_style( conn );
	}
}

#ifdef SVR4_MP

void 
do_lwp_display( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	Boolean lwp;

	DEBUG1( 1, "do_lwp_display client data(%s)\n", client_data);

	lwp = True;

	if ( strcmp( client_data, "on" ) == 0 )
	{
		lwp = True;
	}
	else if ( strcmp( client_data, "off" ) == 0 )
	{
		lwp = False;
	}

	change_LWP_visibility( lwp );
}

#endif




void 
do_auto_size( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	Boolean as;

	DEBUG1( 1, "do_auto_size client data(%s)\n", client_data);

	as = True;

	if ( strcmp( client_data, "on" ) == 0 )
	{
		as = True;
	}
	else if ( strcmp( client_data, "off" ) == 0 )
	{
		as = False;
	}

	AutoSizeMainWindow = as;

}


void 
do_parent_placement( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	ParentPlacement pp;

	DEBUG1( 1, "do_parent_placement client data(%s)\n", client_data);

	pp = PARENT_CENTERED;

	if ( strcmp( client_data, "first" ) == 0 )
	{
		pp = PARENT_ABOVE_FIRST_CHILD;
	}
	else if ( strcmp( client_data, "middle" ) == 0 )
	{
		pp = PARENT_CENTERED;
	}
	else if ( strcmp( client_data, "last" ) == 0 )
	{
		pp = PARENT_ABOVE_LAST_CHILD;
	}

	if ( pp != Current_placement )
	{
		set_button_from_menu_label( w );

		Current_placement = pp;
		set_new_placement( pp );
	}

}


void 
do_hier( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{

	DEBUG1( 1, "do_hier client data(%s)\n", client_data);

	if ( strcmp( client_data, "child" ) == 0 )
	{
		set_process_attach_mode( CHILD_ATTACH );
	}
	else if ( strcmp( client_data, "group" ) == 0 )
	{
		set_process_attach_mode( GROUP_ATTACH );
	}
	else if ( strcmp( client_data, "smart" ) == 0 )
	{
		set_process_attach_mode( SMART_ATTACH );
	}
	New_display_criteria++;
}

void 
do_dec( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int	new_value=0;

	DEBUG1( 1, "do_dec client data(%s)\n", client_data);

	if ( strcmp( client_data, "filter" ) == 0 )
	{
		new_value = FILTER_DESCENDANTS;
	}
	else if ( strcmp( client_data, "always" ) == 0 )
	{
		new_value = SHOW_ALL_DESCENDANTS;
	}

	if ( new_value != Show_descendants )
	{
		Show_descendants = new_value;
		New_display_criteria++;
	}
}


void 
do_user( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int 	   new_value;

	DEBUG1( 1, "do_user client data(%s)\n", client_data);

        if ( strcmp( client_data, "login" ) == 0 )
        {
                new_value = VIEW_LOGGED_IN_USER;
        }
        else if ( strcmp( client_data, "all" ) == 0 )
        {
                new_value = VIEW_ALL_USERS;
        }
        else if ( strcmp( client_data, "root" ) == 0 )
        {
                new_value = VIEW_ROOT_USER;
        }
        else if ( strcmp( client_data, "uucp" ) == 0 )
        {
                new_value = VIEW_UUCP_USER;
        }
        else if ( strcmp( client_data, "User/Group Selection" ) == 0 )
	{
                new_value = VIEW_GUTREE;
	}
	else
	{
		return;
	}

	if ( new_value != User_view )
	{
		set_button_from_menu_label( w );
		set_user_list( client_data );
	}
}

void 
do_find_dialog( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	popup_find_dialog( Top_level );
}

void
start_find( w, find_pw, call_value)
Widget w;
Widget find_pw;
XmAnyCallbackStruct *call_value;
{
	int noop=1;
	Widget tree_node_w=NULL;
	tnode *ptr;
	Process_tree_info    *pti_ptr;
	Time time;
	XButtonEvent *b;
	int len;
	lnode *p;
	search_record match_record;
	char pid_str[8];
	XCrossingEvent cross_event;
	Boolean continueDispatch=True;
	
	xs_wprintf( Find_msg, "%s", "                     ");

	DEBUG2( 1, "start_find: (%s) (%d)\n", Find_process_name, Find_process_id );

	if ( Find_process_id == -1 )  /* Try name only search */
	{
		if ( Find_process_name ==  NULL )
			return;
			
		len = strlen( Find_process_name );
		if ( len == 0 )
		{
	     	        xs_wprintf( Find_msg, "%s", "Process Name Field Empty!!");
			XBell( XtDisplay(w), 100 );
			return;
		}

		/* XmTextSetString( Find_pid_value, buf ); */

		ptr = walk_tree( Head, match_name, Find_process_name ); /* is it visible */
        	if ( ptr == NULL )
		{
		 XmTextSetString( Find_pid_value, "" );
	     	 user_alert( TreeWidget, "Could not find process. Perhaps it's not visible!!");
		 return;

		 /* would be nice to scan process list and check if it's there but those
		  * processes do not always have data loaded.
		  */ 
		}

		pti_ptr = (Process_tree_info *) ptr->data;
		if ( pti_ptr == NULL )
		{
	     	        xs_wprintf( Find_msg, "%s", "No data found for process!!");
			XBell( XtDisplay(w), 100 );
		}

		sprintf( pid_str, "%d", ptr->key );
		XmTextSetString( Find_pid_value, pid_str );
		Find_process_id = ptr->key;    /* In case of find next */

		tree_node_w = pti_ptr->node_w;

	}
	else if(Find_process_name==NULL || strlen(Find_process_name) == 0)/* try pid search */
	{
		ptr = walk_tree( Head, match_key, Find_process_id ); /* is it visible */
        	if ( ptr == NULL )
        	{
		    p = find_lnode( Process_list, Find_process_id ); /* does it exist */
		    if ( p == NULL )
		    {
	     	        xs_wprintf( Find_msg, "%s", "No Match on PID!!");
			XBell( XtDisplay(w), 100 );
		        return;
		    }

	     	        xs_wprintf( Find_msg, "%s", "Process with PID not Visible!!");
			XBell( XtDisplay(w), 100 );

		    /* make visible? can't */
		}
		else
		{
		    pti_ptr = (Process_tree_info *) ptr->data;
		    if ( pti_ptr == NULL )
		    {
	     	        xs_wprintf( Find_msg, "%s", "No data found for process!!");
			XBell( XtDisplay(w), 100 );
		    }

		    /* could check if name matches that in Find_process_name and warn
		     * if not. Why bother?
		     */

		    XmTextSetString( Find_pn_value, ptr->str_key );  /* so find next works */
		    Find_process_name = XmTextGetString( Find_pn_value );

		    tree_node_w = pti_ptr->node_w;
		}
	}
	else  /* Find next */
	{
		match_record.key       = Find_process_id;
		match_record.name      = Find_process_name;
		match_record.found_key = 0;

		ptr = walk_tree( Head, next_name_after_key, &match_record ); 
        	if ( ptr == NULL )
        	{
	     	        xs_wprintf( Find_msg, "%s", "No more matches on NAME!!");
			XBell( XtDisplay(w), 100 );
			Find_process_id = -1;
		 	XmTextSetString( Find_pid_value, "" );
		        return;
		}
		pti_ptr = (Process_tree_info *) ptr->data;
		if ( pti_ptr == NULL )
		{
	     	        xs_wprintf( Find_msg, "%s", "No data found for process!!");
			XBell( XtDisplay(w), 100 );
		}

		sprintf( pid_str, "%d", ptr->key );
		XmTextSetString( Find_pid_value, pid_str );
		Find_process_id = ptr->key;    /* In case of find next */

		tree_node_w = pti_ptr->label_w;
	}

	if ( tree_node_w )
	{
		b = (XButtonEvent *) (call_value->event);
		time = b->time;
		manuallySetSelection( tree_node_w, ptr, time );

		/* display process_tip */

		if ( In_Find_dialog )
		{
		  cross_event.detail  = 99999;
		  process_tip_enter( tree_node_w, pti_ptr, &cross_event, &continueDispatch);
		  Find_pti_ptr = pti_ptr;
		}
	}
}



popup_find_dialog( parent )
Widget parent;
{
	static 		Widget find_pw=NULL;
	Widget 		pn_label, pn_value;
	Widget 		pid_label, pid_value;
	Widget 		sep1, sep2,search_w, exit_w, help_w;
	XmString	title_str;
	char		buf[12];
	int    		n;
	Arg    		wargs[14];


	if ( find_pw == NULL )
	{
	    title_str = XmStringCreate("Find Process",
						XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg( wargs[n], XmNdialogTitle,  title_str ); n++;
 	    XtSetArg( wargs[n], XmNautoUnmanage, FALSE ); n++;
            XtSetArg( wargs[n], XmNfractionBase, 10 ); n++;
	    find_pw = XmCreateFormDialog( parent, "findProcess",
								    wargs, n);

	    Find_pw = find_pw;

	    if ( find_pw == NULL )
	    {
		DEBUG1(1, "popup_find_dialog: Failed parent(%d)\n",parent);
		return;
	    }

	    /* Create done, cancel, help buttons at bottom */

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 1); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 3); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
            search_w = XtCreateManagedWidget("Search", xmPushButtonWidgetClass,
                                                find_pw, wargs, n );

	    XtAddCallback( search_w, XmNactivateCallback, do_start_search, find_pw);

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 4); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 6); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
            exit_w = XtCreateManagedWidget("exit", xmPushButtonWidgetClass,
                                                find_pw, wargs, n );

	    XtAddCallback( exit_w, XmNactivateCallback, do_popdown, find_pw);

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 7); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 9); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
            help_w = XtCreateManagedWidget("Help", xmPushButtonWidgetClass,
                                                find_pw, wargs, n );

	    XtAddCallback( help_w, XmNactivateCallback, (XtCallbackProc) do_help, "find");

	    /* Create message widget */

 	    n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNbottomWidget, help_w); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2 ); n++;
            sep2 = XtCreateManagedWidget( "find_separator1", 
						xmSeparatorGadgetClass,
                                                find_pw, wargs, n );

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNleftOffset, 2); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNbottomWidget, sep2); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2 ); n++;
            Find_msg = XtCreateManagedWidget( "Press Enter to Search", 
						xmLabelWidgetClass,
                                                find_pw, wargs, n);
	    /* Create Process Name widgets */

            n = 0;
            XtSetArg( wargs[n], XmNcolumns, 15 ); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNtopOffset, 4); n++;
            pn_value = XtCreateManagedWidget( "process name", 
						xmTextWidgetClass,
                                                find_pw, wargs, n);
	    Find_pn_value = pn_value;

	    XtAddCallback( pn_value, XmNactivateCallback, do_get_process_name, 0 );

            n = 0;
            XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_END); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNleftOffset, 2); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, pn_value);n++;
            XtSetArg(wargs[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg(wargs[n],XmNbottomWidget,pn_value);n++;
            XtSetArg( wargs[n], XmNtraversalOn, False);n++;
            pn_label = XtCreateManagedWidget( "Process Name:", 
						xmLabelWidgetClass,
                                                find_pw, wargs, n);

	    /* Create pid field */

            n = 0;
            XtSetArg( wargs[n], XmNcolumns, 5 ); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNtopWidget, pn_value); n++;
            XtSetArg( wargs[n], XmNtopOffset, 2); n++;
            pid_value = XtCreateManagedWidget( "Process ID", 
						xmTextWidgetClass,
                                                find_pw, wargs, n);
	    Find_pid_value = pid_value;

	    XtAddCallback( pid_value, XmNactivateCallback, do_get_process_pid, 0 );

            n = 0;
            XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_END); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNleftOffset, 2); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, pid_value);n++;
            XtSetArg(wargs[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg(wargs[n],XmNbottomWidget,pid_value);n++;
            XtSetArg( wargs[n], XmNtraversalOn, False);n++;
            pid_label = XtCreateManagedWidget( "Process ID:", 
						xmLabelWidgetClass,
                                                find_pw, wargs, n);


	    /* Create bottom separator */

 	    n = 0;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNtopWidget, pid_value ); n++;
            XtSetArg( wargs[n], XmNtopOffset, 2 ); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNbottomWidget, Find_msg); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2 ); n++;
            sep1 = XtCreateManagedWidget( "find_separator1", 
						xmSeparatorGadgetClass,
                                                find_pw, wargs, n );


	}

	/*
	sprintf( buf, "%5d", Some_default_process_name );
	XmTextSetString( Find_pn_value, buf );

	sprintf( buf, "%5d", Some_default_pid_value );
	XmTextSetString( Find_pid_value, buf );
	*/

	XtManageChild( find_pw );

	XtAddEventHandler(find_pw, EnterWindowMask, FALSE, enter_find_dialog, 0 );
	XtAddEventHandler(find_pw, LeaveWindowMask, FALSE, leave_find_dialog, 0 );

	XmProcessTraversal( Find_pn_value, XmTRAVERSE_CURRENT );
}


void
do_get_process_name( w, noop, call_data)
Widget w;
int noop;
caddr_t call_data;
{
	char *str;
	int  len=0;
	int  notGoodString=0;
	int  do_find_reset=1;

	str = XmTextGetString( Find_pn_value );

	if ( str ) len = strlen( str );

	/* Sanity check string */

	if ( notGoodString )
	{
	    user_alert( TreeWidget, "Invalid process Name!!");
	    return;
	}

	if ( Find_process_name && ( strcmp( Find_process_name, str ) == 0 ) )
		do_find_reset = 0;

	if ( Find_process_name  ) XtFree( Find_process_name );

	Find_process_name = str;

	if ( noop )	/* called to get snarf values */
		return;

	/* we get here if user actually entered a string */

	/* If it's the same we leave Find_process_id alone so it does a next */

	if ( do_find_reset )
	{
	    Find_process_id = -1;  /* to reset find next */
	    XmTextSetString( Find_pid_value, "" );
	}


	start_find( w, Find_pw, call_data);
}

void
do_get_process_pid( w, noop, call_data)
Widget w;
int noop;
caddr_t call_data;
{
	char *str;
	int  new_value;
	int nonNumerics=0;
	int i, len;
	int valueChanged=0;

	str = XmTextGetString( Find_pid_value );

	if ( str == NULL )
	{
		Find_process_id = -1;
		return;
	}

	len = strlen( str );
	if ( len == 0 )
	{
		Find_process_id = -1;
		XtFree( str );
		return;
	}

	for ( i = 0 ; i < len ; i++ )
	{
		if ( (str[i] >= '0' ) && ( str[i] <= '9' ) )
			continue;

		Find_process_id = -1;
		nonNumerics++;
	}

	/* check for any non numerics */
	if ( nonNumerics )
	{
	    user_alert( TreeWidget, "PID must use numbers only" );
	    XtFree( str );
	    return;
	}

	new_value = atoi( str );
	XtFree( str );

	if ( new_value < 1 )
	{
	    user_alert( TreeWidget, "PID must be  1 or larger!!" );
	    XtFree( str );
	    return;
	}

	if ( new_value > 99999 )  /* 32k? */
	{
	    user_alert( TreeWidget, 
	     "Process ID's don't typically go over 99999" );
	}

	if ( Find_process_id != new_value )
		valueChanged++;

	Find_process_id = new_value;

	if ( noop )
		return;

	/* If we get here user entered a new pid */

	if ( valueChanged )
	{
	   if ( Find_process_name  ) 
	   {
	       	XtFree( Find_process_name ); 
	       	Find_process_name = NULL;

		XmTextSetString( Find_pn_value, "" );
	   }
	}

	start_find( w, Find_pw, call_data);
}

void
do_start_search( w, find_pw, call_value)
Widget w;
Widget find_pw;
XmAnyCallbackStruct *call_value;
{
	int noop=1;
	
	do_get_process_name( Find_pn_value, noop, call_value);
	do_get_process_pid(  Find_pid_value, noop, call_value);


	start_find( w, find_pw, call_value);
}

void
enter_find_dialog( w, noop, event, continueDispatch)
Widget w;
int  noop;
XCrossingEvent *event;
Boolean *continueDispatch;
{
	In_Find_dialog++;

        *continueDispatch = FALSE;
}

void
leave_find_dialog( w, noop, event, continueDispatch)
Widget w;
int  noop;
XCrossingEvent *event;
Boolean *continueDispatch;
{
	if ( Find_pti_ptr == NULL )
		return;

	if ( event->detail == NotifyNonlinear )
        {
		/* printf("leaving find widget nonLinear \n"); */
		In_Find_dialog=0;
		destroy_process_tip_if_displayed( Find_pti_ptr );
        }
	else if ( event->detail == NotifyAncestor )
	{
		/* printf("leaving find widget Ancestor \n"); */
		;
	}
	else if ( event->detail == NotifyVirtual )
	{
		/* Window exited ?*/
		/* printf("leaving find widget Virtual \n"); */

		In_Find_dialog=0;
		destroy_process_tip_if_displayed( Find_pti_ptr );
	}
	else if ( event->detail == NotifyInferior )
	{
		/* going into interior widget - ignore */
		/* printf("leaving find widget Inferior \n"); */
		;
	}
	else if ( event->detail == NotifyNonlinearVirtual )
	{
		/* Something like Alt tab to another window? */
		/* printf("leaving find widget nonLinearVirtual \n"); */

		In_Find_dialog=0;
		destroy_process_tip_if_displayed( Find_pti_ptr );
	}
        else
        {
		printf("leaving find widget via unknown event - border\n");
               DEBUG0(8,"Leaving widget: on widgets border\n");
        }



        *continueDispatch = FALSE;
}


void
do_proc( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int new_value=0;

	DEBUG1( 1, "do_proc client data(%s)\n", client_data);

	if ( strcmp( client_data, "daemons") == 0 )
	{
		new_value = True;
	}
	else if ( strcmp( client_data, "foreground") == 0 )
	{
		new_value = False;
	}

	if ( new_value != Show_daemons )
	{
		set_button_from_menu_label( w );
		Show_daemons = new_value;
		New_display_criteria++;
	}
}

void 
do_color( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int new_value;
	int dynamic_colors=0;

	if ( strcmp( client_data, "uid") == 0 )
	{
		new_value = COLOR_BY_REAL_USER_ID;
	}
	else if ( strcmp( client_data, "euid") == 0 )
	{
		new_value = COLOR_BY_EFFECTIVE_USER_ID;
	}
	else if ( strcmp( client_data, "gid") == 0 )
	{
		new_value = COLOR_BY_REAL_GROUP_ID;
	}
	else if ( strcmp( client_data, "egid") == 0 )
	{
		new_value = COLOR_BY_EFFECTIVE_GROUP_ID;
	}
	else if ( strcmp( client_data, "tcpu") == 0 )
	{
		new_value = COLOR_BY_TOTAL_TIME;
		dynamic_colors++;
	}
	else if ( strcmp( client_data, "load") == 0 )
	{
		new_value = COLOR_BY_LOAD;
		dynamic_colors++;
	}
	else if ( strcmp( client_data, "status") == 0 )
	{
		new_value = COLOR_BY_STATUS;
		dynamic_colors++;
	}
	else if ( strcmp( client_data, "resident") == 0 )
	{
		new_value = COLOR_BY_RESIDENT_PAGES;
		dynamic_colors++;
	}
	else if ( strcmp( client_data, "image") == 0 )
	{
		new_value = COLOR_BY_IMAGE_SIZE;
		dynamic_colors++;
	}
	else if ( strcmp( client_data, "priority") == 0 )
	{
		new_value = COLOR_BY_PRIORITY;
		dynamic_colors++;
	}

	if ( new_value != Color_based_on )
	{
		DEBUG1( 1, "do_color client data(%s)\n", client_data);
		Color_based_on = new_value;
		set_button_from_menu_label( w );

		if ( dynamic_colors )
		{
			Dynamic_colors++;
			recalc_process_colors();
		}
		else	
		{
			Dynamic_colors = 0;
			recalc_process_colors();
		}

		New_display_criteria++;
		change_colors();
		display_title();
	}
}

void 
do_action( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int        new_value;

	DEBUG1( 1, "do_action client data(%s)\n", client_data);

	if ( strcmp( client_data, "select") == 0 )
	{
	 	new_value = ACTION_PROCESS_SELECT;
	}
	else if ( strcmp( client_data, "details") == 0 )
	{
	 	new_value = ACTION_PROCESS_DETAILS;
	}
	else if ( strcmp( client_data, "signal") == 0 )
	{
	 	new_value = ACTION_PROCESS_SIGNAL;
	}
	else if ( strcmp( client_data, "kill") == 0 )
	{
	 	new_value = ACTION_KILL_PROCESS;
	}
	else if ( strcmp( client_data, "outline") == 0 )
	{
	 	new_value = ACTION_OUTLINE_SUBTREE;
	}
	else if ( strcmp( client_data, "hide") == 0 )
	{
	 	new_value = ACTION_HIDE_SUBTREE;
	}
	else if ( strcmp( client_data, "show") == 0 )
	{
	 	new_value = ACTION_SHOW_SUBTREE;
	}

	if ( new_value != Process_action )
	{
		set_button_from_menu_label( w );
		Process_action = new_value;
		set_cursor();
	}
}

void 
do_deco( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int        new_value=0;

	DEBUG1( 3, "do_deco client data(%s)\n", client_data);

	if ( strcmp( client_data, "none") == 0 )
	{
	 	new_value = DECORATE_NONE;
	}
	else if ( strcmp( client_data, "distinguished") == 0 )
	{
	 	new_value = DECORATE_DISTINGUISHED;
	}

	if ( new_value != Decorated_tree )
	{
		set_button_from_menu_label( w );
		change_decor( new_value );
	}
}

void 
do_sig( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int new_sig;

	DEBUG1( 1, "do_sig client data(%s)\n", client_data);

	new_sig = atoi( client_data );

	if ( new_sig == Current_signal )
		return;

	if ( new_sig < 1 || new_sig > MaxSig )
	{
		DEBUG1( 1, "do_sig: Invalid signal(%d)\n", client_data);
	}

	Current_signal = new_sig;

	set_cursor();
}


void 
do_displayManPage( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	int new_sig;

	DEBUG1( 1, "do_displayManPage client data(%s)\n", client_data);

	doMan( (char*) client_data );
}

void 
do_help( w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	DEBUG1( 1, "do_help client data(%s)\n", client_data);

	show_help( w, client_data );
}


popup_timer_controls( parent )
Widget parent;
{
	static 		Widget timer_pw=NULL;
	Widget 		st_label, st_value, st_units;
	Widget 		lt_label, lt_value, lt_units;
	Widget 		sep1, done_w, cancel_w, help_w, mSecDef;
	XmString	title_str;
	char		buf[12];
	int    		n;
	Arg    		wargs[14];

	if ( timer_pw == NULL )
	{
	    title_str = XmStringCreate("Timer Control Panel",
						XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg( wargs[n], XmNdialogTitle,  title_str ); n++;
 	    XtSetArg( wargs[n], XmNautoUnmanage, FALSE ); n++;
            XtSetArg( wargs[n], XmNfractionBase, 10 ); n++;
	    timer_pw = XmCreateFormDialog( parent, "Timer Controls",
								    wargs, n);
	    if ( timer_pw == NULL )
	    {
		DEBUG1(1, "popup_timer_controls: Failed parent(%d)\n",parent);
		return;
	    }

	    /* Create done, cancel, help buttons at bottom */

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 1); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 3); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
            done_w = XtCreateManagedWidget("Done", xmPushButtonWidgetClass,
                                                timer_pw, wargs, n );

	    XtAddCallback( done_w, XmNactivateCallback, do_timer_done,timer_pw);

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 4); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 6); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
            cancel_w = XtCreateManagedWidget("Cancel", xmPushButtonWidgetClass,
                                                timer_pw, wargs, n );

	    XtAddCallback( cancel_w, XmNactivateCallback, do_popdown, timer_pw);

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 7); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 9); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
            help_w = XtCreateManagedWidget("Help", xmPushButtonWidgetClass,
                                                timer_pw, wargs, n );

	    XtAddCallback( help_w, XmNactivateCallback, (XtCallbackProc) do_help, "timer");

	    /* Create short tic widgets */

            n = 0;
            XtSetArg( wargs[n], XmNcolumns, 5 ); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNtopOffset, 4); n++;
            st_value = XtCreateManagedWidget( "short_tic_value", 
						xmTextWidgetClass,
                                                timer_pw, wargs, n);
	    Timer_st_value = st_value;

	    XtAddCallback( st_value, XmNactivateCallback, do_short_tic, 0 );

            n = 0;
            XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_END); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNleftOffset, 2); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, st_value);n++;
            XtSetArg(wargs[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg(wargs[n],XmNbottomWidget,st_value);n++;
            st_label = XtCreateManagedWidget( "Short Tic:", 
						xmLabelWidgetClass,
                                                timer_pw, wargs, n);
            n = 0;
            XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNleftWidget, st_value); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNrightOffset, 2); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, st_value);n++;
            XtSetArg(wargs[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg(wargs[n],XmNbottomWidget,st_value);n++;
            st_units = XtCreateManagedWidget( "Milliseconds", 
						xmLabelWidgetClass,
                                                timer_pw, wargs, n);


	    /* Create long tic widgets */

            n = 0;
            XtSetArg( wargs[n], XmNcolumns, 5 ); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNleftPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNtopWidget, st_value); n++;
            XtSetArg( wargs[n], XmNtopOffset, 2); n++;
            lt_value = XtCreateManagedWidget( "long_tic_value", 
						xmTextWidgetClass,
                                                timer_pw, wargs, n);
	    Timer_lt_value = lt_value;

	    XtAddCallback( lt_value, XmNactivateCallback, do_long_tic, 0 );

            n = 0;
            XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_END); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNleftOffset, 2); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg( wargs[n], XmNrightPosition, 4); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, lt_value);n++;
            XtSetArg(wargs[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg(wargs[n],XmNbottomWidget,lt_value);n++;
            lt_label = XtCreateManagedWidget( "Long Tic:", 
						xmLabelWidgetClass,
                                                timer_pw, wargs, n);

            n = 0;
            XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNleftWidget, lt_value); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNrightOffset, 2); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, lt_value);n++;
            XtSetArg(wargs[n],XmNbottomAttachment,XmATTACH_OPPOSITE_WIDGET);n++;
            XtSetArg(wargs[n],XmNbottomWidget,lt_value);n++;
            lt_units = XtCreateManagedWidget( "Short tics", 
						xmLabelWidgetClass,
                                                timer_pw, wargs, n);

            n = 0;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNleftOffset, 10); n++;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET);n++;
            XtSetArg( wargs[n], XmNtopWidget, lt_value);n++;
            mSecDef =XtCreateManagedWidget("Note: 1 Second = 1000 Milliseconds",
						xmLabelWidgetClass,
                                                timer_pw, wargs, n);

	    /* Create bottom seperator */

 	    n = 0;
            XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNtopWidget, mSecDef ); n++;
            XtSetArg( wargs[n], XmNtopOffset, 2 ); n++;
            XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
            XtSetArg( wargs[n], XmNbottomWidget, done_w); n++;
            XtSetArg( wargs[n], XmNbottomOffset, 2 ); n++;
            sep1 = XtCreateManagedWidget( "timer_separator1", 
						xmSeparatorGadgetClass,
                                                timer_pw, wargs, n );
	}

	sprintf( buf, "%5d", Short_tic );
	XmTextSetString( Timer_st_value, buf );

	sprintf( buf, "%5d", Long_tic );
	XmTextSetString( Timer_lt_value, buf );

	XtManageChild( timer_pw );

	XmProcessTraversal( Timer_st_value, XmTRAVERSE_CURRENT );
}


void
do_short_tic( w, noop, call_data)
Widget w;
int noop;
caddr_t call_data;
{
	char *str;
	int  new_value;

	str = XmTextGetString( Timer_st_value );

	new_value = atoi( str );

	if ( new_value < 1 )
	{
	    user_alert( TreeWidget, "Short tic has to be 1 or larger!!");
	    return;
	}

	if ( new_value < 10 )
	{
	    user_alert( TreeWidget, 
		"That Short Tic value is really low!! You better be on a fast machine");
	}

	Short_tic = new_value;
}

void
do_long_tic( w, noop, call_data)
Widget w;
int noop;
caddr_t call_data;
{
	char *str;
	int  new_value;

	str = XmTextGetString( Timer_lt_value );

	new_value = atoi( str );

	if ( new_value < 1 )
	{
	    user_alert( TreeWidget, "Long tic has to be 1 or larger!!" );
	    return;
	}

	if ( (new_value * Short_tic) > 20000 )
	{
	    user_alert( TreeWidget, 
	     "That Long Tic value is kinda high!! Process info updates will be kinda slow" );
	}

	Long_tic = new_value;
}

void
do_timer_done( w, timer_pw, call_data)
Widget w;
Widget timer_pw;
caddr_t call_data;
{
	int noop=0;
	
	do_short_tic( Timer_st_value, noop, call_data);

	do_long_tic( Timer_lt_value, noop, call_data);

	XtUnmanageChild( timer_pw );
}



void
popup_pref_dialog( parent )
Widget parent;
{

}
