/* File: 	gei_driver.c
 *
 * Description: Code for the gei driver module
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 2002, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	01/12/2002	Created
 *            
 *
 */

#include <stdio.h>		
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <varargs.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef __linux__
#include <sys/mkdev.h>		/* To get major/minor numbers 		 */
#endif
#include <sys/param.h>		/* To get PAGESIZE i.e. system page	 */
#include <unistd.h>

#include "os_info.h"

#include "tree.h"	/* Generic m-ary tree package include file */
#include "list.h"	/* Simple list package			   */
#include "debug.h"	/* Macro based debug package		   */

#include "machine_info.h"	/* Some process wide definitions	   */

extern int Short_tic;
extern int Long_tic;

/* ----------------------- gei routines ----------------- */


typedef struct _GEI_scanner_info
{
	int       first;
	int	  rfd, wfd;
	FILE	 *rfp;
	char	 *pid_list_buf;
	char	 *pid_buf_p;
	int	  helper_pid;
	int 	  helper_can_cache;
} GEI_scanner_info;

extern int   errno;
extern char **Environment;

int 
gei_init_pid_scanner( machine, libdir )
Machine_info *machine;
char         *libdir;
{
	GEI_scanner_info *gei;
	char		  helper_path[256];
	int		  gotHelper=1;
	pid_t 		  child_pid;
        char 		  *argv[4];
	int		  wfd[2];	/* Pipe Write fds */
	int		  rfd[2];	/* Pipe Read fds */
	FILE		  *child_fp;
	int		  rc;
	char		  *p, buf[132];
	int		  EOM=0;

printf("init -> %s\n", libdir );

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) malloc( sizeof( GEI_scanner_info ));
	if ( gei == NULL )
	{
		fprintf( stderr, "Out of memory allocating gei info scanner\n");
		return -1;
	}

	gei->pid_list_buf = NULL;
	gei->pid_buf_p    = NULL;
	gei->helper_pid   = -1;
	gei->rfd   	  = -1;
	gei->wfd   	  = -1;
	gei->helper_can_cache = 0;

	machine->data = (void *) gei;

	sprintf( helper_path, "%s/helper/%s/%s/%s_helper", libdir, 
				machine->osType,
				machine->osRelease,
				machine->osType );

printf("Try helper -> %s\n", helper_path );

	if ( (gotHelper = executableExists( helper_path )) != 1 )
	{
	  sprintf( helper_path, "%s/helpers/%s/%s_helper", libdir, 
				machine->osType,
				machine->osType );

	  printf("Try helper -> %s\n", helper_path );

	  if ( (gotHelper = executableExists( helper_path )) != 1 )
	  {
	    sprintf( helper_path, 
	    		"%s/helpers/Generic/generic_helper", libdir );

	    if ( (gotHelper = executableExists( helper_path )) != 1 )
	    {
	        printf("Got no helper!!! not even -> %s\n", helper_path );
	    }
	  }
	}

	if ( ! gotHelper )
		return -1;

	printf("Use helper -> %s\n", helper_path );

	if ( pipe( wfd ) < 0 )
	{
	  fprintf( stderr, "Can't make helper pipe!!!\n" );
	  return -1;
	}

	if ( pipe( rfd ) < 0 )
	{
	  fprintf( stderr, "Can't make helper pipe!!!\n" );
	  return -1;
	}

	child_pid = fork();

        if ( child_pid == -1 )
	{
	    fprintf( stderr, "Can't fork for helper !!!\n" );
	    return -1;
	}

        if ( child_pid == 0 )  /* In child process */
        {
	    close( rfd[0] );		/* use em half duplex */
	    close( wfd[1] );		

	    if ( wfd[0] != STDIN_FILENO )
	    {
	      if ( dup2( wfd[0], STDIN_FILENO ) != STDIN_FILENO )
	      {
	          fprintf( stderr, "dup2 failed for helper STDIN!!!\n" );
		  exit( 1 );
	      }
	    }
	    close( wfd[0] );

	    if ( rfd[1] != STDOUT_FILENO )
	    {
	      if ( dup2( rfd[1], STDOUT_FILENO ) != STDOUT_FILENO )
	      {
	          fprintf( stderr, "dup2 failed for helper STDOUT!!!\n" );
		  exit( 2 );
	      }
	    }
	    close( rfd[1] );

            setuid( getuid() );		/* remove setuid if any */

            argv[0] = "sh";
            argv[1] = "-c";
            argv[2] = helper_path;
            argv[3] = 0;

            execve("/bin/sh", argv, Environment);

            exit(127);
        }
	else  /* parent */
	{
	    close( rfd[1] );
	    close( wfd[0] );

	    gei->rfd = rfd[0];	/* Save read/write fd's for later */
	    gei->wfd = wfd[1];

	    /* Convert read fd to fp for ease of processing */

	    gei->rfp = fdopen( gei->rfd, "r" );
	    if ( gei->rfp == NULL )
	    {
	      fprintf( stderr, "failed to conver read fd to FILE *\n" );
	      return -1;
	    }

	    setlinebuf( gei->rfp );

	    gei->helper_pid = child_pid;      /* save info for sig handler */

printf("Activating Scanner \n" );

	    rc = write( gei->wfd, "hello\n", 6 );
	    if ( rc < 6 )
	    {
	          fprintf( stderr, "Failed to write to Helper!!!\n" );
		  return -1;
	    }

printf("Wait for response \n" );
	    while ( ! EOM )
	    {
	      p = fgets( buf, 132, gei->rfp );        /* wait for hello response */
	      if ( p == NULL )
	      {
	          fprintf( stderr, "Helper is huffing?\n" );
	      }

	      buf[131] = '\0';

printf("Helper says -> %s\n", buf );

	      if ( (rc = strncmp( buf, ":EOM:", 5 )) == 0 )
	      {
	      	 EOM++;
		 continue;
	      }

	      if ( (rc = strncmp( buf, "short_tic:", 10 )) == 0 )
	      {
	      	 Short_tic = atoi( &(buf[11]) );
	      }
	      else if ( (rc = strncmp( buf, "long_tic:", 9 )) == 0 )
	      {
	      	 Long_tic = atoi( &(buf[10]) );
	      }
	      else if ( (rc = strncmp( buf, "cache: yes", 10 )) == 0 )
	      {
		    gei->helper_can_cache = 1;
	      }
	    }

	}

	gei->first = 1;

	return child_pid;
}

int 
gei_cache_process_details( machine, process_list )
Machine_info *machine;
lnode *process_list;
{
	GEI_scanner_info *gei;
	int	          rc;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	if ( gei->helper_can_cache )
	{
	  /* Send request to cache process info */

	    rc = write( gei->wfd, "cache_details\n", 14 );
	    if ( rc < 6 )
	    {
	        fprintf( stderr, "Failed to write to Helper - cache details!!!\n");
	        return -1;
	    }
printf("request cache_details\n");
	}

	return 0;
}

int 
gei_release_process_details( machine, process_list )
Machine_info *machine;
lnode *process_list;
{
	GEI_scanner_info *gei;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	if ( gei->helper_can_cache )
	{
	  /* Send request to release cache process info */
	}

	return 0;
}

int 
gei_get_machine_status( machine )
Machine_info *machine;
{
	GEI_scanner_info *gei;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	return 0;
}


int
executableExists( path )
char *path;
{
	struct  stat    s;
	int		m, rc, uid, gid;

	rc = stat( path, &s );
	if ( rc < 0 )
		return 0;

	m = s.st_mode;

	if ( ! S_ISREG( m ) )
		return 0;

	uid = getuid();
	gid = getgid();


	if (  ( m & S_IXOTH ) ||
	      ( ( m & S_IXGRP ) && ( gid == s.st_gid ) ) ||
	      ( ( m & S_IXUSR ) && ( uid == s.st_uid ) ) )
	      return 1;

	return 0;
}


#define PROCESS_LIST_BUF_SIZE 10000 /* SWAG - enough for 1000 processes */

int 
gei_get_first_pid( machine )
Machine_info *machine;
{
	GEI_scanner_info *gei;
	int request_len, len, rc;
	char *request, *response;
	char buf[ PROCESS_LIST_BUF_SIZE ];    


	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei )       return -1;

	if ( gei->wfd == -1 ) return -1;

	request = "pid_list short_form\n";
	request_len = strlen( request );

printf("Get first\n" );
printf("Send -> %s", request );

	rc = write( gei->wfd, request, request_len );
	if ( rc < 0 )
	{
	    fprintf( stderr, "Error writing Pid_List request\n" );
	    return -1;
	}

printf("Wait for response\n ");

	/* read and cache list */
	response = fgets( buf, PROCESS_LIST_BUF_SIZE, gei->rfp );
	if ( response == NULL )
	{
	    fprintf( stderr, "Error reading pid_list response\n" );
	    return -1;
	}

printf( "pid_list Response -> %s \n", response );
	
	len = strlen( response );

	if ( response[len-1] == '\n' ) response[len-1] = '\0';

	if ( gei->pid_list_buf != NULL )
		free( gei->pid_list_buf );

	/* Check for error response */

	gei->pid_list_buf = strdup( response );
	gei->pid_buf_p    = gei->pid_list_buf;

	/* get and return first pid of list */

	if ( gei->first )
	    gei->first = 0;
	    
	return  machine->get_next_pid( machine );
}

int
gei_get_next_pid( machine )
Machine_info *machine;
{
	int 		done    = 0;
	int 		pid     = -1;
	GEI_scanner_info *gei;
	char		*p, *e;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;
	if ( gei->wfd == -1 ) return -1;

	p = gei->pid_buf_p;

	pid = strtol( p, &e, 10 );
	if ( e == p )
		return -1;

	gei->pid_buf_p = e;
	
	return pid;
}

#define PROCESS_DETAILS_BUF_SIZE 4096

int 
gei_get_pid_main_details( machine, pid, varlist )
Machine_info *machine;
int pid;
VarList *varlist;	/* Caller free's returned space */
{
	GEI_scanner_info *gei;
	int		  i, k, rc, len, request_len, numvars, lookForValue;
	char		  *response, *name, *value;
	char		  request[30];
	VarList		  v;
	char		  buf[PROCESS_DETAILS_BUF_SIZE];

printf("Get pid details - %d\n", pid );

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	if ( gei->wfd == -1 ) return -1;

	if ( *varlist != NULL )
		return -1;

	sprintf( request, "pid_details %d\n", pid );
	request_len = strlen( request );

printf("Get pid details - %d\n", pid );
printf("Send -> %s", request );

	rc = write( gei->wfd, request, request_len );
	if ( rc < 0 )
	{
	    fprintf( stderr, "Error writing pid_details request\n" );
	    return -1;
	}

printf("Wait for response\n ");

	v = (VarList ) malloc ( sizeof( VarBind ) * 24 );
	if ( v == NULL )
	{
	    fprintf( stderr, "Error out of space VarBind\n" );
	    return -1;
	}

	*varlist = v;

	numvars = 23; 		
	for ( i = 0 ; i < numvars + 1; i++ ) /* Extra for null termination */
	{
	  v[i].name  = NULL;
	  v[i].value = NULL;
	}

	for ( i = 0 ; i < numvars; i++ )
	{
	  response = fgets( buf, PROCESS_DETAILS_BUF_SIZE, gei->rfp );
	  if ( response == NULL )
	  {
	    fprintf( stderr, "Error reading pid_details var(%d)\n", i );
	    freeVarList( v, numvars );
	    return i;
	  }

printf( "pid_details Response -> %s ", response );
	
	  len = strlen( response );
	  if ( response[len-1] == '\n' ) response[len-1] = '\0';

	  if ( (rc = strncmp( response, "Error", 5 )) == 0 )
	  {
	    fprintf( stderr, "Error response reading pid_details var(%d)\n", i );
	    freeVarList( v, numvars );
	    return -1;
	  }

	  name = response;

	  lookForValue = 0;
	  for ( k = 0 ; k < len ; k++ )
	  {
		if ( lookForValue )
		{
		  if ( response[k] == ' ' || response[k] == '\t' )
		  	;
		  else
		  {
	  	    value = &(response[k] );
		    break;
		  }
		}

	  	if ( response[k] == ':' )
		{
			response[k]  = '\0';
			lookForValue = 1;
		}
	  }

	  v[i].name  = strdup( name );
	  v[i].value = strdup( value );

printf( "pid_details name(%s) value(%s)\n", v[i].name, v[i].value );

	}

	return i;
}

int 
gei_get_pid_extra_details( machine, pid )
Machine_info *machine;
int pid;
{
	GEI_scanner_info *gei;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	if ( gei->wfd == -1 ) return -1;

	return 0;
}


int 
gei_close_pid_scanner( machine )
Machine_info *machine;
{
	GEI_scanner_info *gei;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	kill( gei->helper_pid, SIGKILL );   /* cleanup routine will get called */

	return 0;
}

int 
gei_cleanup_pid_scanner( machine, pid )
Machine_info *machine;
int pid;
{
	GEI_scanner_info *gei;

	if ( !machine ) return -1;

	gei = (GEI_scanner_info *) machine->data;

	if ( !gei ) return -1;

	if ( gei->helper_pid != pid )
		return -1;

	if ( gei->wfd == -1 ) return -1;

	if ( gei->pid_list_buf )
		free( gei->pid_list_buf );

printf("GEI Cleanup helper(%d)\n", pid );

	close ( gei->rfd );
	close ( gei->wfd );
	fclose ( gei->rfp );
	
	gei->rfd	= -1;
	gei->wfd	= -1;
	gei->rfp	= NULL;
	gei->pid_buf_p  = NULL;
	gei->helper_pid = -1;

	return 0;
}
