/* File: 	button_bar.c
 *
 * Description: A simple buton bar
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1995, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	2/26/95	Created
 *            
 *
 */

#include <stdio.h>		

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/PushB.h>

#include <Xpsi/Tree.h>

#include <sys/types.h>
#include <unistd.h>

#include "tree.h"		/* M-ary tree package 		   */
#include "list.h"		/* Simple list package		   */
#include "user_group.h"		/* User_group display definitions  */
#include "treeps.h"		/* Process wide defines		   */
#include "color_code.h"
#include "icons.h"

#include "debug.h"		/* Simple debug package		   */

			
#include "icons/mono/16/run"		/* Default Button bar bitmaps   */
#include "icons/mono/16/halt"

#include "icons/mono/16/gutree"
#include "icons/mono/16/cmap"
#include "icons/mono/16/fields"

#include "icons/mono/16/login"
#include "icons/mono/16/all"
#include "icons/mono/16/root"
#include "icons/mono/16/uucp"
#include "icons/mono/16/user_group"

#include "icons/mono/16/user"
#include "icons/mono/16/euid"
#include "icons/mono/16/gid"
#include "icons/mono/16/egid"
#include "icons/mono/16/cpu"
#include "icons/mono/16/load"
#include "icons/mono/16/mem"
#include "icons/mono/16/size"
#include "icons/mono/16/priority"
#include "icons/mono/16/status"

#include "icons/mono/16/fore"
#include "icons/mono/16/daemons"

#include "icons/mono/16/select"
#include "icons/mono/16/info"
#include "icons/mono/16/signal"
#include "icons/mono/16/kill"
#include "icons/mono/16/outline"
#include "icons/mono/16/hide"
#include "icons/mono/16/show"

#include "icons/mono/16/last_posn"
#include "icons/mono/16/center_posn"
#include "icons/mono/16/first_posn"

#include "icons/mono/16/decorate"

#include "button_bar.h"


void do_run();
void do_gutree();
void do_cmap();
void do_fields();
void do_userb();
void do_colorb();
void do_actionb();
void do_daemonb();
void do_placement();
void do_decorate();

int setup_buttons();

void set_arm_indicator();
void arm();
void disarm();
void toggle_button();


/* Note: Toggle buttons must match menu items for proper toggle button
 * and menu toggle indicators to syncronize completely. Therfore always
 * create toggle buttons for each menu item toggle group and vs. versa.
 * They need not be displayed, but they must exist so the set_button
 * function can find a button and hence arm can figure out which 
 * toggle group to operate on. This means that if a menu item
 * is selected from a toggle group that does not have a displayed
 * button, the old armed button will still be disarmed properly.
 */

Button_item  Treeps_buttons[]={
	{ PIXMAP_TYPE, "run", "run",
			"Halt", "Run", 
			True, 0, 
			&(run_bits[0]), &(run_bits[0]), 
 	  		run_width, run_height, 4, 
			do_run, "run", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Run Data Gathering Engine" },



	{ PIXMAP_TYPE, "gutree", NULL,
			"gutree", "gutree", 
			False, 0,
			&(gutree_bits[0]), &(gutree_bits[0]), 
			gutree_width, gutree_height, 8, 
			do_gutree, "gutree", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Group/User Tree Dialog"},

  	{ PIXMAP_TYPE, "cmap", NULL,
			"Color Map", "Color Map", 
			False, 0,
			&(cmap_bits[0]), &(cmap_bits[0]), 
			cmap_width, cmap_height, 1, 
			do_cmap, "Cmap", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Color Map Viewer"},

  	{ PIXMAP_TYPE, "fields", NULL,
			"Fields", "Fields", 
			False, 0,
			&(fields_bits[0]), &(fields_bits[0]), 
			fields_width, fields_height, 1, 
			do_fields, "Fields", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Field Selector"},



	{ PIXMAP_TYPE, "login", NULL, 
			"Login User", "Login User", 
			True, 1, 
			&(login_bits[0]), &(login_bits[0]), 
			login_width, login_height, 8, 
			do_userb, "login", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Currently Logged in Users Processes"},
	{ PIXMAP_TYPE, "all", NULL,
			"All Users", "All Users", 
			False, 2, 
			&(all_bits[0]), &(all_bits[0]), 
			all_width, all_height, 1, 
			do_userb, "all", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display All Users Processes"},
	{ PIXMAP_TYPE, "root", NULL,
			"Root", "Root", 
			False, 3,
			&(root_bits[0]), &(root_bits[0]), 
			root_width, root_height, 1, 
			do_userb, "root", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Root Users Processes"},
	{ PIXMAP_TYPE, "uucp", NULL,
			"Uucp", "Uucp", 
			False, 4,
			&(uucp_bits[0]), &(uucp_bits[0]), 
			uucp_width, uucp_height, 1, 
			do_userb, "uucp", 0, 0, 0, 0, NULL, NULL, NULL,
			False,  /* Change to True to enable in button bar */
			"Display UUCP Processes"},
	{ PIXMAP_TYPE, "user_group", NULL,
			"User/Group Selection", "User/Group Selection", 
			False, 5,
			&(user_group_bits[0]), &(user_group_bits[0]), 
			user_group_width, user_group_height, 1, 
			do_userb, "user_group", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Processes Selected in Group/User Dialog"},



	{ PIXMAP_TYPE, "user", NULL,
			"Real User Id", "Real User Id", 
			False, 1,
			&(user_bits[0]), &(user_bits[0]), 
			user_width, user_height, 8, 
			do_colorb, "uid", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Processes Colored by Real User Id"},
	{ PIXMAP_TYPE, "euid", NULL,
			"Effective User Id", "Effective User Id", 
			True, 2,
			&(euid_bits[0]), &(euid_bits[0]), 
			euid_width, euid_height, 1, 
			do_colorb, "euid", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Processes Colored by Effective User Id"},

	{ PIXMAP_TYPE, "gid", NULL,
			"Real Group Id", "Real Group Id", 
			False, 3,
			&(gid_bits[0]), &(gid_bits[0]), 
			gid_width, gid_height, 1, 
			do_colorb, "gid", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processes Coloered by Real Group Id",
			},
	{ PIXMAP_TYPE, "egid", NULL,
			"Effective Group Id", "Effective Group Id", 
			False, 4,
			&(egid_bits[0]), &(egid_bits[0]), 
			egid_width, egid_height, 1, 
			do_colorb, "egid", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processes Colored by Effective Group Id",
			},
  	{ PIXMAP_TYPE, "cpu", NULL,
			"Total CPU Time", "Total CPU Time", 
			False, 5,
			&(cpu_bits[0]), &(cpu_bits[0]), 
			cpu_width, cpu_height, 1, 
			do_colorb, "CPU", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processed Colored by Total CPU Time"},
	{ PIXMAP_TYPE, "load", NULL,
			"Current CPU Load", "Current CPU Load", 
			False, 6,
			&(load_bits[0]), &(load_bits[0]), 
			load_width, load_height, 1, 
			do_colorb, "Load", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processes Colored by CPU Load"},
  	{ PIXMAP_TYPE, "status", NULL,
			"Process Status", "Process Status", 
			False, 7,
			&(status_bits[0]), &(status_bits[0]), 
			status_width, status_height, 1, 
			do_colorb, "status", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processed Colored by Status"},
  	{ PIXMAP_TYPE, "mem", NULL,
			"Resident Memory", "Resident Memory", 
			False, 8,
			&(mem_bits[0]), &(mem_bits[0]), 
			mem_width, mem_height, 1, 
			do_colorb, "Mem", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processes Colored by Resident Memory"},
  	{ PIXMAP_TYPE, "size", NULL,
			"Image Size", "Image Size", 
			False, 9,
			&(size_bits[0]), &(size_bits[0]), 
			size_width, size_height, 1, 
			do_colorb, "size", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processes Colored by Image Size"},
  	{ PIXMAP_TYPE, "priority", NULL,
			"Priority", "Priority", 
			False, 10,
			&(priority_bits[0]), &(priority_bits[0]), 
			priority_width, priority_height, 1, 
			do_colorb, "priority", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Processes Colored by Priority"},



	{ PIXMAP_TYPE, 	"daemons", "fore",
			"Foreground", "Daemons", 
			True, 0,
			&(daemons_bits[0]), &(fore_bits[0]), 
			fore_width, fore_height, 6, 
			do_daemonb, "daemon", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Daemon Processes"},



	{ PIXMAP_TYPE,  "select", NULL,
			"Select", "Select", 
			False, 1,
			&(select_bits[0]), &(select_bits[0]), 
			select_width, select_height, 6, 
			do_actionb, "select", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Select a Process"},
	{ PIXMAP_TYPE,  "info", NULL,
			"Display Process Details", "Display Process Details", 
			True, 2,
			&(info_bits[0]), &(info_bits[0]), 
			info_width, info_height, 1, 
			do_actionb, "info", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Display Process Details"},
	{ PIXMAP_TYPE,  "signal", NULL,
			"Signal Process", "Signal Process", 
			False, 3,
			&(signal_bits[0]), &(signal_bits[0]), 
			signal_width, signal_height, 1, 
			do_actionb, "signal", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Send Signal to Process"},
	{ PIXMAP_TYPE,  "kill", NULL,
			"Kill Process", "Kill Process", 
			False, 4,
			&(kill_bits[0]), &(kill_bits[0]), 
			kill_width, kill_height, 1, 
			do_actionb, "kill", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Kill a Process(Send Signal #9)"},
	{ PIXMAP_TYPE,  "outline", NULL,
			"Outline Subtree", "Outline Subtree", 
			False, 5,
			&(outline_bits[0]), &(outline_bits[0]), 
			outline_width, outline_height, 1, 
			do_actionb, "outline", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Outline subtree"},
	{ PIXMAP_TYPE,  "hide", NULL,
			"Hide Subtree", "Hide Subtree", 
			False, 6,
			&(hide_bits[0]), &(hide_bits[0]), 
			hide_width, hide_height, 1, 
			do_actionb, "hide", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Hide Subtree"},
	{ PIXMAP_TYPE,  "show", NULL,
			"Show Subtree", "Show Subtree", 
			False, 7,
			&(show_bits[0]), &(show_bits[0]), 
			show_width, show_height, 1, 
			do_actionb, "show", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Show all Processes in Subtree"},

	{ PIXMAP_TYPE,  "last_posn", NULL,
			"Above Last Child", "Above Last Child", 
			False, 1,
			&(last_posn_bits[0]), &(last_posn_bits[0]), 
			last_posn_width, last_posn_height, 6, 
			do_placement, "last", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Parent Positioned Above Last Child"},

	{ PIXMAP_TYPE,  "center_posn", NULL,
			"Centered Above Children", "Centered Above Children", 
			True, 2,
			&(center_posn_bits[0]), &(center_posn_bits[0]), 
			center_posn_width, center_posn_height, 1, 
			do_placement, "center", 0, 0, 0, 0, NULL, NULL, NULL,
			True,
			"Center Parent Above Children"},

	{ PIXMAP_TYPE,  "first_posn", NULL,
			"Above First Child", "Above First Child", 
			False, 3,
			&(first_posn_bits[0]), &(first_posn_bits[0]), 
			first_posn_width, first_posn_height, 6, 
			do_placement, "first", 0, 0, 0, 0, NULL, NULL, NULL,
			False,
			"Parent Positioned Above First Child"},

	{ PIXMAP_TYPE,  "decorate", NULL,
			"No Decorations", "Distinguished", 
			True, 0,
			&(decorate_bits[0]), &(decorate_bits[0]), 
			decorate_width, decorate_height, 6, 
			do_decorate, "distinguished", 0, 0, 0, 0, NULL, 
							NULL, NULL,
			True,
			"Decorate Distinguished Processes"},

};

extern Widget Main_menu;

Button_item  *Buttons=NULL;
int 	      Button_cnt=0;

#define ICON_MODE	0
#define TEXT_MODE	1

int	      	ButtonVisualMode=ICON_MODE;
int		ButtonColorMode=MONOCHROME;
int		ButtonIconSize=16;		/* 16/22/32/48/64 */


extern User_group_info  *Users;		/* From treeps.c */

extern int   CB_dialog_state;		/* From color_code.c */
extern void  (*CB_popdownCB)();

extern Widget Display_list_pw;		/* From treeps.c */
extern void  (*DisplayListPopdownCB)();

int setButtonPixmaps();

/* int ColorDepth=MONOCHROME; */

int ColorDepth=LOW_COLOR;
int IconSize=16;

extern int Decorated_tree;
extern void change_decor();

Widget   Butbar_w;

int 
setButtonParameters( newIconSize, newDepth )
int   newIconSize;
char *newDepth;
{
    switch ( newIconSize ) {
    	case 16:
    	case 22:
    	case 32:
    	case 64:
    		 IconSize = newIconSize;
		 break;

	default:	
	  fprintf( stderr, 
	        "setButtonParameters: Invalid size(%d). try 16/22/32/64\n",
						newIconSize );
	  return -1;
	  break;
    }

    if ( strcmp( newDepth, "mono" ) == 0 )
	ColorDepth=MONOCHROME;
    else if ( strcmp( newDepth, "loColor" ) == 0 )
	ColorDepth=LOW_COLOR;
    else if ( strcmp( newDepth, "hiColor" ) == 0 )
	ColorDepth=HIGH_COLOR;
    else if ( strcmp( newDepth, "highColor" ) == 0 )
	ColorDepth=HIGH_COLOR;
    else 
    {
      fprintf( stderr, 
        "setButtonParameters: Invalid depth(%s). try mono/loColor/hiColor\n",
						newDepth );
	return -1;
    }

    /* Currently must be called prior to creating button bar, later allow
     * these to be changed at run time by user changing preferences
     */

    return 0;
}

Widget
setup_button_bar( parent, attach_to )
Widget parent;
Widget attach_to;
{
	Arg      wargs[20];
	int      n;
	uid_t	 euid;

	n = 0;
	if ( attach_to != NULL )
	{
		XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET);n++;
		XtSetArg( wargs[n], XmNtopWidget, attach_to); n++;
	}
	else
	{
		XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM);n++;
	}

	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
 	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
 	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
 	XtSetArg( wargs[n], XmNmarginHeight, 4); n++;
 	XtSetArg( wargs[n], XmNmarginWidth, 2); n++;
	Butbar_w = XtCreateManagedWidget( "button_bar",  
						xmBulletinBoardWidgetClass,
						 parent, wargs, n );



	setup_buttons( Butbar_w, Treeps_buttons, XtNumber(Treeps_buttons) );


	Buttons    = &(Treeps_buttons[0]);
	Button_cnt = XtNumber(Treeps_buttons);

	if ( ONLY_ROOT_USERS_CAN_READ_ALL_PROC_ENTRIES )
	{
		euid = geteuid();
		if ( euid != 0 )
		{
			n = 0;
 			XtSetArg( wargs[n], XmNsensitive, False); n++;
			XtSetValues( Treeps_buttons[5].w, wargs, n );
			XtSetValues( Treeps_buttons[6].w, wargs, n );
		}
	}

	return( Butbar_w );
}


int
setup_buttons( bbar_w, buttons, num_buttons)
Widget bbar_w;
Button_item buttons[];
int num_buttons;
{
	Pixmap		 icon_pixmap;
	Pixmap		 icon_bitmap;
	int      	 i, j, n;
	Arg 	         wargs[12];
	WidgetList   	 but;
	Position	 x_pos=0;
	XmString	 label_str;
	Widget		 parent;
	Dimension	 wid_width;
	int	 	 src_x, src_y, dest_x, dest_y;
	unsigned int	 width, height;
	GC		 gc;
	unsigned long    plane;
	XGCValues	 values;

	parent = bbar_w;

	src_x = 0; src_y = 0; dest_x = 0; dest_y = 0;

	n = 0;
	XtSetArg( wargs[n], XtNforeground, &values.foreground ); n++;
	XtSetArg( wargs[n], XtNbackground, &values.background ); n++;
	XtGetValues( parent, wargs, n );

	gc = XtGetGC( parent, GCForeground|GCBackground, &values );

	plane = 1;

	but = (WidgetList) XtMalloc( num_buttons * sizeof(Widget) );

        j = 0;
	for ( i = 0; i < num_buttons; i++ )
	{
		if ( buttons[i].active == False )
			continue;

		setButtonPixmaps( &(buttons[i]), parent, gc, 
						ColorDepth, IconSize );

		buttons[i].arm_label_str = XmStringCreate(buttons[i].arm_label,
						     XmSTRING_DEFAULT_CHARSET );
		buttons[i].unarm_label_str =
					XmStringCreate(buttons[i].unarm_label,
						     XmSTRING_DEFAULT_CHARSET );

		x_pos += buttons[i].left_offset;

		n = 0;
 		XtSetArg( wargs[n], XmNx, x_pos );n++;

		if ( buttons[i].armed == True )
		{
			label_str = buttons[i].arm_label_str;
			XtSetArg( wargs[n], XmNlabelPixmap, 
					      buttons[i].arm_pixmap);n++;
		}
		else
		{
			label_str = buttons[i].unarm_label_str;
			XtSetArg( wargs[n], XmNlabelPixmap, 
					      buttons[i].unarm_pixmap);n++;
		}

 		XtSetArg( wargs[n], XmNlabelString, label_str ); n++;

		if ( buttons[i].type == PIXMAP_TYPE )
		{
 			XtSetArg( wargs[n], XmNlabelType, XmPIXMAP );n++;
		}
		else
		{
 			XtSetArg( wargs[n], XmNlabelType, XmSTRING );n++;
		}
 		XtSetArg( wargs[n], XmNmarginHeight, 1 );n++;
 		XtSetArg( wargs[n], XmNmarginWidth,  1 );n++;
		but[j] = XtCreateWidget( "ButtonBarButton",
					xmPushButtonWidgetClass,
						 parent, wargs, n );

		buttons[i].w = but[j];
		if ( buttons[i].armed == True )
 			toggle_button( but[j] );
		
		XtAddCallback(but[j],XmNactivateCallback,
		    			  buttons[i].cb_func, &(buttons[i]) );

		xs_widget_tip( but[j], buttons[i].widgetTip );

		n = 0;
 		XtSetArg( wargs[n], XmNwidth, &wid_width ); n++;
		XtGetValues( but[j], wargs, n );
		x_pos += wid_width;
		j += 1;
	}


	XtManageChildren( but, j );

	return(0);
}

int
setButtonPixmaps( bptr, parent, gc, colorDepth, iconSize )
Button_item *bptr;
Widget	parent;
GC	gc;
int 	colorDepth;
int	iconSize;
{
	Pixmap		 icon_pixmap;
	Pixmap		 icon_bitmap;
	int	 	 rc, src_x, src_y, dest_x, dest_y;
	unsigned int	 width, height;
	unsigned long    plane=1;
	int 		 useBuiltinIcon=0;

	src_x = 0; src_y = 0; dest_x = 0; dest_y = 0;

	if ( (colorDepth == MONOCHROME) && (iconSize == 16) )
	{
		useBuiltinIcon++;
	}
	else	/* Try to load icon */
	{
		DEBUG1( 1, "button(%s) using unarm icon\n", bptr->unarm_fname );

		rc = fetchIcon( bptr->unarm_fname, parent, colorDepth, iconSize, 
					&icon_pixmap, &width, &height);
		if ( rc != 0 )
		{
		    DEBUG1( 1, "Failed to load button icon(%s): Use builtin\n", 
		    					bptr->unarm_fname );
		    useBuiltinIcon++;
		}
	}

	if ( useBuiltinIcon )
	{
	    if ( bptr->default_unarm_pixmap != 0 )
	    {
	    	icon_pixmap = bptr->default_unarm_pixmap;

		width  = 16;  /* Should set these from actual bitmap file */
		height = 16;
	    }
	    else
	    {
		DEBUG1( 1, "button(%s) using unarm builtin\n", bptr->unarm_fname );
		width  = bptr->width;
		height = bptr->height;

        	icon_pixmap = XCreatePixmap( XtDisplay(parent),
                          RootWindowOfScreen( XtScreen(parent)),
                          width,
                          height,
			  DefaultDepthOfScreen( XtScreen(parent)) );

        	icon_bitmap = XCreateBitmapFromData( XtDisplay(parent),
                          RootWindowOfScreen( XtScreen(parent)),
                          BITMAP_DATA_CAST      bptr->unarm_bits,
                          BITMAP_DIMENSION_CAST width,
                          BITMAP_DIMENSION_CAST height );

        	XCopyPlane( XtDisplay(parent), icon_bitmap, icon_pixmap,
				gc, 
				src_x, src_y, 
				width, height, 
				dest_x, dest_y,
				plane );

		XFreePixmap( XtDisplay(parent), icon_bitmap );

	    	bptr->default_unarm_pixmap = icon_pixmap;
	    }
	}
		

	bptr->unarm_pixmap = icon_pixmap;
	bptr->width        = width;
	bptr->height       = height;

	useBuiltinIcon = 0;

	if ( bptr->arm_fname == NULL )
	{
	  bptr->arm_pixmap = icon_pixmap;
	}
	else
	{
	  if ( (colorDepth == MONOCHROME) && (iconSize == 16) )
	  {
		useBuiltinIcon++;
	  }
	  else	/* Try to load icon */
	  {
		DEBUG1( 1, "button(%s) using arm icon\n", bptr->arm_fname );

		rc = fetchIcon( bptr->arm_fname, parent, colorDepth, iconSize, 
					&icon_pixmap, &width, &height);
		if ( rc != 0 )
		{
		    DEBUG1( 1, "Failed to load arm button icon(%s)\n", 
		    					bptr->arm_fname );
		    useBuiltinIcon++;
		}
	  }

	  if ( useBuiltinIcon )
	  {
	    if ( bptr->default_arm_pixmap != 0 )   /* Use cache */
	    {
	    	icon_pixmap = bptr->default_arm_pixmap;
	    }
	    else
	    {
		src_x = 0; src_y = 0; dest_x = 0; dest_y = 0;
		width  = bptr->width;
		height = bptr->height;

		DEBUG1( 1, "button(%s) using arm builtin\n", bptr->arm_fname );

        	icon_pixmap = XCreatePixmap( XtDisplay(parent),
                          RootWindowOfScreen( XtScreen(parent)),
                          width,
                          height,
			  DefaultDepthOfScreen( XtScreen(parent)) );

        	icon_bitmap = XCreateBitmapFromData( XtDisplay(parent),
                          RootWindowOfScreen( XtScreen(parent)),
                          BITMAP_DATA_CAST      bptr->arm_bits,
                          BITMAP_DIMENSION_CAST width,
                          BITMAP_DIMENSION_CAST height );

        	XCopyPlane( XtDisplay(parent), icon_bitmap, icon_pixmap,
				gc, 
				src_x, src_y, 
				width, height, 
				dest_x, dest_y,
				plane );

		XFreePixmap( XtDisplay(parent), icon_bitmap );

		bptr->default_arm_pixmap = icon_pixmap;
	    }
	  }

	  bptr->arm_pixmap = icon_pixmap;
	}
}


void
do_run( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	int rc;
	DEBUG1( 1, "do_run: label(%s)\n", b->arm_label);


	if ( b->armed == False )	   /* Setting to run */
	{
		if ( (rc = set_menu_toggle( Main_menu, "Run", True )) == True )
			return;
		
		Monitor_processes = True;
	}
	else
	{
		if ( (rc = set_menu_toggle( Main_menu, "Halt", True )) == True )
			return;

		Monitor_processes = False;
	}

	set_arm_indicator( w, b );
}

void
toggleGutreeButton()
{
	set_button( Buttons, Button_cnt, "gutree" );
}

void
toggleFieldsButton()
{
	set_button( Buttons, Button_cnt, "Fields" );
}

void
set_fields_callback()
{
	if ( DisplayListPopdownCB == NULL )
	{
	         DisplayListPopdownCB = toggleFieldsButton;
	}
}

void
do_fields( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	DEBUG1( 1, "do_fields: label(%s)\n", b->arm_label);

	set_fields_callback();

        if ( Display_list_pw == NULL )
	{
		show_display_list();
		set_arm_indicator( w, b );
	}
	else if ( XtIsManaged( Display_list_pw ) == True )
	{
		popdown_display_list(); /* Invokes callback which disarms */
	}
	else
	{
		show_display_list();
		set_arm_indicator( w, b );
	}

}

void
set_gutree_callback()
{
	if ( Users->popdownCB == NULL )
	{
	         setPopdownCallback( toggleGutreeButton );
	}
}

void
do_gutree( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	DEBUG1( 1, "do_gutree: label(%s)\n", b->arm_label);

	set_gutree_callback();

	if ( Users->dialog_state == DIALOG_DISPLAYED )
	{
	    popdown_user_tree( Users );
	}
	else 
	{
	    popup_user_tree( Users );

	    /* Toggle group user button?  */

	    /* set_user_list( b->arm_label ); */
	}

	set_arm_indicator( w, b );
}

void
toggleCmapButton()
{
	set_button( Buttons, Button_cnt, "Color Map" );
}

void
set_cmap_callback()
{
	if ( CB_popdownCB == NULL )
	{
	         setCmapPopdownCallback( toggleCmapButton );
	}
}

void
do_cmap( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	DEBUG1( 1, "do_cmap: label(%s)\n", b->arm_label);

	set_cmap_callback();

	if ( CB_dialog_state == DIALOG_DISPLAYED )
	{
	    popdown_color_bar();
	}
	else 
	{
	    show_colors();
	}

	set_arm_indicator( w, b );
}

void
do_userb( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	DEBUG1( 1, "do_userb: label(%s)\n", b->arm_label);

	if ( strcmp( b->arm_label, "User/Group Selection") == 0 )  
	{
	     	DEBUG1(1, "do_userb:  label(%s)\n", b->arm_label);

	     	/* Could popup gutree if not displayed. */
	}

	if ( set_menu_toggle( Main_menu, b->arm_label, True ) )
		return;

	set_arm_indicator( w, b );

	set_user_list( b->arm_label );
}

void
do_colorb( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	int dynamic_colors=0;

	DEBUG1( 1, "do_colorb: label(%s)\n", b->arm_label);

	if ( set_menu_toggle( Main_menu, b->arm_label, True ) )
		return;

	set_arm_indicator( w, b );

	if ( strcmp( b->arm_label, "uid") == 0 )
		Color_based_on = COLOR_BY_REAL_USER_ID;
	else if ( strcmp( b->arm_label, "euid") == 0 )
		Color_based_on = COLOR_BY_EFFECTIVE_USER_ID;
	else if ( strcmp( b->arm_label, "gid") == 0 )
		Color_based_on = COLOR_BY_REAL_GROUP_ID;
	else if ( strcmp( b->arm_label, "egid") == 0 )
		Color_based_on = COLOR_BY_EFFECTIVE_GROUP_ID;
	else if ( strcmp( b->arm_label, "Load") == 0 )
	{
		Color_based_on = COLOR_BY_LOAD;
		dynamic_colors++;
	}
	else if ( strcmp( b->arm_label, "Mem") == 0 )
	{
		Color_based_on = COLOR_BY_RESIDENT_PAGES;
		dynamic_colors++;
	}

	if ( dynamic_colors )
	{
		Dynamic_colors++;
		recalc_process_colors();
	}
	else   
	{
		Dynamic_colors = 0;
		recalc_process_colors();
	}
		
	change_colors();
	display_title();
}


void
do_daemonb( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	DEBUG1( 1, "do_daemonb: label(%s)\n", b->arm_label);


	if ( b->armed == False )	/* setting to daemons */
	{
		if ( set_menu_toggle( Main_menu, b->arm_label, True ) ) 
			return;
		Show_daemons = True;
	}
	else
	{
		if ( set_menu_toggle( Main_menu, b->unarm_label, True ) ) 
			return;
		Show_daemons = False;
	}

	set_arm_indicator( w, b );
	New_display_criteria++;
}

void
do_actionb( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	int newAction=-1;

	DEBUG1( 1, "do_actionb: label(%s)\n", b->arm_label);

	if ( b->armed == True )
	{
		DEBUG1( 1, "do_actionb: Already selected(%s) Ignore\n", 
								b->arm_label);
		return;
	}

	if ( strcmp( b->arm_label, "Select") == 0 )
	{
		newAction = ACTION_PROCESS_SELECT;
	}
	else if ( strcmp( b->arm_label, "Display Process Details") == 0 )
	{
		newAction = ACTION_PROCESS_DETAILS;
	}
	else if ( strcmp( b->arm_label, "Signal Process") == 0 )
	{
		newAction = ACTION_PROCESS_SIGNAL;
	}
	else if ( strcmp( b->arm_label, "Kill Process") == 0 )
	{
		newAction = ACTION_KILL_PROCESS;
	}

	if ( newAction == Process_action )
	{
	    DEBUG1( 1, "do_actionb: newAction(%s) is current action!!\n", 
	    							b->arm_label);
	    return;
	}

	if ( set_menu_toggle( Main_menu, b->arm_label, True ) ) 
	{
	    DEBUG1( 1, "do_actionb: set_menu_toggle for(%s) Failed!!\n", 
	    							b->arm_label);
	}

	set_arm_indicator( w, b );
	set_cursor();
}

void
do_placement( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	ParentPlacement  newPlacement=PARENT_CENTERED;

	DEBUG1( 1, "do_placement: label(%s)\n", b->arm_label);

	if ( b->armed == True )
	{
		DEBUG1( 1, "do_Placement: Already selected(%s) Ignore\n", 
								b->arm_label);
		return;
	}

	if ( set_menu_toggle( Main_menu, b->arm_label, True ) )
		return;


	if ( strcmp( b->arm_label, "Above First Child") == 0 )
	{
		newPlacement = PARENT_ABOVE_FIRST_CHILD;
	}
	else if ( strcmp( b->arm_label, "Centered Above Children") == 0 )
	{
		newPlacement = PARENT_CENTERED;
	}
	else if ( strcmp( b->arm_label, "Above Last Child") == 0 )
	{
		newPlacement = PARENT_ABOVE_LAST_CHILD;
	}

	if ( newPlacement != Current_placement )
        {
                Current_placement = newPlacement;
                set_new_placement( newPlacement );
        }

	set_arm_indicator( w, b );
}

void
do_decorate( w, b, call_data)
Widget w;
Button_item *b;
caddr_t call_data;
{
	int  rc, newDecor=0;

	DEBUG2( 3, "do_decorate: label(%s) armed(%d)\n", b->arm_label, 
								b->armed );

	if ( b->armed )
	{
	    if ((rc = set_menu_toggle(Main_menu,"No Decorations",True)) == True)
		    return;

	    newDecor = DECORATE_NONE;
	}
	else
	{
	    if ((rc = set_menu_toggle( Main_menu,"Distinguished",True)) == True)
		    return;
	    newDecor = DECORATE_DISTINGUISHED;

	}

	if ( newDecor != Decorated_tree )
        {
		change_decor( newDecor );
        }

	set_arm_indicator( w, b );
}

Button_item *
findButtonByName( buttons, num_buttons, button_label )
Button_item *buttons;
int	num_buttons;
char   *button_label;
{
	Button_item *b=NULL;
	Button_item *match=NULL;
	int	     i;

	if ( buttons == NULL )
		return( NULL );

	for ( i = 0; i < num_buttons; i++ )
	{
		b = &(buttons[i]);
		if ( strcmp(b->arm_label, button_label) == 0 )
		{
			match = b;
			break;
		}
		else if ( strcmp(b->unarm_label, button_label) == 0 )
		{
			match = b;
			break;
		}
	}

	return match;
}

int
set_button( buttons, num_buttons, button_label )
Button_item *buttons;
int	num_buttons;
char   *button_label;
{
	Button_item *b;
	int	     rc=-1;

	/* Find the button referenced by it's label and set it, this performs
	 * a toggle or radio action as desired.
	 */

	b = findButtonByName( buttons, num_buttons, button_label );
	if ( b != NULL )
	{
		set_arm_indicator( b->w , b );
		rc = 0;
	}

	return(rc);
}

int
arm_button( buttons, num_buttons, button_label )
Button_item *buttons;
int	num_buttons;
char   *button_label;
{
	Button_item *b;
	int	     rc=-1;

	/* Find the button referenced by it's label and arm it, this is for
	 * single toggle buttons only and does nothing if already armed.
	 */


	b = findButtonByName( buttons, num_buttons, button_label );
	if ( b != NULL )
	{
		if ( b->armed )	
		{
		   	rc = 0;
		}
		else
		{
			arm( b->w, b );
		}
	}

	return(rc);
}

void
set_arm_indicator( w , b )
Widget w;
Button_item *b;
{
	Button_item *p;
	int	     index;
	int	     first=1;

	if ( b->armed )
	{
		if ( b->toggle_number == 0 )
		{
			disarm( w, b );
		}
	}
	else		/* Arm it, disarm any others if in a radio group  */
	{
	    if ( b->toggle_number == 0 )
	    {
	    	arm( w, b );
	    }
	    else
	    {
	    	p = b - b->toggle_number + 1;   /* Pointer magic */
	    	while( p )
	    	{
	    	    /* Test to see if we run off the end of the 
	    	     * toggle group or off the end of the button array.
	    	     */
    
    		    index = p - Buttons;
    
    		    if ( index >= Button_cnt )
    		    {
        	       DEBUG2( 1, "set_arm_indicator: button(%d) overun(%d)\n",
    		        				index, Button_cnt );
    		        break;
    		    }
    
    		    if ( p->toggle_number == 0 )
    		    {
                        DEBUG1(1,"set_arm_indicator: new toggles @button(%d)\n",
	    	        				index );
	    	        break;
	    	    }
	    	    if ( p->toggle_number == 1 )
	    	    {
	    	    	if ( first )
	    	    	{
	    	    	    first = 0;
	    	    	}
	    	    	else
	    	    	{
	       	    	   DEBUG1(1,"set_arm_indicator: new toggle @but(%d)\n",
	    	        		index );
	    	              break;
	    	    	}
	    	    }

	    	    if ( p->armed )
	    	    {
	    	    	disarm( p->w, p );
	    	    	break;
	    	    }

    		    p++;
    		}
    		arm( w, b );
    	    }
	}
}

void
arm( w , b )
Widget w;
Button_item *b;
{
	Arg wargs[2];
	int n=0;

	if ( w == NULL || b == NULL )
	{
		return;
	}

	if ( b->type == PIXMAP_TYPE )
	{
	    if ( b->arm_pixmap )
		XtSetArg( wargs[n], XmNlabelPixmap, b->arm_pixmap );n++;
	}
	else
	{
	    if ( b->arm_label )
		XtSetArg( wargs[n], XmNlabelString, b->arm_label_str );n++;
	}

	XtSetValues( w, wargs, n );

	/* if ( b->toggle_number != 0 ) */
		toggle_button( w );

	b->armed = 1;

}

void
disarm( w , b )
Widget w;
Button_item *b;
{
	Arg wargs[2];
	int n=0;

	if ( w == NULL || b == NULL )
	{
		return;
	}

	if ( b->type == PIXMAP_TYPE )
	{
	    if ( b->unarm_pixmap )
		XtSetArg( wargs[n], XmNlabelPixmap, b->unarm_pixmap );n++;
	}
	else
	{
	    if ( b->unarm_label )
		XtSetArg( wargs[n], XmNlabelString, b->unarm_label_str );n++;
	}

	XtSetValues( w, wargs, n );

	/* if ( b->toggle_number != 0 ) */
		toggle_button( w );

	b->armed = 0;
}


void
toggle_button( w )	/* Invert button shading  popin<->popout  */
Widget w;
{
	int n;
	Arg wargs[6];
	Pixel	top_color, bottom_color;
	Pixmap  top_shade, bottom_shade;

	n = 0;
 	XtSetArg( wargs[n], XmNbottomShadowColor, &bottom_color ); n++;
 	XtSetArg( wargs[n], XmNtopShadowColor, &top_color ); n++;
 	XtSetArg( wargs[n], XmNbottomShadowPixmap, &bottom_shade ); n++;
 	XtSetArg( wargs[n], XmNtopShadowPixmap, &top_shade ); n++;
	XtGetValues( w, wargs, n );

	n = 0;
 	XtSetArg( wargs[n], XmNbottomShadowColor, top_color ); n++;
 	XtSetArg( wargs[n], XmNtopShadowColor, bottom_color ); n++;
 	XtSetArg( wargs[n], XmNbottomShadowPixmap, top_shade ); n++;
 	XtSetArg( wargs[n], XmNtopShadowPixmap, bottom_shade ); n++;
	XtSetValues( w, wargs, n );
}
