/* File: 	treeps.c
 *
 * Description: Main file for treeps contains the main body of code,
 *		including Global declarations, the main routine which sets up
 *		the Motif widgets, registers event handlers, adds translations
 *		and defines the callbacks. Also includes the routines to load
 *		and display process information.
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1993, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	10/10/93	Created
 *            
 *
 */

#include <stdio.h>		
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <varargs.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef __linux__
#include <sys/mkdev.h>		/* To get major/minor numbers 		 */
#endif
#include <sys/param.h>		/* To get PAGESIZE i.e. system page	 */
#include <dirent.h>		/* To get the /proc directory listing	 */
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#include <Xm/Xm.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/DrawnB.h>
#include <Xm/PushBG.h>
#include <Xm/ScrolledW.h>
#include <Xm/Form.h>
#include <Xm/SeparatoG.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/DrawingA.h>

#include <X11/Xmu/Editres.h>


#include "os_info.h"

#include <Xpsi/Tree.h>	/* Custom Tree widget		   	   */

#include "tree.h"	/* Generic m-ary tree package include file */
#include "list.h"	/* Simple list package			   */
#include "debug.h"	/* Macro based debug package		   */

#include "button_bar.h"
#include "user_group.h" /* User_group display related definitions  */
#include "treeps.h"	/* Some process wide definitions	   */
#include "color_code.h" /* Color coding structures, defines        */
#include "ttydb.h"      /* Database of tty names, major & minor #'s */

#include "machine_info.h"
#include "proc_driver.h"
#include "gei_driver.h"

#include "icons.h"

#include "icons/mono/48/treeps48.icon"/* Bitmap for program icon       */

#include "icons/mono/16/signal"	/* Cursor when signaling 	       */
#include "icons/mono/16/signal_mask"
#include "icons/mono/16/target"	/* Cursor when killing 	               */
#include "icons/mono/16/target_mask"
#include "icons/mono/16/info"	/* Cursor when selection popup details */
#include "icons/mono/16/info_mask"		
#include "icons/mono/16/outline"/* Cursor when selection outlines subtree */
#include "icons/mono/16/outline_mask"
#include "icons/mono/16/hide"	/* Cursor when selection hides subtree */
#include "icons/mono/16/hide_mask"	
#include "icons/mono/16/show"	/* Cursor when selection shows subtree */
#include "icons/mono/16/show_mask"

#ifndef lint
static char *treeps_cpr ="@(#) Copyright(c) 1993-2001, Pulsar Software Inc.\n\t All rights reserved.";
static char *treeps_sccs="@(#) %W% 	Last Delta %G% %U%";
static char *treeps_rcs="$Id$ - $name$";
#endif

int Debug=0;


/* ------------------ Application resource structures  -------------------- */



static XtResource Resources[] = {
	{ "debug_level", "Debug_level", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, debug_level ), XtRString, "0"    },
	{ "short_tic", "Refresh_rate", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, short_tic), XtRString, "1000" },
	{ "long_tic", "Refresh_rate", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, long_tic), XtRString, "5" },
	{ "shrink_delay", "Shrink_delay", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, shrink_delay), XtRString, "3" },
	{ "outline_width", "Outline_width", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, outlineWidth ), XtRString, "4"    },
	{ "outline_height", "Outline_height", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, outlineHeight), XtRString, "8" },
	{ "menu_bar", "Menu_bar", XtRBoolean, sizeof(Boolean), 
	   XtOffset(ApplicationDataPtr, menu_bar), XtRString, "TRUE" },
	{ "button_bar", "Button_bar", XtRBoolean, sizeof(Boolean), 
	   XtOffset(ApplicationDataPtr, button_bar), XtRString, "TRUE" },
	{ "display_pid", "Display_pid", XtRBoolean, sizeof(Boolean), 
	   XtOffset(ApplicationDataPtr, disp_pid), XtRString, "TRUE" },
	{ "display_name", "Display_name", XtRBoolean, sizeof(Boolean), 
	   XtOffset(ApplicationDataPtr, disp_pname), XtRString, "TRUE" },
	{ "show_all", "Show_all", XtRBoolean, sizeof(Boolean), 
	   XtOffset(ApplicationDataPtr, show_all), XtRString, "FALSE" },
	{ "show_daemons", "Show_daemons", XtRBoolean, sizeof(Boolean), 
	   XtOffset(ApplicationDataPtr, show_daemons), XtRString, "TRUE" },
	{ "show_process_groups", "Show_process_groups", XtRBoolean,  		/* Deprecate */
	   sizeof(Boolean), XtOffset(ApplicationDataPtr, show_process_groups), 
	   XtRString, "TRUE" },
	{ "process_attach_mode", "Process_attach_mode", XtRInt, 
	   sizeof(int), XtOffset(ApplicationDataPtr, process_attach_mode), 
#ifdef SMART_PROCESS_ATTACHMENT
	   XtRString, "2" },
#else
	   XtRString, "1" },
#endif
	{ "node_foreground", "Node_foreground", XtRPixel, sizeof(Pixel), 
	   XtOffset(ApplicationDataPtr, node_foreground), XtRString, "White" },
	{ "node_background", "Node_background", XtRPixel, sizeof(Pixel), 
	   XtOffset(ApplicationDataPtr, node_background), XtRString, "Black" },
	{ "selected_node_foreground", "Selected_node_foreground", 
	   XtRPixel, sizeof(Pixel), 
	   XtOffset(ApplicationDataPtr, selected_node_foreground), 
	   XtRString, "Black" },
	{ "selected_node_background", "Selected_node_background", 
	   XtRPixel, sizeof(Pixel), 
	   XtOffset(ApplicationDataPtr, selected_node_background), 
	   XtRString, "White" },
	{ "color_based_on", "Color_based_on", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, color_based_on), 
	   XtRString, "effective_user_id" },
	{ "users_color_list", "Users_color_list", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, users_color_list), 
	   XtRString, "" },
	{ "groups_color_list", "Groups_color_list", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, groups_color_list), 
	   XtRString, "" },
	{ "stats_color_list", "Stats_color_list", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, stats_color_list), 
	   XtRString, "" },
	{ "load_color_map", "Load_color_map", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, load_color_map), 
	   XtRString, "" },
	{ "tcpu_color_map", "Tcpu_color_map", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, tcpu_color_map), 
	   XtRString, "" },
	{ "status_color_map", "Status_color_map", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, status_color_map), 
	   XtRString, "" },
	{ "resident_color_map", "Resident_color_map", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, resident_color_map), 
	   XtRString, "" },
	{ "image_color_map", "Image_color_map", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, image_color_map), 
	   XtRString, "" },
	{ "priority_color_map", "Priority_color_map", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, priority_color_map), 
	   XtRString, "" },
	{ "treeps_lib_root", "Treeps_lib_root", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, treepsLibRoot), 
	   XtRString, DEFAULT_LIB_DIR },
	{ "treeps_user_root", "Treeps_user_root", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, treepsUserRoot), 
	   XtRString, ".userStore/Applications/treeps/current" },
	{ "show_process_tips", "Show_process_tips", XtRBoolean, 
	   sizeof(Boolean), XtOffset(ApplicationDataPtr, showProcessTips), 
	   XtRString, "TRUE" },
	{ "process_tip_font", "Process_tip_font", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, processTipFont), 
	   XtRString, "8x13" },
	{ "process_tip_background", "Process_tip_background", XtRPixel, 
	   sizeof(Pixel), XtOffset(ApplicationDataPtr, processTipBackground), 
	   XtRString, "White" },
	{ "process_tip_foreground", "Process_tip_foreground", XtRPixel, 
	   sizeof(Pixel), XtOffset(ApplicationDataPtr, processTipForeground), 
	   XtRString, "Black" },
	{ "button_bar_icon_size", "Button_bar_icon_size", XtRInt, sizeof(int), 
	   XtOffset(ApplicationDataPtr, buttonBarIconSize), XtRString, "22" },
	{ "button_bar_icon_depth", "Button_bar_icon_depth", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, buttonBarIconDepth), 
	   XtRString, "loColor" },
	{ "button_bar_icon_dir", "Button_bar_icon_dir", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, buttonBarIconDir), 
	   XtRString, "icons" },
	{ "decorate_tree", "Decorate_tree", XtRInt, sizeof(int), 
		XtOffset(ApplicationDataPtr, decorate_tree), XtRString, "2" },
	{ "fame_time", "Fame_time", XtRInt, sizeof(int), 
		XtOffset(ApplicationDataPtr, fame_time), XtRString, "20" },
	{ "systemMap", "SystemMap", XtRString, 
	   sizeof(String), XtOffset(ApplicationDataPtr, systemMap), 
	   XtRString, "/boot/System.map" },
	{ "virtualDesktop", "VirtualDesktop", XtRBoolean, 
	   sizeof(Boolean), XtOffset(ApplicationDataPtr, virtualDesktop), 
	   XtRString, "TRUE" },
	{ "autoSizeMainWindow", "AutoSizeMainWindow", XtRBoolean, 
	   sizeof(Boolean), XtOffset(ApplicationDataPtr, autoSizeMainWindow), 
	   XtRString, "TRUE" },
	{ "fitToScreen", "FitToScreen", XtRBoolean, 
	   sizeof(Boolean), XtOffset(ApplicationDataPtr, fitToScreen), 
	   XtRString, "TRUE" },
	{ "rightOffset", "RightOffset", XtRInt, sizeof(int), 
		XtOffset(ApplicationDataPtr, rightOffset), XtRString, "64" },
	{ "bottomOffset", "BottomOffset", XtRInt, sizeof(int), 
		XtOffset(ApplicationDataPtr, bottomOffset), XtRString, "100" },
	{ "displayLWP", "DisplayLWP", XtRBoolean, 
	        sizeof(Boolean), XtOffset(ApplicationDataPtr, displayLWP), 
	   						XtRString, "FALSE" },
	{ "useExternalDataGathering", "UseExternalDataGathering", XtRBoolean, 
	        sizeof(Boolean), XtOffset(ApplicationDataPtr, useExternalInterface), 
	   						XtRString, "FALSE" },
};


/* ----------------  Command line options structure  ---------------------- */

static XrmOptionDescRec Options[] = {
	{"-x",	 "*debug_level",	XrmoptionSepArg, NULL },
/*
 * The following work but only set some of the font resources that need
 * to be set. Doesn't seem to accept multiple resources to set - sigh
 *
	{"-font","*fontList",		XrmoptionSepArg, NULL },
	{"-fn","*fontList",		XrmoptionSepArg, NULL },
*/
};


/* ------------------------------- Actions -------------------------------- */

static void hide_node();
static void expand_node();
static void signal_node();
static void increase_debug();
static void change_spacing();
static void orientation();
static void connect_style();
static void toggle_pid();
static void outline_subtree();
static void toggle_mem();
static void show_find();
static void show_hidden();
static void toggle_outline();
static void toggle_decorations();
static void show_all_processes();
static void show_user_processes();

static XtActionsRec actionsTable[] = {
	{ "hide",   		hide_node   	} ,
	{ "expand", 		expand_node 	} ,
	{ "signal"  , 		signal_node 	} ,
	{ "inc_debug", 		increase_debug 	} ,
	{ "change_space",	change_spacing 	} ,
	{ "orientation",	orientation 	} ,
	{ "connect_style",	connect_style 	} ,
	{ "toggle_pid",		toggle_pid 	} ,
	{ "show_find",		show_find 	} ,
	{ "outline_subtree",	outline_subtree	} ,
	{ "toggle_mem",		toggle_mem 	} ,
	{ "toggle_outline",	toggle_outline 	} ,
	{ "show_hidden",	show_hidden 	} ,
	{ "show_all_processes",	show_all_processes} ,
	{ "show_user_processes",show_user_processes} ,
	{ "toggle_decorations",	toggle_decorations } ,
};


/* ----------------------------- Translations ----------------------------- */

static char 
treeWidgetTranslations[]=" <Key>+:     change_space( 1, 1) \n\
<Key>-:     change_space(-1,-1) \n\
<Key>Left:  change_space(-1, 0) \n\
<Key>Right: change_space( 1, 0) \n\
<Key>Up:    change_space( 0, 1) \n\
<Key>Down:  change_space( 0,-1) \n\
<Key>l:     orientation(LEFT_TO_RIGHT) \n\
<Key>r:     orientation(RIGHT_TO_LEFT) \n\
<Key>t:     orientation(TOP_TO_BOTTOM) \n\
<Key>b:     orientation(BOTTOM_TO_TOP) \n\
<Key>s:     orientation(STAR_TOPOLOGY) \n\
<Key>d:     connect_style(DIAGONAL_CONNECTION) \n\
<Key>e:     connect_style(TIER_CONNECTION) \n\
<Key>p:     connect_style(PATHWAY_CONNECTION) \n\
<Key>n:     connect_style(NO_CONNECTION) \n\
<Key>k:     toggle_pid() \n\
<Key>m:     toggle_mem() \n\
<Key>f:     show_find() \n\
<Key>h:     show_hidden() \n\
<Key>u:     show_user_processes() \n\
<Key>a:     show_all_processes() \n\
<Key>o:     toggle_outline() \n\
<Key>q:     toggle_decorations() \n\
<Key>x:	    inc_debug()";

static char treeNodeTranslations[] = " <Key>s:	 expand() \n\
<Key>h:	 hide() \n\
<Key>i:	 signal(2) \n\
<Key>o:  outline_subtree() \n\
<Key>k:	 signal(9)";


XtTranslations	 Tree_Trans_table;	/* Compiled translation table	*/
XtTranslations	 Node_Trans_table;	/* Compiled translation table	*/

/* -------------------------  Process tree var's --------------------- */

tnode          *Head=NULL;
Widget		TreeWidget;

extern tnode *splice_loose_subtrees();


/* ----------------  Tree Node info description struct  ---------------- */

struct Node_Info_Description Node_Info_Desc [] = {
	{ PR_PID,	"pid",		"The Unique Process Identifier "},
	{ PR_FNAME,	"name",		"Name of Executable File       "},
	{ PR_PSARGS,	"args",		"Initial characters of arg list"},
	{ PR_UID,	"uid",		"User ID of the Process        "},
	{ PR_GID,	"gid",		"Group ID of the Process       "},
	{ PR_EUID,	"euid",		"Effective User ID of Process  "},
	{ PR_EGID,	"egid",		"Effective Group ID of Process "},
	{ PR_PPID,	"ppid",		"Parent Process ID             "},
	{ PR_PGRP,	"pgrp",		"Group Leader PID              "},
	{ PR_SID,	"sid",		"Session Leader PID            "},
	{ PR_SNAME,	"status",	"Process Status                "},
	{ PR_FLAGS,	"flags",	"Process Flags                 "},
	{ PR_NICE,	"nice",		"Nice Value of Process         "},
	{ PR_SIZE,	"size",		"Image Size of Process         "},
	{ PR_RSSIZE,	"rssize",	"Number of Resident Pages      "},
	{ PR_WCHAN,	"wchan",	"Process waiting on address    "},
	{ PR_START,	"start",	"Time the Process was Started  "},
	{ PR_TIME,	"time",		"Total Process Execution Time  "},
	{ PR_PRI,	"priority",	"Process Priority              "},
	{ PR_CLNAME,	"class",	"Process Scheduling Class      "},
	{ PR_TTYDEV,	"ttydev",	"Controlling Terminal          "},
#ifdef SVR4_MP
	{ PR_NLWP,	"numlwp",	"Number of LWPS                "},
	{ PR_ONPRO,	"onpro",	"On Processor                  "},
#endif
};



/* ----------------  Node info display control struct  ------------------- */

/* The default display list for each node */

lnode  *Global_display_list;   

lnode  *set_default_display_list();

void    popup_display_list();
void    popdown_display_list();
void    select_display_field();

void    popdown_display_listCB();

Widget Display_list_pw=NULL;
Widget Display_list_lw1=NULL;
Widget Display_list_lw2=NULL;
Widget Display_list_lw3=NULL;
Widget Display_list_cw=NULL;

void  (*DisplayListPopdownCB)()=NULL;


#define DYNAMIC_INFO	1
#define INFREQUENT_INFO	2

int	Dynamic_info=0;  /* When set, update process info on long tic */

/* --------------  Node detailed info display functions  ----------------- */

void  popup_procinfo();
void  popdown_procinfo();
void  display_procinfo();
void  toggle_procinfo();
void  update_procinfo();
void  refresh_procinfo();

/* ----------------------  Machine List variables  ----------------------- */

Machine_info  *MachineList; 
Machine_info  *Machine; 	                 /* Ptr To Current Machine */
int	      NumMachinesMonitored=0;

char 	      *Process_Info_Dir=NULL;	 /* Local machine typically /proc   */
int	       Local_external_scanner=0; /* Set to 1 if using external prog */

void child_sig_handler();
void pipe_sig_handler();


/* ----------------------  Process list variables  ----------------------- */

lnode *Process_list=NULL;		/* List of all processes	   */

lnode *Active_process_list=NULL;	/* List of active processes	   */

lnode *Displayed_session_list=NULL;	/* List of all displayed sessions  */


/* ------------------------  Process Hash Table  ------------------------- */

lnode **Process_hash_table;
lnode **PHT;                    		/* shortcut for above */

int    Hash_size=1000;

#define PID_HASH( pid )   ( pid % Hash_size )   /* somewhat simplistic */

int    add_to_hash_table();
int    del_from_hash_table();
lnode *find_in_hash_table();

/* ----------------------  Attachment related stuff  ----------------------- */

lnode *Relocated_process_list=NULL;     /* Processes moved around on tree  */

lnode *Attachment_validator=NULL;       /* Checks for valid process attachments */

lnode *push_attachment();
lnode *add_work_task();
lnode *remove_work_task();



/* ----------------------  Menu setting variables  ----------------------- */

int			Monitor_processes=TRUE; /* When false we stop updates */
TreeOrientation		Current_orientation=LEFT_TO_RIGHT;
ConnectionStyle		Current_connection=TIER_CONNECTION;
ParentPlacement		Current_placement=PARENT_CENTERED;
int 			Show_process_groups=True;		/* deprecate */
int 			Process_attach_mode=GROUP_ATTACH;
int 			User_view=VIEW_LOGGED_IN_USER;
int 			Group_view=VIEW_IGNORE_GROUPS;
int 			Show_daemons=True;
int 			Color_based_on=COLOR_BY_EFFECTIVE_USER_ID;
int			Process_action=ACTION_PROCESS_DETAILS;
int			Current_signal=2;
int			Show_descendants=SHOW_ALL_DESCENDANTS;
/* int			Show_descendants=FILTER_DESCENDANTS; */

int			Dynamic_colors=0;  /* Set when color by stats    */

/* ----------------  Default Node display variables  ------------------- */

int Display_name=TRUE;
int Display_pid=TRUE;
int Show_all;		/* Show all processes, on start up */
int Show_Processor=0;	/* Show Processor in node*/


/* int Decorated_tree=DECORATE_ALL; */
int Decorated_tree=DECORATE_DISTINGUISHED;

int Decoration[8];		/* Initialized in setup_decoration_defaults */
int Decoration_count=0;		/* Ditto */
int Decorations_include_load=0;

GC  DecorationGC=0;

/* int Decoration_size=6; */
int Decoration_size=8;

Dimension Decoration_top_size=12;

Dimension Decoration_left_size=5;

int Decoration_top_color=COLOR_BY_REAL_USER_ID;
int Decoration_left_color=COLOR_BY_REAL_GROUP_ID;

/* int Decoration_fame_time=278;  */ /* ~15 Min. of Fame */

int Decoration_fame_time=20;  /* ~1 Min. of Fame - It's a faster world */


int Decoration_spacing=3;

int Decoration_border_dynamic=1;

#define CIRCLE_DECORATION 	1
#define BAR_DECORATION		2

int Decoration_type=CIRCLE_DECORATION;
/* int Decoration_type=BAR_DECORATION; */

int Decoration_borders=0;

void  top_decoration_exposures();
void  left_leader_exposures();
void  decoration_resize();
void  decoration_label_resize();
void  setup_decoration_defaults();
void  redraw_node_decorations();
void  status_decoration();
void  load_decoration();
void  priority_decoration();
void  render_decoration();
void  expand_decoration_left_panel();
void  expand_decoration_top_panel();
void  collapse_decoration_left_panel();
void  collapse_decoration_top_panel();

int Leader_bars=1;

int Leader_bar_size=1;
int Leader_bar_spacing=2;
int Leader_bar_borders=0;

#define STATIC_LEADER_COLOR		0
#define DYNAMIC_LEADER_COLOR		1

int Leader_bar_colors=STATIC_LEADER_COLOR;

/* int Leader_bar_colors=DYNAMIC_LEADER_COLOR; */

void redraw_leader_bars();
void group_leader_bar();
void session_leader_bar();

tnode *redisplay_node();
tnode *outline_node();

int New_display_criteria=0;

int User_Session_Root=1;  /* Used during smart attachments */

/* -------------------  User List variables  -------------------- */

User_group_info  *Users=NULL;

void user_group_process_cb(); 

int Trusted_user=TRUSTED_USER;


/* The following is used to build a list of the login users controlling
 * terminals. This is sometimes used to determine if we can signal a process,
 * i.e. if the process has a different uid, but was started by the user(suid,
 * su ... ) then we allow the signal. 
 *
 */

User_tty_struct *User_ttys=NULL;

User_tty_struct *add_dev_to_users_devs();
User_tty_struct *free_user_ttys_list();


/* The following are used to keep active user process counts, so we can track
 * active user/group id's for the user_group display
 */

lnode *dec_list_count();	
lnode *inc_list_count();
lnode *zero_free_list();

int	_drop_list_entry;
int	_add_list_entry;

/* -------------------  Color List variables  -------------------- */

Object_color  *User_colors=NULL;
Object_color  *Group_colors=NULL;
Object_color  *Stat_colors=NULL;


Pixel   Node_foreground;	/* Default process node fg color    */
Pixel	Node_background;	/* Default process node bg color    */
Pixel   Selected_node_foreground; /* selected node fg color         */
Pixel	Selected_node_background; /* selected node bg color         */

/* -------------------  Dynamic update variables  -------------------- */


/* The following variables control the pulse of the internal process 
 * refresh engine. Each short tic the program checks for:
 *
 *	New processes 		- Added to active process list
 *				  Added to process tree if should be visible
 *
 *	Missing processes	- Removed from process list
 *				  Removed from displayed process tree if visible
 *
 *	If No add's or deletes	- checks for change of status of processes in
 *				  active list. Change of status affecting the
 *				  display could be:
 *
 *					Change of process name  i.e. exec's
 *					Change of ppid  -  reparenting
 *					Change of real user/group  id - 
 *							effects color
 *							and perhaps visibility
 *					Change of effective user/group  id - 
 *							effects color
 *					Change of controlling terminal
 *						     ( Foreground <-> daemon )
 *
 *				   Updates displayed dynamic variables,
 *				   counts, colors for processes in active list
 *
 * Each long tic the program does the following:
 *
 *	Update all process info
 *
 *	Check's all processes for change of states
 *
 *	Stale active processes 	- Remove from active list
 *
 *	Newly active process	- Moved to active list
 *
 *				  Updates displayed variables 
 */
 
int Short_tic=1000;		/* Default every second  		*/
int Long_tic=5;			/* Every five short tics		*/

int Visible_processes_active=0; /* When set all visible processes are active */


int Tree_widget_shrink_delay = 3;   /* Number of refreshes before shrink */
int Last_shrink=0;    
int Pending_shrink=0;

Boolean Doing_auto_resize;
Boolean FitToScreen;
Boolean DisplayLWP;

/* The following are used to limit the size of the automatic window growth.
 * If the window get's larger than this then scroll bars will be enabled,
 * values of -1 mean the window can grow arbitrarily large, this is for
 * those who want such a thing(perhaps on virtual desktops).
 */

int	Max_window_x=-1;
int	Max_window_y=-1;
int	Min_window_x=-1;
int	Min_window_y=-1;

/* If fit to screen is enabled the following are used to limit the max
 * size of the main window */

int ActualScreenWidth=800;
int ActualScreenHeight=600;
int RightOffset=10;
int BottomOffset=60;


void	treeResized();

/* Cursors for the various modes */

Cursor Signal_cursor;
Cursor Kill_cursor;
Cursor Info_cursor;
Cursor Busy_cursor;
Cursor Outline_cursor;
Cursor Hide_cursor;
Cursor Show_cursor;

int Selected_node=-1;

void   clock_event();
lnode *update_process_list();
tnode *update_process_tree();
tnode *recalc_process_tree();

Machine_info *construct_machine_monitor();

/* ------------------ Process List Routines --------------- */

lnode *delete_processes_from_plist();
lnode *add_processes_to_plist();
lnode *construct_process_list_node();
void   destroy_process_list_node();

/* ------------------ Process Tree Routines --------------- */

tnode *delete_processes_from_ptree();
tnode *delete_process_from_ptree();
tnode *add_processes_to_ptree();
tnode *add_process_to_ptree();
tnode *construct_process_tree_node();
tnode *destroy_process_tree_node();
int    attachToParent();
int    attachToNode();

/* ------------------- Active List Routines --------------- */

lnode *update_active_list();
lnode *add_processes_to_active();
lnode *delete_processes_from_active();
lnode *reactivate_processes();
lnode *remove_stale_processes();
							

sysProcInfo *construct_sys_proc_info();

tnode *display_proc_info();
tnode *update_node_info();
Widget create_proc_widget();

tnode *hide_node_widget();
tnode *display_node_widget();
tnode *match_widget();
tnode *reparent_process();
tnode *orphan_process();

#ifdef SVR4_MP
tnode *show_LWP();
tnode *hide_LWP();
#endif


void select_node();
void select_tree();
void select_info();
#ifdef SVR4_MP
void select_lwp();
#endif
void do_detail_rate();
void do_node_rate();
void do_display_rate();

Boolean do_convert_selection();
void    do_lose_selection();

Widget create_button();
Widget create_label();
Widget create_ltext();

extern Widget setup_menu();
extern Widget Menu_widget;	/* Set by above */
Widget Main_menu;		

extern Widget setup_node_popup_menu();
extern Widget setup_tree_popup_menu();
Widget Node_popup_menu;
Widget Tree_popup_menu;

int      Popup_on_pid=-1;

tnode   *match_window();

char *get_connect_menu_str();
char *get_orient_menu_str();

void set_cursor();
void show_display_list();
void set_user_list();

extern void do_help();
extern tnode *outline_all();

extern Widget setup_button_bar();

Button_item  *Main_Buttons=NULL;
int  	      Main_Button_cnt=0;
Widget	      Button_bar_w=0;

char *find_dev_name();

Widget 		Top_level;
Window 		Top_level_window;
XSizeHints	Top_size_hints;
unsigned int 	Display_width;
unsigned int 	Display_height;

Widget 		TreeScrollWidget;

void preTreeWindowOffsets();

struct dimensionalOffsets {
	int	top;
	int	left;
	int	bottom;
	int	right;
};

Dimension OutlineWidth=4;
Dimension OutlineHeight=4;

char *LibRootDir=NULL;
char *UserRootDir=NULL;

int Record_memuse=0;



timestruc_t  TimeZero;   /* Used when running under linux */

size_t PageSize;
int    MaxSig=31;

static XtAppContext MainContext;

/* Here we include a platform specific resource file which defines the
 * resources used when no resource file is found.
 */

#include "fallbackResources"

#ifdef LINUX
extern Widget linuxDetailsPanel();
#endif

#ifdef SVR4_MP
extern Widget detailsPanel();
void    do_draw_lwp_shape();
#endif

int 	VirtualDesktop=True;
int	DoingKdeNameMagic=0;

Boolean AutoSizeMainWindow=True;


ApplicationData  AppData;

Widget Top_separator;
Widget ClipWindow=(Widget)0;

extern Pixmap getDeepIconByName();

extern int loadPixmapToSet();

int InClockEvent=0;

Widget MainFormFW;

XtIntervalId To_id=0;   /* Timeout id */

char **Environment;


void my_error_handler();
void my_warning_handler();

int Priority_Normal_Min;  /* Defines range of priorities considered normal. */
int Priority_Normal_Max; /* Those that are not normal are considered distiguished */

Widget w;
main(argc, argv, envp)
int argc;
char *argv[];
char *envp[];
{
	Widget 		 parent, sw, tree, attach_to;
	int 		 rc, n;
	Arg 		 wargs[16];
	Pixmap		 icon_pixmap;
	char		 path[256];
	Display 	*display=NULL;
	int 		 screen;
	Screen 		*screen_ptr=NULL;
	char		*osRelease, *osType;
	char		*hostname;
	char		*fontName=NULL;
	extern char     *optarg;
        extern int       optind, opterr, optopt;
        int              i, c;
        int              errflg=0;
	XrmDatabase	 xrm_db;
	XrmDatabase	 new_xrm_db;
	char		 res_str[2048];
	char		 *type_str=NULL;
	XrmValue	 xrm_value;
	char		 *xrm_str;


	Environment = envp;

	setvbuf( stderr, NULL, _IOLBF, 0 );  /* Unbuffered output */
	setvbuf( stdout, NULL, _IOLBF, 0 );

        create_hash_table();

	osType     = os_get_sysname();
	osRelease  = os_get_release();
	hostname   = os_get_nodename();

	MaxSig   = 32;   /* Typical number on most unices */

#ifdef _NSIG
	MaxSig   = _NSIG;   /* Override if defined */
#endif

#ifdef MAXSIG
	MaxSig   = MAXSIG;  /* Override if defined */
#endif


#ifdef LINUX
	PageSize = getpagesize();
	Priority_Normal_Min = 0;
	Priority_Normal_Max = 9;

	if ( osMajorReleaseNumber >= 2 )
	{
	    if ( osMinorReleaseNumber >= 4 )
	    {
	       Priority_Normal_Min = 15;
	       Priority_Normal_Max = 19;
	    }
	}

#else
	Priority_Normal_Min = 10;
	Priority_Normal_Max = 90;

	PageSize = PAGESIZE;
#endif

	DEBUG2( 1, "MaxSig(%d), PageSize(%d)\n", MaxSig, PageSize );

	TimeZero.tv_sec  = 0;
	TimeZero.tv_nsec = 0;

	/* Must do this before AppInitialize eats and ignores the font specifier */

	for ( i = 1 ; i < argc; i++ )
	{
	    if ( strcmp( argv[i], "-fn" ) == 0 )
		fontName = argv[i+1] ;
	    else if ( strcmp( argv[i], "-font" ) == 0 )
		fontName = argv[i+1] ;
	    else if ( strcmp( argv[i], "-f" ) == 0 )
		fontName = argv[i+1] ;
	}

	/* Note -fn and -font arguments are checked by the Xt lib to verify
	 * proper font names. A -f does not do this!
	 */

	/* Should now(Motif 2.0+?) be using XtOpenApplication */

	XtSetErrorHandler( my_error_handler );
	XtSetWarningHandler( my_warning_handler );

	Top_level = XtAppInitialize( &MainContext, "Treeps", Options, 
						    XtNumber(Options), 
						    &argc, argv,
						    Fallback_resources,
						    NULL, 0); 

/* To use editres to debug widget resources ...
	XtAddEventHandler( Top_level, NoEventMask, True, _XEditResCheckMessages,(XtPointer)NULL );

*/
	display    = XtDisplay( Top_level );

	XtSetErrorHandler( (XtErrorHandler) 0 );
	XtSetWarningHandler( (XtErrorHandler) 0 );

        icon_pixmap = getDeepIconByName( Top_level, "programIcon",
                          treeps48_bits, 48 );


 	XtVaSetValues( Top_level, XmNiconPixmap, icon_pixmap, NULL );

	/* Load application resources */

	XtGetApplicationResources( Top_level, &AppData, Resources,
				             XtNumber(Resources), 
					     NULL, 0 );


	/*  We process any additoinal args to generate error message if there are any! */

	opterr=0;
        while ( ( c = getopt( argc, argv,"")) != EOF )
        {
                switch ( c )
                {
                        default: errflg++;
                                 break;
                }
        }

        if ( errflg )
        {
          fprintf(stderr, "\nUnexpected command line argument: (-%c) !!!\n", (char) optopt );
                
          fprintf(stderr, "\nusage: %s\n",
                "treeps  [-f fontname] [-x debug level] [ toolkitoptions ]\n");
	  exit( 1 );
	}

	/*  If we want to explicitly get a resource (Not quite right)

	xrm_db     = XrmGetDatabase( display );
	XrmGetResource( xrm_db, "Treeps.fontList", "Treeps.FontList", &type_str, &xrm_value );
	if ( *type_str )
	{
	    if ( strcmp( type_str, "String" ) == 0 )
	    {
		xrm_str = (char *) &xrm_value;
		printf( "xrm_db: font = (%s) (%d): ", type_str, xrm_value );
		for ( i = 0; i < 10 ; i++ )
		    printf("%c(%x) ", xrm_str[i], (int) xrm_str[i] );

		printf( "\n" );
	    }	
	    else
		printf( "xrm_db: font = (%s) (%d)\n", type_str, xrm_value );
	}
	*/



	if ( fontName )
	{
	    DEBUG1(1, "FontName(%s)\n", fontName );

	    if ( strlen( fontName ) > 200 )
	    {
	       fprintf( stderr, "Nice try\n" );
	       exit( 2 );
	    }

	    widget_tip_set_font( fontName );
	    process_tip_set_font( fontName );

	    sprintf( res_str, "%s %s\n%s %s\n%s %s\n", 
			"Treeps*fontList: ", fontName ,
			"userGroupTree*fontList: ", fontName ,
			"processColorMap*fontList: ", fontName 
			);

	    xrm_db     = XrmGetDatabase( display );
	    new_xrm_db = XrmGetStringDatabase( res_str );

	    XrmMergeDatabases( new_xrm_db, &xrm_db );
	}

	/* Register the new actions */

	XtAppAddActions( MainContext, actionsTable, XtNumber(actionsTable) );

	/* Internal setting of translations broke a while back, use resource file */

	Tree_Trans_table = XtParseTranslationTable( treeWidgetTranslations );
	Node_Trans_table = XtParseTranslationTable( treeNodeTranslations );

	Debug = AppData.debug_level;   /* Set the debug packages debug level */


	/* Create a form to hold the menu and scrolled window  */

	MainFormFW = XtCreateManagedWidget( "mainForm", xmFormWidgetClass,
							Top_level, NULL, 0 );
	parent    = MainFormFW;
	attach_to = NULL;

	if ( AppData.menu_bar )		/* User wants a menu bar */
	{
		attach_to = setup_menu( parent );
		Main_menu = Menu_widget;
	}

	LibRootDir		 = AppData.treepsLibRoot;
	UserRootDir		 = AppData.treepsUserRoot;


	if ( AppData.button_bar )	/* User wants a button bar */
	{
		sprintf( path, "%s/%s", LibRootDir, AppData.buttonBarIconDir );

		setIconRoot( path );

		rc = setButtonParameters( AppData.buttonBarIconSize,
						AppData.buttonBarIconDepth );
		if ( rc == -1 )
			exit( 1 );	/* eech no error message !!!! */

  		DEBUG0(1, "setup button bar ");
		Button_bar_w    = setup_button_bar( parent, attach_to );

		attach_to       = Button_bar_w;

		Main_Buttons    = Buttons;
		Main_Button_cnt = Button_cnt;
	}


	n = 0;
	if ( attach_to )
	{
  		DEBUG0(1, "attach to buttonbar/menubar ");
		XtSetArg( wargs[n], XmNtopAttachment,XmATTACH_WIDGET);n++;
		XtSetArg( wargs[n], XmNtopWidget, attach_to); n++;
		XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	}
	else
	{
  		DEBUG0(1, "attach to form ");
		XtSetArg( wargs[n], XmNtopAttachment,XmATTACH_FORM);n++;
	}

	/* Create a scrolled window to hold the process tree widget */

	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;

	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNshadowThickness, 2); n++;

	sw = XtCreateManagedWidget( "scrollWindow", xmScrolledWindowWidgetClass,
						parent, wargs, n );

	n = 0;
	XtSetArg( wargs[n], XmNclipWindow, &ClipWindow); n++;
	XtGetValues( sw, wargs, n );

	/* clipDetails(); */

	TreeScrollWidget = sw;

	/* Create the tree widget */

  	DEBUG0(1, "Create Tree ");

	n = 0;
	tree = XtCreateManagedWidget( "tree", xpsiTreeWidgetClass, sw, wargs,n);
	TreeWidget = tree;

	XtAddCallback( tree, XtNtreeResizeCallback, treeResized, (XtPointer)0 );

	Node_popup_menu = setup_node_popup_menu( tree );
	Tree_popup_menu = setup_tree_popup_menu( tree );


  	DEBUG0(1, "Add Event handler, then translations");

	XtAddEventHandler( tree, ButtonPressMask, FALSE, select_tree, Head );

	/* perhaps this should be the work area of the scroll "widget" */

	XtAugmentTranslations( tree, Tree_Trans_table );

	Short_tic 		 = AppData.short_tic;
	Long_tic 		 = AppData.long_tic;

	Display_name 		 = AppData.disp_pname;
	Display_pid  		 = AppData.disp_pid;
	Tree_widget_shrink_delay = AppData.shrink_delay;
	Show_process_groups 	 = AppData.show_process_groups;		/* Deprecate */
	Process_attach_mode 	 = AppData.process_attach_mode;
	Show_daemons		 = AppData.show_daemons;
	Show_all		 = AppData.show_all;
	Node_foreground		 = AppData.node_foreground;
	Node_background		 = AppData.node_background;

	Selected_node_foreground = AppData.selected_node_foreground;
	Selected_node_background = AppData.selected_node_background;

	OutlineWidth  		 = AppData.outlineWidth;
	OutlineHeight 		 = AppData.outlineHeight;

	Decorated_tree		 = AppData.decorate_tree;
	Decoration_fame_time 	 = AppData.fame_time;

	VirtualDesktop 		 = AppData.virtualDesktop;

	AutoSizeMainWindow 	 = AppData.autoSizeMainWindow;

	FitToScreen 	 	 = AppData.fitToScreen;
	RightOffset 	 	 = AppData.rightOffset;
	BottomOffset 	 	 = AppData.bottomOffset;

	DisplayLWP 	 	 = AppData.displayLWP;

	Local_external_scanner   = AppData.useExternalInterface;

	if ( fontName )
	    setup_process_tips( 100, 10000, fontName, 
	    				AppData.processTipForeground, 
	    				AppData.processTipBackground );
	else
	    setup_process_tips( 100, 10000, AppData.processTipFont, 
	    				AppData.processTipForeground, 
	    				AppData.processTipBackground );

	if ( Show_all )
		User_view=VIEW_ALL_USERS;

	Color_based_on		 = cvt_cbo_str(AppData.color_based_on);

	n = 0;
	XtSetArg( wargs[n], XtNresizePolicy, &Doing_auto_resize ); n++;
	XtGetValues( TreeWidget, wargs, n );



	Global_display_list = set_default_display_list( &AppData );

	Users = setup_user_group( Top_level, user_group_process_cb ); 

	setup_color_maps( &AppData ); 

	/* Set menu and buttons based on X resources or program options */

	set_start_up_menu_toggles();

	setup_decoration_defaults();

	XtRealizeWidget( Top_level );

	Process_Info_Dir = PROCESS_INFO_DIRECTORY;

	/* fire up scanner(s) - for now just one */

	if ( Local_external_scanner )
	{
	    printf("Use external scanner\n" );
	    Machine = construct_machine_monitor( osType, osRelease, hostname, 		
	    						EXTERNAL_POLL_SCANNER );
	}
	else
	{
	    signal( SIGCHLD, child_sig_handler );
            signal( SIGPIPE, pipe_sig_handler );
	    Machine = construct_machine_monitor( osType, osRelease, hostname, 
	    						INTERNAL_POLL_SCANNER );
	}

	if ( Machine )
	{
		MachineList = Machine;
	}
	else
	{
	    fprintf( stderr, "Could not initialize properly - aborting\n");
	    exit( 3 );
	}

	clock_event( tree, NULL );

	/* Get Display Width, height */

	display    = XtDisplay( Top_level );
        screen     = DefaultScreen( display );
        screen_ptr = ScreenOfDisplay( display, screen );

	ActualScreenWidth  = WidthOfScreen( screen_ptr );
	ActualScreenHeight = HeightOfScreen( screen_ptr );

	display_title();

	set_cursor(); 

	n = 0;
	XtSetArg( wargs[n], XtNmaxWidth, &Max_window_x ); n++;
	XtSetArg( wargs[n], XtNmaxHeight, &Max_window_y ); n++;
	XtSetArg( wargs[n], XtNminWidth, &Min_window_x ); n++;
	XtSetArg( wargs[n], XtNminHeight, &Min_window_y ); n++;
	XtGetValues( Top_level, wargs, n );

	DEBUG4(1, "Max_window_x(%d),y(%d) Min x(%d),y(%d)\n", 
					Max_window_x, Max_window_y,
					Min_window_x, Min_window_y );


	XtAppMainLoop( MainContext );
}

void
my_error_handler( message )
String message;
{
	fprintf( stderr, "treeps - Xt error: %s\n", message );
	exit( 1 );
}

void
my_warning_handler( message )
String message;
{
	fprintf( stderr, "\ntreeps - Xt warning: %s\n\n", message );

        fprintf(stderr, "\nusage: %s\n",
                "treeps  [-f fontname] [-x debug level] [ toolkitoptions ]\n");
	exit( 1 ) ;
}

create_hash_table()
{
    int i;

    Process_hash_table = (lnode **) malloc( Hash_size * sizeof(lnode *) );
    if ( Process_hash_table == NULL )
    {
	fprintf( stderr, "ERROR! Failed to create process hash table!\n" );
	exit( 2 );
    }

    for ( i = 0; i < Hash_size ; i++ )
	Process_hash_table[i] = NULL;

    PHT = Process_hash_table;
}

int    
add_to_hash_table( lptr )
lnode *lptr;
{
	int i;
	lnode *bucket_list;

	if ( lptr == NULL )
	{
		DEBUG0(1, "Trying to add null lptr to pid hash table \n");
		return -1;
	}

	i = PID_HASH( lptr->key );

	PHT[i] = push_lnode( PHT[i], lptr->key, lptr );
}

int    
del_from_hash_table( pid )
int pid;
{
	int	i;

	i = PID_HASH( pid );

	PHT[i] = remove_lnode( PHT[i], pid );
}

lnode *
find_in_hash_table( pid )
int pid;
{
	int i;
	lnode *l;

	i = PID_HASH( pid );

	l = find_lnode( PHT[i], pid );

	if ( l ) return ( l->data );

	return NULL;
}

/* some experimental code to check switching views in the scroll window 

void
setNewView( )
{
	int i, n;
	Arg wargs[20];
	Widget done, newWidgetID;
	static int showTree=1;


	if ( showTree )
	{
		XtUnmanageChild( TreeWidget );
		showTree = 0;
	}
	else
	{
		XtManageChild( TreeWidget );
		showTree = 1;
	}

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 50); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 1); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 4); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNscrolledWindowChildType , XmWORK_AREA); n++;

	done = XtCreateManagedWidget("Done", xmPushButtonWidgetClass, 
						TreeScrollWidget, wargs, n );

	newWidgetID = done;

	n = 0;
	XtSetArg( wargs[n], XmNworkWindow, newWidgetID ); n++;
	XtSetArg( wargs[n], XmNscrolledWindowChildType , XmWORK_AREA); n++;
	XtSetValues( TreeScrollWidget, wargs, n );
}

*/

int
refresh_main_window()
{
	int n;
	Arg wargs[20];
	Widget attach_to=0;
	Widget tree, sw;
	Widget parent;

	parent = MainFormFW;

	InClockEvent = 1; 	/* Disable polling processing */

	XtRemoveTimeOut( To_id );  /* Remove old timer */

        startBatchUpdate();
	post_walk_tree( Head, hide_node_widget, 0 );

	XtDestroyWidget( TreeWidget );
	XtDestroyWidget( TreeScrollWidget );


	/* Recreate scrolling window */

	if ( AppData.menu_bar )
		attach_to = Menu_widget;

	if ( AppData.button_bar )
		attach_to = Button_bar_w;


	n = 0;
	if ( attach_to )
	{
  		DEBUG0(1, "attach to buttonbar/menubar ");
		XtSetArg( wargs[n], XmNtopAttachment,XmATTACH_WIDGET);n++;
		XtSetArg( wargs[n], XmNtopWidget, attach_to); n++;
	}
	else
	{
  		DEBUG0(1, "attach to form ");
		XtSetArg( wargs[n], XmNtopAttachment,XmATTACH_FORM);n++;
	}

	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;

	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNshadowThickness, 2); n++;

	sw = XtCreateManagedWidget( "scrollWindow", xmScrolledWindowWidgetClass,
						parent, wargs, n );


	TreeScrollWidget = sw;

	/* Re-create tree widget */

	n = 0;
	tree = XtCreateManagedWidget( "tree", xpsiTreeWidgetClass, sw, wargs,n);
	TreeWidget = tree;

/* Need to destroy these above ?? */

	Node_popup_menu = setup_node_popup_menu( tree );
	Tree_popup_menu = setup_tree_popup_menu( tree );

	XtAddCallback( tree, XtNtreeResizeCallback, treeResized, (XtPointer)0 );

	XtAddEventHandler( tree, ButtonPressMask, FALSE, select_tree, Head );

	XtAugmentTranslations( tree, Tree_Trans_table );


	/* Show processes */

        startBatchUpdate();
	walk_tree( Head, display_node_widget, 0 );
        finishBatchUpdate();

	InClockEvent = 0; 	/* Ensable polling processing */

	/* Reinstall cursor */
	set_cursor(); 

	clock_event( tree, NULL );

	/* 
	XtUnmanageChild( TreeScrollWidget );
	XtManageChild( TreeScrollWidget );
	*/
}

Pixmap
getProgramPixmap( w )
Widget w;
{
	Pixmap  icon_pixmap=-1;
	Display *display=NULL;
	int     screen;
	Screen *screen_ptr=NULL;
	int     numPlanes;
	int	rc, height, width;

	display    = XtDisplay( w );
        screen     = DefaultScreen( display );
        screen_ptr = ScreenOfDisplay( display, screen );

	numPlanes = PlanesOfScreen( screen_ptr );

	if ( numPlanes >= 16 )
	{
	    rc = fetchIcon( "programIcon", w, HIGH_COLOR, 48, 
	    			&icon_pixmap, &width, &height );
	    if ( rc == -1 )
	    	icon_pixmap = -1;
	}
	else if ( numPlanes >= 8 )
	{
                          treeps48_bits,
	    rc = fetchIcon( "programIcon", w, LOW_COLOR, 48, 
	    			&icon_pixmap, &width, &height );
	    if ( rc == -1 )
	    	icon_pixmap = -1;
	}

	if ( icon_pixmap == -1 )
	{
              icon_pixmap = XCreateBitmapFromData( XtDisplay(Top_level),
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      treeps48_bits,
                          BITMAP_DIMENSION_CAST treeps48_width,
                          BITMAP_DIMENSION_CAST treeps48_height );
	}

	return( icon_pixmap );
}



clipDetails()
{
	int n;
	Position x, y;
	Dimension width, height;
	Arg wargs[9];

	n = 0;
	XtSetArg( wargs[n], XmNx, &x); n++;
	XtSetArg( wargs[n], XmNy, &y); n++;
	XtSetArg( wargs[n], XmNwidth, &width); n++;
	XtSetArg( wargs[n], XmNheight, &height); n++;
	XtGetValues( ClipWindow, wargs, n );

printf( "Clip window x(%d) y(%d) width(%d) height(%d)\n", x,y,width, height );
}


void
treeResized( w, data, p )
Widget w;
int    data;
xpsiTreeCallbackStruct *p;
{
	Dimension width, height, border;
	Dimension main_window_width, main_window_height;
	Position  x, y;
	int delta_width, delta_height;
	Dimension new_width, new_height;
	int change_size=0;
	int change_x=0;
	int change_y=0;
	int n;
	Arg wargs[6];
	struct dimensionalOffsets offset;

	DEBUG4( 6, "treeResized from h(%d) v(%d) TO h(%d) v(%d):\n", 
					p->last_width, p->last_height,
					p->width, p->height );

	if ( AutoSizeMainWindow == False )
		return;


	width  = p->width;
	height = p->height;
	border = 0;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &main_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &main_window_height ); n++;
	XtSetArg( wargs[n], XtNx, &x  ); n++;
	XtSetArg( wargs[n], XtNy, &y ); n++;
	XtGetValues( Top_level, wargs, n );

	delta_width  = (int) p->width  - (int) p->last_width;
	delta_height = (int) p->height - (int) p->last_height;

	DEBUG4(6,"treeResized Main Window x(%d),y(%d) Width(%d) height(%d) \n", 
					x, y,
					main_window_width,
					main_window_height );


	DEBUG2( 6, "treeResized Delta Width(%d) height(%d) \n", 
					delta_width, delta_height );

	preTreeWindowOffsets( &offset );

	new_width = p->width + offset.left + offset.right;

	change_size++;
	if ( delta_width > 0 )
	{
	    if ( Max_window_x != -1 )
	    {
	        if ( main_window_width == Max_window_x ) /* Already max */
	        {
		    new_width = Max_window_x;  /* In case height changes */
		    change_size--;
	        }
	    }
	}
	else if ( delta_width < 0 )
	{
	    if ( Last_shrink < Tree_widget_shrink_delay )
	    {
		    change_size--;
		    new_width = main_window_width;
	    }
	    else if ( Min_window_x != -1 )
	    {
	        if ( main_window_width == Min_window_x ) /* Already min */
	        {
		    new_width = Min_window_x;  /* In case height changes */
		    change_size--;
	        }
	    }
	}
	else
	{
	    change_size = 0;
	}

	if ( Max_window_x != -1 )
	{
	    if ( new_width > Max_window_x )
		new_width = Max_window_x;
	}
	if ( Min_window_x != -1 )
	{
	    if ( new_width < Min_window_x )
		new_width = Min_window_x;
	}

	if ( FitToScreen )
	{
	    if ( ( x + new_width ) > ( ActualScreenWidth - RightOffset) )
	    	new_width = ( ActualScreenWidth - x ) - RightOffset;
	}


	new_height = p->height + offset.top + offset.bottom;


	change_size++;

	if ( delta_height > 0 )
	{
	    if ( Max_window_y != -1 )
	    {
	        if ( main_window_height == Max_window_y )
	        {
		    new_height = Max_window_y;
		    change_size--;
	        }
	    }
	}
	else if ( delta_height < 0 )
	{
	    if ( Last_shrink < Tree_widget_shrink_delay )
	    {
		    new_height = main_window_height;
		    change_size--;
	    }
	    else if ( Min_window_y != -1 )
	    {
	        if ( main_window_height == Min_window_y )
	        {
		    new_height = Min_window_y;
		    change_size--;
	        }
	    }
	}
	else
	{
	    change_size--;
	}

	if ( Max_window_y != -1 )
	{
	    if ( new_height > Max_window_y )
		new_height = Max_window_y;
	}
	if ( Min_window_y != -1 )
	{
	    if ( new_height < Min_window_y )
		new_height = Min_window_y;
	}

	if ( FitToScreen )
	{
	    if ( ( y + new_height ) > ( ActualScreenHeight - BottomOffset ) )
	    	new_height = ( ActualScreenHeight - y ) - BottomOffset;
	}


	if ( change_size )
	{
		DEBUG2(8, "Change main window size to h(%d) v(%d)\n", 
						new_width, new_height );

		XtResizeWidget( Top_level, new_width, new_height, border );

		/* clipDetails(); */

	}
}

void
preTreeWindowOffsets( offset )
struct dimensionalOffsets *offset;
{
	Position main_window_x, main_window_y;
	Position clip_window_x, clip_window_y;
	Position mx, my, cx, cy;
	Widget   clip_w;     	    /* Scrolled Windows clip/client widget */
	Dimension main_window_width, main_window_height;
	Dimension clip_window_width, clip_window_height;
	Position mlr_x, mlr_y, clr_x, clr_y;
	int n;
	Arg wargs[6];

	n = 0;
	XtSetArg( wargs[n], XtNx, &main_window_x  ); n++;
	XtSetArg( wargs[n], XtNy, &main_window_y ); n++;
	XtGetValues( Top_level, wargs, n );

	XtTranslateCoords(Top_level, main_window_x, main_window_y, &mx, &my);
	

	n = 0;
	XtSetArg( wargs[n], XmNclipWindow, &clip_w ); n++;
	XtGetValues( TreeScrollWidget, wargs, n );

	n = 0;
	XtSetArg( wargs[n], XtNx, &clip_window_x  ); n++;
	XtSetArg( wargs[n], XtNy, &clip_window_y ); n++;
	XtGetValues( clip_w, wargs, n );

	XtTranslateCoords( clip_w , clip_window_x, clip_window_y, &cx, &cy);


	offset->top    = cy - my;
	offset->left   = cx - mx;

	/* Now get bottom right offsets */

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &main_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &main_window_height ); n++;
	XtGetValues( Top_level, wargs, n );

	mlr_x = mx + main_window_width;
	mlr_y = my + main_window_height;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &clip_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &clip_window_height ); n++;
	XtGetValues( clip_w, wargs, n );

	clr_x = cx + clip_window_width;
	clr_y = cy + clip_window_height;

	offset->bottom  = mlr_y - clr_y;
	offset->right   = mlr_x - clr_x;


	DEBUG4(7, "Offsets top(%d) left(%d) bottom(%d) right(%d)\n", 
				offset->top, offset->left, 
				offset->bottom, offset->right          );
}

void
treeps_exit( exit_code )
int exit_code;
{
	/* Eventually save session info here */

	if ( Button_bar_w != 0 )
	{
		freeAllIconPixmaps( Button_bar_w );
	}

	if ( DecorationGC != 0 )
	    XFreeGC( XtDisplay(Top_level), DecorationGC );

	XtDestroyWidget( TreeWidget );
	
	exit( exit_code );
}

void 
clock_event( w, id )		/* Update display and reset timer callback */
Widget       w;
XtIntervalId id;
{
	lnode *adds=NULL;
	lnode *dels=NULL;
	static int  short_tics=0;
	static int  long_tic=0;
	static int  first=1;
	static int  workDelay=20;
	long next_clock_event;

	if ( InClockEvent )
	{
  		DEBUG0(1, "--- Already Servicing ClockEvent ---");
		return;
	}

	InClockEvent = 1;

	if ( New_display_criteria )
	{
		Last_shrink = 0;

		Head = recalc_process_tree( w, Head, Process_list );

		refresh_display( w );

		New_display_criteria = 0;
	}
	else if ( Monitor_processes )
	{
		DEBUG_LIST("Pre upl Process List:", Process_list );

		Process_list = update_process_list( Machine, Process_list, &adds, &dels);

		TRACE_LIST("Add List:", adds ); 
		TRACE_LIST("Del List:", dels );
		DEBUG_LIST("Process List", Process_list );

		Active_process_list = update_active_list( Active_process_list, 
								  &adds, &dels);

		TRACE_LIST("Active List:", Active_process_list );
  	 	TRACE_LIST("Add List - vis:", adds ); 
		TRACE_LIST("Del List - vis:", dels );

		/* Here we decide to split up the processing load, 
		 * we only update process details/colors if no changes are
		 * being made to the tree. On a machine with lots of power
		 * this may not be such a good idea, probably better to measure
		 * how we are doing at keeping up with changes and only
		 * do this if we are getting behind.
		 */

		if ( display_changes( w, adds, dels ) )
		{
			Head = update_process_tree( w, Head, adds, dels);

			refresh_display( w );
			adds = free_list( adds );  
			dels = free_list( dels );
		}
		else
		{
			shrink_display_if_needed( w );
			if ( long_tic )
			{
				refresh_all_process_info( w, Process_list,
						          &Active_process_list);

				long_tic   = 0;
				short_tics = 0;
			}
			else
			{
				refresh_active_process_info( w, 
							Active_process_list );
			}

			if ( Color_bar_count_mode || Color_bar_averaging )
				recalc_process_colors();

		}
	}

	if ( ++short_tics >= Long_tic )
		long_tic++;

	next_clock_event = Short_tic;

	if ( first )
	{
	    To_id = XtAppAddTimeOut(MainContext, 1000, 
						(XtTimerCallbackProc) clock_event, w);
	    first = 0;
	}
	else 
	    To_id = XtAppAddTimeOut(MainContext, next_clock_event, 
						(XtTimerCallbackProc) clock_event, w);

	if ( workDelay > 0  )
        {
                workDelay--;
        }
        else
        {
                if ( Attachment_validator )
                        perform_attachment_validations();
        }

	InClockEvent = 0;
}

tnode
*recalc_process_tree( treew, head, process_list )
Widget  treew;
tnode  *head;
lnode  *process_list;
{
	lnode		  *work_list=NULL;
	lnode 		  *add_list=NULL;
	lnode 		  *del_list=NULL;
	lnode		  *last_add=NULL;
	lnode		  *new_node;
	lnode 		  *p, *q;
	lnode		  *l_ptr;
	tnode		  *t_ptr;
	Process_list_info *pli_ptr;
	Process_tree_info *pti_ptr;
	sysProcInfo	  *pp;
	int		   add_it;

  	DEBUG0(1, "Recalc_process_tree\n");

	/* We have a new set of display criteria, so we scan the process 
	 * list, applying the new criteria to determine which nodes 
	 * to add and delete, any node's without the process_info are
	 * ignored(they are either loaded specials(sched, init) or are
	 * just about to be deleted).
	 */

	/* List needs to be forward sorted so we add new sessions
	 * before checking for session membership 
	 */

	Displayed_session_list = free_list( Displayed_session_list );

	for ( p = process_list; p != NULL; p = p->next )
		work_list = fwd_insert_lnode( work_list, p->key, p );

	for ( q = work_list; q != NULL; q = q->next )
	{
  		DEBUG1(3, "Recalc_process_tree: Check process(%d)\n", p->key);

		p = (lnode *)q->data;

		if ( p == NULL )
			continue;

		pli_ptr = (Process_list_info *)p->data;
		if ( pli_ptr == NULL )
			continue;


		pp = (sysProcInfo *) pli_ptr->pp;
		if ( pp == NULL )
			continue;

		t_ptr = pli_ptr->t_ptr;

		add_it = 0;
		if (monitor_process(Users, p->key, pp, PROCESS_GROUP_ID(pp), 
					PROCESS_USER_ID(pp) ))
		{
			add_it++;
		}
		else  /* Check for special case session's */
		{
		       if ( visible_session( p->key, pp, add_list, del_list ) )
			   add_it++;
		}

		if ( add_it )
		{
			if ( pp && (PROCESS_ID(pp) == PROCESS_SESSION_ID(pp)) && p->key)
			{
			    if (!match_list( Displayed_session_list, p->key))
			    {
			        DEBUG1(1, "View session(%d)\n", 
						PROCESS_SESSION_ID(pp) );
			        Displayed_session_list = fwd_insert_lnode(
			    		Displayed_session_list, p->key, NULL );
			    }
			}

			if ( t_ptr == NULL ) /* Not in tree, so add */
			{
			    DEBUG0(3, "Recalc_process_tree: Add it\n");
			    if ((new_node= construct_lnode(p->key, p)) == NULL)
			    {
			        DEBUG0(1, 
				"Recalc_process_tree:construct_lnode failed\n");
				continue;
			    }

			    if ( add_list == NULL )
				add_list = new_node;
			    else
				last_add->next = new_node;

			    last_add = new_node;

			}
			else  /* Those who stay may have a color change */
			{
				DEBUG0(3, "Recalc_process_tree: Check color\n");
				set_node_colors(User_colors,Group_colors,t_ptr);
			}
		}
		else
		{
			if ( t_ptr != NULL ) /* In tree, so remove */
			{
			    DEBUG0(3, "Recalc_process_tree: Remove it\n");
			    new_node = construct_lnode(p->key, t_ptr);
			    if ( new_node == NULL )
				continue;
			
			    if ( del_list != NULL )
				new_node->next = del_list;

			    del_list = new_node;  /* Push on stack */

			    /* Set the tree node pointer in the process list
			     * to null as we are about to remove the node and
			     * we don't want an out of date pointer.
			     */

			    pli_ptr->t_ptr = NULL;
			}
		}

		q->data = NULL;
	}

	free_list( work_list );

	if ( add_list || del_list )
	{
		startBatchUpdate();

		head = delete_processes_from_ptree( treew, head, del_list );

		head = add_processes_to_ptree( treew, head, add_list );

        	finishBatchUpdate();

		free_list( add_list );
		free_list( del_list );
	}

	XFlush( XtDisplay(Top_level) );

	return( head );
}


display_changes( treew, add_list, del_list )
Widget  treew;
lnode  *add_list;
lnode  *del_list;
{
	if ( add_list || del_list )
	{
    	    DEBUG0(3, "display_changes: Processing Adds/Deletes\n");

	    /* Turn off layout until we have done all the tree updates */

            startBatchUpdate();

	    return( True );
	}
	
	return( False );
}

refresh_display( treew )
Widget treew;
{
	int n;
	Arg		 wargs[2];

	if ( Doing_auto_resize )
	{
		/* After Tree_widget_shrink_delay's we let tree
		 * shrink, even though there are updates. This
		 * handles the case of one large expansion,
		 * contraction followed by lot's of little updates.
		 * If we did not do this then the display would
		 * remain at the largest size until no activity for
		 * Tree_widget_shrink_delay's. Perhaps this would
		 * be better called persistant shrink delay!!
		 */

		DEBUG3( 6, "refresh_display: Last_shrink(%d) Pending(%d) delay(%d)\n",
			Last_shrink, Pending_shrink, Tree_widget_shrink_delay );

		if ( Last_shrink < Tree_widget_shrink_delay ) /* Waiting a bit */
		{
			n = 0;
			XtSetArg( wargs[n], XtNresizePolicy, False ); n++;
			XtSetValues( TreeWidget, wargs, n );
			Last_shrink++;
			Pending_shrink = 1;
		}
		else 	/* Tree will shrink, after current updates */
		{
			n = 0;
			XtSetArg( wargs[n], XtNresizePolicy, True ); n++;
			XtSetValues( TreeWidget, wargs, n );
			Last_shrink = 0;    
			Pending_shrink = 0;
		}
	}

        finishBatchUpdate(); /* Ok do one layout  */
	/* XFlush( XtDisplay(treew) ); */
	/* XSync(XtDisplay(treew), False ); */
}

refresh_all_process_info( treew, process_list, active_list )
Widget treew;
lnode *process_list;
lnode **active_list;
{
  lnode 		*p;
  Process_list_info	*pli_ptr;
  tnode 		*t_ptr;
  Process_tree_info	*pti_ptr;
  sysProcInfo		*pp;	/* System process info structure   */
  lnode			*stale_list=NULL;
  lnode			*new_alist=NULL;
  int			 changed;

  DEBUG0(3, "Refresh_all_process_info\n");

  TRACE_LIST("rfa-Process List:", process_list );
  TRACE_LIST("rfa-Active:      ", *active_list );

 /* Scan the process lists, calling refresh_process_info hence updating the 
  * sysProcInfo structure and making any required screen updates,
  *
  * Processes on the active list which have had no changes are
  * considered stale, we decrement their activity counter and if it drops
  * to zero, we remove them from the active list. Any process which has
  * changes and is not on the active list is added to it.
  */

		
  /* Request helper, if any, to load and cache process details */

  Machine->cache_process_details( Machine, process_list );

  for ( p = process_list; p != NULL; p = p->next )
  {
        changed = refresh_process_info( treew, p );

        pli_ptr = ( Process_list_info *) p->data;
        if ( pli_ptr == NULL )			/* No list info!!!!   */
	    continue;

        pp = pli_ptr->pp;	
        if ( pp == NULL )				
	    continue;

        if ( changed == 0 )
        {
	    if ( pli_ptr->active > 0 )  /* It's marked as active  */
	    {
		pli_ptr->active--;	
	        if ( pli_ptr->active == 0 ) /* Time to remove from active    */
		{
		    /* It's actually removed below */
		    stale_list = rev_insert_lnode( stale_list, p->key, NULL );
	        }
	    }
        } 
        else if ( pli_ptr->active == 0 )
        {
	    /* Wow it's active again!! Add back onto active list  */
	    pli_ptr->active++;
	    new_alist = rev_insert_lnode( new_alist, p->key, p );
        }

	if ( pli_ptr->t_ptr )	/* It's displayed, update details popup? */
	{
		t_ptr = pli_ptr->t_ptr;

		pti_ptr = (Process_tree_info *) t_ptr->data;
		if ( pti_ptr )
		{
		    if (( pti_ptr->dialog_state == DIALOG_DISPLAYED ) &&
		        ( pti_ptr->dialog_update_rate == UPDATE_ON_LONG_TIC ))
		    {
			/* On long tic update all info */

			display_procinfo( treew, t_ptr );
		    }
		}
	}
  } 


   if ( stale_list )
       *active_list = remove_stale_processes( *active_list, stale_list );

   if ( new_alist )
       *active_list = reactivate_processes( *active_list, new_alist );
}


refresh_process_info( treew, p )
Widget treew;
lnode *p;
{
  tnode			*t_ptr;
  Process_list_info	*pli_ptr;
  Process_tree_info	*pti_ptr=NULL;
  int			 rc;
  sysProcInfo		*pp;	/* System process info structure   */
  int			 possible_color_change=0;
  int			 possible_visibility_change=0;
  int			 visibility;
  int			 last_uid, last_gid;
  int			 last_euid, last_egid;
  int			 last_ppid;
  int			 last_sid;
  dev_t			 last_ttydev;
  char			*new_dev_name;
  char			*last_dev_name;
  major_t		 major_no;
  minor_t		 minor_no;
  int			 changed=0;
  int			 value_changed=0;
  lnode			*q, *next_q;
  int			 not_done_custom;
  char			 last_sname='-';
  u_long		 last_flags;
  char			 last_nice;
  long		 	 last_rssize;
  caddr_t		 last_wchan;
  timestruc_t		 last_time;
  struct 		 timeval tv_now;
  long		         last_pri;
  int			 dynamic_info=0;
  char			 status;
  int			 load_changed=0;
  int			 redraw_decorations=0;
  int 			 possible_deco_label_resize=0;
  int	     		 distinguished_activity=0;


    DEBUG0(3, "Refresh_process_info\n");

   /* Reload the process info, sysProcInfo structure,
    * and check for the following changes:
    *
    *	process name changes, i.e. exec's
    *	uid/group changes     i.e. exec's, setuid by process with P_SETUID
    *	euid/egid changes     i.e. exec's, setuid
    *	ppid changes	      i.e. orphaned processes
    *	controlling terminal   i.e. foreground <-> daemons
    *
    */


    pli_ptr = ( Process_list_info *) p->data;
    if ( pli_ptr == NULL )			/* No list info!!!!   */
	    return( changed );

    pp = pli_ptr->pp;	
    if ( pp )				
    {
	last_gid    = PROCESS_GROUP_ID(pp);
	last_uid    = PROCESS_USER_ID(pp);
	last_egid   = PROCESS_EFFECTIVE_GROUP_ID( pli_ptr );
	last_euid   = PROCESS_EFFECTIVE_USER_ID( pli_ptr );
  	last_ppid   = PROCESS_PARENT_ID( pp );
  	last_ttydev = PROCESS_CONTROL_TTY( pp );
	last_sid    = PROCESS_SESSION_ID( pp );
	last_sname  = PROCESS_SNAME( pp );
	last_nice   = PROCESS_NICE( pp );
	last_rssize = PROCESS_RESIDENT_SET_SIZE( pp ); 

	last_time.tv_sec    = PROCESS_CPU_TIME(pp).tv_sec;
	last_time.tv_nsec   = PROCESS_CPU_TIME(pp).tv_nsec;

	last_pri    = PROCESS_PRIORITY( pp );

	if ( Dynamic_info )
	{
		last_flags = 	PROCESS_FLAG( pp ); 
		last_wchan = 	(caddr_t) PROCESS_WCHAN( pp ); 
	}
    }
    else
    {
	last_gid    = 0;
	last_uid    = 0;
	last_egid   = 0;
	last_euid   = 0;
  	last_ppid   = 0;
  	last_ttydev = -1;
	last_sid    = 0;
	last_time.tv_sec  = 0;
	last_time.tv_nsec = 0;
    }


    if ( Machine->scanner_type == INTERNAL_POLL_SCANNER ) 
        rc = load_sys_proc_info( p->key, pli_ptr );
    else
        rc = load_ext_proc_info( Machine, p->key, pli_ptr );

    if ( rc == IGNORE_PROCESS )
    {
	/* Either we never have access, perhaps were running non suid and
	 * it's sched, init or the process just terminated and we haven't
	 * got round to noticing it yet(this does happen when the system
	 * get's real busy and we just haven't refreshed the process list
	 * yet). If the process is marked active, we mark as inactive.
	 * This allows the caller to get it off the active list quickly.
	 */

	if ( pli_ptr->active != 0 )	  /* it's currently active!!!    */
	{
	    pli_ptr->active = 0;
	}

	return( changed );
    }

    pp = (sysProcInfo *)pli_ptr->pp;
    if ( pp == NULL )				/* Just a sanity check 	*/
        return( ++changed );

    t_ptr = pli_ptr->t_ptr;
    if ( t_ptr )		/* It's in display tree(may be hidden) 	*/
    {
        pti_ptr = (Process_tree_info *)t_ptr->data;
        if (strcmp(t_ptr->str_key, PROCESS_CMD_NAME(pp)) != 0)  /* Exec'ed */
        {
	    if ( t_ptr->str_key != NULL )
		free( t_ptr->str_key );

 	    t_ptr->str_key = strdup( PROCESS_CMD_NAME( pp ) );

       	    if ( pti_ptr->state == NODE_DISPLAYED )
	    {
       		display_proc_info( t_ptr, 0 );	/* Display new label 	*/
		possible_deco_label_resize++;
	    }

	    changed++;
        }

	if ( pti_ptr->display_list )
		dynamic_info = 1;
    }


    if ( last_ppid != PROCESS_PARENT_ID( pp ) )  /* Parent changed */
    {
	  DEBUG2(1, "refresh_process_info: last_ppid(%d) != ppid(%d)\n",
							last_ppid, 
					PROCESS_PARENT_ID( pp ));
	  possible_visibility_change++;

	  /* Reparent??? Can this happen other than orphan to init?? 
	   * As far as I know only the system does this.  We catch 
	   * these when we orphan a process in delete_process_from_ptree. 
	   * If a parent is not visible then we currently hang the process
	   * off of init(so all this does is reparent on init). Future plans 
	   * are to allow non attached processes to be hanged off of
	   * either:
	   *		The closest relative(a named session leader(SMART_ATTACH)),
	   *		Init.
	   *		Perhaps a special symboloic user node.
	   *		Or failing that a special symbolic group node.
	   *
	   * If we do this then when the system orphans a process to init
	   * we can catch it here and do the desired thing.
	   *
	   * I'm not going to do this at the moment, it's computationaly expensive
	   * and there is not much to gain. Users can force this with some operation
	   * that calls the reparenting routine. It means smart attach won't be involved
	   * when processes are orphaned.
	   *
	   * Perhaps a better approach is to mark processes as orphaned when
	   * they are orphaned to init. Then have a work "task" that checks for these
	   * and tries to find parents, perhaps giving up after a while. I suppose it's
	   * possible for such adoptive parents to be hidden while the orphans are not.
	   * Which means visibility changes should reset the orphanes state...
	   *
	   * This sort of stuff does not happen that often anyhow and defaulting to
	   * hanging off init is not that bad a solution.
	   */

	/* there is a bug here, addressed in next release  */

	/* XXX ??? call 	orphan_process( head, t_ptr ); ???		*/
	    changed++;
    }

    if ( last_ttydev != PROCESS_CONTROL_TTY(pp) )
    {
	 if ( Debug > 0 )
	 {
	     major_no = major( PROCESS_CONTROL_TTY(pp) );
	     minor_no = minor( PROCESS_CONTROL_TTY( pp ) );
	     last_dev_name = find_dev_name( major_no, minor_no );

	     major_no = major( PROCESS_CONTROL_TTY( pp ) );
	     minor_no = minor( PROCESS_CONTROL_TTY( pp ) );
	     new_dev_name = find_dev_name( major_no, minor_no );

	     DEBUG2(1, "refresh_process_info: last_ttydev(%s) != ttydev(%s)\n",
						last_dev_name, new_dev_name );
	 }

	 /* Should show onscreen that process is becomming daemon? Perhaps
	  * a Daemon icon for a second 8-).
	  */


	 possible_visibility_change++;
	 changed++;
    }

    if ( last_sid != PROCESS_SESSION_ID( pp ) )	/* Changing session */
    {
	possible_visibility_change++;

	/* If currently visible add to session list, if it also became a
	 * daemon, usually does with setsid, setgrp then the visibility
	 * check will remove it if need be
	 */

	if ( t_ptr )
	{
	    DEBUG1(1, "New session a(%d)\n", PROCESS_SESSION_ID( pp ) );
	    Displayed_session_list = fwd_insert_lnode(
	    			Displayed_session_list, 
				PROCESS_SESSION_ID( pp ), NULL );
	}
	changed++;
    }

    if ( last_gid != PROCESS_GROUP_ID(pp) )		 
    {
	  DEBUG2(1, "refresh_process_info: last_gid(%d) != gid(%d)\n",
							last_gid, pp->pr_gid);
	  possible_color_change++;
	  possible_visibility_change++;
	  changed++;
    }

    if ( last_uid != PROCESS_USER_ID(pp) )	   /* Exec'd? suid???  */
    {
	  DEBUG2(1, "refresh_process_info: last_uid(%d) != uid(%d)\n",
							last_uid, 
						PROCESS_USER_ID( pp ) );
	  changed++;
	  possible_color_change++;
	  possible_visibility_change++;
	  if ( pti_ptr )
          	pti_ptr->uid = PROCESS_USER_ID(pp);		
    }

    if ( last_euid != PROCESS_EFFECTIVE_USER_ID( pli_ptr )  )	 /* Suid'ed   */
    {
	  DEBUG2(1, "refresh_process_info: last_euid(%d) != euid(%d)\n", 
	  		last_euid, PROCESS_EFFECTIVE_USER_ID( pli_ptr ) );
	  possible_color_change++;
	  changed++;
    }

    if ( last_egid != PROCESS_EFFECTIVE_GROUP_ID( pli_ptr ) )		   
    {
	  DEBUG2(1, "refresh_process_info: last_egid(%d) != egid(%d)\n",
	  		last_egid, PROCESS_EFFECTIVE_GROUP_ID( pli_ptr ) );
	  possible_color_change++;
	  changed++;
    }	

    /* If this is the first sample we save the current cpu usage and
     * current timestamp in the stats structure. This keeps the initial
     * calculated load within bounds.
     */

    gettimeofday( &tv_now, NULL );

    if ( t_ptr )
    {
      pti_ptr = (Process_tree_info *)t_ptr->data;

      /* Save sample timestamp for later calculations */

      if ( pti_ptr->stats.current_load == -1.0 ) 
      {
	pti_ptr->stats.lastSampledAt.tv_sec  = tv_now.tv_sec;
	pti_ptr->stats.lastSampledAt.tv_usec = tv_now.tv_usec;

	pti_ptr->stats.last_time.tv_sec =PROCESS_CPU_TIME(pp).tv_sec;
	pti_ptr->stats.last_time.tv_nsec=PROCESS_CPU_TIME(pp).tv_nsec;
      }

      /* Calculate load if needed */

      if ( (Decorated_tree && Decorations_include_load) || 
					Color_based_on == COLOR_BY_LOAD )
      {
	calc_load( pti_ptr, &tv_now );
      }


      /* We want to shrink top decorations before any color changes
       * so we can force a redraw of colors to fill background
       */

      if ( Decorated_tree && ( pti_ptr->state != NODE_HIDDEN ) )
      {
         if ( Decorated_tree == DECORATE_DISTINGUISHED  &&
	 			( pti_ptr->distinguished ) )
	 {
	     /* Remove if node has not distinguished itself in a while */
	     /* When removing - may be able to shrink node width */

	     pti_ptr->distinguished -= 1;

	     if ( pti_ptr->distinguished == 0 )
	     {
		DEBUG3(5, "COLLAPSE Distinguish Node(%d:%s) now(%d)\n", 
				t_ptr->key, 
				t_ptr->str_key,
	     			pti_ptr->distinguished  );

	     	collapse_decoration_top_panel( t_ptr );

       		display_proc_info( t_ptr, 0 );	/* Adjust label size 	*/
	     }
	}
      }

#ifndef LINUX
	/* Perform low res synthesis of Just Ran State, note field size are
 	 * hi res but values in them are low res i.e. microsecs
	 */

	if ( PROCESS_SNAME(pp) == 'S' )
        {
	    if ( last_time.tv_sec == 0 && 
			last_time.tv_nsec == 0 )
	    {
		PROCESS_SNAME(pp) = 'S';
	    }
	    else if ( PROCESS_CPU_TIME(pp).tv_nsec == last_time.tv_nsec )
            {
                if ( PROCESS_CPU_TIME(pp).tv_sec != last_time.tv_sec )
			PROCESS_SNAME(pp) = 'A';
            }
            else
            {
		PROCESS_SNAME(pp) = 'A';
            }
        }

#endif


      switch ( Color_based_on )
      {
        /* These are handled above so no need to test twice 
	 *
	 *    case COLOR_BY_REAL_USER_ID:
	 *    case COLOR_BY_EFFECTIVE_USER_ID:
	 *    case COLOR_BY_REAL_GROUP_ID:
	 *    case COLOR_BY_EFFECTIVE_GROUP_ID:
	 *
	 */
	/* case COLOR_BY_IMAGE_SIZE:    Doesn't change? */

	case COLOR_BY_LOAD: 

      		pti_ptr = (Process_tree_info *)t_ptr->data;

		if ( pti_ptr->stats.last_load != pti_ptr->stats.current_load )
			load_changed++;

		if ( load_changed )
		{
		    if ( Color_bar_count_mode || Color_bar_averaging )
			possible_color_change = 0;
		    else
			possible_color_change++;  
		}
		break;

	case COLOR_BY_TOTAL_TIME:
	  	possible_color_change = 
			((last_time.tv_sec != PROCESS_CPU_TIME(pp).tv_sec) || 
			(last_time.tv_nsec != PROCESS_CPU_TIME(pp).tv_nsec));
		break;

	case COLOR_BY_STATUS:
	  	possible_color_change = (last_sname != PROCESS_SNAME(pp));
		break;

	case COLOR_BY_PRIORITY:
	  	possible_color_change = (last_pri != PROCESS_PRIORITY(pp));
		break;

	case COLOR_BY_RESIDENT_PAGES: possible_color_change = 
			    (last_rssize != PROCESS_RESIDENT_SET_SIZE(pp)); 
		break;

	default: break;
      }

      if ( possible_color_change && ( pti_ptr->state != NODE_HIDDEN ) )
      {
	  set_node_colors( User_colors, Group_colors, t_ptr); 
	  changed++;
      }

      if ( Decorated_tree && ( pti_ptr->state != NODE_HIDDEN ) )
      {
         if ( Decorated_tree == DECORATE_ALL )
	 {
	     redraw_decorations++;
	 }
	 else  
	 {
	     /* Check to see if it's newly distinguished */
	     if ( ( pti_ptr->stats.current_load > 0.0 ) ||
	  		      ( PROCESS_SNAME(pp) != 'S' ) ||
	  		      ( PROCESS_PRIORITY(pp) < Priority_Normal_Min)  ||
	  		      ( PROCESS_PRIORITY(pp) > Priority_Normal_Max)  )
	     {
	     	distinguished_activity++;
	     }


	     if ( pti_ptr->distinguished )
	     {
	        if ( distinguished_activity )
	 		pti_ptr->distinguished = Decoration_fame_time;
	        redraw_decorations++;
	     }
	     else
	     {
		if ( pti_ptr->node_w && distinguished_activity )
		{
		    expand_decoration_top_panel( t_ptr );
		    possible_deco_label_resize++;
	            redraw_decorations++;

	 	    pti_ptr->distinguished = Decoration_fame_time;
		    DEBUG2(5, "Distinguish Node(%d:%s)\n", t_ptr->key, 
		    					t_ptr->str_key );

		}
	     }

	 }

	if ( possible_deco_label_resize )
		decoration_label_resize( pti_ptr );

	 if ( redraw_decorations )
	 	redraw_node_decorations( pti_ptr->top_draw_w, t_ptr );

      }

    }

    if ( !changed )  /* Certain states stay in active list */
    {
    	  status = PROCESS_SNAME(pp);
	  switch ( status ) {
	  	case 'S': break;	/* Sleeping */
	  	case 'I': break;	/* Idle */
	  	case 'Z': break;        /* Zombie */

		default:  changed++;  break;
	  }
    }


    /* Do we need to update any displayed fields, in tree or details popup  */

    if ( pti_ptr && (pti_ptr->state != NODE_HIDDEN) && 
    					(Dynamic_info || dynamic_info) )  
    {
	/* For each member of global display list, check to see if it's
	 * value changed. For each member of custom display list check to
	 * see if it's value changed. If the value changed, call
	 * display_proc_info
	 */
	not_done_custom = 1;
  	for ( q = Global_display_list; q != NULL; q = next_q )
	{
	    switch ( q->key )
	    {
		case PR_SNAME:  if ( PROCESS_SNAME(pp) != last_sname ) 
  					value_changed = 1;
				break;
		case PR_FLAGS:  if ( PROCESS_FLAG( pp ) != last_flags )
  					value_changed = 1;
				break;
		case PR_NICE:   if ( PROCESS_NICE( pp ) != last_nice )
  					value_changed = 1;
				break;
		case PR_RSSIZE: if ( PROCESS_RESIDENT_SET_SIZE( pp ) != last_rssize )
  					value_changed = 1;
				break;
		case PR_WCHAN:  if (((caddr_t)PROCESS_WCHAN(pp)) != last_wchan)
  					value_changed = 1;
				break;
		case PR_TIME:  

			if ( PROCESS_CPU_TIME(pp).tv_sec != last_time.tv_sec )
  				value_changed = 1;
			if ( PROCESS_CPU_TIME(pp).tv_nsec != last_time.tv_nsec )
  				value_changed = 1;
			break;

		case PR_PRI:	if ( PROCESS_PRIORITY( pp ) != last_pri )
  					value_changed = 1;
				break;

		default:  break;
	    }

	    if ( value_changed )	/* Any one change causes update */
		break;

	    if ( q->next == NULL )
	    {
		if ( not_done_custom )	/* Do the custom node list */
		{
			next_q = pti_ptr->display_list;
			not_done_custom = 0;
		}
		else
			next_q = NULL;
	    }
	    else
		next_q = q->next;
	}

	if ( value_changed )
	{
		display_proc_info( t_ptr, 0 );
	}
    }

    if ( pti_ptr )
    {
	if (( pti_ptr->dialog_state == DIALOG_DISPLAYED ) &&
		      ( pti_ptr->dialog_update_rate == UPDATE_ON_SHORT_TIC ))
	{
		/* Should check to see if value actually changed */
		refresh_procinfo( treew, t_ptr );
	}
    }

    if ( possible_visibility_change )
    {
	visibility = monitor_process(Users, p->key, pp, PROCESS_GROUP_ID(pp), 
						PROCESS_USER_ID(pp));
	if ( t_ptr && (visibility == False) )
	{
		DEBUG1(1, "refresh_process_info: Deleting(%d)\n", p->key );
		changed++;
		Head = delete_process_from_ptree( treew, Head, t_ptr );
	}
	else if ( visibility && (t_ptr == NULL) )
	{
		DEBUG1(1, "refresh_process_info: Adding(%d)\n", p->key );
		changed++;
		Head = add_process_to_ptree( treew, Head, p );
	}
    }

    return( changed );
}

lnode
*remove_stale_processes( active_list, stale_list )
lnode *active_list;
lnode *stale_list;
{
	lnode *p;
	lnode *q;
	lnode *last_q;

        TRACE_LIST("remove Stale_list:", stale_list );

	for ( p = stale_list; p != NULL ; p = p->next )
	{
		for ( q = active_list ; q != NULL ; q = q->next )
		{
			if ( p->key == q->key )
				break;
			last_q = q;
		}

		if ( q == NULL )	/* Didn't find it in active list */
			continue;

		if ( q == active_list )		/* Remove head  	*/
			active_list = q->next;
		else				/* Remove middle or end */
			last_q->next = q->next;

		q->data = NULL;		/* You will be, you will be     */
		free( q );

		p->data = NULL;   /* So we don't free the process list entry */
	}

	stale_list = free_list( stale_list );

	return( active_list );
}
lnode
*reactivate_processes( active_list, new_alist )
lnode *active_list;
lnode *new_alist;
{
	lnode *p;

        TRACE_LIST("Reactivate New_alist:", new_alist ); 

	for ( p = new_alist; p != NULL ; p = p->next )
	{
		active_list = rev_insert_lnode( active_list, p->key, p->data );

		p->data = NULL; /* so we don't free the process_list entry */
	}

	new_alist = free_list( new_alist );

	return( active_list );
}

refresh_active_user_info( users )
User_group_info *users;
{
  lnode 		*p;
  Process_list_info	*pli_ptr;
  sysProcInfo		*pp;	/* System process info structure   */
  lnode			*user_count_list=NULL;
  lnode			*group_count_list=NULL;

  	DEBUG0(4, "Refresh_active_user_info\n");

 	/* Scan the process lists, updating the user/group counts */
		
  	for ( p = Process_list; p != NULL; p = p->next )
  	{
    		pli_ptr = ( Process_list_info *) p->data;
    		if ( pli_ptr == NULL )			
	    		continue; 		/* No list info!!!!    */

     		pp = pli_ptr->pp;	
     		if ( pp == NULL )				
			continue;		/* No system proc info */

    		group_count_list = inc_list_count(group_count_list,
						PROCESS_GROUP_ID( pp ));
    		user_count_list  = inc_list_count(user_count_list, 
						PROCESS_USER_ID( pp ) );
	}

	users->active_group_list = zero_free_list( users->active_group_list );
	users->active_user_list  = zero_free_list( users->active_user_list );

	users->active_group_list = group_count_list;    /* assign new lists */
	users->active_user_list  = user_count_list;
}


refresh_active_process_info( treew, active_process_list )
Widget treew;
lnode *active_process_list;
{
  lnode 		*p;
  lnode 		*l_ptr;
  Process_list_info	*pli_ptr;

  	DEBUG0(4, "Refresh_active_process_info\n");

 	/* Scan the active process list, calling refresh_process_info
	 * to update the Process list node's sysProcInfo and to check
	 * for and make any required screen changes:
  	 */

  	Machine->cache_process_details( Machine, active_process_list );

  	for ( p = active_process_list; p != NULL; p = p->next )
  	{
		/* Get actual process list entry pointer */

		l_ptr = (lnode *)p->data;
		if ( l_ptr == NULL )
			continue;
		
		pli_ptr = (Process_list_info *) l_ptr->data;
		if ( pli_ptr == NULL )
			continue;

		if ( pli_ptr->pp == NULL )
			continue;

        	refresh_process_info( treew, l_ptr );
	}
}


shrink_display_if_needed( treew )
Widget treew;
{
	int n;
	Arg		 wargs[2];

	if ( Doing_auto_resize && Pending_shrink )
	{
		DEBUG3( 6, "shrink_delay_if_needed: Last_shrink(%d) Pending(%d) delay(%d)\n",
			Last_shrink, Pending_shrink, Tree_widget_shrink_delay );

		if ( Last_shrink >= Tree_widget_shrink_delay )
		{
			/* Shrink the treewidget to fit children */
			n = 0;
			XtSetArg( wargs[n], XtNresizePolicy, True ); n++;
			XtSetValues( TreeWidget, wargs, n );
			Last_shrink = 0;
			Pending_shrink = 0;
	    	}
		else
		{
			Last_shrink++;
		}
	}
}


Machine_info *
construct_machine_monitor( osType, osRelease, name, type )
char *osType;
char *osRelease;
char *name;
int   type;   /* scanner type */
{
	Machine_info  		*machine; 
	int			 rc;

	DEBUG0(1, "Construct machine \n");

	machine = ( Machine_info  * ) malloc( sizeof( Machine_info ) );
	if ( machine == NULL )
	{
	    fprintf( stderr, "Out of memory! malloc Machine info\n" );
	    return NULL;
	}

	machine->name      	   = strdup( name );
	machine->osType    	   = strdup( osType );
	machine->osRelease         = strdup( osRelease );
	machine->scanner_type      = type;
	machine->next 		   = NULL;

	if ( type == INTERNAL_POLL_SCANNER )  /* Setup local machine monitor */
	{

	    DEBUG0(1, "Internal Scanner\n");

	    machine->get_machine_status = proc_get_machine_status;

	    machine->get_first_pid     = proc_get_first_pid;
	    machine->get_next_pid      = proc_get_next_pid;
	    machine->close_pid_scanner = proc_close_pid_scanner;


	    machine->cleanup_pid_scanner   = proc_cleanup_pid_scanner;
	    machine->get_pid_main_details  = proc_get_pid_main_details;
	    machine->get_pid_extra_details = proc_get_pid_extra_details;

	    machine->cache_process_details    = proc_cache_process_details;
	    machine->release_process_details  = proc_release_process_details;

	    if ( ( rc = proc_init_pid_scanner( machine, Process_Info_Dir )) < 0 )
	    	return NULL;
	}
	else if ( type == EXTERNAL_POLL_SCANNER )  /* Setup local machine monitor */
	{

printf("External Scanner\n");

	    machine->get_machine_status = gei_get_machine_status;

	    machine->get_first_pid     = gei_get_first_pid;
	    machine->get_next_pid      = gei_get_next_pid;
	    machine->close_pid_scanner = gei_close_pid_scanner;

	    machine->cleanup_pid_scanner   = gei_cleanup_pid_scanner;
	    machine->get_pid_main_details  = gei_get_pid_main_details;
	    machine->get_pid_extra_details = gei_get_pid_extra_details;

	    machine->cache_process_details    = gei_cache_process_details;
	    machine->release_process_details  = gei_release_process_details;

	    if ( ( rc = gei_init_pid_scanner( machine, LibRootDir )) < 0 )
	    {
	        fprintf( stderr, "failed to find/start external poller\n");
		user_alert( Top_level, "Failed to find external helper application executable for designated machine");
	    }

	    /* Register the returned child pid as input - for async */

	}
	else
	{
	    printf("Notification(async) scanners not implemented yet \n");
	}

	NumMachinesMonitored += 1;

	return machine;
}

cleanup_machine_monitor( pid )
{
	Machine_info  	*machine; 
	int		type;
	char		message[128];

	machine = MachineList;

	while ( machine )
	{
	    type = machine->scanner_type;
	    if ( type != INTERNAL_POLL_SCANNER )  /* Check external process is ok */
	    {
		printf("Cleanup helper(%d)\n", pid );
	    	if ( machine->cleanup_pid_scanner( machine, pid ) == 0 )
		{
		   sprintf( message, "Helper process(%d) died unexpectedly!", pid );
		   user_alert( TreeWidget, message );
		}
	    }

	    machine = machine->next;
	}
}

void
child_sig_handler( sig_no )
int sig_no;
{
        int pid, status;
	int  sanity_check=100;

        DEBUG0( 1,  "Caught SIG_CHILD\n");

        while(1)
        {
            if ( (pid = waitpid(-1, &status, 0)) == -1)
            {
               if (errno == EINTR)
                   continue;
            }
            else
            {
               break;
            }
	    if ( sanity_check-- < 0 )
		break;
         }

        cleanup_machine_monitor( pid );
}

void
pipe_sig_handler( sig_no )
int sig_no;
{
        int pid, status;

        printf( "Caught SIG_PIPE\n");

	/* typically means child died, should get a SIGCHILD as well */
}



/* Update the process list, producing a list of Additions and Deletions */


lnode
*update_process_list( machine, process_list, add_list, del_list )
Machine_info *machine;
lnode *process_list;
lnode **add_list;
lnode **del_list;
{
	static  lnode   *last_plist=NULL;
	static  lnode  	*new_plist=NULL;
	int		 pid;
	lnode 	        *plist, *p;
	int		 adds=0;
	int		 dels=0;
	char		*e;		/* End of number in file name */
	char		*fname;

	plist = process_list;

	pid = machine->get_first_pid( machine );
	while ( pid != -1 )
	{
		new_plist = rev_insert_lnode( new_plist, pid, NULL );

		pid = machine->get_next_pid( machine );
	}

	diff_rev_lists( last_plist, new_plist, add_list, del_list );

  	DEBUG_LIST("Update_proc_list[1] Add_list:", *add_list ); adds++;
  	DEBUG_LIST("Update_proc_list[1] Del_list:", *del_list ); dels++;

	plist = delete_processes_from_plist( plist, del_list );

	plist = add_processes_to_plist( plist, add_list );

  	DEBUG_LIST("Update_proc_list[2] Add_list:", *add_list );
  	DEBUG_LIST("Update_proc_list[2] Del_list:", *del_list );

	last_plist = free_list( last_plist );
	last_plist = new_plist;
	new_plist  = NULL;

	return( plist );
}


lnode
*update_active_list( list, adds, dels )
lnode *list;
lnode **adds;
lnode **dels;
{
	lnode *new_list;

	/* As a side effect we prune the lists of entries which are of
	 * no interest to update_process_tree. 
	 */

	DEBUG_LIST("Active before update", list );

	new_list = delete_processes_from_active( list, dels );

	DEBUG_LIST("Active after dels", new_list );

	new_list = add_processes_to_active( new_list, adds );

	DEBUG_LIST("Active after adds", new_list );

	return( new_list );
}

lnode
*delete_processes_from_active( active_list, dels )
lnode *active_list;
lnode **dels;
{
	lnode *p;
	lnode *q;
	lnode *last_q=NULL;
	lnode *new_dels=NULL;
	lnode *last_p;
	lnode *next_p;
	tnode *t_ptr;


	if ( active_list == NULL )
		return( NULL );
	
	if ( *dels == NULL )
		return( active_list );

	last_p = *dels; new_dels = *dels;
	for ( p = *dels; p != NULL ; p = next_p )
	{
		for ( q = active_list; q != NULL ; q = q->next )
		{
			if ( p->key == q->key )
				break;
			last_q = q;
		}

		if ( q != NULL )	/* In active list   */
		{
			if ( q == active_list )	/* Remove from head */
				active_list = q->next;
			else			/* Remove from middle or end */
				last_q->next = q->next;
			free( q );
		}


		/* filter out delete requests not in the displayed tree */

		t_ptr = (tnode *)p->data;
		if ( t_ptr == NULL )		/* Remove from del list */
		{
			if ( p == new_dels )	/* remove from head     */
			{
				new_dels = p->next;
				last_p   = new_dels;
				next_p   = new_dels;
			}
			else		/* remove from middle or end    */
			{
				last_p->next = p->next;
				next_p = p->next;
			}

			free( p );
		}
		else
		{
			next_p = p->next;
			last_p = p;
		}
	}

	if ( *dels != new_dels )
		*dels = new_dels;

	return( active_list );
}


lnode
*add_processes_to_active( active_list, adds )
lnode *active_list;
lnode **adds;
{
	lnode 			*new_active_list=NULL;
	lnode 			*new_add_list=NULL;
	lnode 			*p;
	lnode 			*last_p=NULL;
	lnode 			*next_p=NULL;
	lnode 			*l_ptr;
	Process_list_info  	*pli_ptr;
	sysProcInfo		*pp=NULL;
	int    			 add_it;
	int    			 uid, gid;
	int			 do_search;
	tnode			*stp_ptr;
	lnode			*slp_ptr;

	/* The active and add list are both reverse sorted.
	 * We add all of the new processes to the active list, those which
	 * match the current display criteria are put on a new add list
	 * which is forward sorted.
	 */

	/* The adds list data pointer points to the new process node. This
	 * is copied to the new add list.
	 */

	for ( p = *adds; p != NULL ; p = p->next )
	{ 
		new_add_list = fwd_insert_lnode(new_add_list, p->key, p->data);
		last_p = p;
	}

	/* Splice the new active list on the front of the old list */

	if ( last_p )
	{
		last_p->next = active_list;
		new_active_list = *adds;
	}
	else
		new_active_list = active_list;


	/* Now scan the new add's checking for session leaders and removing
	 * processes of no interest. We wait till after the list is
	 * forward sorted, so that session leaders will be identified
	 * before session membership is tested
	 */

	for ( p = new_add_list ; p != NULL ; p = next_p )
	{
		add_it = 0;

		l_ptr = (lnode *)p->data;
		if ( l_ptr )
		{
		    pli_ptr = (Process_list_info *)l_ptr->data;
		    if ( pli_ptr )
		    {
			pp = pli_ptr->pp;
			if ( pp )
			{
			    uid = PROCESS_USER_ID( pp );  
			    gid = PROCESS_GROUP_ID( pp );  
			    add_it++;
			}
			else
		        {
			    if ( p->key == 0 || p->key == 1 )
			    {
				uid = 0;  gid = 0;  add_it++;
			    }
		        }
		    }
		}

		if ( add_it )
		{

	    	    add_it = monitor_process(Users, p->key, pp, gid, uid);
	    	    if ( add_it )
	    	    {
			/* If a new session add to session list */
			if ( pp && (PROCESS_ID(pp) == PROCESS_SESSION_ID(pp)) && p->key)
			{
		    	    DEBUG1(1, "View session(%d)\n", PROCESS_SESSION_ID(pp) );
		    	    Displayed_session_list = fwd_insert_lnode(
		    		Displayed_session_list, p->key, NULL );
			}
	    	    }
		    else
		    {
		      /* SPECIAL CASE:
		       *
		       * If it's a new session and we are showing all
		       * descendants and this is a direct descendant of
		       * a visible tree member or one about to be added as such,
		       * AND it matches the current daemon criteria we add it. 
		       * If and only if we have a valid sysProcInfo structure
		       * and the pid is not 0!! Finally we also ignore sessions
		       * hanging off of init. Talk about convolouted
		       */

		       if ( visible_session( p->key, pp, new_add_list, NULL ) )
			   add_it++;
		    }
		}

		if ( add_it )	
		{
		    next_p = p->next;
		    last_p = p;
		}
		else
		{
		    if ( p == new_add_list )
		    {
			new_add_list = p->next;
			last_p = new_add_list;
			next_p = new_add_list;
		    }
		    else
		    {
			last_p->next = p->next;
			next_p = p->next;
		    }

		    free( p );
		}
	}

	*adds = new_add_list;

	return( new_active_list );
}


visible_session( pid, pp, check_list, expires )
int pid;
sysProcInfo    *pp;
lnode         *check_list;
lnode         *expires;
{
	int add_it=0;
	int do_search;
	tnode *stp_ptr;
	lnode *slp_ptr;

	/* Check for special case visibility of a session:
	 *
	 * 	If it's a new session and we are showing all
	 * 	descendants and this is a direct descendant of
	 * 	a visible tree member or one in the check list,
	 *	but not one in the expires list,
	 * 	AND it matches the current daemon criteria we add it. 
	 * 	If and only if we have a valid sysProcInfo structure
	 * 	and the pid is not 0!! Finally we also ignore sessions
	 * 	hanging off of init. Talk about convolouted
	 */

	if ( pp && (PROCESS_ID(pp) == PROCESS_SESSION_ID( pp )) && pid )
	{
		if ( Show_descendants == SHOW_ALL_DESCENDANTS )
		{
			do_search = 1;
			if ( Show_daemons == False ) 
			{
#ifdef LINUX
				if ( ( PROCESS_CONTROL_TTY(pp) == -1 )  ||
					( PROCESS_CONTROL_TTY(pp) == 0 )  )
#else
				if ( PROCESS_CONTROL_TTY(pp) == -1 )
#endif
				{
					/* A Daemon or Uknown */
					do_search = 0;   /* so we ignore it */
				}
			}

			if ( PROCESS_PARENT_ID( pp ) == 1 )
			    do_search = 0;

			
			if ( match_list( expires, PROCESS_PARENT_ID( pp ) ) == 1 )
				do_search = 0;

			if ( do_search )
			{
				stp_ptr=walk_tree(Head,match_key,
						PROCESS_PARENT_ID( pp ));

				slp_ptr=find_lnode(check_list,
						PROCESS_PARENT_ID( pp ));

				if ( stp_ptr || slp_ptr )
				{
				    add_it++;
				    DEBUG1(1,"VIEW SESSION(%d)\n",
				    		PROCESS_SESSION_ID( pp ));
				    Displayed_session_list = fwd_insert_lnode(
				         Displayed_session_list, pid, NULL );
				}
			}

		}
	}

	return( add_it );
}


/* Update_process_tree is called repetatively to update our internal
 * model of the process tree and modify the display to represent the
 * new state of the viewed processes.
 */

tnode
*update_process_tree( treew, head, add_list, del_list )
Widget  treew;
tnode  *head;
lnode  *add_list;
lnode  *del_list;
{
	lnode *p;

    	DEBUG0(3, "update_process_tree:");

	/* Note: as an efficiency hack the data field of add_list nodes
         * carry pointers to the process list entry's. 
         * Also the del_list data field holds a pointer to the corresponding 
	 * tree node, for quick access.
	 */

	head = delete_processes_from_ptree( treew, head, del_list );

	head = add_processes_to_ptree( treew, head, add_list );

	return( head );
}

tnode
*delete_processes_from_ptree( treew, head, del_list )
Widget  treew;
tnode  *head;
lnode  *del_list;
{
	lnode  		     *p;
	tnode   	     *t_ptr;

    	DEBUG0(3, "delete_processes_from_ptree:");

	for ( p = del_list ; p != NULL ; p = p->next )
	{
		t_ptr = (tnode *)p->data;

		head = delete_process_from_ptree( treew, head, t_ptr );

		p->data = NULL;
	}

	return( head );
}

tnode
*delete_process_from_ptree( treew, head, t_ptr )
Widget  treew;
tnode  *head;
tnode  *t_ptr;
{
	child_ptr            *cptr, *next_cptr;
	sysProcInfo	     *pp;
	Process_tree_info    *pti_ptr;

	/* Note remove_tnode will not remove top node of tree */

	if ( t_ptr != NULL )
	{
	        DEBUG2(3, "delete_process_from_ptree:  del(%d:%s)\n",
					   t_ptr->key, t_ptr->str_key );

		/* Don't allow init(1) or sched(0) to be removed */

		if ( t_ptr->key == 0 || t_ptr->key == 1 ) 
		{
	        	DEBUG0(1, "Can't delete sched or init\n");
			return( head );
		}

		pti_ptr = (Process_tree_info *)t_ptr->data;
		if ( pti_ptr )
			pp = pti_ptr->pp;

		/* If ending a session remove it from the session list */

		if ( match_list(Displayed_session_list, t_ptr->key) )
		{
		    DEBUG1(1, "End session(%d)\n", t_ptr->key );
		    Displayed_session_list = delete_lnode(
		    		          Displayed_session_list, t_ptr->key );
		}

		/* Move children to new parent's or orphan to init(1)*/

		for ( cptr = t_ptr->children ; cptr != NULL ;  )
		{
			next_cptr = cptr->next;
			head = orphan_process( head, cptr->child );
			cptr = next_cptr;
		}

		destroy_proc_widget( t_ptr );
		head = destroy_process_tree_node( head, t_ptr );
	}

	return( head );
}

tnode
*add_processes_to_ptree( treew, head, add_list )
Widget  treew;
tnode  *head;
lnode  *add_list;
{
	lnode 			*p;

    	DEBUG0(3, "add_processes_to_ptree:");

	for ( p = add_list; p != NULL ; p = p->next )
	{
		head = add_process_to_ptree( treew, head, p->data );
		p->data = NULL;
	}

	return( head );
}

/* char *Session_root_list[] = { "gnome-session", "kde-session", "dtsession", "" }; */

int
set_desktop_session_root( cur_uid )
int cur_uid;
{
  lnode 		*p;
  Process_list_info	*pli_ptr;
  sysProcInfo		*pp;	/* System process info structure   */
  int			 uid;
  char 			*proc_name;

  /* Scan the process list and find the process that shares the same user id 
   * as this one and is also a GUI session leader(gnome-session, ksmserver...).
   */

  for ( p = Process_list; p != NULL; p = p->next )
  {
        pli_ptr = ( Process_list_info *) p->data;
        if ( pli_ptr == NULL )			/* No list info!!!!   */
	    continue;

        pp = pli_ptr->pp;	
        if ( pp == NULL )				
	    continue;

	proc_name = PROCESS_CMD_NAME( pp );
	uid 	  = PROCESS_USER_ID( pp );

	if ( uid != cur_uid )
		continue;

	if ( strcmp( proc_name, "gnome-session") == 0 )
	{
    		User_Session_Root = p->key;
		break;
	}

	if ( strcmp( proc_name, "ksmserver") == 0 ) /* kde session manager */
	{
		DoingKdeNameMagic++;
    		User_Session_Root = p->key;
		break;
	}

	if ( strcmp( proc_name, "startkde") == 0 ) /* kde session manager */
	{
		DoingKdeNameMagic++;
    		User_Session_Root = p->key;
		break;
	}

	if ( strcmp( proc_name, "dtsession") == 0 ) /* CDE session manager */
	{
    		User_Session_Root = p->key;
		break;
	}
   }

    DEBUG2(1, "Set session root for uid(%d) - pid(%d)\n", 
    					cur_uid, User_Session_Root );
    return User_Session_Root;
}



tnode
*add_process_to_ptree( treew, head, p)
Widget  treew;
tnode  *head;
lnode  *p;
{
	tnode   		*ptr;
	tnode			*new;
	tnode   		*parent;
	tnode			*init_ptr;
	Process_list_info  	*pli_ptr;
	Process_tree_info       *pti_ptr;
	Process_tree_info    	*ppti_ptr; /* Parent process tree info ptr */
	int			 rc;
	sysProcInfo 		*pp;	/* System process info structure   */
	char			*proc_name;
	int			 uid, cur_uid;
	int			 pid, ppid;
	int			 pgrp;
	int                      sid, attachTo;
	Widget			 new_w;


	if ( p == NULL )
		return( NULL );

	if ( p->data == NULL )
		return( NULL );

	DEBUG1(3, "add_process_to_ptree:  (%d)\n", p->key );

	pli_ptr = (Process_list_info *) p->data;

	if ( pli_ptr == NULL )
		return( NULL );

	pp = (sysProcInfo *) pli_ptr->pp;
	if ( pp == NULL )
	{
		/* XXX Shouldn't really create a psuedo sched process on
		 * all cases(i.e. not on linux). However on linux this
		 * code is not executed so it's left as is.
		 */

		if ( p->key == 0 )	/* Manufacture Special processes */
		{
			uid = 0; ppid = 0; pgrp = 0; sid = 0; proc_name = "Sched";
		}
		else if ( p->key == 1 )
		{
			uid = 0; ppid = 0; pgrp = 0; sid = 0; proc_name = "Init";
		}
		else
		{
			DEBUG0(1,"add_process_to_tree: NULL pp\n");
			return( NULL ); /* Can't access system info */
		}
	}
	else    /* Use system process info structure data */
	{
		proc_name = PROCESS_CMD_NAME( pp );
		uid 	  = PROCESS_USER_ID( pp );
		ppid 	  = PROCESS_PARENT_ID( pp );
		sid       = PROCESS_SESSION_ID(pp);
		pgrp 	  = PROCESS_GROUP_LEADER( pp );
	}

	ptr = construct_process_tree_node( p->key, proc_name, pp );
	if ( ptr == NULL )
	{
		DEBUG1(1, "add_process_to_tree: drop new process(%d\n)",p->key);
		return( head );
	}

	pti_ptr = (Process_tree_info *) ptr->data;

	pti_ptr->uid = uid;

	pli_ptr->t_ptr = ptr; /* Sew together Process list and Process Tree */
	pti_ptr->l_ptr = p;

	DEBUG2(3,"add_process_to_tree: Add(%d) ppid(%d)\n",p->key,ppid);

	if ( pp )	/* Real nodes only, not fake 0 or 1 */
	{
		if ( Visible_processes_active )  /* Lock in active list */
			active_lock( p );
	}


	/* If show process groups is set then,
	 * If the parent pid is set to 1 (init) and the
	 * process group leader id is not the same as the
	 * pid, try and attach based on the process group
	 * leader.  For some reason the mousemgr ppid is
	 * 1 and it's process group leader is not set to
	 * the pid of process, so it ends up getting
	 * attached to sched!! This is not correct. So
	 * we add the following condition:
	 *
	 *	If we are trying to attach to a process
	 *	group leader and it's 0, then ignore it
	 *	and use ppid instead.
	 */

/* It would be nice to allow a psuedo user/group node to be
 * displayed, then if a process is identified with a user it can be
 * placed under that user, similarly the group hierarchy could also be
 * shown. So perhaps define, default node, before we call insert_tnode?
 * this however causes problems with out of sequence parent/child nodes,
 * they would get dumped in the wrong place. We could modify the
 * splice loose subtree's to look in the local dump sites for wayward
 * children.
 *
 */


	pid = p->key;

	ptr->parent_key = ppid;

	attachTo = attachToNode( head, pti_ptr, pid, pgrp, sid, ppid, uid, pp );

	if ( head == NULL )
	{
		head = ptr;
		create_proc_widget( treew, NULL, ptr );
	}
	else
	{
		parent = walk_tree( head, match_key, attachTo );
		if ( parent == NULL )
		{
			parent = walk_tree( head, match_key, 1 );
			if ( parent == NULL )
			{
				/* What!! No Init!!! */
				if ( pti_ptr )
					free( pti_ptr );
				free( ptr );

				pli_ptr->t_ptr = NULL;
				return( head );	 /* drop it on the floor */
			}
		}

		new = add_tnode( parent, ptr );
		if ( new == NULL )
		{
			DEBUG1(0, "add_process_to_ptree: add_sort_failed(%d)\n",
					ptr->key );
			return( head );
		}

		/* Create and display the widget */

		ppti_ptr = (Process_tree_info *) ptr->parent->data;
		if ( ppti_ptr->outline == True )
			pti_ptr->outline = True;

		new_w = create_proc_widget( treew, ppti_ptr->node_w, ptr );

		if ( new_w == NULL )
		{
		    pti_ptr->node_w = NULL;   /* Parent hidden ? */
		    pti_ptr->label_w = NULL;
		    DEBUG1(1, "add_process_to_ptree: create_proc failed(%d)\n",
					ptr->key );
		}

		/* If a new session and not in list, add it */

		if ( pp && (PROCESS_ID(pp) == sid) && p->key)
		{
		    if( !match_list(Displayed_session_list, p->key))
		    {
		        DEBUG1(1, "XXXView session(%d)\n", 
					PROCESS_SESSION_ID( pp ) );
		        Displayed_session_list = fwd_insert_lnode(
		    		Displayed_session_list, p->key, NULL );
		    }
		}

		/* Attach any dangling subtree's */

		init_ptr = walk_tree( head, match_key, 1 );

		splice_loose_subtrees( init_ptr, ptr );
	}


	return( head );
}


int
attachToNode( head, pti_ptr, pid, pgrp, sid, ppid, uid, pp )
tnode  *head;
Process_tree_info *pti_ptr;
int    pid;
int    pgrp;
int    sid;
int    ppid;
int    uid;
sysProcInfo *pp;	/* System process info structure   */
{
	int    cur_uid, attachTo_key;
	tnode *parent;
	int	attach_mode;

	pti_ptr->attachType = ATTACH_PARENT;

	attachTo_key = ppid;
	cur_uid = getuid();

	attach_mode = Process_attach_mode;

	/* Smart attach does not work for root desktops/users !!!! */

	if ( cur_uid == 0 && Process_attach_mode == SMART_ATTACH )
		attach_mode = GROUP_ATTACH;

	if ( ppid == 1 ) 
	{
	    attachTo_key = ppid;	/* Use init if we can't find someplace better */
	    switch ( attach_mode ) {

	    	case CHILD_ATTACH: break;

	    	case GROUP_ATTACH: if ( pid != pgrp ) /* Not a group leader itself */
				   {
				       if ( pgrp != 0 )
				       {
					   attachTo_key = pgrp;
					   pti_ptr->attachType = ATTACH_GROUP_LEADER;
				       }
				   }
				   break;

			
	    	case SMART_ATTACH: if (  pid != pgrp ) /* not grp leader */
				   {
				       if ( pgrp != 0 ) /* Try Attach to grp leader */
				       {
				 	   parent = walk_tree( head, match_key, pgrp );
					   if ( parent != NULL )
					   {
					   	attachTo_key = pgrp; 
					        pti_ptr->attachType = ATTACH_GROUP_LEADER;
					   }
					   else /* look for session leader */
					   {
				 	     parent = walk_tree( head, match_key, sid );
					     if ( parent != NULL )
					     {
					   	attachTo_key = sid; 
					        pti_ptr->attachType = ATTACH_SESSION_LEADER;
					     }
					     else 
					     {
						/* try for desktop session */
					       if ( cur_uid == uid )
					       {
					        if ( User_Session_Root == 1 )
					         set_desktop_session_root( cur_uid );

					         attachTo_key = User_Session_Root;
					         pti_ptr->attachType = ATTACH_DESKTOP_SESSION;
					        }
					     }
					   }
				       }
				   }
				   else /* Process is a group leader */
				   {
				   	if ( pid == sid ) /* And a session leader */
					{
					  if ( cur_uid == uid )
					  {
					     if ( User_Session_Root == 1 )
					     	set_desktop_session_root( cur_uid );

					     attachTo_key = User_Session_Root;
					     pti_ptr->attachType = ATTACH_DESKTOP_SESSION;
					  }
					}
					       /* Process is group leader but not session leader */
					else 
					{
				 	     parent = walk_tree( head, match_key, sid );
					     if ( parent != NULL )
					     {
					   	attachTo_key = sid; 
					        pti_ptr->attachType = ATTACH_SESSION_LEADER;
					     }
					     else 
					     {
					       if ( cur_uid == uid )
					       {
					        if ( User_Session_Root == 1 )
					         set_desktop_session_root( cur_uid );

					         attachTo_key = User_Session_Root;
					         pti_ptr->attachType = ATTACH_DESKTOP_SESSION;
					        }
					     }
				        }
				   }

				   break;

		default: /* printf("Unknown attachment type\n"); */
			 attachTo_key = ppid;
			 break;
	    };
	}

	DEBUG1( 1, "attachTo key(%d)\n", attachTo_key );

	return attachTo_key;
}

tnode
*construct_process_tree_node( pid, name, pp )
int   pid;
char *name;
sysProcInfo *pp;		/* System process info structure   */
{
	tnode 		   *new_tnode;
	Process_tree_info  *new_pnode;
	
	new_pnode = (Process_tree_info *) malloc( sizeof( Process_tree_info ));
	if ( new_pnode == NULL )
	{
		perror("construct_process_tree_node: Malloc");
		return(NULL);
	}

	new_pnode->l_ptr         	= NULL;

	/* The following points to the process lists copy of the 
	 * systems process info structure. This may be NULL, for
	 * example the sched and init nodes that are created dynamicly.
	 * Therefore, whoever uses this field should check it before
	 * trying to accesss the sysProcInfo structure.
	 */

	new_pnode->pp         		= pp;  

	new_pnode->outline       	= False;
	new_pnode->selected       	= False;
	new_pnode->display_list 	= NULL;
	new_pnode->display_update_rate  = UPDATE_ON_LONG_TIC;
	new_pnode->state       		= NODE_HIDDEN;
	new_pnode->attachType           = -1;
	new_pnode->uid                  = -1;
	new_pnode->uid       		= -1;
	new_pnode->label_w     		= NULL;
	new_pnode->node_w     		= NULL;
	new_pnode->top_draw_w  		= NULL;
	new_pnode->left_draw_w 		= NULL;
	new_pnode->dialog_state         = NO_DIALOG;
	new_pnode->dialog_w       	= NULL;
	new_pnode->dialog_info_widgets  = NULL;
	new_pnode->dialog_mode  	= NUMERIC_MODE;
	new_pnode->dialog_update_rate  	= UPDATE_ON_REQUEST;
	new_pnode->distinguished  	= 0;
	new_pnode->leader  		= 0;

	new_pnode->stats.last_load  		= -1.0;
	new_pnode->stats.current_load  		= -1.0;
	new_pnode->stats.last_time.tv_sec  	= 0;
	new_pnode->stats.last_time.tv_nsec  	= 0;

	new_pnode->stats.lastSampledAt.tv_sec  	= 0;
	new_pnode->stats.lastSampledAt.tv_usec 	= 0;

	new_pnode->processTipUp		= False;

	/* Create the generalized tree node */

	new_tnode = construct_tnode( pid, name, new_pnode );
	if ( new_tnode == NULL )
	{
		free( new_pnode );
	}

	return( new_tnode );
}


tnode 
*destroy_process_tree_node( head, p )
tnode *head;
tnode *p;
{
	Process_tree_info  *pti_ptr;
	Process_list_info  *pli_ptr;
	lnode		   *l_ptr;
	char		    title_str[80];
	int		    n;
	Arg		    wargs[3];

	if ( p == NULL )
		return( head );

	pti_ptr = (Process_tree_info *) p->data;
	if ( pti_ptr != NULL )
	{
		l_ptr = pti_ptr->l_ptr; /* Points to node in the process list */
		if ( l_ptr != NULL )  
		{
			pli_ptr = l_ptr->data;	 /* Null out lists pointer to */
			if ( pli_ptr != NULL )   /* this node             */
			{
				pli_ptr->t_ptr = NULL;
				pli_ptr->active = 1;	/* remove all locks */
			}

		}

		pti_ptr->display_list = free_list(pti_ptr->display_list); 


		/* If dialog displayed we could display whatever info we still 
		 * have?? For now, just update title to reflect process is
		 * Exited, or no longer displayed.
		 */

		if ( pti_ptr->dialog_w )
		{
		  sprintf(title_str,"Process (%s:%d) Exited?",
							p->str_key,p->key);
		  n = 0;
		  XtSetArg( wargs[n], XmNtitle, title_str); n++;
		  XtSetValues( XtParent(pti_ptr->dialog_w), wargs, n );
		}
		
		if ( pti_ptr->dialog_info_widgets != NULL )
			free( pti_ptr->dialog_info_widgets );

		free( pti_ptr );
		p->data = NULL;
	}

	head = remove_tnode( head, p );

	return( head );
}

lnode
*delete_processes_from_plist( plist, del_list )
lnode *plist;
lnode **del_list;
{
	lnode *p;
	lnode *q;
	lnode *last_p;
	lnode *last_q;
	lnode *next_q;
	lnode *new_del_list;
	Process_list_info *pli_ptr;
	int drop_list_entry;
	sysProcInfo *pp;

	/* We know that both lists are reverse sorted, so we make use of this
	 * as we process the delete list
	 */

	p = plist;     last_p = plist;
	q = *del_list; last_q = *del_list; new_del_list = *del_list;
	while( q )
	{
		for ( ; p != NULL ; p = p->next )
		{
			if ( p->key == q->key )
				break;
			last_p = p;
		}

		if ( p == NULL )  /* Trying to delete node not in list!!! */
		{
			/* This happen's if we couldn't load the node info
			 * and we dropped the process in the first place, 
			 * so we drop it from the delete list as there is 
			 * naught to do.
			 */

	 	        DEBUG1(1,"delete_process_from_plist: p(%d)\n", p );

			if ( q == new_del_list )	/* Remove from head */
			{
				new_del_list = q->next;
				last_q = new_del_list;
				next_q = new_del_list;
			}
			else
			{
				/* Last_q stay's the same */

				last_q->next = q->next;
				next_q       = q->next;
			}

			free( q );

			q = next_q;

			p = plist;  last_p = plist;  /* Set p to start of list*/
			continue;
		}
		else
		{
			pli_ptr = ( Process_list_info * ) p->data;

			if ( pli_ptr == NULL )
			{
			  DEBUG1(1, 
			  "delete_process_from_plist: p(%d) has NULL pli_ptr\n",
								p->key );
			}
			else
			{
			  pp = (sysProcInfo *)pli_ptr->pp;
			  if ( pp == NULL )
			  {
			    DEBUG1(1, 
			    "delete_process_from_plist: p(%d) has NULL pp\n",
								p->key );
			  }
			  else
			  {

			    /* if monitoring active users
			     *	decrement count on uid/gid
			     *	if need be remove from active_uid/gid list
			     *
			     */
  			    if ( Users->monitor_users )
			    {
	  			Users->active_user_list = 
				   dec_list_count( Users->active_user_list,
						PROCESS_USER_ID( pp ) );
	  			drop_list_entry = _drop_list_entry;
	  			Users->active_group_list = 
				   dec_list_count( Users->active_group_list,
						PROCESS_GROUP_ID( pp ) );

	  			if ( drop_list_entry || _drop_list_entry )
					display_user_group( Users );
			    }
			  }
			}

			/* Set data pointer of the del_list node to the
		         * corresponding tree node. This is used and nulled in
			 * delete_process_from_ptree
			 */
			
			if ( pli_ptr != NULL )
				q->data   = pli_ptr->t_ptr;

			if ( p == plist )  /* Remove from front of list */
			{
				plist = p->next;
				last_p = plist;
			}
			else 		   /* Remove from middle or end */
				last_p->next = p->next;

			destroy_process_list_node( p );

			p = last_p;  /* Set p to valid location in list */
		}
	
		last_q = q;
		q = q->next;
	}

	*del_list = new_del_list;

	return( plist );
}

lnode
*add_processes_to_plist( plist, add_list )
lnode *plist;
lnode **add_list;
{
	lnode *p;
	lnode *q;
	lnode *next_q;
	lnode *last_q;
	lnode *last_p;
	lnode *new_addlist=NULL;
	Process_list_info  *pli_ptr;
	sysProcInfo	*pp;		/* System process info structure   */
	lnode *new_node;
	int    add_list_entry;
	int    uid, gid;
	dev_t  tty_dev;

	/* We know that both lists are reverse sorted, so we make use of this
	 * as we process the add list. New process's are most likely added to
	 * the front of the list, except when pid's wrap arround( a very
	 * infrequent occurence ). As part of the processing we set the data 
	 * pointer of the add list node to the new process list 
	 * node(this is used in update_active_list  and kept when the 
	 * add list is merged with the active process list).  We also keep 
	 * track of user/group id's and if the user/group display is viewing 
	 * the active user/groups then we call a routine to update it's display
	 * to reflect any changes.
	 * 
	 */

	p = plist; last_p = p; 

	new_addlist = *add_list; last_q = *add_list;

	Machine->cache_process_details( Machine, new_addlist );

	for ( q = *add_list ; q != NULL; q = next_q )
	{
		/*  Construct new process_info node and load process info */

		new_node = construct_process_list_node( q->key );
		if ( new_node == NULL )
		{
			DEBUG1(1, "add_to_process_list: Ignore(%d)\n", q->key);

			next_q = q->next;     /* Drop em like hot cakes */

			if ( q == new_addlist )	/* Remove from head of list */
			{
				new_addlist = next_q;
				last_q = new_addlist;
			}
			else	/* Remove from middle or end */
			{
				last_q->next = next_q;
			}

			free( q );
			continue;
		}

		/* Set data pointer of the add_list node to the
		 * new process list node. Note this list is merged into
		 * the active_process_list.
		 */
		last_q  = q;
		next_q  = q->next;
		q->data = (void *)new_node;


		/* Now insert new_node into process list  	*/

		if ( p == NULL )		/* Head of list */
			plist = new_node;
		else				/* Find position in list */
		{
		    for ( /* Start at current loc*/; p != NULL ; p = p->next )
		    {
		    	if ( q->key > p->key )
				break;
			last_p = p;
		    }

		    if ( p == NULL )	  		/* Append to list */
			last_p->next = new_node;
		    else if ( p == plist )	  	/* Insert at head */
		    {
			new_node->next = plist;
			plist = new_node;
		    }
		    else				/* In the middle  */
		    {
			last_p->next   = new_node;
			new_node->next = p;
		    }
		}

		p = new_node; last_p = new_node;   /* Start Search @ new node */

		uid = 0; gid = 0;  /* To allow sched, init */
		tty_dev = -1;

		pli_ptr = (Process_list_info *)new_node->data;
		if ( pli_ptr != NULL )
		{
			pp = (sysProcInfo *) pli_ptr->pp;
			if ( pp )
			{
				uid = PROCESS_USER_ID( pp ); 
				gid = PROCESS_GROUP_ID( pp ) ;
				tty_dev = PROCESS_CONTROL_TTY( pp );
			}
		}

                /* Ensure passwd entry loaded and node created for userid */

		check_for_new_userid( Users, uid );

  		if ( Users->monitor_users )
		{
	  		Users->active_user_list = 
				   inc_list_count( Users->active_user_list,uid);
	  		add_list_entry = _add_list_entry;

	  		Users->active_group_list = 
				  inc_list_count( Users->active_group_list,gid);

	  		if ( add_list_entry || _add_list_entry )
				display_user_group( Users );
	        }
	}

	*add_list = new_addlist;

	return( plist );
}

lnode
*construct_process_list_node( pid )
int pid;
{
	lnode 		   *new_lnode;
	Process_list_info  *new_pnode, *p;
	sysProcInfo	   *pp;		/* System process info structure   */
	int                 ppid, pgrp, sid;
	
	new_pnode = (Process_list_info *) malloc( sizeof( Process_list_info ));
	if ( new_pnode == NULL )
	{
		perror("construct_process_list_node: Malloc");
		return(NULL);
	}

	new_pnode->t_ptr = NULL;
	new_pnode->pp    = NULL;
	new_pnode->euid    = 0;
	new_pnode->egid    = 0;
	new_pnode->active  = 2;	/* -1 locked in, 1-n long tic's    */

	/* Note we set active to 2, to ensure that the process remains in the
	 * active list for at least one long tic, and at most a short tic
	 * shy of two long tic's.
	 */

	pp = construct_sys_proc_info( pid, new_pnode );
	if ( pp != NULL )
	{
		new_pnode->pp  = pp;
	 	DEBUG1(7,"construct_process_list_node: loaded pp for(%d)\n",
									pid);
	}
	else	/* Handle, special case for Sched and init */
	{
		if ( pid == 0 || pid == 1 )
		{
			new_pnode->pp = NULL; /*Create node for sched and init*/
	 	        DEBUG0(7,"construct_process_list_node: 0 or 1\n" );
		}
		else
		{
	 	  DEBUG1(1,"construct_process_list_node: failed pp(%d)\n", pp );
			free( new_pnode );
			return( NULL );
		}
	}

	new_lnode = construct_lnode( pid, new_pnode );
	if ( new_lnode == NULL )
	{
	 	DEBUG1(1,"construct_process_list_node: const_lnode %d\n", pid );
		if ( pp )
		{
			free( pp );
#ifdef LINUX
			if ( pp->pr_psargs != NULL )
				free( pp->pr_psargs );
#endif

		}
		free( new_pnode );
	        return( NULL );
	}

        p              = new_pnode;
        p->attachments = NULL;
        p->hasWork     = FALSE;
        if ( pp )
        {
          if ( pid != 0 && pid != 1 )           /* ignore sched and init */
          {
            ppid    = PROCESS_PARENT_ID( pp );
            pgrp    = PROCESS_GROUP_LEADER( pp );
            sid     = PROCESS_SESSION_ID(pp);

            if ( sid != pid && sid != ppid && sid != pgrp && sid != 1 )
                p->attachments = push_attachment( p->attachments, pid, sid );

            if ( pgrp != pid && pgrp != ppid && pgrp != 1 )
                p->attachments = push_attachment( p->attachments, pid, pgrp );

            p->attachments = push_attachment( p->attachments, pid, ppid );

            /* Now add this process to the attachment validator work
             * list. During idle times the work list is scanned and
             * the attachments of each process in the list are validated and
             * time stamped. Note: If a process is attached to the tree immediately
             * then some of the attachments will be validated at that time. However
             * some may not and hence the remainder of attachments are validated
             * during the work cycle. Also the process nodes are not constructed
             * in forward sorted order so doing the attachment validating now
             * would result in a lot of failures.
             */

            p->hasWork = TRUE;
            Attachment_validator = add_work_task( Attachment_validator, new_lnode );

            p->notifyOnDeath = NULL;
          }
        }

        add_to_hash_table( new_lnode );

	return( new_lnode );
}



lnode *
push_attachment( list, pid, attachTo )
lnode *list;
int pid;
int attachTo;
{
	Attach_point *atp;

	atp = (Attach_point *) malloc( sizeof( Attach_point ) );
	if ( atp == NULL )
	{
		fprintf( stderr, "Error out of memory!!! process attachments will fail for(%d)!\n", pid);
		return list;
	}

	atp->attachToPid        = attachTo;
	atp->startTime.tv_sec   = 0;

	return  push_lnode( list, pid, atp );
}

lnode *
add_work_task( list, ptr )
lnode *list;
lnode *ptr;
{
	DEBUG1(1, "Add work task(%d)\n", ptr->key );

	list = append_lnode( list, ptr->key, ptr );

	return list;
}

lnode *
remove_work_task( list, id )
lnode *list;
int id;
{
	DEBUG1(1, "Find and if it exits, Remove work task(%d) \n", id );
	list = remove_lnode( list, id );

	return list;
}

lnode *
lookup_process_lnode( pid )
{
	return find_in_hash_table( pid );
	/* return find_lnode( Process_list, pid ); */
}

perform_attachment_validations()
{
	int 			num=4;   /* number of nodes to check per dispatch */
	lnode  			*p, *lp, *tp;
	Process_list_info 	*pli, *tpli;
	Attach_list  		at;
	Attach_point		*atp;
	int			noMatch;

	if ( debug > 0 )
	    print_list( "Verify Attachments for", Attachment_validator );

	while ( Attachment_validator )	/* Go through work list */
	{
	  p = Attachment_validator;
	
	  lp = (lnode *) p->data;
	  if ( lp )
	    pli = (Process_list_info *) lp->data;
	  else
	  {
	    DEBUG1(1, "Attachment points to NULL process list node! (%d)\n", p->key );
	    pli = NULL;
	  }

	  if ( pli != NULL )
	  {
	    at = pli->attachments;
	    DEBUG1(1, "Processing (%d)\n", at->key );
	    while( at )
	    {
		noMatch = 1;
		atp = (Attach_point *) at->data;
		if ( atp )
		{
		  if ( atp->attachToPid < 0 )  /* marked as invalid */
		    noMatch = 0;
		  else
		  {
		   /* we check again even if start timestamp already set, this
		    * allows us to use this routine repeatedly and to handle
		    * the rare condition of an attachment dying and being replaced
		    * by a process of the same pid in the same sample.
		    */

		   tp = lookup_process_lnode( atp->attachToPid );
		   if ( tp )
		   {
		     DEBUG1( 1, "\tChecking (%d)\n", atp->attachToPid );
		     tpli = (Process_list_info *) tp->data;
		     if ( tpli )
		     {
		        if ( tpli->pp )
			{
			  /* add some hueristics to validate as could have some older
			   * processes appear as valid when if fact they are not!
			   */

			   DEBUG1(1, "\tTimestamped - attachTo point(%d) as valid\n",
						atp->attachToPid );
			
		     	  if ( atp->startTime.tv_sec  )
			  {
		     	     if ( atp->startTime.tv_sec  != 
					PROCESS_START_TIME(tpli->pp).tv_sec )
			     {
			        DEBUG2(1, 
				  "Attachment for Pid(%d) i.e. (%d) now has different startTime!\n",
						p->key, atp->attachToPid );
			     	atp->attachToPid = 0 - atp->attachToPid;
			     }
				
			  }
			  else
			  {

		     	    atp->startTime.tv_sec  = PROCESS_START_TIME(tpli->pp).tv_sec;
		     	    atp->startTime.tv_nsec = PROCESS_START_TIME(tpli->pp).tv_nsec;

			    /* regiser to be notified on process death - 
			     * so we can mark the attachment as not valid 
			     */

			    registerForDeathNotification( at->key, atp->attachToPid );
			  }

			  noMatch = 0;
			}
		     }
		   }
		  }
	        }

		if ( noMatch )
		{
		   DEBUG1(1, "\tIgnore - attachToPid(%d) is not valid!\n",
			atp->attachToPid );
		   atp->attachToPid = 0 - atp->attachToPid; /* ignore as possible attachment */
		}
		    
		at = at->next;
	    }
	  }

	  Attachment_validator = remove_work_task( Attachment_validator, 
						p->key );

	  pli->hasWork = FALSE;

	  if ( --num <= 0 )
		break;
	}
}

int
registerForDeathNotification( notifyMe, whenIDie )
int notifyMe;
int whenIDie;
{
	lnode *l;
	Process_list_info *pli;

	l = find_in_hash_table( whenIDie );

	/* Lookup process to monitor, add a notifyOnDeath request
         * to it's notifyOnDeath list.
         */

	if ( l )
	{
	  pli = ( Process_list_info *) l->data;
	  if ( pli )
	  {
	    DEBUG2( 1, "\tProcess(%d) registered for death notification of(%d)\n",
					notifyMe, whenIDie );

	    pli->notifyOnDeath = push_lnode( pli->notifyOnDeath,
					notifyMe, NULL );

	  }
	}
}

performDeathNotification( lptr, iDied )
lnode    *lptr;
int	  iDied;
{
	lnode 		  *l;
	Process_list_info *pli;
	Attach_point      *atp;

	while ( lptr )
	{
	  /* Lookup process to notify, then alter any attachment records
           * so it don't try to attach to that process again.
           */

DEBUG2(1, "Looking to notify(%d) of process(%d)'s death\n", lptr->key, iDied );

	  l = find_in_hash_table( lptr->key );

	  if ( l )
	  {
	    pli = (Process_list_info *) l->data;
	    if ( pli )
	    {
	      l = pli->attachments;
	      while ( l )
	      {
		atp = (Attach_point *) l->data;
		if ( atp )
		{
		  if ( atp->attachToPid == iDied )
		  {
		    DEBUG2(1, "Process (%d) - marking(%d) as invalid attachment point\n",
				lptr->key, iDied ); 
		    atp->attachToPid = 0 - atp->attachToPid;   /* negative values are ignored */
		  }
	        }
		l = l->next;
	      }
	    }
	  }
	  lptr = lptr->next;
	}
}


void
destroy_process_list_node( p )
lnode *p;
{
	Process_list_info    *pli_ptr;
	Process_tree_info    *pti_ptr;
	tnode		     *t_ptr;

	if ( p == NULL )
		return;

	pli_ptr = (Process_list_info *) p->data;
	if ( pli_ptr != NULL )
	{
		t_ptr = pli_ptr->t_ptr; /* Points to node in the process tree */
		if ( t_ptr != NULL )  
		{
			pti_ptr = (Process_tree_info *)t_ptr->data;
			if ( pti_ptr != NULL )   
			{
				/* Null out tree's pointer to this list node  */
				pti_ptr->l_ptr = NULL;
				pti_ptr->pp    = NULL;
			}
		}
	
		if ( pli_ptr->pp != NULL )
		{
			free( pli_ptr->pp );
#ifdef LINUX
			if ( pli_ptr->pp->pr_psargs != NULL )
				free( pli_ptr->pp->pr_psargs );
#endif
		}

                if ( pli_ptr->hasWork )
                        Attachment_validator = remove_work_task( Attachment_validator,
                                                p->key );

                if( pli_ptr->notifyOnDeath )
                {
                        performDeathNotification( pli_ptr->notifyOnDeath, p->key );
                        free_list( pli_ptr->notifyOnDeath );
                }

                if ( pli_ptr->attachments )
                        free_list( pli_ptr->attachments );

		free( pli_ptr );
		p->data = NULL;
	}

	del_from_hash_table( p->key );

	destroy_lnode( p );
}

sysProcInfo
*construct_sys_proc_info( key, new_pnode )
int key;
Process_list_info  *new_pnode;
{
	sysProcInfo	*pp;	/* System process info structure   */
	int		 rc;

	pp = (sysProcInfo *) malloc( sizeof(sysProcInfo) );
	if ( pp == NULL )
	{
		perror("construct_sys_proc_info: Malloc\n");
		return( NULL );
	}

	/* The pp structure is filled in by the host specific routine
	 * load_sys_proc_info(). On linux the psargs is dynamically allocated
	 */

#ifdef LINUX
	pp->pr_psargs = NULL;
	pp->last_time.tv_nsec = 0;
	pp->last_time.tv_sec  = 0;
#else
	pp->pr_psargs[0] = NULL_CHAR;
#endif

	new_pnode->pp = pp;

        if ( Machine->scanner_type == INTERNAL_POLL_SCANNER ) 
            rc = load_sys_proc_info(key, new_pnode );
        else
            rc = load_ext_proc_info( Machine, key, new_pnode );
	
	if ( rc == IGNORE_PROCESS )
	{
	 	DEBUG1(1,"construct_sys_proc_info: load failed %d\n", key );
#ifdef LINUX
		if ( pp->pr_psargs != NULL )
			free ( pp->pr_psargs );
#endif
		free( pp );
		new_pnode->pp = NULL;
	}

	return( new_pnode->pp );
}


int
load_ext_proc_info( machine, pid, pli_ptr )
Machine_info *machine;
int pid;
Process_list_info  *pli_ptr;
{
	sysProcInfo	   *pp;		/* System process info structure   */
	int		   i,k, len, nv;
	VarList		   v=NULL;
	char		   buf[128];
	char		   *str, *str2;


printf("load_ext_proc_info for machine(%d) (%d) pli_ptr(%d)\n", machine, pid, pli_ptr );

	if ( machine == NULL ) return IGNORE_PROCESS;
	if ( pli_ptr == NULL ) return IGNORE_PROCESS;
	if ( pli_ptr->pp == NULL ) return IGNORE_PROCESS;

	pp = pli_ptr->pp;

	/* Fetch values using gei interface and convert to internal structure */

	nv = machine->get_pid_main_details( machine, pid, &v );

printf("get_main_details returned (%d) variables\n", nv );

	if ( nv < 0 )
		return IGNORE_PROCESS;

	if ( nv < 23 )   /* Not enough values */
	{
		freeVarList( v, 23 );
		return IGNORE_PROCESS ;
	}


	PROCESS_SNAME( pp ) = v[0].value[0];
	PROCESS_NICE( pp )  = atoi( v[1].value );

	sprintf( buf, "0x%s", v[2].value );

	PROCESS_FLAG( pp )           = strtol( buf, NULL, 16 );
	PROCESS_USER_ID( pp )        = atoi( v[3].value );
	PROCESS_GROUP_ID( pp )       = atoi( v[4].value );
	PROCESS_ID( pp )             = atoi( v[5].value );
	PROCESS_PARENT_ID( pp )      = atoi( v[6].value );
	PROCESS_GROUP_LEADER(pp)     = atoi( v[7].value );
	PROCESS_SESSION_ID( pp )     = atoi( v[8].value );

	sprintf( buf, "0x%s", v[9].value );

	PROCESS_ADDRESS( pp )        = (caddr_t) strtol( buf, NULL, 16 );

	PROCESS_SIZE( pp )                 = atoi( v[10].value );
	PROCESS_RESIDENT_SET_SIZE( pp )    = atoi( v[11].value );

	sprintf( buf, "0x%s", v[12].value );

	PROCESS_WCHAN( pp )                = (caddr_t) strtol( buf, NULL, 16 );

	PROCESS_START_TIME(pp).tv_sec    = atoi( v[13].value );
	PROCESS_START_TIME(pp).tv_nsec   = 0;

	PROCESS_CPU_TIME(pp).tv_sec      = atoi( v[14].value );
	PROCESS_CPU_TIME(pp).tv_nsec     = atoi( v[15].value );

	PROCESS_PRIORITY( pp )             = atoi( v[16].value );

	PROCESS_CONTROL_TTY( pp )          = atoi( v[17].value );

	strcpy( PROCESS_CLASS_NAME( pp ), v[18].value );
          
	if ( v[19].value )
	    strcpy( PROCESS_CMD_NAME( pp ), v[19].value );

#ifdef LINUX
	PROCESS_CMD_ARGS( pp )             = strdup( v[20].value );
#else
	len = strlen( v[20].value );
	if ( len > 79 )
	    v[20].value[79] = '\0';
	strcpy( PROCESS_CMD_ARGS( pp ), v[20].value);
#endif

	PROCESS_EFFECTIVE_USER_ID( pli_ptr )  = atoi( v[21].value );
	PROCESS_EFFECTIVE_GROUP_ID( pli_ptr ) = atoi( v[22].value );

/*

Need to synthesize just ran state

Ensure load is calculated properly 

Do kde name magic

*/

	printf( "Loaded process(%d) ppid(%d)\n", 
				PROCESS_ID( pp ),
				PROCESS_PARENT_ID( pp ) );

	freeVarList( v, 23 );

	return ADD_PROCESS;
}

freeVarList( v, listLen )
VarList v;
int listLen;
{
	int i;

printf("freeVarList: v(%d) (%d)\n", v, listLen );

	if ( !v ) return -1 ;

	for ( i = 0; i < listLen; i++ )
	{
	    if ( v[i].name )  free( v[i].name );
	    if ( v[i].value ) free( v[i].value );
	}

	free( v );

	return 0;
}

tnode
*splice_loose_subtrees( branch, new_node )
tnode *branch;
tnode *new_node;
{
	tnode 		*loose_subtree;
	child_ptr	*cptr;
	child_ptr	*last_ptr;
	int  		n;
	Arg  		wargs[2];
	Process_tree_info  *pti_ptr;
	Process_tree_info  *lpti_ptr;   /* Loose subtree info pointer */
	sysProcInfo	   *pp;

	/* check to see if any of the subtree's dangling off of the
	 * branch(i.e. init) should be attached to the new node. If so we disconnect 
	 * them from the branch(init) and attach them to the new_node.
	 */

	/* XXX This could be more efficient, i.e. create a list of subtrees that
	 * may be re-attachable and then just search that. For each added node it could
	 * save 20 or more node comparisons. Most importantly during switches between
	 * decorated and not.
	 */

	for ( cptr = branch->children ; cptr != NULL ;  )
	{
		DEBUG3(1, "splice_loose_subtrees: check if process(%d) ppid(%d) should attach to (%d)\n",
                                cptr->child->key,
                                cptr->child->parent_key,
                                new_node->key );

		if ( cptr->child->parent_key == new_node->key )
		{
			DEBUG2(3, "splice_loose_subtrees: attach(%d) to (%d)\n",
				cptr->child->key, new_node->key );

			last_ptr = cptr;
			cptr 	 = cptr->next;

			loose_subtree = extract_tnode( last_ptr->child );

			add_tnode( new_node, loose_subtree );

			pti_ptr  = (Process_tree_info *) new_node->data;
			lpti_ptr = (Process_tree_info *) loose_subtree->data;

			if ( pti_ptr == NULL || lpti_ptr == NULL )
				continue;

			if ( lpti_ptr->state == NODE_HIDDEN )
				continue;

			n = 0;
			XtSetArg(wargs[n],XtNparentWidget,pti_ptr->node_w); n++;
			XtSetValues( lpti_ptr->node_w, wargs, n );
		}
		else
		{
			cptr 	 = cptr->next;
		}
	}

	return( new_node );
}




lnode
*inc_list_count( list, key )
lnode *list;
int    key;
{
	lnode *p;
	lnode *last_p;
	
	_add_list_entry=0;
	last_p = list;
	for ( p = list; p != NULL; p = p->next )
	{
		if ( p->key == key )
		{
			p->data = (void *) (((int )p->data) + 1);
			break;
		}

		last_p = p;
	}

	if ( p != NULL )
		return( list );

	/* add new node */

	list = fwd_insert_lnode( list, key, 1 );
	_add_list_entry=1;

	return( list );
}

lnode
*dec_list_count( list, key )
lnode *list;
int    key;
{
	lnode *p;
	lnode *last_p;
	
	_drop_list_entry=0;
	last_p = list;
	for ( p = list; p != NULL; p = p->next )
	{
		if ( p->key == key )
		{
			p->data = (void *) (((int)p->data) - 1);
			break;
		}

		last_p = p;
	}

	if ( p == NULL )
		return( list );

	if ( ((int)p->data) <= 0 )	/* Remove node */
	{
		p->data = NULL;		/* So we don't try and free it */
		list = delete_lnode( list, key );
		_drop_list_entry=1;
	}

	return( list );
}


lnode
*zero_free_list( list )
lnode *list;
{
	lnode *p, *next_p;

	for ( p = list; p != NULL ; p = next_p )
	{
		next_p = p->next;
		p->data = NULL;
		free( p );
	}

	return( NULL );
}

print_list_count( label, list )
char  *label;
lnode *list;
{
	lnode *p;
	
	printf("%s: ", label);
	for ( p = list; p != NULL ; p = p->next )
		printf("%d(%d),", p->key, p->data );
	printf("\n");
}



monitor_process( users, pid, pp, gid, uid )
User_group_info  *users;
int  pid;
sysProcInfo *pp;
int  gid;
int  uid;
{
	int view_group=False;
	int view_user=False;
	int view_process=False;

	if ( pid == 0 || pid == 1 )
		return( True );

	if ( Group_view == VIEW_GUTREE )
	{
		view_group = match_list( users->group_display_list, gid ); 
	}

	if ( User_view == VIEW_LOGGED_IN_USER )
	{
		if ( users->user_uid == uid )
			view_user = True;
	}
	else if ( User_view == VIEW_ROOT_USER )
	{
		if ( uid == 0 )
			view_user = True;
	}
	else if ( User_view == VIEW_UUCP_USER )
	{
		if ( uid == 5 )
			view_user = True;
	}
	else if ( User_view == VIEW_GUTREE  )
	{
		view_user = match_list( users->user_display_list, uid );
	}
	else 
	{
		view_user = True;
	}

	if ( Group_view == VIEW_IGNORE_GROUPS ) 
	{
		view_process = view_user;   /* User setting decides */
	}
	else	/* Either criteria will select it */
	{
		if ( view_group || view_user )
			view_process = True;
	}

	if ( pp )
	{
		if ( view_process == False )  /* Maybe if it's in session */
		{
		    if ( Show_descendants == SHOW_ALL_DESCENDANTS )
		    {
			/* If the session id of the process is in the 
			 * displayed session list then display it.
			 */

			if ( PROCESS_SESSION_ID( pp ) != 0 )  
			{
			    view_process = match_list( Displayed_session_list, 
							PROCESS_SESSION_ID( pp ) );
			}
		    }
		}

		if ( Show_daemons == False ) 
		{
			if ( PROCESS_CONTROL_TTY(pp) == -1 )  /* Filter out daemons */
			{
				view_process = False;
			}
		}

	}
	return( view_process );
}


Widget
create_proc_widget( treew, parent_widget, t_ptr )
Widget treew;
Widget parent_widget;
tnode *t_ptr;
{
	int n;
	Arg wargs[16];
	Process_tree_info  *pti_ptr;
	Process_tree_info  *ppti_ptr;
	Object_color       *obj;
	lnode		   *p;
	int	            uid;
	Pixel	            fg;
	Pixel	            bg, label_fg, label_bg;
	sysProcInfo	   *pp;
	Widget		    tree_node_formw, left_draw, top_draw;
	Widget		    node_labelw;
	int		    saved_color_coding;
	int		    pid, sid_pid, grp_pid;
	Dimension  	    left_draw_width=0;
	Dimension  	    top_draw_width=0;

	if ( t_ptr == NULL )
		return( NULL );

	pti_ptr  = (Process_tree_info *) t_ptr->data;

	/* If parent is hidden ignore request */

	if ( t_ptr->parent )
	{
		ppti_ptr = (Process_tree_info *) t_ptr->parent->data;
		if ( ppti_ptr->state == NODE_HIDDEN )
		{
		   DEBUG0(1, "create_proc_widget: Ignore parent hidden!\n");
		   return( NULL );
		}
	}

	if ( pti_ptr->state == NODE_DISPLAYED || pti_ptr->node_w )
	{
		DEBUG0(1, "create_proc_widget: Widget already exists!\n");
		return( NULL );
	}

	get_node_colors( User_colors, Group_colors, t_ptr, &fg, &bg );  

	n = 0;
	XtSetArg( wargs[n], XtNforeground, fg ); n++;
	XtSetArg( wargs[n], XtNbackground, bg ); n++;
	XtSetArg( wargs[n], XtNbackgroundFill, bg ); n++;
	XtSetArg( wargs[n], XtNparentWidget, parent_widget ); n++;
	XtSetArg( wargs[n], XtNkey, t_ptr->key ); n++;

	/* To squeeze the text inside the process display node
	 * XtSetArg( wargs[n], XmNmarginWidth, 0 ); n++;
	 * XtSetArg( wargs[n], XmNmarginHeight, 0 ); n++;
	 */

	/* 
	 * XtSetArg( wargs[n], XmNhighlightThickness, 0 ); n++;
	 * XtSetArg( wargs[n], XmNshadowType, XmSHADOW_OUT ); n++;
	 * XtSetArg( wargs[n], XmNshadowType, XmSHADOW_IN ); n++;
	 * pti_ptr->w = XtCreateManagedWidget("node", xmDrawnButtonWidgetClass, 
	 *
	 */

        pp = pti_ptr->pp;	

	if ( Decorated_tree == DECORATE_DISTINGUISHED )
	{
            if ( pp )
	    {
	        if ( ( pti_ptr->stats.current_load > 0.0 ) ||
	  		      ( PROCESS_SNAME(pp) != 'S' ) ||
	  		      ( PROCESS_PRIORITY(pp) < Priority_Normal_Min)  ||
	  		      ( PROCESS_PRIORITY(pp) > Priority_Normal_Max)  )
	        {
	 	    pti_ptr->distinguished = Decoration_fame_time;

		    DEBUG2( 7, "Starting as distinguished(%d:%s)\n", 
		    				t_ptr->key, t_ptr->str_key );
	        }
	    }
	}
	else if ( Decorated_tree == DECORATE_ALL )
	{
	      pti_ptr->distinguished = -1;
	}

	if ( Decorated_tree )
	{
	    /* Create an outer composite widget */

	    tree_node_formw = XtCreateManagedWidget("node", xmFormWidgetClass,
						treew, wargs, n );

	    saved_color_coding = Color_based_on;
	    
	    label_bg = bg;
	    label_fg = fg;

	    Color_based_on = Decoration_left_color;
	    get_node_colors( User_colors, Group_colors, t_ptr, &fg, &bg );  


	    if ( pp )
	    {
	        sid_pid = PROCESS_SESSION_ID(pp);
	        grp_pid = PROCESS_GROUP_LEADER(pp);
		pid     = PROCESS_ID(pp);
	    }
	    else
	    {
	        pid = -1; sid_pid = -2; grp_pid = -2;
	    }


	    if ( pid == grp_pid )
	    {
	        left_draw_width = Leader_bar_size + (Leader_bar_spacing*2);
		pti_ptr->leader = 1;
	    }

	    if ( pid == sid_pid )
	    {
	        left_draw_width = (Leader_bar_size * 2) +
						(Leader_bar_spacing * 3);
		pti_ptr->leader = 1;
	    }

	    /* Decoration_left_size = left_draw_width; */

 	    n = 0;
	    XtSetArg( wargs[n], XmNwidth, left_draw_width ); n++;


	    if ( ( ! pti_ptr->leader ) && ( ! pti_ptr->distinguished ) )
	    {
	        XtSetArg( wargs[n], XtNbackground, label_bg ); n++;
	    }
	    else
	    {
	        XtSetArg( wargs[n], XtNbackground, bg ); n++;
	    }

	    XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	    XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;

	    left_draw=XtCreateManagedWidget("node_left",
	    		xmDrawingAreaWidgetClass, tree_node_formw, wargs, n);

	    XtAddCallback( left_draw, XmNexposeCallback,
                                left_leader_exposures, t_ptr );

	    Color_based_on = Decoration_top_color;
	    get_node_colors( User_colors, Group_colors, t_ptr, &fg, &bg );  

 	    n = 0;

	    if ( pti_ptr->distinguished )
	    {
	        XtSetArg( wargs[n], XtNheight, Decoration_top_size ); n++;
	    }
	    else
	    {
	        XtSetArg( wargs[n], XtNheight, 0 ); n++;
	    }

	    if ( ( ! pti_ptr->leader ) && ( ! pti_ptr->distinguished ) )
	    {
	        XtSetArg( wargs[n], XtNbackground, label_bg ); n++;
	    }
	    else
	    {
	        XtSetArg( wargs[n], XtNbackground, bg ); n++;
	    }

	    top_draw_width = Decoration_spacing +
				(( Decoration_size + Decoration_spacing ) * 
	    				( Decoration_count ));

	    XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg( wargs[n], XmNleftWidget, left_draw); n++;
	    XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg( wargs[n], XtNwidth, top_draw_width ); n++;
	    top_draw=XtCreateManagedWidget("node_top",xmDrawingAreaWidgetClass,
	                                          tree_node_formw, wargs, n);


	    XtAddEventHandler(top_draw, ButtonPressMask,
						FALSE, select_node, t_ptr);

	    XtAddCallback( top_draw, XmNexposeCallback,
                                top_decoration_exposures, t_ptr );

	    XtAddCallback( top_draw, XmNresizeCallback,
                                decoration_resize, t_ptr );


	    Color_based_on = saved_color_coding;

	    n = 0;
	    XtSetArg( wargs[n], XmNmarginHeight, 0 ); n++;
	    XtSetArg( wargs[n], XmNmarginBottom, 2 ); n++;
	    XtSetArg( wargs[n], XtNforeground, label_fg ); n++;
	    XtSetArg( wargs[n], XtNbackground, label_bg ); n++;
	    XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
	    XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg( wargs[n], XmNtopWidget, top_draw); n++;
	    XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg( wargs[n], XmNleftWidget, left_draw); n++;
	    XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	    node_labelw = XtCreateManagedWidget("node", xmLabelWidgetClass,
					tree_node_formw, wargs, n );

	    pti_ptr->label_w     = node_labelw;
	    pti_ptr->node_w      = tree_node_formw;
	    pti_ptr->top_draw_w  = top_draw;
	    pti_ptr->left_draw_w = left_draw;
	}
	else
	{
		XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
		pti_ptr->label_w = XtCreateManagedWidget("node", 
					xmLabelWidgetClass, treew, wargs, n );

		pti_ptr->node_w = pti_ptr->label_w;
	}

	if ( AppData.showProcessTips )
	        process_tip( pti_ptr );

	pti_ptr->state = NODE_DISPLAYED;

	display_proc_info( t_ptr, 0 );

	XtAddEventHandler(pti_ptr->label_w, ButtonPressMask,
						FALSE, select_node, t_ptr);

	/* XtAugmentTranslations( pti_ptr->label_w, Node_Trans_table ); */


	DEBUG1(9, "create_proc_widget: finished(%d)!\n", t_ptr->key);

#ifdef SVR4_MP
	if ( DisplayLWP == True )
		addLWPnodes( treew, t_ptr );
#endif
	return( pti_ptr->node_w );
}

#ifdef SVR4_MP

#define MAX_PATH 256

int
addLWPnodes( treew, t_ptr )
Widget treew;
tnode *t_ptr;
{
	int n;
	Arg wargs[16];
	Process_tree_info  *pti_ptr;
	sysProcInfo	   *pp;
	Widget		    parent_widget;
	Widget		    lwp_w=0;
	int 		first, pid, ms_sec, len, i, rc, fd, lwpid;
	char 		path[MAX_PATH];
        lwpsinfo_t 	f;
	DIR 		*dirp;		/* Directory ptr, opened  once */
	struct  dirent 	*direntp;
	char		*e;		/* End of number in file name */
	char		*fname;
	int		key, cnt, half;
	XmString	label_str;

	pti_ptr  = (Process_tree_info *) t_ptr->data;

	if ( pti_ptr == NULL ) return 1;

        pp = pti_ptr->pp;	
	if ( pp == NULL ) return 1;

        if ( pti_ptr->state == NODE_HIDDEN )  return;

	if ( PROCESS_NUM_LWP( pp ) < 2 )   /* Let the process node represent the LWP */
		return 1;
	
	parent_widget = pti_ptr->node_w;

	if ( parent_widget == NULL ) return;

	half = PROCESS_NUM_LWP( pp )/2;

	cnt = 1;

	pid = t_ptr->key;

	sprintf( path, "%s/%d/lwp", PROCESS_INFO_DIRECTORY, pid );

	first = 1;

	dirp = opendir( path );
	if ( dirp == NULL )
	{
	    return 1;
	}
	else
	{
	 while ( (direntp = readdir( dirp )) != NULL )
	 {

	  fname = direntp->d_name;

	  if ( 	( fname[0] == '.' ) &&   /* Skip . */
		( fname[1] == NULL_CHAR ))
	  {
		continue;
	  }

	  if ( 	( fname[0] == '.' ) &&   /* Skip .. */
		( fname[1] == '.' ) &&
		( fname[2] == NULL_CHAR ))
	  {
		continue;
	  }
		
	  lwpid = strtol( fname, &e, 10 );
	  if ( e == fname )
	  {
		continue;		/* not a number */
	  }

	  if ( first )
	  {
	    first = 0;
	    continue;  /* let process node represent the first thread/lwp. is this ok? */
	  }

	  sprintf( path, "%s/%d/lwp/%d/lwpsinfo", PROCESS_INFO_DIRECTORY, pid, lwpid );

	  if ( cnt <= half )
		key = -100000 + lwpid;
	  else
		key = 100000 + lwpid;

	  n = 0;
	  XtSetArg( wargs[n], XtNparentWidget, parent_widget ); n++;
	  XtSetArg( wargs[n], XmNhighlightThickness, 0 ); n++;
	  XtSetArg( wargs[n], XtNkey, key ); n++;

          fd = open( path, O_RDONLY );
          if ( fd == -1 )
          {
	  	sprintf( path, "<%d>", lwpid );
		lwp_w = XtCreateManagedWidget( path, xmLabelWidgetClass,
					treew, wargs, n );
          }
	  else
	  {
              rc = read(fd, &f, sizeof( lwpsinfo_t ) );
              if ( rc < sizeof( lwpsinfo_t ) )
              {
	  	sprintf( path, "<%d>", lwpid );
		lwp_w = XtCreateManagedWidget( path, xmLabelWidgetClass,
					treew, wargs, n );
              }
	      else
	      {
		 if ( f.pr_name && ( strlen( f.pr_name ) > 0 ) )
		      sprintf( path, "<%s>", f.pr_name );
		 else
		      sprintf( path, "<%d.%d>", f.pr_onpro, lwpid );

	 	XtSetArg( wargs[n], XtNdrawShapeFunc, do_draw_lwp_shape ); n++;
		lwp_w = XtCreateManagedWidget( path, xmLabelWidgetClass,
					treew, wargs, n );

	      }

	      close( fd );
	  }

	  if ( lwp_w != 0 )
		XtAddEventHandler(lwp_w, ButtonPressMask, FALSE, 
						select_lwp, (void *) pid);

	  cnt++;
	  lwp_w = 0;
	 }

	}

	if ( dirp ) closedir( dirp );
}

int
delLWPnodes( parent_widget )
Widget parent_widget;
{
	Cardinal	num_widgets;
	int 		i, n, key;
	Arg 		wargs[16];
	Widget		w;
	WidgetList 	wl;

	if ( parent_widget == NULL ) return;

	n = 0;
	XtSetArg( wargs[n], XtNnumTreeNodeChildren, &num_widgets ); n++;
	XtSetArg( wargs[n], XtNkey, &key ); n++;
	XtGetValues( parent_widget, wargs, n ); 
	
	wl = (WidgetList) malloc( num_widgets * sizeof( Widget ) + 1 );
	if ( wl == NULL )
		return;

	XpsiGetChildrenWidgets( TreeWidget, parent_widget, &(wl[0]) ); n++;

	for ( i = 0 ; i < num_widgets ; i++ )
	{
	    w = wl[i];

	    if ( w == NULL )
		continue;

	    n = 0;
	    XtSetArg( wargs[n], XtNkey, &key ); n++;
	    XtGetValues( w, wargs, n ); 

	    if ( key < 0 || key >= 100000 )	/* delete LWP's */
	    {
	 	XtUnmanageChild( w );
	 	XtDestroyWidget( w ); 
	    }
	}

	free( wl );
}

void
do_draw_lwp_shape( w, nothing, call_data )
Widget w;
int nothing;
XmAnyCallbackStruct	*call_data;
{
	/* This routine is currently not called - waiting implementation in Tree widget */
}

void
select_lwp( w, pid, event )
Widget w;
int  pid;
XButtonEvent *event;
{
    if ( event->button == 1 )  /* Select button */
    {
	if ( Process_action == ACTION_PROCESS_SELECT )
	{
		user_alert(w, "Select Action Not Supported for LWP Yet");
	}
	else if ( Process_action == ACTION_PROCESS_DETAILS )
	{
		do_oneLWP( w, pid, 0 );
	}
	else if ( Process_action == ACTION_PROCESS_SIGNAL )
	{
		user_alert(w, "Signal Action Not Supported for LWP Yet");
	}
	else if ( Process_action == ACTION_KILL_PROCESS )
	{
		user_alert(w, "Kill Action Not Supported for LWP Yet");
	}
	else if ( Process_action == ACTION_OUTLINE_SUBTREE )
	{
		user_alert(w, "Outline Action Not Supported for LWP Yet");
	}
	else if ( Process_action == ACTION_HIDE_SUBTREE )
	{
		user_alert(w, "Hide Action Not Supported for LWP Yet");
	}
	else if ( Process_action == ACTION_SHOW_SUBTREE )
	{
		user_alert(w, "Show Action Not Supported for LWP Yet");
	}
    }
    else 
    {
	DEBUG1(1, "select_lwp: Unexpected Button(%d)\n", event->button );
    }
}

change_LWP_visibility( vis_state )
Boolean vis_state;
{
        startBatchUpdate();
	if ( vis_state == True )
		walk_tree( Head, show_LWP, 0 );
	else
		walk_tree( Head, hide_LWP, 0 );

        finishBatchUpdate();

	DisplayLWP = vis_state;
}

tnode
*show_LWP( ptr, data )
tnode *ptr;
int    data;
{

	DEBUG1(1, "Trying to show LWP's on node(%d)\n", ptr->key);

	addLWPnodes( TreeWidget, ptr );

	return( NULL );
}



tnode
*hide_LWP( ptr, data )
tnode *ptr;
int    data;
{
	Process_tree_info  *pti_ptr;
	Widget		    parent_widget;

	pti_ptr  = (Process_tree_info *) ptr->data;

	if ( pti_ptr == NULL ) return NULL;

	parent_widget = pti_ptr->node_w;

	delLWPnodes( parent_widget );

	return( NULL );
}


#endif	/* End LWP code block */

void
expand_decoration_top_panel( t_ptr )
tnode *t_ptr;
{
	int n;
	Arg wargs[4];
	int saved_color_coding;
	Pixel fg, bg;
	Process_tree_info  *pti_ptr;
	int min_width;


	pti_ptr  = (Process_tree_info *) t_ptr->data;

	if ( pti_ptr == NULL )
		return;

	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	/* Set background as it may have been changed by a dynamic
	 * fill.
	 */

	saved_color_coding = Color_based_on;

	Color_based_on = Decoration_top_color;
	get_node_colors( User_colors, Group_colors, t_ptr, &fg, &bg );  

	min_width = Decoration_spacing +
				(( Decoration_size + Decoration_spacing ) * 
	    				( Decoration_count ));
	n = 0;
	XtSetArg( wargs[n], XtNheight, Decoration_top_size ); n++;
	XtSetArg( wargs[n], XtNwidth, (Dimension)min_width ); n++;
	XtSetArg( wargs[n], XtNbackground, bg ); n++;
	XtSetValues( pti_ptr->top_draw_w , wargs, n );

	Color_based_on = saved_color_coding;
}

void
expand_decoration_left_panel( t_ptr )
tnode *t_ptr;
{
	int n;
	Arg wargs[4];
	int saved_color_coding;
	Pixel fg, bg;
	Process_tree_info  *pti_ptr;

	pti_ptr  = (Process_tree_info *) t_ptr->data;

	if ( pti_ptr == NULL )
		return;

	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	saved_color_coding = Color_based_on;

	Color_based_on = Decoration_left_color;

	get_node_colors( User_colors, Group_colors, t_ptr, &fg, &bg );  

	/* XXX Recalc size before using it - not currently used  */

	n = 0;
	XtSetArg( wargs[n], XtNwidth, Decoration_left_size ); n++;
	XtSetArg( wargs[n], XtNbackground, bg ); n++;
	XtSetValues( pti_ptr->left_draw_w , wargs, n );

	Color_based_on = saved_color_coding;
}

void  
collapse_decoration_top_panel( t_ptr )
tnode *t_ptr;
{
	int n;
	Arg wargs[2];
	Process_tree_info  *pti_ptr;
	Pixel fg, bg;

	pti_ptr  = (Process_tree_info *) t_ptr->data;

	if ( pti_ptr == NULL )
		return;

	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	get_node_colors( User_colors, Group_colors, t_ptr, &fg, &bg );  

	n = 0;
	XtSetArg( wargs[n], XtNheight, (Dimension)0 ); n++;
	XtSetArg( wargs[n], XtNbackground, bg ); n++;
	XtSetValues( pti_ptr->top_draw_w , wargs, n );
}

void  
minimize_decoration_top_panel( pti_ptr )
Process_tree_info  *pti_ptr;
{
	int n;
	Arg wargs[2];
	int min_width=1;

	if ( pti_ptr == NULL )
		return;

	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	if ( pti_ptr->distinguished )
	    min_width = Decoration_spacing +
				(( Decoration_size + Decoration_spacing ) * 
	    				( Decoration_count ));
	n = 0;
	XtSetArg( wargs[n], XtNwidth, (Dimension)min_width ); n++;
	XtSetValues( pti_ptr->top_draw_w , wargs, n );
}

void  
collapse_decoration_left_panel( t_ptr )
tnode *t_ptr;
{
	int n;
	Arg wargs[2];
	Process_tree_info  *pti_ptr;

	pti_ptr  = (Process_tree_info *) t_ptr->data;

	if ( pti_ptr == NULL )
		return;

	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	n = 0;
	XtSetArg( wargs[n], XtNwidth, (Dimension)0 ); n++;
	XtSetValues( pti_ptr->left_draw_w , wargs, n );
}

void  
setup_decoration_defaults()
{
	Decoration[0] = COLOR_BY_STATUS;
	Decoration[1] = COLOR_BY_LOAD;
	Decoration[2] = COLOR_BY_PRIORITY;
	Decoration[3] = 0;

	Decorations_include_load++;

	Decoration_count = 3;
}

void  
decoration_label_resize( pti_ptr )
Process_tree_info  *pti_ptr;
{
	int 	n;
	Arg 	wargs[16];
	Dimension  label_width, min_width;
	Dimension  deco_width;

	if ( pti_ptr == NULL )
		return;

	if ( Decorated_tree == DECORATE_DISTINGUISHED )
	{
	        n = 0;
	        XtSetArg( wargs[n], XmNwidth, &label_width ); n++;
	        XtGetValues( pti_ptr->label_w, wargs, n );

		if ( pti_ptr->distinguished )
	            min_width = Decoration_spacing +
				(( Decoration_size + Decoration_spacing ) * 
	    				( Decoration_count ));
		else
		    min_width = 1;

	        n = 0;
	        XtSetArg( wargs[n], XmNwidth, &deco_width ); n++;
	        XtGetValues( pti_ptr->top_draw_w, wargs, n );

		if ( label_width < min_width )
		{
	    	    n = 0;
	    	    XtSetArg( wargs[n], XmNwidth, min_width ); n++;
	    	    XtSetValues( pti_ptr->label_w, wargs, n );
		}
		else if ( label_width > min_width )
		{
			if ( deco_width != label_width )
			{
	    	    	    n = 0;
	    	    	    XtSetArg( wargs[n], XmNwidth, label_width ); n++;
	    	    	    XtSetValues( pti_ptr->top_draw_w, wargs, n );
			}
		}
	}
}

void
top_decoration_exposures( w, t_ptr, cb )
Widget  w;
tnode	*t_ptr;
XmDrawingAreaCallbackStruct *cb;
{
        Region region;
	int x, y;
        unsigned int width, height;

	/*
        region = XCreateRegion();
        XtAddExposureToRegion( cb->event, region );
        XSetRegion( XtDisplay(w), DecorationGC, region );
	x = 0; y = 0; 
	width  =  draw area width
	height =  draw area height
        if ( XRectInRegion( region, x, y, width, height) != RectangleOut )
        {
	*/

	redraw_node_decorations( w, t_ptr );

        /* } XDestroyRegion( region ); */
}

void
left_leader_exposures( w, t_ptr, cb )
Widget  w;
tnode	*t_ptr;
XmDrawingAreaCallbackStruct *cb;
{
	redraw_leader_bars( w, t_ptr );
}

void
redraw_node_decorations( w, t_ptr )
Widget w;
tnode *t_ptr;
{
	int i;
	int saved_color_coding;
	Process_tree_info *pti_ptr;
	sysProcInfo	   *pp;

	if ( t_ptr == NULL )
		return;

	pti_ptr = (Process_tree_info *) t_ptr->data;
	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	saved_color_coding = Color_based_on;

	for ( i = 0 ; i < Decoration_count; i++ )
	{
	    switch ( Decoration[i] ) {

	    	case COLOR_BY_STATUS: status_decoration( w, t_ptr, i ); break;

	    	case COLOR_BY_LOAD: load_decoration( w, t_ptr, i ); break; 	

	    	case COLOR_BY_PRIORITY: priority_decoration(w, t_ptr, i); break; 	
		default: break;
	    }
	}
	Color_based_on = saved_color_coding;
}

void
redraw_leader_bars( w, t_ptr )
Widget w;
tnode *t_ptr;
{
	Process_tree_info *pti_ptr;
	sysProcInfo	   *pp;
	Dimension	   left_draw_width;
	Dimension	   desired_left_draw_width=1;
	int		   n;
	Arg		   wargs[4];

	if ( t_ptr == NULL )
		return;

	pti_ptr = (Process_tree_info *) t_ptr->data;
	if ( pti_ptr == NULL || pti_ptr->state == NODE_HIDDEN )
		return;

	pp = pti_ptr->pp;
	if ( pp )
	{
	    /* if node is grp leader draw seperator bar */

	    if ( PROCESS_ID(pp) == PROCESS_GROUP_LEADER(pp) )
	    {
	    	group_leader_bar( w, t_ptr );
	    }

	    /* if node is sesion leader add tty/pty ident */

	    if ( PROCESS_ID(pp) == PROCESS_SESSION_ID(pp) )
	    {
	    	session_leader_bar( w, t_ptr );
	    }
	}
}

void
group_leader_bar( w, t_ptr )
tnode	*t_ptr;
Widget  w;
{
	int n, x, y;
	unsigned int width, height;
	Dimension  widget_width, widget_height;
	Pixel border_fg, bar_fg, fg, bg;
	Arg wargs[4];


	if ( DecorationGC == 0 )
		create_decoration_gc( );

	if ( Leader_bar_colors == STATIC_LEADER_COLOR )
	    bar_fg = WhitePixel( XtDisplay(w), DefaultScreen( XtDisplay(w) ) );
	else
	{
	    get_dynamic_color( t_ptr, &fg, &bg );

	    bar_fg = fg;
	}

	XSetForeground(XtDisplay(w), DecorationGC, bar_fg );
	XSetBackground(XtDisplay(w), DecorationGC, bar_fg );


	n = 0;
	XtSetArg( wargs[n], XmNwidth,  &widget_width ); n++;
	XtSetArg( wargs[n], XmNheight, &widget_height ); n++;
	XtGetValues( w, wargs, n );

	
	x      = (widget_width - Leader_bar_size) - Leader_bar_spacing;
	y      = 0;
	width  = Leader_bar_size;
	height = widget_height;

        XFillRectangle( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height );

	if ( Leader_bar_borders )
	{
	    border_fg = WhitePixel(XtDisplay(w),DefaultScreen( XtDisplay(w)));
	    XSetForeground(XtDisplay(w), DecorationGC, border_fg );
	    XSetBackground(XtDisplay(w), DecorationGC, border_fg );
            XDrawRectangle( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height );
	}
}

void
session_leader_bar( w, t_ptr )
tnode	*t_ptr;
Widget  w;
{
	int n, x, y;
	unsigned int width, height;
	Dimension widget_width, widget_height;
	Pixel fg, bg, border_fg, bar_fg;
	Arg wargs[4];


	if ( DecorationGC == 0 )
		create_decoration_gc( );

	if ( Leader_bar_colors == STATIC_LEADER_COLOR )
	    bar_fg = WhitePixel( XtDisplay(w), DefaultScreen( XtDisplay(w) ) );
	else
	{
	    get_dynamic_color( t_ptr, &fg, &bg );

	    bar_fg = fg;
	}

	XSetForeground(XtDisplay(w), DecorationGC, bar_fg );
	XSetBackground(XtDisplay(w), DecorationGC, bar_fg );

	n = 0;
	XtSetArg( wargs[n], XmNwidth,  &widget_width ); n++;
	XtSetArg( wargs[n], XmNheight, &widget_height ); n++;
	XtGetValues( w, wargs, n );
	
	x      = widget_width - ((2 * Leader_bar_size) + 
					(Leader_bar_spacing * 2));
	y      = 0;
	width  = Leader_bar_size;
	height = widget_height;

        XFillRectangle( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height );

	if ( Leader_bar_borders )
	{
	    border_fg = WhitePixel(XtDisplay(w),DefaultScreen( XtDisplay(w)));
	    XSetForeground(XtDisplay(w), DecorationGC, border_fg );
	    XSetBackground(XtDisplay(w), DecorationGC, border_fg );
            XDrawRectangle( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height );
	}
}

void
status_decoration( w, t_ptr, pos )
tnode	*t_ptr;
Widget  w;
int pos;
{
	Pixel fg, bg;

	Color_based_on = COLOR_BY_STATUS;

	get_dynamic_color( t_ptr, &fg, &bg );

	render_decoration( w, pos, fg, bg );
}

void
priority_decoration( w, t_ptr, pos )
tnode	*t_ptr;
Widget  w;
int pos;
{
	Pixel fg, bg;

	Color_based_on = COLOR_BY_PRIORITY;

	get_dynamic_color( t_ptr, &fg, &bg );

	render_decoration( w, pos, fg, bg );
}

void
load_decoration( w, t_ptr, pos )
tnode	*t_ptr;
Widget  w;
int pos;
{
	Pixel fg, bg;

	Color_based_on = COLOR_BY_LOAD;

	get_dynamic_color( t_ptr, &fg, &bg );

	render_decoration( w, pos, fg, bg );
}

void
render_decoration( w, pos, fg, bg )
Widget  w;
int   pos;
Pixel fg;
Pixel bg;
{
	int x, y=0;
	unsigned int width, height;
	Pixel border_fg;


	if ( DecorationGC == 0 )
		create_decoration_gc( );

	XSetForeground(XtDisplay(w), DecorationGC, bg );
	XSetBackground(XtDisplay(w), DecorationGC, bg );

	width  = Decoration_size;

	x = (pos * (Decoration_size + Decoration_spacing)) + 
					Decoration_spacing/2 ;

	if ( Decoration_border_dynamic )
	  border_fg = fg;
	else
	  border_fg = WhitePixel( XtDisplay(w), DefaultScreen( XtDisplay(w) ) );

	if ( Decoration_type == CIRCLE_DECORATION )
	{
	    height = Decoration_size;
	    y      = (Decoration_top_size - Decoration_size) / 2;

            XFillArc( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height,
				  0, (64 * 360) );

	    if ( Decoration_borders )
	    {
	        XSetForeground(XtDisplay(w), DecorationGC, border_fg );
	        XSetBackground(XtDisplay(w), DecorationGC, border_fg );
                XDrawArc( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height,
				  0, (64 * 360) );
	    }
	}
	else if ( Decoration_type == BAR_DECORATION )
	{
	    height = Decoration_top_size - 1;

            XFillRectangle( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height );

	    if ( Decoration_borders )
	    {
	        XSetForeground(XtDisplay(w), DecorationGC, border_fg );
	        XSetBackground(XtDisplay(w), DecorationGC, border_fg );
                XDrawRectangle( XtDisplay(w), XtWindow(w), DecorationGC,
                                  x, y, 
				  width, height );
	    }
	}

}



void
decoration_resize( w, pti_ptr, call_data )
Widget w;
Process_tree_info  *pti_ptr;
caddr_t    call_data;
{
	/* Handle resized drawing areas */
}

create_decoration_gc( )
{
        XGCValues       gcv;
        Display         *dpy;
        Window          w;
        int             mask;
        int             n;
        Arg             wargs[10];

        dpy  = XtDisplay( Top_level );
        w    = XtWindow( Top_level );
        mask = GCForeground | GCBackground ;

        n = 0;
        XtSetArg( wargs[n], XtNforeground, &gcv.foreground ); n++;
        XtSetArg( wargs[n], XtNbackground, &gcv.background ); n++;
        XtGetValues( Top_level, wargs, n );

	DecorationGC = XCreateGC( dpy, w, mask, &gcv );
}


tnode
*display_proc_info( t_ptr, noop )
tnode *t_ptr;
int    noop;
{
	Process_tree_info  *pti_ptr;
	Process_list_info  *pli;
	lnode  		   *p, *next_p;
	XmString	    label_str;
	Arg		    wargs[4];
	int		    first, n, len;
	sysProcInfo	   *pp;
	lnode		   *l_ptr;
	lnode		   *dis_list;
	int		    custom_list=0;
	char		   *dev_name;
	major_t		    major_no;
	minor_t		    minor_no;
	char		   *str, *time_str;
	time_t		    start_time;
	int		    hours, min, sec;
	long		    hsec;
	char		    buf[10000];
	char		    label_buf[10000];
	int		    pro=0;


	pti_ptr = (Process_tree_info *) t_ptr->data;

	/* Traverse the display list and build up the list of displayed
	 * values.
	 */

	if ( pti_ptr->state == NODE_HIDDEN )
		return( NULL );

	if ( pti_ptr->outline == True )		/* Empty label */
	{
		if ( Decorated_tree )
			minimize_decoration_top_panel( pti_ptr );

		xs_wprintf( pti_ptr->label_w, "%s", "");
		n = 0;
		XtSetArg(wargs[n], XmNwidth, OutlineWidth); n++;
		XtSetArg(wargs[n], XmNheight, OutlineHeight); n++;
		XtSetValues( pti_ptr->label_w, wargs, n);

		if ( Decorated_tree )
			decoration_label_resize( pti_ptr );

		return(NULL);
	}

	if ( pti_ptr->display_list )
	{
		custom_list++;
		dis_list=merge_lists(Global_display_list,pti_ptr->display_list);
	}
	else
		dis_list = Global_display_list;

	p      = dis_list;
	next_p = NULL;

	strcpy( buf, t_ptr->str_key );

	/* Might be nice to have the PR_PRO up front on one liners
	 *
	 *  i.e.   CPU.PID Name   ->  2.23 Tray Racer
	 */

#ifdef SVR4_MP
	Show_Processor = 1;

	pp = pti_ptr->pp;
	if ( pp == NULL )
		pro = -1;
	else
		pro = PROCESS_ON_PROCESSOR( pp );
#endif
	
	if ( p )
	{
	  if ( p->key == PR_PID )  /* See if we also want name */
	  {
		if ( p->next && ( p->next->key == PR_FNAME ))  /* Yup */
		{
		  if ( Show_Processor )
			sprintf(buf,"%d.%d: %s", pro,t_ptr->key,t_ptr->str_key);
		  else
			sprintf(buf,"%d: %s",t_ptr->key,t_ptr->str_key);
		  
		  next_p = p->next->next;	
		}
		else
		{
		  if ( Show_Processor )
			sprintf(buf,"%d.%d", pro, t_ptr->key);
		  else
			sprintf(buf, "%d", t_ptr->key);

		  next_p = p->next;
		}
	  }
	  else if ( p->key == PR_FNAME )
	  {
		next_p = p->next;
	  }
	  else
		next_p = p;
	}

	strcpy( label_buf, buf );

	pp = pti_ptr->pp;
	if ( pp == NULL )
		next_p = NULL;

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
		next_p = NULL;

	pli = (Process_list_info *)l_ptr->data;
	if ( pli == NULL )
		next_p = NULL;

	buf[0] = NULL_CHAR;
	for ( p = next_p ; p != NULL; p = p->next )
	{
	    /* For each field, format appropriately and add to widget */

	    switch ( p->key )
	    {
		case PR_PID: 
		case PR_FNAME:  break;

		case PR_PSARGS: sprintf(buf,"Args:  %s", PROCESS_CMD_ARGS(pp)); 
				break;

		case PR_UID: sprintf(buf,   "UID:   %d", PROCESS_USER_ID(pp));
			     break;

		case PR_GID: sprintf(buf,   "GID:   %d", PROCESS_GROUP_ID(pp));
		       	     break;

		case PR_PPID: sprintf(buf,  "PPID:  %d", PROCESS_PARENT_ID(pp));
			      break;

		case PR_EUID: sprintf(buf,  "EUID:  %d", 
				PROCESS_EFFECTIVE_USER_ID(pli));break;

		case PR_EGID: sprintf(buf,  "EGID:  %d", 
				PROCESS_EFFECTIVE_GROUP_ID(pli));break;

		case PR_PGRP: sprintf(buf,  "Pgrp:  %d", 
					    PROCESS_GROUP_LEADER(pp) );break;

		case PR_SID: sprintf(buf,   "SID:   %d", PROCESS_SESSION_ID(pp));
			     break;

		case PR_SNAME: /*sprintf(buf,"status:%c",PROCESS_SNAME(pp) );*/

		    switch( PROCESS_SNAME(pp) ) {

			case 'O': sprintf(buf, "%s",  "Running" ); break;
			case 'A': sprintf(buf, "%s",  "Just Ran" ); break;
			case 'S': sprintf(buf, "%s",  "Sleeping" ); break;
			case 'D': sprintf(buf, "%s",   "Wait/Swap" ); break;
			case 'R': sprintf(buf, "%s",  "Runable" ); break;
			case 'I': sprintf(buf, "%s",  "Idle" ); break;
			case 'Z': sprintf(buf, "%s",  "Zombie" ); break;
			case 'T': sprintf(buf, "%s",  "Stopped" ); break;
			case 'X': sprintf(buf, "%s",  "SXBRK" ); break;

			default : sprintf(buf, "%s",  "Unknown" ); break;
				break;
		    }
		    break;

		case PR_FLAGS: sprintf(buf, "Flags: %x", PROCESS_FLAG(pp));break;
		case PR_NICE:  sprintf(buf, "Nice:  %d", PROCESS_NICE(pp));break;

#ifdef SOLARIS_7
		case PR_SIZE:   sprintf(buf, "Size:  %dk", PROCESS_SIZE(pp) );
				break;
		case PR_RSSIZE: sprintf(buf,"Rsize: %dk", 
						PROCESS_RESIDENT_SET_SIZE(pp) );
				break;
#else
		case PR_SIZE:  sprintf(buf, "Size:  %dk",
					( PROCESS_SIZE(pp) * PageSize)/1024);
				break;
		case PR_RSSIZE: sprintf(buf,"Rsize: %dk", 
				(PROCESS_RESIDENT_SET_SIZE(pp) * PageSize)/1024);
				break;
#endif

		case PR_WCHAN: 
		
			       /* Later after we cache the looked up values
			        * display the text symbol? */

			       sprintf(buf, "Wchan: %xX", PROCESS_WCHAN( pp ) );
			       break;

		case PR_START: 
		
	    		start_time = PROCESS_START_TIME(pp).tv_sec;
	    		time_str   = ctime( &start_time );
	    		len = strlen( time_str );
	    		time_str[len-1] = NULL_CHAR;/* Remove new line char */

			sprintf(buf,"Start: %s", time_str );
			break;

		case PR_TIME:  

	    		hours = PROCESS_CPU_TIME(pp).tv_sec / 3600;

	    		if ( hours )
				min  = (PROCESS_CPU_TIME(pp).tv_sec % 3600) / 60;
	    		else
				min  = PROCESS_CPU_TIME(pp).tv_sec / 60;
		
	    		sec  = PROCESS_CPU_TIME(pp).tv_sec % 60;
	    		hsec = PROCESS_CPU_TIME(pp).tv_nsec / 10000000;
		
			if ( hours > 0 )
	    		    sprintf(buf, "Time: %d:%0.2d:%0.2d.%0.2d",
	    						hours, min, sec, hsec);
			else if ( min > 0 )
	    		    sprintf(buf, "Time: %0.2d:%0.2d.%0.2d",
	    						 min, sec, hsec);
			else
	    		    sprintf(buf, "Time: %0.2d.%0.2d", sec, hsec);

		        break;

#ifdef SVR4_MP
		case PR_NLWP: sprintf(buf,"Nlwp:  %d", PROCESS_NUM_LWP( pp ) );
			     break;

		case PR_ONPRO: 
			sprintf(buf,"Proc:  %d", PROCESS_ON_PROCESSOR( pp ) );

			     break;
#endif

		case PR_PRI: sprintf(buf,"Pri:   %d", PROCESS_PRIORITY( pp ) );
			     break;

		case PR_CLNAME: sprintf(buf,"Class: %s", PROCESS_CLASS_NAME(pp));
			        break;

		case PR_TTYDEV: /* sprintf(buf,"ttydev %d", 
						PROCESS_CONTROL_TTY(pp) ); */

#ifdef LINUX
			if  ( ( PROCESS_CONTROL_TTY( pp ) == -1 ) ||
					( PROCESS_CONTROL_TTY( pp ) == 0 ) )
#else
			if ( PROCESS_CONTROL_TTY( pp ) == -1 )
#endif
			{
				sprintf(buf, "%s", "TTY:   None" );
			}
			else
			{
				major_no = major( PROCESS_CONTROL_TTY(pp) );
				minor_no = minor( PROCESS_CONTROL_TTY(pp) );

				dev_name = find_dev_name( major_no, minor_no );
				if (  dev_name != NULL )
					sprintf(buf, "TTY:   %s", dev_name);

				else
					sprintf(buf, "TTY:   %ld:%ld",
							major_no, minor_no);
			}
			break;

		default: fprintf( stderr, "Unknown field!!\n");
			 break;
	    }

	    strcat( label_buf, "\n" );
	    strcat( label_buf, buf );  /* XXX Should check for overflow */
	}

/*
#ifdef LESSTIF
*/
	if ( Decorated_tree )
		minimize_decoration_top_panel( pti_ptr );
/*
#endif
*/

        label_str = XmStringCreateLtoR( label_buf, XmSTRING_DEFAULT_CHARSET );

	XtSetArg(wargs[0], XmNlabelString, label_str);
	XtSetValues( pti_ptr->label_w, wargs, 1);

	if ( Decorated_tree )
		decoration_label_resize( pti_ptr );

	XmStringFree( label_str );

	if ( custom_list )
		free_list( dis_list );

	return(NULL);
}

destroy_proc_widget( t_ptr )
tnode *t_ptr;
{
	Process_tree_info  *pti_ptr;

	pti_ptr = (Process_tree_info *) t_ptr->data;

	if ( pti_ptr->state == NODE_DISPLAYED )
	{
		/* This must be done before the node_w is destroyed */

		if ( AppData.showProcessTips )
	        	destroy_process_tip_if_displayed( pti_ptr );

		/* note: node_w also refers to label_w in a non-decorated tree
		 * in which case we just need to destroy label_w, not both!
		 */

#ifdef SVR4_MP

		delLWPnodes( pti_ptr->node_w ); /* TTT Destroy LWP */
#endif

		XtUnmanageChild( pti_ptr->label_w );
		XtDestroyWidget( pti_ptr->label_w );
		pti_ptr->label_w = NULL;

		if ( Decorated_tree ) /* in a decorated tree do both */
		{
 		    /* Node_w children also destroyed */

		    XtUnmanageChild( pti_ptr->node_w );
		    XtDestroyWidget( pti_ptr->node_w );
		}

		pti_ptr->node_w = NULL;

	}

	pti_ptr->state = NODE_HIDDEN;
}

/*  --------------------------- Callbacks ---------------------------- */

void
manuallySetSelection( w, ptr, time )
Widget w;
tnode *ptr;
Time time;
{
    	Process_tree_info  *pti_ptr;

	if ( ptr == NULL ) return;
	pti_ptr = (Process_tree_info *)ptr->data;
	if ( pti_ptr == NULL )
		return;

	/* Note Time is a long measured in milliseconds 
	 * provided by the Xserver in event messages. Here we get
	 * passed the value and the caller supplies the time
	 * using the most recent event they have.
	 */

	if ( XtOwnSelection( w, XA_PRIMARY, time, do_convert_selection,
						do_lose_selection,
						NULL))
	{
		Selected_node = ptr->key;

		pti_ptr->selected = True;
		set_node_colors( User_colors, Group_colors, ptr);
	}
}

void
select_node( w, ptr, event )
Widget w;
tnode  *ptr;
XButtonEvent *event;
{
    String params[1];
    Cardinal num_params=1;
    char pid_str[10];
    Process_tree_info  *pti_ptr;


    if ( event->button == 1 )  /* Select button */
    {
	if ( Process_action == ACTION_PROCESS_SELECT )
	{
		DEBUG2(3, "Selected %s:%d\n", ptr->str_key, ptr->key );

	    pti_ptr = (Process_tree_info *)ptr->data;
	    if ( pti_ptr == NULL )
		return;

	    if ( pti_ptr->selected == True )
	    {
		XtDisownSelection( w, XA_PRIMARY, event->time);
	    }
	    else
	    {
		if ( XtOwnSelection( w, XA_PRIMARY, event->time,
							do_convert_selection,
							do_lose_selection,
							NULL))
		{
			Selected_node     = ptr->key;
			pti_ptr->selected = True;
			set_node_colors( User_colors, Group_colors, ptr);
		}
	    }
	}
	else if ( Process_action == ACTION_PROCESS_DETAILS )
	{
		popup_procinfo( Top_level, ptr );
	}
	else if ( Process_action == ACTION_PROCESS_SIGNAL )
	{
		sprintf( pid_str, "%d", Current_signal );
		params[0] = &(pid_str[0]);
		signal_node( w, event, params, &num_params );
	}
	else if ( Process_action == ACTION_KILL_PROCESS )
	{
		sprintf( pid_str, "%d", 9 );
		params[0] = &(pid_str[0]);
		signal_node( w, event, params, &num_params );
	}
	else if ( Process_action == ACTION_OUTLINE_SUBTREE )
	{
                startBatchUpdate();
		walk_tree( ptr, outline_node, 0 );
		walk_tree( ptr, redisplay_node, 0 );
        	finishBatchUpdate();
	}
	else if ( Process_action == ACTION_HIDE_SUBTREE )
	{
		if ( ptr == Head )
		{
			XBell( XtDisplay(w), 100 );
			return;
		}

                startBatchUpdate();
		post_walk_tree( ptr, hide_node_widget, 0 );
        	finishBatchUpdate(); /* Ok do one layout  */
	}
	else if ( Process_action == ACTION_SHOW_SUBTREE )
	{
                startBatchUpdate();
		walk_tree( ptr, display_node_widget, 0 );
        	finishBatchUpdate();
	}
    }
    else 
    {
	DEBUG1(1, "select_node: Unexpected Button(%d)\n", event->button );
    }
}

void
change_decor( newDecor )
int newDecor;
{
        startBatchUpdate();
	post_walk_tree( Head, hide_node_widget, 0 );

	Decorated_tree = newDecor;

	walk_tree( Head, display_node_widget, 0 );
        finishBatchUpdate();
}

void
select_tree( w, head, event )
Widget w;
tnode  *head;
XButtonEvent *event;
{
	tnode 		  *t_ptr;
	Process_tree_info *pti_ptr;
	Arg 		  wargs[2];
	int 		  n, post_button;
    	Widget 		  popup_menu=NULL;
    	Window 		  tree_window;

    	n = 0;
    	XtSetArg( wargs[n], XmNwhichButton, &post_button ); n++;
    	XtGetValues( Node_popup_menu, wargs, n );


	if ( event->button == post_button )
	{
	    tree_window = XtWindow( TreeWidget );
	    if ( event->subwindow == 0 )
	    {
		DEBUG0(1, "Post tree popup menu??\n");
	     	XmMenuPosition( Tree_popup_menu, (XButtonPressedEvent *)event );
	     	XtManageChild( Tree_popup_menu );
	    }
	    else
	    {
		DEBUG0(1, "Post node menu??\n");
		/* Find the widget which has the window matching subwindow */

		t_ptr = walk_tree( Head, match_window, event->subwindow);
		if ( t_ptr == NULL )
		{
			DEBUG0(0, "select_tree: Failed to find matching sub\n");
			return;
		}

		Popup_on_pid = t_ptr->key;

		/* Check to see if we have a custom menu for this process */

		/* popup_menu = xs_getDynamicMenuForObject( "process_tree_node", 
							t_ptr->str_key ); */
		if ( popup_menu == NULL )
			popup_menu = Node_popup_menu;

		/* If not then popup the generic Node_popup_menu */

	     	XmMenuPosition( popup_menu, (XButtonPressedEvent *)event );
	     	XtManageChild( popup_menu );
	    }
	}
	else if ( event->button == 1 )
	{
	    if ( Process_action == ACTION_PROCESS_SELECT )
	    {
		DEBUG1(1, "Deselect %d\n", Selected_node );
		if ( Selected_node != -1 )
		{
			t_ptr = walk_tree( Head, match_key, Selected_node );
			if ( t_ptr == NULL )
			{
				XBell( XtDisplay(w), 100 );
				return;
			}
			
			pti_ptr = (Process_tree_info *)t_ptr->data;
			if ( pti_ptr && (pti_ptr->state != NODE_HIDDEN ) )
			{
				XtDisownSelection( pti_ptr->label_w,XA_PRIMARY, 
							event->time);

				/* For some reason the above does not always generate
				 * the do_lose_selection call. This happens with find
				 * selected nodes(sometimes, not sure of the exact
				 * senarios. Anyhow we force the deselection here
				 * even though do_lose_selection may also be called.
				 */

				Selected_node = -1;
				pti_ptr->selected = False;
				set_node_colors( User_colors, Group_colors, t_ptr);
			}

			DEBUG2(1, "Deselected %s:%d\n", t_ptr->str_key,
								 t_ptr->key );
		}
		else
			XBell( XtDisplay(w), 100 );
	    }
	    else	/* Other Action sensitive procedures ?? */
	    {
		XBell( XtDisplay(w), 100 );
	    }
	}
	else
	{
		DEBUG1(1,"select_tree: Unexpected button(%d)\n",event->button);
	}
}

tnode
*match_window( t_ptr, window )
tnode   *t_ptr;
Window   window;
{
	Process_tree_info   *pti_ptr;
	Window   	     widget_window;
	
	if ( t_ptr == NULL )
		return( NULL );

	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr == NULL )
		return( NULL );

	if ( pti_ptr->state == NODE_HIDDEN )
		return( NULL );

	widget_window = XtWindow( pti_ptr->node_w );
	if ( widget_window == window )
		return( t_ptr );

	return( NULL );
}

Boolean
do_convert_selection( w, selection, target, type, value, length, format )
Widget		w;
Atom		*selection;
Atom		*target;
Atom		*type;
caddr_t		*value;
unsigned long	*length;
int		*format;
{
	char *str;
	int   len;
	char  buf[20];
	tnode *t_ptr;

	DEBUG0(3, "do_convert_selection:  \n");

	if ( *selection == XA_PRIMARY )
	{
		DEBUG0(9, "Request for primary selection\n");
	}
	else if ( *selection == XA_SECONDARY )
	{
		DEBUG0(0,"WARNING: Request for secondary  selection\n");
		return( False );
	}
	else
	{
		DEBUG1(0, "WARNING: unknown selection(%d)\n", *selection );
		return( False );
	}

	t_ptr = walk_tree( Head, match_key, Selected_node );
	if ( t_ptr == NULL )
	{
		DEBUG1(1,"Failed to find node(%d) to convert\n", Selected_node);
		return( False );
	}

	if ( *target == XA_STRING )
	{
		*type = XA_STRING;

		len = strlen( t_ptr->str_key );
		str = strdup( t_ptr->str_key );
		*value = &(str[0]);
		*length = len+1;	/* Transfer NULL as well */
		*format = 8;

		/* If we have registered a done_selection procedure we own
		 * the space and must free it in the done procedure. 
		 * Otherwise we do not own the space!!.
		 */
		
		DEBUG0(3, "Converted string to proc name string\n");
	}
	else if ( *target == XA_INTEGER )
	{
		*type = XA_STRING;

		sprintf( buf, "%d", t_ptr->key);
		str = strdup(buf);
		len = strlen(str);

		*value  = &(str[0]);
		*length = len + 1;	/* Send NULL as well */
		*format = 8;

		DEBUG1(3, "Converted integer to pid string(%s)\n", str);
	}
	else
	{
		str = XGetAtomName( XtDisplay(w), *target );
		if ( str == NULL )
		{
			DEBUG1(0,"Uknown selection target type(%s)", *target);
	    	}
		else
		{
			DEBUG1(0, "Requested type(%s) not yet handled\n", str);
			XtFree( str );
		}

		return( False );
	}

	DEBUG0(3, "Selection successfully converted\n");

	return( True );
}

void
do_lose_selection( w, selection )
Widget w;
Atom   *selection;
{
	tnode 		  *t_ptr;
	Process_tree_info *pti_ptr;

	DEBUG2(1,"do_lose_selection:  widget(%d) selection(%d)\n",w,*selection);

	if ( Selected_node == -1 )
		return;

	t_ptr = walk_tree( Head, match_key, Selected_node );
	if ( t_ptr == NULL )
	{
		DEBUG1(0, "Failed to find node(%d)\n", Selected_node );
		return;
	}

	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr )
	{
	    if ( pti_ptr->selected == True )
	    {
		pti_ptr->selected = False;
		set_node_colors( User_colors, Group_colors, t_ptr);
	    }
	    else
	    {
	        DEBUG2(0, 
	    "do_lose_selection: ERROR process(%s:%d) apparently not selected\n",
					t_ptr->str_key, t_ptr->key );
	    }
	}

	Selected_node = -1;
}

void
hide_node( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	tnode *ptr;

	/* Find the node associated with the widget */

	ptr = walk_tree( Head, match_widget, w );
	if ( ptr == NULL )
	{
		DEBUG1(1, "hide_node: can't find widget for (%d)\n", w );
		return;
	}

	if ( ptr == Head )	/* Hmm, need to be able to expand! */
	{
		DEBUG0(1, "Trying to hide head\n");
		return;
	}

	/* walk the subtree marking all nodes as hidden  */

        startBatchUpdate();
	post_walk_tree( ptr, hide_node_widget, 0 );
        finishBatchUpdate();
}

tnode
*match_widget( ptr, w )
tnode *ptr;
Widget w;
{
	Process_tree_info  *pti_ptr;

	pti_ptr = (Process_tree_info *) ptr->data;

	if ( pti_ptr->state == NODE_HIDDEN )
		return( NULL );

	if ( ( pti_ptr->label_w == w ) ||
	     ( pti_ptr->distinguished && ( pti_ptr->top_draw_w == w ) ) )
		return( ptr );
	else
		return( NULL );
}

tnode
*hide_node_widget( ptr, data )
tnode *ptr;
int    data;
{

	DEBUG1(1, "Trying to hide node(%d)\n", ptr->key);
	destroy_proc_widget( ptr );
	return( NULL );
}

void
expand_node( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	tnode *ptr;

	/* Find the node associated with the widget */

	ptr = walk_tree( Head, match_widget, w );
	if ( ptr == NULL )
	{
		DEBUG1(1, "expand_node: can't find widget for (%d)\n", w );
		return;
	}

        startBatchUpdate();
	walk_tree( ptr, display_node_widget, 0 );
        finishBatchUpdate();
}


tnode
*display_node_widget( ptr, data )
tnode *ptr;
int    data;
{
	Widget         parent_widget;
	Process_tree_info  *pti_ptr;
	Process_tree_info  *ppti_ptr;

	if ( ptr == NULL )
		return( NULL );

	pti_ptr = (Process_tree_info *) ptr->data;

	if ( pti_ptr->state == NODE_HIDDEN )
	{
		parent_widget = NULL;
		if ( ptr->parent != NULL )
		{
		    ppti_ptr = (Process_tree_info *) ptr->parent->data;
		    if ( ppti_ptr != NULL )
		        parent_widget = ppti_ptr->node_w;
		}

		create_proc_widget( TreeWidget, parent_widget, ptr );
		pti_ptr->state = NODE_DISPLAYED;
		DEBUG0(1, "succeded\n");
	}

	return( NULL );
}

void
signal_node( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	tnode *ptr;
	Process_tree_info  *pi_info;
	sysProcInfo 	*pp;
	int    sig_value;
	int    process_is_descendant;
	int    allow_signal=0;

	DEBUG2(1, "signal_node: num_params(%d) param(%s)\n", 
						*num_params, params[0]);

/* Note: The sigsend(2) features are not yet used, i.e. sending signals to
 *
 *	Processes
 *	Group members
 *	Session members
 *	Any process with effective UID 
 *	Any process with effective GID 
 *	Any process with schedular Class
 *	All Processes
 *
 * These might be best implemented as a signal popup, (press s over node)
 * with toggle buttons for the type, with appropriate feedback. Perhaps
 * even highlighting the process nodes that would be affected.
 *
 */

	ptr = walk_tree( Head, match_widget, w );
	if ( ptr == NULL )
	{
		DEBUG1(1, "signal_node: can't find widget for (%d)\n", w );
		return;
	}
	else
	{
		pi_info = (Process_tree_info *) ptr->data;

		/* In General we only allow the user to send signal's to
		 * their own processes. Since this process may be suid to root
		 * we need to ensure the userid of the process receiving
		 * the signal is the same as the user's real uid.
		 *
		 * We make an exception if the following is true:
		 *
		 *	The controlling terminal of the process we are
		 *	sending the signal too is one of those associated
		 *	with the real userid of the current process.
		 *	( i.e. the process getting the signal is a descendant
		 *	of the user, perhaps setuid, or after su ....
		 *	or another login.
		 *
		 *	Or the TRUSTED_USER #define was set in treeps.h in
		 *	which case, you can shutdown the system!!!
		 */
		
		pp = pi_info->pp;
		if ( pp == NULL )
			return;

		if ( Users->user_uid == PROCESS_USER_ID(pp) )
		{
			allow_signal++;
		}
		else 
		{
			calc_current_users_ttys();
			process_is_descendant =find_dev_in_users_devs(User_ttys,
						PROCESS_CONTROL_TTY(pp) );
			if ( process_is_descendant )
			{
				/* On linux the user may be running treeps
				 * non setuid, we verify that here so
				 * we can issue the error dialog if not
				 */

				if ( geteuid() == 0 )
				    allow_signal++;
				else
				{
			 	    user_alert(w, 
		    "Cannot signal process, treeps needs to be setuid root");
				    return;
				}
			}
			else if ( Trusted_user )
			{
				if ( geteuid() == 0 )
				    allow_signal++;
				else
				{
			 	    user_alert(w, 
		    "Cannot signal process, treeps needs to be setuid root");
				    return;
				}
			}
		}


		if ( allow_signal )
		{
		    if ( ptr->key > 1 )  /* Never kill(0:sched), kill(1:init)*/
		    {
			if ( *num_params > 0 )
			{
				DEBUG2(1, "signal_node: pid(%d) signal(%s)\n", 
						ptr->key, params[0]);

				sig_value = atoi( params[0] );
				if ( sig_value < 1 || sig_value > MaxSig)
				{
					DEBUG1(1, "kill_node: sig(%d) OOR\n",
							sig_value );
					return;
				}
				kill( ptr->key, sig_value );
			}
			else
			{
				DEBUG0(1, "signal_node: no arg, use kill(9)\n");
				kill( ptr->key, 9 );
			}
		    }
		}
		else
		{
			/* Display signal ignored, invalid user id or
			 * not a direct descendant of the current process.
			 *
			 */
			 user_alert(w, 
				"Cannot signal process, Invalid Permissions");
		}
	}
}

tnode
*calc_user_ttys( ptr, noop )
tnode *ptr;
void  *noop;
{
	sysProcInfo 	   *pp;
	Process_tree_info  *pti_ptr;
	int		    in_list;

	if ( ptr == NULL )
		return( NULL );

	pti_ptr = (Process_tree_info *)ptr->data;
	if( pti_ptr == NULL )
		return( NULL );

	pp = (sysProcInfo *)pti_ptr->pp;
	if ( pp == NULL )
		return( NULL );

	if ( PROCESS_USER_ID(pp) != Users->user_uid )  /* Only login user count */
		return( NULL );

	if ( PROCESS_ID(pp) != PROCESS_SESSION_ID(pp))  /* Only check session leaders */
		return( NULL );

	in_list = find_dev_in_users_devs(User_ttys, PROCESS_CONTROL_TTY(pp) );

	if ( in_list == False )
	{
#ifdef LINUX
		if ( ( PROCESS_CONTROL_TTY(pp) == -1 ) ||
			( PROCESS_CONTROL_TTY(pp) == 0 ) )	/* No tty!! */
#else
		if ( PROCESS_CONTROL_TTY(pp) == -1 )
#endif
		{
			return( NULL );
		}

		User_ttys = add_dev_to_users_devs( User_ttys, 
					PROCESS_CONTROL_TTY( pp ) );
	}
	
	return( NULL );
}


calc_current_users_ttys()
{
	User_tty_struct *p;
	major_t			 major_no;
	minor_t			 minor_no;
	char			*dev_name;

	User_ttys = free_user_ttys_list( User_ttys );
	walk_tree( Head, calc_user_ttys, 0 );
}


find_dev_in_users_devs( user_ttys, ttydev )
User_tty_struct *user_ttys;
dev_t ttydev;
{
	User_tty_struct *p;

	/* Scan list of users ttydev's , Allows kill's on other sessions? */

	for ( p = user_ttys; p != NULL ; p = p->next )
		if ( p->ttydev == ttydev )
			return( True );

	return( False );
}

User_tty_struct
*add_dev_to_users_devs( user_ttys, ttydev )
User_tty_struct *user_ttys;
dev_t ttydev;
{
	User_tty_struct *new;

	new = ( User_tty_struct * ) malloc( sizeof( User_tty_struct ) );
	if ( new == NULL )
	{
		fprintf( stderr, "add_dev_to_users_devs: Malloc failed\n");
		return( NULL );
	}

	new->ttydev = ttydev;
	if ( user_ttys == NULL )
		new->next = NULL;
	else
		new->next = user_ttys;

	return( new );
}

User_tty_struct 
*free_user_ttys_list()
{
	User_tty_struct *p, *next_p;

	for ( p = User_ttys; p != NULL; )
	{
		next_p = p->next;
		free( p );
		p = next_p;
	}

	return( NULL );
}

load_cursors()
{
	Pixmap		 signal_pixmap, kill_pixmap;
	Pixmap		 info_pixmap, outline_pixmap, hide_pixmap, show_pixmap;
	Pixmap		 signal_mask_pixmap, kill_mask_pixmap, info_mask_pixmap;
	Pixmap		 outline_mask_pixmap, hide_mask_pixmap, show_mask_pixmap;
	XColor		 fg_color, bg_color;
	Display *dpy;
	int      scr;
	Colormap cmap;


	dpy  = XtDisplay( TreeWidget );
	scr  = DefaultScreen( dpy );
	cmap = DefaultColormap( dpy, scr );

	bg_color.pixel = WhitePixel( dpy, scr );
	fg_color.pixel = BlackPixel( dpy, scr );

	XQueryColor( XtDisplay(TreeWidget), cmap, &fg_color );
	XQueryColor( XtDisplay(TreeWidget), cmap, &bg_color );


        signal_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      signal_bits,
                          BITMAP_DIMENSION_CAST signal_width,
                          BITMAP_DIMENSION_CAST signal_height );

        signal_mask_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      signal_mask_bits,
                          BITMAP_DIMENSION_CAST signal_mask_width,
                          BITMAP_DIMENSION_CAST signal_mask_height );

 	Signal_cursor = XCreatePixmapCursor( dpy, signal_pixmap,
						  signal_mask_pixmap,
						 &fg_color,
						 &bg_color,
						  signal_x_hot,
						  signal_y_hot );

        kill_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      target_bits,
                          BITMAP_DIMENSION_CAST target_width,
                          BITMAP_DIMENSION_CAST target_height );

        kill_mask_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      target_mask_bits,
                          BITMAP_DIMENSION_CAST target_mask_width,
                          BITMAP_DIMENSION_CAST target_mask_height );

 	Kill_cursor = XCreatePixmapCursor( dpy, kill_pixmap,
						  kill_mask_pixmap,
						 &fg_color,
						 &bg_color,
						  target_x_hot,
						  target_y_hot );

        info_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      info_bits,
                          BITMAP_DIMENSION_CAST info_width,
                          BITMAP_DIMENSION_CAST info_height );

        info_mask_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      info_mask_bits,
                          BITMAP_DIMENSION_CAST info_mask_width,
                          BITMAP_DIMENSION_CAST info_mask_height );

 	Info_cursor = XCreatePixmapCursor( dpy,   info_pixmap,
					 	  info_mask_pixmap,
						 &fg_color,
						 &bg_color,
						  info_x_hot,
						  info_y_hot);

 	Busy_cursor   = XCreateFontCursor(XtDisplay(TreeWidget), XC_watch);

        outline_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      outline_bits,
                          BITMAP_DIMENSION_CAST outline_width,
                          BITMAP_DIMENSION_CAST outline_height );

        outline_mask_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      outline_mask_bits,
                          BITMAP_DIMENSION_CAST outline_mask_width,
                          BITMAP_DIMENSION_CAST outline_mask_height );

 	Outline_cursor = XCreatePixmapCursor( dpy,   outline_pixmap,
					 	  outline_mask_pixmap,
						 &fg_color,
						 &bg_color,
						  outline_x_hot, 
						  outline_y_hot  );

        hide_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      hide_bits,
                          BITMAP_DIMENSION_CAST hide_width,
                          BITMAP_DIMENSION_CAST hide_height );

        hide_mask_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      hide_mask_bits,
                          BITMAP_DIMENSION_CAST hide_mask_width,
                          BITMAP_DIMENSION_CAST hide_mask_height );

 	Hide_cursor = XCreatePixmapCursor( dpy,   hide_pixmap,
					 	  hide_mask_pixmap,
						 &fg_color,
						 &bg_color,
						  hide_x_hot,
						  hide_y_hot);

        show_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      show_bits,
                          BITMAP_DIMENSION_CAST show_width,
                          BITMAP_DIMENSION_CAST show_height );

        show_mask_pixmap = XCreateBitmapFromData( dpy,
                          RootWindowOfScreen( XtScreen(Top_level)),
                          BITMAP_DATA_CAST      show_mask_bits,
                          BITMAP_DIMENSION_CAST show_mask_width,
                          BITMAP_DIMENSION_CAST show_mask_height );

 	Show_cursor = XCreatePixmapCursor( dpy,   show_pixmap,
					 	  show_mask_pixmap,
						 &fg_color,
						 &bg_color,
						  show_x_hot,
						  show_y_hot);
}

void
set_cursor( )
{
	Widget w;
	Cursor new_cursor;
	static int first=1;

	if ( first )
	{
		load_cursors();
		first--;
	}

	w = TreeWidget;
	if ( Process_action == ACTION_PROCESS_SELECT )
	{
		XUndefineCursor( XtDisplay(w), XtWindow(w));
		XFlush( XtDisplay(w) );
		return;
	}
	else if ( Process_action == ACTION_PROCESS_SIGNAL )
		new_cursor = Signal_cursor;
	else if ( Process_action == ACTION_KILL_PROCESS )
		new_cursor = Kill_cursor;
	else if ( Process_action == ACTION_PROCESS_DETAILS )
		new_cursor = Info_cursor;
	else if ( Process_action == ACTION_OUTLINE_SUBTREE )
		new_cursor = Outline_cursor;
	else if ( Process_action == ACTION_HIDE_SUBTREE )
		new_cursor = Hide_cursor;
	else if ( Process_action == ACTION_SHOW_SUBTREE )
		new_cursor = Show_cursor;
	else
	{
		XUndefineCursor( XtDisplay(w), XtWindow(w));
		XFlush( XtDisplay(w) );
		return;
	}

	XDefineCursor( XtDisplay(w), XtWindow(w), new_cursor );
	XFlush( XtDisplay(w) );
}

void
change_spacing( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	int n;
	Arg wargs[3];
	Dimension  h_pad;
	Dimension  v_pad;

	n = 0;
	XtSetArg( wargs[n], XtNhorizontalSpace, &h_pad ); n++;
	XtSetArg( wargs[n], XtNverticalSpace,   &v_pad ); n++;
	XtGetValues( TreeWidget, wargs, n );


	h_pad = h_pad + atoi( params[0] );
	v_pad = v_pad + atoi( params[1] );

	DEBUG2(8, "change_spacing: h_pad(%d) v_pad(%d)\n", h_pad, v_pad );

	n = 0;
	XtSetArg( wargs[n], XtNhorizontalSpace, h_pad ); n++;
	XtSetArg( wargs[n], XtNverticalSpace,   v_pad ); n++;
	XtSetValues( TreeWidget, wargs, n );
}

void
orientation( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	int n;
	Arg wargs[2];
	XrmValue  from, to;
	TreeOrientation  new_orientation;
	char *str;

	DEBUG2(8, "orientation: num_params(%d) params[0](%s)\n", 
				*num_params, params[0] );

	if ( *num_params > 0 )
	{

		from.size = strlen(params[0]);
		from.addr = params[0];

		to.size = sizeof( TreeOrientation );
		to.addr = (caddr_t) &new_orientation;

		if ( XtConvertAndStore( w, XtRString, &from,
					XtRTreeOrientation, &to ) == False )
		{
			XtAppError( XtWidgetToApplicationContext(w),
			    "orientation action: String to TreeOrientation \
				conversion failed");

			return;
		}

		DEBUG1(8,"orientation: new_orientation(%d)\n", *(to.addr) );

		str = get_orient_menu_str( new_orientation );
		if ( set_menu_toggle( Main_menu, str, True ) )
			return;

		set_new_orientation( new_orientation );
	}
}

char
*get_orient_menu_str( orientation )
TreeOrientation orientation;
{
	char *str=NULL_STR;

	switch ( orientation )
	{
	    case RIGHT_TO_LEFT:		str = "Right to Left"; break;
	    case LEFT_TO_RIGHT:		str = "Left to Right"; break;
	    case TOP_TO_BOTTOM:		str = "Top to Bottom"; break;
	    case BOTTOM_TO_TOP:		str = "Bottom to Top"; break;
	    case STAR_TOPOLOGY:		str = "Star Topology"; break;

	    default:    fprintf( stderr, "get_orient_menu_str: Uknown type\n");
			break;
	}

	return( str );
}

set_new_orientation( new_orient )
TreeOrientation new_orient;
{
 	int 			n, k;
	Arg 			wargs[6];
	Arg 			in_args[4];
	TreeOrientation 	orient;
	ConnectionStyle		connect;
	NodeSize		node_size;
        Dimension  		h_pad;
        Dimension  		v_pad;
	static int 		was_star=-1;
	static int 		was_vertical=-1;
	static ConnectionStyle	last_conn;
	static NodeSize		last_node_size;
	int			swap_padding=0;


	DEBUG1( 1, "set_new_orientation(%d)\n", new_orient);

	if ( was_star == -1 )	/* First time through so set was_* */
	{
		/* Determine if currently a star orientation */

        	n = 0;
		XtSetArg( wargs[n], XtNtreeOrientation, &orient ); n++;
        	XtGetValues( TreeWidget, wargs, n );

		was_vertical = 0;
		was_star = 0;

		if ( orient == TOP_TO_BOTTOM || orient == BOTTOM_TO_TOP )
		{
			was_vertical = 1;
		}
		else if ( orient == STAR_TOPOLOGY )
		{
			was_star = 1;
		}
	}

 	n = 0;
	if ( new_orient == STAR_TOPOLOGY )
	{
		if ( was_star == 0 ) /* Snarf old style and set to diagonal */
		{
        		n = 0;
			XtSetArg(wargs[n], XtNconnectionStyle, &connect);n++;
			XtSetArg(wargs[n], XtNconnectionStyle, &node_size);n++;
        		XtGetValues( TreeWidget, wargs, n );

			last_conn      = connect;
			last_node_size = node_size;

        		n = 0;
			connect = DIAGONAL_CONNECTION;
			if ( last_node_size == VARIABLE_SIZE )
				node_size = FIXED_SIZE_FOR_LEVEL;

			XtSetArg( wargs[n], XtNconnectionStyle, connect ); n++;
			/* XtSetArg( wargs[n], XtNnodeSize, node_size ); n++;*/
		}

		orient = STAR_TOPOLOGY;
		was_star = 1;
	}
	else
	{
		if ( new_orient == LEFT_TO_RIGHT )
		{
			orient = LEFT_TO_RIGHT;

			if ( was_vertical == 1 )
				swap_padding = 1;

			was_vertical = 0;
		}
		else if ( new_orient == RIGHT_TO_LEFT )
		{
			orient = RIGHT_TO_LEFT;
			if ( was_vertical == 1 )
				swap_padding = 1;

			was_vertical = 0;
		}
		else if ( new_orient == TOP_TO_BOTTOM )
		{
			orient = TOP_TO_BOTTOM;
			if ( was_vertical == 0 )
				swap_padding = 1;

			was_vertical = 1;
		}
		else if ( new_orient == BOTTOM_TO_TOP )
		{
			orient = BOTTOM_TO_TOP;
			if ( was_vertical == 0 )
				swap_padding = 1;

			was_vertical = 1;
		}
	
		/* If we change back from star restore connection style */

		if ( was_star )
		{
			/* Restore connection style */
			XtSetArg( wargs[n], XtNconnectionStyle, last_conn); n++;

			/*
			 *XtSetArg( wargs[n], XtNnodeSize, last_node_size); n++;
			 */

			was_star = 0;
		}
	}

	/* If we change from vertical to horizontal, swap spacing ?? */

	if ( swap_padding == 1)
	{
        	k = 0;
        	XtSetArg( in_args[k], XtNhorizontalSpace, &h_pad ); k++;
        	XtSetArg( in_args[k], XtNverticalSpace,   &v_pad ); k++;
        	XtGetValues( TreeWidget, in_args, k );

        	XtSetArg( wargs[n], XtNhorizontalSpace, v_pad ); n++;
        	XtSetArg( wargs[n], XtNverticalSpace,   h_pad ); n++;
	}

	XtSetArg( wargs[n], XtNtreeOrientation, orient ); n++;
	XtSetValues( TreeWidget, wargs, n );
}

void
connect_style( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	int n;
	Arg wargs[2];
	XrmValue  from, to;
	ConnectionStyle connect_style;
	char *str;

	DEBUG2(8, "connect_style: num_params(%d) params[0](%s)\n", 
				*num_params, params[0] );

	if ( *num_params > 0 )
	{

		from.size = strlen(params[0]);
		from.addr = params[0];

		to.size = sizeof( ConnectionStyle );
		to.addr = (caddr_t) &connect_style;

		if ( XtConvertAndStore( w, XtRString, &from,
					XtRConnectionStyle, &to ) == False )
		{
			XtAppError( XtWidgetToApplicationContext(w),
			    "connect_style action: String to ConnectionStyle \
				conversion failed");

			return;
		}

		DEBUG1(8,"connect_style: new_connectStyle(%d)\n", *(to.addr) );

		/*
		n = 0;
		XtSetArg( wargs[n], XtNconnectionStyle, *(to.addr) ); n++;
		XtSetValues( TreeWidget, wargs, n );
		*/

		str = get_connect_menu_str( connect_style );
		if ( set_menu_toggle( Main_menu, str, True ) )
			return;

		set_new_conn_style( connect_style );
	}
}

char
*get_connect_menu_str( connect_style )
ConnectionStyle connect_style;
{
	char *str=NULL_STR;

	switch ( connect_style )
	{
	    case DIAGONAL_CONNECTION:	str = "Diagonal"; break;
	    case TIER_CONNECTION:	str = "Tier"; break;
	    case PATHWAY_CONNECTION:	str = "Pathway"; break;
	    case NO_CONNECTION:		str = "None"; break;

	    default:    fprintf( stderr, "get_connect_menu_str: Uknown type\n");
			break;
	}

	return( str );
}

void
toggle_decorations( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	int newDecor;

	DEBUG2(8, "toggle_decorations: num_params(%d) params[0](%s)\n", 
				*num_params, params[0] );

	if ( Decorated_tree == DECORATE_NONE )
	{
 		newDecor = DECORATE_DISTINGUISHED;
		
		set_menu_toggle( Main_menu,"Distinguished",True);
	}
	else
	{
 		newDecor = DECORATE_NONE;
		
		set_menu_toggle( Main_menu,"No Decorations",True);
	}
}

void
toggle_pid( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	DEBUG2(8, "toggle_pid: num_params(%d) params[0](%s)\n", 
				*num_params, params[0] );

	if ( Display_pid == True )
		Display_pid = False;
	else
		Display_pid = True;

	startBatchUpdate();
	walk_tree( Head, redisplay_node, 0 );
	finishBatchUpdate();
}

void
toggle_outline( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	Process_tree_info  *pti_ptr;

        XpsiDoTreeLayout( TreeWidget, False );

        pti_ptr = (Process_tree_info *)Head->data;
        if ( pti_ptr->outline == True )
                walk_tree( Head, outline_all, False );
        else
                walk_tree( Head, outline_all, True );
        walk_tree( Head, redisplay_node, 0 );

        XpsiDoTreeLayout( TreeWidget, True );
}


void
show_all_processes( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	set_menu_toggle( Main_menu,"All Users",True);
}

void
show_user_processes( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	set_menu_toggle( Main_menu,"Login User",True);
}

void
show_hidden( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	XpsiDoTreeLayout( TreeWidget, False );
        walk_tree( Head, display_node_widget, 0 );
        XpsiDoTreeLayout( TreeWidget, True );	
}

void
show_find( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	popup_find_dialog( Top_level );
}

startBatchUpdate()
{
	/* XpsiDoTreeRedisplay( TreeWidget, False ); */
	XpsiDoTreeLayout( TreeWidget, False );
}

finishBatchUpdate()
{
	XpsiDoTreeLayout( TreeWidget, True );
	/* XpsiDoTreeRedisplay( TreeWidget, True ); */
}

void
toggle_mem( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	if ( Record_memuse )
		Record_memuse = 0;
	else
		Record_memuse = 1;

printf("Record_memuse now(%d)\n", Record_memuse );

}

tnode
*redisplay_node( ptr, noop )
tnode *ptr;
void  *noop;
{
	display_proc_info( ptr, 0 );
	return( NULL );
}

void
outline_subtree( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	tnode *ptr;
	DEBUG2(8, "outline_subtree: num_params(%d) params[0](%s)\n", 
				*num_params, params[0] );

	ptr = walk_tree( Head, match_widget, w );
	if ( ptr == NULL )
	{
		DEBUG1(1, "outline_subtree: can't find widget for (%d)\n", w );
		return;
	}

	startBatchUpdate();
	walk_tree( ptr, outline_node, 0 );
	walk_tree( ptr, redisplay_node, 0 );
        finishBatchUpdate();
}

tnode
*outline_node( ptr, noop )
tnode *ptr;
void  *noop;
{
	Process_tree_info  *pti_ptr;

	pti_ptr = (Process_tree_info *) ptr->data;
	if ( pti_ptr == NULL )
		return( NULL );

	if ( pti_ptr->outline == True )
		pti_ptr->outline = False;
	else
		pti_ptr->outline = True;

	return( NULL );
}

set_new_conn_style( conn )
ConnectionStyle conn;
{
	int n;
	Arg wargs[2];

	n = 0;
	XtSetArg( wargs[n], XtNconnectionStyle, conn ); n++;
	XtSetValues( TreeWidget, wargs, n );
}

set_new_placement( pp )
ParentPlacement pp;
{
	int n;
	Arg wargs[2];

	n = 0;
	XtSetArg( wargs[n], XtNparentPlacement, pp ); n++;
	XtSetValues( TreeWidget, wargs, n );
}

void
increase_debug( w, event, params, num_params )
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	Debug += 1;

	if ( Debug > 15 )
		Debug = 0;

	DEBUG1( 1, "Debug level now(%d)\n", Debug );
}


set_process_attach_mode( attach_mode )
int attach_mode;
{
	if ( attach_mode != Process_attach_mode )
	{
	        DEBUG1(1, "set_process_attach_mode: attach_mode(%d)\n", attach_mode);
		startBatchUpdate();
		Process_attach_mode = attach_mode;
		post_walk_tree( Head, reparent_process, attach_mode ); 
        	finishBatchUpdate();
	}
}

tnode
*orphan_process( head, ptr )
tnode *head;
tnode *ptr;
{
	Process_tree_info    	*pti_ptr;
	Process_tree_info    	*ppti_ptr;
	Process_list_info    	*pli;
	Attach_point		*atp;
	Attach_list		 l;
	sysProcInfo		*pp;
	int		 	show_groups;
	int		 	pid, ppid, pgrp, sid, uid, attachTo_key;
	Widget		 	parent_widget;
	tnode			*new_parent;
	int 			n;
	Arg 			wargs[2];

	if ( ptr == NULL )
		return(head);

	pti_ptr = (Process_tree_info *) ptr->data;

	if ( pti_ptr == NULL || pti_ptr->pp == NULL )
		return(head);

	pp = pti_ptr->pp;
	if ( pp == NULL )
		return;

	ppid = PROCESS_PARENT_ID( pp );
	pgrp = PROCESS_GROUP_LEADER( pp ); 
        uid  = PROCESS_USER_ID( pp );
        sid  = PROCESS_SESSION_ID( pp );

        attachTo_key = ppid;

	pli = (Process_list_info *) pti_ptr->l_ptr->data;

	DEBUG0(1, "orphan_process: Attach_list = ");

	l = pli->attachments;
	while ( l )
	{
		atp = (Attach_point *) l->data;
		DEBUG1(1, "%d, ", atp->attachToPid );
		l = l->next;
	}
	DEBUG0(1, "\n");

	pid = ptr->key;
/* 
	attachTo_key = attachToNode( Head, pti_ptr, pid, pgrp, sid, ppid, uid, pp ); 

finds and attaches to parent again!!! Then parent gets deleted and child has wrong parent_key!!!
Widget get orphaned though! Need to ignore current parent as a target attachment point.
*/
        attachTo_key = 1;

	/* Look for parent */
		
	new_parent = walk_tree( Head, match_key, attachTo_key );
		
	if ( new_parent == NULL ) /*Parent or proc leader, not visible*/
	{
		new_parent = walk_tree( head, match_key, 1 );
		if ( new_parent == NULL )
		{
			DEBUG0(1,"orphan_process: can't find init !!!\n");
			return( head );
		}
	}

	DEBUG4(0,"orphan_process: %s(%d) to %s(%d)\n", ptr->str_key, ptr->key,
					new_parent->str_key, new_parent->key);

	move_tnode( ptr, new_parent );
	ppti_ptr = (Process_tree_info *) new_parent->data;
	parent_widget   = ppti_ptr->node_w;
	
	if ( pti_ptr->state == NODE_HIDDEN )
		return( head );
	n = 0;
	XtSetArg( wargs[n], XtNparentWidget, parent_widget ); n++;
	XtSetValues( pti_ptr->node_w, wargs, n );

	return( head );
}



tnode
*reparent_process( ptr, data )		/* move to new parent if required */
tnode *ptr;
int data;
{
	sysProcInfo	*pp;
	Process_tree_info    *pti_ptr;
	Process_tree_info    *ppti_ptr;
	int		 process_attach_mode;
	Widget		 parent_widget;
	tnode		*new_parent=NULL;
	int 		n, pid, ppid, pgrp, sid, uid, cur_uid, attachTo_key;
	Arg wargs[2];

	process_attach_mode = data;

	if ( ptr == NULL )
		return(NULL);

	pti_ptr = (Process_tree_info *) ptr->data;

	if ( pti_ptr == NULL || pti_ptr->pp == NULL )
		return(NULL);

	pp = pti_ptr->pp;
	if ( pp == NULL )
		return;

	ppid = PROCESS_PARENT_ID( pp );
	pgrp = PROCESS_GROUP_LEADER( pp ); 
        uid  = PROCESS_USER_ID( pp );
	sid  = PROCESS_SESSION_ID( pp );

        attachTo_key = ppid;
	if ( PROCESS_PARENT_ID(pp) == 1 ) 
	{
	    pid = ptr->key;
	    attachTo_key = attachToNode( Head, pti_ptr, pid, pgrp, sid, ppid, uid, pp );
        }

	/* Look for new attachment position(parent) */
		
	if ( attachTo_key != ptr->parent_key )  /* should it be != current attachment */
	{
	    new_parent = walk_tree( Head, match_key, attachTo_key );
		
	    if ( new_parent == NULL ) /*Parent or proc leader, not visible*/
	    {
		new_parent = walk_tree( Head, match_key, 1 ); /* use init */
		if ( new_parent == NULL )
			return( NULL );
		attachTo_key = 1;
	    }

	    DEBUG3(0, "Move(%d) from parent(%d) to (%d)\n", ptr->key, 
	    				ptr->parent_key, attachTo_key );

	    /* ptr->parent_key = attachTo_key; */

	    move_tnode( ptr, new_parent );
	    ppti_ptr =(Process_tree_info *)new_parent->data;
	    parent_widget   = ppti_ptr->node_w;

	    if ( pti_ptr->state != NODE_HIDDEN )
	    {
	      n = 0;
	      XtSetArg( wargs[n], XtNparentWidget, parent_widget ); n++;
	      XtSetValues( pti_ptr->node_w, wargs, n );
	    }
	}
	
	return( NULL );
}


/* ----------------- Routines affecting the node displays -------------- */

lnode
*set_default_display_list( data )
ApplicationData *data;
{
	lnode  *list;

	/* Should parse a list from a resource */

	list = (lnode *) NULL;

	if ( data->disp_pid == TRUE )
		list = fwd_insert_lnode( list, PR_PID, NULL );

	if ( data->disp_pname == TRUE )
		list = fwd_insert_lnode( list, PR_FNAME, NULL );

	return( list );
}


void	
show_display_list()
{
	XmString		 *items;	/* Selected items  */
	int			 nitems;
	int		 	 i, n;
	Arg		 	 wargs[4];
	lnode			 *p;
	Widget			  w;

	if ( Display_list_pw == NULL )
	{
		popup_display_list( Top_level );	/* Create and manage */

		nitems = PR_LAST + 1;	   /* In case they are all selected */

		items = (XmString *) XtMalloc( sizeof( XmString ) * nitems );
		if ( items == NULL )
		{
			fprintf(stderr, "Malloc failed for selected list\n");
			return;
		}

		i=0;
		for ( p = Global_display_list; p != NULL ; p = p->next )
		{
			items[i++] = XmStringCreate( 
					      Node_Info_Desc[p->key].short_name,
					      XmSTRING_DEFAULT_CHARSET );
		}

		n = 0;
		XtSetArg( wargs[n], XmNselectedItems, items); n++;
		XtSetArg( wargs[n], XmNselectedItemCount, i); n++;
		XtSetValues( Display_list_lw1, wargs, n );

		if ( Global_display_list->key == 1 )
		    xs_wprintf(Display_list_cw,"%s",Node_Info_Desc[1].description);
		else
		    xs_wprintf(Display_list_cw,"%s",Node_Info_Desc[0].description);
	}
	else
	{
		popup_display_list( Top_level );	/* Manages it */
	}
}

void    
popup_display_list( parent_w )
Widget parent_w;
{
	Widget  	 	 dw, list1_w, list2_w, list3_w, sep1, comment;
	Widget  	 	 attachto, sep2, done, help;
	Widget			 gni_update_rate, sep3;
	Widget			 parent;
        Widget			 pulldown;
        Widget  		 button[3];
	XmString  		 option_label;
	Arg		 	 wargs[16];
	int		 	 i, n;
	char			 title_str[80];
	int			 nitems;
	XmString		 *items;
	int			 itemsPerPanel=7;


	 if ( parent_w == NULL )
	 {
		user_alert(parent_w, "Internal error");
		return;
	 }


	if ( Display_list_pw != NULL )
	{
		XtManageChild( Display_list_pw );
		XmProcessTraversal( Display_list_lw1, XmTRAVERSE_CURRENT );
		return;
	}

	/* If we get here we need to create a form dialog with appropriate
	 * widgets to allow the user to select fields for the global
	 * display_list.
	 */


	sprintf( title_str, "Display List" );


	n = 0;
	XtSetArg( wargs[n], XmNautoUnmanage, FALSE ); n++;
	XtSetArg( wargs[n], XmNfractionBase, 10 ); n++;
	XtSetArg( wargs[n], XmNtitle, title_str); n++;
	dw = XmCreateFormDialog( parent_w, "display_list", wargs,  n);
	if ( dw == NULL )
	{
		DEBUG0( 0, "popup_display_list: failed to create form.\n" );
		return;
	}


	Display_list_pw = dw;
	parent = dw; 

	/* Create Button to close display list  popup */

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 1); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 4); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;

	done = XtCreateManagedWidget("Done", xmPushButtonWidgetClass, 
						parent, wargs, n );

	XtAddCallback( done, XmNactivateCallback, popdown_display_list, (XtPointer)0 );

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 6); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 9); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;

	help = XtCreateManagedWidget("Help", xmPushButtonWidgetClass, 
						parent, wargs, n );

	XtAddCallback( help, XmNactivateCallback, do_help, (XtPointer)"display_list" );


	n = 0;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, done); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep3 = XtCreateManagedWidget( "dl_separator3", xmSeparatorGadgetClass, 
						parent, wargs, n );

	pulldown = (Widget) XmCreatePulldownMenu( parent, "p1", NULL, 0);

        XtSetArg( wargs[0], XmNmnemonic, 'S'); 
	button[0] = XmCreatePushButtonGadget(pulldown, "Short Tic", wargs, 1);
    	XtAddCallback(button[0], XmNactivateCallback, do_display_rate, (XtPointer)0);

        XtSetArg( wargs[0], XmNmnemonic, 'L'); 
	button[1] = XmCreatePushButtonGadget(pulldown, "Long Tic", wargs, 1);
    	XtAddCallback(button[1], XmNactivateCallback, do_display_rate, (XtPointer)1);

	XtManageChildren( button, 2 );

	option_label = XmStringCreate( "Update On:", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, sep3 ); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;

        XtSetArg( wargs[n], XmNlabelString, option_label); n++; 
        XtSetArg( wargs[n], XmNoptionMnemonic, 'U'); n++;  
        XtSetArg( wargs[n], XmNsubMenuId, pulldown); n++;
        XtSetArg( wargs[n], XmNmenuHistory, button[1]); n++;
        gni_update_rate = (Widget)XmCreateOptionMenu( parent, 
							"Display_update_rate",
								   wargs, n );
	XmStringFree( option_label );
	XtManageChild( gni_update_rate );


	n = 0;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, gni_update_rate); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep2 = XtCreateManagedWidget( "dl_separator2", xmSeparatorGadgetClass, 
						parent, wargs, n );

	/* Create comment label  to display meaning of field   		*/

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, sep2); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	comment = XtCreateManagedWidget("dl_label", xmLabelWidgetClass, 
						parent, wargs, n );

	Display_list_cw = comment;

	/* Build xmstring table to hold the selectable items for item  0 -  6 */

#ifdef SVR4_MP
	itemsPerPanel=8;
#endif

	nitems = PR_LAST + 1;

	items = (XmString *) XtMalloc( sizeof( XmString ) * nitems );
	if ( items == NULL )
	{
		fprintf(stderr, "Malloc failed for display list\n" );
		return;
	}

	for ( i = 0 ; i < itemsPerPanel ; i ++ )
	{
		items[i] = XmStringCreate( Node_Info_Desc[i].short_name,
						    XmSTRING_DEFAULT_CHARSET);
	}

	/* Create list widget to hold list of displayable fields */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 3); n++;
	XtSetArg( wargs[n], XmNselectionPolicy, XmMULTIPLE_SELECT); n++;
	XtSetArg( wargs[n], XmNitems, items); n++;
	XtSetArg( wargs[n], XmNitemCount, itemsPerPanel); n++;
#ifdef LESSTIF
	XtSetArg( wargs[n], XmNvisibleItemCount, itemsPerPanel+1); n++;
#else
	XtSetArg( wargs[n], XmNvisibleItemCount, itemsPerPanel); n++;
#endif
	XtSetArg( wargs[n], XmNlistSizePolicy, XmRESIZE_IF_POSSIBLE); n++;
	list1_w = XtCreateManagedWidget( "dl_list1", xmListWidgetClass, 
						parent, wargs, n );
	attachto = list1_w;

	Display_list_lw1 = list1_w;

	XtAddCallback( list1_w, XmNmultipleSelectionCallback, 
						select_display_field, (XtPointer)0 );



	/* Build xmstring table to hold the second panel */

	for ( i = 0 ; i < itemsPerPanel ; i ++ )
	{
		items[i] = XmStringCreate( Node_Info_Desc[i+itemsPerPanel].short_name,
						    XmSTRING_DEFAULT_CHARSET);
	}

	/* Create list widget to hold list of displayable fields */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 3); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 7); n++;
	XtSetArg( wargs[n], XmNselectionPolicy, XmMULTIPLE_SELECT); n++;
	XtSetArg( wargs[n], XmNitems, items); n++;
	XtSetArg( wargs[n], XmNitemCount, itemsPerPanel); n++;
#ifdef LESSTIF
	XtSetArg( wargs[n], XmNvisibleItemCount, itemsPerPanel+1); n++;
#else
	XtSetArg( wargs[n], XmNvisibleItemCount, itemsPerPanel); n++;
#endif
	XtSetArg( wargs[n], XmNlistSizePolicy, XmRESIZE_IF_POSSIBLE); n++;
	list2_w = XtCreateManagedWidget( "dl_list2", xmListWidgetClass, 
						parent, wargs, n );

	Display_list_lw2 = list2_w;

	XtAddCallback( list2_w, XmNmultipleSelectionCallback, 
						select_display_field, (XtPointer)0 );



	/* Build xmstring table to hold the last panel */

	for ( i = 0 ; i < itemsPerPanel ; i ++ )
	{
		items[i] = XmStringCreate( Node_Info_Desc[i+(2*itemsPerPanel)].short_name,
						    XmSTRING_DEFAULT_CHARSET);
		if ( (i+(2*itemsPerPanel)) == PR_LAST )
			break;
	}

	/* Create list widget to hold list of displayable fields */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 7); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNselectionPolicy, XmMULTIPLE_SELECT); n++;
	XtSetArg( wargs[n], XmNitems, items); n++;
	XtSetArg( wargs[n], XmNitemCount, i+1); n++;
#ifdef LESSTIF
	XtSetArg( wargs[n], XmNvisibleItemCount, i+2); n++;
#else
	XtSetArg( wargs[n], XmNvisibleItemCount, i+1); n++;
#endif
	XtSetArg( wargs[n], XmNlistSizePolicy, XmRESIZE_IF_POSSIBLE); n++;
	list3_w = XtCreateManagedWidget( "dl_list3", xmListWidgetClass, 
						parent, wargs, n );
	Display_list_lw3 = list3_w;

	XtAddCallback( list3_w, XmNmultipleSelectionCallback, 
						select_display_field, (XtPointer)0 );


	/* Create separator at bottom of popup */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, attachto); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, comment); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	sep1 = XtCreateManagedWidget( "dl_separator1", xmSeparatorGadgetClass, 
						parent, wargs, n );

	XtManageChild( dw ); 

	XtAddCallback( XtParent(dw), XmNpopdownCallback, popdown_display_listCB, 
						(XtPointer)0 ); 

	XmProcessTraversal( Display_list_lw1, XmTRAVERSE_CURRENT );
}

void
do_display_rate( w, button_number, call_data )
Widget w;
int    button_number;
XmAnyCallbackStruct *call_data;
{
	lnode 		  *p;
	Process_list_info *pli_ptr;

	DEBUG1( 1, "Display_rate: button(%d)\n", button_number );

	if ( button_number == 0 )
	{
		Visible_processes_active = 1; 

		/* Move all inactive visible processes to active list 
		 * and lock in.
		 */

		for ( p = Process_list; p != NULL ; p = p->next )
		{
		    pli_ptr = (Process_list_info *) p->data;
		    if ( pli_ptr == NULL )
			continue;

		    if ( pli_ptr->t_ptr )	/* It's visible */
			active_lock( p );
		}
	}
	else
	{
		Visible_processes_active = 0; 

		for ( p = Process_list; p != NULL ; p = p->next )
		{
		    pli_ptr = (Process_list_info *) p->data;
		    if ( pli_ptr == NULL )
			continue;

		    if ( pli_ptr->t_ptr )	/* It's visible */
			active_unlock( p );
		}
	}
}

void
select_display_field( w, client_data, call_data )
Widget w;
caddr_t *client_data;
XmListCallbackStruct *call_data;
{
	char *text=NULL_STR;
	int   i;
	int   field=-1;
	int   in_list=0;
	Boolean rc;
	int   info_type;


	rc = XmStringGetLtoR( call_data->item, XmSTRING_DEFAULT_CHARSET, &text);
	if ( rc != True )
	{
		fprintf( stderr, "Failed to find text segment!!\n" );
		return;
	}

	for ( i = 0 ; i <= PR_LAST ; i++ )
	{
		if ( strcmp( text, Node_Info_Desc[i].short_name ) == 0 )
		{
			field = Node_Info_Desc[i].field;
			break;
		}
	}

	if ( field == -1 )
	{
		fprintf( stderr, "Failed to find field(%s)!!\n", text );
		return;
	}

	if ( match_list( Global_display_list, field ) )
	{
		Global_display_list = delete_lnode(Global_display_list, field);
	}
	else
	{
		Global_display_list=fwd_insert_lnode(Global_display_list,field,
								NULL);

		xs_wprintf(Display_list_cw, "%s", Node_Info_Desc[i].description );
	}

	Dynamic_info = is_list_dynamic( Global_display_list );

	startBatchUpdate();
	walk_tree( Head, display_proc_info, 0 );
        finishBatchUpdate();

	if ( text )
		free( text ); 
}


is_list_dynamic( list )
lnode *list;
{
	int    dynamic=0;
	int    infrequent=0;
	lnode *p;

	for ( p = list; p != NULL ; p = p->next )
	{
		switch ( p->key )
		{
			case PR_FNAME:
			case PR_PPID:
			case PR_EUID:
			case PR_EGID:
			case PR_PGRP:
			case PR_SID:
			case PR_CLNAME:
			case PR_TTYDEV:		infrequent = 1;
						break;

			case PR_SNAME:
			case PR_FLAGS:
			case PR_NICE:
			case PR_RSSIZE:
			case PR_WCHAN:
			case PR_TIME:
#ifdef SVR4_MP
			case PR_NLWP:
			case PR_ONPRO:
#endif
			case PR_PRI:		dynamic = 1;
						break;

			default:  break;

		}
	}

	if ( dynamic )
		return( DYNAMIC_INFO );
	else if ( infrequent )
		return( INFREQUENT_INFO );
	else
		return( 0 );
}

void    
popdown_display_list()
{
	XtUnmanageChild( Display_list_pw );

	/* When the widget is poped down the following routine 
	 * is called.  It calls the routine registered by the button
	 * to toggle off the fields button.
	 */
}

void    
popdown_display_listCB()
{
	if ( DisplayListPopdownCB != NULL )
		(*DisplayListPopdownCB)();
}

void
popup_procinfo( parent_w, ptr )
Widget parent_w;
tnode  *ptr;
{
	Process_tree_info    	*pti_ptr;
	Process_list_info    	*pli_ptr;
	Node_info_widgets 	*diw;
	Widget  	 	 dw, parent, attachto, label, sep1, sep2, sep3;
	Widget  	 	 update_label, update_now, sw, sep4;
	Widget  	 	 ni_update_rate, di_update_rate;
	Widget			 done, toggle_view, help, frame;
	Widget			 pulldown, top_button;
	WidgetList		 widgets;
	XmString  		 button_string;
	Widget			 button[4];
	XmString  		 option_label;
	Arg		 	 wargs[23];
	int		 	 i, n;
	char			 title_str[80];
	lnode			*l_ptr;
	Dimension		height;

	pti_ptr = (Process_tree_info *)    ptr->data;
	if ( pti_ptr == NULL )
		return;


	l_ptr = (lnode *)pti_ptr->l_ptr;
	if ( l_ptr == NULL )
		return;


	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
		return;

	/* Need to refresh info, if not on the active list */

	if ( pli_ptr->active == 0 )
		refresh_process_info( TreeWidget, l_ptr );

	/* We could potentially remove the node of interest by calling,
	 * refresh_process_info!!! This would occur if the process just
	 * became non-visible under the current display criteria!
	 *
	 * This would make ptr point to free'd space, perhaps reassigned!
	 *
	 * However l_ptr will still be ok, as the node has not been removed
	 * from the process list. So we do a sanity check, and tell the
	 * disgruntled user what happened.
	 *
	 *
	 *
	 * XXX
	 *
	 * The process display list, popup window widget and details widget
	 * should be moved to the process_list structure. This would solve
	 * above problem as well as keeping this info arround if user 
	 * changes views(something that will happen)
	 *
	 */

	 if ( pli_ptr->t_ptr == NULL )
	 {
		user_alert(parent_w, 
		       "Process just changed state, change display criteria");
		return;
	 }


	if ( pti_ptr->dialog_state == DIALOG_HIDDEN )
	{
		DEBUG1( 5, "popup_procinfo: dialog already created popup(%s)\n",
							        ptr->str_key );
		display_procinfo( parent_w, ptr );

		pti_ptr->dialog_state = DIALOG_DISPLAYED;
		XtManageChild( pti_ptr->dialog_w );
		return; 
	}
	else if ( pti_ptr->dialog_state != NO_DIALOG )
	{
		DEBUG1( 5, "popup_procinfo: dialog already displayed (%s)\n",
							        ptr->str_key );
		return; 
	}

	/* If we get here we need to create a form dialog with appropriate
	 * widgets to display the process info.
	 */

	/* first create the structure to hold the updatable widget pointers */

	diw = (Node_info_widgets *) malloc(sizeof(Node_info_widgets));
	if ( diw == NULL )
	{
		DEBUG1( 0, "popup_procinfo: malloc of Node_info_widgets(%s)\n",
							       ptr->str_key );
		return; 
	}

	/* Create space to hold widgets, so we can manage all at once */

	widgets = (WidgetList) XtMalloc( 85 * sizeof( Widget ) );
	i=0;

	sprintf( title_str, "Details for:  %s:%d", ptr->str_key, ptr->key );

	n = 0;
	XtSetArg( wargs[n], XmNautoUnmanage, FALSE ); n++;
	XtSetArg( wargs[n], XmNtitle, title_str); n++;
	XtSetArg( wargs[n], XmNuserData,  ptr->key); n++;
	dw = XmCreateFormDialog( parent_w, "Treeps procInfo", wargs,  n);
	if ( dw == NULL )
	{
		DEBUG1( 0, "popup_procinfo: failed to create form(%s)\n",
							        ptr->str_key );
		return;
	}

	parent = dw; 

	/* Create Button to close info popup */

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 4); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 24); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;

	done = XtCreateWidget("Done", xmPushButtonWidgetClass, 
						parent, wargs, n );
 	widgets[i++] = done;

	XtAddCallback( done, XmNactivateCallback, popdown_procinfo, (XtPointer)ptr->key );

	/* Create Button to toggle between text/numbers */

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 28); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 48); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	toggle_view = XtCreateWidget("Text", xmPushButtonWidgetClass, 
						parent, wargs, n );
 	widgets[i++] = toggle_view;

	XtAddCallback(toggle_view,XmNactivateCallback,toggle_procinfo,(XtPointer)ptr->key);


	/* Create Button to update info popup */

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 52); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 72); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;

	update_now = XtCreateWidget("Update", xmPushButtonWidgetClass, 
						parent, wargs, n );
 	widgets[i++] = update_now;

	XtAddCallback( update_now, XmNactivateCallback, update_procinfo,
								     (XtPointer)ptr->key);

	/* Create Button to display info popup help */

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 76); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 96); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	help = XtCreateWidget("Help", xmPushButtonWidgetClass, 
						parent, wargs, n );
 	widgets[i++] = help;

	XtAddCallback( help, XmNactivateCallback, do_help, (XtPointer)"details" );
	top_button = done;

#ifdef LINUX

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, top_button); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 3); n++;
	sep4 = XtCreateWidget( "separator4", xmSeparatorGadgetClass, 
						parent, wargs, n );
 	widgets[i++] = sep4;

	top_button = linuxDetailsPanel( parent, sep4, ptr->key, ptr->str_key );

#else

#ifdef SVR4_MP

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, top_button); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 3); n++;
	sep4 = XtCreateWidget( "separator4", xmSeparatorGadgetClass, 
						parent, wargs, n );
 	widgets[i++] = sep4;

	top_button = detailsPanel( parent, sep4, ptr->key, ptr->str_key );

#endif

#endif

	/* Process Name */

	label 	    = create_label("Command:", parent, select_info, PR_FNAME);
	diw->name_w = create_ltext("name_w", 22,  parent, &frame);
	align_attach_widgets( NULL, 20, label, frame );
 	widgets[i++] = label;

	/* Process ID */

	label 	    = create_label("PID:", parent, select_info, PR_PID);
	diw->pid_w  = create_ltext("pid_w", 6, parent, &frame);
	align_attach_widgets( NULL, 80, label, frame );
 	widgets[i++] = label;

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, frame); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep1 = XtCreateWidget( "separator1", xmSeparatorGadgetClass, 
						parent, wargs, n );
 	widgets[i++] = sep1;

	attachto = sep1;

	/* Process arguments */

	label = create_label("Args:", parent, select_info, PR_PSARGS );

	n = 0;
	XtSetArg( wargs[n], XmNheight, &height); n++;
	XtGetValues( diw->pid_w, wargs, n );

	n = 0;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNscrollBarDisplayPolicy, XmAS_NEEDED); n++;
	XtSetArg( wargs[n], XmNscrollingPolicy, XmAUTOMATIC); n++;

	/* This is to create enough space for the scroll bar, it's
	 * not as precise as it should be but works in most cases.
	 */

	if ( strlen( PROCESS_CMD_ARGS(pli_ptr->pp)) > 49 )
	{
		XtSetArg( wargs[n], XmNheight, height*3); n++;
	}
	else
	{
		XtSetArg( wargs[n], XmNheight, height+6); n++;
	}

        XtSetArg( wargs[n], XmNshadowThickness, 1); n++;
	XtSetArg( wargs[n], XmNspacing, 0); n++;
	sw = XtCreateManagedWidget( "argsScrolledWindow", 
					xmScrolledWindowWidgetClass,
					parent, wargs, n );

	n = 0;
	XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	XtSetArg( wargs[n], XmNmarginHeight, 1); n++;
	XtSetArg( wargs[n], XmNmarginWidth, 3); n++;
	diw->args_w = XtCreateManagedWidget( "args_w", xmLabelWidgetClass, 
						sw, wargs, n);


	/* diw->args_w  = create_ltext("args_w", 40 , sw, &frame); */
	
	align_attach_widgets( attachto, 20, label, sw );

	/* align_attach_widgets( attachto, 20, label, frame ); */
	/* attachto = frame; */

 	widgets[i++] = label;

	attachto = sw;

	/* Process started at */

	label 	    = create_label("Started:", parent, select_info, PR_START);
	diw->start_w  = create_ltext("start_w", 24, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;

	/* Process run time */

	label = create_label("CPU Time:", parent, select_info, PR_TIME);
	diw->time_w  = create_ltext("time_w", 11, parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	n = 0;
	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetValues( frame, wargs, n );

	attachto = frame;

	/* Process Status */

	label = create_label("Status:", parent, select_info, PR_SNAME);
	diw->status_w  = create_ltext("status_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;

	/* Scheculing Class Name */

	label = create_label("Scheduling Class:",parent, select_info, PR_CLNAME);
	diw->class_name_w  = create_ltext("class_name_w", 11, parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;


	attachto = frame;

	/* Waiting On */

	label = create_label("Wait Addr:", parent, select_info, PR_WCHAN);
	diw->wait_addr_w  = create_ltext("wait_addr_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;



	/* Flags  */

	label 	    = create_label("Flags:", parent, select_info, PR_FLAGS);
	diw->flags_w  = create_ltext("flags_w", 11, parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	/* Process Priority */

	label = create_label("Priority:", parent, select_info, PR_PRI);
	diw->priority_w  = create_ltext("priority_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;

#ifdef LINUX
	/* Nice Value */

	label 	    = create_label("Nice:", parent, select_info, PR_NICE);
	diw->nice_w  = create_ltext("nice_w", 11, parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;
#else

	/* Old Process Priority */

#ifdef SVR4_MP
	label = create_label("Num LWP:", parent, select_info, PR_NLWP );
#else
	label = create_label("Old Priority:", parent, select_info, -1 );
#endif
	diw->old_priority_w  = create_ltext("old_priority_w", 11, parent, 
								&frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	/* Nice Value */

	label 	    = create_label("Nice:", parent, select_info, PR_NICE);
	diw->nice_w  = create_ltext("nice_w", 3, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;


	/* Old CPU Usage */

#ifdef SVR4_MP
	label = create_label("On CPU:", parent, select_info, PR_ONPRO );
#else
	label = create_label("Old CPU Usage:", parent, select_info, -1 );
#endif
	diw->old_cpu_usage_w  = create_ltext("old_cpu_usage_w", 11, parent, 
								&frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;
#endif
	/* User ID */

	label = create_label("User ID:", parent, select_info, PR_UID);
	diw->uid_w  = create_ltext("uid_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;


	/* Effective User ID */

	label = create_label("Effective UID:", parent, select_info, PR_EUID);
	diw->euid_w  = create_ltext("euid_w", 11, parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	/* Group ID */

	label = create_label("Group ID:", parent, select_info, PR_GID);
	diw->gid_w  = create_ltext("gid_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;

	/* Effective Group ID */

	label = create_label("Effective GID:", parent, select_info, PR_EGID);
	diw->egid_w  = create_ltext("gid_w", 11, parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	/* Parent PID */

	label = create_label("Parent PID:", parent, select_info, PR_PPID);
	diw->ppid_w  = create_ltext("ppid_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;


	/* Group Leader PID */

	label = create_label("Group Leader:", parent, select_info, PR_PGRP);
	diw->group_leader_pid_w  = create_ltext("group_leader_pid_w", 11, 
							    parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	
	/* Session ID */

	label = create_label("Session ID:", parent, select_info, PR_SID);
	diw->session_id_w  = create_ltext("session_id_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;


	/* Controlling TTY*/

	label = create_label("Controlling TTY:", parent, select_info,PR_TTYDEV);
	diw->controlling_tty_w  = create_ltext("controlling_tty_w", 11, 
							    parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	/* Image Size */

	label = create_label("Image Size:", parent, select_info, PR_SIZE);
	diw->image_size_w  = create_ltext("image_size_w", 11, parent, &frame);
	align_attach_widgets( attachto, 20, label, frame );
 	widgets[i++] = label;


	/* Memory Resident */

	label = create_label("Memory Resident:", parent, select_info,PR_RSSIZE);
	diw->memory_resident_w  = create_ltext("memory_resident_w", 11, 
							    parent, &frame);
	align_attach_widgets( attachto, 80, label, frame );
 	widgets[i++] = label;

	attachto = frame;

	/* Create separator for update controls */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, attachto); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep2 = XtCreateWidget( "separator2", xmSeparatorGadgetClass, 
						parent, wargs, n );
 	widgets[i++] = sep2;
	attachto = sep2;


	/* Create Popup Update_rate option menu */

	pulldown = (Widget) XmCreatePulldownMenu( parent, "p1", NULL, 0);

	n = 0;
        XtSetArg( wargs[0], XmNmnemonic, 'R'); n++;
	button[0] = XmCreatePushButtonGadget(pulldown, "Request", wargs, n);
    	XtAddCallback(button[0], XmNactivateCallback, do_detail_rate, (XtPointer)0);

	n = 0;
        XtSetArg( wargs[0], XmNmnemonic, 'S'); n++;
	button[1] = XmCreatePushButtonGadget(pulldown, "Short Tic", wargs, n);
    	XtAddCallback(button[1], XmNactivateCallback, do_detail_rate, (XtPointer)1);

	n = 0;
        XtSetArg( wargs[0], XmNmnemonic, 'L'); n++;
	button[2] = XmCreatePushButtonGadget(pulldown, "Long Tic", wargs, n);
    	XtAddCallback(button[2], XmNactivateCallback, do_detail_rate, (XtPointer)2);

	XtManageChildren( button, 3 );

        option_label = XmStringCreate( "Popup On", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 15); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, attachto); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
        XtSetArg( wargs[n], XmNlabelString, option_label); n++; 
        XtSetArg( wargs[n], XmNoptionMnemonic, 'P'); n++;  
        XtSetArg( wargs[n], XmNsubMenuId, pulldown); n++;
        XtSetArg( wargs[n], XmNmenuHistory, button[0]); n++;
        di_update_rate = (Widget)XmCreateOptionMenu( parent, "Detail_update",
								wargs,n);
 	widgets[i++] = di_update_rate;

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, di_update_rate); n++;
	/* XtSetArg( wargs[n], XmNtopOffset, 4); n++; */
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, di_update_rate); n++;
	XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	update_label = XtCreateWidget("Update:", xmLabelWidgetClass, 
						parent, wargs, n );
 	widgets[i++] = update_label;

	/* Create Node Update_rate option menu */

	pulldown = (Widget) XmCreatePulldownMenu( parent, "p2", NULL, 0);

        XtSetArg( wargs[0], XmNmnemonic, 'S'); 
	button[0] = XmCreatePushButtonGadget(pulldown, "Short Tic", wargs, 1);
    	XtAddCallback(button[0], XmNactivateCallback, do_node_rate, (XtPointer)0);

        XtSetArg( wargs[0], XmNmnemonic, 'L'); 
	button[1] = XmCreatePushButtonGadget(pulldown, "Long Tic", wargs, 1);
    	XtAddCallback(button[1], XmNactivateCallback, do_node_rate, (XtPointer)1);

	XtManageChildren( button, 2 );

        option_label = XmStringCreate( "Tree On", XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 60); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, attachto); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
        XtSetArg( wargs[n], XmNlabelString, option_label); n++; 
        XtSetArg( wargs[n], XmNoptionMnemonic, 'T'); n++;  
        XtSetArg( wargs[n], XmNsubMenuId, pulldown); n++;
        XtSetArg( wargs[n], XmNmenuHistory, button[1]); n++;
        ni_update_rate = (Widget)XmCreateOptionMenu( parent, "Node_update",
								wargs,n);
 	widgets[i++] = ni_update_rate;

	attachto = ni_update_rate;


	/* Create separator at bottom of popup */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, attachto); n++;
	XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, top_button); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 3); n++;
	sep3 = XtCreateWidget( "separator3", xmSeparatorGadgetClass, 
						parent, wargs, n );
 	widgets[i++] = sep3;

	pti_ptr->dialog_w 		= dw;
	pti_ptr->dialog_info_widgets 	= diw;
	pti_ptr->dialog_state 		= DIALOG_DISPLAYED;

	XtManageChildren( widgets, i );

	display_procinfo( parent_w, ptr );

	XtManageChild( dw ); 

	XtFree( (char *) widgets );
}


void
do_detail_rate( w, button_number, call_data )
Widget w;
int    button_number;
XmAnyCallbackStruct	*call_data;
{
	Widget   	    parent;
	int		    n, pid;
	Arg		    wargs[3];
	tnode		   *ptr;
	lnode		   *l_ptr;
	Process_tree_info  *pti_ptr;
	Process_list_info  *pli_ptr;

	parent = XtParent( w );   
	parent = XtParent( parent );   
	parent = XtParent( parent );   

	n = 0;
	XtSetArg( wargs[n], XmNuserData,  &pid); n++;
	XtGetValues( parent, wargs, n); n++;

	ptr = walk_tree( Head, match_key, pid );
	if ( ptr == NULL )
	{
		DEBUG1(1, "do_detail_rate: Can't find ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	pti_ptr = (Process_tree_info *) ptr->data;
	if ( pti_ptr == NULL )
	{
		DEBUG1(1, "do_detail_rate: NULL pti_ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
	{
		DEBUG1(1, "do_detail_rate: NULL l_ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
	{
		DEBUG1(1, "do_detail_rate: NULL pli_ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	if ( button_number == UPDATE_ON_REQUEST )
	{
		if ( pti_ptr->dialog_update_rate == UPDATE_ON_SHORT_TIC )
			active_unlock( l_ptr );
	}
	else if ( button_number == UPDATE_ON_SHORT_TIC )
	{
		/* If not active and need be add to active list and lock  */

		if ( pti_ptr->dialog_update_rate != UPDATE_ON_SHORT_TIC )
			active_lock( l_ptr );
	}
	else if ( button_number == UPDATE_ON_LONG_TIC )
	{
		if ( pti_ptr->dialog_update_rate == UPDATE_ON_SHORT_TIC )
			active_unlock( l_ptr );
	}

	pti_ptr->dialog_update_rate = button_number;

}

active_unlock( l_ptr )
lnode *l_ptr;
{
	Process_list_info  *pli_ptr;
	int		    rc;

	if ( l_ptr == NULL )
		return( False );

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
		return( False );

DEBUG1(1, "Request to unlock(%d)\n", l_ptr->key );

	if ( pli_ptr->active < 0 )	/* Currently locked */
	{

DEBUG1(1, "Current locks(%d)\n", pli_ptr->active * -1 );

		pli_ptr->active++;
		if ( pli_ptr->active == 0 )	/* All locks gone */
		{
DEBUG1(1, "Removed last lock so drop(%d) from active \n", l_ptr->key );
			pli_ptr->active = 0; 	
	    		Active_process_list = remove_lnode( Active_process_list,
							         l_ptr->key );
			rc = True;
		}
		else
			rc = False;
	}
	else
		rc = True;   /* Wasn't locked so it's now unlocked!! */

	return( rc );
}

active_lock( l_ptr )
lnode *l_ptr;
{
	Process_list_info  *pli_ptr;
	int		    rc;

	if ( l_ptr == NULL )
		return( False );

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
		return( False );

	rc = True;   
	if ( pli_ptr->active < 0 )	/* Currently locked */
	{
		pli_ptr->active--;	/* Add another lock */
	}
	else if ( pli_ptr->active > 0 )   /* Already active, so just lock it */
	{
		pli_ptr->active = -1;
	}
	else	/* Currently inactive, so add to list AND lock */
	{
	    	Active_process_list = rev_insert_lnode( Active_process_list, 
							   l_ptr->key, l_ptr);
		pli_ptr->active = -1;
	}

	return( rc );
}

void
do_node_rate( w, button_number, call_data )
Widget w;
int    button_number;
XmAnyCallbackStruct	*call_data;
{
	Widget   	    parent;
	int		    n, pid;
	Arg		    wargs[3];
	tnode		   *ptr;
	lnode		   *l_ptr;
	Process_tree_info  *pti_ptr;
	Process_list_info  *pli_ptr;
	String		    widget_name;

	parent = XtParent( w );   	/* menu pane */
	parent = XtParent( parent );   	/* Menu shell*/
	parent = XtParent( parent );   	/* popup form */

	n = 0;
	XtSetArg( wargs[n], XmNuserData,  &pid); n++;
	XtGetValues( parent, wargs, n); n++;

	ptr = walk_tree( Head, match_key, pid );
	if ( ptr == NULL )
	{
		DEBUG1(1, "do_node_rate: Can't find ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	pti_ptr = (Process_tree_info *) ptr->data;
	if ( pti_ptr == NULL )
	{
		DEBUG1(1, "do_node_rate: NULL pti_ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
	{
		DEBUG1(1, "do_node_rate: NULL l_ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
	{
		DEBUG1(1, "do_node_rate: NULL pli_ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	if ( button_number == 0 )	/* Short tic */
	{
		if ( pti_ptr->display_update_rate != UPDATE_ON_SHORT_TIC )
			active_lock( l_ptr );

		pti_ptr->display_update_rate = UPDATE_ON_SHORT_TIC;
	}
	else 	/* Long Tic */
	{
		if ( pti_ptr->display_update_rate == UPDATE_ON_SHORT_TIC )
			active_unlock( l_ptr );

		pti_ptr->display_update_rate = UPDATE_ON_LONG_TIC;
	}
}

void
toggle_procinfo( w, pid, call_data )
Widget w;
int    pid;
XmAnyCallbackStruct	*call_data;
{
	tnode			*ptr;
	Process_tree_info       *pti_ptr;
	Process_list_info       *pli_ptr=NULL;
	lnode			*l_ptr;
	sysProcInfo		*pp;
	int			 euid, egid;


	ptr = walk_tree( Head, match_key, pid );
	if ( ptr == NULL )
	{
		DEBUG1(1, "update_procinfo: Can't find ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	pti_ptr = (Process_tree_info *) ptr->data;
	pp      = (sysProcInfo *) pti_ptr->pp;

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr )
		pli_ptr = (Process_list_info *)l_ptr->data;

	/* Set toggle switch, update label */

	if ( pti_ptr->dialog_mode == ALPHA_MODE )
	{
		pti_ptr->dialog_mode = NUMERIC_MODE;
		xs_wprintf(w, "%s", "Text" );
	}
	else
	{
		pti_ptr->dialog_mode = ALPHA_MODE;
		xs_wprintf(w, "%s", "Numeric" );

	}

	display_procinfo( w, ptr );

#ifdef LINUX
	if ( pti_ptr->dialog_mode == ALPHA_MODE )
		displayWaitSymbol( pti_ptr );
#endif

}

#ifdef LINUX

#define MAX_SYMBOL_SIZE  32

displayWaitSymbol( pti_ptr )
Process_tree_info *pti_ptr;
{
	sysProcInfo		*pp;
	Node_info_widgets 	*diw;
	char			buf[MAX_SYMBOL_SIZE];

	pp = (sysProcInfo *) pti_ptr->pp;
	if ( pp == NULL )
		return;

	diw = pti_ptr->dialog_info_widgets;
	if ( diw == NULL )
	{
		DEBUG1( 5, "displayWaitSymbol: no dialog widgets (%d)\n",
						PROCESS_ID(pp) );
		return; 
	}

	buf[0] = NULL_CHAR;

	findWaitSymbol( AppData.systemMap, PROCESS_WCHAN(pp), 
						MAX_SYMBOL_SIZE-1, buf );

	xs_wprintf(diw->wait_addr_w, "%-11s", buf );
}
#endif

char
*user_name(uid)
int uid;
{
	return( find_name_by_uid( Users, uid ) );
}

char
*group_name(gid)
int gid;
{
	return( find_name_by_gid( Users, gid ) );
}

char
*proc_name(pid)
int pid;
{
	lnode 		  *p;
	Process_list_info *pli_ptr;
	sysProcInfo        *pp;

	for ( p = Process_list ; p != NULL ; p = p->next )
	{
		if ( p->key == pid )
		{
			pli_ptr = (Process_list_info *)p->data;
			if ( pli_ptr )
			{
				pp = (sysProcInfo *) pli_ptr->pp;
				if ( pp )
				{
					return( PROCESS_CMD_NAME( pp ) );
				}
			}
		}
	}

	return( "-" );  /* Process name is missing so return hyphen */

}

void
display_procinfo( parent, ptr )
Widget parent;
tnode  *ptr;
{
	sysProcInfo 		*pp;
	Process_tree_info    	*pti_ptr;
	Process_list_info    	*pli_ptr;
	lnode			*l_ptr;
	Node_info_widgets 	*diw;
	Widget  	 	 dw;
	Arg		 	 wargs[10];
	int		 	 n, len;
	time_t			 start_time;
	char			*time_str;
	int 			 hours, min, sec, hsec;
	major_t			 major_no;
	minor_t			 minor_no;
	char			*dev_name=NULL_STR;
	char			 buf[20];


	if ( ptr == NULL )
	{
		DEBUG0( 1, "display_procinfo: ptr == NULL\n" );
		return; 
	}

	pti_ptr = (Process_tree_info *)    ptr->data;
	if ( pti_ptr == NULL )
	{
		DEBUG0( 1, "display_procinfo: pti_ptr == NULL\n" );
		return; 
	}

	pp = (sysProcInfo *) pti_ptr->pp;
	if ( pp == NULL )
	{
		DEBUG0( 1, "display_procinfo: pp == NULL\n" );
		return; 
	}

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
	{
		DEBUG0( 1, "display_procinfo: No process list entry!!!\n" );
		return; 
	}

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
	{
		DEBUG0( 1, "display_procinfo: No process list data!!!\n" );
		return; 
	}

	if ( pti_ptr->dialog_state == NO_DIALOG )
	{
		DEBUG1( 5, "display_procinfo: state == no dialog  (%d)\n",
						PROCESS_ID(pp) );
		return; 
	}

	/* Stash widget pointer structure in a local var for quick access */

	diw = pti_ptr->dialog_info_widgets;
	if ( diw == NULL )
	{
		DEBUG1( 5, "display_procinfo: no dialog widgets (%d)\n",
						PROCESS_ID(pp) );
		return; 
	}

	
	xs_wprintf(diw->name_w, "%-16s", PROCESS_CMD_NAME(pp) );
	
	xs_wprintf(diw->pid_w,  "%6d", ptr->key );
	/* xs_wprintf(diw->args_w, "%-40s", PROCESS_CMD_ARGS(pp) ) */;
	xs_wprintf(diw->args_w, "%s", PROCESS_CMD_ARGS(pp) );
	

	    start_time = PROCESS_START_TIME(pp).tv_sec;
	    time_str   = ctime( &start_time );

	    if ( time_str )
	    {
	        len = strlen( time_str );
	        time_str[len-1] = NULL_CHAR;	/* Remove new line char */

	        xs_wprintf(diw->start_w, "%-24s", time_str );
	    }
	    else
	        xs_wprintf(diw->start_w, " " );
	        

	    hours = PROCESS_CPU_TIME(pp).tv_sec / 3600;

	    if ( hours )
		min  = (PROCESS_CPU_TIME(pp).tv_sec % 3600) / 60;
	    else
		min  = PROCESS_CPU_TIME(pp).tv_sec / 60;

	    sec  = PROCESS_CPU_TIME(pp).tv_sec % 60;
	    hsec = PROCESS_CPU_TIME(pp).tv_nsec / 10000000;

	    if ( hours > 0 )
	        xs_wprintf(diw->time_w, "%d:%0.2d:%0.2d.%0.2d",
	    					hours, min, sec, hsec);
	    else if ( min > 0 )
	        xs_wprintf(diw->time_w, "%0.2d:%0.2d.%0.2d", min, sec, hsec);
	    else
	        xs_wprintf(diw->time_w, "%0.2d.%0.2d", sec, hsec);


	/* xs_wprintf(diw->status_w,"%x %c",pp->pr_state,PROCESS_SNAME(pp) ); */

	switch( PROCESS_SNAME(pp) ) {

		case 'O': xs_wprintf(diw->status_w, "%-11s","Running" ); break;
		case 'A': xs_wprintf(diw->status_w, "%-11s","Just Ran" ); break;
		case 'S': xs_wprintf(diw->status_w, "%-11s","Sleeping" ); break;
		case 'D': xs_wprintf(diw->status_w, "%-11s","Wait/Swap"); break;
		case 'R': xs_wprintf(diw->status_w, "%-11s","Runable" ); break;
		case 'I': xs_wprintf(diw->status_w, "%-11s","Idle" ); break;
		case 'Z': xs_wprintf(diw->status_w, "%-11s","Zombie" ); break;
		case 'T': xs_wprintf(diw->status_w, "%-11s","Stopped" ); break;
		case 'X': xs_wprintf(diw->status_w, "%-11s","SXBRK" ); break;

		default : xs_wprintf(diw->status_w, "%-11s","Unknown" ); break;
			break;
	}


	if ( pti_ptr->dialog_mode == ALPHA_MODE )
	{
		xs_wprintf(diw->uid_w, "%-11s", user_name( PROCESS_USER_ID(pp) ));

		xs_wprintf(diw->gid_w, "%-11s", group_name( PROCESS_GROUP_ID(pp) ));

		xs_wprintf(diw->euid_w,"%-11s", 
			user_name( PROCESS_EFFECTIVE_USER_ID(pli_ptr) ));

		xs_wprintf(diw->egid_w,"%-11s", 
			group_name( PROCESS_EFFECTIVE_GROUP_ID(pli_ptr) ));

		xs_wprintf(diw->ppid_w,"%-11s", 
			proc_name( PROCESS_PARENT_ID(pp) ));

		xs_wprintf(diw->group_leader_pid_w,"%-11s", 
					proc_name(PROCESS_GROUP_LEADER(pp)));

		xs_wprintf(diw->session_id_w, "%-11s", 
			proc_name( PROCESS_SESSION_ID(pp) ));
	}
	else
	{
		xs_wprintf(diw->uid_w, "%-11d",  PROCESS_USER_ID(pp) );
		xs_wprintf(diw->gid_w, "%-11d",  PROCESS_GROUP_ID(pp) );

		xs_wprintf(diw->euid_w,"%-11d",  
			 	PROCESS_EFFECTIVE_USER_ID(pli_ptr) );

		xs_wprintf(diw->egid_w,"%-11d",  
				PROCESS_EFFECTIVE_GROUP_ID(pli_ptr) );

		xs_wprintf(diw->ppid_w,"%-11d",  PROCESS_PARENT_ID(pp) );

		xs_wprintf(diw->group_leader_pid_w, "%-11d", 
						PROCESS_GROUP_LEADER(pp) );

		xs_wprintf(diw->session_id_w, "%-11d", 
						PROCESS_SESSION_ID(pp) );
	}
	
	
	xs_wprintf(diw->nice_w, "%-11d",  PROCESS_NICE(pp) );

	sprintf( buf, "%lxX",  PROCESS_FLAG(pp) );
	xs_wprintf(diw->flags_w, "%-11s",  buf );

	xs_wprintf(diw->priority_w,     "%-11ld", PROCESS_PRIORITY(pp) );

#ifdef LINUX
#else

#ifdef SVR4_MP
	xs_wprintf(diw->old_priority_w, "%-11d",  PROCESS_NUM_LWP(pp) );
	xs_wprintf(diw->old_cpu_usage_w,"%-11d",  PROCESS_ON_PROCESSOR(pp) ); 
#else
	
	xs_wprintf(diw->old_priority_w, "%-11d",  PROCESS_OLD_PRIORITY(pp) );
	xs_wprintf(diw->old_cpu_usage_w,"%-11d",  PROCESS_CPU(pp) ); 
#endif

#endif

	xs_wprintf(diw->class_name_w,   "%-11s",  PROCESS_CLASS_NAME(pp) );


#ifdef LINUX
	if ( (PROCESS_CONTROL_TTY(pp) == -1) || (PROCESS_CONTROL_TTY(pp) == 0) )
#else
	if ( PROCESS_CONTROL_TTY(pp) == -1 ) 
#endif
	{
		xs_wprintf(diw->controlling_tty_w, "%-11s", "None" );
	}
	else
	{
		major_no = major( PROCESS_CONTROL_TTY(pp) );
		minor_no = minor( PROCESS_CONTROL_TTY(pp) );

		dev_name = find_dev_name( major_no, minor_no );
		if (  dev_name != NULL_STR )
			xs_wprintf(diw->controlling_tty_w, "%-11s", dev_name);

		else
			xs_wprintf(diw->controlling_tty_w, " %5ld:%-5ld",
							major_no, minor_no);
	}

#ifdef SOLARIS_7

	     sprintf( buf, "%d", PROCESS_SIZE(pp) );
	     strcat(  buf, "k" );
	     xs_wprintf( diw->image_size_w, "%-11s", buf );

	     sprintf( buf, "%d", PROCESS_RESIDENT_SET_SIZE(pp) );
	     strcat(  buf, "k" );
	     xs_wprintf(diw->memory_resident_w, "%-11s",  buf );

#else
	if ( pti_ptr->dialog_mode == ALPHA_MODE )
	{
	     sprintf( buf, "%d", (PROCESS_SIZE(pp) * PageSize)/1024);
	     strcat( buf, "k" );
	     xs_wprintf( diw->image_size_w, "%-11s", buf );

	     sprintf( buf, "%d", (PROCESS_RESIDENT_SET_SIZE(pp) *
	     					 PageSize)/1024 ) ;
	     strcat( buf, "k" );
	     xs_wprintf(diw->memory_resident_w, "%-11s",  buf );
	}
	else
	{
	     sprintf( buf, "%d", PROCESS_SIZE(pp) );
	     strcat(  buf, " pages" );
	     xs_wprintf( diw->image_size_w, "%-11s", buf );

	     sprintf( buf, "%d", PROCESS_RESIDENT_SET_SIZE(pp) );
	     strcat(  buf, " pages" );
	     xs_wprintf(diw->memory_resident_w, "%-11s",  buf );
	}
	
#endif
	sprintf( buf, "%xX",  PROCESS_WCHAN(pp) );
	xs_wprintf(diw->wait_addr_w, "%-11s", buf );
}

void
refresh_procinfo( parent, ptr )
Widget parent;
tnode  *ptr;
{
	sysProcInfo	 	*pp;
	Process_tree_info    	*pti_ptr;
	Process_list_info    	*pli_ptr;
	lnode			*l_ptr;
	Node_info_widgets 	*diw;
	Arg		 	 wargs[10];
	int		 	 n, len;
	int 			 hours, min, sec, hsec;
	char			 buf[20];


	if ( ptr == NULL )
	{
		DEBUG0( 1, "refresh_procinfo: ptr == NULL\n" );
		return; 
	}

	pti_ptr = (Process_tree_info *)    ptr->data;
	if ( pti_ptr == NULL )
	{
		DEBUG0( 1, "refresh_procinfo: pti_ptr == NULL\n" );
		return; 
	}

	pp = (sysProcInfo *) pti_ptr->pp;
	if ( pp == NULL )
	{
		DEBUG0( 1, "refresh_procinfo: pp == NULL\n" );
		return; 
	}

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
	{
		DEBUG0( 1, "refresh_procinfo: No process list entry!!!\n" );
		return; 
	}

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
	{
		DEBUG0( 1, "refresh_procinfo: No process list data!!!\n" );
		return; 
	}

	if ( pti_ptr->dialog_state == NO_DIALOG )
	{
		DEBUG1( 5, "refresh_procinfo: state == no dialog  (%d)\n",
						PROCESS_ID(pp) );
		return; 
	}

	/* Stash widget pointer structure in a local var for quick access */

	diw = pti_ptr->dialog_info_widgets;
	if ( diw == NULL )
	{
		DEBUG1( 5, "refresh_procinfo: no dialog widgets (%d)\n",
						PROCESS_ID(pp) );
		return; 
	}


	/* Only update those dynamic var's that changed */


	    hours = PROCESS_CPU_TIME(pp).tv_sec / 3600;
	    if ( hours )
		min  = (PROCESS_CPU_TIME(pp).tv_sec % 3600) / 60;
	    else
		min  = PROCESS_CPU_TIME(pp).tv_sec / 60;
    
	    sec  = PROCESS_CPU_TIME(pp).tv_sec % 60;
	    hsec = PROCESS_CPU_TIME(pp).tv_nsec / 10000000;

	    xs_wprintf(diw->time_w, "%0.2d:%0.2d:%0.2d.%0.2d", 
	    					hours, min, sec, hsec);


	switch( PROCESS_SNAME(pp) ) {

		case 'O': xs_wprintf(diw->status_w, "%-11s","Running" ); break;
		case 'A': xs_wprintf(diw->status_w, "%-11s","Just Ran" ); break;
		case 'S': xs_wprintf(diw->status_w, "%-11s","Sleeping" ); break;
		case 'D': xs_wprintf(diw->status_w, "%-11s","Wait/Swap"); break;
		case 'R': xs_wprintf(diw->status_w, "%-11s","Runable" ); break;
		case 'I': xs_wprintf(diw->status_w, "%-11s","Idle" ); break;
		case 'Z': xs_wprintf(diw->status_w, "%-11s","Zombie" ); break;
		case 'T': xs_wprintf(diw->status_w, "%-11s","Stopped" ); break;
		case 'X': xs_wprintf(diw->status_w, "%-11s",  "SXBRK" ); break;

		default : xs_wprintf(diw->status_w, "%-11s", "Unknown" ); break;
			break;
	}

	xs_wprintf(diw->nice_w, "%-3d",  PROCESS_NICE( pp ) );


	sprintf(buf, "%lxX  ", PROCESS_FLAG(pp) );
	xs_wprintf(diw->flags_w, "%-11s", buf );

	xs_wprintf(diw->priority_w,      "%-11ld", PROCESS_PRIORITY( pp ) );	

#ifdef LINUX
#else

	xs_wprintf(diw->old_cpu_usage_w, "%-11d",  PROCESS_CPU( pp ) ); 
	xs_wprintf(diw->old_priority_w,  "%-11d",  PROCESS_OLD_PRIORITY( pp ) );
#endif

#ifdef SOLARIS_7
	sprintf( buf, "%d", PROCESS_SIZE(pp) );
	strcat(  buf, "k" );
	xs_wprintf( diw->image_size_w, "%-11s", buf );
	sprintf( buf, "%d", PROCESS_RESIDENT_SET_SIZE(pp) ) ;
	strcat(  buf, "k" );
	xs_wprintf(diw->memory_resident_w, "%-11s",  buf );
#else
	if ( pti_ptr->dialog_mode == ALPHA_MODE )
	{
	     sprintf( buf, "%d",( PROCESS_SIZE(pp) * PageSize)/1024);
	     strcat(  buf, " pages" );
	     xs_wprintf( diw->image_size_w, "%-11s", buf );
	     sprintf( buf, "%d", (  PROCESS_RESIDENT_SET_SIZE(pp) *
	     						PageSize)/1024 ) ;
	     strcat(  buf, " pages" );
	     xs_wprintf(diw->memory_resident_w, "%-11s",  buf );
	}
	else
	{
	     sprintf( buf, "%ld", PROCESS_SIZE(pp) );
	     strcat(  buf, " pages" );
	     xs_wprintf( diw->image_size_w, "%-11s", buf );

	     sprintf( buf, "%ld",  PROCESS_RESIDENT_SET_SIZE(pp) );
	     strcat( buf, " pages" );
	     xs_wprintf(diw->memory_resident_w, "%-11s",  buf );
	}
	
#endif

	sprintf( buf, "%xX",  PROCESS_WCHAN( pp ) );
	xs_wprintf(diw->wait_addr_w, "%-11s", buf );
}

char
*find_dev_name( major_no, minor_no )
minor_t	 major_no;
minor_t	 minor_no;
{
	static  char 	name[30];
	char	  	buf[30];
	int		found_major, len, i, minor_digits;
	struct _ttydb 	*p;

	if ( major_no < 0 || major_no > 255 )
		return( NULL_STR );

	if ( minor_no < 0 || minor_no > 255 )
		return( NULL_STR );

	if ( Ttydb == NULL )
		return( NULL_STR );

	found_major = 0;
	for ( p = &(Ttydb[0]) ; p->major_no != -1 ; p++ )
	{
		if ( p->major_no == major_no )
		{
			if ( p->minor_no == minor_no )
				return( p->name );

			found_major = 1;
		}


		if ( p->major_no > major_no )	 
			break;
	}

	if ( found_major )	/* Wowa ttydb must be outa date */
	{
		/* So we atempt to craft one */

		for ( p = &(Ttydb[0]) ; p->major_no != -1 ; p++ )
		{
			if ( p->major_no == major_no )
				break;
		}

		strcpy( buf, p->name );	/* Use existing one as basis */

		/* Rubble old number and add new one based on minor no */

		len = strlen( buf );
		minor_digits = 0;	/* Count digits so they look similar */

		for ( i = len - 1 ; i >= 0 ; i-- )
		{
			if ( buf[i] >= '0' && buf[i] <= '9' )
			{
				buf[i] = NULL_CHAR;
				minor_digits++;
			}
		}

		if ( minor_digits == 0 )
			minor_digits = 3;

		sprintf( name, "%s%0*d", buf, minor_digits, minor_no );

		return( name );
	}

	return( NULL_STR );
}


void
update_procinfo( w, pid, call_data )
Widget w;
int    pid;
XmAnyCallbackStruct	*call_data;
{
	tnode  		   *ptr;
	lnode     	   *l_ptr;
	Process_list_info  *pli_ptr;
	Process_tree_info  *pti_ptr;

	ptr = walk_tree( Head, match_key, pid );
	if ( ptr == NULL )
	{
		DEBUG1(1, "update_procinfo: Can't find ptr for pid(%d)\n",pid);
		user_alert(w, "Process either non-visible or gone");
		return;
	}

	/* Need to refresh info, perhaps node is not on the active list */

	pti_ptr = (Process_tree_info *)ptr->data;
	if ( pti_ptr )
	{
		l_ptr = pti_ptr->l_ptr;
		if ( l_ptr )
		{
			pli_ptr = (Process_list_info *)l_ptr->data;

			if ( pli_ptr && pli_ptr->active == 0 )
				refresh_process_info( TreeWidget, l_ptr );
		}	
	}

	if ( pli_ptr->t_ptr == NULL )	/* ptr now invalid */
		user_alert(w, "Process changed state, change display criteria");
	else
		display_procinfo( w, ptr );
}


void
popdown_procinfo( w, pid, call_data )
Widget w;
int    pid;
XmAnyCallbackStruct	*call_data;
{
	tnode  		     *ptr;
	Process_tree_info    *pti_ptr;
	Widget		      pw;

	ptr = walk_tree( Head, match_key, pid );
	if ( ptr == NULL )
	{
		DEBUG1(1, "popdown_procinfo: Can't find ptr for pid(%d)\n",pid);

		/* Process either non-visible or gone, so just zap dialog */

		pw = XtParent( w );	/* Button's parent is form */
		pw = XtParent( pw );    /* Forms parent is dialog  */
		XtDestroyWidget( pw );
		return;
	}
	else    /* Found a node, Ah but does the popup belong to the node! */
	{
		pti_ptr = (Process_tree_info *) ptr->data;

		pw = XtParent( w );	/* Button's parent is form */

		if ( pw != pti_ptr->dialog_w )
		{
			/* Stale popup, pid now refers to a new tree node */
			pw = XtParent( pw );    /* Forms parent is dialog  */
			XtDestroyWidget( pw );
		}
		else if ( pti_ptr->dialog_w != NULL )
		{
			XtUnmanageChild( pti_ptr->dialog_w );
			pti_ptr->dialog_state = DIALOG_HIDDEN;
		}
	}
}




/* Convenience layout routine , attaches a label(l) centered vertically
 * on a text(t) widget, such that the label is  to the left of the text area.
 * Both widgets are then managed. Note this ensures the label is the same
 * height as the text widget. Assume text widget font >= label font.
 */

int
align_attach_manage( above, pos, l, t )
Widget above;
int    pos;
Widget l,t;
{
	int n;
	Arg 	wargs[10];

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, pos); n++;

	if ( above == NULL )
	{
		XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
		XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	}
	else
	{
		XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
		XtSetArg( wargs[n], XmNtopOffset, 2); n++;
		XtSetArg( wargs[n], XmNtopWidget, above); n++;
	}
	XtSetValues( t, wargs, n );

	XtManageChild( t );

	n = 0;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, t); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, pos); n++;
	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, t); n++;
	XtSetValues( l, wargs, n );

	XtManageChild( l );
}

/* As above but without managing the widgets			*/

int
align_attach_widgets( above, pos, l, t )
Widget above;
int    pos;
Widget l,t;
{
	int n;
	Arg 	wargs[10];

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, pos); n++;

	if ( above == NULL )
	{
		XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
		XtSetArg( wargs[n], XmNtopOffset, 2); n++;
	}
	else
	{
		XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
		XtSetArg( wargs[n], XmNtopOffset, 2); n++;
		XtSetArg( wargs[n], XmNtopWidget, above); n++;
	}
	XtSetValues( t, wargs, n );


	n = 0;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, t); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, pos); n++;
	XtSetArg( wargs[n], XmNrightOffset, 2); n++;
	XtSetArg( wargs[n], XmNleftOffset, 2); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, t); n++;
	XtSetValues( l, wargs, n );
}


Widget 
create_label(name, parent, func, field)
char *name;
Widget parent;
void (*func)();
int field;
{
	Widget label;
	Arg wargs[10];
	int n;

	n = 0;
	XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_END); n++;
	label = XtCreateWidget( name, xmLabelWidgetClass,
			parent, wargs, n);

	if ( field != -1 )
		XtAddEventHandler(label, ButtonPressMask, FALSE, func, 
						(XtPointer) field);

	return( label );
}


Widget 
create_button(name, parent, func)
char *name;
Widget parent;
void (*func)();
{
	Widget button;

	button = XtCreateWidget( name, xmPushButtonGadgetClass,
			parent, NULL, 0);

	XtAddCallback( button, XmNactivateCallback, func, (XtPointer)name);

	return( button );
}

Widget 
create_ltext(name, cols, parent, frame)
char *name;
int   cols;
Widget parent;
Widget *frame;
{
	Widget text;
	Widget border;
	Arg wargs[10];
	int n;

 	n = 0;
        XtSetArg( wargs[n], XmNshadowType, XmSHADOW_IN); n++;
        XtSetArg( wargs[n], XmNshadowThickness, 1); n++;
        border = XtCreateManagedWidget( "text_frame", xmFrameWidgetClass,
                                                parent, wargs, n );

	n = 0;
	XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	XtSetArg( wargs[n], XmNmarginHeight, 1); n++;
	XtSetArg( wargs[n], XmNmarginWidth, 3); n++;
	text = XtCreateManagedWidget( name, xmLabelWidgetClass, border, 
								wargs, n);

	*frame = border;

	return( text );
}

void
select_info( w, field, event )
Widget w;
int field;
XEvent *event;
{
	Widget  		 parent;
	tnode  			*t_ptr;
	int     		 n;
	Arg     		 wargs[2];
	int 			 pid;
	Process_tree_info  	*pti_ptr;


	/* Get the pid of the process from the widget parent userData pointer */

	parent = XtParent(w );
	if ( parent == NULL )
	{
		user_alert(w, "Can't find parent widget, oops");
		return;
	}

	n = 0;
	XtSetArg( wargs[n], XmNuserData,  &pid); n++;
	XtGetValues( parent, wargs, n);

	t_ptr = walk_tree( Head, match_key, pid );
	if ( t_ptr == NULL )
	{
		user_alert(w, "Process Either Gone or no Longer Displayed");
		return;
	}


	DEBUG3(1, "select_info: node(%s:%d) field ->%d\n", t_ptr->str_key, 
							t_ptr->key,
							field );
	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr == NULL )
	{
		user_alert(w, "Process Either Gone or no Longer Displayed[2]");
		return;
	}

	/* Check popup came from the node found  */
	if ( parent != pti_ptr->dialog_w )
	{
		user_alert(w, "Popup belongs to missing node, respecify");
		return;
	}

	/* If field not tagged for display, then tag it(Underline or Mark
	 * with + sign i.e.  CPU Time(+) Then add to process'es info list
	 * for display in the main process tree view area.
	 *
	 * Else if already tagged, then unmark it and remove it from the
	 *      list of fields displayed for the process.
	 */

	if ( match_list( pti_ptr->display_list, field ) )
	{
		DEBUG1(1, "select_info: Remove (%d) from display list\n",field);

		pti_ptr->display_list = delete_lnode( pti_ptr->display_list,
								field );
	}
	else
	{
		DEBUG1(1, "select_info: Add (%d) to display list\n",field);

		pti_ptr->display_list = fwd_insert_lnode( pti_ptr->display_list,
								field, NULL );
	}

	/* Invert fg/bg colors to show it's selected   */

	xs_invert_fgbg( w );

	display_proc_info( t_ptr, 0 );
}

/* ------------------- User/Group List management ------------------- */

void
set_user_list( user )
char *user;
{

	Group_view = VIEW_IGNORE_GROUPS;

	if ( strcmp( user, "login" ) == 0 )
	{
		User_view  = VIEW_LOGGED_IN_USER;
	}
	else if ( strcmp( user, "all" ) == 0 )
	{
		User_view  = VIEW_ALL_USERS;
	}
	else if ( strcmp( user, "root" ) == 0 )
	{
		User_view  = VIEW_ROOT_USER;
	}
	else if ( strcmp( user, "uucp" ) == 0 )
	{
		User_view  = VIEW_UUCP_USER;
	}
	else if ( strcmp( user, "User/Group Selection" ) == 0 )
	{
		User_view  = VIEW_GUTREE;
	}

	/* Set flag to Update tree to reflect new set of users being displayed*/

	New_display_criteria++;

	display_title();
}

void 
user_group_process_cb( users )
User_group_info  *users;
{
	/* Update Group/User View and set flag to change process view */

	/* If the view is not currently based on the user group tree then
	 * change to that view. Update button bar/menu_bar if needed 
	 */


	if ( users->num_grp_selected > 0 )
	{
        	Group_view = VIEW_GUTREE;
	}
	else
        	Group_view = VIEW_IGNORE_GROUPS;

	if ( User_view != VIEW_GUTREE )  /* User view change */
	{
		set_menu_toggle(Main_menu,"User/Group Selection",True);
		/* set_button( Main_Buttons, Main_Button_cnt, "User/Group" ); */
	}
	else
	{
        	User_view  = VIEW_GUTREE;
		display_title();
        	New_display_criteria++;
	}
}

char
*get_color_menu_str( color_by )
int color_by;
{
	char *str;

	switch( color_by ) 
	{
		case COLOR_BY_REAL_USER_ID:	  str = "   Real User Id   "; 
						  break;
		case COLOR_BY_EFFECTIVE_USER_ID:  str = " Effective User Id";
						  break;

		case COLOR_BY_REAL_GROUP_ID:      str = "  Real Group Id   "; 
						  break;
		case COLOR_BY_EFFECTIVE_GROUP_ID: str = "Effective Group Id";
						  break;

		case COLOR_BY_LOAD:	      str = " Current CPU Load "; break;
		case COLOR_BY_TOTAL_TIME:     str = "  Total CPU Time  "; break;
		case COLOR_BY_STATUS:	      str = "  Process Status  "; break;

		case COLOR_BY_RESIDENT_PAGES: str = "  Resident Memory ";break;
		case COLOR_BY_IMAGE_SIZE:     str = "    Image Size    ";break;
		case COLOR_BY_PRIORITY:	      str = "     Priority     ";break;

		default: str = "unknown"; break;
	}
	return( str );
}

char
*get_color_menu_str2( color_by )
int color_by;
{
	char *str;

	switch( color_by ) 
	{
		case COLOR_BY_REAL_USER_ID:	  str = "Real User Id"; 
						  break;
		case COLOR_BY_EFFECTIVE_USER_ID:  str = "Effective User Id";
						  break;

		case COLOR_BY_REAL_GROUP_ID:      str = "Real Group Id"; 
						  break;
		case COLOR_BY_EFFECTIVE_GROUP_ID: str = "Effective Group Id";
						  break;

		case COLOR_BY_LOAD:	      str = "Current CPU Load"; break;
		case COLOR_BY_TOTAL_TIME:     str = "Total CPU Time"; break;
		case COLOR_BY_STATUS:	      str = "Process Status"; break;

		case COLOR_BY_RESIDENT_PAGES: str = "Resident Memory";break;
		case COLOR_BY_IMAGE_SIZE:     str = "Image Size";break;
		case COLOR_BY_PRIORITY:	      str = "Priority";break;

		default: str = "unknown"; break;
	}
	return( str );
}

char
*get_color_str()
{
	char *str;

	switch( Color_based_on ) 
	{
		case COLOR_BY_REAL_USER_ID:	  str = "UID";        break;
		case COLOR_BY_EFFECTIVE_USER_ID:  str = "EUID";       break;
		case COLOR_BY_REAL_GROUP_ID:      str = "GID";        break;
		case COLOR_BY_EFFECTIVE_GROUP_ID: str = "EGID";       break;
		case COLOR_BY_LOAD:		  str = "CPU Load";   break;
		case COLOR_BY_TOTAL_TIME:	  str = "CPU Time";   break;
		case COLOR_BY_STATUS:		  str = "Status"  ;   break;
		case COLOR_BY_RESIDENT_PAGES:     str = "Resident";   break;
		case COLOR_BY_IMAGE_SIZE:	  str = "Image";      break;
		case COLOR_BY_PRIORITY:		  str = "Priority";   break;

		default: str = "unknown"; break;
	}
	return( str );
}

display_title( )
{
	Arg  wargs[2];
	int len, n;
	char title_str[256];
	char user_str[256];
	char group_str[256];
	char *color_str;

	group_str[0] = NULL_CHAR;
	user_str[0]  = NULL_CHAR;

	if ( Group_view == VIEW_LOGGED_IN_GROUP )
		strcpy( group_str, Users->user_gid_str );
	else if ( Group_view == VIEW_LOGGED_IN_GROUPS )
		strcpy( group_str, "Logged In" );
	else if ( Group_view == VIEW_BACKGROUND_GROUPS )
		strcpy( group_str, "Background" );
	else if ( Group_view == VIEW_SELECTED_GROUP_LIST )
		strcpy( group_str, "Selected" );
	else if ( Group_view == VIEW_IGNORE_GROUPS ) 
		strcpy( group_str, "" );
	else /* VIEW_ALL_GROUPS || VIEW_ACTIVE_GROUPS */
		strcpy( group_str, "All" );

	if ( User_view == VIEW_LOGGED_IN_USER )
		strcpy( user_str, Users->user_uid_str );
	else if ( User_view == VIEW_ROOT_USER )
		strcpy( user_str, "root" );
	else if ( User_view == VIEW_UUCP_USER )
		strcpy( user_str, "uucp" );
	else if ( User_view == VIEW_LOGGED_IN_USERS )
		strcpy( user_str, "Logged In" );
	else if ( User_view == VIEW_BACKGROUND_USERS )
		strcpy( user_str, "Background" );
	else if ( User_view == VIEW_SELECTED_USER_LIST  )
		strcpy( user_str, "Selected" );
	else if ( User_view == VIEW_GUTREE  )
		strcpy( user_str, "Group/User Tree Selection" );
	else if ( User_view == VIEW_ALL_USERS || 
				User_view == VIEW_ACTIVE_USERS  )
	{
		strcpy( user_str, "All" );
	}

	color_str = get_color_str();

	if ( (len = strlen(group_str)) > 0 )
	    sprintf(title_str, "Treeps: Group(%s) User(%s) Color(%s)",
							group_str, 
							user_str,
							color_str);
	else
	    sprintf(title_str, "Treeps: User(%s) Color(%s)",user_str,color_str);
	

	n = 0;
	XtSetArg( wargs[n], XmNtitle, title_str); n++;
	XtSetValues( Top_level, wargs, n );
}


