/* File: 	load_tree.c:  
 *
 * Description:	A generic tree load routine, assumes the following file
 *		format:
 *
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1995, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	2/19/95	Created
 *            
 */

#include <string.h>

#include "tree.h"
#include "debug.h"

tnode *splice_loose_subtrees();

int	_max_branch=0;

tnode
*load_tree( branch, filename )
tnode *branch;
char  *filename;
{
	FILE		*fp;
	int		key, parent_key, i, len, branch_num;
	char		line[132];
	char		*str, *str_key;
	tnode   	*ptr, *head;
	int		word=0;
	int		in_word;

	fp = fopen(filename, "r");
	if ( fp == NULL )
	{
		(void) fprintf( stderr, "fopen: (%s)\n", filename );
		perror( "Error: " );

		return( branch );
	}

	/* read the tree info file */

	if ( branch == NULL_TNODE_PTR )
		head = construct_tnode( -1, "Top", NULL );
	else
		head = branch;

	while ( (str = fgets( line, 132, fp )) != NULL )
	{
		if (line[0] == '#' )	/* Skip comments  */
			continue;

		(void) sscanf( line , "%d%d", &key, &parent_key );

		len = strlen( line );
		if ( len > 0 )
			line[len-1] = NULL_CHAR;

		branch_num = key/1000;
		if ( branch_num > _max_branch )
			_max_branch = branch_num;

		in_word = 0; word = 0; str_key = NULL;
		for ( i = 0 ; line[i] != NULL_CHAR; i++ )
		{
			if ( line[i] == ' ' || line[i] == '\t' )
			{
				if ( in_word )
				{
					word++; in_word = 0;
				}
			}
			else
				in_word = 1;

			if ( word >= 2 && ( line[i] != ' ' && line[i] != '\t' ))
			{
				str_key = strdup( &(line[i]) );
				break;
			}
		}

		DEBUG3(3, "load_tree: Adding(%d) parent(%d) line(%s)\n",
					key, parent_key, str);
	
		/* Need to read an info block (type, size, data) 
		 * set type_ptr
		 */
	
		ptr = make_tnode( key, str_key, NULL );
	
		ptr->parent_key = parent_key;
	
		/* Add the new node at the appropriate place in the tree */
	
		head = insert_tnode( head, ptr, parent_key );

		ptr->parent_key = parent_key; /* Set to desired parent */
	
		(void) splice_loose_subtrees( head, ptr );
	}

	(void) fclose( fp );

	/* If we had no root node and we added only one child, then we
	 * assume it's the top of the tree. Otherwise we return either
	 * the node passed or the top node we created to hold all of the
	 * branches.
	 */

	if ( branch == NULL_TNODE_PTR )
	{
		if ( head->num_children == 1 )	/* Child is head */
		{
			ptr = head;

			head = head->children->child;
			head->parent = NULL_TNODE_PTR;

			destroy_tnode( ptr );	/* Remove tmp head */
		}
	}

	return( head );
}

tnode
*splice_loose_subtrees( head, new_node )
tnode *head;
tnode *new_node;
{
	child_ptr	*cptr;
	child_ptr	*next_cptr;

	/* check to see if any of the subtree's dangling off of the
	 * root should be attached to the new node. If so we disconnect 
	 * them from the root and attach them to the new_node.
	 */

	for ( cptr = head->children ; cptr != NULL ; cptr = next_cptr )
	{
		next_cptr = cptr->next;
		if ( cptr->child->parent_key == new_node->key )
		{
			if ( cptr->child->parent_key == cptr->child->key )
			{
				/* Node's attached to itself probably root */
				continue;
			}

			DEBUG2(9, "splice_loose_subtrees: attach(%d) to (%d)\n",
				cptr->child->key, new_node->key );

			(void) move_tnode( cptr->child, new_node );
		}
	}

	return( new_node );
}

