/* File:        cmd_tip.c
 *
 * Description: Code to display a "cmd tip" for a specified widget
 *              A command tip is a window which is pop'ed when the mouse 
 *		moves into the the specified widget. It's
 *              similar to a tooltip/widget_tip/ballon_help ...
 *		Instead of help it just displays the result of a command
 *		that is executed.
 *
 * Author:      George MacDonald
 *
 * Copyright:   GPL - see http://www.gnu.org
 *
 * History:     George MacDonald        19/Nov/2001        Created
 *
 *
 */


#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <time.h>

#include <stdio.h>

#include "../src/debug.h"

void 		cmd_tip();

static void 	cmd_tip_enter();
static void 	cmd_tip_leave();
static void 	cmd_tip_press();
static void 	cmd_tip_startup_timeout();
static void 	cmd_tip_uptime_timeout();

static void 	cmd_tip_motion();

static Window 	create_cmd_tip_window();
void 		destroy_cmd_tip();

void 		cmd_tip_cancel_timer();

void 		cmd_tip_set_bg_color();
void 		cmd_tip_set_fg_color();
void 		cmd_tip_set_font();


Window 		Cmd_tip_window=0;

XtIntervalId 	Cmd_tip_interval_id=-1;

char 		*Cmd_tip_str=NULL;

time_t 		Last_cmd_tip_time=0;

Position 	Cmd_tip_x, Cmd_tip_y;

Widget   	Cmd_tip_w=NULL;

int		Default_cmd_tip_uptime=15000;
int		Default_cmd_tip_startup_delay=400;
char 		*Default_cmd_tip_fontname="9x15";
Pixel		Default_cmd_tip_bg=-1;
Pixel		Default_cmd_tip_fg=-1;

int		Cmd_tip_uptime;
int		Cmd_tip_startup_delay;
char 		*Cmd_tip_fontname;
Pixel		Cmd_tip_bg=-1;
Pixel		Cmd_tip_fg=-1;

extern int debug;



void
xs_cmd_tip( w, tipStr )
Widget w;
char *tipStr;
{
	static int first=1;

	if ( first )
	{
		cmd_tip_set_bg_color( w, "green" );
		cmd_tip_set_fg_color( w, "black" );

		Default_cmd_tip_bg = Cmd_tip_bg;
		Default_cmd_tip_fg = Cmd_tip_fg;

		Cmd_tip_uptime	 = Default_cmd_tip_uptime;
		Cmd_tip_startup_delay = Default_cmd_tip_startup_delay;
		Cmd_tip_fontname      = Default_cmd_tip_fontname;
		Cmd_tip_bg		 = Default_cmd_tip_bg;
		Cmd_tip_fg		 = Default_cmd_tip_fg;

		first = 0;
	}

	XtAddEventHandler(w, EnterWindowMask, FALSE, cmd_tip_enter, tipStr);

	XtAddEventHandler(w, LeaveWindowMask, FALSE, cmd_tip_leave, NULL);
	XtAddEventHandler(w, ButtonPressMask, FALSE, cmd_tip_press, NULL);

	/*XtAddEventHandler(w,PointerMotionMask,FALSE,cmd_tip_motion,NULL);*/
}

void
cmd_tip_enter( w, tipStr, event, continueDispatch)
Widget w;
char *tipStr;
XCrossingEvent *event;
Boolean *continueDispatch;
{
	Arg 		wargs[20];
	int 		n;
	time_t		time_now;
	Widget 		parent;
	XtAppContext 	ctx;
	Display 	*display;
	Position	widget_x, widget_y;
	Position	root_x, root_y;

	DEBUG1( 8, "cmd_tip_enter: tipStr(%s)\n", tipStr );


	display = XtDisplay( w );
	ctx     = XtWidgetToApplicationContext( w );

	*continueDispatch = TRUE;

	Cmd_tip_w = w;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &widget_x ); n++;
	XtSetArg( wargs[n], XtNheight, &widget_y ); n++;
	XtGetValues( w, wargs, n );

	XtTranslateCoords( w, widget_x, widget_y, &root_x, &root_y );

	Cmd_tip_x = root_x - ( widget_x / 3 );
	if ( root_y - 40 > 0 )
	    Cmd_tip_y = root_y - 40;
	else
	    Cmd_tip_y = root_y + 40;

	DEBUG4( 8, "cmd_tip_enter: x(%d) y(%d) root:x(%d)y(%d)\n", 
				widget_x, widget_y, root_x, root_y);

	DEBUG1( 8, "event->detail(%d)\n", event->detail );

	time( &time_now );
	if ( tipStr == NULL )
	{
		DEBUG0(8,"No tip string!!!\n");
		return ;
	}
	else	/* Create a cmd tip? */
	{

		if ( event->detail == NotifyAncestor ) /* 0 - Normal Enter */
		{
			DEBUG0(8,"Entering widget: Notify Ancestor\n" );
		}
		if ( event->detail == NotifyVirtual ) 
		{
			DEBUG0(8,"Entering widget: Notify Virtual\n" );
		}
		if ( event->detail == NotifyInferior ) 
		{
			DEBUG0(8,"Entering widget: Notify Inferior\n" );
		}
		if ( event->detail == NotifyNonlinear ) /* 3 */
	        {
			DEBUG0(8,"Entering widget: Notify NonLinear\n" );
		}

		if ( event->detail == NotifyNonlinearVirtual ) 
		{
			DEBUG0(8,"Entering widget: Notify NonLinearVirtual\n" );
		}


		if ( Cmd_tip_window != 0 ) /* Already got one up!? */
		{

			if ( strcmp( Cmd_tip_str, tipStr) != 0 )
			{
				DEBUG0(8,"Entering widget: Teardown old tip!\n");
			        destroy_cmd_tip(w);
			}
			else
			{
				DEBUG0(8,"Entering widget: Tip already up!!\n");
				return;
			}
		}

		/* We are go for cmd tip */


		/* If a cmd tip was recently active, then user is probably
		 * looking for a tip so put one up right away.  Otherwise wait 
		 * till the mouse is inactive for a small amount of time 
		 * while in the widget before poping up tip.
		 */

		if ( Cmd_tip_str == NULL )
		{
			Cmd_tip_str = strdup(tipStr);
		}

		if ( Last_cmd_tip_time + 0 > time_now )
		{
		    DEBUG0(8,"Popup cmd tip: Immediately \n");
		    Cmd_tip_window = create_cmd_tip_window( w, tipStr,
		    				Cmd_tip_x, Cmd_tip_y);

		    Cmd_tip_interval_id = XtAppAddTimeOut( ctx,
		    				Cmd_tip_uptime,
		    				cmd_tip_uptime_timeout, 
						w 			   );
		}
		else
		{
		    DEBUG0(8,"Popup cmd tip: firing startup timer \n");

		    Cmd_tip_interval_id = XtAppAddTimeOut( ctx,
		    				Cmd_tip_startup_delay,
		    				cmd_tip_startup_timeout, w );
		}

		Last_cmd_tip_time = time_now;
	}
	
	*continueDispatch = TRUE;

	return;
}

void
cmd_tip_press( w, tipStr, event, continueDispatch)
Widget w;
char *tipStr;
XButtonPressedEvent *event;
Boolean *continueDispatch;
{
	destroy_cmd_tip(w);

	*continueDispatch = TRUE;
}


void
cmd_tip_leave( w, tipStr, event, continueDispatch)
Widget w;
char *tipStr;
XCrossingEvent *event;
Boolean *continueDispatch;
{
	Arg 		wargs[20];
	int 		n;
	time_t		time_now;
	Widget 		parent;
	XtAppContext 	ctx;
	Display 	*display;

	display = XtDisplay( w );
	ctx     = XtWidgetToApplicationContext( w );

	*continueDispatch = TRUE;

	Cmd_tip_w = w;
	Cmd_tip_x = event->x_root; 
	Cmd_tip_y = event->y_root;  


	DEBUG3( 8, "cmd_tip_leave: tipStr(%s) event->x(%d) event->y(%d)\n", 
				Cmd_tip_str, event->x_root, event->y_root);

	DEBUG1( 8, "event->detail(%d)\n", event->detail );


	if ( event->detail == NotifyAncestor ) /* 0 - Normal Leave */
	{
		DEBUG0(8,"Leaving widget: Notify Ancestor\n" );
	}
	if ( event->detail == NotifyVirtual ) 
	{
		DEBUG0(8,"Leaving widget: Notify Virtual\n" );
	}
	if ( event->detail == NotifyInferior ) 
	{
		DEBUG0(8,"Leaving widget: Notify Inferior\n" );
	}
	if ( event->detail == NotifyNonlinear ) /* 3 - Into tip */
	{
		DEBUG0(8,"Leaving widget: Notify NonLinear\n" );
	}
	if ( event->detail == NotifyNonlinearVirtual ) 
	{
		DEBUG0(8,"Leaving widget: Notify NonLinearVirtual\n" );
	}

	time( &time_now );
	if ( tipStr == NULL )
	{
		if ( event->detail == NotifyNonlinear )
	        {
			Last_cmd_tip_time = time_now;
			destroy_cmd_tip(w);
		}
		else
		{
			DEBUG0(8,"Leaving widget: on widgets border\n");
			destroy_cmd_tip(w);
		}

		Cmd_tip_w = NULL;
	}
	else	/* Eh? */
	{
		DEBUG0(8,"Leave widget(%d) with non null string!\n" );
		return;
	}
	
	*continueDispatch = TRUE;

	return;
}

void
cmd_tip_motion( w, tipStr, event, continue_dispatch)
Widget w;
char *tipStr;
XMotionEvent *event;
Boolean *continue_dispatch;
{
	unsigned long 		timeout;
	XtTimerCallbackProc 	timeout_func;
	XtAppContext 		ctx;

	ctx = XtWidgetToApplicationContext( w );

	DEBUG2(8,"Cmd_tip_motion x(%d)/y(%d)\n",event->x_root,event->y_root);

	/* Cancel old timer */

	if ( Cmd_tip_interval_id != -1 )
	{
		XtRemoveTimeOut( Cmd_tip_interval_id );
		Cmd_tip_interval_id = -1;
	}


	/* Reset timer */

	if ( Cmd_tip_window == 0 ) 	/* Waiting to popup cmd tip */
	{
		DEBUG0(8,"Reset cmd tip startup timer \n");
		timeout      = Cmd_tip_startup_delay;
		timeout_func = cmd_tip_startup_timeout;
	}
	else
	{
		DEBUG0(8,"Reset cmd tip uptime timer \n");
		timeout      = Cmd_tip_uptime;
		timeout_func = cmd_tip_uptime_timeout;
	}

	Cmd_tip_interval_id = XtAppAddTimeOut( ctx, timeout, 
						timeout_func, w );

	*continue_dispatch = TRUE;
}

void
cmd_tip_startup_timeout( w, id )  
Widget       w;
XtIntervalId id;
{
	XtAppContext 	ctx;

	ctx = XtWidgetToApplicationContext( w );


	if ( Cmd_tip_window == 0 )
	{
		DEBUG1(8,"cmd_tip_startup_timeout: creating cmd tip(%s)\n",
						Cmd_tip_str );

		if ( Cmd_tip_w == NULL || Cmd_tip_str == NULL )
		{
			DEBUG0(8,"cmd_tip_timeout: Too Late\n");
			return;
		}

		Cmd_tip_window = create_cmd_tip_window(
					Cmd_tip_w,
					Cmd_tip_str, 
					Cmd_tip_x, 
					Cmd_tip_y		);

		Cmd_tip_interval_id = XtAppAddTimeOut( ctx,
		    				Cmd_tip_uptime,
		    				cmd_tip_uptime_timeout, 
						w 			   );
	}
	else
	{
		DEBUG0(8,"cmd_tip_startup_timeout: Already up!!!\n");
	}
}

void
cmd_tip_uptime_timeout( w, id )  
Widget       w;
XtIntervalId id;
{

	if ( Cmd_tip_window == 0 )
	{
		DEBUG0(8, "cmd_tip_uptime_timeout: no cmd tip!!\n");
	}
	else 
	{
		Cmd_tip_interval_id = -1;
		destroy_cmd_tip( w );
	}

}

void
destroy_cmd_tip( w )
Widget w;
{
	time_t	time_now;

	time( &time_now );

	if ( Cmd_tip_window != 0 )
	{
		DEBUG1(8,"destroying_cmd_tip(%s)\n", Cmd_tip_str );
		XDestroyWindow( XtDisplay( w ), Cmd_tip_window );
		Cmd_tip_window = 0;
	}

	if ( Cmd_tip_str != NULL )
	{
		free( Cmd_tip_str );
		Cmd_tip_str = NULL;
	}

	Last_cmd_tip_time = time_now;

	cmd_tip_cancel_timer();
}

void
cmd_tip_cancel_timer()
{

	if ( Cmd_tip_interval_id != -1 )
	{
		DEBUG0(8,"Canceling cmd tip Timer\n");
		XtRemoveTimeOut( Cmd_tip_interval_id );
		Cmd_tip_interval_id = -1;
	}
}

Window
create_cmd_tip_window( w, cmd_str, x, y )
Widget w;
char *cmd_str;
int x;
int y;
{
	static int first=1;
	static Display *display=NULL;
	static int screen;
	static Screen *screen_ptr=NULL;
        static char *window_name = "Cmd_tip";
        static XFontStruct *font_info;
        static int depth;
        static Visual *visual;
        static unsigned int class;
        static unsigned long valuemask;
        static unsigned long GCvaluemask=0;
        static XSetWindowAttributes setwinattr;
        static XGCValues values;
	static int border_width;
        int 		window_size = 0;
        XSizeHints 	size_hints;
        XEvent 		report;
        int  		str_x, str_y;
	unsigned int 	width, height;
	int 		len, nb;
        GC 		gc;		/* Need to free this */
	Window		win;
	int             win_x, win_y;
	static char	result_str[256];
	char		*str;
	FILE		*fp;

	if ( first )
	{
	    border_width = 1;

	    display = XtDisplay( w );
	    screen  = DefaultScreen( display );
  
	    screen_ptr = ScreenOfDisplay( display, screen );

            if ((font_info = XLoadQueryFont(display, Cmd_tip_fontname))==NULL)
            {
                fprintf( stderr, "Can't open %s font!!!!\n",Cmd_tip_fontname);
                return( 0 );
            }

	    depth = CopyFromParent;
            class = CopyFromParent;
            visual = CopyFromParent;

            valuemask = CWBackPixel | CWBorderPixel | CWOverrideRedirect ;

	    if ( Cmd_tip_bg == -1 )
	    	cmd_tip_set_bg_color( w, "white" );

	    if ( Cmd_tip_fg == -1 )
	    	cmd_tip_set_fg_color( w, "black" );

            setwinattr.background_pixel = Cmd_tip_bg;

            setwinattr.border_pixel = BlackPixel( display, screen );
            setwinattr.override_redirect = True;

	    first = 0;
	}

	fp = popen( cmd_str, "r" );
	if ( fp == NULL )
	{
	    sprintf(result_str, "command(%s): failed to execute", cmd_str );
	}
	else
	{
	    nb = fread( result_str, 1, 255, fp );
	    if ( nb == 0 )
	        sprintf(result_str, "command(%s): failed to execute", cmd_str );
	    else
	    {
	        result_str[nb-1] = '\0';
	    }
	    pclose( fp );
	}



	str = &(result_str[0]);

        len = strlen( str );
	width = XTextWidth( font_info, str, len ) + 6;
        height = font_info->max_bounds.ascent + 
					font_info->max_bounds.descent +4;

        str_x = 3;
        str_y = font_info->max_bounds.ascent + 2;


	win_x = x;

        if ( (win_x + width) > WidthOfScreen( screen_ptr ) )
        {
                win_x = ((int)WidthOfScreen( screen_ptr )) - width;
                if ( win_x < 0 )
                    win_x = 0;
        }

        win_y = y;
        if ( win_y + (height*2) > HeightOfScreen( screen_ptr ) )
        {
                win_y = ((int)HeightOfScreen( screen_ptr )) - (4*height);
                if ( win_y < 0 )
                    win_y = 0;
        }

        win = XCreateWindow( display, RootWindow(display, screen ),
                        win_x, win_y,
                        width, height,
                        border_width,
                        depth,
                        class,
                        visual,
                        valuemask,
                        &setwinattr );

        gc = XCreateGC( display, win, GCvaluemask, &values );

        XSetFont( display, gc, font_info->fid );


        /* XSetForeground( display, gc, BlackPixel(display, screen) ); */

        XSetForeground( display, gc, Cmd_tip_fg );

	size_hints.flags = PPosition | PSize | PMinSize;
        size_hints.x = win_x;
        size_hints.y = win_y;
        size_hints.width = width;
        size_hints.height = height;
        size_hints.min_width = width;
        size_hints.min_height = height;

        XSetStandardProperties( display, win, window_name, NULL,
        				   0, NULL, 0, &size_hints );

        XMapWindow( display, win );

        XDrawString( display, win, gc, str_x, str_y, str, len );

	XFlush( display );

	XFreeGC( display, gc );

        return( win );
}

/* These must be called before first creation of a cmd tip, otherwise
 * the default values will be used
 */

void
cmd_tip_set_bg_color( w, colorName )
Widget w;
char *colorName;
{
	Display *dpy;
        int      scr;
        Colormap cmap;
        XColor   color;
        XColor   exact;
        int      rc;

        dpy  = XtDisplay( w );
        scr  = DefaultScreen( dpy );
        cmap = DefaultColormap( dpy, scr );

        /* Lookup Pixel colors based on character string symbolic names */

        rc = XAllocNamedColor( dpy, cmap, colorName, &color, &exact );
        if ( rc )
                Cmd_tip_bg = color.pixel;
        else
        {
                fprintf(stderr, 
			"Warning: Couldn't allocate cmd tip bg color %s\n",
                                                                colorName);
                Cmd_tip_bg = WhitePixel(dpy, scr);
        }
}


void
cmd_tip_set_fg_color( w, colorName )
Widget w;
char *colorName;
{
	Display *dpy;
        int      scr;
        Colormap cmap;
        XColor   color;
        XColor   exact;
        int      rc;

        dpy  = XtDisplay( w );
        scr  = DefaultScreen( dpy );
        cmap = DefaultColormap( dpy, scr );

        /* Lookup Pixel colors based on character string symbolic names */

        rc = XAllocNamedColor( dpy, cmap, colorName, &color, &exact );
        if ( rc )
                Cmd_tip_fg = color.pixel;
        else
        {
                fprintf(stderr, 
			"Warning: Couldn't allocate cmd tip fg color %s\n",
                                                                colorName);
                Cmd_tip_fg = BlackPixel(dpy, scr);
        }
}

void
cmd_tip_set_font( fontName )
char *fontName;
{
	/* Small memory leak here */

	Default_cmd_tip_fontname = strdup( fontName );

	Cmd_tip_fontname = Default_cmd_tip_fontname;
}
