/* File: 	Tree.c
 *
 * Description: Methods file for the tree constraint widget. Includes all
 *		space allocation for the tree widget, requires Tree.h and
 *		TreeP.h
 *
 *		The tree widget will layout a number of supplied widgets
 *		using the current orientation, packing, node shape, 
 *		connection style, padding, etc.
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1993, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	10/01/93	Created
 *
 */

#ifndef lint
static char *treew_cpr ="@(#) Copyright(c) 1993, Pulsar Software Inc.\n\t All rights reserved.";
static char *treew_sccs="@(#) %W% 	Last Delta %G% %U%";
static char *treew_rcs="$id$ - $name$";
#endif

/* The following is used for X memory related type casting, just in
 * case somebody decides to make it a (void *) one day
 */

typedef char * Xmem;	/* Used for typecasting args to XtFree ... */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Core.h>
#include <X11/CompositeP.h>
#include <X11/ConstrainP.h>

#include "TreeP.h"
#include "Tree.h"

#include "debug.h"

#define RADIANS(x)      ((M_PI * 2.0 * (x)) / 360.0)
#define DEGREES(x)      ((x) / (M_PI * 2.0) * 360.0)

#define MAX(a,b)	((((int)a) > ((int)b)) ? (a) :  (b))


static void 		Class_Initialize();
static void 		psi_cvt_str_to_treeOrientation();
static void 		psi_cvt_str_to_treePacking();
static void 		psi_cvt_str_to_connectionStyle();
static void 		psi_cvt_str_to_nodeSize();
static void 		psi_cvt_str_to_parentPlacement();

static void 		Initialize();
static void 		ConstraintInitialize();
static void 		Destroy();
static void 		ConstraintDestroy();
static Boolean 		ConstraintSetValues();
static void 		Resize();
static Boolean 		SetValues();
static XtGeometryResult GeometryManager();
static void 		ChangeManaged();
       void 		XpsiDoTreeLayout();
       void             XpsiLayoutUnmanaged();
       void             XpsiDoTreeRedisplay();
static void 		insert_node();
static void 		delete_node();
static void 		Redisplay();
static void 		drawTreeNode();

static void 		new_layout();
static int		compute_LtoR_positions();
static int		compute_TtoB_positions();
static int		compute_Star_positions();
static void 		compute_node_diameters();
static void 		compute_node_extents();
static void 		compute_node_shapes();
static void 		calculate_tree_rings();
static void 		compute_tree_rings();
static void 		compute_connections();
static void		diagonal_connect();
static void		tier_connect();
static void		draw_tier_bar();
static Position		tier_bar_pos();
static void 		compute_relative_LRpositions();
static void 		compute_relative_TBpositions();
static void		pathway_connect();
static void 		shift_subtree();
static void 		set_positions();
static void 		center_star_tree();
static TreeTablePtr	create_table();
static void		destroy_table();
static void 		reset_table();
static Position 	current_value();
static void 		set_current_value();
static Position 	sum_table();
static void 		adjust_connection_segments();
static void 		adjust_arcs();
static void 		adjust_rings();
static void 		adjust_rectangles();

static XtResource resources[] = {
	{ XtNtreeOrientation, XtCTreeOrientation, XtRTreeOrientation, 
		sizeof( TreeOrientation ),
	        XtOffset(XpsiTreeWidget,tree.orientation),
		XtRString, "LEFT_TO_RIGHT" },
	{ XtNtreePacking, XtCTreePacking, XtRTreePacking, 
		sizeof( TreePacking ),
	        XtOffset(XpsiTreeWidget,tree.packing),
		XtRString, "ISOMORPHIC_PACKING" },
	{ XtNconnectionStyle, XtCConnectionStyle, XtRConnectionStyle, 
		sizeof( ConnectionStyle ),
	        XtOffset(XpsiTreeWidget,tree.connect_style),
		XtRString, "TIER_CONNECTION" },
	{ XtNnodeShape, XtCNodeShape, XtRBoolean, sizeof( Boolean ),
	        XtOffset(XpsiTreeWidget,tree.node_shape),XtRString,"TRUE" },
	{ XtNnodeSize, XtCNodeSize, XtRNodeSize, sizeof( NodeSize ),
	        XtOffset(XpsiTreeWidget,tree.node_size),
		XtRString,"VARIABLE_SIZE" },
	{ XtNresizePolicy, XtCResizePolicy, XtRBoolean, sizeof( Boolean ),
	        XtOffset(XpsiTreeWidget,tree.resize_policy),XtRString,"FALSE" },
	{ XtNhorizontalSpace, XtCSpace, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.min_x_pad), XtRString, "15" },
	{ XtNverticalSpace, XtCSpace, XtRDimension, sizeof( Dimension ),
	         XtOffset( XpsiTreeWidget, tree.min_y_pad), XtRString, "5" },
	{ XtNconnectionSpace, XtCConnectionSpace, XtRDimension, 
	        sizeof( Dimension ),
	        XtOffset( XpsiTreeWidget, tree.path_pad), 
		XtRString, "2" },
	{ XtNforeground, XtCForeground, XtRPixel, sizeof( Pixel),
	        XtOffset(XpsiTreeWidget, tree.foreground), XtRString, "Black" },
	{ XtNnodeForeground, XtCForeground, XtRPixel, sizeof( Pixel),
	        XtOffset(XpsiTreeWidget, tree.node_foreground), 
		XtRString, "Black" },
	{ XtNnodeBackground, XtCBackground, XtRPixel, sizeof( Pixel),
	        XtOffset(XpsiTreeWidget, tree.node_background), 
		XtRString, "White" },
	{ XtNleftMargin, XtCTreeMargin, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.left_margin),   XtRString, "4" },
	{ XtNrightMargin, XtCTreeMargin, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.right_margin),  XtRString, "4" },
	{ XtNtopMargin, XtCTreeMargin, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.top_margin),    XtRString, "4" },
	{ XtNbottomMargin, XtCTreeMargin, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.bottom_margin), XtRString, "4" },
	{ XtNminWidth, XtCMinWidth, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.minWidth),  XtRDimension, 0 },
	{ XtNminHeight, XtCMinHeight, XtRDimension, sizeof(Dimension),
	         XtOffset(XpsiTreeWidget, tree.minHeight),  XtRDimension, 0 },
	{ XtNparentPlacement, XtCParentPlacement, XtRParentPlacement,
		 sizeof(ParentPlacement),
	         XtOffset(XpsiTreeWidget, tree.parents_place), 
		 XtRString, "PARENT_CENTERED" },
	{ XtNshowRings, XtCShowRings, XtRBoolean, sizeof( Boolean ),
	        XtOffset(XpsiTreeWidget,tree.show_rings),XtRString,"True" },
	{ XtNringBackground, XtCBackground, XtRPixel, sizeof( Pixel),
	        XtOffset(XpsiTreeWidget, tree.ring_background), 
		XtRString, "Black" },
	{ XtNringForeground, XtCForeground, XtRPixel, sizeof( Pixel),
	        XtOffset(XpsiTreeWidget, tree.ring_foreground), 
		XtRString, "White" },
	{ XtNcenterRootX, XtCCenterRootX, XtRBoolean, sizeof( Boolean ),
	        XtOffset(XpsiTreeWidget,tree.center_root_x),XtRString,"TRUE" },
	{ XtNcenterRootY, XtCCenterRootY, XtRBoolean, sizeof( Boolean ),
	        XtOffset(XpsiTreeWidget,tree.center_root_y),XtRString,"FALSE" },
	{ XtNtreeResizeCallback, XtCCallback, XtRCallback, sizeof( caddr_t ),
	        XtOffset(XpsiTreeWidget, tree.treeResize), XtRCallback, NULL },
};

static XtResource treeConstraintResources[] = {
	{ XtNparentWidget, XtCParentWidget, XtRPointer, sizeof( Widget ),
	  XtOffset(TreeConstraints, tree.parent), XtRPointer, NULL	},
	{ XtNkey, XtCKey, XtRInt, sizeof( int ),
	  XtOffset(TreeConstraints, tree.key), XtRString, "0"	  	},
	{ XtNbackgroundFill, XtCBackgroundFill, XtRPixel, sizeof( Pixel),
	        XtOffset(TreeConstraints, tree.backgroundFill), 
		XtRString, "Black" },

	{ XtNdrawShapeFunc, XtCDrawShapeFunc, XtRPointer, 
			sizeof( caddr_t ),
	        	XtOffset(TreeConstraints, tree.drawShape), 
			XtRPointer, NULL },

	{ XtNdrawShapeFuncArg, XtCDrawShapeFuncArg, XtRPointer, 
			sizeof( caddr_t ), 
			XtOffset(TreeConstraints, tree.drawShapeArg), 
			XtRPointer, NULL },

	{ XtNdrawConnectionFunc, XtCDrawConnectionFunc, XtRPointer, 
			sizeof( caddr_t ), 
			XtOffset(TreeConstraints, tree.drawConnection), 
			XtRPointer, NULL },

	{ XtNdrawConnectionFuncArg, XtCDrawConnectionFuncArg, XtRPointer, 
			sizeof( caddr_t ), 
			XtOffset(TreeConstraints, tree.drawConnectionArg), 
			XtRPointer, NULL },
	{ XtNnumTreeNodeChildren, XtCNumTreeNodeChildren, XtRInt, 
			sizeof( int ),
			XtOffset(TreeConstraints, tree.num_children), 
			XtRString, "0"	  	},
};


XpsiTreeClassRec	xpsiTreeClassRec = {

	{					/* Core Class part */
		(WidgetClass) &constraintClassRec,/* Superclass	 	*/
		"Tree",				/* class_name		*/
		sizeof(XpsiTreeRec),		/* widget_size		*/
		Class_Initialize,		/* class_initialize	*/
		NULL,				/* class_part_initialize*/
		FALSE,				/* class_inetd		*/
		Initialize,			/* initialize		*/
		NULL,				/* initialize_hook	*/
		XtInheritRealize,		/* realize		*/
		NULL,				/* actions		*/
		0,				/* num_actions		*/
		resources,			/* resources		*/
		XtNumber(resources),		/* num_resources	*/
		NULLQUARK,			/* xrm_class		*/
		TRUE,				/* compress_motion	*/
		 TRUE,		/*	   compress_exposure	*/
		/*XtExposeCompressMultiple,	/* compress_exposure	*/
		TRUE,				/* compress_enterleave	*/
		TRUE,				/* visibility_interest	*/
		Destroy,			/* destroy		*/
		Resize,				/* resize		*/
		Redisplay,			/* expose		*/
		SetValues,			/* set_values		*/
		NULL,				/* set_values_hook	*/
		XtInheritSetValuesAlmost,	/* set_values_almost	*/
		NULL,				/* get_values_hook	*/
		NULL,				/* accept_focus		*/
		XtVersion,			/* version		*/
		NULL,				/* callback private	*/
		NULL,				/* tm_table		*/
		NULL,				/* query_geometry	*/
		NULL,				/* display_accelerator	*/
		NULL,				/* extension		*/
	},
	{				/* Composite Class members 	*/
		GeometryManager,		/* geometry_manger 	*/
		ChangeManaged,			/* change_managed  	*/
		XtInheritInsertChild,		/* insert_child		*/
		XtInheritDeleteChild,		/* delete_child		*/
		NULL,				/* extension		*/
	},
	{				/* constraint Class members 	*/
		treeConstraintResources,	  /* subresources 	*/
		XtNumber(treeConstraintResources),/* subresources_count	*/
		sizeof(TreeConstraintRec),	  /* constraint_size	*/
		ConstraintInitialize,		  /* initialize		*/
		ConstraintDestroy,		  /* destroy		*/
		ConstraintSetValues,		  /* set_values		*/
		NULL,				  /* extension		*/
	},
	{				/* Tree Class Part */
		0,			/* Ignore */
	}
};

WidgetClass xpsiTreeWidgetClass = (WidgetClass) &xpsiTreeClassRec;

/* --------------------------- Class_Initialize --------------------------- */

static void
Class_Initialize( request, newW )
XpsiTreeWidget request, newW;
{
	/* Add new resource converters */

	XtAddConverter( XtRString, XtRTreeOrientation,
				   psi_cvt_str_to_treeOrientation, 
				   NULL, 0 );

	XtAddConverter( XtRString, XtRTreePacking,
				   psi_cvt_str_to_treePacking, 
				   NULL, 0 );

	XtAddConverter( XtRString, XtRConnectionStyle,
				   psi_cvt_str_to_connectionStyle, 
				   NULL, 0 );

	XtAddConverter( XtRString, XtRNodeSize,
				   psi_cvt_str_to_nodeSize, 
				   NULL, 0 );

	XtAddConverter( XtRString, XtRParentPlacement,
				   psi_cvt_str_to_parentPlacement, 
				   NULL, 0 );
}

extern int Debug;

/* ------------------- psi_cvt_str_to_treeOrientation ------------------- */

void
psi_cvt_str_to_treeOrientation( args, nargs, fromVal, toVal )
XrmValue	*args;
Cardinal	*nargs;
XrmValue	*fromVal;
XrmValue	*toVal;
{
	static	TreeOrientation result;
	char 	str[20];
	int 	i;

	if ( *nargs != 0 )
	   XtWarning("String to TreeOrientation conversion needs no arguments");

	/* Convert the string in the fromVal to one of the valid 
	 * orientation values.
	 */

	/* upper case the string */

	for ( i = 0 ; i < fromVal->size && i < 19 ; i++ )
		str[i] = (char) toupper( fromVal->addr[i] );
	str[i] = NULL_CHAR;


	toVal->size = sizeof( TreeOrientation );
	toVal->addr = (caddr_t) &result;

	if ( strcmp( str, "TOP_TO_BOTTOM" ) == 0 )
		result = TOP_TO_BOTTOM;
	else if ( strcmp( str, "BOTTOM_TO_TOP" ) == 0 )
		result = BOTTOM_TO_TOP;
	else if ( strcmp( str, "LEFT_TO_RIGHT" ) == 0 )
		result = LEFT_TO_RIGHT;
	else if ( strcmp( str, "RIGHT_TO_LEFT" ) == 0 )
		result = RIGHT_TO_LEFT;
	else if ( strcmp( str, "STAR_TOPOLOGY" ) == 0 )
		result = STAR_TOPOLOGY;
	else
	{
		result = LEFT_TO_RIGHT;
		XtStringConversionWarning( (char *) fromVal->addr,
							"TreeOrientation");
	}
}

/* ------------------- psi_cvt_str_to_treePacking ------------------- */

void
psi_cvt_str_to_treePacking( args, nargs, fromVal, toVal )
XrmValue	*args;
Cardinal	*nargs;
XrmValue	*fromVal;
XrmValue	*toVal;
{
	static	TreePacking result;
	char 	str[30];
	int 	i;

	if ( *nargs != 0 )
	   XtWarning("String to TreePacking conversion needs no arguments");

	/* Convert the string in the fromVal to one of the valid 
	 * tree packing values.
	 */

	/* upper case the string */

	for ( i = 0 ; i < fromVal->size && i < 29 ; i++ )
		str[i] = (char) toupper( fromVal->addr[i] );
	str[i] = NULL_CHAR;


	toVal->size = sizeof( TreePacking );
	toVal->addr = (caddr_t) &result;

	if ( strcmp( str, "ISOMORPHIC_PACKING" ) == 0 )
		result = ISOMORPHIC_PACKING;
	else if ( strcmp( str, "MINIMUM_WIDTH_PACKING" ) == 0 )
		result = MINIMUM_WIDTH_PACKING;
	else
	{
		result = ISOMORPHIC_PACKING;
		XtStringConversionWarning( (char *) fromVal->addr,
							"TreePacking");
	}

}

/* ------------------- psi_cvt_str_to_connectionStyle ------------------- */

void
psi_cvt_str_to_connectionStyle( args, nargs, fromVal, toVal )
XrmValue	*args;
Cardinal	*nargs;
XrmValue	*fromVal;
XrmValue	*toVal;
{
	static	ConnectionStyle result;
	char 	str[30];
	int 	i;

	if ( *nargs != 0 )
	   XtWarning("String to ConnectionStyle conversion needs no arguments");

	/* Convert the string in the fromVal to one of the valid 
	 * connection style values.
	 */

	/* upper case the string */

	for ( i = 0 ; i < fromVal->size && i < 29 ; i++ )
		str[i] = (char) toupper( fromVal->addr[i] );
	str[i] = NULL_CHAR;


	toVal->size = sizeof( ConnectionStyle );
	toVal->addr = (caddr_t) &result;

	if ( strcmp( str, "DIAGONAL_CONNECTION" ) == 0 )
		result = DIAGONAL_CONNECTION;
	else if ( strcmp( str, "TIER_CONNECTION" ) == 0 )
		result = TIER_CONNECTION;
	else if ( strcmp( str, "PATHWAY_CONNECTION" ) == 0 )
		result = PATHWAY_CONNECTION;
	else if ( strcmp( str, "NO_CONNECTION" ) == 0 )
		result = NO_CONNECTION;
	else
	{
		result = DIAGONAL_CONNECTION;
		XtStringConversionWarning( (char *) fromVal->addr,
							"ConnectionStyle");
	}

}

/* ------------------- psi_cvt_str_to_nodeSize ------------------- */

void
psi_cvt_str_to_nodeSize( args, nargs, fromVal, toVal )
XrmValue	*args;
Cardinal	*nargs;
XrmValue	*fromVal;
XrmValue	*toVal;
{
	static	NodeSize result;
	char 	str[30];
	int 	i;

	if ( *nargs != 0 )
		XtWarning("String to NodeSize conversion needs no arguments");

	/* Convert the string in the fromVal to one of the valid 
	 * node size values.
	 */

	/* upper case the string */

	for ( i = 0 ; i < fromVal->size && i < 29 ; i++ )
		str[i] = (char) toupper( fromVal->addr[i] );
	str[i] = NULL_CHAR;

	toVal->size = sizeof( NodeSize );
	toVal->addr = (caddr_t) &result;

	if ( strcmp( str, "VARIABLE_SIZE" ) == 0 )
		result = VARIABLE_SIZE;
	else if ( strcmp( str, "FIXED_SIZE_FOR_LEVEL" ) == 0 )
		result = FIXED_SIZE_FOR_LEVEL;
	else if ( strcmp( str, "FIXED_SIZE_FOR_TREE" ) == 0 )
		result = FIXED_SIZE_FOR_TREE;
	else 
	{
		result = VARIABLE_SIZE;
		XtStringConversionWarning( (char *) fromVal->addr,
							"NodeSize");
	}
}

/* ------------------- psi_cvt_str_to_parentPlacement ------------------- */

void
psi_cvt_str_to_parentPlacement( args, nargs, fromVal, toVal )
XrmValue	*args;
Cardinal	*nargs;
XrmValue	*fromVal;
XrmValue	*toVal;
{
	static	ParentPlacement result;
	char 	str[30];
	int 	i;

	if ( *nargs != 0 )
		XtWarning("String to ParentPlacement conversion needs no arguments");

	/* Convert the string in the fromVal to one of the valid 
	 * parent Placement values.
	 */

	/* upper case the string */

	for ( i = 0 ; i < fromVal->size && i < 29 ; i++ )
		str[i] = (char)toupper( fromVal->addr[i] );
	str[i] = NULL_CHAR;

	toVal->size = sizeof( ParentPlacement );
	toVal->addr = (caddr_t) &result;

	if ( strcmp( str, "PARENT_CENTERED" ) == 0 )
		result = PARENT_CENTERED;
	else if ( strcmp( str, "PARENT_ABOVE_FIRST_CHILD" ) == 0 )
		result = PARENT_ABOVE_FIRST_CHILD;
	else if ( strcmp( str, "PARENT_ABOVE_LAST_CHILD" ) == 0 )
		result = PARENT_ABOVE_LAST_CHILD;
	else 
	{
		result = VARIABLE_SIZE;
		XtStringConversionWarning( (char *) fromVal->addr,
							"ParentPlacement");
	}
}

/* ------------------------------ Initialize ------------------------------ */

static void
Initialize( request, newTW )
XpsiTreeWidget request, newTW;
{
	Arg		wargs[2];
	XGCValues	values;
	XtGCMask	valueMask;
	int 		n;
	Widget		newW;

	/* Make sure the window size is not zero , The Core Initialize
	 * does not do this
	 */


	if ( request->core.width <= 0 )
	{
	    if ( request->tree.minWidth > 0 )
		newTW->core.width = request->tree.minWidth;
	    else
		newTW->core.width = 4;
	}

	if ( request->core.height <= 0 )
	{
	    if ( request->tree.minHeight > 0 )
		newTW->core.height = request->tree.minHeight;
	    else
		newTW->core.height = 4;
	}

	/* Create graphics context for the connecting lines */

	valueMask = GCForeground | GCBackground;
	values.foreground = newTW->tree.foreground;
	values.background = newTW->core.background_pixel;


	newW = (Widget) newTW;

	newTW->tree.gc = XtGetGC( newW, valueMask, &values );

	/* Create graphics context for drawing nodes */

/*
	valueMask = GCForeground;
	values.foreground = newTW->tree.node_foreground;
	newTW->tree.node_fgc = XtGetGC( newW, valueMask, &values );

*/

	/* Note: GC's that are modified can't be shared so are created */

	valueMask = GCForeground | GCBackground;
	values.foreground = newTW->tree.node_foreground;
	values.background = newTW->core.background_pixel;
	newTW->tree.node_fgc = XCreateGC( XtDisplay(newTW), 
				RootWindowOfScreen(XtScreen( request )),
						valueMask, &values );

	valueMask = GCForeground | GCBackground;
	values.foreground = newTW->tree.node_background;
	values.background = newTW->tree.node_background;
	newTW->tree.node_bgc = XCreateGC( XtDisplay(newTW), 
				RootWindowOfScreen(XtScreen( request )),
						valueMask, &values );

	valueMask = GCForeground | GCBackground;
	values.foreground = newTW->tree.node_background;
	values.background = newTW->tree.node_background;
	newTW->tree.draw_gc = XCreateGC( XtDisplay(newTW), 
				RootWindowOfScreen(XtScreen( request )),
						valueMask, &values );

	/* Create graphics context for drawing rings */

	valueMask = GCForeground;
	values.foreground = newTW->tree.ring_foreground;
	newTW->tree.ring_fgc = XtGetGC( newW, valueMask, &values );

	valueMask = GCForeground;
	values.foreground = newTW->tree.ring_background;
	newTW->tree.ring_bgc = XtGetGC( newW, valueMask, &values );

	/* create the hidden root widget */

	newTW->tree.tree_root = (Widget) NULL;

	n = 0;
	XtSetArg( wargs[0], XtNwidth,  1 ); n++;
	XtSetArg( wargs[1], XtNheight, 1 ); n++;
	newTW->tree.tree_root = XtCreateWidget( "root", widgetClass, newW, 
							wargs, n);



	newTW->tree.do_layout = True;  /* Update layout when ChangeManaged */
	newTW->tree.do_redisplay = True;  /* redisplay on expose events */


				/* Unmanged widgets ignored during layout */

	newTW->tree.layout_unmanaged = False;  

	newTW->tree.max_x = 0;	/* Used to calculate minimum parent size */
	newTW->tree.max_y = 0;

	newTW->tree.max_node_width  = 0;      /* Used for symetric layouts */
	newTW->tree.max_node_height = 0;

	/* Start total_parents at -1, since we don't count the psuedo parent */

	newTW->tree.total_parents      = -1;
	newTW->tree.total_children     =  0;
	newTW->tree.num_connect_lines  =  0;
	newTW->tree.max_connect_lines  =  0;

	newTW->tree.connect_lines      = (XSegment *) NULL;

	newTW->tree.max_arcs  	     =  0;
	newTW->tree.num_arcs  	     =  0;

	newTW->tree.arcs  	     =  (XArc *) NULL;

	newTW->tree.max_rectangles     =  0;
	newTW->tree.num_rectangles     =  0;

	newTW->tree.rectangles  	     =  (XRectangle *) NULL;

	newTW->tree.max_rings  	     =  0;
	newTW->tree.num_rings  	     =  0;
	newTW->tree.rings  	     =  (XArc *) NULL;

	newTW->tree.fillColor  	     =  (Pixel *) NULL;

	/* Allocate the tables used by the layout algorithms  */

	newTW->tree.horizontal_pos   = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.vertical_pos     = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.max_width        = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.max_height       = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.breadth          = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.cur_breadth      = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.calc_breadth     = create_table(DEFAULT_TABLE_DEPTH);
	newTW->tree.ring_width       = create_table(DEFAULT_TABLE_DEPTH);

	newTW->tree.depth = 0;

	if ( newTW->tree.min_x_pad < 1 )
		newTW->tree.min_x_pad = 1;
	if ( newTW->tree.min_y_pad < 1 )
		newTW->tree.min_y_pad = 1;
	if ( newTW->tree.path_pad < 1 )
		newTW->tree.path_pad = 1;

	if ( newTW->tree.left_margin < 1 )
		newTW->tree.left_margin = 1;
	if ( newTW->tree.right_margin < 1 )
		newTW->tree.right_margin = 1;
	if ( newTW->tree.top_margin < 1 )
		newTW->tree.top_margin = 1;
	if ( newTW->tree.bottom_margin < 1 )
		newTW->tree.bottom_margin = 1;
}


/* ----------------------  ConstraintInitialize    ------------------------- */

static
void ConstraintInitialize( request, newW )
Widget request, newW;
{
	Arg	wargs[2];
	int 	n;
	Pixel	bg;
	Display *display;
	int      screen;
	XGCValues	values;
	XtGCMask	valueMask;


	TreeConstraints tree_const = TREE_CONSTRAINT(newW);
	XpsiTreeWidget   tw	   = (XpsiTreeWidget) newW->core.parent;

	/* Initialize the widget to have no children */

 	display = XtDisplay(newW);
        screen  = DefaultScreen( display );

	/* The parent and key values are set by the resource manager */

	tree_const->tree.num_children   = 0;
	tree_const->tree.layout_cnt     = 0;
	tree_const->tree.children       = (ChildList) NULL;
	tree_const->tree.x		= 0;
	tree_const->tree.y		= 0;
	tree_const->tree.radius		= 0;
	tree_const->tree.inferior_cnt	= 0;
	tree_const->tree.superior_cnt	= 0;
	tree_const->tree.shapeIndex	= 0;

	/* If no node_background constraint resource is specified we use
	 * the widgets background. This is fine for nodes with static color.
	 * For nodes that want dynamic color the calling code needs to 
	 * explicitly set the constraint resource as there is no way to 
	 * detect the widgets resource change(well at least no clean way 
	 * that I know of).
	 *
	 * This also allows the fill background to be different than the
	 * widgets background, perhaps desirable for pixmaps...
	 *
	 */

 	n = 0;
        XtSetArg( wargs[n], XtNbackground, &bg ); n++;
	XtGetValues( newW, wargs, n );

	tree_const->tree.node_background = bg;

	/* If Fill color resource specified then use it */

	if ( tree_const->tree.backgroundFill != BlackPixel( display, screen ) )
	{
	    tree_const->tree.node_background = tree_const->tree.backgroundFill;

	    /* Note that this obviously fails if the user specifies the fill
	     * as black. However the node will default to the initial widgets
	     * color, most likely the same(i.e. black). The caller should then
	     * change the values with calls to SetValues as needed. So in effect
	     * this is just to set the node_background IF the caller specifies
	     * a Fill color when the widget is created(which the really
	     * shouldnt). It handles all of these except black.
	     */
	}


	valueMask                = GCForeground;
	values.foreground        = tree_const->tree.node_background;
	tree_const->tree.node_gc = XtGetGC( newW, valueMask, &values );

	/* If this widget has a parent, add it to that widget's
	 * child list. Otherwise if we have a tree_root(we should) then
	 * make it a child of the tree_root widget.
	 */

	if ( tree_const->tree.parent )
	{
		if ( tree_const->tree.parent == ( Widget )TREE_SUPER_ROOT )
			insert_node( tw->tree.tree_root, newW );
		else
			insert_node( tree_const->tree.parent, newW );
	}
	else if ( tw->tree.tree_root )
	{
		insert_node( tw->tree.tree_root, newW );
	}
}

/* ------------------------------  SetValues  ----------------------------- */

static
Boolean SetValues( current, request, newTW )
XpsiTreeWidget current, request, newTW;
{
	int		redraw = FALSE;
	XGCValues	values;
	XtGCMask	valueMask;
	Widget		newW;

	DEBUG0(1, "XpsiTree: SetValues: \n" );

	/* If the foreground color has changed, redo the GC's and
	 * indicate a redraw
	 */

	newW = (Widget) newTW;

	if ( newTW->tree.foreground != current->tree.foreground ||
		newTW->core.background_pixel != current->core.background_pixel )
	{
		valueMask	  = GCForeground | GCBackground;
		values.foreground = newTW->tree.foreground;
		values.background = newTW->core.background_pixel;
		XtReleaseGC( newW, newTW->tree.gc );
		newTW->tree.gc      = XtGetGC( newW, valueMask, &values );
		redraw = TRUE;
	}

	/* If the node_background color has changed, redo the GC's and
	 * indicate a redraw
	 */

	if ( newTW->tree.node_foreground != current->tree.node_foreground )
	{
	        valueMask = GCForeground | GCBackground;
		values.foreground = newTW->tree.node_foreground;
		values.background = newTW->core.background_pixel;
		XFreeGC( XtDisplay(newTW), newTW->tree.node_fgc );

		newTW->tree.node_fgc = XCreateGC( XtDisplay(newTW), 
				RootWindowOfScreen(XtScreen( request )),
						valueMask, &values );
		redraw = TRUE;
	}

	if ( newTW->tree.node_background != current->tree.node_background )
	{
	        valueMask = GCForeground | GCBackground;
		values.foreground = newTW->tree.node_background;
		values.background = newTW->tree.node_background;
		XFreeGC( XtDisplay(newTW), newTW->tree.node_bgc );

		newTW->tree.node_fgc = XCreateGC( XtDisplay(newTW), 
				RootWindowOfScreen(XtScreen( request )),
						valueMask, &values );
		redraw = TRUE;
	}

	/* If the ring foreground or background color has changed, redo 
	 * the GC's and indicate a redraw
	 */

	if ( newTW->tree.ring_background != current->tree.ring_background )
	{
		valueMask = GCForeground;
		values.foreground = newTW->tree.ring_background;
		XtReleaseGC( newW, newTW->tree.ring_bgc );
		newTW->tree.ring_bgc      = XtGetGC( newW, valueMask, &values );
		redraw = TRUE;
	}

	if ( newTW->tree.ring_foreground != current->tree.ring_foreground )
	{
		valueMask = GCForeground;
		values.foreground = newTW->tree.ring_foreground;
		XtReleaseGC( newW, newTW->tree.ring_fgc );
		newTW->tree.ring_fgc      = XtGetGC( newW, valueMask, &values );
		redraw = TRUE;
	}

	/* If the minimum spacing has changed, recalculate the tree
	 * layout. new_layout() does a redraw, so we don't need SetValues
	 * to do another. Also call new_layout if the orientation has
	 * changed.
	 */

	if (      newTW->tree.min_y_pad     != current->tree.min_y_pad      ||
	          newTW->tree.min_x_pad     != current->tree.min_x_pad      ||
	          newTW->tree.connect_style != current->tree.connect_style  ||
	          newTW->tree.node_shape    != current->tree.node_shape     ||
	          newTW->tree.node_size     != current->tree.node_size      ||
	          newTW->tree.path_pad      != current->tree.path_pad       ||
	          newTW->tree.resize_policy != current->tree.resize_policy  ||
	          newTW->tree.left_margin   != current->tree.left_margin    ||
	          newTW->tree.right_margin  != current->tree.right_margin   ||
	          newTW->tree.top_margin    != current->tree.top_margin     ||
	          newTW->tree.bottom_margin != current->tree.bottom_margin  ||
	          newTW->tree.minWidth      != current->tree.minWidth       ||
	          newTW->tree.minHeight     != current->tree.minHeight      ||
	          newTW->tree.parents_place != current->tree.parents_place  ||
	          newTW->tree.center_root_x != current->tree.center_root_x  ||
	          newTW->tree.center_root_y != current->tree.center_root_y  ||
	          newTW->tree.show_rings    != current->tree.show_rings     ||
		  newTW->tree.orientation   != current->tree.orientation       )
	{
		if ( newTW->tree.min_x_pad < 1 )
			newTW->tree.min_x_pad = 1;
		if ( newTW->tree.min_y_pad < 1 )
			newTW->tree.min_y_pad = 1;
		if ( newTW->tree.path_pad < 1 )
			newTW->tree.path_pad = 1;

		if ( newTW->tree.left_margin < 1 )
			newTW->tree.left_margin = 1;
		if ( newTW->tree.right_margin < 1 )
			newTW->tree.right_margin = 1;
		if ( newTW->tree.top_margin < 1 )
			newTW->tree.top_margin = 1;
		if ( newTW->tree.bottom_margin < 1 )
			newTW->tree.bottom_margin = 1;

		if ( newTW->tree.do_layout == True )
			new_layout( newTW );

		redraw = FALSE;
	}
	      
	return(redraw);
}

/* -------------------------  ConstraintSetValues   -------------------- */

static
Boolean ConstraintSetValues( current, request, newW )
Widget current, request, newW;
{
	TreeConstraints newconst      = TREE_CONSTRAINT(newW);
	TreeConstraints current_const = TREE_CONSTRAINT(current);
	XpsiTreeWidget        tw = (XpsiTreeWidget) newW->core.parent;
	TreeConstraints      pc;
	TreeConstraints      cc;
	TreeConstraintPart  *pct;
	TreeConstraintPart  *cct;
	ChildPtr	    *p;
	ChildPtr	    *last_p;
	ChildPtr	    *node_p;
	int		    key;
	XGCValues	    values;
	XtGCMask	    valueMask;
	Boolean		    redraw=FALSE;

	DEBUG0(1, "XpsiTree: ConstraintSetValues: \n" );

	/* If the parent has changed, remove the widget
	 * from the old widget's child list and add it to the
	 * new one.
	 */

	if ( current_const->tree.parent != newconst->tree.parent )
	{
		if ( current_const->tree.parent )
			delete_node( current_const->tree.parent, newW );
		if ( newconst->tree.parent )
		{
			if ( newconst->tree.parent == (Widget)TREE_SUPER_ROOT )
				insert_node( tw->tree.tree_root, newW );
			else
				insert_node( newconst->tree.parent, newW );
		}

		/* If the Tree widget has been realized, compute the new
		 * layout
		 */

		if ( XtIsRealized((Widget)tw) && (tw->tree.do_layout == True))
			new_layout( tw );
	}

	if ( current_const->tree.key != newconst->tree.key )
	{
		/* We now have a node which has a new key, this new
		 * key may be out of order, so we find the node with the
		 * new key, extract it then do an insertion sort.
		 */

		/* Scan list of parent's children, if any, and move nodes
		 * position to it's new position if required.
		 */

		if ( newconst->tree.parent == NULL )
			return(FALSE);

		key = newconst->tree.key;

		pc = TREE_CONSTRAINT(newconst->tree.parent);  
		pct = &(pc->tree);  
		last_p = pct->children;
		for ( p = last_p ; p != NULL ; p = p->next )
		{
			cc=TREE_CONSTRAINT(p->child);
			cct = &(cc->tree);  
			if ( cct->key == key )
				break;
			last_p = p;
		}

		if ( p == NULL )
		{
			(void)fprintf(stderr, "Xpsi:Tree New Key not found\n");
			return(FALSE);
		}

		node_p = p;   /* Save child node pointer for later */

		if ( p == pct->children )
		{
			pct->children = p->next;  /* Remove from head */
		}
		else
			last_p->next = p->next;  /* Remove from child list */

		node_p->next = NULL;	/* Prepare for re-insertion */

		 /* Sorted insertion in child list */
		last_p = pct->children;
		for ( p = last_p ; p != NULL ; p = p->next )
		{
			cc=TREE_CONSTRAINT(p->child);
			cct = &(cc->tree);  
			if ( key < cct->key )
				break;
			last_p = p;
		}

		/* Check for a now empty list of children */

		if ( pct->children == NULL )	/* Insert at head of list   */
		{
			node_p->next = pct->children;
			pct->children = node_p;
		}
		else if ( p == NULL )		/* Append to child list     */
		{
			last_p->next = node_p;
		}
		else if ( p == pct->children )	/* Insert at head of list   */
		{
			node_p->next = pct->children;
			pct->children = node_p;
		}
		else				/* Insert in middle of list */
		{
			last_p->next = node_p;
			node_p->next   = p;
		}

		if (XtIsRealized((Widget)tw) && ( tw->tree.do_layout == True ))
			new_layout( tw );
	}

	if ( current_const->tree.backgroundFill != 
						newconst->tree.backgroundFill )
	{
	    newconst->tree.node_background = newconst->tree.backgroundFill;

	    /* Update nodes pixel value in shape array */

	    tw->tree.fillColor[newconst->tree.shapeIndex] =
	    					newconst->tree.backgroundFill;

	    /* Redraw node background with changed color */

	    drawTreeNode( tw, newconst->tree.shapeIndex );

	    redraw = TRUE;
	}

	/* Ignore any other values */

	return( redraw );
}

/* -------------------------  insert_node  ---------------------------- */

static
void insert_node( parent, node )
Widget parent, node;
{
	TreeConstraints      pc=TREE_CONSTRAINT(parent);
	TreeConstraints      nc=TREE_CONSTRAINT(node);
	TreeConstraints      cc;
	TreeConstraintPart  *pct;
	TreeConstraintPart  *nct;
	TreeConstraintPart  *cct;
	ChildPtr	    *p;
	ChildPtr	    *last_p;
	ChildPtr	    *cptr;
	XpsiTreeWidget        tw = (XpsiTreeWidget) node->core.parent;


	/* Set pointers to tree part of constraints record */

	pct = &(pc->tree);  
	nct = &(nc->tree);  

	/* Create a new pointer node to hold the child pointer for the
	 * new node. This pointer node is then inserted into the list of
	 * child pointer nodes.
	 */

	cptr = ( ChildPtr * ) malloc( sizeof( ChildPtr ) );
	if ( cptr == NULL )
	{
		(void)fprintf( stderr, "TreeWidget: Malloc failed\n");
		return;
	}

	cptr->child = node;	/* Set the child pointer */
	cptr->next  = NULL;

	/* Set the parent field of the tree record in the new nodes 
	 * constraint record to point to the parent widget.
	 */

	nct->parent = parent;	

	/* Now perform a sorted insertion of the child pointer into the list
	 * of children in the tree structure of parents constraint record.
	 */

	if ( pct->children == NULL )
	{
		/* This is the first child so we have a new parent */

		tw->tree.total_parents  += 1;

		pct->children = cptr;
	}
	else
	{	/* Find position to insert new node  		*/

		last_p = pct->children;
		for ( p = last_p ; p != NULL ; p = p->next )
		{
			cc=TREE_CONSTRAINT(p->child);
			cct = &(cc->tree);  
			if ( nct->key < cct->key )
				break;
			last_p = p;
		}

		if ( p == NULL )		/* Append to child list     */
		{
			last_p->next = cptr;
		}
		else if ( p == pct->children )	/* Insert at head of list   */
		{
			cptr->next = pct->children;
			pct->children = cptr;
		}
		else				/* Insert in middle of list */
		{
			last_p->next = cptr;
			cptr->next   = p;
		}
	}

	tw->tree.total_children += 1;
	pct->num_children++;
}

/* -------------------------  delete_node  ---------------------------- */

static
void delete_node( parent, node )
Widget parent, node;
{
	TreeConstraints 	 pc;
	TreeConstraintPart  	*pct;
	ChildPtr	    	*p;
	ChildPtr	    	*last_p;
	XpsiTreeWidget            tw = (XpsiTreeWidget) node->core.parent;

	/* Make sure the parent node exists */

	if ( !parent )
		return;

	pc = TREE_CONSTRAINT( parent );

	/* Set pointers to the tree part of the constraint record */

	pct = &(pc->tree);  

	/* Find the child node in the parent's child pointer list */

	last_p = pct->children;
	for ( p = last_p ; p != NULL ; p = p->next )
	{
		if ( p->child == node )
			break;
		last_p = p;
	}

	if ( p == NULL ) 			/* Failed to find node!      */
		return;

	if ( p == pct->children )  	
		pct->children = p->next; 	/* Remove from head 	     */
	else				
		last_p->next = p->next; 	/* Remove from middle or end */

	if ( p != NULL )			/* Free up pointer node      */
		free( p );

	pct->num_children--; 	/* Decrement the number of children */

	if ( pct->num_children == 0 )		/* No longer parent */
		tw->tree.total_parents  -= 1;

	tw->tree.total_children -= 1;
}

/* -------------------------  Resize  -------------------------- */

static void Resize( tw )
XpsiTreeWidget tw;
{
	/* We could change the graph drawing rules based on the 
	 * size that we have just changed to. i.e. auto scale to fit.
	 *
	 * This is left for future updates.
	 *
	 */
}

/* -------------------------  Destroy  -------------------------- */

static
void Destroy( tw )
XpsiTreeWidget  tw;
{
	Widget tW;

	DEBUG0(1, "XpsiTree: Destroy: \n" );

	tW = (Widget) tw;

	XtReleaseGC( tW, tw->tree.gc );
	XtReleaseGC( tW, tw->tree.ring_fgc );
	XtReleaseGC( tW, tw->tree.ring_bgc );

	XFreeGC( XtDisplay(tW), tw->tree.node_fgc );
	XFreeGC( XtDisplay(tW), tw->tree.node_bgc );
	XFreeGC( XtDisplay(tW), tw->tree.draw_gc );

	destroy_table( tw->tree.horizontal_pos );
	destroy_table( tw->tree.vertical_pos );
	destroy_table( tw->tree.max_width );
	destroy_table( tw->tree.max_height );
	destroy_table( tw->tree.breadth );
	destroy_table( tw->tree.cur_breadth );
	destroy_table( tw->tree.calc_breadth );
	destroy_table( tw->tree.ring_width );

	if ( tw->tree.rectangles != NULL )
		XtFree( (Xmem) tw->tree.rectangles );
	if ( tw->tree.rings != NULL )
		XtFree( (Xmem) tw->tree.rings );
	if ( tw->tree.arcs != NULL )
		XtFree( (Xmem) tw->tree.arcs );
	if ( tw->tree.connect_lines != NULL )
		XtFree( (Xmem) tw->tree.connect_lines );
	if ( tw->tree.fillColor != NULL )
		XtFree( (Xmem) tw->tree.fillColor );
}


/* -------------------------  ConstraintDestroy  -------------------------- */

static
void ConstraintDestroy( w )
Widget	w;
{
	TreeConstraints	tree_const = TREE_CONSTRAINT( w );
	TreeConstraints	child_const;
	TreeConstraintPart  	*t;
	ChildPtr	    	*p;
	ChildPtr	    	*last_p;
	XpsiTreeWidget            tw = (XpsiTreeWidget) w->core.parent;

	DEBUG0(1, "XpsiTree: ConstraintDestroy: \n" );

	/* Normally we Remove the widget from parent's child list and
	 * make all it's child widgets of the parent. However during
	 * the destruction of the tree widget we just remove the nodes
	 * child pointer records.
	 */

	t = &(tree_const->tree);  

	if ( tw->tree.tree_root == w )
	{
		/* Removing root node, this should only occur during the
		 * destruction of the tree widget. When this occurs we
		 * set the tree_root node pointer to NULL. This then
		 * acts as a flag to further calls to this routine with
		 * other members of the tree, in which case we don't
		 * connect children to grand($k) parents 
		 */

		 tw->tree.tree_root = NULL;
		 tw->tree.do_layout = False;
	}

	if ( tree_const->tree.parent && tw->tree.tree_root != NULL )
	{
		/* Remove child pointer to this widget */
		delete_node( tree_const->tree.parent, w );
	}

	for ( p = t->children ; p != NULL ; )
	{
		if ( tw->tree.tree_root != NULL )
		{
			/* Add child to parents list of children */
			if ( t->parent )
				insert_node( t->parent, p->child );
			else
			{
				child_const = TREE_CONSTRAINT( p->child );
				child_const->tree.parent = NULL;
			}
		}
		else
		{
			child_const = TREE_CONSTRAINT( p->child );
			child_const->tree.parent = NULL;
		}

		last_p = p; 		/* Free up child pointer record */
		p      = p->next;
		free( last_p );
	}

	t->num_children = 0;
	t->children = NULL;

	XtReleaseGC( w, tree_const->tree.node_gc );

	if ( tw->tree.do_layout == True )
		new_layout( tw );
}


/* -----------------------  Geometry Manager    --------------------------- */

static
XtGeometryResult GeometryManager( w, request, reply )
Widget 			w;
XtWidgetGeometry	*request;
XtWidgetGeometry	*reply;
{
	XpsiTreeWidget	  tw = (XpsiTreeWidget) w->core.parent;

	/* No position changes allowed */

	if ( (request->request_mode & CWX && request->x != w->core.x)	||
	     (request->request_mode & CWY && request->y != w->core.y) )
	{
	        DEBUG0(0, "XpsiTree: GeometryManager:  No moves Allowed\n" );
		return( XtGeometryNo);
	}

	DEBUG0(1, "XpsiTree: GeometryManager: \n" );

	/* Allow all resize requests */

	if ( request->request_mode & CWWidth  )
		w->core.width = request->width;
	if ( request->request_mode & CWHeight )
		w->core.height = request->height;
	if ( request->request_mode & CWBorderWidth )
		w->core.border_width = request->border_width;

	/* Compute the new layout based on the new widget sizes */

	if ( tw->tree.do_layout == True )
		new_layout(tw);

	return( XtGeometryYes );
}

/* -----------------------  ChangedManaged   --------------------------- */

static
void ChangeManaged( tw )
XpsiTreeWidget	tw;
{
	DEBUG1(1, "XpsiTree: ChangedManaged: do_layout(%d)\n", 
							tw->tree.do_layout );

	/* If the needs_layout  flag is true then doit */

	if ( tw->tree.do_layout == True )
		new_layout( tw );
}

void
XpsiDoTreeLayout( tw, layout_flag )
XpsiTreeWidget tw;
Boolean layout_flag;
{

	DEBUG2(1, "XpsiTree: DoTreeLayout: do_layout(%d) new layout_flag(%d)\n",
			tw->tree.do_layout, layout_flag );

	if ( ( tw->tree.do_layout == False ) && layout_flag )
	{
		new_layout( tw );
	}

	tw->tree.do_layout = layout_flag;
}

void
XpsiLayoutUnmanaged( tw, unmanaged_flag )
XpsiTreeWidget tw;
Boolean unmanaged_flag;
{

	/* Sets flag such that when true, widgets which are unmanaged are 
	 * included in the layout algorithm, i.e. space is provided for
	 * them even though they are not visible. This allows for
	 * quick hide/expands.
	 */

	DEBUG2(1,"XpsiTree:LayoutUnmanaged: layout_unmanaged(%d) newflag(%d)\n",
			tw->tree.layout_unmanaged, unmanaged_flag );

	tw->tree.layout_unmanaged = unmanaged_flag;
}


void
XpsiDoTreeRedisplay( tw, redisplay_flag )
XpsiTreeWidget tw;
Boolean redisplay_flag;
{
	XEvent		event;
	Region		region;

	/* Sets flag such that when true, all expose events cause a redraw
	 * of the region specified. When set false all expose events are
	 * ignored. When set from false to true, we automatically call
	 * one redraw. Default is True. This is useful when doing lot's
	 * of moves, add's, deletes, as it cuts way down on the number
	 * of redundant redraws. There is however a timing problem, the
	 * caller must wait  until the server has returned all the
	 * appropriate expose events, before enabling the flag to do
	 * a final redraw. This can be accomplished using the XSync
	 * function.
	 */

	DEBUG2(1,"XpsiTree:DoTreeRedisplay: redaw_flag(%d) newflag(%d)\n",
			tw->tree.do_redisplay, redisplay_flag);

	if ( ( tw->tree.do_redisplay == False ) && redisplay_flag )
	{
		/* XXX Create psuedo event and region */
		
		/* Ignore event for now as it's not used */

		Redisplay( tw, &event, region );
	}

	tw->tree.do_redisplay = redisplay_flag;
}


/* Return a list of the children of the supplied widget
 *
 * Children in this sense are the ones drawn as children in the
 * tree diagram and not the children in the widget tree.
 *
 */

void
XpsiGetChildrenWidgets( tw, tn, list)
XpsiTreeWidget tw;
Widget tn;
WidgetList list;  /* Caller creates and releases this space */
{
	TreeConstraints		tree_const;
	TreeConstraintPart      *ct;
	int			i=0;
	ChildPtr	 	*p;

	tree_const = TREE_CONSTRAINT(tn);
	ct = &(tree_const->tree);

	for ( p = ct->children; p != NULL; p = p->next )
	{
		list[i++] = p->child;
	}
}


/* ----------------------------  Redisplay   --------------------------- */

static void
Redisplay( w, event, region )
XpsiTreeWidget    w;
XEvent		*event;
Region		 region;
{
	TreeConstraints		tree_const;
	TreeConstraintPart      *ct;
	int			 i;


	/* If the tree widget is visible and has a root node, display 
	 * connections.
	 */

	if ( w->tree.do_redisplay && w->core.visible && w->tree.tree_root )
	{

		DEBUG0(1, "XpsiTree: Redisplay\n" );

		/* Get the pointer to the psuedo root tree constraint 
		 * record, it contains the pointer to the root widget of
		 * the tree.
		 */

		tree_const = TREE_CONSTRAINT(w->tree.tree_root);
		ct = &(tree_const->tree);

		if ( w->tree.orientation == STAR_TOPOLOGY && 
						w->tree.show_rings )
		{
			XSetRegion(XtDisplay(w), w->tree.ring_bgc, region );

	   		XFillArcs(XtDisplay(w), XtWindow(w), 
					w->tree.ring_bgc,
					w->tree.rings,
					w->tree.num_rings );

			XSetRegion(XtDisplay(w), w->tree.ring_fgc, region );

	   		XDrawArcs(XtDisplay(w), XtWindow(w), 
					w->tree.ring_fgc,
					w->tree.rings,
					w->tree.num_rings );
		}

		if ( ct->num_children > 0 )
		{
			if ( w->tree.connect_style != NO_CONNECTION )
			{
				XSetRegion(XtDisplay(w), w->tree.gc, region );
				XDrawSegments(  XtDisplay(w), 
						XtWindow(w), w->tree.gc, 
				     		w->tree.connect_lines,
				     		w->tree.num_connect_lines );
			}
		}

		if ( w->tree.orientation == STAR_TOPOLOGY && 
						w->tree.num_arcs > 0 )
		{
		    XSetForeground(XtDisplay(w), w->tree.node_bgc, 
						w->tree.node_background );

		    XSetRegion(XtDisplay(w), w->tree.node_bgc, region );

		    for ( i = 0 ; i < w->tree.num_arcs ; i++ )
		    {
			XSetForeground(XtDisplay(w), w->tree.node_bgc, 
							w->tree.fillColor[i] );

	   		XFillArcs(XtDisplay(w), XtWindow(w), w->tree.node_bgc,
					&(w->tree.arcs[i]), 1 );
		    }

		    XSetRegion(XtDisplay(w), w->tree.node_fgc, region );
		    XSetForeground(XtDisplay(w), w->tree.node_fgc, 
						w->tree.node_foreground );

	   	    XDrawArcs(XtDisplay(w), XtWindow(w), 
					w->tree.node_fgc,
					w->tree.arcs,
					w->tree.num_arcs );
		}


		if ( w->tree.orientation != STAR_TOPOLOGY && 
						w->tree.num_rectangles > 0 )
		{
		  XSetRegion(XtDisplay(w), w->tree.node_bgc, region );
		  XSetForeground(XtDisplay(w), w->tree.node_bgc, 
						w->tree.node_background );

		  if ( w->tree.node_size != VARIABLE_SIZE )
		  {
		    for ( i = 0 ; i < w->tree.num_rectangles ; i++ )
		    {
			XSetForeground(XtDisplay(w), w->tree.node_bgc, 
							w->tree.fillColor[i] );

			XFillRectangles( XtDisplay(w), 
					XtWindow(w), w->tree.node_bgc, 
			     		&(w->tree.rectangles[i]), 1 );
		    }
		  }
		  else
		  {
			XFillRectangles(  XtDisplay(w), 
					XtWindow(w), w->tree.node_bgc, 
			     		w->tree.rectangles,
			     		w->tree.num_rectangles );
		  }

		  XSetRegion(XtDisplay(w), w->tree.node_fgc, region );

		  XDrawRectangles(  XtDisplay(w), 
					XtWindow(w), w->tree.node_fgc, 
			     		w->tree.rectangles,
			     		w->tree.num_rectangles );
		}

   	}
}

static void
drawTreeNode( w, shapeIndex )
XpsiTreeWidget    w;
int shapeIndex;
{

	XSetForeground(XtDisplay(w), w->tree.draw_gc, 
					w->tree.fillColor[shapeIndex] );

	if ( w->tree.orientation == STAR_TOPOLOGY && w->tree.num_arcs > 0 )
	{
   		XFillArcs(XtDisplay(w), XtWindow(w), w->tree.draw_gc,
				&(w->tree.arcs[shapeIndex]), 1 );

		XSetForeground(XtDisplay(w), w->tree.draw_gc, 
					w->tree.node_foreground );

   		XDrawArcs(XtDisplay(w), XtWindow(w), w->tree.draw_gc,
				&(w->tree.arcs[shapeIndex]), 1 );
	}
	else if ( w->tree.num_rectangles > 0 )
	{
		if ( w->tree.node_size != VARIABLE_SIZE )
		{
		    XFillRectangles(XtDisplay(w), XtWindow(w), 
				w->tree.draw_gc, 
		     		&(w->tree.rectangles[shapeIndex]), 1 );
		}

		XSetForeground(XtDisplay(w), w->tree.draw_gc, 
					w->tree.node_foreground );

		XDrawRectangles(  XtDisplay(w), 
				XtWindow(w), w->tree.draw_gc, 
		     		&(w->tree.rectangles[shapeIndex]), 1 );
	}
}

/* ----------------------------  new_layout ------------------------------ */

static
void new_layout( tw )
XpsiTreeWidget	tw;
{
	Dimension	 	replyWidth = 0, replyHeight = 0;
	XtGeometryResult 	result;
	TreeConstraints  	tree_const;
	TreeConstraintPart      *ct;
	int			rc, not_done = 1;
	int		  	sanity_cnt=0;
	xpsiTreeCallbackStruct	cb;
	int			doResize=0;

	DEBUG0(1, "XpsiTree: new_layout\n");

	/* Reset the layout tables */

	reset_table( tw->tree.vertical_pos );
	reset_table( tw->tree.horizontal_pos );
	reset_table( tw->tree.max_width );
	reset_table( tw->tree.max_height );

	if ( tw->tree.orientation == STAR_TOPOLOGY )
	{
		reset_table( tw->tree.breadth );
		reset_table( tw->tree.calc_breadth );
		reset_table( tw->tree.ring_width );
	}

	tw->tree.depth = 0;
	tw->tree.max_x = 0;  tw->tree.max_y = 0;
	tw->tree.min_x = 0;  tw->tree.min_y = 0;

	tw->tree.max_node_width  = 0;  
	tw->tree.max_node_height = 0;

	/* Compute each widget's x,y position */

	switch ( tw->tree.orientation )
	{
		case LEFT_TO_RIGHT :
		case RIGHT_TO_LEFT :
			compute_node_extents( tw, tw->tree.tree_root, 0);
			compute_LtoR_positions( tw, tw->tree.tree_root, 0 );
			break;


		case TOP_TO_BOTTOM :
		case BOTTOM_TO_TOP :
			compute_node_extents( tw, tw->tree.tree_root,0);
			compute_TtoB_positions( tw, tw->tree.tree_root, 0 );
			break;

		case STAR_TOPOLOGY:
			compute_node_diameters( tw, tw->tree.tree_root, 0 );
			while ( not_done )
			{
				reset_table( tw->tree.cur_breadth );
				calculate_tree_rings( tw );
				rc = compute_Star_positions( tw, 
						     tw->tree.tree_root, 0);

				if ( rc >= 0 )
				{
				    not_done = False;
				}
				else if ( sanity_cnt > 1000 )
				{
				    (void)fprintf(stderr,
					   "Exceeded 1000 tree expansions\n");
				    not_done = False;
				}
				else
				{
					reset_table( tw->tree.horizontal_pos );
				}
				sanity_cnt++;
			}

			break;


		default:
			compute_node_extents(   tw, tw->tree.tree_root, 0);
			compute_LtoR_positions( tw, tw->tree.tree_root, 0 );
			break;
	}

	/* move each widget into place. */

	set_positions( tw, tw->tree.tree_root, 0 );

	/* If the orientation is a star then we need to translate the
	 * node coordinates from the 4 quadrant 0 centered values to
	 * the widgets coordinate space, i.e. 0,0 in upper left, y increasing
	 * downwards. We also adjust coordinates to account for centering
	 * of the tree in either or both the X and Y axis. Finally, if the
	 * tree rings are requested to be fully visible we adjust accordingly.
	 */


	if ( tw->tree.orientation == STAR_TOPOLOGY )
	{
		/* If we desire tree root to be centered  on  x */

		if ( tw->tree.center_root_x == True )
		{
			if ( abs(tw->tree.min_x) > ((int)tw->tree.max_x) )
			        tw->tree.max_x = (Dimension)abs(tw->tree.min_x);
			else
				tw->tree.min_x = 0 - ((int) tw->tree.max_x);
		}

		/* If we desire tree root to be centered  on  y */

		if ( tw->tree.center_root_y == True )
		{
			if ( (int)tw->tree.max_y > abs(tw->tree.min_y) )
				tw->tree.min_y = 0 - ( (int) tw->tree.max_y );
			else
				tw->tree.max_y = (Dimension)abs(tw->tree.min_y);
		}

		if ( tw->tree.show_rings == True ) /* Overide above */
		{
			tw->tree.max_x = sum_table( tw->tree.ring_width,
							 tw->tree.depth + 1);

			tw->tree.min_x = 0 - ( tw->tree.max_x +
							tw->tree.left_margin );
			tw->tree.max_y = sum_table( tw->tree.ring_width,
						tw->tree.depth + 1);

			tw->tree.min_y = 0 - ( tw->tree.max_y +
							tw->tree.top_margin );
		}

		center_star_tree( tw, tw->tree.tree_root, 0 );

		/* Adjust max values to represent polar->X translation */

		tw->tree.max_x -= tw->tree.min_x; /*  Offset for resize */
		tw->tree.max_y -= tw->tree.min_y;
	}

	/* Compute position of connections, node shapes and tree rings */

	tree_const = TREE_CONSTRAINT(tw->tree.tree_root);
	ct = &(tree_const->tree);
	if ( ct->num_children > 0 )
	{
		adjust_arcs( tw );
		adjust_rings( tw );
		adjust_rectangles( tw );
		if ( tw->tree.node_shape == True )
		{
			compute_node_shapes( tw, ct->children->child, 1 ); 
		}

		if ( tw->tree.connect_style != NO_CONNECTION )
		{
			adjust_connection_segments( tw );
			compute_connections( tw, ct->children->child, 1);
		}

		if ( tw->tree.orientation == STAR_TOPOLOGY &&
						  tw->tree.show_rings == True )
			compute_tree_rings( tw );
	}

	/* Add right and bottom margins */
	tw->tree.max_x += tw->tree.right_margin;
	tw->tree.max_y += tw->tree.bottom_margin;


	if ( tw->tree.minWidth > tw->tree.max_x )
	{
		tw->tree.max_x = tw->tree.minWidth;
	}

	if ( tw->tree.minHeight > tw->tree.max_y )
	{
		tw->tree.max_y = tw->tree.minHeight;
	}

	/* If the tree has auto_resize policy enabled, then the tree
	 * request it's parent to grow and shrink as it grows and shrinks.
	 * Otherwise we implement a "growth only" policy. This is a mixed
	 * blessing, if a tree grow's and shrinks frequently then growing
	 * and shrinking the window is wasteful. The only real problem with
	 * this is if a tree gets very large, then shrinks significantly
	 * and never grows again. On most systems the "wasted" memory will
	 * get paged out anyhow. So the only real drawback would be for
	 * a scrolled window which will have a small viewport on the
	 * the large client window. In such rare cases the treeResize
	 * callback can be used to change the window size if desired.
	 */

	/* We perform the callback first to so the outer window can
	 * be resized before the tree is 
	 */

	if ( tw->tree.max_x != tw->core.width || 
				tw->tree.max_y != tw->core.height)
	{
		cb.reason = Xpsi_TREE_RESIZED;
		cb.width  = tw->tree.max_x;
		cb.height = tw->tree.max_y;

		cb.last_width  = tw->core.width;
		cb.last_height = tw->core.height;

		XtCallCallbacks( (Widget)tw, XtNtreeResizeCallback, &cb );
	}

	result = XtGeometryDone;

	if ( tw->tree.resize_policy )	/* Min size */
	{
		result = XtMakeResizeRequest( (Widget) tw, 
						tw->tree.max_x, tw->tree.max_y,
					  	&replyWidth, &replyHeight );

#ifdef LESSTIF
		tw->core.width = replyWidth;
		tw->core.height = replyHeight;
		doResize++;
#endif

	}
	else if ( tw->tree.max_x > tw->core.width || 
			tw->tree.max_y > tw->core.height )
	{
		result = XtMakeResizeRequest( (Widget) tw, 
					MAX( tw->core.width, tw->tree.max_x),
					MAX( tw->core.height, tw->tree.max_y),
					&replyWidth, &replyHeight );

#ifdef LESSTIF
		tw->core.width = replyWidth;
		tw->core.height = replyHeight;
		doResize++;
#endif
	}


	/* Accept any compromise */

	if ( result == XtGeometryAlmost )
	{
		XtMakeResizeRequest( (Widget) tw, replyWidth, replyHeight,
							NULL, NULL );
	}

	if ( XtIsRealized( (Widget) tw ) )
	{
		/* If realized, clear the Tree widget's window.
	 	 * The resulting Expose event will result in a
	 	 * call to the widget's Redisplay() method.
	 	 */

#ifdef LESSTIF
if ( doResize )
    XResizeWindow( XtDisplay(tw), XtWindow(tw), replyWidth, replyHeight);
#endif
		XClearArea( XtDisplay(tw), XtWindow(tw), 0, 0, 0, 0, TRUE );
	}
}


/* --------------------  compute_LtoR_positions  ------------------------ */

static
int compute_LtoR_positions( tw, w, level )
XpsiTreeWidget   tw;
Widget		w;
int		level;
{
	Position  	 current_hpos, hpos, current_vpos, pos;
	Position  	 space_req;
	Position  	 width=0, height=0;
	int	  	 i, depth = 0;
	int		 max_subtree_depth=0;
	TreeConstraints	 tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;
	ChildPtr	 *first_p;
	ChildPtr	 *last_p;
	int		 h, max_width_on_level;



	if ( level > tw->tree.depth ) 		/* Set the depth of tree */
		tw->tree.depth = level;

	/* Get the current positions for this level  */

	current_hpos = current_value( tw->tree.horizontal_pos, level );
	current_vpos = current_value( tw->tree.vertical_pos,   level );

	switch ( tw->tree.node_size )
	{
		case VARIABLE_SIZE:
			width  = w->core.width;
			height = w->core.height;
			break;

		case FIXED_SIZE_FOR_TREE:
			width  = tw->tree.max_node_width;
			height = tw->tree.max_node_height;
			break;

		case FIXED_SIZE_FOR_LEVEL:
			width   = current_value( tw->tree.max_width, level );
			height  = current_value( tw->tree.max_height, level );
			break;
	}

	hpos = MAX( current_hpos, width );

	if ( tree_const->tree.layout_cnt == 0 ) /* No visible children */
	{
		tree_const->tree.y = current_vpos;
		max_subtree_depth  = level;
	}
	else
	{
		Widget		  first_kid, last_kid;
		TreeConstraints   const1, const2;
		Position	  top, bottom;

		/* If the node has children, recursively figure the
		 * positions of each child.
		 */

		first_p = last_p = NULL;
		for ( p = tree_const->tree.children; p != NULL; p = p->next )
		{
		    if ( XtIsManaged( p->child ) || tw->tree.layout_unmanaged )
		    {
			if ( first_p == NULL )
				first_p = p;

			depth = compute_LtoR_positions(tw,p->child,level + 1);

			max_subtree_depth = MAX( max_subtree_depth, depth );

			last_p = p;
		    }
		}
		
		/* Now that the vertical positions of all children are 
		 * known, find the vertical extent of all sub_trees.
		 */

		first_kid = first_p->child;
		last_kid  = last_p->child;

		const1 = TREE_CONSTRAINT(first_kid);
		const2 = TREE_CONSTRAINT(last_kid);

		top    = const1->tree.y + first_kid->core.height / 2;
		bottom = const2->tree.y + last_kid->core.height  / 2;


		switch ( tw->tree.parents_place ) /* Position parent */
		{
		    case PARENT_CENTERED:
				tree_const->tree.y = (top + bottom)/2 - 
							(height/2);
				break;

		    case PARENT_ABOVE_FIRST_CHILD: tree_const->tree.y = 
		    					    const1->tree.y;
				break;

		    case PARENT_ABOVE_LAST_CHILD: 
		    

			if ( last_kid->core.height < (2 * height) )
		    	    h = bottom - ( height/2 );
			else
		    	    h = const2->tree.y;

			if ( h < 0 ) h = 0;

		    	tree_const->tree.y = h;

			break;
		}


		/* If this position is less than the next available
		 * position, correct it to be the next available
		 * position, calculate the amount by which all children
		 * must be shifted, and shift the entire sub-tree.
		 */

		if ( tree_const->tree.y < current_vpos )
		{
			Dimension offset = current_vpos - tree_const->tree.y;
			for ( p = first_p; p != NULL; p = p->next )
			{
			    if ( XtIsManaged( p->child ) ||
			    			tw->tree.layout_unmanaged)
			    {
				shift_subtree( p->child, offset,
							tw->tree.orientation );
			    }
			}

			/* Adjust the next available space at all levels in
			 * the subtree below the current level.
			 */

			for ( i = level + 1; i <= max_subtree_depth ; i++ )
			{
				pos = current_value(tw->tree.vertical_pos, i);
				set_current_value(tw->tree.vertical_pos, i,
					(Dimension)pos+offset );
			}
			tree_const->tree.y = current_vpos;
		}
	}

	if ( tw->tree.connect_style == PATHWAY_CONNECTION )
	{
		/* If space requirements exceed current spacing parameters
		 * automatically increase to support routing lines
		 */

	   	compute_relative_LRpositions( tw, w, level);

		space_req = (tw->tree.path_pad * 
				(tree_const->tree.maxNumParallelRouteLines) );

		/* We need to use the max width on this level to ensure we
		 * don't bump into any siblings with the routing lines. This
		 * could be optimized a bit more as it really only needs to
		 * be the max width of any node that we would encounter while
		 * routing(a more expensive calculation).
		 */

		if ( tw->tree.node_size == FIXED_SIZE_FOR_TREE )
		    max_width_on_level = tw->tree.max_node_width;
		else
		    max_width_on_level =current_value(tw->tree.max_width,level);

		if ( max_width_on_level + space_req > hpos  )
			hpos = max_width_on_level + space_req;
	}

	if ( hpos > current_hpos && level > 0 ) /* Ignore psuedo root */
		set_current_value( tw->tree.horizontal_pos, level, 
						(Dimension) hpos );

	/* record the current vertical position at this level */

	set_current_value( tw->tree.vertical_pos, level,
			      tw->tree.min_y_pad +
			      tree_const->tree.y + height );

	return( max_subtree_depth );
}

/* --------------------  compute_TtoB_positions  ------------------------ */

static
int compute_TtoB_positions( tw, w, level )
XpsiTreeWidget   tw;
Widget		w;
int		level;
{
	Position  	 current_hpos, current_vpos, vpos, pos;
	Position	 space_req;
	Position  	 width=0, height=0;
	int	  	 i, depth = 0;
	int		 max_subtree_depth=0;
	TreeConstraints	 tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;
	ChildPtr	 *first_p;
	ChildPtr	 *last_p;
	int		  max_height_on_level;


	if ( level > tw->tree.depth ) /* Set the depth of tree */
		tw->tree.depth = level;

	/* Get the current positions for this level  */

	current_hpos = current_value( tw->tree.horizontal_pos, level );
	current_vpos = current_value( tw->tree.vertical_pos,   level );

	switch ( tw->tree.node_size )
	{
		case VARIABLE_SIZE:
			width  = w->core.width;
			height = w->core.height;
			break;

		case FIXED_SIZE_FOR_TREE:
			width  = tw->tree.max_node_width;
			height = tw->tree.max_node_height;
			break;

		case FIXED_SIZE_FOR_LEVEL:
			width   = current_value( tw->tree.max_width, level );
			height  = current_value( tw->tree.max_height, level );
			break;
	}

	/* Set the current vertical height to the max height of
	 * all widgets at this level. 
	 */

	vpos = MAX( current_vpos, height );


	if ( tree_const->tree.layout_cnt == 0 )  /* No visible children */
	{
		tree_const->tree.x = current_hpos;
		max_subtree_depth = level;
	}
	else
	{
		Widget		  first_kid, last_kid;
		TreeConstraints   const1, const2;
		Position	  left, right;

		/* If the node has children, recursively figure the
		 * positions of each child.
		 */

		first_p = last_p = NULL;
		for ( p = tree_const->tree.children ; p != NULL; p = p->next )
		{
		    if ( XtIsManaged( p->child ) || tw->tree.layout_unmanaged )
		    {
		        if ( first_p == NULL )
			     first_p = p;

			depth = compute_TtoB_positions(tw,p->child,level + 1);

			max_subtree_depth = MAX( max_subtree_depth, depth );

			last_p = p;
		    }
		}

		/* Now that the horizontal positions of all children are 
		 * known, find the horizontal extent of all sub_trees.
		 */

		first_kid = first_p->child;
		last_kid  = last_p->child;

		const1 = TREE_CONSTRAINT(first_kid);
		const2 = TREE_CONSTRAINT(last_kid);

		left    = const1->tree.x + first_kid->core.width / 2;
		right   = const2->tree.x + last_kid->core.width  / 2;

		switch ( tw->tree.parents_place )
		{
		    case PARENT_CENTERED: tree_const->tree.x = 
		    				(left + right)/2 - (width/2);
				break;

		    case PARENT_ABOVE_FIRST_CHILD: tree_const->tree.x = 
		    		const1->tree.x + (first_kid->core.width/2) - 
							(width/2);
				break;

		    case PARENT_ABOVE_LAST_CHILD: tree_const->tree.x = 
		    		const2->tree.x + (last_kid->core.width/2) - 
							    (width/2);
				break;
		}


		/* If this position is less than the next available
		 * position, correct it to be the next available
		 * position, calculate the amount by which all children
		 * must be shifted, and shift the entire sub-tree
		 */

		if ( tree_const->tree.x < current_hpos )
		{
			Dimension offset = current_hpos - tree_const->tree.x;
			for ( p = first_p; p != NULL; p = p->next )
			{
			    if ( XtIsManaged( p->child ) ||
						tw->tree.layout_unmanaged )
			    {
				shift_subtree( p->child, offset,
							tw->tree.orientation );
			    }
			}

			/* Adjust the next available space at all levels below
			 * the current level
			 */

			for ( i = level + 1; i <= max_subtree_depth ; i++ )
			{
				pos = current_value(tw->tree.horizontal_pos,i);
				set_current_value(tw->tree.horizontal_pos, i,
							pos+offset );
			}

			tree_const->tree.x = current_hpos;
		}


	}

	if ( tw->tree.connect_style == PATHWAY_CONNECTION )
	{
		/* if space requirements exceed current spacing parameters
		 * automatically increase to support routing lines
		 */

	   	compute_relative_TBpositions( tw, w, level);

		if ( tw->tree.node_size == FIXED_SIZE_FOR_TREE )
		  max_height_on_level = tw->tree.max_node_height;
		else
		  max_height_on_level =current_value(tw->tree.max_height,level);


		space_req = (tw->tree.path_pad * 
				(tree_const->tree.maxNumParallelRouteLines) );

		if ( max_height_on_level + space_req > vpos  )
			vpos = max_height_on_level + space_req;
	}

	if ( vpos > current_vpos && level > 0 ) /* Ignore psuedo root */
		set_current_value( tw->tree.vertical_pos, level, 
						(Dimension)vpos );

	/* record the current horizontal position at this level */

	set_current_value( tw->tree.horizontal_pos, level,
			      tw->tree.min_x_pad +
			      tree_const->tree.x + width );


	return( max_subtree_depth );
}

/* --------------------  compute_node_diameters  ------------------------ */

static
void compute_node_diameters( tw, w, level )
XpsiTreeWidget   tw;
Widget		w;
int		level;
{
	Position  	 current_diameter;
	Position	 diameter;
	double		 radius;
	TreeConstraints	 tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;
	int		 num_nodes_in_ring;

	/* Set the depth of tree */

	if ( level > tw->tree.depth )
		tw->tree.depth = level;

	if ( tw->tree.node_size != VARIABLE_SIZE )
	{
		if ( (int) w->core.height > tw->tree.max_node_height )
			tw->tree.max_node_height = w->core.height;
		if ( (int) w->core.width > tw->tree.max_node_width )
			tw->tree.max_node_width = w->core.width;
	}

	current_diameter  = current_value( tw->tree.vertical_pos, level );
	num_nodes_in_ring = current_value( tw->tree.breadth,  level ) + 1;


	radius = hypot( (((double) w->core.width ) / 2.0 ) , 
			        (((double) w->core.height) / 2.0 ));

	diameter = 2.0 * radius;

	/* Store actual radius now , changed later in compute_node_shapes */

	tree_const->tree.radius = radius;  

	/* Set the current diameter to the max diameter of all widgets 
	 * at this level 
	 */

	set_current_value( tw->tree.vertical_pos, level, 
			(Dimension) MAX(current_diameter, diameter) );

	/* Increment count of the number of nodes on this level  */

	set_current_value( tw->tree.breadth,      level, 
						(Dimension)num_nodes_in_ring );
	set_current_value( tw->tree.calc_breadth, level, 
						(Dimension)num_nodes_in_ring);

	for ( p = tree_const->tree.children; p != NULL; p = p->next )
	{
		compute_node_diameters( tw, p->child, level + 1);
	}
}

/* --------------------  compute_node_extents  ------------------------ */

static
void compute_node_extents( tw, w, level )
XpsiTreeWidget   tw;
Widget		w;
int		level;
{
	Position  	 current_hpos;
	Position  	 current_vpos;
	Position  	 max_width, max_height;
	TreeConstraints	 tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;
	int		 num_layout_kids;

	if ( level > 0 )
	{
		max_width    = current_value( tw->tree.max_width, level );
		max_height   = current_value( tw->tree.max_height, level );

		/* Set level maximums */

		set_current_value( tw->tree.max_width, level, 
					MAX(max_width, w->core.width) );
		set_current_value( tw->tree.max_height, level, 
					MAX(max_height, w->core.height) );


		/* Set tree maximums */

		if ( tw->tree.node_size != VARIABLE_SIZE )
		{
			if ( (int) w->core.height > tw->tree.max_node_height )
				tw->tree.max_node_height = w->core.height;
			if ( (int) w->core.width > tw->tree.max_node_width )
				tw->tree.max_node_width = w->core.width;
		}

		switch ( tw->tree.orientation )
		{

		    case TOP_TO_BOTTOM :
		    case BOTTOM_TO_TOP :
		 	current_vpos = current_value( tw->tree.vertical_pos, 
									level );
		 	set_current_value( tw->tree.vertical_pos, level, 
					MAX(current_vpos, w->core.height) );
			break;

		    case LEFT_TO_RIGHT :
		    case RIGHT_TO_LEFT :

		        current_hpos = current_value( tw->tree.horizontal_pos, 
									level );
		 	set_current_value( tw->tree.horizontal_pos, level, 
					MAX(current_hpos, w->core.width) );
			break;

		    default:
		        current_hpos = current_value( tw->tree.horizontal_pos, 
									level );
		 	set_current_value( tw->tree.horizontal_pos, level, 
					MAX(current_hpos, w->core.width) );
			break;
		}
	}

	num_layout_kids = 0;
	for ( p = tree_const->tree.children; p != NULL; p = p->next )
	{
		if ( XtIsManaged( p->child ) || tw->tree.layout_unmanaged )
		{
			compute_node_extents( tw, p->child, level + 1);
			num_layout_kids += 1;
		}
	}

	tree_const->tree.layout_cnt = num_layout_kids;
}

/* -----------------------  calculate_tree_rings  --------------------------- */

static
void calculate_tree_rings( tw )
XpsiTreeWidget   tw;
{
	Position  	diameter, node_space, ring_width, min_ring_width;
	double		inner_radius, center_radius, outer_radius;
	double		new_center_radius, opposite, rad;
	double		angular_rotation, total_angular_rotation;
	double		desired_angular_rotation;
	int 	  	level, stop, num_nodes;

	node_space = tw->tree.min_y_pad;

	/* Note level 0 is the psuedo root node and is ignored, level 1
	 * is the root widget and we only need to set the ring radius to
	 * the widgets radius + node_spacing. For all the remaining levels 
	 * we need to use the nodes diameters + node_spacing to determine 
	 * the outer ring radius. Once we have that we test to see if it's
	 * sufficient to hold all the nodes on that level, if not we increase
	 * the ring width to accomodate the nodes.
	 */

	set_current_value( tw->tree.ring_width, 0, 0 );

	if ( tw->tree.node_size == FIXED_SIZE_FOR_TREE )
	{
	      diameter = 2.0 * hypot((((double)tw->tree.max_node_width)/2.0),
			              (((double)tw->tree.max_node_height)/2.0));
	}
	else 
	{
	      diameter =  current_value( tw->tree.vertical_pos, 1 );
	}

	outer_radius = ((double)diameter)/2.0 + node_space;
	ring_width   = outer_radius;
	set_current_value( tw->tree.ring_width, 1, (Dimension)ring_width );

	stop = tw->tree.depth;
	for ( level = 2; level <= stop ; level++ )
	{
	    inner_radius = outer_radius;

	    if ( tw->tree.node_size == FIXED_SIZE_FOR_TREE )
	    {
	        diameter = 2.0 * hypot((((double)tw->tree.max_node_width)/2.0),
			              (((double)tw->tree.max_node_height)/2.0));
	    }
	    else 
	    {
		diameter =  current_value( tw->tree.vertical_pos, level );
	    }

	    min_ring_width  = node_space + diameter;
	    ring_width      = min_ring_width;
	    outer_radius    = inner_radius +  (1.0 * min_ring_width);
	    center_radius   = inner_radius + ((1.0 * min_ring_width)/2.0);

	    num_nodes = current_value( tw->tree.calc_breadth,  level );

	    /* Calculate the amount of rotation required to show
	     * all the nodes at this level. Assume all nodes are same
	     * size, i.e. the max size.
	     */

	    opposite     = ((int)diameter + (int)tw->tree.min_x_pad) * 1.0;
	    rad                    = atan2( opposite, center_radius );
	    angular_rotation       = DEGREES( rad );
	    total_angular_rotation = angular_rotation * num_nodes;

DEBUG3(4, "  angular_rot(%lg) * num_nodes(%d) = total(%lg)\n", angular_rotation,
			num_nodes, total_angular_rotation );

	    if ( total_angular_rotation >= 360.0 )
	    {
	    	/* We need to increase the ring width so that all
	    	 * of the nodes would fit when centered in the new
	    	 * ring width. Note this is a crude
	    	 * estimate since we can't determine the subtree
	    	 * requirements yet. The goal here is to cut down
	    	 * on the amount of ring expansion and sub-tree
	    	 * shifting, both of these are very expensive compared
	    	 * to this approximation.
	    	 *
	    	 */


	    	desired_angular_rotation = 359.0 / ((double)num_nodes);

	    	rad = RADIANS( desired_angular_rotation );

	    	new_center_radius = min_ring_width / tan( rad );

	    	ring_width = (new_center_radius - inner_radius) * 2; 

		if ( ring_width < min_ring_width )
			ring_width = min_ring_width;

	    	outer_radius = inner_radius + ring_width;

DEBUG4(1,"  Expand ring(%d) desired_angular_rot(%lg) new_center_radius(%lg) ring_width(%d)\n", level, desired_angular_rotation, new_center_radius, ring_width );

	    }

	    set_current_value(tw->tree.ring_width, level,(Dimension)ring_width);
	}
}

/* -----------------------  compute_tree_rings  --------------------------- */

static
void compute_tree_rings( tw )
XpsiTreeWidget   tw;
{
	Position  		cx, cy, x, y, width;
	double			radius;
	int      		stop;
	int 	  		level;
	TreeConstraints		tree_const;
	TreeConstraintPart      *ct;
	Widget			w;
	XArc			*ap;


	/* If the tree widget is visible and has a root node, compute
	 * rings
	 */

	tw->tree.num_rings = 0;

	if ( tw->core.visible && tw->tree.tree_root && tw->tree.show_rings )
	{
		/* Get the pointer to the psuedo root tree constraint 
		 * record, it contains the pointer to the root widget of
		 * the tree.
		 */
		tree_const = TREE_CONSTRAINT(tw->tree.tree_root);
		ct = &(tree_const->tree);

		w = tw->tree.tree_root;
		cx = w->core.x;
		cy = w->core.y;
		if ( ct->num_children > 0 )
		{
			w = ct->children->child;
			stop = tw->tree.depth;
			for ( level = 1; level <= stop ; level++ )
			{
				radius= 1.0 * sum_table(
							tw->tree.ring_width,
								level + 1);
				x = cx - radius;
				y = cy - radius;
				width = radius * 2;
				if ( level == 1 )
					width -= 1;

				ap = &(tw->tree.rings[ tw->tree.num_rings ]);
				ap->x      = x;
				ap->y      = y;
				ap->width  = width;
				ap->height = width;
				ap->angle1 = 0;
				ap->angle2 = (360 * 64);
				tw->tree.num_rings += 1;
			}
		}
	}
}

static
void compute_node_shapes( tw, w, level )
XpsiTreeWidget   tw;
Widget		w;
int		level;
{
	double	 	 radius;
	TreeConstraints	 tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;
	Position	  cx, cy, x, y, ox;
	XArc		 *ap;
	XRectangle	 *rp;

	if ( !XtIsManaged( w ) )   
		return;

	switch ( tw->tree.orientation )
	{
		case STAR_TOPOLOGY:

		    if ( tree_const->tree.drawShape ) /*user drawn*/
		    {
			/* XXX Add code Here to call users function which
			 * will return drawing primitives.
			 */

			break;
		    }

		    radius = tree_const->tree.radius;

		    cx = w->core.x + w->core.width/2;
		    cy = w->core.y + w->core.height/2;
		    x = cx - radius;
		    y = cy - radius;
    
		    ap = &(tw->tree.arcs[ tw->tree.num_arcs ]);
    
		    ap->x      = x;
		    ap->y      = y;
		    ap->width  = radius * 2;
		    ap->height = radius * 2;
		    ap->angle1 = 0;
		    ap->angle2 = (360 * 64);
    

		    tw->tree.fillColor[tw->tree.num_arcs] = 
		    			tree_const->tree.backgroundFill;

	    	    tree_const->tree.shapeIndex = tw->tree.num_arcs;

		    tw->tree.num_arcs += 1;
    
		    break;
    
		default:

		    if ( tree_const->tree.drawShape ) /*user drawn*/
		    {
			/* XXX Add code Here to call users function which
			 * will return drawing primitives. For now just
			 * ignore this node.
			 */
			break;
		    }

			rp = &(tw->tree.rectangles[ tw->tree.num_rectangles ]);

		        tree_const->tree.radius = (w->core.width/2) + 1;

			rp->x      = w->core.x - 1;
			rp->y      = w->core.y - 1;

		    	ox = current_value( tw->tree.max_width, level);

			rp->width  = w->core.width  + 1;
			rp->height = w->core.height + 1;

		    	if ( tw->tree.node_size == FIXED_SIZE_FOR_LEVEL )
		    	{
				rp->width  = ox + 1;
		    	}
		    	else if ( tw->tree.node_size == FIXED_SIZE_FOR_TREE )
			{
				rp->width  = tw->tree.max_node_width  + 1;
				rp->height = tw->tree.max_node_height + 1;
			}

		        tw->tree.fillColor[tw->tree.num_arcs] = 
		    			tree_const->tree.backgroundFill;

	    	        tree_const->tree.shapeIndex = tw->tree.num_rectangles;

		        tw->tree.fillColor[tw->tree.num_rectangles] = 
		    			tree_const->tree.backgroundFill;

	    	        tree_const->tree.shapeIndex = tw->tree.num_rectangles;

			tw->tree.num_rectangles += 1;
			break;
	}

	for ( p = tree_const->tree.children; p != NULL; p = p->next )
	{
		compute_node_shapes( tw, p->child, level + 1);
	}

}

/* --------------------  compute_Star_positions  ------------------------ */

static
int compute_Star_positions( tw, w, level )
XpsiTreeWidget   tw;
Widget		w;
int		level;
{
	Position  	 current_theta, pos;
	Position	 diameter, ring_width;
	int	  	 i, depth = 0;
	TreeConstraints	 tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;
	ChildPtr	 *first_p;
	ChildPtr	 *last_p;
	double	 	  adjacent, opposite, radius, rad, rotation_angle;
	double		  default_radius, default_rotation=0.0;
	double		  new_theta, adjust_theta;
	int		  node_space;
	int		  cur_pos; /* Node Position at this level */
	int		  breadth, nodes_left;

	/* Depth of tree set in compute_node_diameters */

	/* Get the current theta for this level, note this position is
	 * the angle of the center of a node of default size. The actual
	 * size is calculated and then the position is adjusted accordingly
	 */

	current_theta = current_value( tw->tree.horizontal_pos, level );

	/* Note max diameter for this level is stored in tw->tree.vertical_pos 
	 * and was set in compute_node_diameter 
	 */

	cur_pos = current_value( tw->tree.cur_breadth, level) + 1;
	set_current_value( tw->tree.cur_breadth, level, (Dimension)cur_pos );

	/* Determine the radius and diameter to use for the node */

	diameter         =  2.0 + current_value( tw->tree.vertical_pos, level );
	default_radius   =  ((double) diameter) / 2.0;
	if ( tw->tree.node_size == FIXED_SIZE_FOR_LEVEL )
	{
		diameter = current_value( tw->tree.vertical_pos, level );
		radius   = 1 + (((double) diameter) / 2.0);
	}
	else if ( tw->tree.node_size == FIXED_SIZE_FOR_TREE )
	{
	        radius = 1 + hypot((((double)tw->tree.max_node_width)/2.0),
			         (((double)tw->tree.max_node_height)/2.0));
		default_radius   = radius;
	}
	else
	{
	        radius = 1 + hypot( (((double) w->core.width ) / 2.0 ), 
				        (((double) w->core.height) / 2.0 ));
	}

	/* Store node radius for use later */

	tree_const->tree.radius = radius;


	node_space =  tw->tree.min_x_pad;
	opposite   =  (radius + (((double)node_space) * 0.5));
	ring_width =  current_value(tw->tree.ring_width, level);

	/* The following centers the node in the ring */

	adjacent   = ((double) sum_table(tw->tree.ring_width, level)) +
						((double) ring_width/2.0);
			
	rad            = atan2( opposite, adjacent );
	rotation_angle = DEGREES( rad );

	if ( default_radius != 0.0 )
	{
		opposite   =  (default_radius + (((double)node_space) * 0.5));
		rad = atan2( opposite, adjacent );
		default_rotation = DEGREES( rad );
	}

	/* If the node has no children, just set the angle
	 * to the current angle - (half the default node size angle - half
	 * the calculated node size). 
	 */
	
	adjust_theta = default_rotation - rotation_angle;

DEBUG3(1, "default_rotation(%lg) calulated_rotation_angle(%lg)  adjust(%lg)\n",
			default_rotation, rotation_angle, adjust_theta );

	if ( current_theta > 0 )
		current_theta = current_theta - adjust_theta;

DEBUG1(1, "Current_theta(%d)\n", current_theta );

	if ( tree_const->tree.num_children == 0 )
	{
		tree_const->tree.x = current_theta;
	}
	else
	{
		Widget		  first_kid, last_kid;
		TreeConstraints   const1, const2;
		Position	  left, right;

		/* If the node has children, recursively figure the
		 * positions of each child.
		 */

		first_p = last_p = tree_const->tree.children;
		for ( p = first_p; p != NULL; p = p->next )
		{
			depth = compute_Star_positions(tw,p->child,level + 1);

			if ( depth < 0 )	       /* Expanding tree */
				return( depth );
			last_p = p;
		}

		/* Now that the radial positions of all children are 
		 * known, find the angular extent of all sub_trees.
		 */

		first_kid = first_p->child;
		last_kid  = last_p->child;

		const1 = TREE_CONSTRAINT(first_kid);
		const2 = TREE_CONSTRAINT(last_kid);

		left    = const1->tree.x; 
		right   = const2->tree.x;

		switch ( tw->tree.parents_place )
		{
			case PARENT_CENTERED:
				tree_const->tree.x = (left + right)/2;
				break;

			case PARENT_ABOVE_FIRST_CHILD:
				tree_const->tree.x = left;
				break;

			case PARENT_ABOVE_LAST_CHILD:
				if ( level > 1 )
					tree_const->tree.x = right;
				else
					tree_const->tree.x = left;

				break;
		}


		/* If this position is less than the next available
		 * position, correct it to be the next available
		 * position, calculate the amount by which all children
		 * must be shifted, and shift the entire sub-tree
		 */

		if ( tree_const->tree.x < current_theta )
		{
			Dimension offset;

			offset = current_theta - tree_const->tree.x;
			for ( p = first_p; p != NULL; p = p->next )
			{
				shift_subtree( p->child, offset,
							tw->tree.orientation );
			}

			/* Adjust the next available space at all levels below
			 * the current level
			 */

			for ( i = level + 1; i <= depth ; i++ )
			{
				pos = current_value(tw->tree.horizontal_pos, i);
				set_current_value(tw->tree.horizontal_pos, i,
							pos+offset );
			}

			tree_const->tree.x = current_theta;
		}
	}

	/* record the current angle at this level */

	/* Increase the angle by enough to clear the newly positioned
	 * subtree. If the position of the current node would cause it to
	 * run into the first node, i.e. it's > 360.0 degrees then 
	 * we increase the calculated breadth by the number
	 * of remaining unprocessed nodes on this level and immediately
	 * return -1. This forces the recursion to immediately bubble up to
	 * the top where the ring width's are recalculated, and the layout
	 * starts again.
	 */

	new_theta      = ((double)tree_const->tree.x) + rotation_angle;

DEBUG2(1, "tree.x(%d) rotation_angle(%lg)\n", tree_const->tree.x, rotation_angle);

	/* If we have wrapped arround we need to increase ring width */

	if ( new_theta > ( 360.0 - default_rotation) )
	{
		breadth    = current_value( tw->tree.breadth, level);
		nodes_left = (breadth - cur_pos) + 1;

DEBUG4(1, "Star layout level[%d] node(%d) ang(%lg) >360.0 nodes_left(%d)\n", level,
		cur_pos, new_theta, nodes_left );

		breadth = current_value( tw->tree.calc_breadth, level);
		set_current_value( tw->tree.calc_breadth, level, 
					(Dimension)  breadth + nodes_left );

DEBUG1(1, "	New calculated breadth(%d)\n", breadth + nodes_left);

		return( -1 );
	}

	set_current_value( tw->tree.horizontal_pos, level, 
					(Dimension) tree_const->tree.x + 
					((int)(default_rotation * 2.0)) );

	return( MAX(depth, level) );
}

/* ------------------------  compute_connections --------------------------- */

static
void compute_connections( tw, parent, level )
XpsiTreeWidget  tw;
Widget	    parent;
int	    level;
{
	TreeConstraints		tree_const;
	TreeConstraintPart      *ct;
	ChildPtr                *p;
	ChildPtr                *first_p;
	ChildPtr                *first_vis_p=NULL;
	ChildPtr                *last_vis_p=NULL;
	int			child_num;

	/* Set pointer to the tree part of the constraint record.  */

	tree_const = TREE_CONSTRAINT(parent);
	ct = &(tree_const->tree);

	/* Depending on the connection style, draw lines to connect the nodes */

	if ( ct->num_children )
	{
	   child_num = 1;
	   first_p = ct->children;
	   for ( p = first_p; p != NULL; p = p->next )
	   {

		if ( !XtIsManaged( p->child ) )   /* Ignore hidden branch */
		{
			/* Could indicate that there is a hidden branch
			 * by drawing a stub. Perhaps a future enhancement
			 */

			continue;
		}
		else	/* Keep track of first and last visible pointers */
		{
			/* Used later to connect children on either end of
			 * a tier bar
			 */

			last_vis_p = p;
			if ( first_vis_p == NULL )
				first_vis_p = p;
		}

		compute_connections( tw, p->child, level + 1 );

		switch( tw->tree.connect_style )
		{
			case DIAGONAL_CONNECTION:
				diagonal_connect( tw, parent, p->child, level );
				break;

			case TIER_CONNECTION:
				tier_connect( tw, parent, p->child, level );
				break;

			case PATHWAY_CONNECTION:
				pathway_connect( tw, parent, p->child, level,
						 child_num++ );
				break;

			case NO_CONNECTION:
				break;
		}
	   }

	   if ( tw->tree.connect_style == TIER_CONNECTION )
	   {
		if ( last_vis_p ) 
		{
			draw_tier_bar(tw, parent, first_vis_p->child, 
						last_vis_p->child, level);
		}
	    }
	}
}


/* --------------------------  diagonal_connect ----------------------------- */

#define SET_SEGMENT(w, p,a1,b1,a2,b2) \
p->x1 = a1; \
p->y1 = b1; \
p->x2 = a2; \
p->y2 = b2; \
w->tree.num_connect_lines += 1;




static
void diagonal_connect( w, parent, child, level )
XpsiTreeWidget  w;
Widget	parent;
Widget  child;
int	level;
{
	XSegment	*seg;
	Position	 cx1, cy1, cx2, cy2;
	TreeConstraints	 tree_const;
	double		 opposite, adjacent, rad;
	Position	 width, height;

	seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);

	switch ( w->tree.orientation )
	{
	   case LEFT_TO_RIGHT:

		width = parent->core.width;
		if ( 	w->tree.node_shape == True )
		{
			if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
			{
		    	    width = current_value(w->tree.max_width,level) + 1;
			}
			else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
			{
		    	    width = w->tree.max_node_width + 1;
			}
			else
			    width += 1;
		}

/* XXX Note should use width as determined from w->tree.node_size */

	   	seg->x1 = parent->core.x   + width;
		seg->y1 = parent->core.y   + parent->core.height/2;

		seg->x2 = child->core.x    - 1;
		seg->y2 = child->core.y    + child->core.height/2;
		break;

	   case RIGHT_TO_LEFT:
		width = child->core.width;
		if ( w->tree.node_shape == True )
		{
			if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
			{
		    	    width = current_value(w->tree.max_width,level+1)+1;
			}
			else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
			{
		    	    width = w->tree.max_node_width + 1;
			}
			else
				width += 1;
		}

/* XXX Note should use width as determined from w->tree.node_size */

		seg->x1 = parent->core.x  - 1;
		seg->y1 = parent->core.y  + parent->core.height/2;

		seg->x2 = child->core.x   + width;
		seg->y2 = child->core.y   + child->core.height/2;
		break;

	   case TOP_TO_BOTTOM:
		height = parent->core.height + 1;
		if ( w->tree.node_shape == True )
		{
			if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
			{
		    	    height = current_value(w->tree.max_height,level)+1;
			}
			else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
			{
		    	    height = w->tree.max_node_height + 1;
			}
		}

/* XXX Note should use height as determined from w->tree.node_size */

		seg->x1 = parent->core.x  + parent->core.width/2;
		seg->y1 = parent->core.y  + height;
		seg->x2 = child->core.x   + child->core.width/2;
		seg->y2 = child->core.y   - 1;
		break;

	   case BOTTOM_TO_TOP:
		height = child->core.height + 1;
		if ( w->tree.node_shape == True )
		{
			if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
			{
		    	    height =current_value(w->tree.max_height,level+1)+1;
			}
			else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
			{
		    	    height = w->tree.max_node_height + 1;
			}
		}

/* XXX Note should use height as determined from w->tree.node_size */

		seg->x1 = parent->core.x    + parent->core.width/2;
		seg->y1 = parent->core.y    - 1;
		seg->x2 = child->core.x     + child->core.width/2;
		seg->y2 = child->core.y     + height;
		break;

	   case STAR_TOPOLOGY:

			 /* Determine the angle between the center
			  * of the parent and the center of the child.
			  * calculate the intersection of the
			  * direction vector and the widgets 
			  * surrounding circle/oval
			  */

			/* Center of widgets */
			cx1 = parent->core.x + (parent->core.width/2);
			cy1 = parent->core.y + (parent->core.height/2);
			cx2 = child->core.x + (child->core.width/2);
			cy2 = child->core.y + (child->core.height/2);

	    		opposite    =  cy1 - cy2;
			adjacent    =  cx2 - cx1;
	    		rad         = atan2( opposite, adjacent );

			tree_const = TREE_CONSTRAINT(parent);
			seg->x1 = cx1 + (tree_const->tree.radius * cos( rad ));
			seg->y1 = cy1 - (tree_const->tree.radius * sin( rad ));

	    		opposite    =  cy2 - cy1;
			adjacent    =  cx1 - cx2;
	    		rad         = atan2( opposite, adjacent );

			tree_const = TREE_CONSTRAINT(child);
			seg->x2 = cx2 + (tree_const->tree.radius * cos( rad ));
			seg->y2 = cy2 - (tree_const->tree.radius * sin( rad ));

		break;
	}

	w->tree.num_connect_lines += 1;
}

/* --------------------------  tier_connect ----------------------------- */

static
void tier_connect( w, parent, child, level )
XpsiTreeWidget  w;
Widget	parent;
Widget  child;
int	level;
{
	Position	tier_pos;
	XSegment	*seg;
	Position	width, height;

	seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);

	/* Draw connection from the tier_bar(see next function) to the
	 * child widget.
	 */

	tier_pos = tier_bar_pos( w, parent, child, level );


	switch ( w->tree.orientation )
	{
	   case LEFT_TO_RIGHT:

/* XXX Note should use height as determined from w->tree.node_size */

			    seg->x1 = tier_pos;
			    seg->y1 = child->core.y     + child->core.height/2;
			    seg->x2 = child->core.x - 1;
			    seg->y2 = child->core.y     + child->core.height/2;
		break;

	   case RIGHT_TO_LEFT:
		width = child->core.width;
		if ( w->tree.node_shape == True )
		{
			if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
			{
		    	    width = current_value(w->tree.max_width,level+1)+1;
			}
			else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
			{
		    	    width = w->tree.max_node_width + 1;
			}
			else
			    width += 1;
		}

/* XXX Note should use height as determined from w->tree.node_size */

		seg->x1 = tier_pos;
		seg->y1 = child->core.y     + child->core.height/2;
		seg->x2 = child->core.x     + width;
		seg->y2 = child->core.y     + child->core.height/2;
		break;

	   case TOP_TO_BOTTOM:

/* XXX Note should use width as determined from w->tree.node_size */

			    seg->x1 = child->core.x     + child->core.width/2;
			    seg->y1 = tier_pos;
			    seg->x2 = child->core.x     + child->core.width/2;
			    seg->y2 = child->core.y - 1;
		break;

	   case BOTTOM_TO_TOP:
		height = child->core.height;
		if ( 	w->tree.node_shape == True )
		{
		    if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
		    {
		       height = current_value(w->tree.max_height,level+1)+1;
		    }
		    else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
		    {
		       height = w->tree.max_node_height + 1;
		    }
		    else
		        height += 1;
		}

/* XXX Note should use width as determined from w->tree.node_size */

		seg->x1 = child->core.x     + child->core.width/2;
		seg->y1 = tier_pos;
		seg->x2 = child->core.x     + child->core.width/2;
		seg->y2 = child->core.y     + height;
		break;

	   case STAR_TOPOLOGY:

			 /* Determine the angle between the center
			  * of the parent and the center of the child.
			  * calculate the intersection of the
			  * direction vector and the widgets 
			  * surrounding circle/oval
			  */

			/* Simple center to center */

			    seg->x1 = parent->core.x    + parent->core.width/2;
			    seg->y1 = parent->core.y    + parent->core.height/2;
			    seg->x2 = child->core.x     + child->core.width/2;
			    seg->y2 = child->core.y     + child->core.height/2;
		break;
	}

	w->tree.num_connect_lines += 1;
}

/* ----------------------------  tier_bar ------------------------------ */

static
void draw_tier_bar( w, parent, first, last, level )
XpsiTreeWidget  w;
Widget	parent;
Widget  first;
Widget  last;
int    level;
{
	Position	tier_pos;
	Position	parent_pos_x;
	Position	parent_pos_y;
	XSegment	*seg;
	Position	width, height;

	seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);

	tier_pos = tier_bar_pos( w, parent, first, level );

	switch ( w->tree.orientation )
	{
	   case LEFT_TO_RIGHT: 
	   case RIGHT_TO_LEFT:
		width = parent->core.width;
		if ( 	w->tree.node_shape == True )
		{
		    if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
		    {
		       width = current_value(w->tree.max_width,level)+1;
		    }
		    else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
		    {
		       width = w->tree.max_node_width + 1;
		    }
		    else
		        width += 1;
		}

		parent_pos_x = parent->core.x;
		if ( w->tree.orientation == LEFT_TO_RIGHT )
			    parent_pos_x += width;
	
		parent_pos_y = parent->core.y + 
					parent->core.height/2;

		if ( first != last )	/* Draw the vertical bar */
		{

/* XXX Note should use height as determined from w->tree.node_size */

		    seg->x1 = tier_pos;
		    seg->y1 = first->core.y     + first->core.height/2;
		    seg->x2 = tier_pos;
		    seg->y2 = parent_pos_y;

		    w->tree.num_connect_lines += 1;
		    seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);

		    seg->x1 = tier_pos;
		    seg->y1 = parent_pos_y;
		    seg->x2 = tier_pos;
		    seg->y2 = last->core.y      + last->core.height/2;

		    w->tree.num_connect_lines += 1;
		    seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);
		}


		if ( w->tree.parents_place == PARENT_CENTERED )
		{
			seg->x1 = parent_pos_x;
			seg->y1 = parent_pos_y;
			seg->x2 = tier_pos;     
			seg->y2 = parent_pos_y;
		}
		else if ( w->tree.parents_place == PARENT_ABOVE_FIRST_CHILD )
		{
			seg->x1 = parent_pos_x;
			seg->y1 = parent_pos_y;
			seg->x2 = tier_pos;     
			seg->y2 = parent_pos_y;
			/*
	   		seg->x1 = parent->core.x   + width/2;
			seg->y1 = parent->core.y   + parent->core.height;
	   		seg->x2 = seg->x1;
		    	seg->y2 = first->core.y     + first->core.height/2;
			*/

		        w->tree.num_connect_lines += 1;
		        seg=&(w->tree.connect_lines[w->tree.num_connect_lines]);

			seg->x1 = tier_pos;     
			seg->y1 = parent_pos_y;
			seg->x2 = tier_pos;     
		    	seg->y2 = first->core.y     + first->core.height/2;

			/*
	   		seg->x1 = parent->core.x   + width/2;
		    	seg->y1 = first->core.y     + first->core.height/2;
			seg->x2 = tier_pos;     
		    	seg->y2 = seg->y1;
			*/
		}
		else
		{
			seg->x1 = parent_pos_x;
			seg->y1 = parent_pos_y;
			seg->x2 = tier_pos;     
			seg->y2 = parent_pos_y;
			/*
	   		seg->x1 = parent->core.x   + width/2;
			seg->y1 = parent->core.y;
			seg->x2 = seg->x1;
		        seg->y2 = last->core.y      + last->core.height/2;
			*/

		        w->tree.num_connect_lines += 1;
		        seg=&(w->tree.connect_lines[w->tree.num_connect_lines]);

			seg->x1 = tier_pos;     
			seg->y1 = parent_pos_y;
			seg->x2 = tier_pos;     
		    	seg->y2 = last->core.y     + last->core.height/2;
			/*
	   		seg->x1 = parent->core.x   + width/2;
		    	seg->y1 = last->core.y     + last->core.height/2;
			seg->x2 = tier_pos;     
		    	seg->y2 = seg->y1;
			*/
		}

		w->tree.num_connect_lines += 1;

		break;

	   case TOP_TO_BOTTOM:
		height = parent->core.height;
		if ( 	w->tree.node_shape == True )
		{
		    if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
		    {
		       height = current_value(w->tree.max_height,level)+1;
		    }
		    else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
		    {
		       height = w->tree.max_node_height + 1;
		    }
		    else
		        height += 1;
		}

			    /* ----  Fall Through ---- */

	   case /*FALLTHRU*/ BOTTOM_TO_TOP:

		parent_pos_x = parent->core.x + parent->core.width/2;

		if ( first != last )
		{

/* XXX Note should use width as determined from w->tree.node_size */

		    seg->x1 = first->core.x    + first->core.width/2;
		    seg->y1 = tier_pos;
		    seg->x2 = last->core.x     + last->core.width/2;
		    seg->y2 = tier_pos;
		    w->tree.num_connect_lines += 1;
		    seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);
		}
		else
		{
		    seg->x1 = first->core.x    + first->core.width/2;
		    seg->y1 = tier_pos;
		    seg->x2 = parent_pos_x;
		    seg->y2 = tier_pos;
		    w->tree.num_connect_lines += 1;
		    seg = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);
		}

		parent_pos_y = parent->core.y;
		if ( w->tree.orientation == TOP_TO_BOTTOM )
			parent_pos_y += height;
		
	   	
		seg->x1 = parent_pos_x;
		seg->y1 = parent_pos_y;
		seg->x2 = parent_pos_x;
		seg->y2 = tier_pos;

		w->tree.num_connect_lines += 1;

		break;


	   case STAR_TOPOLOGY:

		/* Determine the arc required for the tier bar.
		 */

		break;
	}
}

static
Position tier_bar_pos( w, parent, child, level )
XpsiTreeWidget  w;
Widget	parent;
Widget  child;
int    level;
{
	Position	pos;
	Position	cpos, ppos;

	switch ( w->tree.orientation )
	{
	   case LEFT_TO_RIGHT:  ppos = parent->core.x + 
				   current_value( w->tree.horizontal_pos,level);
				cpos = child->core.x;
				pos = ppos + (cpos - ppos)/2;
				break;

	   case RIGHT_TO_LEFT:  ppos = parent->core.x;
				cpos = child->core.x + 
				  current_value(w->tree.horizontal_pos,level+1);
				pos = cpos + (ppos - cpos)/2;
				break;

	   case TOP_TO_BOTTOM:  ppos = parent->core.y  + 
				   current_value( w->tree.vertical_pos, level);
				cpos = child->core.y;
				pos = ppos + (cpos - ppos)/2;
				break;

	   case BOTTOM_TO_TOP:  ppos = parent->core.y;
				cpos = child->core.y + 
				   current_value(w->tree.vertical_pos,level+1);
				pos = cpos + (ppos - cpos)/2;
				break;

	   case STAR_TOPOLOGY:
				pos = 0;
				break;

	}

	return( pos );
}

/* --------------------------  pathway_connect ----------------------------- */

static void
compute_relative_LRpositions( tw, parent, level )
XpsiTreeWidget  tw;
Widget	parent;
int	level;
{
	TreeConstraints		tree_const;
	TreeConstraintPart      *pcp;  /* Parent constraint pointer */
	TreeConstraintPart      *ccp;  /* Child constraint ponter */
	ChildPtr                *p;
	int			vis_posn=1;
	int			parent_mid_y;
	int			child_mid_y;
	int			max_posn=0;
	int			abs_posn=0;

	/* Determines the relative position of children based on the
	 * parent. 0 Means directly inline with parent, numbers above mean
	 * offset above parent, negative numbers are offset below.
	 *
	 * Note: There may not be any node directly inline.
	 *
	 * The numbers start at 1 and increase, so we have to do a little
	 * arithmetic later to figure the offset(better than two passes).
	 *
	 * We also keep track of the max number of parallel routing lines
	 * required so we can do some space calculations prior to drawing the
	 * pathways.
	 */

	/* Set pointer to the tree part of the constraint record.  */

	tree_const = TREE_CONSTRAINT(parent);
	pcp = &(tree_const->tree);

	parent_mid_y = pcp->y + (parent->core.height / 2);

	if ( pcp->num_children != 0 )
	{
	   for ( p = pcp->children; p != NULL; p = p->next )
	   {
		if ( !XtIsManaged( p->child ) )   /* Ignore hidden branch */
		{
		    continue;
		}
		else	
		{
		   /* Get pointer to childs constraint structure */

		   tree_const = TREE_CONSTRAINT( p->child );
		   ccp = &(tree_const->tree);

		   child_mid_y = ccp->y + (p->child->core.height / 2);

		   if ( child_mid_y < parent_mid_y )
		   {
		   	abs_posn      = vis_posn;
		        ccp->rel_posn = vis_posn++;
		   }
		   else if ( child_mid_y > parent_mid_y )
		   {
		   	if ( vis_posn > 0 )
				vis_posn = -1;

		   	abs_posn      = 0 - vis_posn;
		        ccp->rel_posn = vis_posn--;
		   }
		   else /* They are equal */
		   {
			if ( tw->tree.parents_place == PARENT_CENTERED )
			{
		   	    vis_posn      = 0;
		   	    abs_posn      = 0;
		            ccp->rel_posn = vis_posn--;
			}
			else if(tw->tree.parents_place==PARENT_ABOVE_FIRST_CHILD)
			{
		   	    abs_posn      = 1;
		            ccp->rel_posn = -1;
			    vis_posn      = -2;
			}
			else
			{
			    abs_posn      = vis_posn;
		            ccp->rel_posn = vis_posn++;
			}
		   }

		   if ( abs_posn > max_posn )
		   	max_posn = abs_posn;
		}
	   }
	   pcp->maxNumParallelRouteLines = max_posn;
	}
	else
	{
	   pcp->maxNumParallelRouteLines = 0;
	}
}

static void
compute_relative_TBpositions( tw, parent, level )
XpsiTreeWidget  tw;
Widget	parent;
int	level;
{
	TreeConstraints		tree_const;
	TreeConstraintPart      *pcp;  /* Parent constraint pointer */
	TreeConstraintPart      *ccp;  /* Child constraint ponter */
	ChildPtr                *p;
	int			vis_posn=1;
	int			parent_mid_x;
	int			child_mid_x;
	int			max_posn=0;
	int			abs_posn=0;

	/* Determines the relative position of children based on the
	 * parent. 0 Means directly inline with parent, numbers above mean
	 * offset left of parent, negative numbers are offset right.
	 *
	 * Note: There may not be any node directly inline.
	 *
	 * The numbers start at 1 and increase, so we have to do a little
	 * arithmetic later to figure the offset(better than two passes).
	 *
	 * We also keep track of the max number of parallel routing lines
	 * required so we can do some space calculations prior to positioning
	 * a parent node.
	 */

	/* Set pointer to the tree part of the constraint record.  */

	tree_const = TREE_CONSTRAINT(parent);
	pcp = &(tree_const->tree);

	parent_mid_x = pcp->x + (parent->core.width / 2);


	pcp->superior_cnt=0;
	pcp->inferior_cnt=0;
	if ( pcp->num_children != 0 )
	{
	   for ( p = pcp->children; p != NULL; p = p->next )
	   {
		if ( !XtIsManaged( p->child ) )   /* Ignore hidden branch */
		{
		    continue;
		}
		else	
		{
		   /* Get pointer to childs constraint structure */

		   tree_const = TREE_CONSTRAINT( p->child );
		   ccp = &(tree_const->tree);


		   child_mid_x = ccp->x + (p->child->core.width / 2);

		   if ( child_mid_x < parent_mid_x )
		   {
		   	abs_posn      = vis_posn;
		        ccp->rel_posn = vis_posn++;
			pcp->superior_cnt++;
		   }
		   else if ( child_mid_x > parent_mid_x )
		   {
		   	if ( vis_posn > 0 )
				vis_posn = -1;

		   	abs_posn      = 0 - vis_posn;
		        ccp->rel_posn = vis_posn--;
			pcp->inferior_cnt++;
		   }
		   else /* They are equal */
		   {
			if ( tw->tree.parents_place == PARENT_CENTERED )
			{
		   	    vis_posn      = 0;
		   	    abs_posn      = 0;
		            ccp->rel_posn = vis_posn--;
			}
			else if(tw->tree.parents_place==PARENT_ABOVE_FIRST_CHILD)
			{
		   	    abs_posn      = 1;
		            ccp->rel_posn = -1;
			    vis_posn      = -2;
			    pcp->inferior_cnt++;
			}
			else
			{
			    abs_posn      = vis_posn;
		            ccp->rel_posn = vis_posn++;
			    pcp->superior_cnt++;
			}
		   }

		   if ( abs_posn > max_posn )
		   	max_posn = abs_posn;
		}
	   }
	   pcp->maxNumParallelRouteLines = max_posn;
	}
	else
	{
	   pcp->maxNumParallelRouteLines = 0;
	}
}

static
void pathway_connect( w, parent, child, level, child_num )
XpsiTreeWidget  w;
Widget	parent;
Widget  child;
int	level;
int	child_num;
{
	Position		pt1_x, pt1_y;
	Position		pt2_x, pt2_y;
	Position		pt3_x, pt3_y;
	Position		pt4_x, pt4_y;
	Position		pt5_x, pt5_y;
	TreeConstraints		tree_const;
	TreeConstraintPart      *ct;   /* Parent constraint struct ptr */
	TreeConstraintPart      *ccp;  /* Child constraint struct ptr */
	XSegment		*seg1;
	XSegment		*seg2;
	XSegment		*seg3;
	XSegment		*seg4;
	Position		width, height, space_width;
	Position		parent_width, parent_height,parent_half_height;
	Position		child_width,  child_height, child_half_height;
	Position		child_x, child_y, parent_x, parent_y;
	int			num_kids, pad, rel_posn, mid_child, is_odd;
	int			half_Hpad, half_Vpad;
	int			num_rte;
	int			inferiors, superiors, adj;
	double			d_adj;


	seg1 = &(w->tree.connect_lines[ w->tree.num_connect_lines ]);
	seg2 = &(w->tree.connect_lines[ w->tree.num_connect_lines + 1 ]);
	seg3 = &(w->tree.connect_lines[ w->tree.num_connect_lines + 2 ]);
	seg4 = &(w->tree.connect_lines[ w->tree.num_connect_lines + 3 ]);

	/* Draw connection from the child widget to the parent using simple 
	 * line router. The 4th segment is to cover the case when the lines
	 * extend beyond the edge of the parent widget. In such cases a 
	 * diagonal line is shown connecting to the center of the 
	 * parents widget.
	 */

	tree_const = TREE_CONSTRAINT(parent);
	ct = &(tree_const->tree);

	if ( ct->num_children == 0 )	/* Oops bail out */
		return;
				
	tree_const = TREE_CONSTRAINT(child);
	ccp = &(tree_const->tree);

	child_x            = child->core.x;
	child_y            = child->core.y;
	child_width        = child->core.width;
	child_height       = child->core.height;
	child_half_height  = ((int)child_height)/2;

	parent_x           = parent->core.x;
	parent_y           = parent->core.y;
	parent_height      = parent->core.height;
	parent_width       = parent->core.width;
	parent_half_height = ((int)parent_height)/2;
	
	half_Hpad = w->tree.min_x_pad/2;
	half_Vpad = w->tree.min_y_pad/2;

	/* Determine size of nodes for layout connection purposes */



	if( w->tree.node_size == FIXED_SIZE_FOR_LEVEL )
	{
		parent_height = current_value(w->tree.max_height,level);
	   	parent_width  = current_value(w->tree.max_width, level);
		child_height  = current_value(w->tree.max_height,level+1);
	   	child_width   = current_value(w->tree.max_width, level+1);
	}
	else if ( w->tree.node_size == FIXED_SIZE_FOR_TREE )
	{
		parent_height = w->tree.max_node_height;
	        parent_width  = w->tree.max_node_width;
		child_height  = parent_height;
	        child_width   = parent_width;
	}

	if ( w->tree.node_shape == True ) /* Add space for surrounding box */
	{
		parent_height += 1;  
		parent_width  += 1;  
		child_height  += 1;  
		child_width   += 1;  
	}

	num_kids = ct->num_children;
	pad      = w->tree.path_pad;
	rel_posn = ccp->rel_posn;

	inferiors = ct->inferior_cnt;
	superiors = ct->superior_cnt;
	num_rte   = ct->maxNumParallelRouteLines;

	switch ( w->tree.orientation )
	{
	   case LEFT_TO_RIGHT:   

		pt1_x = child_x   - 1; 
		pt1_y = child_y   + child_half_height;
		pt4_x = parent_x  + parent_width;
		pt4_y = parent_y  + parent_half_height;

		if ( pt1_y == pt4_y && ct->layout_cnt == 1) /* Single inline */
		{
		    SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt4_x, pt4_y );
		    break;
		}

		pt2_y = pt1_y;

		if ( rel_posn > 0 )
		{
		    pt2_x = child_x - ( (num_rte + 1 - rel_posn) * pad);
		}
		else if ( rel_posn < 0 )
		    pt2_x = child_x - ((0 - rel_posn) * pad);
		else
		    pt2_x = child_x - pad;

		pt2_x -= 1;  		/* For shape box */
		pt2_x -= half_Hpad;  	/* Center in normal horizonal pad */

		pt3_x = pt2_x;		pt3_y = pt4_y;

		if ( ct->num_children > 1 ) /* Now adjust the y position */
		    pt3_y += (((num_kids-1)*pad)/2) -((num_kids-child_num)*pad);

		pt4_x = parent_x + parent_width; 	pt4_y = pt3_y;

		SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt2_x, pt2_y );
		SET_SEGMENT( w, seg2, pt2_x, pt2_y, pt3_x, pt3_y );
		SET_SEGMENT( w, seg3, pt3_x, pt3_y, pt4_x, pt4_y );

		if ( pt4_y < parent_y || pt4_y > (parent_y + parent_height) )
		{
		    pt5_x =  parent_x + parent_width/2;
		    pt5_y =  parent_y + parent_half_height; 

		    SET_SEGMENT( w, seg4, pt4_x, pt4_y, pt5_x, pt5_y );
		}
		break;

	   case RIGHT_TO_LEFT:

		pt1_x = child_x   + child_width; 
		pt1_y = child_y   + child_half_height;
		pt4_x = parent_x  - 1;
		pt4_y = parent_y  + parent_half_height;

		if ( pt1_y == pt4_y && ct->layout_cnt == 1) /* Single inline */
		{
		    SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt4_x, pt4_y );
		    break;
		}

		pt2_y = pt1_y;
		if ( rel_posn > 0 )
		    pt2_x = parent_x - (rel_posn * pad);
		else if ( rel_posn < 0 )
		    pt2_x = parent_x - (((num_rte + 1) + rel_posn) * pad);
		else
		    pt2_x = parent_x - pad;

		pt2_x -= 1;  		/* For shape box */
		pt2_x -= half_Hpad;  	/* Center in normal horizonal pad */


		pt3_x = pt2_x;		pt3_y = pt4_y;

		if ( ct->num_children > 1 ) /* Now adjust the y position */
			pt3_y += ( ((num_kids - 1) * pad)/2) -
			           ((num_kids - child_num) * pad);

		pt4_x = parent_x - 1;
		pt4_y = pt3_y;

		SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt2_x, pt2_y );
		SET_SEGMENT( w, seg2, pt2_x, pt2_y, pt3_x, pt3_y );
		SET_SEGMENT( w, seg3, pt3_x, pt3_y, pt4_x, pt4_y );

		if ( pt4_y < parent_y || pt4_y > (parent_y + parent_height) )
		{
		    pt5_x =  parent_x + parent_width/2;
		    pt5_y =  parent_y + parent_half_height; 

		    SET_SEGMENT( w, seg4, pt4_x, pt4_y, pt5_x, pt5_y );
		}
		break;

	   case TOP_TO_BOTTOM:

		pt1_x = child_x + child_width/2;
		pt1_y = child_y - 1;

		pt4_x = parent_x + parent_width/2;
		pt4_y = parent_y + parent_height;


		if ( pt1_x == pt4_x && ct->layout_cnt == 1) /* Single inline */
		{
		    SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt4_x, pt4_y );
		    break;
		}

		pt2_x = pt1_x;

		if ( rel_posn > 0 )
		{
		  adj = 0;

		  /* The following is experimental, to see what spreading
		   * the route lines looks like(only when unevenly distributed)
		   * This is currently only for top to bottom.

		  adj = num_rte - superiors;
		  if ( superiors < num_rte )
		  {
		    if ( adj != 0 && superiors != 0)
		    {
		        d_adj = (((double)((adj+1) * pad))/(double)superiors) * 
					((double)(superiors + rel_posn));

			adj = d_adj;
		    }
		  }
		   */
		  
		    pt2_y = child_y - (((num_rte + 1) - rel_posn) * pad) + adj;
		}
		else if ( rel_posn < 0 )
		{
		    /* pt2_y = child_y - ((((num_rte + 1) -
			  	ct->inferior_cnt) + rel_posn) * pad); */
		    		
		  adj = 0;

		  /*
		  adj = num_rte - inferiors;
		  if ( inferiors < num_rte ) 
		  {
		    if ( adj != 0 && inferiors != 0)
		    {
		        d_adj = (((double)((adj+1) * pad))/(double)inferiors) * 
					((double)(inferiors + rel_posn));

			adj = d_adj;
		    }
		  }
		  */
		  
		  pt2_y = (child_y - (((0 - rel_posn) +
				((num_rte) - inferiors)   ) * pad)) + adj;

		}
		else
		    pt2_y = child_y;

		pt2_y -= 1;  		/* For shape box */
		pt2_y -= half_Vpad;  	/* Center in normal horizonal pad */

		pt3_x = pt4_x; /* Already calculated above */

		if ( ct->num_children > 1 )
		{
		    pt3_x += ( ((num_kids - 1) * pad)/2) -
			           ((num_kids - child_num) * pad);
		}
		pt3_y = pt2_y;

		pt4_x = pt3_x;   /* In case it changes, pt4_y does not */

		SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt2_x, pt2_y );
		SET_SEGMENT( w, seg2, pt2_x, pt2_y, pt3_x, pt3_y );
		SET_SEGMENT( w, seg3, pt3_x, pt3_y, pt4_x, pt4_y );

		if ( pt4_x < parent_x || pt4_x > (parent_x + parent_width) )
		{
		    pt5_x =  parent_x + parent_width/2;
		    pt5_y =  parent_y + parent_half_height; 

		    SET_SEGMENT( w, seg4, pt4_x, pt4_y, pt5_x, pt5_y );
		}

		break;

	   case BOTTOM_TO_TOP:

		pt1_x = child_x + child_width/2;
		pt1_y = child_y + child_height;

		pt4_x = parent_x + parent_width/2;
		pt4_y = parent_y;


		if ( pt1_x == pt4_x && ct->layout_cnt == 1) /* Single inline */
		{
		    SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt4_x, pt4_y );
		    break;
		}

		pt2_x = pt1_x;

		if ( rel_posn > 0 )
		{
		    pt2_y = (child_y + child_height) + 
		    		 (((num_rte + 1) - rel_posn) * pad);
		}
		else if ( rel_posn < 0 )
		{
		  pt2_y = ((child_y + child_height) + (((0 - rel_posn) +
				((num_rte) - inferiors)   ) * pad));
		}
		else
		    pt2_y = child_y + child_height;

		pt2_y += 1;  		/* For shape box */
		pt2_y += half_Vpad;  	/* Center in normal horizonal pad */

		pt3_x = pt4_x; /* Already calculated above */

		if ( ct->num_children > 1 )
		{
		    pt3_x += ( ((num_kids - 1) * pad)/2) -
			           ((num_kids - child_num) * pad);
		}
		pt3_y = pt2_y;

		pt4_x = pt3_x;   /* In case it changes, pt4_y does not */

		SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt2_x, pt2_y );
		SET_SEGMENT( w, seg2, pt2_x, pt2_y, pt3_x, pt3_y );
		SET_SEGMENT( w, seg3, pt3_x, pt3_y, pt4_x, pt4_y );

		if ( pt4_x < parent_x || pt4_x > (parent_x + parent_width) )
		{
		    pt5_x =  parent_x + parent_width/2;
		    pt5_y =  parent_y + parent_half_height; 

		    SET_SEGMENT( w, seg4, pt4_x, pt4_y, pt5_x, pt5_y );
		}

		break;

	   case STAR_TOPOLOGY:
		pt1_x = parent->core.x    + parent->core.width/2;
		pt1_y = parent->core.y    + parent->core.height/2;
		pt2_x = child->core.x     + child->core.width/2;
		pt2_y = child->core.y     + child->core.height/2;

		SET_SEGMENT( w, seg1, pt1_x, pt1_y, pt2_x, pt2_y );


		break;
	}
}
/* ---------------------------  shift_subtree  -------------------------- */

static
void shift_subtree( w, offset, orientation )
Widget		w;
Dimension	offset;
TreeOrientation	orientation;
{
	TreeConstraints tree_const = TREE_CONSTRAINT(w);
	ChildPtr	 *p;

	/* shift the node by the offset */

	switch ( orientation )
	{
	    case TOP_TO_BOTTOM:
	    case BOTTOM_TO_TOP:
	    case STAR_TOPOLOGY: 	tree_const->tree.x += offset;
					break;
	    case LEFT_TO_RIGHT:
	    case RIGHT_TO_LEFT:
	    default: 			tree_const->tree.y += offset;
					break;
	}

	/* descend tree and shift each child into place */

	for ( p = tree_const->tree.children; p != NULL; p = p->next )
	{
		if ( XtIsManaged( p->child ) )
			shift_subtree( p->child, offset, orientation );
	}
}


/* ---------------------------  set_positions  -------------------------- */

static
void set_positions( tw, w, level )
XpsiTreeWidget	tw;
Widget		w;
int		level;
{
	ChildPtr	 *p;
	double		  angle, radius;
	int		  depth, ring_width;
	Dimension	  total_width, total_height;
	int		  level_width=0, level_height=0, offset=0;

	if ( w )
	{
	    TreeConstraints tree_const = TREE_CONSTRAINT( w );

	    /* Find max width, height for layouts and determining maxx & maxy */
	    if ( tw->tree.node_size  == FIXED_SIZE_FOR_TREE )
	    {
		if ( level > 0 )
		{
			level_width  = tw->tree.max_node_width;
			level_height = tw->tree.max_node_height;
		}
	    }
	    else
	    {
		level_width  = current_value(tw->tree.max_width, level);
		level_height = current_value(tw->tree.max_height, level);
	    }

	    switch ( tw->tree.orientation )
	    {
	        case LEFT_TO_RIGHT:
			
	    	/* Add up the sum of the width's of all nodes to 
	    	 * this depth, and use it as the x position.
	      	 */

	    	    tree_const->tree.x = ((level-1) * tw->tree.min_x_pad ) +
	    		 sum_table(tw->tree.horizontal_pos, level );
			       
	    	    break;

	        case RIGHT_TO_LEFT:
	    
	    	    depth = tw->tree.depth;
	    	    total_width = ((depth - 1) * tw->tree.min_x_pad )      +
	    		sum_table( tw->tree.horizontal_pos, depth + 1);

	    	    tree_const->tree.x = ( total_width - 
	    		    (((level-1) * tw->tree.min_x_pad ) +
	    		    sum_table(tw->tree.horizontal_pos, level+1))) ;

		    if ( tw->tree.connect_style == PATHWAY_CONNECTION )
		    {
			offset = (current_value(tw->tree.horizontal_pos,level) -
		       			level_width);

			tree_const->tree.x += offset;
		    }

	    	    break;

	    
	        case TOP_TO_BOTTOM:

	    	    /* Add up the sum of the height's of all nodes to this
	     	     * depth, and use it as the y position.
	     	     */

	    	    tree_const->tree.y = ( (level-1) * tw->tree.min_y_pad ) +
	    		sum_table(tw->tree.vertical_pos, level );
	    	    break;

	        case BOTTOM_TO_TOP:
		
		    depth = tw->tree.depth;
		    total_height = ((depth) * tw->tree.min_y_pad )    +
			sum_table( tw->tree.vertical_pos, depth+1);

		    tree_const->tree.y = total_height - 
			    (( level * tw->tree.min_y_pad ) +
			     sum_table(tw->tree.vertical_pos,level + 1)) +
			    (current_value(tw->tree.vertical_pos,level)
							   - w->core.height);

		    break;

	        case STAR_TOPOLOGY:

	    	 /* Convert from polar coordinates to 4 quadrant based
	    	  * x, y. Later, in center_star_tree we adjust these
	    	  * values to the widgets 0,0 upper left based coords.
	    	  */

	    	radius= 1.0 * sum_table(tw->tree.ring_width, level);

	    	if ( level > 1 )  /* Ignore psuedo root */
	    	{
	    	    /* Adjust node radius to middle of ring */
		    ring_width = current_value(tw->tree.ring_width, level);
		    radius   = ((double) sum_table(tw->tree.ring_width,
					  level)) + ((double) ring_width/2.0);
		}
		
		/* Angle stored in widgets constraint x value */

		angle = RADIANS(((double)tree_const->tree.x));

		tree_const->tree.x = (radius * cos( angle ))      - 
						((double)w->core.width/2) ;
		tree_const->tree.y = (radius * sin( angle ))      -
						((double)w->core.height/2) ;

		break;
	    }


	   /* If orientation is not star topology we have sufficient
	    * information to move the widget now. If it is a star we
	    * keep track of the minimum x and y so we can later
	    * translate the coordinates to the widgets 0,0 upper left
	    * coordinate space.
	    */

	   
	   if ( tw->tree.orientation == STAR_TOPOLOGY )
	   {
	   	if (((int)tree_const->tree.x) < tw->tree.min_x)
		{
	   		tw->tree.min_x =  tree_const->tree.x;
		}

	   	if (((int)tree_const->tree.y) < tw->tree.min_y)
		{
			tw->tree.min_y =  tree_const->tree.y;
		}
	   }
	   else
	   {
	    	/* Offset for left and top margins */
	    	tree_const->tree.x += tw->tree.left_margin;
	    	tree_const->tree.y += tw->tree.top_margin;
	   	XtMoveWidget(w, tree_const->tree.x, tree_const->tree.y);
	   }


	   /* If the widget position plus its width or height 
	    * doesn't fit in the tree, bump the appropriate 
	    * max size value.
	    */

	   if ( level > 0 )
	   {
	      if ( (tree_const->tree.x + level_width)  > ((int) tw->tree.max_x))
	      {
	   		tw->tree.max_x = tree_const->tree.x + level_width;
	      }

	      if ( (tree_const->tree.y + level_height) > ((int) tw->tree.max_y))
	      {
	   		tw->tree.max_y = tree_const->tree.y + level_height;
	      }
	   }							

	   /* Decend further into tree, positioning as we go */

	   for ( p = tree_const->tree.children; p != NULL; p = p->next )
	   {
		if ( XtIsManaged( p->child ) || tw->tree.layout_unmanaged )
	   		set_positions( tw, p->child, level+1 );
	   }
        }
}

/* ---------------------------  center_star_tree  -------------------------- */

static
void center_star_tree( tw, w, level )
XpsiTreeWidget	tw;
Widget		w;
int		level;
{
	ChildPtr	 *p;
	int		 x_shift;

	if ( w )
	{
		TreeConstraints tree_const = TREE_CONSTRAINT( w );

		/* Shift tree right to bring all x values > 0 */

		if ( tw->tree.show_rings == True )
		{
			x_shift=  sum_table( tw->tree.ring_width,
							 tw->tree.depth + 1) +
							 tw->tree.left_margin;
		}
		else if ( tw->tree.node_shape == True )
		{
			/* need to add enough to see all of node circle */

			x_shift = tw->tree.left_margin - tw->tree.min_x;

			/* tree_const->tree.radius - width/2 */
		}
		else	/* just offset by minimum x amount + margin */
		{
			x_shift = tw->tree.left_margin - tw->tree.min_x;
		}


		tree_const->tree.x += x_shift;

		/* Need to invert y, to make counter clockwise display */

		tree_const->tree.y = (tw->tree.max_y - tree_const->tree.y) +
					tw->tree.top_margin;

		XtMoveWidget(w, tree_const->tree.x,
				tree_const->tree.y - w->core.height);

		for ( p = tree_const->tree.children; p != NULL; p = p->next )
		{
			center_star_tree( tw, p->child, level+1 );
		}
	}
}

/* --------------------------  create_table   ---------------------------- */

static
TreeTablePtr create_table( size )
int size;
{
	TreeTablePtr  table = (TreeTablePtr) XtMalloc(sizeof(TreeTable));

	table->size = size;
	table->array = (Dimension *) XtMalloc(size * sizeof(Dimension));

	return( table );
}

static
void destroy_table( table )
TreeTablePtr  table;
{
	if ( table != NULL )
	{
		if ( table->array != NULL )
		{
			XtFree( (Xmem) table->array );
		}
		XtFree( (Xmem) table );
		table = NULL;
	}
}

/* --------------------------  reset_table    ------------------------- */

static
void reset_table( table )
TreeTablePtr table;
{
	int i;

	for ( i = 0; i < table->size; i++ )
		table->array[i] = 0;
}

/* -----------------------  current_value  --------------------------- */

static
Position current_value( table, index )
TreeTablePtr   table;
int		index;
{
	if ( index >= table->size )
		return( 0 );

	return( table->array[index]);
}

/* ----------------------  set_current_value  ------------------------- */

static
void set_current_value( table, index, value )
TreeTablePtr	table;
int		index;
Dimension	value;
{
	if ( index >= table->size )
	{
		table->size = index + index / 2;
		table->array = (Dimension *) XtRealloc( (Xmem) table->array,
						table->size *
							sizeof(Dimension));
	}

	table->array[index] = value;
}

/* ---------------------------  sum_table  -------------------------- */

static
Position sum_table( table, index )
TreeTablePtr   table;
int		index;
{
	int i;

	Position  sum = 0;
	int      stop = index;

	if ( index > table->size )
		stop = table->size;

	for ( i = 0; i < stop ; i++ )
		sum += table->array[i];

	return( sum );
}

/* --------------------  adjust_connection_segments  ----------------------- */

static
void adjust_connection_segments( tw )
XpsiTreeWidget	tw;
{
	int num_segments=0;

	switch( tw->tree.connect_style )
	{
		case DIAGONAL_CONNECTION:
			num_segments = tw->tree.total_children + 1;
			break;

		case TIER_CONNECTION:
			num_segments = (tw->tree.total_children + 1);
			if ( tw->tree.parents_place == PARENT_CENTERED )
				num_segments += (tw->tree.total_parents * 3 );
			else
				num_segments += (tw->tree.total_parents * 3 );
			break;

		case PATHWAY_CONNECTION:
			num_segments = (tw->tree.total_children * 4);
			break;

		default:
			num_segments=0;
			break;
	}

	if ( tw->tree.max_connect_lines == 0 && num_segments > 0 )
	{
	    tw->tree.connect_lines = (XSegment *) 
					XtMalloc( num_segments *
						     sizeof(XSegment));
	}
	else if ( num_segments > tw->tree.max_connect_lines )
	{
	    tw->tree.connect_lines = (XSegment *) 
				XtRealloc( (Xmem) tw->tree.connect_lines,
						   num_segments *
						     sizeof(XSegment));
	}

	/* Simple cache, never free up connection_lines  */

	tw->tree.max_connect_lines = num_segments;
	tw->tree.num_connect_lines = 0;
}




/* -------------------------  adjust_arcs  ---------------------------- */

static
void adjust_arcs( tw )
XpsiTreeWidget	tw;
{
	int num_arcs=0;

	if ( tw->tree.node_shape == True && 
			tw->tree.orientation == STAR_TOPOLOGY)
	{
		num_arcs = tw->tree.total_children + 1;
	}

	if ( tw->tree.max_arcs == 0 && num_arcs > 0 )
	{
	    tw->tree.arcs = (XArc *) XtMalloc( num_arcs * sizeof(XArc));
	    tw->tree.fillColor = (Pixel *) XtMalloc( num_arcs * sizeof(Pixel));
	}
	else if ( num_arcs > tw->tree.max_arcs )
	{
	    tw->tree.arcs = (XArc *) XtRealloc( (Xmem) tw->tree.arcs, 
						num_arcs * sizeof(XArc));
	    tw->tree.fillColor = (Pixel *) XtRealloc((Xmem) tw->tree.fillColor, 
						num_arcs * sizeof(Pixel));
	}

	/* Simple cache, never free up arcs  */

	tw->tree.max_arcs = num_arcs;
	tw->tree.num_arcs = 0;
}

/* -------------------------  adjust_rings  ---------------------------- */

static
void adjust_rings( tw )
XpsiTreeWidget	tw;
{
	int num_rings=0;

	if ( tw->tree.show_rings )
	{
		num_rings += (tw->tree.depth + 2);
	}

	if ( tw->tree.max_rings == 0 && num_rings > 0 )
	{
	    tw->tree.rings = (XArc *) XtMalloc( num_rings *
						     sizeof(XArc));
	}
	else if ( num_rings > tw->tree.max_rings )
	{
	    tw->tree.rings = (XArc *) XtRealloc( (Xmem) tw->tree.rings, 
						num_rings * sizeof(XArc));
	}

	/* Simple cache, never free up rings  */

	tw->tree.max_rings = num_rings;
	tw->tree.num_rings = 0;
}

/* ------------------------  adjust_rectangles  --------------------------- */

static
void adjust_rectangles( tw )
XpsiTreeWidget	tw;
{
	int num_rectangles=0;

	if ( tw->tree.node_shape == True && 
				tw->tree.orientation != STAR_TOPOLOGY )
	{
		num_rectangles = tw->tree.total_children + 1;
	}


	if ( tw->tree.max_rectangles == 0 && num_rectangles > 0 )
	{
	    tw->tree.rectangles = (XRectangle *) XtMalloc( num_rectangles *
						     sizeof(XRectangle));

	    tw->tree.fillColor = (Pixel *) XtMalloc( num_rectangles * 
	    					sizeof(Pixel));
	}
	else if ( num_rectangles > tw->tree.max_rectangles )
	{
	    tw->tree.rectangles = (XRectangle *) 
	    			   XtRealloc( (Xmem) tw->tree.rectangles,
					num_rectangles * sizeof(XRectangle));

	    tw->tree.fillColor = (Pixel *) XtRealloc( (Xmem) tw->tree.fillColor,
					num_rectangles * sizeof(Pixel));

	}

	/* Simple cache, never free up rectangles  */

	tw->tree.max_rectangles = num_rectangles;

	tw->tree.num_rectangles = 0;  /* So we can draw less if desired */
}
