/* File: 	icons.c
 *
 * Description: Icon handling Routines
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1999, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	2/8/99	Created
 *            
 *
 */

#include <stdio.h>		
#include <sys/stat.h>		
#include <unistd.h>		

#include <X11/Intrinsic.h>

#include "localized.h"

#ifdef USE_PIXMAPS
#include <xpm.h>
#endif

#include <Xm/Xm.h>

#include "icons.h"

#include "debug.h"		/* Simple debug package		   */


Icons  IconCache;



static char *DefaultIconRootDir=DEFAULT_ICON_DIR;

char *IconRoot=NULL;

extern int Debug;

int fetchIcon();
int setIconRoot();
void freeAllIconPixmaps();


static Icon_list *getIconList();
static Icon_record *getIcon();

static void freeIconCacheSetsPixmaps();
static void freeIconCacheListPixmaps();

int 
setIconRoot( newRootDir )
char *newRootDir;
{
	if ( IconRoot != NULL )
		free( IconRoot );

	IconRoot = NULL;

	if ( newRootDir != NULL )
		IconRoot = strdup( newRootDir );

	return 0;
}

void
freeAllIconPixmaps( w )
Widget w;
{
	if ( IconCache.mono != NULL )
		freeIconCacheSetsPixmaps( w, IconCache.mono );

	if ( IconCache.loColor != NULL )
		freeIconCacheSetsPixmaps( w, IconCache.loColor );

	if ( IconCache.hiColor != NULL )
		freeIconCacheSetsPixmaps( w, IconCache.hiColor );
}


void
freeIconCacheListPixmaps( w, iconList )
Widget w;
Icon_list *iconList;
{
	Icon_record *p;

	if ( iconList == NULL )
		return;

	for ( p = iconList->first ; p != NULL ; p = p->next )
	{
		if (  p->icon_pixmap != 0 )
		{
		    XFreePixmap( XtDisplay(w), p->icon_pixmap );
		    p->icon_pixmap = 0;

		    /* Could remove record, but in this case we
		     * are only calling this when program exits
		     * so that pixmap resources can be properly freed.
		     */
		}
	}

}

void
freeIconCacheSetsPixmaps( w, iconSet )
Widget w;
Icon_set *iconSet;
{
	if ( iconSet == NULL )
		return;

	freeIconCacheListPixmaps( w, iconSet->size16 );
	freeIconCacheListPixmaps( w, iconSet->size22 );
	freeIconCacheListPixmaps( w, iconSet->size32 );
	freeIconCacheListPixmaps( w, iconSet->size48 );
	freeIconCacheListPixmaps( w, iconSet->size64 );
}


Icon_list *
createIconList( listName )
char *listName;
{
	Icon_list *newList = (Icon_list *) malloc( sizeof(Icon_list) );
	if ( newList == NULL )
	{
		DEBUG0( 0, "icons: createIconList Malloc Failed");
		return NULL;
	}

	if ( listName != NULL )
		newList->name = strdup( listName );
	else
		newList->name = NULL;

	newList->first = NULL;

	return( newList );
}

Icon_set *
createIconSet( setName )
char *setName;
{
	char listName[80];

	Icon_set  *newSet = (Icon_set *) malloc( sizeof(Icon_set) );
	if ( newSet == NULL )
	{
		DEBUG0( 0, "icons: createIconSet Malloc Failed\n");
		return NULL;
	}

	newSet->size16 = createIconList( "16" );
	newSet->size22 = createIconList( "22" );
	newSet->size32 = createIconList( "32" );
	newSet->size48 = createIconList( "48" );
	newSet->size64 = createIconList( "64" );

	if ( newSet->size16 == NULL || newSet->size22 == NULL ||
	     newSet->size32 == NULL || newSet->size48 == NULL ||
	     newSet->size64 == NULL )
	{
		/* Should free up unused ones, but not worth the code */

		DEBUG0( 0, "icons: createIconSet Initialization Failed\n");

		return NULL;
	}

	if ( setName != NULL )
		newSet->name = strdup( setName );
	else
		newSet->name = NULL;

	return newSet;
}

int
fetchIcon( iconName, w, colorMode, iconSize, iconPixmap, 
						iconWidth, iconHeight )
char    *iconName;
Widget 	w;
int	colorMode;
int	iconSize;
Pixmap  *iconPixmap;
int 	*iconWidth;
int	*iconHeight;
{
	static int first=1;
	int	   rc=-1;

	*iconPixmap = (int)NULL;	/* Set defaults for error return */
	*iconWidth  = 0;
	*iconHeight = 0;

	DEBUG1( 3, "icons: fetchIcon name(%s)\n", iconName);

	if ( first )
	{
		DEBUG0( 1, "icons: fetchIcon Initializing Icon Cache\n");
		IconCache.mono    = createIconSet("mono");
		IconCache.loColor = createIconSet("loColor");
		IconCache.hiColor = createIconSet("hiColor");

		if ( IconCache.mono    == NULL ||
		     IconCache.loColor == NULL ||
		     IconCache.hiColor == NULL            )
		{
		    DEBUG0( 0, "icons: fetchIcon Initialization Failed\n");
		    return rc;
		}

		first = 0;
	}

	if ( IconRoot == NULL )
	{
		IconRoot = strdup( DefaultIconRootDir );
	}

	if ( w == NULL )
		return( rc );

	if ( colorMode == MONOCHROME )
	{
		rc = loadBitmapToSet( IconCache.mono, iconName, w, iconSize, 
					iconPixmap, iconWidth, iconHeight );
	}
	else if ( colorMode == LOW_COLOR )
	{
		rc = loadPixmapToSet( IconCache.loColor, iconName, w, iconSize,
					iconPixmap, iconWidth, iconHeight );
	}
	else if ( colorMode == HIGH_COLOR )
	{
		rc = loadPixmapToSet( IconCache.hiColor, iconName, w, iconSize,
					iconPixmap, iconWidth, iconHeight );
	}

	return( rc );	/* Return OK  */
}

static Icon_list *
getIconList( iconSet, iconSize )
Icon_set *iconSet;
int iconSize;
{
	Icon_list *iconList = NULL;

	switch ( iconSize ) {

		case 16:  iconList = iconSet->size16; break;
		case 22:  iconList = iconSet->size22; break;
		case 32:  iconList = iconSet->size32; break;
		case 48:  iconList = iconSet->size48; break;
		case 64:  iconList = iconSet->size64; break;

		default: iconList = NULL; break;
	};

	return ( iconList );
}

static Icon_record *
getIcon( iconList, iconName )
Icon_list *iconList;
char 	  *iconName;
{
	Icon_record *p;

	for ( p = iconList->first ; p != NULL ; p = p->next )
	{
		if ( strcmp( iconName, p->name) == 0 )
			return p;
	}

	return( NULL );
}

int
addNewIconTo( iconList, iconName, iconPixmap, iconWidth, iconHeight )
Icon_list *iconList; 
char 	*iconName; 
Pixmap 	*iconPixmap; 
int 	*iconWidth; 
int	*iconHeight;
{
	Icon_record  *newIcon=NULL;
	Icon_record  *nextIcon=NULL;

	if ( iconList == NULL )
	{
		DEBUG0( 0, "addNewIconTo Invalid list\n" );
		return -1;
	}

	newIcon = (Icon_record *) malloc( sizeof( Icon_record ) );
	if ( newIcon == NULL )
	{
		DEBUG0( 0, "addNewIconTo: malloc newIcon failed\n" );
		return -1;
	}

	newIcon->name = strdup( iconName );
	newIcon->icon_pixmap = *iconPixmap;
	newIcon->icon_width  = *iconWidth;
	newIcon->icon_height = *iconHeight;
	
	if ( iconList->first != NULL )
		nextIcon = iconList->first->next;
	else
		nextIcon = NULL;

	newIcon->next = nextIcon;

	iconList->first = newIcon;	/* Push on front */
}



int
loadPixmapToSet( iconSet, iconName, w, iconSize, iconPixmap, 
						iconWidth, iconHeight )
Icon_set *iconSet;
char     *iconName;
Widget 	w;
int	iconSize;
Pixmap  *iconPixmap;
int 	*iconWidth;
int	*iconHeight;
{
	int 	     	rc = -1;
	int	     	status;
	Icon_record 	*icon = NULL;
	Pixmap 	     	pixmap;
	char		iconPath[256];
	Icon_list 	*iconList = NULL;
	struct  stat    s;
	Pixel		fg_color, bg_color;

#ifdef USE_PIXMAPS
	XpmColorSymbol  symbol[2];
	XpmAttributes   attributes;
#endif

	int		newIcon=1;

	iconList = getIconList( iconSet, iconSize );
	if ( iconList == NULL )
	{
		DEBUG1( 0, "icons: loadPixmapToSet Invalid iconSize(%d)\n", 
								iconSize);
	        return rc;
	}

	/* Scan list to see if already loaded, if so return it */

	if ( ( icon = getIcon( iconList, iconName ) ) != NULL )
	{
		if ( icon->icon_pixmap != 0 ) /* Already loaded */
		{
			*iconPixmap = icon->icon_pixmap;
			*iconWidth  = icon->icon_width;
			*iconHeight = icon->icon_height;
			return 0;
		}
		else  
			newIcon = 0; 	/* Need to Reload Pixmap */
	}

	/* Otherwise get from file */

	sprintf( iconPath, "%s/%s/%s/%s.xpm", IconRoot, iconSet->name,
					iconList->name, iconName );

	/* Stat file to ensure it's there and readable */

        DEBUG1(3,"loadPixmapToSet: loading %s\n", iconPath );

	rc = stat( iconPath, &s );
        if ( rc == -1 )
        {
                if ( Debug > 0 )
                        perror("loadPixmapToSet: stat");
                DEBUG1(0,"loadPixmapToSet:can't stat %s\n", iconPath );
                return( rc );
        }


	XtVaGetValues( w, XtNforeground, &fg_color, NULL );
	XtVaGetValues( w, XtNbackground, &bg_color, NULL );

#ifdef USE_PIXMAPS
	symbol[0].name  = NULL;
	symbol[0].value = "none";
	symbol[0].pixel = bg_color;

	symbol[1].name  = NULL;
	symbol[1].value = "fore";
	symbol[1].pixel = fg_color;

	attributes.colorsymbols = &symbol[0];
	attributes.numsymbols   = 2;
     	attributes.valuemask    = XpmColorSymbols; 

    	status = XpmReadFileToPixmap(XtDisplay(w),
                                 XRootWindowOfScreen(XtScreen(w)),
                                 iconPath, 
				 &pixmap, 
				 NULL, 
				 &attributes);

    	if (status != XpmSuccess) 
	{
          DEBUG1(0,"icon.c:loadPixmapToSet:XpmReadFileToPixmap Failed on(%s)\n",
	  			iconPath );
          fprintf(stderr, "XpmError: %s\n", XpmGetErrorString(status));
          return -1;
    	}

	rc = 0;

 	*iconWidth  = attributes.width;
        *iconHeight = attributes.height;
	*iconPixmap = pixmap;

	if ( newIcon )
	  addNewIconTo( iconList, iconName, iconPixmap, iconWidth, iconHeight );

#else
          rc = -1;
#endif

	return rc;
}


int
loadBitmapToSet( iconSet, iconName, w, iconSize, iconPixmap, 
						iconWidth, iconHeight )
Icon_set *iconSet;
char     *iconName;
Widget 	w;
int	iconSize;
Pixmap  *iconPixmap;
int 	*iconWidth;
int	*iconHeight;
{
	int 	        rc  = -1;
	int	     	status;
	Icon_record 	*icon=NULL;
	Pixmap 	     	pixmap;
	char		iconPath[256];
	int 		x_hot, y_hot;
	unsigned int	icon_width, icon_height;
	Icon_list 	*iconList=NULL;
	struct  stat    s;

	iconList = getIconList( iconSet, iconSize );
	if ( iconList == NULL )
	{
		DEBUG1( 0, "icons: loadBitmapToSet Invalid iconSize(%d)\n", 
								iconSize);
	        return rc;
	}

	/* Scan list to see if already loaded, if so return it */

	if ( ( icon = getIcon( iconList, iconName ) ) != NULL )
	{
		*iconPixmap = icon->icon_pixmap;
		*iconWidth  = icon->icon_width;
		*iconHeight = icon->icon_height;
	}

	/* Otherwise get from file */

	sprintf( iconPath, "%s/%s/%s/%s", IconRoot, iconSet->name,
					iconList->name, iconName );

        DEBUG1(1,"loadBitmapToSet: loading %s\n", iconPath );

	rc = stat( iconPath, &s );
        if ( rc == -1 )
        {
                if ( Debug > 0 )
                        perror("loadBitmapToSet: stat");
                DEBUG1(1,"loadBitmapToSet:can't stat %s\n", iconPath );
                return( rc );
        }

 	status = XReadBitmapFile( XtDisplay(w),
                                 XRootWindowOfScreen(XtScreen(w)),
				 iconPath,
				 &icon_width, &icon_height, &pixmap, 
				 &x_hot, &y_hot);

	if ( status != 0 )
	{
          DEBUG2(0,"icon.c:loadBitmapToSet:XReadBitmapFile Failed(%d) on(%s)\n",
	  					status, iconPath );
          return -1;
	}
	else
		rc = 0;

 	*iconWidth  = icon_width;
        *iconHeight = icon_height;
	*iconPixmap = pixmap;

	addNewIconTo( iconList, iconName, iconPixmap, iconWidth, iconHeight );

	return rc;
}
