/* File: 	color_code.c
 *
 * Description: Routines to color by user, group, load or other stats
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1995, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	3/8/95	Created
 *            
 *
 */

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/ArrowBG.h>
#include <Xm/PushBG.h>
#include <Xm/Form.h>
#include <Xm/SeparatoG.h>
#include <Xm/ScrolledW.h>
#include <Xm/Label.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <Xpsi/Tree.h>

#include "tree.h"	/* Generic m-ary tree package include file */
#include "list.h"	/* Simple list package			   */
#include "debug.h"	/* Macro based debug package		   */

#include "treeps.h"
#include "user_group.h"

#include "color_code.h"

extern Pixmap getDeepIconByName();

#include "icons/mono/48/color_bar48.icon"

extern User_group_info  *Users;
extern Object_color	*User_colors;
extern Object_color	*Group_colors;
extern Object_color	*Stat_colors;

extern lnode *sorted_group_list();
extern lnode *sorted_passwd_list();

extern int find_gid_by_name();
extern int find_uid_by_name();
extern char *get_color_menu_str();

extern Widget Top_level;
extern Widget Main_menu;


Range_color       Default_range_colors[MAX_COLOR_RANGES];

Range_color_map   Default_color_map; /* Using above colors */

Range_color_map   *User_color_map;		
Range_color_map   *Group_color_map;		
Range_color_map   *Load_color_map;		
Range_color_map   *Time_color_map;	
Range_color_map   *Image_color_map;
Range_color_map   *Status_color_map;
Range_color_map   *Priority_color_map;
Range_color_map   *Resident_color_map;

Range_color_map  *current_color_map();

Range_color_map	 *create_psuedo_range_map();

long        Max_isize=0; 
long        Max_rsize=0; 
long 	    Max_time=0;


Widget Color_bar_shw=NULL;	/* Shell used to display color bar */
Widget Color_bar_lw=NULL;	/* Color bar label		   */
Widget Color_bar_law=NULL;	/* Color bar left arrow		   */
Widget Color_bar_raw=NULL;	/* Color bar right arrow	   */
Widget Color_bar_sw=NULL;	/* Color bar scrolled window widget*/
Widget Color_bar_sfw=NULL;	/* Color bar scrolled form widget  */
Widget Color_bar_ltw=NULL;	/* Color bar label toggle widget   */

Widget  *Color_bar_w=NULL;

int    Color_bar=-1;		/* Currently displayed colors      */

#define LABEL_MODE	1
#define RANGE_MODE	2
#define COUNT_MODE	3

int Color_bar_label_mode=1;
int Color_bar_range_mode=0;
int Color_bar_count_mode=0;
int Color_bar_averaging=0;

Widget popup_color_bar();
void   cb_mode();
void   cb_mv();
void   cb_range();

int Reset_load_avg=1;

#define USER   1
#define GROUP  2


#define NO_DIALOG               0
#define DIALOG_DISPLAYED        1
#define DIALOG_HIDDEN           2


int CB_dialog_state=NO_DIALOG;

static Atom deleteWindow=0;
static Atom protocols=0;
static void deleteWindowHandler();

void          (*CB_popdownCB)()=NULL;

void    cbShellResize();
void    cbPreResizeWindowOffsets();

/* The following are used to limit the size of the automatic window growth.
 * If the window get's larger than this then scroll bars will be enabled,
 * values of -1 mean the window can grow arbitrarily large, this is for
 * those who want such a thing(perhaps on virtual desktops).
 */

extern int     Max_window_x;
extern int     Max_window_y;
static int     Min_window_x=130;
static int     Min_window_y=100;

extern Boolean FitToScreen;

extern int ActualScreenWidth;
extern int ActualScreenHeight;
extern int BottomOffset;

static int RightOffset=1;

struct dimensionalOffsets {
        int     top;
        int     left;
        int     bottom;
        int     right;
};

struct winSizeInfo {
        int     last_width;
        int     last_height;
        int     width;
        int     height;
};

extern int Decorated_tree;

/* ------------------- Color processing routines ------------------ */


extern int PageSize;

Object_color
*create_object_color( id, name, fg_str, bg_str )
int id;
char *name;
char *fg_str;
char *bg_str;
{
	Object_color *p;
	Display *dpy;
	int      scr;

	dpy  = XtDisplay( Top_level );
	scr  = DefaultScreen( dpy );

	p = (Object_color *) malloc( sizeof(Object_color) );
	if ( p == NULL )
	{
		fprintf( stderr, "create_object_color: Malloc failed\n");
		return( NULL );
	}

	p->id	   = id;
	p->name	   = strdup( name );
	p->fg_str  = strdup( fg_str );
	p->bg_str  = strdup( bg_str );

	p->bg      = BlackPixel(dpy, scr);
	p->fg      = WhitePixel(dpy, scr);
	p->next    = NULL;

	return( p );
}

lookup_object_color( obj )
Object_color *obj;
{
	Display *dpy;
	int      scr;
	Colormap cmap;
	XColor   color;
	XColor   exact;
	int      rc;

	dpy  = XtDisplay( Top_level );
	scr  = DefaultScreen( dpy );
	cmap = DefaultColormap( dpy, scr );

	/* Lookup Pixel colors based on character string symbolic names */

	rc = XAllocNamedColor( dpy, cmap, obj->fg_str, &color, &exact ); 
	if ( rc )
		obj->fg = color.pixel;
	else
	{
		fprintf(stderr, "Warning: Couldn't allocate color %s\n",
								obj->fg_str);
		obj->fg = WhitePixel(dpy, scr);
	}

	rc = XAllocNamedColor( dpy, cmap, obj->bg_str, &color, &exact );
	if ( rc )
		obj->bg = color.pixel;
	else
	{
		fprintf(stderr, "Warning: Couldn't allocate color %s\n", 
								obj->bg_str);
		obj->bg = BlackPixel(dpy, scr);
	}
}

Object_color
*push_object_list( olist, new )
Object_color *olist;
Object_color *new;
{
	if ( olist != NULL )
		new->next = olist;
	return( new );
}

Object_color
*parse_object_colors( def_str )
char *def_str;
{
	Object_color *olist=NULL;
	Object_color *new=NULL;
	char	     *tok;
	char	     *name=NULL_STR;
	char	     *fg_str=NULL_STR;
	char	     *bg_str=NULL_STR;

	/* Parse the string, building a list of object color nodes. */


	tok = strtok( def_str, ", 	" );
	while ( tok != NULL_STR )
	{
		name = strdup(tok);
		tok = strtok( NULL_STR, ", 	" );
		if ( tok == NULL_STR )
			break;

		fg_str = strdup(tok);

		tok = strtok( NULL_STR, ", 	" );
		if ( tok == NULL_STR )
			break;

		bg_str = strdup(tok);
		
		new = create_object_color( 0, name, fg_str, bg_str );

		lookup_object_color( new );

		olist = push_object_list( olist, new );

		free( name ); free( fg_str ); free( bg_str );
		name = NULL_STR; fg_str = NULL_STR; bg_str = NULL_STR;
		tok = strtok( NULL_STR, ", " );  /* Get beginning of next rec*/
	}

	return( olist );
}

void
match_color_to_username( users, user_colors )
User_group_info  *users;
Object_color *user_colors;
{
	Object_color *p;

	if ( users == NULL || user_colors == NULL )
		return;

	for ( p = user_colors ; p != NULL; p = p->next )
		p->id = find_uid_by_name( users, p->name );
}

void
match_color_to_groupname( users, group_colors )
User_group_info  *users;
Object_color *group_colors;
{
	Object_color *p;

	if ( users == NULL || group_colors == NULL )
		return;

	for ( p = group_colors ; p != NULL; p = p->next )
		p->id = find_gid_by_name( users, p->name );
}

print_object_list( obj_list )
Object_color *obj_list;
{
	Object_color *p;

	for ( p = obj_list ; p != NULL; p = p->next )
	{
		fprintf(stderr,"name(%s), uid(%d), fg(%s)(%d), bg(%s)(%d)\n",
					p->name, p->id,
					p->fg_str, p->fg,
					p->bg_str, p->bg );
	}
}

Object_color 
*find_obj_color_by_id( group_colors, id )
Object_color *group_colors;
int id;
{
	Object_color *p;

	if ( group_colors == NULL )
		return NULL;

	for ( p = group_colors ; p != NULL; p = p->next )
	{
		if ( p->id == id )
			return( p );
	}

	return( NULL );
}

cvt_cbo_str( str )
char *str;
{
	int rc=1;

	if ( strcmp( str, "real_user_id" ) == 0 )
	{
		rc = COLOR_BY_REAL_USER_ID;
		Dynamic_colors = 0;
	}
	else if ( strcmp( str, "effective_user_id" ) == 0 )
	{
		rc = COLOR_BY_EFFECTIVE_USER_ID;
		Dynamic_colors = 0;
	}
	else if ( strcmp( str, "real_group_id" ) == 0 )
	{
		rc = COLOR_BY_REAL_GROUP_ID;
		Dynamic_colors = 0;
	}
	else if ( strcmp( str, "effective_group_id" ) == 0 )
	{
		rc = COLOR_BY_EFFECTIVE_GROUP_ID;
		Dynamic_colors = 0;
	}
	else if ( strcmp( str, "load" ) == 0 )
	{
		rc = COLOR_BY_LOAD;
		Dynamic_colors++;
	}
	else if ( strcmp( str, "total_cpu_time" ) == 0 )
	{
		rc = COLOR_BY_TOTAL_TIME;
		Dynamic_colors++;
	}
	else if ( strcmp( str, "status" ) == 0 )
	{
		rc = COLOR_BY_STATUS;
		Dynamic_colors++;
	}
	else if ( strcmp( str, "resident_pages" ) == 0 )
	{
		rc = COLOR_BY_RESIDENT_PAGES;
		Dynamic_colors++;
	}
	else if ( strcmp( str, "image_size" ) == 0 )
	{
		rc = COLOR_BY_IMAGE_SIZE;
		Dynamic_colors++;
	}
	else if ( strcmp( str, "priority" ) == 0 )
	{
		rc = COLOR_BY_PRIORITY;
		Dynamic_colors++;
	}

	return( rc );
}



/* --------------------- Range color map Routines ----------------- */


Range_color_map
*parse_map_colors( def_str )
char *def_str;
{
	Range_color_map	*map;
	int		 i, nitems;
	Object_color     obj;		/* Used to lookup colors */
	char	        *tok;
	char	        *name;
	char		*min_str;
	char		*max_str;

	DEBUG1(1, "parse_map_colors: def_str(%s)\n", def_str );

	map = (Range_color_map *) malloc( sizeof( Range_color_map ) );
	if ( map == NULL )
	{
		fprintf(stderr, "Malloc error: range_color_map\n");
		return( NULL );
	}

	map->count = 0;
	map->range = NULL;
	map->selections = False;

	nitems = MAX_COLOR_RANGES;   	/* Should be based on actual need */

	map->range = (Range_color *) malloc( nitems * sizeof( Range_color ) );
	if ( map->range == NULL )
	{
		fprintf(stderr, "Malloc error: range_colors\n");
		return( NULL );
	}

	DEBUG0(3, "parse_map_colors: Allocated 256 entries\n" );

	/* Parse the string, populating the range color map entries. */

	i = 0;
	tok = strtok( def_str, "," );
	while ( tok != NULL_STR )
	{
		name = strdup(tok);

		tok = strtok( NULL_STR, ", 	" );
		if ( tok == NULL_STR )
			break;
		min_str = strdup(tok);

		tok = strtok( NULL_STR, ", 	" );
		if ( tok == NULL_STR )
			break;
		max_str = strdup(tok);

		tok = strtok( NULL_STR, ", 	" );
		if ( tok == NULL_STR )
			break;
		obj.fg_str = strdup(tok);

		tok = strtok( NULL_STR, ", 	" );
		if ( tok == NULL_STR )
			break;
		obj.bg_str = strdup(tok);

		lookup_object_color( &obj );

		map->range[i].label = name;
		map->range[i].min   = atof( min_str );
		map->range[i].max   = atof( max_str );
		map->range[i].fg    = obj.fg;
		map->range[i].bg    = obj.bg;
		map->range[i].selected  = False;
		map->range[i].count = 0;

		i++; map->count++;

		free( min_str ); free( max_str); 
		free( obj.fg_str ); free( obj.bg_str );

		tok = strtok( NULL_STR, "," ); /*Get first tok of next record*/
	}

	return( map );
}



Range_color_map  
*current_color_map()
{
	Range_color_map  *map;

	switch ( Color_based_on )
	{
		case COLOR_BY_REAL_USER_ID:   
		case COLOR_BY_EFFECTIVE_USER_ID:   map = User_color_map; break;

		case COLOR_BY_REAL_GROUP_ID:   
		case COLOR_BY_EFFECTIVE_GROUP_ID:  map = Group_color_map; break;

		case COLOR_BY_TOTAL_TIME:     map = Time_color_map; break;
		case COLOR_BY_LOAD:  	      map = Load_color_map; break;
		case COLOR_BY_STATUS: 	      map = Status_color_map; break;
		case COLOR_BY_IMAGE_SIZE:     map = Image_color_map; break;
		case COLOR_BY_PRIORITY:       map = Priority_color_map; break;
		case COLOR_BY_RESIDENT_PAGES: map = Resident_color_map; break;

		default: map = &Default_color_map; break;
	}
	return( map );
}

/* Massage the statistics, choose the appropriate color mapping and lookup
 * colors.
 */

get_dynamic_color( t_ptr, fg, bg )
tnode *t_ptr;
Pixel *fg;
Pixel *bg;
{
	sysProcInfo 	  *pp;
	Process_tree_info *pti_ptr;
	lnode		  *l_ptr;
	Process_list_info *pli_ptr;
	timestruc_t	   *ot;
	double		   val;
	double		   run_time, sample_time;
	long		   sec, x_sec;
	int		   n;
	Range_color_map	   *map;
	Display *dpy;
	int      scr;
	dpy  = XtDisplay( Top_level );
	scr  = DefaultScreen( dpy );


	pti_ptr = (Process_tree_info *) t_ptr->data;
	if ( pti_ptr == NULL )
		return((int)NULL);

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
		return( (int) NULL);

	pli_ptr = (Process_list_info *) l_ptr->data;
	if ( pli_ptr == NULL )
		return( (int) NULL );

	map = &Default_color_map;   /* Don't remove */

	pp = pti_ptr->pp;
	if ( pp == NULL )
	{
	    val = 0.0;
	}
	else
	{
	    switch ( Color_based_on )
	    {
		case COLOR_BY_REAL_USER_ID:   

			val = pp->pr_uid;
			map = User_color_map; 
			break;

		case COLOR_BY_EFFECTIVE_USER_ID:   
		
			val = pli_ptr->euid;
			map = User_color_map; 
			break;

		case COLOR_BY_REAL_GROUP_ID:   

			val = pp->pr_gid;
			map = Group_color_map;
			break;

		case COLOR_BY_EFFECTIVE_GROUP_ID:  
	
			val = pli_ptr->egid;
			map = Group_color_map;
			break;
						

		case COLOR_BY_TOTAL_TIME: 
		
			if ( Time_color_map != &Default_color_map)
			{
				map = Time_color_map;
				val = (pp->pr_time.tv_sec * 1.0) +
				      ((pp->pr_time.tv_nsec/10000000) * 0.01);
			}
			else  /* Use a percentage */
			{
				 val=(100.0*(pp->pr_time.tv_sec+1)) /
			       /*    -----------------------------  */
					      Max_time;
			}

			break;
				

		case COLOR_BY_LOAD:  
				 
			if ( Time_color_map != &Default_color_map)
			{
			    val = pti_ptr->stats.current_load;
			    map = Load_color_map;
			}
			else  /* Use percentage of all process usage */
			{
			   /* Last_time - current_time */

			   val = 0.0;  /* XXX Add percentage calc */
			}

			break;


		case COLOR_BY_STATUS: 

			map = Status_color_map;

			switch( PROCESS_SNAME( pp ) ) {

                		case 'O': val = 95.0;   /* Running  */  break;
                		case 'R': val = 80.0;   /* Runable  */  break;
                		case 'A': val = 78.0;   /*Recently Ran*/ break;
                		case 'X': val = 70.0;   /* SXBRK    */  break;
                		case 'S': val = 50.0;   /* Sleeping */  break;
                		case 'I': val = 40.0;   /* Idle     */  break;
                                case 'D': val = 30.0;   /* Wait/Swap*/  break;
                		case 'T': val = 20.0;   /* Stopped  */  break;
                		case 'Z': val =  5.0;   /* Zombie   */  break;

                		default : val = 0;
                        		   break;
			}
			break;

		case COLOR_BY_IMAGE_SIZE:     
		
			if ( Image_color_map != &Default_color_map)
			{
				map = Image_color_map;
#ifdef SOLARIS_7
				val = pp->pr_size;
#else
				val = (pp->pr_size * PageSize ) / 1024;
#endif

			}
			else	/* Percentage */
			{
				val = ( 100.0 * pp->pr_size ) / 
				    /*  -------------------  */
					    Max_isize;
			}
		        break;

		case COLOR_BY_PRIORITY:       
		
			val = PROCESS_PRIORITY( pp );
			map = Priority_color_map;
			break;

		case COLOR_BY_RESIDENT_PAGES: 

			if ( Resident_color_map != &Default_color_map)
			{
				map = Resident_color_map;
#ifdef SOLARIS_7
				val = pp->pr_rssize;
#else
				val = (pp->pr_rssize * PageSize) / 1024;
#endif
			}
			else	/* Percentage */
			{
				val = ( 100.0 * pp->pr_rssize )/
				     /* ---------------------- */
					(1.0 *  Max_rsize );
			}
			break;

		default: val = 0;
			 map = &Default_color_map;
		 	 break;
	    }
	}


	n = find_range(map, val);

	if ( map->selections )
	{
		if ( map->range[n].selected )
		{
			*fg = map->range[n].fg;
			*bg = map->range[n].bg;
		}
		else  /* To filter out unselected ranges */
		{
			*fg = BlackPixel(dpy, scr);
			*bg = BlackPixel(dpy, scr);
		}
	}
	else
	{
		*fg = map->range[n].fg;
		*bg = map->range[n].bg;
	}

	if ( Color_bar_count_mode == True )
		map->range[n].count++;

	DEBUG4(7, "get_dynamic_color: node(%s) val(%g)\n", t_ptr->str_key, 
						val, *fg, *bg );
	return 0;
}

void
calc_load( pti_ptr, tv_now )
Process_tree_info *pti_ptr;
struct timeval    *tv_now;
{
	timestruc_t	   *ot;
	double		   val, run_time, sample_time;
	long		   sec, x_sec;
	sysProcInfo 	  *pp;

	pp = pti_ptr->pp;
	if ( pp == NULL )
	{
		pti_ptr->stats.current_load = 0.0;
		return;
	}

	ot    = &(pti_ptr->stats.last_time);
	sec   = pp->pr_time.tv_sec  - ot->tv_sec;
	x_sec = pp->pr_time.tv_nsec - ot->tv_nsec;
	val   = 0.0;

	if ( (sec + x_sec) != 0 )
	{
  	    run_time = ((double)sec) + 
    		(((double)(x_sec / 100000)) * 0.0001);

	    sec   = tv_now->tv_sec  - pti_ptr->stats.lastSampledAt.tv_sec;
	    x_sec = tv_now->tv_usec - pti_ptr->stats.lastSampledAt.tv_usec;

	    if ( (sec + x_sec) != 0 )
	    {
	        sample_time = ((double) sec ) +
	      		(((double)x_sec) * 0.000001);

	        val =  (run_time / sample_time) * 100.0;
	    }
	}


	pti_ptr->stats.last_load  = pti_ptr->stats.current_load;

	pti_ptr->stats.lastSampledAt.tv_sec  = tv_now->tv_sec;
	pti_ptr->stats.lastSampledAt.tv_usec = tv_now->tv_usec;

	ot->tv_sec     = pp->pr_time.tv_sec;
	ot->tv_nsec    = pp->pr_time.tv_nsec;

	pti_ptr->stats.current_load = val;
}

tnode
*set_dynamic_color( t_ptr, noop )
tnode *t_ptr;
int noop;
{
	Process_tree_info *pti_ptr;
	int	           n;
	Arg	           wargs[4];
	Pixel              fg;
	Pixel              bg;

	pti_ptr = (Process_tree_info *) t_ptr->data;
	if ( pti_ptr == NULL )
		return(NULL);

	if ( pti_ptr->state == NODE_HIDDEN )
		return;

	if ( pti_ptr->selected == True )
	{
		fg = Selected_node_foreground;
		bg = Selected_node_background;
	}
	else
	{
		get_dynamic_color( t_ptr, &fg, &bg );
	}

	DEBUG3(7, "set_dynamic_color: node(%s) fg(%d) bg(%d)\n", 
						t_ptr->str_key, fg, bg );
	n = 0;
	XtSetArg( wargs[n], XtNforeground, fg ); n++;
	XtSetArg( wargs[n], XtNbackground, bg ); n++;
	XtSetArg( wargs[n], XtNbackgroundFill, bg ); n++;
	XtSetValues( pti_ptr->label_w, wargs, n );

	/* If decorated tree is using dynamic coloring then do it here.
	 * for now we update non-decorated nodes in a decorated tree to
	 * work arround an apparent bug in drawingArea(min size is 1 ).
	 *
	 */

	if ( Decorated_tree )
	{
	    n = 0;
	    XtSetArg( wargs[n], XtNbackgroundFill, bg ); n++;
	    XtSetValues( pti_ptr->node_w, wargs, n );


	    if ( ! pti_ptr->distinguished )
	    {
		n = 0;
		XtSetArg( wargs[n], XtNbackground, bg ); n++;
		XtSetArg( wargs[n], XtNforeground, bg ); n++;
		XtSetValues( pti_ptr->top_draw_w, wargs, n );

	    }

	    if ( ! pti_ptr->leader )
	    {
		n = 0;
		XtSetArg( wargs[n], XtNbackground, bg ); n++;
		XtSetArg( wargs[n], XtNforeground, bg ); n++;
		XtSetValues( pti_ptr->left_draw_w, wargs, n );
	    }

	}

	return( NULL );
}

tnode
*calc_stat_extents( t_ptr, noop )
tnode *t_ptr;
int noop;
{
	sysProcInfo 	  *pp;
	Process_tree_info *pti_ptr;

	pti_ptr = (Process_tree_info *) t_ptr->data;
	if ( pti_ptr == NULL )
		return(NULL);

	pp = pti_ptr->pp;
	if ( pp != NULL )
	{
	    switch ( Color_based_on )
	    {
		case COLOR_BY_LOAD:     
					if ( Reset_load_avg )
						pti_ptr->stats.last_load = 0;

					/* Fall through to get max time */

		case COLOR_BY_TOTAL_TIME:     
		
					if( pp->pr_time.tv_sec > Max_time )
						Max_time = pp->pr_time.tv_sec;
					break;


		case COLOR_BY_IMAGE_SIZE: if ( pp->pr_size > Max_isize )
						Max_isize = pp->pr_size;
					break;

		case COLOR_BY_RESIDENT_PAGES: 
		
					if ( pp->pr_rssize > Max_rsize )
						Max_rsize = pp->pr_rssize;
					break;
		default: 
		 	 break;
	    }
	}

 	return( NULL );
}

recalc_process_colors( )
{
	lnode *p;

	/* Calculate and display process colors based on current values */

	Max_isize=1; Max_rsize=1; Max_time=1;

	if ( Color_bar_averaging )
	    walk_tree( Head, calc_stat_extents, 0 );

	/* If counts are displayed, zero out counts so get_dynamic_color
	 * can set them.
	 */

	if ( Color_bar_count_mode == True )
		zero_range_counts();

	DEBUG3(9, "Max_isize(%ld) Max_rsize(%ld) Max_time(%ld)\n", Max_isize, 
						Max_rsize, Max_time );

	if ( Color_based_on == COLOR_BY_LOAD )
		Reset_load_avg = 0;
	else
		Reset_load_avg = 1;

	walk_tree( Head, set_dynamic_color, 0 );


	if ( Color_bar_count_mode == True )	/* To watch count changes */
		display_color_map();
}

setup_color_maps( data )
ApplicationData  *data;
{

        User_colors  = parse_object_colors( data->users_color_list  );
        Group_colors = parse_object_colors( data->groups_color_list );
        Stat_colors  = parse_object_colors( data->stats_color_list  );

        /* Resolve user_id's in User_colors using Users passwd entries */

        match_color_to_username( Users, User_colors );

        /* Resolve group_id's in Group_colors using Users group entries */

        match_color_to_groupname( Users, Group_colors );

	/* Manufacture psuedo color maps for User and groups		*/

	User_color_map  = create_psuedo_range_map( Users, USER, User_colors,
								Group_colors );

	Group_color_map = create_psuedo_range_map( Users, GROUP, User_colors,
								Group_colors );


	set_default_stat_color_map( Stat_colors );

	Time_color_map     = &Default_color_map;	
	Load_color_map     = &Default_color_map;	
	Status_color_map   = &Default_color_map;
	Image_color_map    = &Default_color_map;
	Priority_color_map = &Default_color_map;
	Resident_color_map = &Default_color_map;

	/* If any custom color maps are provided, load them */

	if ( strlen(data->load_color_map) > 0 )
	{
		DEBUG0(1, "Loading custom color map for: CPU load\n");
        	Load_color_map  = parse_map_colors( data->load_color_map );

		Load_color_map->label = "CPU Load";
	}

	if ( strlen(data->tcpu_color_map) > 0 )
	{
		DEBUG0(1,"Loading custom color map for: Total CPU Time\n");
        	Time_color_map  = parse_map_colors( data->tcpu_color_map );

		Time_color_map->label = "CPU Time";
	}

	if ( strlen(data->status_color_map) > 0 )
	{
		DEBUG0(1, "Loading custom color map for: Status\n");
        	Status_color_map  = parse_map_colors( data->status_color_map );

		Status_color_map->label = "Status";
	}

	if ( strlen(data->image_color_map) > 0 )
	{
		DEBUG0(1, "Loading custom color map for: Image Size\n");
        	Image_color_map  = parse_map_colors( data->image_color_map );

		Image_color_map->label = "Image";
	}

	if ( strlen(data->priority_color_map) > 0 )
	{
	    DEBUG0(1, "Loading custom color map for: Priority\n");
            Priority_color_map  = parse_map_colors( data->priority_color_map );

	    Priority_color_map->label = "Priority";
	}

	if ( strlen(data->resident_color_map) > 0 )
	{
	    DEBUG0(1, "Loading custom color map for: Resident Pages\n");
            Resident_color_map  = parse_map_colors( data->resident_color_map );

	    Resident_color_map->label = "Resident";
	}
}


Range_color_map	
*create_psuedo_range_map( users, type, user_colors, group_colors )
User_group_info  *users;
int		  type;
Object_color     *user_colors;
Object_color     *group_colors;
{
	Object_color	 *obj;
	Range_color_map	 *map;
	int		  i, gid, nitems;
	lnode		 *list;
	lnode	 	 *p;
	Pixel		  fg, bg;
	Display *dpy;
	int      scr;

	dpy  = XtDisplay( Top_level );
	scr  = DefaultScreen( dpy );

	map = (Range_color_map *) malloc( sizeof( Range_color_map ) );
	if ( map == NULL )
	{
		fprintf(stderr, "Malloc error: range_color_map\n");
		return( NULL );
	}

	map->count = 0;

	if ( type == GROUP )
	{
		map->label = "Group";
		list = sorted_group_list( users, &nitems );
	}
	else
	{
		map->label = "User";
		list = sorted_passwd_list( users, &nitems );
	}

	map->selections = False;
	map->range = (Range_color *) malloc( nitems * sizeof( Range_color ) );
	if ( map->range == NULL )
	{
		fprintf(stderr, "Malloc error: range_colors\n");
		return( NULL );
	}

	DEBUG1(1, "create_psuedo_range_map: Allocated (%d) entries\n", nitems );

	/* Scan the list and create color map entries. */

	i = 0;
	for ( p = list ; p != NULL_LNODE_PTR ; p = p->next )
	{
		map->range[i].label = strdup( (( char *) p->data) );
		map->range[i].min   = p->key;
		map->range[i].max   = p->key;

		fg = WhitePixel(dpy, scr);
		bg = BlackPixel(dpy, scr);


		if ( type == USER )
		{
			obj = find_obj_color_by_id( user_colors, p->key);
			if ( obj )
			{
				fg = obj->fg;
				bg = obj->bg;
			}
			else	/* Inherit group color  */
			{
				gid = find_gid_by_uid( users, p->key );
				obj = find_obj_color_by_id( group_colors, gid);
				if ( obj )
				{
					fg = obj->fg;
					bg = obj->bg;
				}
			}
		}
		else
		{
			obj = find_obj_color_by_id( group_colors, p->key);
			if ( obj )
			{
				fg = obj->fg;
				bg = obj->bg;
			}
		}

		map->range[i].fg    = fg;
		map->range[i].bg    = bg;
		map->range[i].count = 0;
		map->range[i].selected = False;

		i++; map->count++;

		p->data = NULL;  /* So we don't free, providers data */
	}

	free_list( list );

	return( map );
}


set_default_stat_color_map( stat_colors )
Object_color *stat_colors;
{
	Object_color   *p;
	int	        i;
	int 		count=0;

	for ( p = stat_colors; p != NULL ; p = p->next )
		p->id = count++;

	Default_color_map.range = Default_range_colors;
	Default_color_map.count = count;
	Default_color_map.selections = False;
	Default_color_map.label = "Default Colors";

	set_linear_color_range( &Default_color_map, stat_colors, 100.0);

	if ( Debug > 0 )
	{
	    p = stat_colors;
	    for ( i = 0; i < count ; i++ )
	    {
		DEBUG3(1, "%s: range[%g-%g] ", p->name,
				        Default_color_map.range[i].min, 
					Default_color_map.range[i].max );

		DEBUG2(1, "fg(%d) bg(%d)\n", Default_color_map.range[i].fg, 
						Default_color_map.range[i].bg );
		p = p->next;
	    }
	}
}

set_linear_color_range( map, colors, max_val )
Range_color_map *map;
Object_color *colors;
double max_val;
{
	int i;
	double incr;
	Object_color *s;
	Range_color  *range;
	int n;
	char buf[40];

	n     = map->count;
	range = &(map->range[0]);

	if ( n > MAX_COLOR_RANGES )
		n = MAX_COLOR_RANGES;

	s = colors;

	incr = max_val / ( 1.0 * n );
	for ( i = 0 ; i < n ; i++ )
	{
		if ( i == 0 )
			range[i].min = 0;
		else
			range[i].min = range[i-1].max;

		range[i].max = range[i].min + incr;

		sprintf( buf, "%2.2g%% - %3.3g%%", range[i].min, range[i].max );

		range[i].label = strdup(buf);
		range[i].fg = s->fg;
		range[i].bg = s->bg;
		range[i].count = 0;
		range[i].selected = False;

		s = s->next;
	}
}

find_range( map, value )
Range_color_map  *map;
double value;
{
	int 		offset,n,i;
	int 		not_found=1;
	Range_color    *range;
	int 		count;
	int		maxValue;

	range = &(map->range[0]);
	count   = map->count;

	n = count/2;
	offset = n; i = 0;

	maxValue = count;

	while( not_found )
	{
		/* The following is needed when we are range checking
		 * user or group values. In these cases the value matches
		 * exactly, so they would fail the normal range test.
		 */

		if ( value == range[n].min )	/* DON'T REMOVE !!!!!  */
			return( n );

		if (  value >= range[n].min && value < range[n].max )
			return( n );

		offset = (1+offset)/2;
		if ( value < range[n].min )
		{
			if ( n == 1 )  			/* must be 0 */
				return( n-1 );   	/* Close as we got */

			n = n - offset;
		}
		else
		{
			if ( n >= (count) )
				return( count-1 );   /* Close as we got */

			n = n + offset;
		}

		/* Bogus map pointer?? or range spec */
		if ( n < 0 || n > maxValue || i++ > 12 )	
		{
			fprintf(stderr, "(%d)Bogus color range spec\n", i);
			fprintf(stderr, 
			      "Somewhere near entry(%d)(%s) of map(%s): \n",
			      			n,range[n].label, map->label );
			      
			fprintf(stderr, "Value(%g) fell in a hole!!\n", value);
			return( 0 );
		}
	}

	return( n );
}


void
get_node_colors( user_colors, group_colors, t_ptr, fg, bg )
Object_color  *user_colors;
Object_color  *group_colors;
tnode	      *t_ptr;
Pixel         *fg;
Pixel         *bg;
{
	sysProcInfo 	  *pp;
  	Object_color	  *obj;
	int		   uid=0;
	int		   gid=0;
	int		   euid=0;
	int		   egid=0;
	Process_tree_info *pti_ptr;
	Process_list_info *pli_ptr;
	lnode		  *l_ptr;
	Display *dpy;
	int      scr;

	dpy  = XtDisplay( Top_level );
	scr  = DefaultScreen( dpy );

	*fg = Node_foreground; 		/* default, from X resource */
	*bg = Node_background;


	pti_ptr = (Process_tree_info *) t_ptr->data;
	if ( pti_ptr == NULL )
		return;

	l_ptr = pti_ptr->l_ptr;
	if ( l_ptr == NULL )
		return;

	pli_ptr = (Process_list_info *)l_ptr->data;
	if ( pli_ptr == NULL )
		return;

	pp = pli_ptr->pp;
	if ( pp == NULL )
		return;

	uid  = pp->pr_uid;
	gid  = pp->pr_gid;

	egid = pli_ptr->egid;
	euid = pli_ptr->euid;

	switch ( Color_based_on )
	{
		case COLOR_BY_REAL_USER_ID:

			obj = find_obj_color_by_id( user_colors, uid);
			if ( obj )
			{
				*fg = obj->fg;
				*bg = obj->bg;
			}
			else	/* Inherit group color  */
			{
				/* Lookup the group for the uid */

				gid = find_gid_by_uid( Users, uid );
				obj = find_obj_color_by_id( group_colors, gid);
				if ( obj )
				{
					*fg = obj->fg;
					*bg = obj->bg;
				}
			}
			break;
	
		case COLOR_BY_EFFECTIVE_USER_ID:

			obj = find_obj_color_by_id( user_colors, euid);
			if ( obj )
			{
				*fg = obj->fg;
				*bg = obj->bg;
			}
			else	/* Inherit effective users group color  */
			{
				/* Lookup the group for the euid */

				gid = find_gid_by_uid( Users, euid );
				obj = find_obj_color_by_id( group_colors, gid);
				if ( obj )
				{
					*fg = obj->fg;
					*bg = obj->bg;
				}
			}
			break;
	
		case COLOR_BY_REAL_GROUP_ID:

			obj = find_obj_color_by_id( group_colors, gid);
			if ( obj )
			{
				*fg = obj->fg;
				*bg = obj->bg;
			}


			break;

		case COLOR_BY_EFFECTIVE_GROUP_ID:

			obj = find_obj_color_by_id( group_colors, egid);
			if ( obj )
			{
				*fg = obj->fg;
				*bg = obj->bg;
			}

			break;

		case COLOR_BY_LOAD:
		case COLOR_BY_TOTAL_TIME:
		case COLOR_BY_STATUS:
		case COLOR_BY_IMAGE_SIZE:
		case COLOR_BY_PRIORITY:
		case COLOR_BY_RESIDENT_PAGES:
						 get_dynamic_color(t_ptr,fg,bg);
						 break;

		default: break;
	}

}

void
set_node_colors( user_colors, group_colors, t_ptr )
Object_color  *user_colors;
Object_color  *group_colors;
tnode	      *t_ptr;
{
	Pixel               fg;
	Pixel               bg;
	int	            n;
	Arg	            wargs[4];
	Process_tree_info  *pti_ptr;


	/* Replace calls to set_dynamic_color with this? Cant so 
	 * replace calls to set_node_colors, or better yet change
	 * interface to set_node_colors so it can be passed to
	 * walk_tree 
	 */

	if ( t_ptr == NULL )
		return;
	
	pti_ptr = (Process_tree_info *)t_ptr->data;
	if ( pti_ptr == NULL )
		return;

	if ( pti_ptr->state == NODE_HIDDEN )
		return;

	if ( pti_ptr->selected == True )
	{
		fg = Selected_node_foreground;
		bg = Selected_node_background;
	}
	else
	{
		get_node_colors( user_colors, group_colors, t_ptr, &fg, &bg );
	}

	DEBUG3(7, "set_node_colors: node(%s) fg(%d) bg(%d)\n", t_ptr->str_key,
						fg, bg );

	n = 0;
	XtSetArg( wargs[n], XtNforeground, fg ); n++;
	XtSetArg( wargs[n], XtNbackground, bg ); n++;
	XtSetArg( wargs[n], XtNbackgroundFill, bg ); n++;

	XtSetValues( pti_ptr->label_w, wargs, n );

	if ( Decorated_tree )
	{
	    if ( ! pti_ptr->distinguished )
	    {
		n = 0;
		XtSetArg( wargs[n], XtNforeground, bg ); n++;
		XtSetArg( wargs[n], XtNbackground, bg ); n++;
		XtSetValues( pti_ptr->top_draw_w, wargs, n );
	    }

	    if ( ! pti_ptr->leader )
	    {
		n = 0;
		XtSetArg( wargs[n], XtNforeground, bg ); n++;
		XtSetArg( wargs[n], XtNbackground, bg ); n++;
		XtSetValues( pti_ptr->left_draw_w, wargs, n );
	    }

	    n = 0;
	    XtSetArg( wargs[n], XtNbackgroundFill, bg ); n++;
	    XtSetValues( pti_ptr->node_w, wargs, n );

	}
}




/* -------------------- Show colors routines -------------------- */


show_colors(parent)
Widget parent;
{

	if ( CB_dialog_state != DIALOG_DISPLAYED ) 
	{
		popup_color_bar( Top_level );
	}

	change_colors();
}

change_colors()
{
	Arg  wargs[2];

	if ( (Color_bar_shw == NULL) )  /* Needed sanity check */
		return;

	/* If the current color map in the color bar does not represent the
	 * current color map, change it.
	 */

	if ( Color_based_on != Color_bar )
	{
		switch ( Color_based_on )
		{
			case COLOR_BY_STATUS:
			case COLOR_BY_REAL_USER_ID:
			case COLOR_BY_EFFECTIVE_USER_ID:
			case COLOR_BY_REAL_GROUP_ID:
			case COLOR_BY_EFFECTIVE_GROUP_ID:

				if ( Color_bar_label_mode == 0 )
				{
					Color_bar_label_mode=1;
					XtSetArg( wargs[0], XmNset, True );
					XtSetValues( Color_bar_ltw, wargs, 1);
				}
				break;

			default:
				  break;
		}

		display_color_map( );
		Color_bar = Color_based_on;
	}
}


display_color_map()
{
	static int widgets_created=0;
	static int widgets_managed=0;
	int 		  i,j,k,f,n;
	Arg  		  wargs[10];
	Range_color_map   *map;		
	static Range_color_map   *last_map=NULL;		
	static int	  lastFieldCnt=0;
	static int	  lastColoring=0;
	int		  numFields;
	int		  nw=0;
	int		  len, pad, performanceHack=0;
	char 		  *title_str;
	char		  *fmt;
	char		   format[30];
	char		  *str[3];
	char		  buf1[40];
	char		  buf2[30];
	char		  buf3[30];
	char		  centeredString[40];
	static int	  maxWidgets=0;
	static Widget	  *widgets=NULL;
	struct winSizeInfo p;
	Dimension	  widget_width, widget_height;


	str[0] = &(buf1[0]); str[1] = &(buf2[0]); str[2] = &(buf3[0]);

	/* For each color in color map, display label and color coded max val*/

	n = 0;
        XtSetArg( wargs[n], XtNwidth,  &widget_width  ); n++;
        XtSetArg( wargs[n], XtNheight, &widget_height ); n++;
        XtGetValues( Color_bar_sfw, wargs, n );

	p.last_width = widget_width;
	p.last_height= widget_height;

	map = current_color_map();

	title_str = get_color_menu_str( Color_based_on );

	if ( title_str == NULL_STR )
		xs_wprintf( Color_bar_lw, "%s", map->label );
	else	
		xs_wprintf( Color_bar_lw, "%s", title_str );

	/* Note: could probably remove the widgets array and calculate
	 * the ranges of widgets to be managed/unmanaged then use address
	 * from start of range.
	 */

	if ( map->count > maxWidgets )
	{
	    if ( widgets )
	        XtFree( (char *)widgets );

	    widgets = (Widget *) XtMalloc( map->count * sizeof( Widget ) );

	    Color_bar_w = (Widget *) XtRealloc((char *) Color_bar_w,
	    					map->count * sizeof( Widget ) );
	    if ( Color_bar_w == NULL )
	    {
		DEBUG0( 0, "display_color_map: failed to malloc for widgets\n");
		return; 
	    }

	    maxWidgets = map->count;
	}

	if ( widgets_managed > 0 )
	{
	    if ( map->count > 50 || widgets_managed > 50 )
	    {
	    	/* Updating labels appears to take N! time, so if there
		 * are more than 50 we unmanage them all to stop the
		 * layout thrashing about. The effect is to shrink the
		 * window real small and then enlarge again. I find
		 * this displeasing, hence the minimal managing code.
		 */

		/* If the color map is the same as the last call and
		 * all we are doing is updating the labels, then in
		 * such case we don't do the performance hack
		 */

		numFields = Color_bar_label_mode + Color_bar_range_mode + 
							Color_bar_count_mode;

		if ( map == User_color_map && last_map == User_color_map )
		{
		    if ( lastColoring != Color_based_on )
		    {
		    	if ( ( numFields == 0 ) ||
				(( Color_bar_range_mode == 0 ) &&
				 ( Color_bar_count_mode == 0 ) )    )
			{
				/* Don't need to do anything !! */
				lastColoring = Color_based_on;
				return ;
			}

		    	performanceHack++;
		    }
		}

		if ( map == Group_color_map && last_map == Group_color_map )
		{
		    if ( lastColoring != Color_based_on )
		    {
		    	if ( ( numFields == 0 ) ||
				(( Color_bar_range_mode == 0 ) &&
				 ( Color_bar_count_mode == 0 ) )    )
			{
				/* Don't need to do anything !! */
				lastColoring = Color_based_on;
				return ;
			}
		    	performanceHack++;
		    }
		}

		if ( map != last_map )
		    performanceHack++;

		if ( numFields != lastFieldCnt )
		    performanceHack++;

		if ( performanceHack )
		    XtUnmanageChildren( Color_bar_w, widgets_managed );
	    }
	}

	for ( i = 0 ; i < map->count ; i++ )
	{
	    if ( i >= widgets_managed )
	    {
		if ( i >= widgets_created )
		{
		    n = 0;
		    if ( i == 0 )
		    {
		        XtSetArg(wargs[n],XmNtopAttachment,XmATTACH_FORM);n++;
		    }
		    else
		    {
		        XtSetArg(wargs[n],XmNtopAttachment,XmATTACH_WIDGET);n++;
		        XtSetArg(wargs[n],XmNtopWidget,Color_bar_w[i-1]);n++;
		    }
		    XtSetArg(wargs[n],XmNleftAttachment,XmATTACH_FORM);n++;
		    XtSetArg(wargs[n],XmNrightAttachment,XmATTACH_FORM);n++;
		    Color_bar_w[i] = XtCreateWidget( "range_label", 
					xmLabelWidgetClass, 
					Color_bar_sfw, wargs, n);

		    /* Add event to do_range_select with offset i */
		    XtAddEventHandler(Color_bar_w[i], ButtonPressMask,
					FALSE, cb_range, (XtPointer) i);
		}

		widgets[nw++] = Color_bar_w[i];  /* Save in list to Manage */
	    }

	    f = 0;
	    if ( Color_bar_label_mode )
	    {
	        sprintf( str[f++], "%s", map->range[i].label );
		
	    	if ( Color_bar_range_mode || Color_bar_count_mode )
		{
			len = strlen( str[0] );

			str[0][len]   = ':';
			str[0][len+1] = NULL_CHAR;
		}
	    }

	    if ( Color_bar_range_mode )
	        sprintf( str[f++], "%g", map->range[i].max );

	    if ( Color_bar_count_mode )
	        sprintf( str[f++], "%d", map->range[i].count );

	    if ( f == 0 )  /* Gotta have something!!! so label it */
	    {
	        sprintf( str[f++], "%s", map->range[i].label );

		/* Should force the label toggle, but I guess the user 
		 * will figure it out.
		 */
	    }

	    n = 0;
	    if ( f > 1 )  /* Create a left justifyable string */
	    {
	        /* XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_BEGINNING ); n++; */

		if ( f == 2 )
			fmt = "%-20s  %9s";
		else
			fmt = "%-20s  %9s  %9s";
	    }
	    else	/* Center string in label widget */
	    {
	        /*
	        XtSetArg( wargs[n], XmNalignment, XmALIGNMENT_CENTER ); n++;
		*/

		/* Center the string in a fixed width field */

		len = strlen( str[0] );

		pad = (20 - len)/2;
		for ( j = 0 ; j < pad ; j++ )
			centeredString[j] = ' ';
		
		for ( k = 0 ; k < len ; k++ )
		    centeredString[j++] = str[0][k];

		for ( ; j < 20 ; j++ )
			centeredString[j] = ' ';

		centeredString[j] = (char) 0;

		strcpy( str[0], centeredString );
	        fmt = "%s";
	    }

	    if ( map->selections )
	    {
	    	if ( map->range[i].selected == True )
			sprintf( format, "(+)%s", fmt );
	    	else
			sprintf( format, "   %s", fmt );
	    }
	    else
		sprintf( format, "%s", fmt );

	    xs_wprintf( Color_bar_w[i], format, str[0], str[1], str[2] );

	    /* Set color for range */

	    XtSetArg( wargs[n], XtNforeground, map->range[i].fg ); n++;
	    XtSetArg( wargs[n], XtNbackground, map->range[i].bg ); n++;
	    XtSetValues( Color_bar_w[i], wargs, n );
	}


	if ( performanceHack )
	{
	    XtManageChildren( Color_bar_w, i );
	}
	else
	{
	    if ( ( widgets_managed - i ) > 0 )
		XtUnmanageChildren( &(Color_bar_w[i]), widgets_managed - i );

	    if ( nw > 0 )
		XtManageChildren( widgets, nw );
	}

	widgets_managed = i;
	if ( i > widgets_created )
		widgets_created = i;

	lastFieldCnt = Color_bar_label_mode + Color_bar_range_mode + 
							Color_bar_count_mode;

	last_map     = map;
	lastColoring = Color_based_on;

	n = 0;
        XtSetArg( wargs[n], XtNwidth,  &widget_width  ); n++;
        XtSetArg( wargs[n], XtNheight, &widget_height ); n++;
        XtGetValues( Color_bar_sfw, wargs, n );

	p.width = widget_width;
	p.height= widget_height;

	cbShellResize( &p );
}

void
cbShellResize( p )
struct winSizeInfo *p;
{
	Dimension width, height, border;
	Dimension main_window_width, main_window_height;
	Position  x, y;
	int delta_width, delta_height;
	Dimension new_width, new_height;
	int change_size=0;
	int change_x=0;
	int change_y=0;
	int n;
	Arg wargs[6];
	struct dimensionalOffsets offset;

	if ( ! XtIsRealized( Color_bar_shw ) )
		return;


	DEBUG4( 5, "cbShellResize from h(%d) v(%d) TO h(%d) v(%d):\n", 
					p->last_width, p->last_height,
					p->width, p->height );

	width  = p->width;
	height = p->height;
	border = 0;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &main_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &main_window_height ); n++;
	XtSetArg( wargs[n], XtNx, &x  ); n++;
	XtSetArg( wargs[n], XtNy, &y ); n++;
	XtGetValues( Color_bar_shw, wargs, n );

	delta_width  = (int) p->width  - (int) p->last_width;
	delta_height = (int) p->height - (int) p->last_height;

	DEBUG4(5,"cbShellResize Window x(%d),y(%d) Width(%d) height(%d) \n", 
					x, y,
					main_window_width,
					main_window_height );


	DEBUG2( 5, "cbShellResize Delta Width(%d) height(%d) \n", 
					delta_width, delta_height );


	cbPreResizeWindowOffsets( &offset );

	new_width = p->width + offset.left + offset.right;

	change_size++;
	if ( delta_width > 0 )
	{
	    if ( Max_window_x != -1 )
	    {
	        if ( main_window_width == Max_window_x ) /* Already max */
	        {
		    new_width = Max_window_x;  /* In case height changes */
		    change_size--;
	        }
	    }
	}
	else if ( delta_width < 0 )
	{
	    if ( Min_window_x != -1 )
	    {
	        if ( main_window_width == Min_window_x ) /* Already min */
	        {
		    new_width = Min_window_x;  /* In case height changes */
		    change_size--;
	        }
	    }
	}
	else
	{
	    change_size = 0;
	}

	if ( Max_window_x != -1 )
	{
	    if ( new_width > Max_window_x )
		new_width = Max_window_x;
	}
	if ( Min_window_x != -1 )
	{
	    if ( new_width < Min_window_x )
		new_width = Min_window_x;
	}

	if ( FitToScreen )
	{
	    if ( ( x + new_width ) > ( ActualScreenWidth - RightOffset) )
	    	new_width = ( ActualScreenWidth - x ) - RightOffset;
	}


	new_height = p->height + offset.top + offset.bottom;


	change_size++;

	if ( delta_height > 0 )
	{
	    if ( Max_window_y != -1 )
	    {
	        if ( main_window_height == Max_window_y )
	        {
		    new_height = Max_window_y;
		    change_size--;
	        }
	    }
	}
	else if ( delta_height < 0 )
	{
	    if ( Min_window_y != -1 )
	    {
	        if ( main_window_height == Min_window_y )
	        {
		    new_height = Min_window_y;
		    change_size--;
	        }
	    }
	}
	else
	{
	    change_size--;
	}

	if ( Max_window_y != -1 )
	{
	    if ( new_height > Max_window_y )
		new_height = Max_window_y;
	}
	if ( Min_window_y != -1 )
	{
	    if ( new_height < Min_window_y )
		new_height = Min_window_y;
	}

	if ( FitToScreen )
	{
	    if ( ( y + new_height ) > ( ActualScreenHeight - BottomOffset ) )
	    	new_height = ( ActualScreenHeight - y ) - BottomOffset;
	}


	if ( change_size )
	{
		DEBUG2(8, "Change main window size to h(%d) v(%d)\n", 
						new_width, new_height );

		XtResizeWidget( Color_bar_shw, new_width, new_height, border );
	}
}

void
cbPreResizeWindowOffsets(offset)
struct dimensionalOffsets *offset;
{
	Position main_window_x, main_window_y;
	Position clip_window_x, clip_window_y;
	Position mx, my, cx, cy;
	Widget   clip_w;     	    /* Scrolled Windows clip/client widget */
	Dimension main_window_width, main_window_height;
	Dimension clip_window_width, clip_window_height;
	Position mlr_x, mlr_y, clr_x, clr_y;
	int n;
	Arg wargs[6];

	n = 0;
	XtSetArg( wargs[n], XtNx, &main_window_x  ); n++;
	XtSetArg( wargs[n], XtNy, &main_window_y ); n++;
	XtGetValues( Color_bar_shw, wargs, n );

	XtTranslateCoords( Color_bar_shw, main_window_x, main_window_y, &mx, &my);
	

	n = 0;
	XtSetArg( wargs[n], XmNclipWindow, &clip_w ); n++;
	XtGetValues( Color_bar_sw, wargs, n );


	n = 0;
	XtSetArg( wargs[n], XtNx, &clip_window_x  ); n++;
	XtSetArg( wargs[n], XtNy, &clip_window_y ); n++;
	XtGetValues( clip_w, wargs, n );

	XtTranslateCoords( clip_w , clip_window_x, clip_window_y, &cx, &cy);


	offset->top    = cy - my;
	offset->left   = cx - mx;

	/* Now get bottom right offsets */

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &main_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &main_window_height ); n++;
	XtGetValues( Color_bar_shw, wargs, n );

	mlr_x = mx + main_window_width;
	mlr_y = my + main_window_height;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &clip_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &clip_window_height ); n++;
	XtGetValues( clip_w, wargs, n );

	clr_x = cx + clip_window_width;
	clr_y = cy + clip_window_height;

	offset->bottom  = mlr_y - clr_y;
	offset->right   = mlr_x - clr_x;


	DEBUG4(7, "Offsets top(%d) left(%d) bottom(%d) right(%d)\n", 
				offset->top, offset->left, 
				offset->bottom, offset->right          );
}



setCmapPopdownCallback( popdown_cb )
void (*popdown_cb) ();
{
        CB_popdownCB = popdown_cb;
}


static void
deleteWindowHandler(w, p, e, b)
Widget w;
XtPointer p;
XEvent *e;
Boolean *b;
{
	if (((XClientMessageEvent *)e)->type != ClientMessage)
	{
              DEBUG1( 1, "Not Client Message: type(%d)\n",
		 	    ((XClientMessageEvent *)e)->type );
              return;
	}

	if ((((XClientMessageEvent *)e)->message_type == protocols) &&
               (((XClientMessageEvent *)e)->data.l[0] == deleteWindow)) 
	{
              DEBUG2( 1, "Got Delete Message: type(%d) code(%d)\n",
		((XClientMessageEvent *)e)->message_type,
                ((XClientMessageEvent *)e)->data.l[0]  );


		 /* Dialog is unmapped by protocol handling code */

		 CB_dialog_state = DIALOG_HIDDEN;

 		if ( CB_popdownCB != NULL )
                    (*CB_popdownCB)();

	}
	else
	{
              DEBUG2( 1, "Not right Message: type(%d) code(%d)\n",
		((XClientMessageEvent *)e)->message_type,
                ((XClientMessageEvent *)e)->data.l[0] );

		return;
	}


	return;

}

void
popdown_color_bar( )
{
	if ( Color_bar_shw == NULL )
                return;

        if ( CB_dialog_state == DIALOG_DISPLAYED )
        {
                DEBUG0( 1, "popdown_color_bar: poping down\n");

                XtPopdown( Color_bar_shw );

		CB_dialog_state = DIALOG_HIDDEN;
                return;
        }
}


Widget
popup_color_bar(parent)
Widget parent;
{
	Widget  	 	 color_shell;
	Widget			 form_w, sep1, sfw, sep2, b;
	Arg		 	 wargs[18];
	int		 	 n;
	char			 title_str[80];
	Pixmap			 icon_pixmap;
	Display			*display;
	Screen			*screen;
	Window			 icon_win;
	Dimension		 height;


	if ( Color_bar_shw )
	{
		DEBUG0( 1, "popup_color_bar: already created\n");
		
		if ( CB_dialog_state == DIALOG_HIDDEN )
		{
		    DEBUG0( 1,"popup_color_bar: Was hidden so calling Popup\n");
		    XtPopup( Color_bar_shw, XtGrabNone ); 
		    CB_dialog_state = DIALOG_DISPLAYED;
		}
		return( Color_bar_shw ); 
	}

	/* If we get here we need to create an app shell with a form widget
	 * which then contains the appropriate widgets to display the 
	 * color bar
	 */

	sprintf( title_str, "%s", "ColorMap" );

	display = XtDisplay(parent);
	screen  = XtScreen(parent);


	icon_pixmap = getDeepIconByName( parent, "colorBar", 
                          			color_bar48_bits, 48 );

	n = 0;
	XtSetArg( wargs[n], XmNtitle, title_str); n++;
	XtSetArg( wargs[n], XmNiconName, "ColorMap"); n++;
	XtSetArg( wargs[n], XmNiconPixmap, icon_pixmap); n++;
	XtSetArg( wargs[n], XmNdeleteResponse, XmUNMAP); n++;

	/*
	color_shell = XtCreateApplicationShell( "color_bar", 
					   topLevelShellWidgetClass,
					   wargs, n);
	*/

	color_shell = XtAppCreateShell( "processColorMap", "ProcessColorMap",
					   topLevelShellWidgetClass,
					   display,
					   wargs, n);
	Color_bar_shw = color_shell;



 	/* The following are for the Window Delete Event handler */

        protocols = XInternAtom( XtDisplay(color_shell), "WM_PROTOCOLS", False);

        deleteWindow = XInternAtom( XtDisplay(color_shell), "WM_DELETE_WINDOW",
                                                                        False);



	if ( XmIsMotifWMRunning( color_shell ) )
	{
		DEBUG0(1, "popup_color_bar: Motif Window Manager\n");
	}
	else
	{
		DEBUG0(1, "popup_color_bar: No Motif Window Manager\n");
	}


	n = 0;
	XtSetArg( wargs[n], XmNfractionBase, 96 ); n++;
	form_w = XmCreateForm( color_shell, "cbform", wargs, n);
	if ( form_w == NULL )
	{
		DEBUG0( 0, "popup_color_bar: failed to create form\n");
		return( NULL ); 
	}

	n = 0;
	Color_bar_lw = XtCreateWidget("cbLabel",xmLabelWidgetClass, 
						form_w, wargs, n );


	n = 0;
	XtSetArg( wargs[n], XmNheight, &height); n++;
	XtGetValues( Color_bar_lw, wargs, n );

	height = height + 4;

	/* Create color map arrow buttons, Note must be smaller in hieght
	 * than label  
	 */

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNleftOffset, 1); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNtopOffset, 1); n++;
	XtSetArg( wargs[n], XmNheight, height); n++;
	XtSetArg( wargs[n], XmNwidth, height); n++;
	XtSetArg( wargs[n], XmNarrowDirection, XmARROW_LEFT ); n++;
	Color_bar_law = XtCreateManagedWidget("cbl_arrow",
						xmArrowButtonGadgetClass, 
						form_w, wargs, n );

	XtAddCallback( Color_bar_law, XmNactivateCallback, cb_mv, 
						(XtPointer) XmARROW_LEFT);

	n = 0;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightOffset, 1); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNtopOffset, 1); n++;
	XtSetArg( wargs[n], XmNheight, height); n++;
	XtSetArg( wargs[n], XmNwidth, height); n++;
	XtSetArg( wargs[n], XmNarrowDirection, XmARROW_RIGHT ); n++;
	Color_bar_raw = XtCreateManagedWidget("cbr_arrow",
						xmArrowButtonGadgetClass, 
						form_w, wargs, n );

	XtAddCallback(Color_bar_raw, XmNactivateCallback, cb_mv, 
						(XtPointer) XmARROW_RIGHT);

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNleftWidget, Color_bar_law ); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNrightWidget, Color_bar_raw ); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, Color_bar_law); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, Color_bar_law); n++;
	XtSetValues( Color_bar_lw, wargs, n );

	XtManageChild( Color_bar_lw );

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, Color_bar_lw); n++;
	sep1 = XtCreateManagedWidget( "cbseparator1", xmSeparatorGadgetClass, 
						form_w, wargs, n );


	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 1 ); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 31 ); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNindicatorOn, False ); n++;
	XtSetArg( wargs[n], XmNshadowThickness, 2 ); n++;
	XtSetArg( wargs[n], XmNindicatorType, XmN_OF_MANY ); n++;
	XtSetArg( wargs[n], XmNset, True ); n++;
	b = XtCreateManagedWidget( "Label", xmToggleButtonWidgetClass, 
						form_w, wargs, n );

	Color_bar_ltw = b;

	XtAddCallback( b, XmNvalueChangedCallback, cb_mode, 
						(XtPointer) LABEL_MODE );

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 33 ); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 63 ); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNindicatorOn, False ); n++;
	XtSetArg( wargs[n], XmNshadowThickness, 2 ); n++;
	XtSetArg( wargs[n], XmNindicatorType, XmN_OF_MANY ); n++;
	b = XtCreateManagedWidget( "Range", xmToggleButtonWidgetClass, 
						form_w, wargs, n );

	XtAddCallback( b, XmNvalueChangedCallback, cb_mode, 
						(XtPointer) RANGE_MODE );

	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNleftPosition, 65 ); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg( wargs[n], XmNrightPosition, 95 ); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
	XtSetArg( wargs[n], XmNindicatorOn, False ); n++;
	XtSetArg( wargs[n], XmNshadowThickness, 2 ); n++;
	XtSetArg( wargs[n], XmNindicatorType, XmN_OF_MANY ); n++;
	b = XtCreateManagedWidget( "Count", xmToggleButtonWidgetClass, 
						form_w, wargs, n );

	XtAddCallback( b, XmNvalueChangedCallback, cb_mode, 
						(XtPointer) COUNT_MODE );


	n = 0;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, b); n++;
	sep2 = XtCreateManagedWidget( "cbseparator2", xmSeparatorGadgetClass, 
						form_w, wargs, n );


	/* Create scrolled window */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment,XmATTACH_WIDGET);n++;
	XtSetArg( wargs[n], XmNtopWidget, sep1); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNbottomWidget, sep2); n++;
	Color_bar_sw = XtCreateWidget( "cbslist",  xmScrolledWindowWidgetClass,
						form_w, wargs, n );

	/* Create form for interior of scrolled window */
	n = 0;
	XtSetArg( wargs[n], XmNfractionBase, 10 ); n++;
	sfw = XmCreateForm( Color_bar_sw, "cbsform", wargs, n);
	if ( form_w == NULL )
	{
		DEBUG0( 0, "popup_color_bar: failed to create scroll form\n");
		return( NULL ); 
	}

	Color_bar_sfw = sfw;

	/* Create space for color bar widgets, Color_bar_mode controls the
	 * content. 
	 */

	
	Color_bar_w = (Widget *) malloc( MAX_COLOR_RANGES * sizeof( Widget ) );
	if ( Color_bar_w == NULL )
	{
		DEBUG0( 0, "popup_color_bar: failed to malloc for widgets\n");
		return( NULL ); 
	}

	XtManageChild( Color_bar_sfw );
	XtManageChild( Color_bar_sw );
	XtManageChild( form_w );

	XtPopup( Color_bar_shw, XtGrabNone );


        /* Add Window Delete Event handler */

        XSetWMProtocols( XtDisplay(color_shell), XtWindow(color_shell),
                                        &deleteWindow, 1 );

        XtAddRawEventHandler( color_shell, 0, True, deleteWindowHandler, NULL );

	CB_dialog_state = DIALOG_DISPLAYED;

	return( Color_bar_shw ); 
}

void
cb_mv( w, direction, call_data )
Widget w;
int    direction;
XmArrowButtonCallbackStruct     *call_data;
{
	int color_by;
	char *str=NULL_STR;
	char noWhiteSpaceStr[40];
	int i, len;

	color_by = Color_based_on;

	if ( direction == XmARROW_LEFT )
	{
		color_by--;
		if ( color_by <= NO_COLOR )
			color_by = COLOR_BY_PRIORITY;
		
	}
	else if ( direction == XmARROW_RIGHT )
	{
		color_by++;
		if ( color_by > COLOR_BY_PRIORITY )
			color_by = COLOR_BY_REAL_USER_ID;
	}
	else
	{
		color_by = COLOR_BY_REAL_USER_ID;
	}

	str = get_color_menu_str( color_by );

	len = strlen( str );
	for ( i = 0 ; i < len; i++ )  	/* Remove leading white space */
	{
	    if ( str[i] != ' ' )
	    	break;
	}

	strcpy( noWhiteSpaceStr, &(str[i]) );

	len = strlen( noWhiteSpaceStr );

	for ( i = len-1; i > 0; i-- )	/* Remove trailing white space */
	{
	    if ( noWhiteSpaceStr[i] == ' ' )
	        noWhiteSpaceStr[i] = NULL_CHAR;
	    else
	    	break;
	}



	set_menu_toggle( Main_menu, noWhiteSpaceStr, True );
}

void
cb_mode( w, mode, call_data )
Widget w;
int    mode;
XmAnyCallbackStruct     *call_data;
{

	if( mode == LABEL_MODE )
	{
		if ( Color_bar_label_mode )
			Color_bar_label_mode = 0;
		else
			Color_bar_label_mode = 1;
	}
	else if( mode == RANGE_MODE )
	{
		if ( Color_bar_range_mode )
			Color_bar_range_mode = 0;
		else
			Color_bar_range_mode = 1;
	}
	else if( mode == COUNT_MODE )
	{
		if ( Color_bar_count_mode )
			Color_bar_count_mode = 0;
		else
		{
			Color_bar_count_mode = 1;
			recalc_process_colors();  /* So we have counts */
		}
	}

	if ( Color_bar_count_mode == False )/*If true, display done in recalc*/
		display_color_map();
}

void
cb_range( w, i, event )
Widget w;
int i;
XEvent *event;
{
	Range_color_map   *map;		
	int		   j; 
	int		   num_selected=0;

	map = current_color_map();

	if ( map->selections )
	{
		/* If we are unsetting last one we need to unset the flag */

		for ( j = 0 ; j < map->count ; j++ )
		{
			if ( map->range[j].selected == True )
				num_selected++;
		}

		if ( map->range[i].selected == True )
		{
			map->range[i].selected = False;

			if ( num_selected == 1 )  /* We just unset the last */
				map->selections	= False;
		}
		else
		{
			map->range[i].selected = True;
		}
	}
	else
	{
		map->selections 	= True;
		map->range[i].selected  = True;
	}

	if ( Color_bar_count_mode == False )
		display_color_map();		/* Redisplay to see selection */

	recalc_process_colors();	/* If count true, does display */
}

zero_range_counts()
{
	Range_color_map   *map;		
	int		   i;

	map = current_color_map();

	for ( i = 0 ; i < map->count ; i++ )
		map->range[i].count = 0;
}
