/* File: 	user_group_tree.c
 *
 * Description: Contains code to load, display and select users and groups
 *
 * Author:	George MacDonald
 *
 * Copyright:	Copyright (c) 1993, Pulsar Software Inc.
 *		All Rights Reserved, Unpublished rights reserved.
 *
 * History:	George MacDonald	2/9/95	Created
 *            
 *
 */

#include <stdio.h>		
#include <varargs.h>
#include <pwd.h>
#include <grp.h>
#include <utmp.h>

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/ScrolledW.h>
#include <Xm/Label.h>
#include <Xm/SeparatoG.h>
#include <Xm/PushB.h>


#include <Xpsi/Tree.h>	/* Tree widget		   */

#include "icons/mono/48/group_user48.icon"

#include "icons.h"

#include "tree.h"	/* Generic m-ary tree package include file */
#include "list.h"	/* Simple list struct & routine defs	   */

#include "user_group.h"

#include  "treeps.h"	/* Process wide definitions		   */

#include  "color_code.h"

#include "debug.h"	/* Macro based debug package		   */

#ifndef lint
static char *usr_grp_cpr ="@(#) Copyright(c) 1995, Pulsar Software Inc.\n\t All rights reserved.";
static char *usr_grp_sccs="@(#) %W% 	Last Delta %G% %U%";
#endif

Pixmap getDeepIconByName();

struct group  *copy_group();
struct passwd *copy_passwd();

/* ----------------  User Group Tree Node Structure  ------------------- */

/* The following structure is replicated for each node in the user/group tree.
 */

/* State of each node in the tree */

#define NODE_EMPTY	0
#define NODE_DISPLAYED  1	
#define NODE_HIDDEN	2

/* Types of nodes in the tree */

#define ALL_GROUPS_USERS	1
#define GROUP			2
#define USER			3

typedef struct 	_user_group_info_node {
	int			 state; /* NODE_HIDDEN or NODE_DISPLAYED */
	int			 type;  /* GROUP/USER/ALL_GROUPS_USERS   */
	int			 selected; /* True or False 		 */
	Widget			 w; 	/* The label widget              */
	int			 dialog_state; /* NO_DIALOG, DIALOG_DISPLAYED */
	Widget			 dialog_w;     /* The dialog widget     */
	Pixel			 fg;	/* Foreground color		*/
	Pixel			 bg;	/* Background color		*/
} User_group_info_node;


void 			select_usr_node();
User_group_info_node   *make_ugi_node();
tnode 		       *find_group_node();

void perform_all_users();
void create_usr_widget();
tnode *match_userid();

tnode *add_ug_node();

static void show_node();
static void hide_node();
static void show_node_value();
static void monitor_selected_node();
static void remove_selected_node();

static void in_list();
static void show_background_group();
static void show_background_user();
static void grey_node();

static void login_user();

extern Widget setup_user_group_menu();

extern Object_color   *User_colors;
extern Object_color   *Group_colors;
extern Object_color   *find_obj_color_by_id();


static Atom deleteWindow=0;
static Atom protocols=0;
static void deleteWindowHandler();

/* To handle WM delete events we need to find our way back to the 
 * user_group info structure. It would be cleaner to pass a
 * pointer to the structure into the event engine so we could
 * maintain proper context, but we know there is only ever one
 * user_group structure, so we just use the following  single 
 * global variable for this purpose.
 */

static User_group_info *User_group=NULL;


void    ugTreeResized();
void    ugPreTreeWindowOffsets();

/* The following are used to limit the size of the automatic window growth.
 * If the window get's larger than this then scroll bars will be enabled,
 * values of -1 mean the window can grow arbitrarily large, this is for
 * those who want such a thing(perhaps on virtual desktops).
 */

extern int     Max_window_x;
extern int     Max_window_y;
static int     Min_window_x=170;
static int     Min_window_y=60;

extern Boolean FitToScreen;

extern int ActualScreenWidth;
extern int ActualScreenHeight;
extern int BottomOffset;

static int RightOffset=1;


Widget UserGroupShell=NULL;
Widget UserGroupScrollWidget=NULL;


struct dimensionalOffsets {
        int     top;
        int     left;
        int     bottom;
        int     right;
};

/* -------------------  Func's  -------------------- */


User_group_info
*setup_user_group( parent, process_cb )
Widget parent;
void (*process_cb) ();
{
	User_group_info *usr;
	User_group_info_node  *ugi;

	if ( User_group != NULL )
	{
	    fprintf( stderr, "WARNING!!! Trying to create a second User_group structure! There can be only one!!!\n");
	    return NULL;
	}

	usr = ( User_group_info * ) malloc( sizeof( User_group_info ));
	if ( usr == NULL )
	{
		fprintf(stderr, "Setup_user_group: malloc failed\n");
		exit( 2 );
	}

	usr->state        = INITIALIZED;
	usr->dialog_state = NO_DIALOG;
	usr->group_view   = VIEW_IGNORE_GROUPS;
	usr->user_view    = VIEW_ACTIVE_USERS;

	/* Add current user id as the default selection */

	usr->user_display_list = NULL; /* Needs to be null for next statement */

	usr->user_display_list = fwd_insert_lnode( usr->user_display_list, 
							     getuid(), NULL );

	usr->group_display_list         = NULL;

	usr->monitor_users		= False;  /* Set true on popup */

	usr->active_user_list           = NULL;
	usr->active_group_list          = NULL;
	usr->group_login_list           = NULL;
	usr->user_login_list            = NULL;

	usr->head			= NULL;

	usr->no_group			= NULL;

	usr->num_grp_selected		= 0;  

	usr->info_loaded 		= 0;

	ugi = make_ugi_node();		/* Create tree top  All */
	if ( ugi == NULL )
	{
		fprintf(stderr, "Setup_user_group: malloc failed\n");
		exit( 2 );
	}

	/* Set color for All node(tree root) */

	ugi->fg	  	  = -1;
	ugi->bg	  	  = -1;
	ugi->type	  = ALL_GROUPS_USERS;

	usr->head	  = construct_tnode( -1, "All", ugi ); 

	usr->ug_shell	  = NULL;
	usr->form_w	  = NULL;
	usr->parent_w	  = parent;
	usr->tree_w	  = NULL;
	usr->user_uid     = getuid();
	usr->user_gid     = getgid();

	usr->user_uid_str = NULL;
	usr->user_gid_str = NULL;

	usr->group_info   = NULL;
	usr->passwd_info  = NULL;

	usr->process_cb = process_cb;
	usr->popdownCB  = NULL;

	load_and_store_group_info( usr );
	load_and_store_passwd_info( usr );

	get_ug_str( usr );

	User_group = usr;

	return( usr );
}


setPopdownCallback( popdown_cb )
void (*popdown_cb) ();
{
	User_group->popdownCB = popdown_cb;
}


static void
deleteWindowHandler(w, p, e, b)
Widget w;
XtPointer p;
XEvent *e;
Boolean *b;
{
	if (((XClientMessageEvent *)e)->type != ClientMessage)
	{
              DEBUG1( 1, "Not Client Message: type(%d)\n",
		 	    ((XClientMessageEvent *)e)->type );
              return;
	}

	if ((((XClientMessageEvent *)e)->message_type == protocols) &&
               (((XClientMessageEvent *)e)->data.l[0] == deleteWindow)) 
	{
              DEBUG2( 1, "Got Delete Message: type(%d) code(%d)\n",
		((XClientMessageEvent *)e)->message_type,
                ((XClientMessageEvent *)e)->data.l[0]  );


		 /* Dialog is unmapped by protocol handling code */

		 User_group->dialog_state = DIALOG_HIDDEN;

		 /* Invoke callback to tell button bar to disarm
		  * the gutree icon.
		  */

		if ( User_group->popdownCB != NULL )
		    (*User_group->popdownCB)();
	}
	else
	{
              DEBUG2( 1, "Not right Message: type(%d) code(%d)\n",
		((XClientMessageEvent *)e)->message_type,
                ((XClientMessageEvent *)e)->data.l[0] );
	}

	return;
}

popdown_user_tree( users )
User_group_info  *users;
{
	if ( users->state != INITIALIZED )
		return;

	if ( users->dialog_state == DIALOG_DISPLAYED )
	{
		DEBUG0( 1, "popdown_user_tree: Displayed dialog poping down\n");

		XtPopdown( users->ug_shell );

		users->dialog_state = DIALOG_HIDDEN;
	}
}


popup_user_tree( users)
User_group_info  *users;
{
	Widget  	 	 app_shell2;
	Widget			 form_w, sep1, sw, tw, sep2, monitor;
	Arg		 	 wargs[10];
	int		 	 n;
	Position		 x, y;
	Dimension		 width;
	char			 title_str[80];
	Pixmap			 icon_pixmap;
	Display			*display;
	Screen			*screen;
	Window			 icon_win;

	if ( users == NULL )
		return;

	if ( users->state != INITIALIZED )
		return;

	if ( users->dialog_state == DIALOG_HIDDEN )
	{
		DEBUG0( 1, "popup_user_tree: dialog already created=> popup\n");

		XtPopup( users->ug_shell, XtGrabNone );

		users->dialog_state = DIALOG_DISPLAYED;
		return; 
	}
	else if ( users->dialog_state != NO_DIALOG )
	{
		DEBUG0( 5, "popup_user_tree: dialog already displayed\n");
		return; 
	}

	/* If we get here we need to create an app shell with a form widget
	 * which then contains the appropriate widgets to display the 
	 * users tree.
	 */

	sprintf( title_str, "%s", "Groups and Users"  );

	display = XtDisplay(users->parent_w);
	screen  = XtScreen(users->parent_w);

	icon_pixmap = getDeepIconByName( users->parent_w, "groupUser",
					group_user48_bits, 48 );



	/* icon_win = XCreateSimpleWindow( XtDisplay(users->parent_w),
                          RootWindowOfScreen( XtScreen(users->parent_w)),
			  10,10,
			  200,10,
			  2,     
			  BlackPixelOfScreen(screen),
			  WhitePixelOfScreen(screen)); */

	/* XtSetArg( wargs[n], XmNiconWindow, icon_win); n++; */


	n = 0;
	XtSetArg( wargs[n], XmNtitle, title_str); n++;
	XtSetArg( wargs[n], XmNiconName, "Grp/User"); n++;
	XtSetArg( wargs[n], XmNiconPixmap, icon_pixmap); n++;
	XtSetArg( wargs[n], XmNdeleteResponse, XmUNMAP); n++;


	app_shell2 = XtAppCreateShell( "userGroupTree", "UserGroupTree",
					   topLevelShellWidgetClass,
					   display,
					   wargs, n);


	UserGroupShell = app_shell2;

	/* The following magic is for the Window Delete Event handler */

	protocols = XInternAtom( XtDisplay(app_shell2), "WM_PROTOCOLS", False);

	deleteWindow = XInternAtom( XtDisplay(app_shell2), "WM_DELETE_WINDOW",
									False);

	users->ug_shell = app_shell2;

	n = 0;
	XtSetArg( wargs[n], XmNfractionBase, 10 ); n++;
	form_w = XmCreateForm( users->ug_shell, "ugform", wargs, n);
	if ( form_w == NULL )
	{
		DEBUG0( 0, "popup_user_tree: failed to create form\n");
		return;
	}

	users->form_w	    = form_w;

	sep1 = setup_user_group_menu( users );


	/* Create scrolled window to hold tree */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment,XmATTACH_WIDGET);n++;
	XtSetArg( wargs[n], XmNtopWidget, sep1); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	sw = XtCreateManagedWidget( "ugswindow",  xmScrolledWindowWidgetClass,
						form_w, wargs, n );

	UserGroupScrollWidget = sw;

	/* Create group/user tree display */

	n = 0;
	XtSetArg( wargs[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg( wargs[n], XmNtopWidget, sep1); n++;
	XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
	tw = XtCreateWidget( "userTree", xpsiTreeWidgetClass, sw, wargs, n);
	users->tree_w = tw;

	XtAddCallback( tw, XtNtreeResizeCallback, ugTreeResized, (XtPointer)0 );

	/* Create top widget of tree */

	create_usr_widget( users->tree_w, NULL, users->head );

	if ( users->info_loaded != 1 )
	{
	    load_group_info( users );
	    load_password_info( users );

	    /* Load logged in user info */

	    load_logged_in( users );

	    users->info_loaded = 1;

	    set_default_selection( users );
	}

	users->dialog_state = DIALOG_DISPLAYED; /* Must be before next call */



	XtManageChild( tw );
	XtManageChild( form_w );

	/* XtRealizeWidget( users->ug_shell ); */

	XtPopup( users->ug_shell, XtGrabNone );

	/* Add Window Delete Event handler */

	XSetWMProtocols( XtDisplay(app_shell2), XtWindow(app_shell2),
					&deleteWindow, 1 );

	XtAddRawEventHandler( app_shell2, 0, True, deleteWindowHandler, NULL );

	display_user_group( users );
}


void
ugTreeResized( w, data, p )
Widget w;
int    data;
xpsiTreeCallbackStruct *p;
{
	Dimension width, height, border;
	Dimension main_window_width, main_window_height;
	Position  x, y;
	int delta_width, delta_height;
	Dimension new_width, new_height;
	int change_size=0;
	int change_x=0;
	int change_y=0;
	int n;
	Arg wargs[6];
	struct dimensionalOffsets offset;

	if ( ! XtIsRealized( UserGroupShell ) )
		return;

	DEBUG4( 5, "ugTreeResized from h(%d) v(%d) TO h(%d) v(%d):\n", 
					p->last_width, p->last_height,
					p->width, p->height );

	width  = p->width;
	height = p->height;
	border = 0;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &main_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &main_window_height ); n++;
	XtSetArg( wargs[n], XtNx, &x  ); n++;
	XtSetArg( wargs[n], XtNy, &y ); n++;
	XtGetValues( UserGroupShell, wargs, n );

	delta_width  = (int) p->width  - (int) p->last_width;
	delta_height = (int) p->height - (int) p->last_height;

	DEBUG4(5,"ugTreeResized Main Window x(%d),y(%d) Width(%d) height(%d) \n", 
					x, y,
					main_window_width,
					main_window_height );


	DEBUG2( 5, "ugTreeResized Delta Width(%d) height(%d) \n", 
					delta_width, delta_height );


	ugPreTreeWindowOffsets( &offset );

	new_width = p->width + offset.left + offset.right;

	change_size++;
	if ( delta_width > 0 )
	{
	    if ( Max_window_x != -1 )
	    {
	        if ( main_window_width == Max_window_x ) /* Already max */
	        {
		    new_width = Max_window_x;  /* In case height changes */
		    change_size--;
	        }
	    }
	}
	else if ( delta_width < 0 )
	{
	    if ( Min_window_x != -1 )
	    {
	        if ( main_window_width == Min_window_x ) /* Already min */
	        {
		    new_width = Min_window_x;  /* In case height changes */
		    change_size--;
	        }
	    }
	}
	else
	{
	    change_size = 0;
	}

	if ( Max_window_x != -1 )
	{
	    if ( new_width > Max_window_x )
		new_width = Max_window_x;
	}
	if ( Min_window_x != -1 )
	{
	    if ( new_width < Min_window_x )
		new_width = Min_window_x;
	}

	if ( FitToScreen )
	{
	    if ( ( x + new_width ) > ( ActualScreenWidth - RightOffset) )
	    	new_width = ( ActualScreenWidth - x ) - RightOffset;
	}


	new_height = p->height + offset.top + offset.bottom;


	change_size++;

	if ( delta_height > 0 )
	{
	    if ( Max_window_y != -1 )
	    {
	        if ( main_window_height == Max_window_y )
	        {
		    new_height = Max_window_y;
		    change_size--;
	        }
	    }
	}
	else if ( delta_height < 0 )
	{
	    if ( Min_window_y != -1 )
	    {
	        if ( main_window_height == Min_window_y )
	        {
		    new_height = Min_window_y;
		    change_size--;
	        }
	    }
	}
	else
	{
	    change_size--;
	}

	if ( Max_window_y != -1 )
	{
	    if ( new_height > Max_window_y )
		new_height = Max_window_y;
	}
	if ( Min_window_y != -1 )
	{
	    if ( new_height < Min_window_y )
		new_height = Min_window_y;
	}

	if ( FitToScreen )
	{
	    if ( ( y + new_height ) > ( ActualScreenHeight - BottomOffset ) )
	    	new_height = ( ActualScreenHeight - y ) - BottomOffset;
	}


	if ( change_size )
	{
		DEBUG2(8, "Change main window size to h(%d) v(%d)\n", 
						new_width, new_height );

		XtResizeWidget( UserGroupShell, new_width, new_height, border );

		/* clipDetails(); */

	}
}

void
ugPreTreeWindowOffsets( offset )
struct dimensionalOffsets *offset;
{
	Position main_window_x, main_window_y;
	Position clip_window_x, clip_window_y;
	Position mx, my, cx, cy;
	Widget   clip_w;     	    /* Scrolled Windows clip/client widget */
	Dimension main_window_width, main_window_height;
	Dimension clip_window_width, clip_window_height;
	Position mlr_x, mlr_y, clr_x, clr_y;
	int n;
	Arg wargs[6];

	n = 0;
	XtSetArg( wargs[n], XtNx, &main_window_x  ); n++;
	XtSetArg( wargs[n], XtNy, &main_window_y ); n++;
	XtGetValues( UserGroupShell, wargs, n );

	XtTranslateCoords( UserGroupShell, main_window_x, main_window_y, &mx, &my);
	

	n = 0;
	XtSetArg( wargs[n], XmNclipWindow, &clip_w ); n++;
	XtGetValues( UserGroupScrollWidget, wargs, n );


	n = 0;
	XtSetArg( wargs[n], XtNx, &clip_window_x  ); n++;
	XtSetArg( wargs[n], XtNy, &clip_window_y ); n++;
	XtGetValues( clip_w, wargs, n );

	XtTranslateCoords( clip_w , clip_window_x, clip_window_y, &cx, &cy);


	offset->top    = cy - my;
	offset->left   = cx - mx;

	/* Now get bottom right offsets */

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &main_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &main_window_height ); n++;
	XtGetValues( UserGroupShell, wargs, n );

	mlr_x = mx + main_window_width;
	mlr_y = my + main_window_height;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &clip_window_width  ); n++;
	XtSetArg( wargs[n], XtNheight, &clip_window_height ); n++;
	XtGetValues( clip_w, wargs, n );

	clr_x = cx + clip_window_width;
	clr_y = cy + clip_window_height;

	offset->bottom  = mlr_y - clr_y;
	offset->right   = mlr_x - clr_x;


	DEBUG4(7, "Offsets top(%d) left(%d) bottom(%d) right(%d)\n", 
				offset->top, offset->left, 
				offset->bottom, offset->right          );
}


Pixmap
getDeepIconByName( w, name, defaultBitmap, iconSize )
Widget w;
char *name;
unsigned char *defaultBitmap;
int iconSize;
{
        Pixmap  icon_pixmap=-1;
        Display *display=NULL;
        int     screen;
        Screen *screen_ptr=NULL;
        int     numPlanes;
        int     rc, height, width;

        display    = XtDisplay( w );
        screen     = DefaultScreen( display );
        screen_ptr = ScreenOfDisplay( display, screen );

        numPlanes = PlanesOfScreen( screen_ptr );

        if ( numPlanes >= 16 )
        {
            rc = fetchIcon( name, w, HIGH_COLOR, iconSize,
                                &icon_pixmap, &width, &height );
            if ( rc == -1 )
                icon_pixmap = -1;
        }
        else if ( numPlanes >= 8 )
        {
            rc = fetchIcon( name, w, LOW_COLOR, iconSize,
                                &icon_pixmap, &width, &height );
            if ( rc == -1 )
                icon_pixmap = -1;
        }

        if ( icon_pixmap == -1 )
        {
	    icon_pixmap = XCreateBitmapFromData( XtDisplay(w),
                          RootWindowOfScreen( XtScreen(w)),
                          BITMAP_DATA_CAST      defaultBitmap,
                          BITMAP_DIMENSION_CAST iconSize,
                          BITMAP_DIMENSION_CAST iconSize );
	}
	return( icon_pixmap );
}


set_default_selection( users )
User_group_info  *users;
{
	User_group_info_node  *ugi;
	tnode		      *tptr;

	/* Set the default selection to the current user */


	tptr = walk_tree( users->head, match_userid, users->user_uid );
	if ( tptr == NULL )
		return;


	ugi = (User_group_info_node *) tptr->data;

	if ( ugi != NULL )
	{
		ugi->selected = True;
		display_usr_info( tptr );
	}
}

tnode
*match_userid( ptr, key )
tnode *ptr;
int key;
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi == NULL )
                return( NULL_TNODE_PTR );

        if ( ptr->key == key && ugi->type == USER )
                return( ptr );
        else
                return( NULL_TNODE_PTR );
}

load_and_store_group_info( users )
User_group_info  *users;
{
	struct group *g, *grp;
	Group_info   *grp_info;
	Group_info   *last_grp=NULL;

	if ( users->group_info != NULL )
	{
		DEBUG0( 0, "load_and_store_group_info: Already done it!!\n");
		return;
	}

	while ( ( g = getgrent()) != NULL )
	{
		grp_info = ( Group_info * ) malloc( sizeof( Group_info ) );
		grp = ( struct group * ) malloc( sizeof( struct group ) );
		if ( grp == NULL || grp_info == NULL )
		{
			fprintf(stderr, "Malloc: Not enough memory\n");
			exit(3);
		}

		grp_info->g    = copy_group( g, grp );
		grp_info->next = NULL;

		/* Append to list */

		if ( last_grp == NULL )
			users->group_info  = grp_info;
		else
			last_grp->next = grp_info;

		last_grp = grp_info;
	}

	endgrent();
}

struct group
*copy_group( old, new )
struct group *old;
struct group *new;
{
	int   i;
	int   num_str=0;

	new->gr_name   = strdup( old->gr_name );
	new->gr_passwd = strdup( old->gr_passwd );
	new->gr_gid    = old->gr_gid;

	/* Count number of groups which are in membership list */

	for ( i = 0 ; i < 100 ; i++ )	/* 100, is a sanity checker */
	{
		if ( old->gr_mem == NULL )
			break;

		if ( old->gr_mem[i] == NULL )
			break;
	}

	num_str = i;
	if ( num_str == 0 )
	{
		new->gr_mem = NULL;
		return( new );
	}

	new->gr_mem = (char **) malloc( (num_str + 1) * sizeof( char *) );
	if ( new->gr_mem == NULL )
	{
		perror("Copy_group: Malloc");
		return( NULL );
	}

	for ( i = 0; i < num_str ; i++ )
	{
		new->gr_mem[i] = strdup( old->gr_mem[i] );
	}

	new->gr_mem[i] = NULL;
	return( new );
}

int
find_gid_by_name( users , name )
User_group_info  *users;
char *name;
{
	Group_info   *grp_info;

	for ( grp_info = users->group_info; grp_info != NULL ; 
						grp_info = grp_info->next )
	{
		if ( strcmp( grp_info->g->gr_name, name ) == 0 )
		{
			return( grp_info->g->gr_gid );
		}
	}

	return( -1 );
}

char
*find_name_by_gid( users , gid )
User_group_info  *users;
int  gid;
{
	Group_info   *grp_info;

	for ( grp_info = users->group_info; grp_info != NULL ; 
						grp_info = grp_info->next )
	{
		if ( grp_info->g->gr_gid == gid )
		{
			return( grp_info->g->gr_name );
		}
	}

	return( NULL );
}

load_and_store_passwd_info( users )
User_group_info  *users;
{
	struct passwd *p, *pwd;
	Passwd_info   *pwd_info;
	Passwd_info   *last_pwd=NULL;
	int		maxInitialCount=256;
	int		cnt=0;

	if ( users->passwd_info != NULL )
	{
		/* Later do a drop, i.e. allow runtime reloads */

		DEBUG0( 0, "load_and_store_passwd_info: Already done it!!\n");
		return;
	}
	while ( ( p = getpwent()) != NULL )
	{
		pwd_info = ( Passwd_info * ) malloc( sizeof( Passwd_info ) );
		pwd = ( struct passwd * ) malloc( sizeof( struct passwd ) );
		if ( pwd == NULL || pwd_info == NULL )
		{
			fprintf(stderr, "Malloc: Not enough memory\n");
			exit(3);
		}

		pwd_info->p    = copy_passwd( p, pwd );
		pwd_info->next = NULL;

		/* Append to list */

		if ( last_pwd == NULL )
			users->passwd_info  = pwd_info;
		else
			last_pwd->next = pwd_info;

		last_pwd = pwd_info;

		/* Note the following limits the number of
		 * passwd entries initially loaded. Others
		 * are loaded as needed
		 */

		if ( cnt++ > maxInitialCount )
			break;  /* Load others dynamically */
	}
	endpwent();
}

char *
load_passwd_info_by_uid( users, uid )
User_group_info  *users;
int 		  uid;
{
	struct passwd *p, *pwd;
	Passwd_info   *pwd_info;
	char	      *name=NULL;
	static char   *unknown="Unknown";

	if ( users == NULL )
		return;

	if ( ( p = getpwuid( uid )) != NULL )
	{
		pwd_info = ( Passwd_info * ) malloc( sizeof( Passwd_info ) );
		pwd = ( struct passwd * ) malloc( sizeof( struct passwd ) );
		if ( pwd == NULL || pwd_info == NULL )
		{
			fprintf(stderr, "Malloc: load_passwd_info_by_uid \n");
			return unknown;
		}

		pwd_info->p         = copy_passwd( p, pwd );
		pwd_info->next      = users->passwd_info;
		users->passwd_info  = pwd_info;
		name = pwd_info->p->pw_name;
	}
	endpwent();

	if ( name == NULL )
	   name = unknown;

	return( name );
}

int
load_passwd_info_by_name( users, name )
User_group_info  *users;
char 		 *name;
{
	struct passwd *p, *pwd;
	Passwd_info   *pwd_info;
	int	       uid=-1;

	if ( users == NULL )
		return;

	if ( ( p = getpwnam( name )) != NULL )
	{
		pwd_info = ( Passwd_info * ) malloc( sizeof( Passwd_info ) );
		pwd = ( struct passwd * ) malloc( sizeof( struct passwd ) );
		if ( pwd == NULL || pwd_info == NULL )
		{
			fprintf(stderr, "Malloc: load_passwd_info_by_uid \n");
			return -1;
		}

		pwd_info->p         = copy_passwd( p, pwd );
		pwd_info->next      = users->passwd_info;
		users->passwd_info  = pwd_info;

		uid = pwd_info->p->pw_uid;
	}

	endpwent();

	return uid;
}

check_for_new_userid( users, uid )
User_group_info  *users;
int uid;
{
	Passwd_info   *pi;
	struct passwd *p;
   
	for ( pi = users->passwd_info; pi != NULL ; pi = pi->next )
	{
		if ( pi->p->pw_uid == uid )
			return;
	}
	
	if ( load_passwd_info_by_uid( users, uid ) )
	{

	    if ( users->dialog_state == NO_DIALOG )
		return;

	    p = users->passwd_info->p; /* It was just pushed on the front */
	    add_ug_node( users, p->pw_gid, USER, p->pw_name, p->pw_uid); 

	}
}

int
find_uid_by_name( users , name )
User_group_info  *users;
char *name;
{
	Passwd_info *pwd_info;

	for ( pwd_info = users->passwd_info; pwd_info != NULL ; 
						pwd_info = pwd_info->next )
	{
		if ( strcmp( pwd_info->p->pw_name, name ) == 0 )
		{
			return( pwd_info->p->pw_uid );
		}
	}

	return load_passwd_info_by_name( users, name );
}

char
*find_name_by_uid( users , uid )
User_group_info  *users;
int  uid;
{
	Passwd_info *pwd_info;

	/* Note: can have > 1 occurance of any uid - ignore for now  */

	for ( pwd_info = users->passwd_info; pwd_info != NULL ; 
						pwd_info = pwd_info->next )
	{
		if ( pwd_info->p->pw_uid == uid )
		{
			return( pwd_info->p->pw_name );
		}
	}

	return( load_passwd_info_by_uid( users, uid ) );
}

int
find_gid_by_uid( users , uid )
User_group_info  *users;
int  uid;
{
	Passwd_info *pwd_info;

	for ( pwd_info = users->passwd_info; pwd_info != NULL ; 
						pwd_info = pwd_info->next )
	{
		if ( pwd_info->p->pw_uid == uid )
		{
			return( pwd_info->p->pw_gid );
		}
	}

	return( -1 );
}

struct passwd
*copy_passwd( old, new )
struct passwd *old;
struct passwd *new;
{
	new->pw_name   = strdup( old->pw_name );
	new->pw_passwd = strdup( old->pw_passwd );
	new->pw_uid    = old->pw_uid;
	new->pw_gid    = old->pw_gid;

#ifndef LINUX
	new->pw_age     = strdup( old->pw_age );
	new->pw_comment = strdup( old->pw_comment );
#endif
	new->pw_gecos   = strdup( old->pw_gecos );
	new->pw_dir     = strdup( old->pw_dir );
	new->pw_shell   = strdup( old->pw_shell );

	return( new );
}

load_group_info( users )
User_group_info  *users;
{
	Group_info   *grp_info;
	struct group *g;

	for ( grp_info = users->group_info; grp_info != NULL ; 
						grp_info = grp_info->next )
	{
		g = grp_info->g;
		add_ug_node( users, -1, GROUP, g->gr_name, g->gr_gid); 
	}
}

lnode
*sorted_group_list( users, num_groups )
User_group_info  *users;
int		 *num_groups;
{
	Group_info   *grp_info;
	struct group *g;
	lnode        *list=NULL;

	*num_groups=0;
	for ( grp_info = users->group_info; grp_info != NULL ; 
						grp_info = grp_info->next )
	{
		g = grp_info->g;
		*num_groups = *num_groups + 1;
		list = fwd_insert_lnode( list, g->gr_gid, g->gr_name);
	}
	
	return( list );
}

load_password_info( users )
User_group_info  *users;
{
	Passwd_info *pwd_info;
	struct passwd *p;

	for ( pwd_info = users->passwd_info; pwd_info != NULL ; 
						pwd_info = pwd_info->next )
	{
		p = pwd_info->p;
		add_ug_node( users, p->pw_gid, USER, p->pw_name, p->pw_uid); 
	}

}


lnode
*sorted_passwd_list( users, num_passwd )
User_group_info  *users;
int		 *num_passwd;
{
	Passwd_info *pwd_info;
	struct passwd *p;
	lnode        *list=NULL;

	*num_passwd = 0;

	for ( pwd_info = users->passwd_info; pwd_info != NULL ; 
						pwd_info = pwd_info->next )
	{
		p = pwd_info->p;
		*num_passwd = *num_passwd + 1;
		list = fwd_insert_lnode( list, p->pw_uid, p->pw_name);
	}

	return( list );
}

get_ug_str( users )
User_group_info  *users;
{
	Passwd_info *pwd_info;
	Group_info   *grp_info;
	struct passwd *p;
	struct group *g;
	char *str=NULL;

	for ( pwd_info = users->passwd_info; pwd_info != NULL ; 
						pwd_info = pwd_info->next )
	{
		p = pwd_info->p;
		if ( p->pw_uid == users->user_uid )
		{
			str = strdup(p->pw_name);
			users->user_uid_str = strdup( str );
			break;
		}
	}

	if ( str == NULL )
	{
		str = load_passwd_info_by_uid( users, users->user_uid );
		if ( str == NULL )
			users->user_uid_str = strdup("Unknown");
		else
			users->user_uid_str = strdup( str );
	}

	str = NULL;
	for ( grp_info = users->group_info; grp_info != NULL ; 
						grp_info = grp_info->next )
	{
		g = grp_info->g;
		if ( g->gr_gid == users->user_gid )
		{
			str = strdup(g->gr_name);
			users->user_gid_str = strdup( str );
			break;
		}
	}

	if ( str == NULL )
	{
	    users->user_gid_str = strdup( "Unknown" );
	}
}

tnode *
add_ug_node( users, parent_key, type, name, number)
User_group_info  *users;
int     parent_key;
int     type;
char   *name;
int	number;
{
	tnode 		      *ptr, *pptr;
	tnode 		      *nogroup_group=NULL;
	User_group_info_node  *ugi;
	User_group_info_node  *p_ugi;
	Widget  	       parent_widget;
	Object_color	      *obj;

	ugi = make_ugi_node();
	if ( ugi == NULL )
	{
		fprintf(stderr, "add_ug_node: malloc failed\n");
		return( NULL );
	}

	ugi->type = type;
	ugi->fg   = -1;  		/* By default Use resource values */
	ugi->bg   = -1;

	ptr  = construct_tnode( number, name, ugi );

	if ( type == GROUP )
	{
		obj = find_obj_color_by_id( Group_colors, ptr->key );
		if ( obj )
		{
			ugi->fg = obj->fg;
			ugi->bg = obj->bg;
		}

		add_tnode( users->head, ptr );
	}
	else if ( type == USER )
	{
		pptr = find_group_node( users, parent_key );

		if ( pptr == NULL )
		{
		    if ( users->no_group == NULL )
		    {
			pptr = add_ug_node( users, -1, GROUP, 
						 ":NO-GROUP:", 999999 );

			users->no_group = pptr;
		    }
		    else
		    {
			pptr = users->no_group;
		    }
		}


		obj = find_obj_color_by_id( User_colors, ptr->key );
		if ( obj )
		{
			ugi->fg = obj->fg;
			ugi->bg = obj->bg;
		}
		else /* If group has a color use it, else use default */
		{
			if ( pptr )
			{
				p_ugi = (User_group_info_node *)pptr->data;
				if ( p_ugi )
				{
					if ( p_ugi->fg != -1 )
						ugi->fg = p_ugi->fg;

					if ( p_ugi->bg != -1 )
						ugi->bg = p_ugi->bg;
				}
			}
		}

		add_tnode( pptr, ptr );
	}

	if ( ptr->parent )
	{
		p_ugi = (User_group_info_node *) ptr->parent->data;
		parent_widget = p_ugi->w;
	}

	create_usr_widget( users->tree_w, parent_widget, ptr);

	return( ptr );
}

tnode
*find_group_node( users, group_id )
User_group_info  *users;
int		  group_id;
{
	child_ptr *p;
	User_group_info_node  *ugi;

	if ( users->head == NULL )
		return( NULL );

	for ( p = users->head->children; p != NULL ; p = p->next )
	{
		if ( p->child->key == group_id )
		{
			return( p->child );
		}
	}

	return( NULL );
}

User_group_info_node
*make_ugi_node()
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) malloc( sizeof(User_group_info_node ) );
	if ( ugi == NULL )
	{
		fprintf(stderr, "Setup_user_group: malloc failed\n");
		return( NULL );
	}

	ugi->state 	  = NODE_EMPTY;
	ugi->type 	  = 0;
	ugi->selected 	  = False;
	ugi->w   	  = NULL;
	ugi->dialog_state = NO_DIALOG;
	ugi->dialog_w     = NULL;
	ugi->fg     	  = -1;
	ugi->bg           = -1;

	return( ugi );
}

void
create_usr_widget( treew, parent_widget, ptr )
Widget treew;
Widget parent_widget;
tnode *ptr;
{
	int n;
	Arg wargs[8];
	User_group_info_node   *ugi;
	User_group_info_node   *p_ugi;
	Pixel			bg;
	Pixel			fg;


	if ( ptr == NULL )
		return;

	if ( ptr->parent )
	{
		p_ugi = (User_group_info_node *) ptr->parent->data;
		if ( p_ugi->state == NODE_EMPTY )
		{
		   DEBUG0(1, "create_usr_widget: Ignore parent hidden!\n");
		   return;
		}
	}

	ugi = (User_group_info_node *) ptr->data;
	if ( ugi->state == NODE_DISPLAYED || ugi->w )
	{
		DEBUG0(0, "create_usr_widget: Widget already exists!\n");
		return;
	}

	if ( ugi->fg == -1 )
		fg = Node_foreground;
	else
		fg = ugi->fg;
	
	if ( ugi->bg == -1 )
		bg = Node_background;
	else
		bg = ugi->bg;

	n = 0;
        XtSetArg( wargs[n], XmNforeground, fg); n++;
        XtSetArg( wargs[n], XmNbackground, bg); n++;
	XtSetArg( wargs[n], XtNparentWidget, parent_widget ); n++;
	XtSetArg( wargs[n], XtNkey, ptr->key ); n++;
	ugi->w = XtCreateManagedWidget("ugInfo", xmLabelWidgetClass, 
						treew, wargs, n );

	DEBUG1(9, "create_usr_widget: created widget for(%s)!\n", ptr->str_key);

	display_usr_info( ptr );

	XtAddEventHandler(ugi->w, ButtonPressMask, FALSE, select_usr_node, ptr);

	ugi->state = NODE_DISPLAYED;

	DEBUG1(9, "create_usr_widget: finished(%d)!\n", ptr->key);
}

display_usr_info( ptr )
tnode *ptr;
{
	User_group_info_node   *ugi;
	int 			display_numeric=0;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi->selected == True )
	{
		if ( display_numeric )
			xs_wprintf(ugi->w, "(*)%d:%s", ptr->key , ptr->str_key );
		else
			xs_wprintf(ugi->w, "(*)%s", ptr->str_key );
	}
	else
	{
		if ( display_numeric )
			xs_wprintf(ugi->w, "%d:%s", ptr->key , ptr->str_key );
		else
			xs_wprintf(ugi->w, "%s", ptr->str_key );
	}
}

destroy_usr_widget( ptr )
tnode *ptr;
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi->state == NODE_DISPLAYED )
	{
		XtUnmanageChild( ugi->w );
		XtDestroyWidget( ugi->w );
		ugi->w = NULL;
	}

	ugi->state = NODE_EMPTY;
}

void
select_usr_node( w, ptr, event )
Widget w;
tnode  *ptr;
XEvent *event;
{
	User_group_info       *users;
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi->type == ALL_GROUPS_USERS )
		return;

	if ( event->xbutton.state & ControlMask )
	{
		if ( ugi->type == USER )
		{
			DEBUG1(1, "Selected USER(%s)\n", ptr->str_key );
	 		/*	popup_user_info( w, ptr );  */
		}
		else if ( ugi->type == GROUP )
		{
			DEBUG1(1, "Selected GROUP(%s)\n", ptr->str_key );
	 		/*	popup_group_info( w, ptr );  */
		}
	}
	else
	{
		users = User_group;
		if ( ugi->selected == True )
		{
			ugi->selected = False;

			if ( ugi->type == GROUP )
			{
			    remove_selected_node( users, ptr,
			    			&(users->group_display_list) );

			    users->num_grp_selected -= 1;
			}
			else if ( ugi->type == USER )
			{
			    remove_selected_node( users, ptr, 
			    			&(users->user_display_list) );
			}

		}
		else
		{
			ugi->selected = True;

			if ( ugi->type == GROUP )
			{
			    monitor_selected_node( users, ptr,
						  &(users->group_display_list));

			    users->num_grp_selected += 1;
			}
			else if ( ugi->type == USER )
			{
			    monitor_selected_node( users, ptr,
						  &(users->user_display_list));
			}

		}

		display_usr_info( ptr );

		/* Notify main window to update display with new criteria */

		if ( users->process_cb != NULL )
			(*users->process_cb) ( users );
	}
}

display_user_group( users )
User_group_info         *users;
{
	if ( users->dialog_state == NO_DIALOG )
		return;

	set_user_monitor_switch( users );

	XpsiDoTreeLayout( users->tree_w, False );

	/* Based on current view, either hide or show nodes */

	if ( users->group_view == VIEW_ALL_GROUPS  )
		perform_all_groups( users, show_node, 0 );
	else if ( users->group_view == VIEW_IGNORE_GROUPS  )
		perform_all_groups( users, hide_node, 0 );
	else if ( users->group_view == VIEW_LOGGED_IN_GROUP )
		perform_all_groups( users, show_node_value, users->user_gid );
	else if ( users->group_view == VIEW_LOGGED_IN_GROUPS )
		perform_all_groups( users, in_list, users->group_login_list);
	else if ( users->group_view == VIEW_BACKGROUND_GROUPS )
		perform_all_groups( users, show_background_group, 0 );
	else if ( users->group_view == VIEW_ACTIVE_GROUPS )
		perform_all_groups( users, in_list, users->active_group_list);

	if ( users->user_view == VIEW_ALL_USERS  )
		perform_all_users( users, show_node, 0 );
	else if ( users->user_view == VIEW_NO_USERS  )
		perform_all_users( users, hide_node, 0 );
	else if ( users->user_view == VIEW_LOGGED_IN_USER  )
		perform_all_users( users, show_node_value, users->user_uid );
	else if ( users->user_view == VIEW_LOGGED_IN_USERS  )
		perform_all_users( users, in_list, users->user_login_list);
	else if ( users->user_view == VIEW_UUCP_USER  )
		perform_all_users( users, show_node_value, 4 );
	else if ( users->user_view == VIEW_ROOT_USER  )
		perform_all_users( users, show_node_value, 0 );
	else if ( users->user_view == VIEW_BACKGROUND_USERS  )
		perform_all_users( users, show_background_user, 0);
	else if ( users->user_view == VIEW_ACTIVE_USERS  )
		perform_all_users( users, in_list, users->active_user_list);

	XpsiDoTreeLayout( users->tree_w, True );

	display_ug_title( users );
}

set_user_monitor_switch( users )
User_group_info   *users;
{
	int monitor;
	int old_monitor;

	old_monitor = users->monitor_users;

	switch( users->user_view ) {

		case VIEW_ACTIVE_USERS:		/* Fall thru */

		case VIEW_BACKGROUND_USERS: monitor = True;
					    break;
		default: monitor = False; 
			 break;
	}

	switch( users->group_view ) {

		case VIEW_ACTIVE_GROUPS:		/* Fall thru */

		case VIEW_BACKGROUND_GROUPS: monitor = True;
					    break;

		default: 
			 break;
	}

	users->monitor_users = monitor;  

	if ( old_monitor == False && monitor == True )
		refresh_active_user_info( users );
}

int
perform_all_groups( users, func, arg )
User_group_info   *users;
void 		 (*func) ();
void		  *arg;
{
	child_ptr *p;
	User_group_info_node  *ugi;

	if ( users->head == NULL )
		return( (int)NULL );

	for ( p = users->head->children; p != NULL ; p = p->next )
	{
		(*func)(users, p->child, arg);
	}

	return( (int)NULL );
}

void
perform_all_users( users, func, arg )
User_group_info         *users;
void (*func) ();
void *arg;
{
	child_ptr *p;
	child_ptr *g;
	tnode     *ptr;

	if ( users->head == NULL )
		return;

	for ( g = users->head->children; g != NULL ; g = g->next )
	{
		ptr = g->child;
		for ( p = ptr->children; p != NULL; p = p->next )
		{
			(*func)(users, p->child, arg);
		}
	}
}

void
show_node( users, ptr, noop)
User_group_info   *users;
tnode *ptr;
void   *noop;
{
	User_group_info_node  *ugi;
	User_group_info_node  *pugi;

	if ( ptr == NULL || ptr->data == NULL )
		return;

	ugi = (User_group_info_node *) ptr->data;


	/* We only show a node if parent is visible, otherwise we hide it.
	 * except in the case where we are ignoring groups, then we
	 * ensure the parent is visible and show the node as requested
	 */

	if ( ugi->type == USER && ptr->parent ) /* Only for user nodes */
	{
		/* If node is a user and parent widget is not visible, 
		 *   if ignoring groups, then make parent visible
		 */

		pugi= (User_group_info_node *)ptr->parent->data;
		if ( users->group_view == VIEW_IGNORE_GROUPS  )
		{
			if ( pugi->state == NODE_HIDDEN )
			{
				show_node( users, ptr->parent, noop );
			}
		}
		else
		{
			if ( pugi->state == NODE_HIDDEN )
			{
				if ( ugi->state == NODE_DISPLAYED )
					hide_node( users, ptr, noop );
				return;
			}
		}
	}



	if ( ugi->state == NODE_HIDDEN )
	{
		XtManageChild( ugi->w );
		ugi->state = NODE_DISPLAYED;
	}
}

void
hide_node( users, ptr, noop )
User_group_info         *users;
tnode *ptr;
void *noop;
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi->state == NODE_DISPLAYED )
	{
		XtUnmanageChild( ugi->w );
		ugi->state = NODE_HIDDEN;
	}
}

void
show_node_value( users, ptr, value )
User_group_info         *users;
tnode *ptr;
int    value;
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ptr->key == value )
		show_node( users, ptr, 0 );
	else
		hide_node( users, ptr, 0 );
}

void
in_list( users, ptr, list )
User_group_info         *users;
tnode *ptr;
lnode *list;
{
	lnode *p;

	p = find_lnode( list, ptr->key );
	if ( p == NULL )
		hide_node( users, ptr, 0 );
	else
		show_node( users, ptr, 0 );
}

void
show_background_group( users, ptr, noop )
User_group_info         *users;
tnode *ptr;
void  *noop;
{
	lnode *p;

	p = find_lnode( users->active_group_list, ptr->key );

	if ( p == NULL )	/* Group has no processes, so hide it */
	{
		hide_node( users, ptr, 0 );
		return;
	}

	p = find_lnode( users->group_login_list, ptr->key );

	if ( p == NULL )	/* Active and not a logged in user */
		show_node( users, ptr, 0 );
	else
		hide_node( users, ptr, 0 );
}

void
show_background_user( users, ptr, noop )
User_group_info         *users;
tnode *ptr;
void  *noop;
{
	lnode *p;

	p = find_lnode( users->active_user_list, ptr->key );
	if ( p == NULL )	
	{
		hide_node( users, ptr, 0 );
		return;
	}

	p = find_lnode( users->user_login_list, ptr->key );

	if ( p == NULL )	/* Active and not a logged in user */
		show_node( users, ptr, 0 );
	else
		hide_node( users, ptr, 0 );
}

void
monitor_selected_node( users, ptr, list )
User_group_info         *users;
tnode *ptr;
lnode **list;
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi->selected == True )
		*list = fwd_insert_lnode( *list, ptr->key, NULL );
}

void
remove_selected_node( users, ptr, list )
User_group_info         *users;
tnode *ptr;
lnode **list;
{
	User_group_info_node  *ugi;

	ugi = (User_group_info_node *) ptr->data;

	if ( ugi->selected == False )
		*list = remove_lnode( *list, ptr->key );
}
void
grey_node( users, ptr, noop )
User_group_info         *users;
tnode *ptr;
void   *noop;
{
	User_group_info_node  *ugi;
        Pixel   fg;
        Pixel   bg;
        Arg     wargs[4];
        int     n;

	ugi = (User_group_info_node *) ptr->data;

        n = 0;
        XtSetArg( wargs[n], XmNforeground,  &fg); n++;
        XtSetArg( wargs[n], XmNbackground,  &bg); n++;
        XtGetValues( ugi->w, wargs, n);

        n = 0;
        XtSetArg( wargs[n], XmNforeground,  bg); n++;
        XtSetArg( wargs[n], XmNbackground,  fg); n++;
        XtSetValues( ugi->w, wargs, n);
}


display_ug_title( users )
User_group_info         *users;
{
	Arg  wargs[2];
	int len, n;
	char title_str[256];
	char user_str[256];
	char group_str[256];

	group_str[0] = NULL_CHAR;
	user_str[0] = NULL_CHAR;

	if ( users->group_view == VIEW_LOGGED_IN_GROUP )
		strcpy( group_str, users->user_gid_str );
	else if ( users->group_view == VIEW_LOGGED_IN_GROUPS )
		strcpy( group_str, "Logged In" );
	else if ( users->group_view == VIEW_BACKGROUND_GROUPS )
		strcpy( group_str, "Background" );
	else if ( users->group_view == VIEW_IGNORE_GROUPS ) 
		strcpy( group_str, "" );
	else if ( users->group_view == VIEW_ALL_GROUPS ) 
		strcpy( group_str, "All" );
	else if ( users->group_view == VIEW_ACTIVE_GROUPS ) 
		strcpy( group_str, "Active" );

	if ( users->user_view == VIEW_ALL_USERS )
		strcpy( user_str, "All" );
	else if ( users->user_view == VIEW_NO_USERS )
		strcpy( user_str, "None" );
	else if ( users->user_view == VIEW_LOGGED_IN_USER )
		strcpy( user_str, users->user_uid_str );
	else if ( users->user_view == VIEW_LOGGED_IN_USERS )
		strcpy( user_str, "Logged In" );
	else if ( users->user_view == VIEW_BACKGROUND_USERS )
		strcpy( user_str, "Background" );
	else if ( users->user_view == VIEW_ACTIVE_USERS  )
		strcpy( user_str, "Active" );

	if ( (len = strlen(group_str)) != 0 )
	    sprintf(title_str, "Group(%s) User(%s)", group_str, user_str);
	else
	    sprintf(title_str, "User(%s)",user_str);

	n = 0;
	XtSetArg( wargs[n], XmNtitle, title_str); n++;
	XtSetValues( users->ug_shell, wargs, n );
}


load_logged_in( users )
User_group_info  *users;
{
	static int 	first=1;
	int	   	reload=0;
	static long     old_utmp_time;
	long	 	new_utmp_time;
	struct utmp  	*p;

	if ( first )
	{
		time( &old_utmp_time ); 
		reload = 1;
	}
	else
	{
		/* UTMP_FILE */

		/* Get timestamp of utmp file */

		/* If utmp file has changed, reload login info */

		if ( new_utmp_time > old_utmp_time )
		{
		 users->group_login_list = free_list( users->group_login_list );
		 users->user_login_list = free_list( users->user_login_list );

		 reload = 1;
		}
	}

	if ( reload == 0 )
		return;

#ifndef ACCOUNTING
#define ACCOUNTING 9
#endif

	while ( (p = getutent()) != NULL )
	{
		/* printf("ut_user=%s\n", p->ut_user );
		DEBUG1(1, "ut_id=%s\n", p->ut_id );
		DEBUG1(1, "ut_line=%s\n", p->ut_line );
		DEBUG1(1, "ut_pid=%d\n", p->ut_pid ); */

		switch( p->ut_type ) {

#ifndef LINUX
			case EMPTY:
#endif
			case RUN_LVL:
			case BOOT_TIME:
			case OLD_TIME:
			case NEW_TIME:
			case INIT_PROCESS:
			case LOGIN_PROCESS:
			case DEAD_PROCESS:
			case ACCOUNTING:
				break;

			case USER_PROCESS:
				/* printf("ut_type=User Process(%d)\n", 
								p->ut_type );
								*/

				/* Using the string user name, lookup the
				 * user_id and group_id and insert each in
				 * the appropriate logged_in list
				 */

				perform_all_users( users, login_user, 
								p->ut_user );

				break;

			default:
				DEBUG1(1,"utmp unknown type=%d\n", p->ut_type );
		};

		/* printf("ut_time=%ld\n", p->ut_time ); */
	}

	endutent();
}

void
login_user( users, ptr, usr)
User_group_info   *users;
tnode *ptr;
char  *usr;
{
	int		       uid, gid;

	if ( ptr == NULL )
		return;

	/* XXX Should check utmp occasionally for new logins */

	if ( strcmp( usr, ptr->str_key ) == 0 )
	{
		uid = ptr->key;
		gid = ptr->parent->key;

		users->group_login_list = 
			    fwd_insert_lnode( users->group_login_list, gid,
								NULL );

		users->user_login_list = 
		 	     fwd_insert_lnode( users->user_login_list, uid,
								NULL);
	}
}
