/* File:        process_tip.c
 *
 * Description: Code to display a "process tip" for each displayed node in
 *		the process tree. A process tip is a window which is pop'ed
 *		when the mouse moves into the process tree node(widget).It's
 *		similar to a tooltip/widget_tip/ballon_help ...
 *
 * Author:      George MacDonald
 *
 * Copyright:   GPL - see http://www.gnu.org
 *
 * History:     George MacDonald        3/30/99        Created
 *
 *
 */

#ifndef __linux__
#include <sys/types.h>          /* For major/minor number */
#include <sys/mkdev.h>          /* To get major/minor number macros   */
#else
#include <sys/sysmacros.h>
#endif



#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <time.h>

#include "debug.h"

#include <Xpsi/Tree.h>

#include "list.h"
#include "tree.h"
#include "treeps.h"

void 		process_tip();

void 	process_tip_enter();
static void 	process_tip_leave();
static void 	process_tip_press();
static void 	process_tip_startup_timeout();
static void 	process_tip_uptime_timeout();

static Window 	create_process_tip_window();
void 		destroy_process_tip();

void 		process_tip_cancel_timer();

void 		process_tip_set_bg_color();
void 		process_tip_set_fg_color();
void 		process_tip_set_font();


Window 		Process_tip_window=0;

XtIntervalId 	Process_tip_interval_id=-1;

char 		*Process_tip_str=NULL;

Process_tree_info  *Current_pti_ptr=NULL;

time_t 		Last_process_tip_time=0;

Position 	Process_tip_x, Process_tip_y;

Widget   	Process_tip_w=NULL;


int		Process_tip_uptime=6000;
int		Process_tip_startup_delay=100;
char 		*Process_tip_fontname="8x13";
Pixel		Process_tip_bg=-1;
Pixel		Process_tip_fg=-1;

extern int debug;

extern char *find_dev_name();


void
setup_process_tips( startup_delay, uptime, fontname, fg, bg )
int    startup_delay;
int    uptime;
char   *fontname;
Pixel  fg, bg;
{
	Process_tip_startup_delay = startup_delay;
	Process_tip_uptime	  = uptime;
	Process_tip_fontname      = fontname;  /* Assumes it's persistent */
	Process_tip_bg		  = bg;
	Process_tip_fg		  = fg;
}

void
process_tip( pti_ptr )
Process_tree_info  *pti_ptr;
{
	Widget w = pti_ptr->node_w;

	if ( Process_tip_bg == -1 )
	{
		process_tip_set_bg_color( w, "white" );
		process_tip_set_fg_color( w, "black" );
	}
	
	XtAddEventHandler(w,EnterWindowMask, FALSE, process_tip_enter, pti_ptr);
	XtAddEventHandler(w,LeaveWindowMask, FALSE, process_tip_leave, pti_ptr);
	XtAddEventHandler(w,ButtonPressMask, FALSE, process_tip_press, pti_ptr);
}

void
process_tip_enter( w, pti_ptr, event, continueDispatch)
Widget w;
Process_tree_info  *pti_ptr;
XCrossingEvent *event;
Boolean *continueDispatch;
{
	Arg 		wargs[20];
	int 		n;
	time_t		time_now;
	Widget 		parent;
	XtAppContext 	ctx;
	Display 	*display;
	Position	widget_x, widget_y;
	Position	root_x, root_y;
	char		buf[8192];  /* Way more than needed */
	sysProcInfo     *pp;
	char		tty_str[8];
	char            *dev_name=NULL;
        major_t         major_no;
        minor_t         minor_no;
	int             hours, min, sec, hsec;
	

	DEBUG1( 8, "process_tip_enter: process info ptr(%d)\n", pti_ptr );


	display = XtDisplay( w );
	ctx     = XtWidgetToApplicationContext( w );

	*continueDispatch = TRUE;

	Process_tip_w = w;

	n = 0;
	XtSetArg( wargs[n], XtNwidth,  &widget_x ); n++;
	XtSetArg( wargs[n], XtNheight, &widget_y ); n++;
	XtGetValues( w, wargs, n );

	XtTranslateCoords( w, widget_x, widget_y, &root_x, &root_y );

	Process_tip_x = root_x - ( widget_x / 3 );
	Process_tip_y = root_y + 6;

	DEBUG4( 8, "process_tip_enter: x(%d) y(%d) root:x(%d)y(%d)\n", 
				widget_x, widget_y, root_x, root_y);

	DEBUG1( 8, "event->detail(%d)\n", event->detail );

	time( &time_now );
	if ( pti_ptr == NULL )
	{
		DEBUG0(8,"No tree ptr!!!\n");
		return ;
	}
	else	/* Create a tip? */
	{
		if ( Process_tip_window != 0 ) /* Already got one up!? */
		    destroy_process_tip(w);

		if ( Process_tip_str == NULL ) /* Should be */
		{
		    pp = pti_ptr->pp;
		    if ( pp == NULL )
		    {
		      Process_tip_str = strdup("Process Details Missing");
		    }
		    else /* Make tip string - Later make these user definable */
		    {
		      strcpy( tty_str, "  -    ");
#ifdef LINUX
		      if ( (PROCESS_CONTROL_TTY(pp) != -1) &&
					  (PROCESS_CONTROL_TTY(pp) != 0) )
#else
		      if ( PROCESS_CONTROL_TTY(pp) != -1) 
#endif
        	      {
                	  major_no = major( PROCESS_CONTROL_TTY(pp) );
                	  minor_no = minor( PROCESS_CONTROL_TTY(pp) );

                	  dev_name = find_dev_name( major_no, minor_no );
                	  if ( dev_name != NULL_STR )
			  {
				if ( strcmp( dev_name, "console") == 0 )
				    strncpy( &(tty_str[0]), "console", 7 );
				else
				    strncpy( &(tty_str[0]), &(dev_name[0]), 7 );
			  }
		      }

		      /* "  PID  UID  PRI   NI   RSS TTY    TIME COMMAND", */

			hours = PROCESS_CPU_TIME(pp).tv_sec / 3600;

            		if ( hours )
                		min = (PROCESS_CPU_TIME(pp).tv_sec % 3600) / 60;
            		else
                		min = PROCESS_CPU_TIME(pp).tv_sec / 60;
		
            		sec  = PROCESS_CPU_TIME(pp).tv_sec % 60;

		      sprintf( buf, 
			    "%5d %4d %4d %4d %5d %-7s%3d:%0.2d:%0.2d.%0.2d %s", 
				PROCESS_ID(pp),
				PROCESS_USER_ID(pp),
				PROCESS_PRIORITY(pp),
				PROCESS_NICE(pp),
				PROCESS_RESIDENT_SET_SIZE(pp),
				tty_str,
				hours, min, sec,
                        	PROCESS_CPU_TIME(pp).tv_nsec / 10000000,
				PROCESS_CMD_ARGS(pp) );

		      Process_tip_str = strdup(buf);
		    }
		}

		Current_pti_ptr = pti_ptr;

		/* If a tip was recently active, then user is probably
		 * looking for a tip so put one up right away.  Otherwise wait 
		 * till the mouse is inactive for a small amount of time 
		 * while in the widget before poping up tip.
		 */

		if ( Last_process_tip_time + 0 > time_now )
		{
		    DEBUG0(8,"Popup tip: Immediately \n");
		    Process_tip_window = create_process_tip_window( w, 
						Process_tip_str,
		    				Process_tip_x, Process_tip_y);


		    Process_tip_interval_id = XtAppAddTimeOut( ctx,
		    				Process_tip_uptime,
		    				process_tip_uptime_timeout, 
						w 			   );
		}
		else
		{
		    DEBUG0(8,"Popup tip: firing startup timer \n");

		    Process_tip_interval_id = XtAppAddTimeOut( ctx,
		    				Process_tip_startup_delay,
		    				process_tip_startup_timeout, w);
		}

		Last_process_tip_time = time_now;
	}
	
	*continueDispatch = FALSE;

	return;
}

void
process_tip_press( w, tipStr, event, continueDispatch)
Widget w;
char *tipStr;
XButtonPressedEvent *event;
Boolean *continueDispatch;
{
	destroy_process_tip(w);

	*continueDispatch = TRUE;
}


void
process_tip_leave( w, pti_ptr, event, continueDispatch)
Widget w;
Process_tree_info  *pti_ptr;
XCrossingEvent *event;
Boolean *continueDispatch;
{
	Arg 		wargs[20];
	int 		n;
	time_t		time_now;
	Widget 		parent;
	XtAppContext 	ctx;
	Display 	*display;

	display = XtDisplay( w );
	ctx     = XtWidgetToApplicationContext( w );

	*continueDispatch = TRUE;

	Process_tip_w = w;
	Process_tip_x = event->x_root; 
	Process_tip_y = event->y_root;  

	time( &time_now );
	if ( pti_ptr != NULL )
	{
		if ( event->detail == NotifyNonlinear )
	        {
			Last_process_tip_time = time_now;
			destroy_process_tip(w);
		}
		else
		{
			DEBUG0(8,"Leaving widget: on widgets border\n");
			destroy_process_tip(w);
		}

		Process_tip_w = NULL;
	}
	else	/* Eh? */
	{
		DEBUG0(8,"Leave widget with null pti_ptr!\n" );
		return;
	}
	
	*continueDispatch = FALSE;

	return;
}

void
process_tip_startup_timeout( w, id )  
Widget       w;
XtIntervalId id;
{
	XtAppContext 	ctx;

	ctx = XtWidgetToApplicationContext( w );

	if ( Process_tip_window == 0 )
	{
	    DEBUG1(8,"process_tip_startup_timeout: creating tip(%s)\n",
						Process_tip_str );

		if ( Process_tip_w == NULL || Process_tip_str == NULL )
		{
			DEBUG0(8,"process_tip_timeout: Too Late\n");
			return;
		}

		Process_tip_window = create_process_tip_window(
					Process_tip_w,
					Process_tip_str, 
					Process_tip_x, 
					Process_tip_y		);

		Process_tip_interval_id = XtAppAddTimeOut( ctx,
		    				Process_tip_uptime,
		    				process_tip_uptime_timeout, 
						w 			   );
	}
	else
	{
		DEBUG0(8,"process_tip_startup_timeout: Already up!!!\n");
	}
}

void
process_tip_uptime_timeout( w, id )  
Widget       w;
XtIntervalId id;
{

	if ( Process_tip_window == 0 )
	{
		DEBUG0(8, "process_tip_uptime_timeout: no tip!!\n");
	}
	else 
	{
		Process_tip_interval_id = -1;
		destroy_process_tip( w );
	}

}

void
destroy_process_tip_if_displayed( pti_ptr )
Process_tree_info  *pti_ptr;
{
	if ( Current_pti_ptr != pti_ptr )
		return;

	destroy_process_tip( pti_ptr->node_w );
}

void
destroy_process_tip( w )
Widget w;
{
	time_t	time_now;

	time( &time_now );

	if ( Process_tip_window != 0 )
	{
		DEBUG1(8,"destroying_process_tip(%s)\n", Process_tip_str );
		XDestroyWindow( XtDisplay( w ), Process_tip_window );
		Process_tip_window = 0;
	}

	if ( Process_tip_str != NULL )
	{
		free( Process_tip_str );
		Process_tip_str = NULL;
	}

	if ( Current_pti_ptr != NULL )
		Current_pti_ptr->processTipUp = False;;

	Current_pti_ptr = NULL;;

	Last_process_tip_time = time_now;

	process_tip_cancel_timer();
}

void
process_tip_cancel_timer()
{

	if ( Process_tip_interval_id != -1 )
	{
		DEBUG0(8,"Canceling tip Timer\n");
		XtRemoveTimeOut( Process_tip_interval_id );
		Process_tip_interval_id = -1;
	}
}

Window
create_process_tip_window( w, str, x, y )
Widget w;
char *str;
int x;
int y;
{
	static int first=1;
	static Display *display=NULL;
	static int screen;
	static Screen *screen_ptr=NULL;
        static char *window_name = "Process_tip";
        static XFontStruct *font_info;
        static int depth;
        static Visual *visual;
        static unsigned int class;
        static unsigned long valuemask;
        static unsigned long GCvaluemask=0;
        static XSetWindowAttributes setwinattr;
        static XGCValues values;
	static int border_width;
	static char *title_str=NULL;
	static int   title_len=0;
        int 		window_size = 0;
        XSizeHints 	size_hints;
        XEvent 		report;
        int  		str_x, str_y;
	unsigned int 	width, height;
	int 		len;
        GC 		gc;		/* Need to free this */
	Window		win;
	int 		win_x, win_y;

	if ( first )
	{
	    border_width = 1;

	    display = XtDisplay( w );
	    screen  = DefaultScreen( display );
 
	    screen_ptr = ScreenOfDisplay( display, screen );

            if ((font_info=XLoadQueryFont(display, Process_tip_fontname))==NULL)
            {
                fprintf(stderr,"Can't open %s font!!!!\n",Process_tip_fontname);
                return( 0 );
            }

	    depth = CopyFromParent;
            class = CopyFromParent;
            visual = CopyFromParent;

            valuemask = CWBackPixel | CWBorderPixel | CWOverrideRedirect ;

	    if ( Process_tip_bg == -1 )
	    	process_tip_set_bg_color( w, "yellow" );

	    if ( Process_tip_fg == -1 )
	    	process_tip_set_fg_color( w, "black" );

            setwinattr.background_pixel = Process_tip_bg;

            setwinattr.border_pixel = BlackPixel( display, screen );
            setwinattr.override_redirect = True;

	    first = 0;

	    title_str = 
		"  PID  UID  PRI   NI   RSS TTY            TIME COMMAND",
	    				   title_len = strlen( title_str );
	}

        len = strlen( str );

	if ( len < title_len )
	    width = XTextWidth( font_info, title_str, title_len ) + 6;
	else
	    width = XTextWidth( font_info, str, len ) + 6;

        height = (( font_info->max_bounds.ascent + 
				   font_info->max_bounds.descent ) * 2) + 4;

        str_x = 3;
        str_y = font_info->max_bounds.ascent + 2;


	win_x = x;

	if ( (win_x + width) > WidthOfScreen( screen_ptr ) )
	{
	        win_x = ((int)WidthOfScreen( screen_ptr )) - width;
                if ( win_x < 0 )
            	    win_x = 0;
	}

	win_y = y;
	if ( win_y + (height*2) > HeightOfScreen( screen_ptr ) )
	{
	        win_y = ((int)HeightOfScreen( screen_ptr )) - (4*height);
	        if ( win_y < 0 )
		    win_y = 0;
	}


        win = XCreateWindow( display, RootWindow(display, screen ),
                        win_x, win_y,
                        width, height,
                        border_width,
                        depth,
                        class,
                        visual,
                        valuemask,
                        &setwinattr );

        gc = XCreateGC( display, win, GCvaluemask, &values );

        XSetFont( display, gc, font_info->fid );

        XSetForeground( display, gc, Process_tip_fg );

	size_hints.flags = PPosition | PSize | PMinSize;
        size_hints.x = win_x;
        size_hints.y = win_y;
        size_hints.width = width;
        size_hints.height = height;
        size_hints.min_width = width;
        size_hints.min_height = height;

        XSetStandardProperties( display, win, window_name, NULL,
        				   0, NULL, 0, &size_hints );

        XMapWindow( display, win );


        XDrawString( display, win, gc, str_x, str_y, title_str, title_len );

        str_y += ( font_info->max_bounds.ascent + 
				   font_info->max_bounds.descent );

        XDrawString( display, win, gc, str_x, str_y, str, len );

	XFlush( display );

	XFreeGC( display, gc );

	if ( Current_pti_ptr != NULL )
		Current_pti_ptr->processTipUp = True;;

        return( win );
}

/* These must be called before first creation of a tip, otherwise
 * the default values will be used
 */

void
process_tip_set_bg_color( w, colorName )
Widget w;
char *colorName;
{
	Display *dpy;
        int      scr;
        Colormap cmap;
        XColor   color;
        XColor   exact;
        int      rc;

        dpy  = XtDisplay( w );
        scr  = DefaultScreen( dpy );
        cmap = DefaultColormap( dpy, scr );

        /* Lookup Pixel colors based on character string symbolic names */

        rc = XAllocNamedColor( dpy, cmap, colorName, &color, &exact );
        if ( rc )
                Process_tip_bg = color.pixel;
        else
        {
                fprintf(stderr, 
			"Warning: Couldn't allocate tip bg color %s\n",
                                                                colorName);
                Process_tip_bg = WhitePixel(dpy, scr);
        }
}


void
process_tip_set_fg_color( w, colorName )
Widget w;
char *colorName;
{
	Display *dpy;
        int      scr;
        Colormap cmap;
        XColor   color;
        XColor   exact;
        int      rc;

        dpy  = XtDisplay( w );
        scr  = DefaultScreen( dpy );
        cmap = DefaultColormap( dpy, scr );

        /* Lookup Pixel colors based on character string symbolic names */

        rc = XAllocNamedColor( dpy, cmap, colorName, &color, &exact );
        if ( rc )
                Process_tip_fg = color.pixel;
        else
        {
                fprintf(stderr, 
			"Warning: Couldn't allocate tip fg color %s\n",
                                                                colorName);
                Process_tip_fg = BlackPixel(dpy, scr);
        }
}

void
process_tip_set_font( fontName )
char *fontName;
{
	Process_tip_fontname = strdup( fontName );
}
