/* 
 * loadValues -  load linux process values
 *
 * Loads linux process values from 
 *
 * 	/proc/<pid>/*
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/times.h>
#include <sys/sysmacros.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <varargs.h>
#include <string.h>
#include <sys/param.h>          /* To get PAGESIZE i.e. system page      */
#include <dirent.h>             /* To get the /proc directory listing    */

#include <Xm/Xm.h>

#include <Xpsi/Tree.h>  /* Custom Tree widget */

#include "tree.h"       /* Generic m-ary tree package include file */
#include "list.h"       /* Simple list package                     */
#include "debug.h"      /* Macro based debug package               */

#include "button_bar.h"

#include "user_group.h" /* User_group display related definitions  */

#include "treeps.h"

#include "loadValues.h"

#include "os_info.h"



#ifndef NULL_CHAR
#define NULL_CHAR (char) 0
#endif


extern size_t 	PageSize;		/* From treeps.c */
extern int 	DoingKdeNameMagic;

int 	
load_sys_proc_info( pid, pli_ptr )
int   pid;
Process_list_info  *pli_ptr;
{
	char		proc_path[64];
	char		proc_info[64];
	int 		fd;
	int		rc;
	struct  stat    s;

	/* sprintf( proc_dir, "/proc/%d", pid ); */

	sprintf( proc_path, "%s/%d", PROCESS_INFO_DIRECTORY, pid );

	/* We call stat to see if we have access and to determine
	 * the effective uid/gid  of the process. 
	 *
	 */

	rc = stat( proc_path, &s );   
	if ( rc == -1 )
	{
		if ( Debug > 0 )
			perror("load_sys_proc_info: stat");
	 	DEBUG1(1,"load_sys_proc_info:can't stat %s\n", proc_path );
	 	return( IGNORE_PROCESS );
	}

	if ( pli_ptr == NULL )
	{
	 	DEBUG1(1,"load_sys_proc_info: null pli_ptr for %d\n", pid );
		return( IGNORE_PROCESS );
	}

	if ( pli_ptr->pp == NULL)
	{
	 	DEBUG1(1,"load_sys_proc_info: null pp for %d\n", pid );
		return( IGNORE_PROCESS );
	}

	/* On linux, the effective uid/euid are loaded from the
	 * /proc/<pid>/status files, since the /proc/pid doesn't
	 * show the effective group. 
	 */

	rc = load_proc_info( pid, pli_ptr, &(proc_path[0]) );
	if ( rc ==  IGNORE_PROCESS )
	{
	 	DEBUG1(1,"load_sys_proc_info: load_proc_info failed(%d)\n", pid );
		return( IGNORE_PROCESS );
	}

	return( ADD_PROCESS );
}

int 	
load_proc_info( pid, pli_ptr, proc_path )
int   pid;
Process_list_info  *pli_ptr;
char *proc_path;
{
	char		proc_info[64];
	int 		fd;
	int		i, rc;
	struct  stat    s;
	sysProcInfo	*pp;
	int		first_time=0;

	static struct linuxProcessStat lps;

	DEBUG2(1,"load_proc_info: pid(%d) path(%s)\n", pid, proc_path );

	pp = pli_ptr->pp;


	rc = loadProcessStat( pid, &lps );  /* Gets most fields */
	if ( rc != 0 )
	{
		DEBUG1(1,"load_proc_info: loadProcesStat failed(%d)\n", pid );
		return( IGNORE_PROCESS );
	}

	/* The following could probably be skipped if we already have loaded the
	 * command line as in almost all cases it doesn't change
	 */

        rc = loadCmdLine( pid, &lps );	/* Gets just command line */
	if ( rc != 0 )
	{
		DEBUG1(1,"load_proc_info: loadCmdLine failed(%d)\n", pid );
		return( IGNORE_PROCESS );
	}

        rc = loadProcessStatus( pid, &lps ); /* Get uid/euid/gid/egid */
	if ( rc != 0 )
	{
		DEBUG1(0,"load_proc_info: loadProcesStatus failed(%d)\n", pid );
		return( IGNORE_PROCESS );
	}


	pli_ptr->euid = lps.euid;
	pli_ptr->egid = lps.egid;


	pp->pr_pid	= pid;


	pp->pr_sname = lps.state; 


	if ( (pp->last_time.tv_sec == 0) && ( pp->last_time.tv_nsec == 0 ) )
	{
		first_time++;
	}

	pp->last_time.tv_sec = pp->pr_time.tv_sec;
	pp->last_time.tv_nsec = pp->pr_time.tv_nsec;

	pp->pr_nice	= lps.nice;
	pp->pr_flag	= lps.flags;
	pp->pr_uid	= lps.uid;
	pp->pr_gid	= lps.gid;
	pp->pr_ppid	= lps.ppid;
	pp->pr_pgrp	= lps.pgrp;
	pp->pr_sid	= lps.sid;
	pp->pr_addr	= (caddr_t)lps.startcode;
	pp->pr_size	= lps.vsize / PageSize;
	pp->pr_rssize	= lps.rss;
	pp->pr_wchan	= (caddr_t)lps.wchan;

	pp->pr_start.tv_sec	= startTimeSec( lps.starttime );
	pp->pr_start.tv_nsec	= 0;

	/* The following assumes 100 HZ
	 *
	 * millisec = .001
	 * microsec = .000001
	 * nanosec  = .000000001
	 *
	 * 1 HZ = .01
	 *
	 * Convert HZ to Nanosecs:   1HZ = 10 millisec
	 *			     1HZ = 10000 microsec
	 *			     1HZ = 10000000 nanosec
	 */

	pp->pr_time.tv_sec	= (lps.utime + lps.stime) / HZ;
	pp->pr_time.tv_nsec	= ((lps.utime + lps.stime) % HZ) * 10000000;

	if ( pp->pr_sname == 'S' ) /* If recently run, synthesize new state */
	{
	    if ( first_time )
	    {
		pp->pr_sname = 'S';
	    }
	    else if ( pp->pr_time.tv_nsec == pp->last_time.tv_nsec )
	    {
	    	if ( pp->pr_time.tv_sec != pp->last_time.tv_sec )
			pp->pr_sname = 'A';
	    }
	    else
	    {
		pp->pr_sname = 'A';
	    }
	}
	pp->pr_pri	= lps.priority;
	pp->pr_ttydev	= lps.tty;

	strcpy( pp->pr_clname, "LinuxTS");

	strncpy( &(pp->pr_fname[0]), &(lps.fname[0]), LINUX_FILE_NAME_SIZE );
	pp->pr_fname[LINUX_FILE_NAME_SIZE-1] = NULL_CHAR;


        if ( lps.cmdLine == NULL )
	{
	    /* If we already have the arg string, then keep it as it is, 
	     * otherwise in desperation we use the filename with a marker to 
	     * indicate the lack of args.
	     */
	    if ( pp->pr_psargs == NULL )
	    {
	        pp->pr_psargs = (char *) malloc( strlen( lps.fname ) + 2 );
	        if ( pp->pr_psargs != NULL )
		{
	        	pp->pr_psargs[0] = '>';
			strcpy( &(pp->pr_psargs[1]), lps.fname );
		}
	    }
	}
	else
	{
	    if ( pp->pr_psargs != NULL )
	    {
	    	free( pp->pr_psargs );
	    	pp->pr_psargs = NULL;
	    }

	    /* Note we pass the malloced space upwards, generally we free it
	     * in this routine(as we get called to reload the info), however
	     * when the process node is destructed the calling code 
	     * MUST free it.
	     */

	    pp->pr_psargs = lps.cmdLine;   /* Passing malloced space up */
	}

	/* If fname == kdeinit we do some special processing to pull the real
 	* command name from the first arg of ps_args
 	*/

	if ( DoingKdeNameMagic )
	{
	    if ( strcmp( lps.fname, "kdeinit" ) == 0 )
	    {
	    	if ( lps.cmdLine != NULL )
		{
			strncpy( lps.fname, &(lps.cmdLine[9]),
						LINUX_FILE_NAME_SIZE );
			lps.fname[LINUX_FILE_NAME_SIZE-1] = NULL_CHAR;

			if ( strncmp( lps.fname, "Running", 7 ) == 0 )
			    strcpy( lps.fname, "kdeinit" );

			for ( i = 0 ; i < LINUX_FILE_NAME_SIZE ; i++ )
			{
			    pp->pr_fname[i] = lps.fname[i];
			    if (( lps.fname[i] == ' ' ) ||
			    	( lps.fname[i] == NULL_CHAR ))
			    {
			        pp->pr_fname[i] = NULL_CHAR;
				break;
			    }
			}
		}
	    }
	}


	return( ADD_PROCESS );
}


#define MAX_ALLOWED_CMD_LINE_INPUT_LENGTH  8192

loadCmdLine( pid, lps )
int pid;
struct linuxProcessStat *lps;
{
	char 	cmdLinePath[60];
	FILE 	*fp;
	int	len, maxlen;
	char	*buf[MAX_ALLOWED_CMD_LINE_INPUT_LENGTH];

	/* Note on Entry we assume lps->cmdLine is undefined */

	sprintf( cmdLinePath, "/proc/%d/cmdline", pid );

	fp = fopen( cmdLinePath, "r" );
	if ( fp == NULL )
	{
		DEBUG1( 1, "Can't open cmdLine file(%s)", cmdLinePath );
		return -1;
	}

	/* Note: may get null string if swapped out or zombie !!! */

	maxlen = MAX_ALLOWED_CMD_LINE_INPUT_LENGTH;

	len = fread( &(buf[0]), 1, maxlen, fp );

	if ( len > 0 )
	{
	    /* len = fread( &(lps->cmdLine[0]), 1, maxlen, fp ); */
	    /* lps->cmdLine[ maxlen - 1 ] = NULL_CHAR; */

	    buf[ maxlen - 1 ] = NULL_CHAR;

	    lps->cmdLine = (char *) malloc( len + 1 );
	    if ( lps->cmdLine == NULL )
	    {
		DEBUG1( 0, "Can't malloc space for cmdLine pid(%d)", pid );
		fclose( fp );
		return -1;
	    }
	    else
	    {
	        memcpy( (void *) lps->cmdLine, (void *) buf, len );
	    }
	}
	else
	{
	    lps->cmdLine = NULL;
	}

	fclose( fp );

	if ( len >= maxlen )
		len = maxlen - 1;
	
	nullToSpace( len, lps->cmdLine );

	/* Note caller is responsible for freeing space */

	return 0;
}

#define CONVERT_INT() 	( (str = strtok( NULL, " " )) == NULL) ? 0 : atoi( str )
#define CONVERT_LONG() 	( (str = strtok( NULL, " " )) == NULL) ? 0 : strtoul(str, NULL, 10)

loadProcessStatus( pid, lps )
int pid;
struct linuxProcessStat *lps;
{
	char 	statusPath[60];
	FILE 	*fp;
	int 	i=0;
	char 	*p;
	int 	len=128;
	char	*tok;
	char  	buf[128];   
	int	gotAllFields=0;
	static  int uidLine=0;
	static  int gidLine=0;


	sprintf( statusPath, "/proc/%d/status", pid );

	fp = fopen( statusPath, "r" );
	if ( fp == NULL )
	{
		fprintf( stderr, "Can't open status file(%s)\n", statusPath );
		return -1;
	}

	/* Figure out the position of the uid and gid line. This can
 	 * vary depending on kernel version. Only need to do this once
	 * at startup.
	 */

	if ( uidLine == 0 )  
	{
		while ( (p = fgets( buf, len, fp )) != NULL )
		{
			if ( strncmp( "Uid:", buf, 4 ) == 0 )
				uidLine = i;

			if ( strncmp( "Gid:", buf, 4 ) == 0 )
				gidLine = i;
			i++;
		}

		rewind( fp );
		i = 0;
	}

	while ( (p = fgets( buf, len, fp )) != NULL )
	{
		/* printf("Status line(%d): %s", i, buf ); */

		/* Only add some members to structure */

		if ( i == uidLine )   /* get uid/euid */
		{
			/* printf("User line(%d): %s", i, buf ); */

			tok = strtok( buf, "	" );  /* Toss label */

			tok = strtok( NULL, " 	" ); /* Note tab separator */
			if ( tok == NULL )
				lps->uid = -1;
			else
				lps->uid = atoi( tok );

			tok = strtok( NULL, " 	" ); /* Note tab separator */
			if ( tok == NULL )
				lps->euid = -1;
			else
				lps->euid = atoi( tok );
		}

		if ( i == gidLine )  /* get gid/egid */
		{
			/* printf("Group line(%d): %s", i, buf ); */

			tok = strtok( buf, "	" );  /* Toss label */

			tok = strtok( NULL, " 	" ); /* Note tab separator */
			if ( tok == NULL )
				lps->gid = -1;
			else
				lps->gid = atoi( tok );

			tok = strtok( NULL, " 	" ); /* Note tab separator */
			if ( tok == NULL )
				lps->egid = -1;
			else
				lps->egid = atoi( tok );

			gotAllFields++;
		}
		i++;

		if ( gotAllFields )
			break;
	}

	fclose( fp );

	return 0;
}


loadProcessStat( pid, lps )
int pid;
struct linuxProcessStat *lps;
{
	char 	statPath[60];
	FILE 	*fp;
	char  	buf[512];   /* 37 * 10 - rounded up to power of 2 */
	int 	i=0;
	char 	*p;
	int 	len=256;
	char	*tok;
	char    *str;
	int	rc=0;


	sprintf( statPath, "/proc/%d/stat", pid );

	fp = fopen( statPath, "r" );
	if ( fp == NULL )
	{
		fprintf( stderr, "Can't open stat file(%s)\n", statPath );
		return -1;
	}

	p = fgets( buf, len, fp );
	if ( p != NULL )
	{
		/* printf("Stat line(%d): %s\n", i++, buf ); */

		/* Convert to structure */

		tok = strtok( buf, " " );
		if ( tok == NULL )
		  	lps->pid = -1;
		else
		  	lps->pid = atoi( tok );

		tok = strtok( NULL, ")" );
		if ( tok == NULL )
		  	lps->fname[0] = NULL_CHAR;
		else
		{
		    	strncpy( &(lps->fname[0]), &(tok[1]), LINUX_FILE_NAME_SIZE-1 );
			lps->fname[LINUX_FILE_NAME_SIZE-1] = NULL_CHAR;
		}

		tok = strtok( NULL, " " );
		if ( tok == NULL )
			lps->state = 'Z';
		else
			lps->state = tok[0];

		lps->ppid 	= CONVERT_INT();
		lps->pgrp 	= CONVERT_INT();
		lps->sid 	= CONVERT_INT();
		lps->tty 	= CONVERT_INT();
		if ( lps->tty == 0 )
			lps->tty = -1;

		lps->tpgid 	= CONVERT_INT();

		lps->flags 	= CONVERT_INT();
		lps->minflt 	= CONVERT_INT();
		lps->cminflt 	= CONVERT_INT();
		lps->majflt 	= CONVERT_INT();
		lps->cmajflt 	= CONVERT_INT();

		lps->utime 	= CONVERT_INT();
		lps->stime 	= CONVERT_INT();
		lps->cutime 	= CONVERT_INT();
		lps->cstime 	= CONVERT_INT();
		lps->priority 	= CONVERT_INT();
		lps->nice 	= CONVERT_INT();
		lps->timeout 	= CONVERT_INT();
		lps->itrealvalue= CONVERT_INT();
	
		lps->starttime	= CONVERT_LONG();
		lps->vsize	= CONVERT_LONG();
		lps->rss	= CONVERT_LONG();
		lps->rlim	= CONVERT_LONG();
		lps->startcode	= CONVERT_LONG();
		lps->endcode	= CONVERT_LONG();

		lps->startstack	= CONVERT_LONG();
		lps->kstkesp	= CONVERT_LONG();
		lps->kstkeip	= CONVERT_INT();
		lps->signal	= CONVERT_INT();
		lps->blocked	= CONVERT_INT();
		lps->sigIgnore	= CONVERT_INT();
		lps->sigCatch	= CONVERT_INT();

		/* lps->wchan	= CONVERT_INT(); */
		/* Not really a long, just using macro for the unsigned conversion */
		lps->wchan	= CONVERT_LONG(); 

		lps->nswap	= CONVERT_INT();
		lps->cnswap	= CONVERT_INT();

		if ( osMajorReleaseNumber > 1 )
		      lps->exitcode	= CONVERT_INT();
	}
	else
	{
		rc = -1;
	}

	fclose( fp );

	return rc;
}

printProcessStat( lps )
struct linuxProcessStat *lps;
{
	printf( "Name:\t\t%s\n", lps->fname );

	if ( lps->cmdLine[0] == NULL_CHAR )
		printf( "CmdLine:\t(%s)\n", &(lps->fname[0]) );
	else
		printf( "CmdLine:\t%s\n", &(lps->cmdLine[0]) );


	printf( "Pid:\t\t%d\n", lps->pid );
	printf( "State:\t\t%c\n", lps->state );
	printf( "Ppid:\t\t%d\n", lps->ppid );
	printf( "Pgrp:\t\t%d\n", lps->pgrp );
	printf( "SessionId:\t%d\n", lps->sid        );


	printf( "UserId:\t\t%d\n", lps->uid        );
	printf( "EUserId:\t%d\n", lps->euid        );

	printf( "GroupId:\t%d\n", lps->gid        );
	printf( "EGroupId:\t%d\n", lps->egid        );


	printf( "Tty:\t\tmajor(%d), minor(%d)\n", major(lps->tty), minor(lps->tty) );

	printf( "Tty_gpid:\t%d\n", lps->tpgid      );

	printf( "Flags:\t\t%x\n", lps->flags      );
	printf( "MinorFlt:\t%d\n", lps->minflt     );
	printf( "CMinorFlt:\t%d\n", lps->cminflt    );
	printf( "MajorFlt:\t%d\n", lps->majflt     );
	printf( "CMajorFlt:\t%d\n", lps->cmajflt    );

	printf( "UserTime:\t%d\n", lps->utime      );
	printf( "SysTime:\t%d\n", lps->stime      );
	printf( "CUserTime:\t%d\n", lps->cutime     );

	printf( "Total Process CPU time:  min %d, %d sec, %d nanosec\n", 
				((lps->utime + lps->stime) / HZ)/60,
				((lps->utime + lps->stime) / HZ)%60,
				((lps->utime + lps->stime) % HZ) * 10000000 );

	printf( "CSysTime:\t%d\n", lps->cstime     );
	printf( "Priority:\t%d\n", lps->priority   );
	printf( "Nice:\t\t%d\n", lps->nice       );
	printf( "Timeout:\t%d\n", lps->timeout    );
	printf( "ItRealValue:\t%d\n", lps->itrealvalue);

	printf( "StartTime:\t(%d)%s", startTimeSec( lps->starttime ),
			startTimeStr( lps->starttime ));


	printf( "Vsize:\t\t%d k\n", lps->vsize / 1024      );
	printf( "RSS:\t\t%d Pages\n", lps->rss        );
	printf( "RLimit:\t\t%d M\n", lps->rlim / (1024*1024) );

	printf( "StartCode:\t0x%8.8x\n", lps->startcode  );
	printf( "EndCode:\t0x%8.8x\n", lps->endcode    );
	printf( "StartStack:\t0x%8.8lX\n", lps->startstack );
	printf( "KernStackkESP:\t0x%8.8lX\n", lps->kstkesp );
	printf( "KernStackEIP:\t0x%8.8lX\n", lps->kstkeip    );
	printf( "Signals:\t0x%x\n", lps->signal     );
	printf( "SignalsBlocked:\t0x%x\n", lps->blocked    );
	printf( "SignalsIgnore:\t0x%x\n", lps->sigIgnore  );
	printf( "SignalsCatch:\t0x%x\n", lps->sigCatch   );
	printf( "WaitChannel:\t0x%x\n", lps->wchan      );
	printf( "NSwap:\t\t%d\n", lps->nswap      );
	printf( "CNSwap:\t\t%d\n", lps->cnswap     );

	if ( osMajorReleaseNumber > 1 )
		printf( "ExitCode:\t\t%d\n", lps->exitcode     );

}



/* Convert process start time in jiffies to a human readable
 * time string
 */

char *
startTimeStr( start_time )
unsigned long start_time;
{
        static time_t 	boot_at=0;
        struct tms 	tbuf;
        time_t 		pst;

	if ( boot_at == 0 )
        	boot_at = time((time_t *)0) - (times(&tbuf)/HZ);

	pst = boot_at + (start_time/HZ);
	return ( ctime( &pst ) );
}

int
startTimeSec( start_time )
unsigned long start_time;
{
        static time_t 	boot_at=0;
        struct tms 	tbuf;
        time_t 		pst;

	if ( boot_at == 0 )
        	boot_at = time((time_t *)0) - (times(&tbuf)/HZ);

	return boot_at + (start_time/HZ);
}

/* Convert a buffer of N characters, replacing nulls with spaces 
 * leave null terminator
 */

nullToSpace( len, buf )
int len;
char *buf;
{
	int i;

	if ( buf == NULL )
		return;

	for ( i = 0; i < len ; i++ )
	{
		if ( buf[i] == NULL_CHAR )
			buf[i] = ' ';
	}

	buf[i] = NULL_CHAR;
}

