/* 
 * linuxDetails -  display linux specific details
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/times.h>
#include <sys/sysmacros.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <varargs.h>
#include <string.h>
#include <sys/param.h>          /* To get PAGESIZE i.e. system page      */
#include <dirent.h>             /* To get the /proc directory listing    */
#include <errno.h>

#include <Xm/Xm.h>
#include <Xm/PushB.h>
#include <Xm/MessageB.h>
#include <Xm/Text.h>


#include "tree.h"       /* Generic m-ary tree package include file */
#include "list.h"       /* Simple list package                     */
#include "debug.h"      /* Macro based debug package               */

#include <Xpsi/Tree.h>

#include "button_bar.h"
#include "user_group.h" /* User_group display related definitions  */

#include "os_info.h"

#include "treeps.h"
#include "loadValues.h"



#ifndef NULL_CHAR
#define NULL_CHAR (char) 0
#endif

static int ProcHasResolvedPaths=0;	/* Set true when proc entries contain resolved pathnames */

extern size_t PageSize;		/* From treeps.c */

void do_dirInfo();
void do_envInfo();
void do_memInfo();
void do_detailInfo();
void do_detailPopup();
char *getProcessInfo();
char *sprintfProcessStat();

static void
initializeLinuxDetails()
{
	/* Don't know when this was introduced, this is a guess */

	if ( osMajorReleaseNumber > 1  && osMinorReleaseNumber > 1 )
	{
		ProcHasResolvedPaths++;
	}
}

Widget
linuxDetailsPanel( parent, attachto, pid, pName )
Widget parent;
Widget attachto;
int pid;
char *pName;
{
	Widget w;
	int n;
	Arg wargs[12];
	static int first=1;

	if ( first )
	{
		initializeLinuxDetails();
		first = 0;
	}

	/* Combine CWD, Root Dir, Open File Descriptor Info */

 	n = 0;
        XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
        XtSetArg( wargs[n], XmNbottomWidget, attachto); n++;
        XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
        XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNleftPosition, 4); n++;
        XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNrightPosition, 24); n++;
XtSetArg( wargs[n], XmNuserData, ( XtPointer ) strdup( pName )); n++; /* Small mem leak */
        w = XtCreateManagedWidget("File/Dir", xmPushButtonWidgetClass,
                                                parent, wargs, n );

        XtAddCallback( w, XmNactivateCallback, do_dirInfo, 
						   (XtPointer) pid );

	/* Environment */
 	n = 0;
        XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
        XtSetArg( wargs[n], XmNbottomWidget, attachto); n++;
        XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
        XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNleftPosition, 28 ); n++;
        XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNrightPosition, 48 ); n++;
        w = XtCreateManagedWidget("Environment", xmPushButtonWidgetClass,
                                                parent, wargs, n );

        XtAddCallback( w, XmNactivateCallback, do_envInfo, 
						   (XtPointer) pid );

	/* Memory ( statm, maps??  */

 	n = 0;
        XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
        XtSetArg( wargs[n], XmNbottomWidget, attachto); n++;
        XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
        XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNleftPosition, 52); n++;
        XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNrightPosition, 72); n++;
        w = XtCreateManagedWidget("Mem Maps", xmPushButtonWidgetClass,
                                                parent, wargs, n );

        XtAddCallback( w, XmNactivateCallback, do_memInfo, 
						   (XtPointer) pid );

	/* Status Details ( combine status, stat ) */

 	n = 0;
        XtSetArg( wargs[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
        XtSetArg( wargs[n], XmNbottomWidget, attachto); n++;
        XtSetArg( wargs[n], XmNbottomOffset, 2); n++;
        XtSetArg( wargs[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNleftPosition, 76); n++;
        XtSetArg( wargs[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg( wargs[n], XmNrightPosition, 96); n++;
        w = XtCreateManagedWidget("Raw Details", xmPushButtonWidgetClass,
                                                parent, wargs, n );

        XtAddCallback( w, XmNactivateCallback, do_detailInfo, 
						   (XtPointer) pid );

	return( w );
}


#define MAX_FILE_INFO 20000  /* SWAG */
#define MAX_PATH 256

#define P 	pos = strlen( buf )



void
do_dirInfo( w, pid, call_data )
Widget w;
int pid;
void *call_data;
{
	char 		path[MAX_PATH];
	char 		exe_path[MAX_PATH];
	char 		cwd_path[MAX_PATH];
	char 		root_path[MAX_PATH];
	char 		resolved_path[MAX_PATH];
	char 		buf[MAX_FILE_INFO];
	char		dialogLabel[128];
	char 		*pName;
        static DIR      *dirp;
        struct dirent   *direntp;
        struct stat 	s;
        char            *fname;
	int 		pos, rc, n;
	Arg 		wargs[3];


	n = 0;
	XtSetArg( wargs[n], XmNuserData, &pName ); n++;
	XtGetValues( w, wargs, n );

 	sprintf( path, "/proc/%d/root", pid );
	absPath( path, root_path );

 	sprintf( path, "/proc/%d/cwd", pid );
	absPath( path, cwd_path );


	findExe( pid, pName, exe_path );

	sprintf(exe_path, "%s", exe_path );
	sprintf(cwd_path, "%s", cwd_path );
	sprintf(root_path, "%s", root_path );

	sprintf( buf, "%s %s\n%s %s\n%s %s\n\n",

	    "Executable Location: ", exe_path,
	    "Processes CWD:       ", cwd_path,
	    "Processes Root Dir:  ", root_path );

	pos = strlen( buf );

	if ( ProcHasResolvedPaths )
	{
	    sprintf( &(buf[pos]), "%s\n\n%s\n%s\n",
	        "Open File Descriptor Information i.e. ls -l /proc/$pid/fd",
"Permission Ln  UID   GID     Size          Last Mod       Filename-> Resolved Entity",
"========== == ===== ===== ===========   === == ==== ===== ==========================");
	}
	else
	{
	    sprintf( &(buf[pos]), "%s\n\n%s\n%s\n",
	        "Open File Descriptor Information i.e. ls -lUL /proc/$pid/fd",
"Permission Ln  UID   GID     Size          Last Mod       Filename->[maj,min]:inode",
"========== == ===== ===== ===========   === == ==== ===== =========================");
	}

	pos = strlen( buf );


	/* Add Open File descriptor info */

 	sprintf( path, "/proc/%d/fd", pid );

        dirp = opendir( path );
        if ( dirp == NULL )
        {
            sprintf( &(buf[pos]), "%s\n", "Failed to open /proc/{pid}/fd" );
	}
	else
	{
	  while ( (direntp = readdir( dirp )) != NULL )
          {
                fname = direntp->d_name;

                if ( fname[0] == '.' )  /* Skip ., .., any .files */
                        continue;

 		sprintf( path, "/proc/%d/fd/%s", pid, fname );

        	rc = stat( path, &s );
        	if ( rc != 0 )
        	{
                   sprintf( &(buf[pos]), "%s %s\n", "Cant stat file -", fname );
		   pos = strlen( buf );
        	}
		else
		{
			getPermStr( s.st_mode, &(buf[pos]) ); P;
			sprintf( &(buf[pos]), " %2d ", s.st_nlink ); P;
			sprintf( &(buf[pos]), "%5d", s.st_uid ); P;
			sprintf( &(buf[pos]), " %5d", s.st_gid ); P;
			sprintf( &(buf[pos]), " %11d   ", s.st_size ); P;

			getFileTime( s.st_mtime, &(buf[pos]) ); P;

			sprintf( &(buf[pos]), " %4s -> ", fname ); P;

			if ( ProcHasResolvedPaths )
			{
			    rc = readlink( path, resolved_path, MAX_PATH-1 );

			    if ( rc > 0 )
			    {
			    	resolved_path[rc] = NULL_CHAR;
			        sprintf( &(buf[pos]), "%s\n", resolved_path ); P;
			    }
			}
			else
			{
			    sprintf( &(buf[pos]), "[%02x,%02x]", 
				major(s.st_dev),minor(s.st_dev) ); P;
			     sprintf( &(buf[pos]), ":%ld\n", s.st_ino ); P;
			}

		}
	   }
	   closedir( dirp );
	}

	sprintf( dialogLabel, "File and Directory info for pid(%d)", pid );

	do_detailPopup( w, dialogLabel, 25, 120, False, buf );
}

getFileTime( ft, buf )
time_t ft;
char *buf;
{
  	struct tm *t;
	char      *month;

  	t = localtime( &ft );
	if ( t == NULL )
		return;

	switch( t->tm_mon ) {
	    case  0: month = "Jan"; break;
	    case  1: month = "Feb"; break;
	    case  2: month = "Mar"; break;
	    case  3: month = "Apr"; break;
	    case  4: month = "May"; break;
	    case  5: month = "Jun"; break;
	    case  6: month = "Jul"; break;
	    case  7: month = "Aug"; break;
	    case  8: month = "Sep"; break;
	    case  9: month = "Oct"; break;
	    case 10: month = "Nov"; break;
	    case 11: month = "Dec"; break;

	    default: month = "XXX"; break;
	}

	sprintf( buf, "%s %2d %4d %02d:%02d", month, t->tm_mday, 
			t->tm_year + 1900, t->tm_hour, t->tm_min );
}

getPermStr( m, buf )
int m;
char *buf;
{
	char t, ur,uw,ux,gr,gw,gx,ar,aw,ax;

	if ( S_ISLNK(m) ) 		t = 'l';
	else if ( S_ISDIR(m) )		t = 'd';
	else if ( S_ISREG(m) )		t = '-';
	else if ( S_ISCHR(m) )		t = 'c';
	else if ( S_ISBLK(m) )		t = 'b';
	else if ( S_ISFIFO(m) )		t = 'p';
	else if ( S_ISSOCK(m) )		t = 's';

	t = ( m & S_ISVTX ) ? 'S' : t;		/* Sticky bit */

	ur = ( m & S_IRUSR ) ? 'r' : '-';
	uw = ( m & S_IWUSR ) ? 'w' : '-';
	ux = ( m & S_IXUSR ) ? 'x' : '-';
	ux = ( m & S_ISUID ) ? 's' : ux;	/* Set UID */

	gr = ( m & S_IRGRP ) ? 'r' : '-';
	gw = ( m & S_IWGRP ) ? 'w' : '-';
	gx = ( m & S_IXGRP ) ? 'x' : '-';
	gx = ( m & S_ISGID ) ? 's' : gx;	/* Set GID */

	ar = ( m & S_IROTH ) ? 'r' : '-';
	aw = ( m & S_IWOTH ) ? 'w' : '-';
	ax = ( m & S_IXOTH ) ? 'x' : '-';

	buf[0] = t;
	buf[1] = ur; buf[2] = uw; buf[3] = ux;
	buf[4] = gr; buf[5] = gw; buf[6] = gx;
	buf[7] = ar; buf[8] = aw; buf[9] = ax;
	buf[10] = (char) 0;
}

#define MAX_SYM_SIZE 128

findWaitSymbol( systemMap, addr, len, buf )
char *systemMap;
caddr_t addr;
int len;
char *buf;
{
	caddr_t 	iaddr;
	FILE 		*fp;
	caddr_t   	sym_addr=0;
	char 		sym_class;
	char 		sym_name[MAX_SYM_SIZE];
	char 		last_sym_name[MAX_SYM_SIZE];
	int  		rc, notDone=1;

	strcat( buf, "-------" );

	fp = fopen( systemMap, "r" );
	if ( fp == NULL )
		return;

	last_sym_name[0] = NULL_CHAR;
	while ( notDone )
	{
		rc = fscanf( fp, "%x %c %s\n", &sym_addr, &sym_class,
					sym_name );
		if ( rc <= 0 )
		{
		    notDone = 0;
		}
		else if ( sym_addr > addr )
		{
		    strncpy( buf, last_sym_name, len-2 );
		    buf[len-1] = NULL_CHAR;
		    notDone = 0;
		}
		else
		{
		    strcpy( last_sym_name , sym_name );
		}
	}

	fclose( fp );

	/* could cache the sym table */
}

findExe( pid, pName, exe_path )
int pid;
char *pName;
char *exe_path;
{
 	char path[MAX_PATH];
        FILE *fp;
        struct stat s;
        long exe_ino;
        int  rc, len;
        char cmdStr[256];
        char line[256];

        sprintf( path, "/proc/%d/exe", pid );

	if ( ProcHasResolvedPaths )
	{
	    rc = readlink( path, exe_path, MAX_PATH-1 );

	    if ( rc > 0 )
	    	exe_path[rc] = NULL_CHAR;
	    else
                strcpy( exe_path, "Unreadable" );
	}
	else
	{
	    /* requires use of locate, should use different method if suid */
	    /* Fails if process name is truncated form of filename */

            strcpy( exe_path, "Error trying to locate executable\n" );

            rc = stat( path , &s );
            if ( rc != 0 )
            {
                strcpy( exe_path, "Invalid permissions or program exited\n" );
		return ;
            }
            exe_ino = s.st_ino;

	    sprintf( cmdStr, "/usr/bin/locate *%s", pName);
            fp = popen( cmdStr, "r" );
            if ( fp == NULL )
            {
                strcpy( exe_path, "Error trying to locate executable\n" );
                return;
            }

            while ( fgets( line, MAX_PATH - 1, fp ) != NULL  )
            {
                len = strlen( line );
                line[ len - 1 ] = (char) 0;

                rc = stat( line , &s );
                if ( rc == 0 )
                {
                    if ( s.st_ino == exe_ino )
                    {
                	strcpy( exe_path, line );
                        break;
                    }
		}
            }

	    pclose( fp );
	}
}

absPath( path, buf )
char *path;
char *buf;
{
	int rc;
	char savePath[MAX_PATH];
	char absPath[MAX_PATH];
	char *ap;

	if ( ProcHasResolvedPaths )
	{
	    rc = readlink( path, buf, MAX_PATH-1 );

	    if ( rc > 0 )
	    	buf[rc] = NULL_CHAR;
	    else
                strcpy( buf, "Unreadable" );
	}
	else
	{
            ap = getcwd( savePath, MAX_PATH );
	    if ( ap == NULL )
	    {
                strcpy( buf, "Uknown or Unreadable" );
		return;
	    }

            rc = chdir( path );
            if ( rc != 0 )
            {
                strcpy( buf, "Uknown or Unreadable" );
                return;
            }

            ap = getcwd( absPath, MAX_PATH );
            if ( ap != NULL )
            	strcpy( buf, absPath  );
            else
                strcpy( buf, "Unreadable" );

            chdir( savePath );
	}
}


#define MAX_ENV 10000

void
do_envInfo( w, pid, call_data )
Widget w;
int pid;
void *call_data;
{
	FILE 	*fp;
	int	len;
	char	buf[MAX_ENV];  /* A guess, we truncate to max */
	char 	path[MAX_PATH];
	char    dialogTitle[128];

	sprintf( path, "/proc/%d/environ", pid );

	fp = fopen( path, "r" );
	if ( fp == NULL )
	{
	    DEBUG1( 1, "Can't open environ file(%s)", path );
	    sprintf( buf, "%s(%s)\n\n%s\n\n%s",
		"Error opening environment info file", path, 
		"The reason as reported by the Operating System is:", 
		strerror( errno ) );
	}
	else
	{
	    len = fread( &(buf[0]), 1, MAX_ENV-1, fp );
	    if ( len > 0 )
	    {
	        buf[ MAX_ENV - 1 ] = NULL_CHAR;
	    	nullToNewline( len, buf );
	    }
	    else
	    {
	        sprintf( buf, "%s\n", "The Environment Appears to be Empty" );
	    }
    
	    fclose( fp );
	}

	sprintf( dialogTitle, "Environment for pid(%d)", pid);

	do_detailPopup( w, dialogTitle, 25, 132, False, buf);
}

#define MAX_MAPINFO 40000	/* A WAG */

void
do_memInfo( w, pid, call_data )
Widget w;
int pid;
void *call_data;
{
	FILE 	*fp;
	int	len1, len;
	char 	path[MAX_PATH];
	char	buf[MAX_MAPINFO];  /* A guess, we truncate to max */
	char    dialogTitle[128];


	sprintf( path, "/proc/%d/maps", pid );

	fp = fopen( path, "r" );
	if ( fp == NULL )
	{
	    DEBUG1( 1, "Can't open memory map file(%s)", path );
	    sprintf( buf, "%s(%s)\n\n%s\n\n%s",
		"Error opening memory map info file", path, 
		"The reason as reported by the Operating System is:", 
		strerror( errno ) );
	}
	else
	{
	    sprintf( buf, "%s\n%s\n",
	        "Address           Perm Offset   Dev   Inode",
		"----------------- ---- -------- ----- -----" );

            len1 = strlen( buf );

	    len = fread( &(buf[ len1 ]), 1, MAX_MAPINFO - len1, fp );
	    if ( len > 0 )
	    {
	        buf[ len1 + len ] = NULL_CHAR;

strcat( buf, "
==================================================
Address is the address space in the process that 
the memory occupies, perm are the permissions 
decoded using the following:

         r = read
         w = write
         x = execute
         s = shared
         p = private (copy on write)

Offset is the offset into the file/..., Dev is
the devices (major:minor) numbers, and inode is 
file system index on that device. A 0 indicates 
that no inode is associated with the memory 
region, as the case would be with bss(The 
Uninitialized Data Segment)");


	    }
	    else
	    {
	        sprintf( buf, "%s\n", "There appears to be no memory maps" );
	    }
    
	    fclose( fp );
	}

        sprintf( dialogTitle, "Memory Map Info for pid(%d)", pid );

	do_detailPopup( w, dialogTitle, 20, 90, False, buf );
}

#define STAT_BUF_SIZE 40000

void
do_detailInfo( w, pid, call_data )
Widget w;
int pid;
void *call_data;
{
	char buf[STAT_BUF_SIZE];
	char dialogTitle[128];

	getProcessInfo( pid, &(buf[0]) );

	sprintf( dialogTitle, "Process Details for pid(%d)", pid );

	do_detailPopup( w, dialogTitle, 25, 80, False, buf );

}


void
do_detailPopup( w, title, rows, cols, wrap, text )
Widget w;
char *title;
int rows, cols;
Boolean wrap;
char *text;
{
	int             n;
	XmString	title_str, exit_str, message_str, infoStr;
	Widget 		textw, textSW, dw;
        Arg             wargs[16];

        title_str   = XmStringCreate( title, XmSTRING_DEFAULT_CHARSET );
        exit_str    = XmStringCreate("Done", XmSTRING_DEFAULT_CHARSET );
        message_str = XmStringCreate( title, XmSTRING_DEFAULT_CHARSET );

        n = 0;
        XtSetArg( wargs[n], XmNdialogTitle,   title_str ); n++;
        XtSetArg( wargs[n], XmNokLabelString, exit_str ); n++;
        XtSetArg( wargs[n], XmNmessageString, message_str ); n++;
        XtSetArg( wargs[n], XmNmessageAlignment,  XmALIGNMENT_CENTER ); n++;
        XtSetArg( wargs[n], XmNdefaultButtonType, XmDIALOG_OK_BUTTON ); n++;
#ifdef LESSTIF
        XtSetArg( wargs[n], XmNsymbolPixmap, XmUNSPECIFIED_PIXMAP ); n++;
#else
        XtSetArg( wargs[n], XmNsymbolPixmap, NULL ); n++;
#endif
        dw = XmCreateInformationDialog(w, title, wargs, n);
        if ( dw == NULL )
        {
            DEBUG0(1, "do_detailPopup: Failed to create\n");
            return;
        }

        XtUnmanageChild( XmMessageBoxGetChild( dw, XmDIALOG_HELP_BUTTON ));
        XtUnmanageChild( XmMessageBoxGetChild( dw, XmDIALOG_CANCEL_BUTTON));

        n = 0;
        XtSetArg( wargs[n], XmNdialogTitle,   title_str ); n++;
        XtSetArg( wargs[n], XmNeditMode, XmMULTI_LINE_EDIT ); n++;
        XtSetArg( wargs[n], XmNcursorPositionVisible, False ); n++;
        XtSetArg( wargs[n], XmNeditable, False ); n++;
        XtSetArg( wargs[n], XmNcolumns, cols ); n++;
        XtSetArg( wargs[n], XmNrows, rows ); n++;
        XtSetArg( wargs[n], XmNwordWrap, wrap ); n++;
	textw = XmCreateScrolledText( dw, "detailsText", wargs, n );

        n = 0;
        XtSetArg( wargs[n], XmNvalue, text ); n++;
        XtSetValues( textw, wargs, n );

        XtManageChild( textw );

        XtManageChild( dw );
}

char * 	
getProcessInfo( pid, buf )
int   pid;
char  *buf;
{
	char		proc_path[64];
	char		proc_info[64];
	int 		fd;
	int		rc;
	struct  stat    s;
	sysProcInfo	*pp;
	static          int	first_time=1;
	static struct linuxProcessStat lps;


	if ( first_time )
	{
	    lps.cmdLine = NULL;
	    first_time = 0;
	}
	else
	{
	    if ( lps.cmdLine != NULL )
	    {
	    	free( lps.cmdLine );
		lps.cmdLine = NULL;
	    }
	}

	sprintf( proc_path, "/proc/%d", pid );

	/* We call stat to see if we have access */

	rc = stat( proc_path, &s );   
	if ( rc == -1 )
	{
	  sprintf( buf, "%s - %s\n%s %s\n",
		"Error: Can't stat process directory",
		"Process probably exited",
		"System says:",
		strerror( errno ) );
	 	return( buf );
	}

	rc = loadProcessStat( pid, &lps );  /* Gets most fields */
	if ( rc != 0 )
	{
	  sprintf( buf, "%s - %s\n", "Error: Can't load stat values",
					    "Process probably exited" );
	  return( buf );
	}

        rc = loadCmdLine( pid, &lps );	/* Gets just command line */
	if ( rc != 0 )
	{
	  sprintf( buf, "%s - %s\n", "Error: Can't load Cmd Line",
					    "Process probably exited" );
	  return( buf );
	}

        rc = loadProcessStatus( pid, &lps ); /* Get uid/euid/gid/egid */
	if ( rc != 0 )
	{
	  sprintf( buf, "%s - %s\n", "Error: Can't load Status values",
					    "Process probably exited" );
	  return( buf );
	}

	return ( sprintfProcessStat( &lps, buf ) );
}



#define A strcat(buf, b)

char *
sprintfProcessStat( lps, buf )
struct linuxProcessStat *lps;
char *buf;
{
	char b[STAT_BUF_SIZE];


	sprintf( buf, "Name:\t\t%s\n", lps->fname );

	if ( lps->cmdLine == NULL || lps->cmdLine[0] == NULL_CHAR )
		sprintf( b, "CmdLine:\t(%s)\n", &(lps->fname[0]) );
	else
		sprintf( b, "CmdLine:\t%s\n", &(lps->cmdLine[0]) );

	strcat( buf, b );

	sprintf( b,  "Pid:\t\t%d\n", lps->pid ); A;
	sprintf( b,  "State:\t\t%c\n", lps->state ); A;
	sprintf( b,  "Ppid:\t\t%d\n", lps->ppid ); A;
	sprintf( b,  "Pgrp:\t\t%d\n", lps->pgrp ); A;
	sprintf( b,  "SessionId:\t%d\n", lps->sid ); A;


	sprintf( b,  "UserId:\t\t%d\n", lps->uid ); A;
	sprintf( b,  "EUserId:\t%d\n", lps->euid ); A;

	sprintf( b,  "GroupId:\t%d\n", lps->gid ); A;
	sprintf( b,  "EGroupId:\t%d\n", lps->egid ); A;


	sprintf( b,  "Tty:\t\tmajor(%d), minor(%d)\n", major(lps->tty),
							 minor(lps->tty) ); A;

	sprintf( b,  "Tty_gpid:\t%d\n", lps->tpgid ); A;

	sprintf( b,  "Flags:\t\t%x\n", lps->flags ); A;
	sprintf( b,  "MinorFlt:\t%d\n", lps->minflt ); A;
	sprintf( b,  "MajorFlt:\t%d\n", lps->majflt ); A;
	sprintf( b,  "CumMinorFlt:\t%d\n", lps->cminflt ); A;
	sprintf( b,  "CumMajorFlt:\t%d\n", lps->cmajflt ); A;

	sprintf( b,  "UserTime:\t%d\n", lps->utime ); A;
	sprintf( b,  "SysTime:\t%d\n\n", lps->stime ); A;

	sprintf( b,  "Total Process CPU time:  %d min, %d sec, %d nanosec\n\n", 
			((lps->utime + lps->stime) / HZ)/60,
			((lps->utime + lps->stime) / HZ)%60,
			((lps->utime + lps->stime) % HZ) * 10000000 ); A;

	sprintf( b,  "CumulativeUserTime:\t%d\n", lps->cutime ); A;
	sprintf( b,  "CumulativeSysTime:\t%d\n\n", lps->cstime ); A;
	sprintf( b,  "Priority:\t%d\n", lps->priority ); A;
	sprintf( b,  "Nice:\t\t%d\n", lps->nice ); A;
	sprintf( b,  "Timeout:\t%d\n", lps->timeout ); A;
	sprintf( b,  "ItRealValue:\t%d\n", lps->itrealvalue); A;

	sprintf( b,  "StartTime:\t(%d) = %s", startTimeSec( lps->starttime ),
			startTimeStr( lps->starttime )); A;


	sprintf( b,  "Vsize:\t\t%d k\n", lps->vsize / 1024 ); A;
	sprintf( b,  "RSS:\t\t%d Pages\n", lps->rss ); A;
	sprintf( b,  "RLimit:\t\t%d M\n", lps->rlim / (1024*1024) ); A;

	sprintf( b,  "StartCode:\t0x%8.8x\n", lps->startcode ); A;
	sprintf( b,  "EndCode:\t0x%8.8x\n", lps->endcode ); A;
	sprintf( b,  "StartStack:\t0x%8.8lX\n", lps->startstack ); A;
	sprintf( b,  "KernStackkESP:\t0x%8.8lX\n", lps->kstkesp ); A;
	sprintf( b,  "KernStackEIP:\t0x%8.8lX\n", lps->kstkeip ); A;
	sprintf( b,  "Signals:\t0x%x\n", lps->signal ); A;
	sprintf( b,  "SignalsBlocked:\t0x%x\n", lps->blocked ); A;
	sprintf( b,  "SignalsIgnore:\t0x%x\n", lps->sigIgnore ); A;
	sprintf( b,  "SignalsCatch:\t0x%x\n", lps->sigCatch ); A;
	sprintf( b,  "WaitChannel:\t0x%x\n", lps->wchan ); A;
	sprintf( b,  "NSwap:\t\t%d\n", lps->nswap ); A;

	if (( osMajorReleaseNumber > 1 ) && ( osMinorReleaseNumber > 1 ))
	{
	    sprintf( b,  "exitcode:\t%d\n", lps->exitcode ); A;
	}

	return buf;
}

nullToNewline( len, buf )
int len;
char *buf;
{
	int i;

	if ( buf == NULL )
		return;

	for ( i = 0; i < len ; i++ )
	{
		if ( buf[i] == NULL_CHAR )
			buf[i] = '\n';
	}

	buf[i] = NULL_CHAR;
}
