/*
 * Source of main code for clientd.
 * Jason Carlyle 5/12/97
 * Phil White    5/12/97
 *
 * Some code from an example of a generic server using the TCP protocol.
 * Eric Jeschke  4/20/1997
 *
 * Some of this code is adapted from "Unix Network Programming"
 * by W. Richard Stevens, ISBN 0-13-949876-1
 */

#include        <stdio.h>
#include        <string.h>
#include        <unistd.h>
#include        <stdlib.h>
#include        <errno.h>
#include        <stdarg.h>
#include        <time.h>
#include        <fcntl.h>
#include        <sys/types.h>
#include        <sys/socket.h>
#include        <netdb.h>
#include        <netinet/in.h>
#include        <sys/ipc.h>
#include        <sys/msg.h>
#include        <signal.h>

#include	"sig.h"
#include	"daemon.h"
#include        "handlers.h"
#include        "getstatus.h"
#include        "server.h"


/*
 * Log an error message to the log file, along with the time,
 * and process ID of the server.
 */
void
printlog (char *fmt, ...)
{
	char buf[256];
	time_t t;

	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	time (&t);
	fprintf (log, "[%s] %6d: %s\t%s\n",
                    pname, (int) getpid(), ctime (&t), buf);
	fflush (log);
}


/* Handler to take care of connections when the timeout */
void command_timeout(int signo) 
{
  printlog ("Time Out Connection.\n");
  exit(127);
}


/*
 * This function handles all our interaction with the client.
 * 'client_id' is a unique integer representing the client.
 */
void
client_interaction (int client_id, FILE *fdin, FILE*fdout)
{
	char   buf[512], stat[16384];
	int    i;
	struct mymsg           *message;

	printlog ("Opened connection to client %d", client_id);

	/* Set timeout for connection */
	signal(SIGALRM,SIG_DFL);
	signal(SIGALRM,command_timeout); 
	alarm(WAIT_MULT*poll_time);


	/* malloc space for the possible outgoing message */
	message = (struct mymsg *)  malloc(sizeof(struct mymsg));

	/*
         * Loop, handling the client's requests, until the client
	 * closes the connection or we take action to drop it.
         */
	while (fgets (buf, sizeof(buf), fdin) != NULL) {
	  
	  
	  /* Set timeout for connection */
	  alarm(WAIT_MULT*poll_time);
	  

	  /* get rid of control characters. */
	  for (i = 0; i < sizeof(buf); ++i)
	    if (buf[i] < 32) buf[i] = 0;
	  
	  printlog ("serverd %d request is '%s'", client_id, buf);
	  
	  /* BYE command -- serverd wants to quit. */
	  if (! strcmp(buf, "QUIT")) {
	    break;
	  }
	  
	  /* Serverd is requesting the information string */
	  else if (! strncmp(buf, "RETRIEVEINFO", 12)) {
	    getstatus(stat);
	    fprintf (fdout, "%s\n",stat);
	  }
	  
	  /* Server is going down (i.e. you should re-register) */
	  else if (! strncmp(buf, "NUKEME!", 7)) {

	    /* Send a message to parent to notify */
	    message->mtype = 2;
	    message->registered = FALSE;
	    msgsnd(msqid, (void *) message, 2 * sizeof(int), IPC_NOWAIT);
	    kill(0,SIGUSR1);
	    
#ifdef DEBUG
	    fprintf (stderr, "Nuking Machine\n");
#endif
	  }
	  
	  /* Setting the polling interval. */
	  else if (! strncmp(buf, "POLL", 4)) {
	    poll_time = atoi(buf+5);

	    /* Send a message to parent to notify of new poll_time */
	    message->mtype = 1;
	    message->poll_time = poll_time;
	    msgsnd(msqid, (void *) message, 2* sizeof(int) , IPC_NOWAIT);
	    kill(0,SIGUSR1);
	  }
	  
	  /* We don't understand the request... */
	  else {
	    fprintf (fdout, "ERROR Invalid request: '%s'\n", buf);
	    printlog ("unrecognized request: '%s'", buf);
	  }
	}
	
	/* Close everything that is open and exit */
	fclose (fdin);
	fclose (fdout);
	printlog ("closed connection to client %d, errno: %d",
		  client_id, errno);
	fclose (log);
	exit (0);
}




/*
 * Main program.  Command line options are:
 *	-p port		-- specify alternate port
 *	-d		-- initialize as a daemon
 *      -s server       -- machine to register to
 */
int
main (int argc, char *argv[])
{
	int                     i, serv_port, flag_daemon = 1;
	int			sockfd, newsockfd, clilen;
	struct sockaddr_in	cli_addr, serv_addr;
	pid_t                   pid;
        char                   *logfile;
	FILE		       *fdin, *fdout;


	/* Initialize some variables */
	pname      = argv[0];
	serv_port  = CLIENTD_PORT;  
	poll_time = DEFAULT_WAIT;
	register_to = 0;


	/* Process command line args. */
	for (i = 1; i < argc; ++i) {
		if (! strcmp(argv[i], "-d"))
			flag_daemon = 1;
		else if (! strcmp(argv[i], "-p"))
			serv_port = atoi (argv[++i]);
		else if (! strcmp(argv[i], "-s"))
		        register_to = argv[++i];
	}

	/* Complain if a server is not specified. */
	if (register_to == 0) {
	  fprintf(stderr,"You must specify a serverd.\ne.g. clientd -s serverd.sysmanager.org\n");
	  exit(127);
	}

	

	/* Initialize ourself as a daemon if necessary. */
	if ((flag_daemon) && (daemon_start () < 0)) {
		fprintf (stderr, "can't initialize as a daemon.\n");
		exit (127);
	}

 	/* Let's prevent any nasty zombie processes. */
	no_zombies ();

	/* Setup Timeout re-registration signal */ 
	signal(SIGALRM,try_to_register);  
	
	/* Register Message in Queue signal */
	signal(SIGUSR1,check_msg_queue);

	/* Setup communication between child and parent for poll info */
	msqid = msgget(IPC_PRIVATE, 0777);
	signal(SIGINT, remove_msq);
	signal(SIGTERM, remove_msq);
	
	/* Register machine for the first time */	
	raise(SIGALRM);
	
#ifdef DEBUG
	fprintf(stderr,"Should've just raised sigalrm");
#endif 
	
	/* Open a log file for us to write error messages to. */
	logfile = SERVER_LOG;
	log = fopen (logfile, "a");
	if (log == NULL)
	  exit (127);
	
	/* Open a TCP socket (an Internet stream socket). */
	if ( (sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	  printlog ("can't open stream socket-- errno: %d", errno);
	
	/* Bind our local address so that the client can send to us. */
	memset ((char *) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family      = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port        = htons(serv_port);
	
#ifdef DEBUG
	fprintf(stderr,"Binding\n");
#endif
	
	if (bind (sockfd, (struct sockaddr *) &serv_addr,
		  sizeof(serv_addr)) < 0)
	  printlog ("can't bind local address-- errno: %d", errno);

	
#ifdef DEBUG
	fprintf(stderr,"Listening\n");
#endif
	listen (sockfd, 5);
	
	printlog ("Server open for connections.");

        /* Loop and accept connections. */	
	for ( ; ; ) {
	  /*
	   * Wait for a connection from a client.  When we get one
	   * 'newsockfd' is the descriptor of the new socket that was
	   * created to handle the connection.
	   */

	  /* set the appropriate registration timeout */
	  alarm(WAIT_MULT*poll_time);
	  
	  
	  clilen = sizeof(cli_addr);
	  newsockfd = accept(sockfd, (struct sockaddr *) &cli_addr,
						     &clilen);
	  
	  if (newsockfd < 0) {
	    printlog ("accept error-- errno: %d", errno);
	    continue;
	  }
	  
#ifdef DEBUG
	  fprintf(stderr,"poll is set to %d\n",poll_time);
#endif
	  
	  /*
	   * Fork to handle the client.
	   */
	  if ( (pid = fork()) < 0)
	    printlog ("fork error");
	  
	  else if (pid > 0) {	/* PARENT */
	    close (newsockfd);
	    signal(SIGALRM, SIG_IGN);
	    alarm(0); /* Set alarm to 0 so it doesn't 
			 go off while in child */
	    continue;
	    
	  }
	  
	  /* CHILD */

	  /* Remanipulate signal handlers because we're in the child. */
	  signal(SIGUSR1, SIG_IGN);
	  signal(SIGINT, SIG_DFL);
	  signal(SIGTERM, SIG_DFL);
	  
	  close (sockfd);		/* close original socket */

	  /*
	   * Associate stdio lib file descriptors with the socket.
	   * This allows us to use fprintf, fscanf, etc. instead of
	   * read, write.
	   */
	  fdin  = fdopen (newsockfd, "r");
	  fdout = fdopen (newsockfd, "w");

	  /*
	   * Non-tty devices are not line-buffered by default.
	   * This allows us to get and send lines as soon as they
	   * are available.
	   */
	  setvbuf (fdin,  NULL, _IOLBF, 0);
	  setvbuf (fdout, NULL, _IOLBF, 0);
	  
#ifdef DEBUG
	  fprintf(stderr,"before client interaction\n");
#endif
	  
	  /* Go into command mode with the calling serverd. */
	  client_interaction (newsockfd, fdin, fdout);

#ifdef DEBUG
	  fprintf(stderr,"after client interaction\n");
#endif

	  close (newsockfd);    /* For good measure... */
	  exit (0);
	}
	
	printlog ("Server shutting down.");
	fclose (log);
	close (sockfd);
	exit (0);
}
