/* handlers.c

   This file contains all of the functions needed to handle signals that
   may be raised.

   Authors:
      Jason Carlyle
      Phil White

*/

#include        <stdio.h>
#include        <string.h>
#include        <unistd.h>
#include        <stdlib.h>
#include        <errno.h>
#include        <stdarg.h>
#include        <time.h>
#include        <fcntl.h>
#include        <sys/types.h>
#include        <sys/socket.h>
#include        <netdb.h>
#include        <netinet/in.h>
#include        <sys/ipc.h>
#include        <sys/msg.h>
#include        <signal.h>

#include	"sig.h"
#include	"daemon.h"

#include        "handlers.h"
#include        "server.h"


/* When called, usually as a result of loosing connection with the server,
   this function tries to recontact and re-register itself with that 
   server. */
void
try_to_register(int signo) 
{
  int			  sockfd, serv_port, i, done;
  struct hostent         *hp;
  struct sockaddr_in	  serv_addr;
  FILE		         *fdin, *fdout;
  char                   *serv_name;
  sigset_t                newmask, oldmask;
  
#ifdef DEBUG
  fprintf(stderr,"try_to_register: start\n");
#endif
  done = FALSE;
  signal(SIGALRM,SIG_IGN);
  
  for (i=0; !done; i++) {
    done = TRUE;
#ifdef DEBUG
    fprintf(stderr,"try_to_register: in the for loop\n");
#endif
    /* if it is not the first time through the loop (i.e. registration failed
       once already), allow SIGALRM to be raised and then set an alarm */
    if (i>0) { 
      sigemptyset(&newmask);         /* Create a new mask */
      sigaddset(&newmask,SIGALRM);   /* Allow SIGALRM to be raised */
      sigprocmask(SIG_UNBLOCK, &newmask, &oldmask);
      sleep(WAIT_MULT*poll_time);
      sigprocmask(SIG_SETMASK, &oldmask, 0);
    }
#ifdef DEBUG
    fprintf(stderr,"try_to_register: after the sleep\n");
#endif
    serv_name = "localhost"; 	/* This is our default server. */
    serv_port = SERVERD_PORT;	/* This is our default server port. */
    
    /* Doing this lets 'write' fail cleanly if server closes socket: */
    signal (SIGPIPE, SIG_IGN);
    
    /* Lookup the host name and get an internet address. */
    hp = gethostbyname (register_to);
    
    if (hp == NULL) { /* host lookup failed */
      printlog ("Host lookup failed: %s\n",serv_name);
      done = FALSE;
      continue;
    }
    
    /*
     * Fill in the structure "serv_addr" with the address of the
     * server that we want to connect with.
     */
    memset((char *)&serv_addr, 0, sizeof(serv_addr));
    memcpy((char *)&serv_addr.sin_addr, hp->h_addr, hp->h_length);
    serv_addr.sin_family = hp->h_addrtype;
    serv_addr.sin_port   = htons(serv_port);
    

    
    /* Open a TCP socket (an Internet stream socket). */
    if ( (sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      printlog ("can't open stream socket\n");
      done = FALSE;
      continue;
    }
    

    
    /* Connect to the server. */
    if (connect(sockfd, (struct sockaddr *) &serv_addr,
		sizeof(serv_addr)) < 0) {
      printlog ("can't connect to server\n");
      done = FALSE;
      continue;
    }
    
    /*
     * Associate stdio lib file descriptors with the socket.
     * This allows us to use fprintf, fscanf, etc. instead of
     * read, write.
     */
    fdin  = fdopen (sockfd, "r");
    fdout = fdopen (sockfd, "w");
    
    /*
     * Non-tty devices are not line-buffered by default.
     * This allows us to get and send lines as soon as they
     * are available.
     */
    setvbuf (fdin,  NULL, _IOLBF, 0);
    setvbuf (fdout, NULL, _IOLBF, 0);

    /* Register yourself as a NEWCLIENT with the NEWCLIENT command word. */
    fprintf(fdout,"NEWCLIENT\n");

    /* Then remember to QUIT */
    fprintf(fdout,"QUIT");
    
    fflush(fdout);
    
    fclose(fdin);	/* This will close sockfd. */
    fclose(fdout);
    
  }
  
#ifdef DEBUG  
  fprintf(stderr,"Registered!!\n"); 
#endif

  /* Set Timeout for server communication */
  alarm(WAIT_MULT*poll_time);      
  
  /* Register the SIGALRM signal handler again*/
  signal(SIGALRM,try_to_register); 
  return;
}


/* This function is meant to be called whenever a SIGUSR1 is raised.  A 
   SIGUSR1 indicates that a child just put a message in the queue and the
   parent should check for it. 
   */
void check_msg_queue(int signo)
{
  struct mymsg *message;

#ifdef DEBUG  
  fprintf(stderr,"Checking message queue\n");
#endif

  /* If it times out, let it re-register.
     This is also needed by below in case the message is telling the client
     to re-register.*/
  signal(SIGALRM, try_to_register);
  
  message = (struct mymsg *)  malloc(sizeof(struct mymsg));

  /* Check for messages of type 1 (i.e. the fourth field) */
  if ( msgrcv(msqid, (void *) message, 2 * sizeof(int), 1, IPC_NOWAIT) 
       > 0) {
    /* Cast message to struct mymsg type */
    message = (struct mymsg *) message;

#ifdef DEBUG
    fprintf(stderr,"poll_time = %d\n",message->poll_time);
#endif

    /* Update the global polling interval */
    poll_time = message->poll_time;
  }
  
  /* Check for messages of type 2 (i.e. the fourth field) */
  if (msgrcv(msqid, (void *) message, 2 * sizeof(int), 2, IPC_NOWAIT) 
      > 0) {
    /* Cast message to struct mymsg type */
    message = (struct mymsg *) message;
    
#ifdef DEBUG
    fprintf(stderr,"regis = %d\n",message->registered);
#endif

    if (!message->registered)
      raise(SIGALRM);
  }


  signal(SIGUSR1,check_msg_queue); /* re-register signal handler */
  alarm(WAIT_MULT*poll_time);      /* Set Timeout for server communication */
  
}

/* To be called on a SIGINT or SIGTERM.  This function does some cleaning up
   so that message queues aren't left just laying around on the system */
void remove_msq(int signo)
{
  msgctl(msqid, IPC_RMID, 0);
  printlog("Exiting.\n");
  fprintf(stderr,"Goodbye\n");
  exit(0);
}



