/* server.c

   This is the main chunk of code for serverd.

   Authors:
      Jason Carlyle
      Phil White

*/

#include        <stdio.h>
#include        <string.h>
#include        <unistd.h>
#include        <stdlib.h>
#include        <errno.h>
#include        <stdarg.h>
#include        <time.h>
#include        <fcntl.h>
#include        <sys/types.h>
#include        <sys/socket.h>
#include        <sys/time.h>
#include        <netdb.h>
#include        <netinet/in.h>
#include        <sys/ipc.h>
#include        <sys/msg.h>
#include        <signal.h>

#include	"sig.h"
#include	"daemon.h"
#include        "handlers.h"
#include        "server.h"
#include        "machinefuncs.h"

FILE *log; /* Daemon log file */


/*
 * Log an error message to the log file, along with the time,
 * and process ID of the server.
 */
void
printlog (char *fmt, ...)
{
	char buf[256];
	time_t t;

	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	time (&t);
	fprintf (log, "[%s] %6d: %s\t%s\n",
                    pname, (int) getpid(), ctime (&t), buf);
	fflush (log);
}

/* Handle connection timeouts */
void command_timeout(int signo) 
{
  printlog ("Timed Out Connection.\n");
  exit(127);
}



/*
 * This function handles all our interaction with the clientd's.
 */
void server_interaction (FILE *fdin, FILE *fdout, char ip[16])
{
        char     response[16384];
	struct mymsg           *message;
	int      temp;

#ifdef DEBUG
	fprintf(stderr,"In the server Interaction\n"); 
#endif

	message = (struct mymsg *)  malloc(sizeof(struct mymsg));

	/*
         * Just Get their info, tell them some things and leave.
	 */
	
	fprintf (fdout, "RETRIEVEINFO\n");
	fflush (fdout);


	/* Wait for a response. */
	if (fgets (response, sizeof(response), fdin) == NULL) {
	  printlog("Server closed connection, errno: %d\n", errno);
	  strcpy(message->ip,ip);
	  message->mtype = 4; /* Send message to complain */
	  msgsnd(msqid, (void *) message, MSGSIZE , IPC_NOWAIT);
	  free(message);
	  kill(0,SIGUSR1);
	  exit(127);
	}

	/* Process the server response. */
	response[strlen(response)-1] = 0;
	message->mtype = 3;
	strcpy(message->ip,ip);

	/* add the ip# to the response if respone is not null*/
	if (response != 0) {

#ifdef DEBUG
	  fprintf(stderr,"!strcmp(response,''), ip address is %s\n",ip);
#endif
	  strcat(response, ip);
	  strcat(response, "|");
	} 

	strcpy(message->info, response);

#ifdef DEBUG
	fprintf(stderr,"The ip address is %s\n",ip);
	fprintf(stderr,"Going to send: %s\n",message->info); 
	fprintf(stderr,"msqid = %d\n",msqid);
#endif

	/* Put the message in the message queue */
	temp = msgsnd(msqid, (void *) message, 2000 , 0 );

#ifdef DEBUG
	perror("msgsnd");
	fprintf(stderr,"Status of msgsnd is: %d\n",temp); 
#endif

	/* Notify the parent of the message */
	kill(0,SIGUSR1);

	free(message);
	fflush (stdout);	

	/* Set polling interval and quit */

	fprintf (fdout, "POLL %d\n",poll_time);
	fflush (fdout);
	fprintf (fdout, "QUIT\n");
	fflush (fdout);

	
	exit(0);

}


/* Checks the incoming connections port for some change.  

   Note:  This would have been handled more nicely with Asynchronous
          I/O, but I couldn't get that working under linux. 
*/
void check_for_incoming()
{
  fd_set                  server;
  struct timeval          tv;
  int                     status;

  FD_ZERO(&server);
  FD_SET(isockfd, &server);
  tv.tv_sec = 0; tv.tv_usec = 250;

  status = select(isockfd+1, &server, NULL, NULL, &tv);
  if (status > 0 && FD_ISSET (isockfd, &server)) {
    /* Some one has said something, go check it out */
    async_request(0);
  }
}



/*
 * Main program.  Command line options are:
 *	-p port		-- specify alternate port
 *	-d		-- initialize as a non-daemon
 *      -s server       -- machine to register to
 */
int
main (int argc, char *argv[])
{
	int                     i, sleep_time, serv_port, flag_daemon = 1;
	int			osockfd, newsockfd, testing;
	struct sockaddr_in	serv_addr;
	pid_t                   pid;
        char                   *logfile;
	FILE		       *fdin, *fdout;
	struct hostent         *hp;
	struct mymsg           *wmessage;



	/* Initialize a few variables */
	pname      = argv[0];
	serv_port  = SERVERD_PORT;  /* This is our default server port. */
	poll_time = DEFAULT_WAIT;
	testing = FALSE;
	NumberOfMachines = 0;
	current = 0;


	/* Process command line args. */
	for (i = 1; i < argc; ++i) {
		if (! strcmp(argv[i], "-d"))
			flag_daemon = 0;
		else if (! strcmp(argv[i], "-p"))
			serv_port = atoi (argv[++i]);
		else if (! strcmp(argv[i], "-s"))
		        register_to = argv[++i];
	}

	/* Initialize ourself as a daemon if necessary. */
	if ((flag_daemon) && (daemon_start () < 0)) {
		fprintf (stderr, "can't initialize as a daemon.\n");
		exit (127);
	}

 	/* Let's prevent any nasty zombie processes. */
	no_zombies ();


	/* Let the 'read' fail cleanly */
	signal(SIGPIPE, SIG_IGN);

	/* Ignore alarms for 'good measure' */
	signal(SIGALRM, SIG_IGN); 
	
	/* Register Message in Queue signal */
	signal(SIGUSR1,check_msg_queue);
	
	/* Setup communication between child and parent for poll info */
	
	msqid = msgget(IPC_PRIVATE, 0777);

	signal(SIGTERM, remove_msq);
	signal(SIGINT,  remove_msq);

	/* Open a log file for us to write error messages to. */
	logfile = SERVER_LOG;
	log = fopen (logfile, "a");
	if (log == NULL)
		exit (127);

	/* Open a TCP socket (an Internet stream socket). */
	if ( (isockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		printlog ("can't open stream socket-- errno: %d", errno);


	/* Bind our local address so that the client can send to us. */
	memset ((char *) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family      = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port        = htons(serv_port);


	/* Set up socket for incoming communication */
	if (bind (isockfd, (struct sockaddr *) &serv_addr,
                                             sizeof(serv_addr)) < 0)
		printlog ("can't bind local address-- errno: %d", errno);

	listen (isockfd, 5);
	fcntl(isockfd,F_SETOWN, getpid());

        /* Server dispatch loop. */
	printlog ("Server open for connections.");

	for ( ; ; ) {
		/*
	         * Wait for a connection from a client.  When we get one
	         * 'newsockfd' is the descriptor of the new socket that was
	         * created to handle the connection.
	         */
	  /* Necessary because you just can't set NumberOfMachines to 0
	     at any time you want. */
	  if (ZeroMachines) {
	    NumberOfMachines = 0;
	    ZeroMachines = FALSE;
	  }

	  /* Simulate asynchronous I/O */
	  check_for_incoming();


	  while (NumberOfMachines == 0) {/* Watch out for nasty division */
	    check_for_incoming();
	    sleep(4);
	  }

	  
	  /* Delay between information gathering */	  
	  sleep_time = sleep((int) (poll_time/(NumberOfMachines)));
	  
	  /* If a signal interrupts sleep, sleep again until time is up */
	  while (sleep_time != 0)
	    sleep_time = sleep((int) (poll_time/(NumberOfMachines)));
	  
	  
	  /*
	   * Fork to gather information.  If we wanted to be a
	   * single-threaded server, we wouldn't do this.
	   */
	  if ( (pid = fork()) < 0)
	    printlog ("fork error");
	  
	  else if (pid > 0) {	/* PARENT */
	    advance_machine();
	    continue;
	    
	  }
	  
	  signal(SIGINT, SIG_DFL);
	  signal(SIGTERM, SIG_DFL);
	  signal(SIGUSR1,SIG_IGN);
	  
	  /* NumberOfMachines suddenly changed to 0 */
	  if (current->ip == 0) 
	    exit(0);
	  
	  
	  

	  /* Fill in the hostpointer so we can connect a little later */
	  hp = gethostbyname(current->ip);

	  if (hp == NULL) { /* Couldn't look up host */
	    printlog ("Host lookup failed: %s\n", current->ip);
	    exit (1);
	  }

	  /* Define port to connect to */
	  serv_port = CLIENTD_PORT;
	  
	  /*
	   * Fill in the structure "serv_addr" with the address of the
	   * server that we want to connect with.
	   */
	  memset((char *)&serv_addr, 0, sizeof(serv_addr));
	  memcpy((char *)&serv_addr.sin_addr, hp->h_addr, hp->h_length);
	  serv_addr.sin_family = hp->h_addrtype;
	  serv_addr.sin_port   = htons(serv_port);
	  

	  /* Open a TCP socket (an Internet stream socket). */
	  if ( (osockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	    printlog ("can't open stream socket\n");
	    exit (1);
	  }
	  
	  
	  /* Connect to the server. */
	  if (connect(osockfd, (struct sockaddr *) &serv_addr,
		      sizeof(serv_addr)) < 0) {
	    printlog ("can't connect to server\n");
	    
	    /* Send a message if connection failed.
	       This will raise the warncount of the machine.*/
	    wmessage = (struct mymsg *) malloc(sizeof(struct mymsg));
	    wmessage->mtype = 4; /* Send message to complain */
	    strcpy(wmessage->info, current->info);
	    strcpy(wmessage->ip, current->ip);
	    msgsnd(msqid, (void *) wmessage, MSGSIZE , IPC_NOWAIT);

	    /* Notify Parent and free message */
	    kill(0,SIGUSR1);
	    free(wmessage);
	    exit(127);
	  }
	  
	  
	  
	  /*
	   * Associate stdio lib file descriptors with the socket.
	   * This allows us to use fprintf, fscanf, etc. instead of
	   * read, write.
	   */
	  fdin  = fdopen (osockfd, "r");
	  fdout = fdopen (osockfd, "w");
	  
	  /*
	   * Non-tty devices are not line-buffered by default.
	   * This allows us to get and send lines as soon as they
	   * are available.
	   */
	  setvbuf (fdin,  NULL, _IOLBF, 0);
	  setvbuf (fdout, NULL, _IOLBF, 0);
	  
	  /* We have a connection; go handle it. */
	  server_interaction (fdin, fdout, current->ip);
	  
	  fclose(fdin);	/* ?? This will close sockfd. */
	  fclose(fdout);
	  
	  close (newsockfd);    /* For good measure... */
	  
	  
	}
	
	printlog ("Server shutting down.\n");
	fclose (log);
	close (isockfd);
	exit (0);
}

