/* machinfuncs.c

   This file contains the code for manipulation of all functions related to
   the machine list.

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "server.h"
#include "machinefuncs.h"


int add_machine(char ip[16]) {
  struct machine *newmachine;

  /* Watch for special cases */
  if (NumberOfMachines == 0)
    current = 0;
  
  /* Don't add it again */
  if (current != 0 && find_machine(ip) != 0) 
    return -1;
  

  newmachine = malloc(sizeof(struct machine));

  /*Malloc it so it can later be freed like all subsequent cases */
  newmachine->info = (char *) malloc(2); 
  strcpy(newmachine->info,""); /* No info to start out with */

  /* Copy in the ip address of the machine to check */
  newmachine->ip = (char *) malloc(strlen(ip)+1);
  strcpy(newmachine->ip,ip);
  

  if (current == 0) {     /* Adding the first machine */
    current = newmachine;
    newmachine->next = current;
    newmachine->prev = current;
    newmachine->warncount = 0;
  } else {                /* Already on machine added. */
    newmachine->next = current;
    newmachine->prev = current->prev;
    (current->prev)->next = newmachine;
    current->prev = newmachine;
    newmachine->warncount = 0;
    current = newmachine; /* Make it so this one gets new info soon. */
  }

  /* Increment the number of Machines */
  NumberOfMachines++;


  /* Don't allow the NumberOfMachines to exceed the polling interval */
  if (NumberOfMachines > poll_time)
	poll_time = NumberOfMachines;
  return 0;
}

void advance_machine() { /* Move current pointer to the next machine. */

#ifdef DEBUG
  fprintf(stderr,"advance_machine: current->info is %s\n",current->info);
#endif

  current = current->next;
}


/* Remove the given machine from the list */

int remove_machine(char ip[16]) { 
  struct machine *removeme;

#ifdef DEBUG
  fprintf(stderr,"remove_machine:\n");
#endif

  /* If you can't find it, you can't remove it!!! */
  if ( (removeme = (struct machine *) find_machine(ip)) == 0)
    return -1;


  if (NumberOfMachines - 1 == 0) {   /* Special Case for last machine */
    free(current->ip);
    free(current->info);
    current->ip = 0;
    current->info = 0;
    ZeroMachines = TRUE;
    return 0;
  } else {                           /* Case for everyone else... */
    NumberOfMachines--;
  }
  
  /* If current is being deleted, move to next machine */ 
  if (current == removeme)
    current = current->next;

  /* set appropriate pointers and free malloced memory */
  free(removeme->ip);
  (removeme->prev)->next = removeme->next;
  (removeme->next)->prev = removeme->prev;
  free(removeme->info);
  free(removeme);

  return 0;
}

/* Update the machine specified by ip with the information string info */
int update_machine(char ip[16], char *info) {
  struct machine *found;


  /* If not found, you can't update it */
  if ( (found = (struct machine *) find_machine(ip)) == 0)
    return -1;

#ifdef DEBUG
  fprintf(stderr,"Updating %s\n",found->ip);
#endif

  /* Free previous information and copy in the new info */
  free(found->info);
  found->info = (char *) malloc(strlen(info)+1);
  strcpy(found->info,info);

  found->warncount = 0; /* Reset warn count */

  return 0;
}

/* Update the number of warnings for the machine specified by ip */
int warn_machine(char ip[16]) {
  struct machine *mach;
  
#ifdef DEBUG
  fprintf(stderr,"warn_machine:\n");
#endif 


  /* You can't warn if it doesn't exists in the list */
  if ( (mach = (struct machine *) find_machine(ip)) == 0)
    return -1;
  
  /* Get rid of machine after WARNCOUNTLIMIT number of warnings */
  if (++mach->warncount > WARNCOUNTLIMIT)
    remove_machine(ip);
  
  return 0;
}


/* Often used function to return a machine struct based on ip address */
struct machine *find_machine(char ip[16]) {
  struct machine *checker;

#ifdef DEBUG
    fprintf(stderr,"find_machine: %s\n",ip);
#endif


    /* If no IP address, or no machines in list, fail */
    if (ip == NULL || current == 0)
      return 0;
  

    /* compare current with the passed in ip address */
    if (!strcmp(current->ip,ip)) {
      return current;
    }
    
    
    /* Loop and check for the machine */
    checker = current->next;
    while (checker != current) {
      if (!strcmp(checker->ip,ip))
	return checker;
      checker = checker->next;
    }
    
    /* If it's made it this far, ip is not in the list */
    return 0;
} 




