/*
 * Daemon help functions.
 * Eric Jeschke  4/20/1997
 *
 * This code is adapted from "Unix Network Programming"
 * by W. Richard Stevens, ISBN 0-13-949876-1
 */

#include	<stdio.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<signal.h>
#include	<sys/param.h>
#include	<sys/stat.h>
#include	<termios.h>
#include	<errno.h>
extern int	errno;

#ifdef	SIGTSTP		/* true if BSD system */
#include	<sys/file.h>
#include	<sys/ioctl.h>
#endif

#include	"sig.h"
#include	"daemon.h"

extern void err_log (char *msg);


/*
 * Detach a daemon process from login session context.
 */
int
daemon_start (void)
{
	int	            childpid, fd;

	/*
	 * If we were started by init (process 1) from the /etc/inittab file
	 * there's no need to detach.
	 * This test is unreliable due to an unavoidable ambiguity
 	 * if the process is started by some other process and orphaned
	 * (i.e., if the parent process terminates before we are started).
  	 */

	if (getppid() == 1)
  		goto out;

	/*
	 * Ignore the terminal stop signals (BSD).
	 */

#ifdef SIGTTOU
	signal(SIGTTOU, SIG_IGN);
#endif
#ifdef SIGTTIN
	signal(SIGTTIN, SIG_IGN);
#endif
#ifdef SIGTSTP
	signal(SIGTSTP, SIG_IGN);
#endif

	/*
	 * If we were not started in the background, fork and
	 * let the parent exit.  This also guarantees the first child
	 * is not a process group leader.
	 */

	if ( (childpid = fork()) < 0)
		return -1;
	else if (childpid > 0)
		exit(0);	/* parent */

	/*
	 * First child process.
	 *
	 * Disassociate from controlling terminal and process group.
	 * Ensure the process can't reacquire a new controlling terminal.
	 */

#ifdef	SIGTSTP		/* BSD */

	if (setpgid(getpid(), 0) == -1)
		return -1;

	if ( (fd = open("/dev/tty", O_RDWR)) >= 0) {
		ioctl(fd, TIOCNOTTY, (char *)NULL); /* lose controlling tty */
		close(fd);
	}

#else	/* System V */

	if (setpgrp() == -1)
		return -1;

	signal(SIGHUP, SIG_IGN);	/* immune from pgrp leader death */

	if ( (childpid = fork()) < 0)
		return -1;
	else if (childpid > 0)
		exit(0);	/* first child */

	/* second child */
#endif

out:
	/*
	 * Close any open files descriptors.
	 */

	for (fd = 0; fd < NOFILE; fd++)
		close(fd);

	errno = 0;		/* probably got set to EBADF from a close */

	/*
	 * Move the current directory to root, to make sure we
	 * aren't on a mounted filesystem.
	 */

	chdir("/");

	/*
	 * Clear any inherited file mode creation mask.
	 */

	umask(0);

	return 0;
}
