
/* $Id: nwmain.c,v 1.11 2008/10/31 13:58:14 jayrfink Exp $*/

#include "nw.h"

/* Default command line options */
struct options opt = {
	NULL,			/* Interface */
	0,			/* Deamon mode */
	NULL,			/* This is defaulted in the parser below! */
	1,			/* Default refresh delay 1s */
	0,			/* Default timeout */
	100,			/* Default warning score */
	300,			/* Default critical score */
	NULL,			/* Warning script */
	NULL,			/* Critical script */
};

/* Builtin scoring rules */
char default_rules[MAX_RULES][MAX_RULE_LEN] = {
    "quick proto tcp from any port any to any port 135 flags SYN score 40",
    "quick proto tcp from any port any to any port 139 flags SYN score 40",
    "quick proto tcp from any port any to any port 1025 flags SYN score 40",
    "quick proto tcp from any port any to any port 3306 flags SYN score 40",
    "quick proto tcp from any port any to any port 1433 flags SYN score 40",
    "quick proto udp from any port any to any port 1026 score 20",
    "quick proto udp from any port any to any port 1027 score 20",
    "quick proto udp from any port any to any port 1028 score 20",
    "quick proto tcp from any port any to any port any flags SYN score 1",
    "quick proto icmp from any to any score 10",
    0,
    0,
    0,
    0,
    0
};

/* hosts score table */
struct host *hosts;
struct host *tmphosts;
struct net *netfilter;

uint32_t max_id = 1;
uint32_t level_id = 1;
uint32_t hosts_num = 0;
uint32_t packets_max = 0;
uint32_t netfilter_num = 0;
uint32_t vflag = 0;

/* rules table */
struct rule *rules;
uint8_t rules_num;

/* Mutexes */
short m_packet = 0;		/* Get packet function mutex */
short m_display = 0;		/* Display function mutex */

/* Local function */
void usage(void);

/* Main function */
int main(int argc, char *argv[])
{
	int c;			/* Temporary variable */
	char errbuf[PCAP_ERRBUF_SIZE];	/* pcap error buffer */
	pcap_t *handle;		/* pcap handle */
	char *filter;		/* pcap filter */
	struct bpf_program program;	/* BPF filter program */
	struct itimerval timer;	/* Timer characteristic */
	bpf_u_int32 net;	/* our IP adx */
	bpf_u_int32 mask;	/* our netmask */

	/* Init network filter */
	netfilter = calloc(1, sizeof(struct net));
	filter = NULL;

	/* Options data structure- see usage() below for details */
	while (1) {
		static struct option long_options[] = {
			{"critical", required_argument, 0, 'c'},
			{"exec-crit", required_argument, 0, 'C'},
			{"daemon", no_argument, 0, 'd'},
			{"exit", required_argument, 0, 'e'},
			{"interface", required_argument, 0, 'i'},
			{"log", required_argument, 0, 'l'},
			{"packets", required_argument, 0, 'p'},
			{"refresh", required_argument, 0, 'r'},
			{"score", required_argument, 0, 's'},
			{"usage", no_argument, 0, 'u'},
			{"verbose", no_argument, 0, 'v'},
			{"warning", required_argument, 0, 'w'},
			{"exec-warn", required_argument, 0, 'W'},
			{0, 0, 0, 0}
		};

		int option_index = 0;

		c = getopt_long(argc, argv, "c:C:de:i:l:p:r:uvw:W:",
				long_options, &option_index);

		/* Parse command line arguments */
		if (c == -1)
			break;
		switch (c) {
		case 'c':
			opt.critical = atoi(optarg);
			break;
		case 'C':
			opt.script_crit = optarg;
			break;
		case 'd':
			opt.daemon = 1;
			break;
		case 'e':
			if (optarg != NULL && isdigit(*optarg)) {
				opt.timeout = atol(optarg);
				if (optarg[strlen(optarg) - 1] == 's');
				else if (optarg[strlen(optarg) - 1] == 'm')
					/* timeout must be in seconds */
					opt.timeout *= 60;
				else if (optarg[strlen(optarg) - 1] == 'h')
					opt.timeout *= 3600;	/* look up */
				else if (optarg[strlen(optarg) - 1] == 'd')
					opt.timeout *= 3600 * 24;
				else if (!isdigit
					 (optarg[strlen(optarg) - 1]))
					fprintf(stderr,
						"Error: timeout incorrect\n");
				usage();
				exit(1);
			}
			break;
		case 'i':
			opt.pcap_dev = optarg;
			break;
		case 'l':
			opt.log = optarg;
			break;
		case 'p':
			if (optarg != NULL && isdigit(*optarg)) {
				packets_max = atol(optarg);
				if (packets_max < 0) {
					fprintf(stderr,
						"Packets must be > than 0\n");
					exit(1);
				}
			} else {
				fprintf(stderr, "Invalid packet number\n");
				exit(1);
			}
			break;
		case 'r':
			opt.refresh = atoi(optarg);
			break;
		case 'u':
			usage();
			exit(0);
			break;
		case 'v':
			vflag++;
			break;
		case 'w':
			opt.warning = atoi(optarg);
			break;
		case 'W':
			opt.script_warn = optarg;
			break;
		default:
			usage();
			exit(1);
			break;
		}
	}

	/* Make sure it is the root user */
	if (getuid()) {
		fprintf(stderr, "Must be root user.\n");
		exit(1);
	}

	if (!filter)
		filter = copy_argv(&argv[optind]);

	/* Daemonize */
	if (opt.daemon) {
		if (opt.log == NULL)
			opt.log = DEFAULT_LOG_FILE;

		if (daemon(0, 0) != 0) {
			perror("Error entering daemon mode: ");
			exit(-1);
		} else
			message_handler(2, "netward started", opt.log);
	}

	/* Init tables */
	hosts = calloc(2, sizeof(struct host));
	rules = calloc(1, sizeof(struct rule));
	rules_num = 0;
	bzero(hosts, sizeof(hosts));
	bzero(rules, sizeof(rules));

	/*init_rules(opt.rules); */
	if (init_rules2() != 0) {
		fprintf(stderr, "Could not init scoring rules !\n");
		exit(-1);
	}

#ifdef DEBUG
	fprintf(stderr, "debug: loaded %d active rules...\n", rules_num);
#endif

	/* Set up interface */
	if ((!opt.pcap_dev)
	    && ((opt.pcap_dev = pcap_lookupdev(errbuf)) == NULL)) {
		fprintf(stderr, "%s\n", errbuf);
		exit(-1);
	}

	if ((handle =
	     pcap_open_live(opt.pcap_dev, 68, 0, 0, errbuf)) == NULL) {
		fprintf(stderr, "%s\n", errbuf);
		exit(-1);
	}

	pcap_lookupnet(opt.pcap_dev, &net, &mask, errbuf);
	if (filter) {
		if (pcap_compile(handle, &program, filter, 0, net) == -1) {
			fprintf(stderr, "Error - `pcap_compile()'\n");
			return 1;
		}

		if (pcap_setfilter(handle, &program) == -1) {
			fprintf(stderr, "Error - `pcap_setfilter()'\n");
			return 1;
		}

		pcap_freecode(&program);
	}

	/* Set timer to show statistics every delay timeout */
	bzero(&timer, sizeof(timer));
	timer.it_interval.tv_sec = 1;
	timer.it_value.tv_sec = 1;

	if (setitimer(ITIMER_REAL, &timer, NULL) != 0) {
		perror("Timer");
		exit(5);
	}

	if (signal(SIGALRM, ckpkt) == SIG_ERR) {
		perror("Signal");
		exit(6);
	}

	/* Main loop */
	printf("Starting capturing engine on %s...\n", opt.pcap_dev);
	pcap_loop(handle, -1, nwcore, NULL);

	/* Exit program */
	printf("Closing capturing engine...\n");
	pcap_close(handle);
}

/*
 * usage - Simple usage message for 
 */
void usage()
{
	printf(PACKAGE " [option][arguments]\n"
	       PACKAGE
	       " "
	       "[-c|--critical <int>][-C|-exec-crit <path>][-d|--daemon]\n"
	       "[-e|--exit <int>][-i|--interface <dev>][-l|--log <file>]\n"
	       "[-p|--packets <int>][-r|--refresh <int>][-s|--score <path>]\n"
	       "[-u|--usage][-v|--verbose][-vv][-w|--warning <int>]\n"
	       "[-W|-exec-warn <path>][filter-expr]\n"
	       "Options:\n"
	       " -c|--critical  <int>   Set Critical level to int (default 300)\n"
	       " -C|--exec-crit <path>  Execute script at <path> when criticals flag.\n"
	       " -d|--daemon            Go into daemon mode (implies logging)\n"
	       " -e|--exit      <int>   Exit after int seconds\n"
	       " -i|--interface <dev>   Specify the interface to watch\n"
	       " -l|--log       <file>  Log to file default is\n"
	       "                        %s\n"
	       " -p|--packets   <int>   Exit after analyzing int packets\n"
	       " -r|--refresh   <int>   Set report refresh rate to int seconds.\n"
	       " -u|--usage             Display help\n"
	       " -v|--verbose           Turn on verbosity\n"
	       " -vv                    Print all packet header info\n"
	       " -w|--warning   <int>   Set warning level to int (default 200)\n"
	       " -W|--exec-warn <path>  Execute script at path when warning flag.\n",
	       DEFAULT_LOG_FILE);

}
